/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
const jsonSchemaService_1 = require("./jsonSchemaService");
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
class YAMLValidation {
    constructor(jsonSchemaService, promiseConstructor) {
        this.jsonSchemaService = jsonSchemaService;
        this.promise = promiseConstructor;
        this.validationEnabled = true;
    }
    configure(shouldValidate) {
        if (shouldValidate) {
            this.validationEnabled = shouldValidate.validate;
        }
    }
    doValidation(textDocument, yamlDocument) {
        if (!this.validationEnabled) {
            return this.promise.resolve([]);
        }
        return this.jsonSchemaService.getSchemaForResource(textDocument.uri).then(function (schema) {
            var diagnostics = [];
            var added = {};
            let newSchema = schema;
            if (schema) {
                let documentIndex = 0;
                for (let currentYAMLDoc in yamlDocument.documents) {
                    let currentDoc = yamlDocument.documents[currentYAMLDoc];
                    if (schema.schema && schema.schema.schemaSequence && schema.schema.schemaSequence[documentIndex]) {
                        newSchema = new jsonSchemaService_1.ResolvedSchema(schema.schema.schemaSequence[documentIndex]);
                    }
                    let diagnostics = currentDoc.getValidationProblems(newSchema.schema);
                    for (let diag in diagnostics) {
                        let curDiagnostic = diagnostics[diag];
                        currentDoc.errors.push({ location: { start: curDiagnostic.location.start, end: curDiagnostic.location.end }, message: curDiagnostic.message });
                    }
                    documentIndex++;
                }
            }
            if (newSchema && newSchema.errors.length > 0) {
                for (let curDiagnostic of newSchema.errors) {
                    diagnostics.push({
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Error,
                        range: {
                            start: {
                                line: 0,
                                character: 0
                            },
                            end: {
                                line: 0,
                                character: 1
                            }
                        },
                        message: curDiagnostic
                    });
                }
            }
            for (let currentYAMLDoc in yamlDocument.documents) {
                let currentDoc = yamlDocument.documents[currentYAMLDoc];
                currentDoc.errors.concat(currentDoc.warnings).forEach(function (error, idx) {
                    // remove duplicated messages
                    var signature = error.location.start + ' ' + error.location.end + ' ' + error.message;
                    if (!added[signature]) {
                        added[signature] = true;
                        var range = {
                            start: textDocument.positionAt(error.location.start),
                            end: textDocument.positionAt(error.location.end)
                        };
                        diagnostics.push({
                            severity: idx >= currentDoc.errors.length ? vscode_languageserver_types_1.DiagnosticSeverity.Warning : vscode_languageserver_types_1.DiagnosticSeverity.Error,
                            range: range,
                            message: error.message
                        });
                    }
                });
            }
            return diagnostics;
        });
    }
}
exports.YAMLValidation = YAMLValidation;
//# sourceMappingURL=yamlValidation.js.map