(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define(["require", "exports", "../parser/jsonParser", "vscode-languageserver-types", "vscode-nls"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    var jsonParser_1 = require("../parser/jsonParser");
    var vscode_languageserver_types_1 = require("vscode-languageserver-types");
    var nls = require("vscode-nls");
    var localize = nls.loadMessageBundle();
    var JSONValidation = /** @class */ (function () {
        function JSONValidation(jsonSchemaService, promiseConstructor) {
            this.jsonSchemaService = jsonSchemaService;
            this.promise = promiseConstructor;
            this.validationEnabled = true;
        }
        JSONValidation.prototype.configure = function (raw) {
            if (raw) {
                this.validationEnabled = raw.validate;
                this.commentSeverity = raw.allowComments ? jsonParser_1.ProblemSeverity.Ignore : jsonParser_1.ProblemSeverity.Error;
            }
        };
        JSONValidation.prototype.doValidation = function (textDocument, jsonDocument, documentSettings) {
            var _this = this;
            if (!this.validationEnabled) {
                return this.promise.resolve([]);
            }
            var diagnostics = [];
            var added = {};
            var addProblem = function (problem) {
                if (problem.severity === jsonParser_1.ProblemSeverity.Ignore) {
                    return;
                }
                // remove duplicated messages
                var signature = problem.location.start + ' ' + problem.location.end + ' ' + problem.message;
                if (!added[signature]) {
                    added[signature] = true;
                    var range = {
                        start: textDocument.positionAt(problem.location.start),
                        end: textDocument.positionAt(problem.location.end)
                    };
                    var severity = problem.severity === jsonParser_1.ProblemSeverity.Error ? vscode_languageserver_types_1.DiagnosticSeverity.Error : vscode_languageserver_types_1.DiagnosticSeverity.Warning;
                    diagnostics.push({ severity: severity, range: range, message: problem.message });
                }
            };
            return this.jsonSchemaService.getSchemaForResource(textDocument.uri, jsonDocument).then(function (schema) {
                var trailingCommaSeverity = documentSettings ? documentSettings.trailingCommas : jsonParser_1.ProblemSeverity.Error;
                var commentSeverity = documentSettings ? documentSettings.comments : _this.commentSeverity;
                if (schema) {
                    if (schema.errors.length && jsonDocument.root) {
                        var astRoot = jsonDocument.root;
                        var property = astRoot.type === 'object' ? astRoot.getFirstProperty('$schema') : null;
                        if (property) {
                            var node = property.value || property;
                            addProblem({ location: { start: node.start, end: node.end }, message: schema.errors[0], severity: jsonParser_1.ProblemSeverity.Warning });
                        }
                        else {
                            addProblem({ location: { start: astRoot.start, end: astRoot.start + 1 }, message: schema.errors[0], severity: jsonParser_1.ProblemSeverity.Warning });
                        }
                    }
                    else {
                        var semanticErrors = jsonDocument.validate(schema.schema);
                        if (semanticErrors) {
                            semanticErrors.forEach(addProblem);
                        }
                    }
                    if (schemaAllowsComments(schema.schema)) {
                        trailingCommaSeverity = commentSeverity = jsonParser_1.ProblemSeverity.Ignore;
                    }
                }
                jsonDocument.syntaxErrors.forEach(function (p) {
                    if (p.code === jsonParser_1.ErrorCode.TrailingComma) {
                        p.severity = trailingCommaSeverity;
                    }
                    addProblem(p);
                });
                if (commentSeverity !== jsonParser_1.ProblemSeverity.Ignore) {
                    var message_1 = localize('InvalidCommentToken', 'Comments are not permitted in JSON.');
                    jsonDocument.comments.forEach(function (c) {
                        addProblem({ location: c, severity: commentSeverity, message: message_1 });
                    });
                }
                return diagnostics;
            });
        };
        return JSONValidation;
    }());
    exports.JSONValidation = JSONValidation;
    function schemaAllowsComments(schemaRef) {
        if (schemaRef && typeof schemaRef === 'object') {
            if (schemaRef.allowComments) {
                return true;
            }
            if (schemaRef.allOf) {
                return schemaRef.allOf.some(schemaAllowsComments);
            }
        }
        return false;
    }
});
//# sourceMappingURL=jsonValidation.js.map