/**
 * Copyright (c) 2010-2012, Balazs Grill, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Balazs Grill - initial API and implementation
 */
package org.eclipse.viatra.query.tooling.ui.migrator;

import com.google.common.base.Objects;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import org.eclipse.core.filebuffers.FileBuffers;
import org.eclipse.core.filebuffers.ITextFileBuffer;
import org.eclipse.core.filebuffers.ITextFileBufferManager;
import org.eclipse.core.filebuffers.LocationKind;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.ImportDeclaration;
import org.eclipse.jdt.core.dom.Name;
import org.eclipse.jdt.core.dom.QualifiedName;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.SimpleType;
import org.eclipse.jdt.core.dom.rewrite.ASTRewrite;
import org.eclipse.jface.text.IDocument;
import org.eclipse.swt.widgets.Display;
import org.eclipse.text.edits.TextEdit;
import org.eclipse.viatra.query.tooling.core.project.ProjectGenerationHelper;
import org.eclipse.viatra.query.tooling.ui.migrator.FileStringReplacer;
import org.eclipse.viatra.query.tooling.ui.migrator.JavaProjectMigratorData;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;

@SuppressWarnings("all")
public class JavaProjectMigrator extends JavaProjectMigratorData {
  private final IJavaProject javaProject;
  
  public JavaProjectMigrator(final IProject project) {
    this.javaProject = JavaCore.create(project);
  }
  
  public JavaProjectMigrator(final IJavaProject project) {
    this.javaProject = project;
  }
  
  public void migrate(final IProgressMonitor monitor) {
    try {
      final SubMonitor m = SubMonitor.convert(monitor);
      final LinkedList<ICompilationUnit> list = CollectionLiterals.<ICompilationUnit>newLinkedList();
      IPackageFragment[] _packageFragments = this.javaProject.getPackageFragments();
      for (final IPackageFragment p : _packageFragments) {
        int _kind = p.getKind();
        boolean _equals = (_kind == IPackageFragmentRoot.K_SOURCE);
        if (_equals) {
          CollectionExtensions.<ICompilationUnit>addAll(list, p.getCompilationUnits());
        }
      }
      int _size = list.size();
      int _multiply = (_size * 2);
      int _plus = (_multiply + 1);
      m.beginTask("Migrating project", _plus);
      final IProject project = this.javaProject.getProject();
      boolean _isOpenPDEProject = ProjectGenerationHelper.isOpenPDEProject(project);
      if (_isOpenPDEProject) {
        ProjectGenerationHelper.replaceBundleDependencies(project, JavaProjectMigratorData.bundleRenames, JavaProjectMigratorData.bundleVersions, m.newChild(1));
        ProjectGenerationHelper.ensureBundleDependenciesAndPackageImports(project, ProjectGenerationHelper.DEFAULT_VIATRA_BUNDLE_REQUIREMENTS, ProjectGenerationHelper.DEFAULT_VIATRA_IMPORT_PACKAGES, m.newChild(1));
      }
      final ITextFileBufferManager bufferManager = FileBuffers.getTextFileBufferManager();
      for (final ICompilationUnit unit : list) {
        Display _default = Display.getDefault();
        _default.syncExec(new Runnable() {
          @Override
          public void run() {
            try {
              final ASTNode ast = JavaProjectMigrator.this.parse(unit, m.newChild(1));
              final ASTRewrite rewrite = JavaProjectMigrator.this.collectChanges(ast);
              m.worked(1);
              final TextEdit textEdit = rewrite.rewriteAST();
              final IPath path = unit.getPath();
              try {
                bufferManager.connect(path, LocationKind.IFILE, null);
                final ITextFileBuffer textFileBuffer = bufferManager.getTextFileBuffer(path, LocationKind.IFILE);
                final IDocument document = textFileBuffer.getDocument();
                textEdit.apply(document);
                textFileBuffer.commit(null, false);
              } finally {
                bufferManager.disconnect(path, LocationKind.IFILE, null);
                m.worked(1);
              }
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        });
      }
      final LinkedList<IFile> xtendlist = CollectionLiterals.<IFile>newLinkedList();
      IPackageFragmentRoot[] _packageFragmentRoots = this.javaProject.getPackageFragmentRoots();
      for (final IPackageFragmentRoot p_1 : _packageFragmentRoots) {
        int _kind_1 = p_1.getKind();
        boolean _equals_1 = (_kind_1 == IPackageFragmentRoot.K_SOURCE);
        if (_equals_1) {
          final IResourceVisitor _function = (IResource it) -> {
            if (((it instanceof IFile) && "xtend".equalsIgnoreCase(it.getFileExtension()))) {
              xtendlist.add(((IFile) it));
            }
            return (it instanceof IContainer);
          };
          ResourcesPlugin.getWorkspace().getRoot().getFolder(p_1.getPath()).accept(_function);
        }
      }
      for (final IFile file : xtendlist) {
        this.updateXTend(file);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void updateXTend(final IFile xtendFile) {
    try {
      final FileStringReplacer replacer = new FileStringReplacer(xtendFile);
      Set<Map.Entry<String, String>> _entrySet = JavaProjectMigratorData.qualifiedNameRenames.entrySet();
      for (final Map.Entry<String, String> entry : _entrySet) {
        boolean _endsWith = entry.getKey().endsWith(".");
        if (_endsWith) {
          replacer.replacePattern(entry.getKey(), entry.getValue());
        } else {
          String _key = entry.getKey();
          String _plus = ("import " + _key);
          String _value = entry.getValue();
          String _plus_1 = ("import " + _value);
          boolean _replacePattern = replacer.replacePattern(_plus, _plus_1);
          if (_replacePattern) {
            replacer.replacePattern(this.getLastSegment(entry.getKey()), this.getLastSegment(entry.getValue()));
          } else {
            replacer.replacePattern(entry.getKey(), entry.getValue());
          }
        }
      }
      replacer.save();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ASTNode parse(final ICompilationUnit unit, final IProgressMonitor monitor) {
    final ASTParser parser = ASTParser.newParser(AST.JLS8);
    parser.setKind(ASTParser.K_COMPILATION_UNIT);
    parser.setSource(unit);
    parser.setResolveBindings(true);
    return parser.createAST(monitor);
  }
  
  public String replaceName(final String oldValue, final Map.Entry<String, String> entry) {
    return oldValue.replace(entry.getKey(), entry.getValue());
  }
  
  public String getLastSegment(final String fqn) {
    boolean _endsWith = fqn.endsWith(".");
    if (_endsWith) {
      return "";
    }
    final int i = fqn.lastIndexOf(".");
    if ((i >= 0)) {
      return fqn.substring((i + 1));
    }
    return fqn;
  }
  
  public void createChange(final ASTRewrite rewrite, final ImportDeclaration importDeclaration, final Map<String, String> typeRenames) {
    final String fullyQualifiedName = importDeclaration.getName().getFullyQualifiedName();
    Set<Map.Entry<String, String>> _entrySet = JavaProjectMigrator.qualifiedNameRenames.entrySet();
    for (final Map.Entry<String, String> entry : _entrySet) {
      boolean _startsWith = fullyQualifiedName.startsWith(entry.getKey());
      if (_startsWith) {
        final String newName = this.replaceName(fullyQualifiedName, entry);
        final String tn_old = this.getLastSegment(fullyQualifiedName);
        final String tn_new = this.getLastSegment(newName);
        boolean _notEquals = (!Objects.equal(tn_old, tn_new));
        if (_notEquals) {
          typeRenames.put(tn_old, tn_new);
        }
        rewrite.set(importDeclaration, ImportDeclaration.NAME_PROPERTY, importDeclaration.getAST().newName(newName), null);
        return;
      }
    }
  }
  
  public void createChange(final ASTRewrite rewrite, final SimpleType type, final Map<String, String> typeRenames) {
    final Name name = type.getName();
    if ((name instanceof QualifiedName)) {
      final String fullyQualifiedName = ((QualifiedName)name).getFullyQualifiedName();
      Set<Map.Entry<String, String>> _entrySet = JavaProjectMigrator.qualifiedNameRenames.entrySet();
      for (final Map.Entry<String, String> entry : _entrySet) {
        boolean _startsWith = fullyQualifiedName.startsWith(entry.getKey());
        if (_startsWith) {
          rewrite.set(type, ImportDeclaration.NAME_PROPERTY, type.getAST().newName(this.replaceName(fullyQualifiedName, entry)), null);
          return;
        }
      }
    }
    if ((name instanceof SimpleName)) {
      final String n = ((SimpleName)name).getFullyQualifiedName();
      boolean _containsKey = typeRenames.containsKey(n);
      if (_containsKey) {
        rewrite.set(type, SimpleType.NAME_PROPERTY, type.getAST().newName(typeRenames.get(n)), null);
      }
    }
  }
  
  public ASTRewrite collectChanges(final ASTNode node) {
    final ASTRewrite rewrite = ASTRewrite.create(node.getAST());
    final HashMap<String, String> typeRenames = CollectionLiterals.<String, String>newHashMap();
    node.accept(new ASTVisitor() {
      @Override
      public boolean visit(final ImportDeclaration node) {
        boolean _xblockexpression = false;
        {
          JavaProjectMigrator.this.createChange(rewrite, node, typeRenames);
          _xblockexpression = super.visit(node);
        }
        return _xblockexpression;
      }
      
      @Override
      public boolean visit(final SimpleType node) {
        boolean _xblockexpression = false;
        {
          JavaProjectMigrator.this.createChange(rewrite, node, typeRenames);
          _xblockexpression = super.visit(node);
        }
        return _xblockexpression;
      }
    });
    return rewrite;
  }
}
