package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link SwarmJoin}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableSwarmJoin.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableSwarmJoin implements SwarmJoin {
  private final String listenAddr;
  private final @Nullable String advertiseAddr;
  private final List<String> remoteAddrs;
  private final String joinToken;

  private ImmutableSwarmJoin(
      String listenAddr,
      @Nullable String advertiseAddr,
      List<String> remoteAddrs,
      String joinToken) {
    this.listenAddr = listenAddr;
    this.advertiseAddr = advertiseAddr;
    this.remoteAddrs = remoteAddrs;
    this.joinToken = joinToken;
  }

  /**
   * @return The value of the {@code listenAddr} attribute
   */
  @JsonProperty("ListenAddr")
  @Override
  public String listenAddr() {
    return listenAddr;
  }

  /**
   * @return The value of the {@code advertiseAddr} attribute
   */
  @JsonProperty("AdvertiseAddr")
  @Override
  public @Nullable String advertiseAddr() {
    return advertiseAddr;
  }

  /**
   * @return The value of the {@code remoteAddrs} attribute
   */
  @JsonProperty("RemoteAddrs")
  @Override
  public List<String> remoteAddrs() {
    return remoteAddrs;
  }

  /**
   * @return The value of the {@code joinToken} attribute
   */
  @JsonProperty("JoinToken")
  @Override
  public String joinToken() {
    return joinToken;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link SwarmJoin#listenAddr() listenAddr} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for listenAddr
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableSwarmJoin withListenAddr(String value) {
    String newValue = Objects.requireNonNull(value, "listenAddr");
    if (this.listenAddr.equals(newValue)) return this;
    return new ImmutableSwarmJoin(newValue, this.advertiseAddr, this.remoteAddrs, this.joinToken);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link SwarmJoin#advertiseAddr() advertiseAddr} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for advertiseAddr (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableSwarmJoin withAdvertiseAddr(@Nullable String value) {
    if (Objects.equals(this.advertiseAddr, value)) return this;
    return new ImmutableSwarmJoin(this.listenAddr, value, this.remoteAddrs, this.joinToken);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link SwarmJoin#remoteAddrs() remoteAddrs}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableSwarmJoin withRemoteAddrs(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableSwarmJoin(this.listenAddr, this.advertiseAddr, newValue, this.joinToken);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link SwarmJoin#remoteAddrs() remoteAddrs}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of remoteAddrs elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableSwarmJoin withRemoteAddrs(Iterable<String> elements) {
    if (this.remoteAddrs == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableSwarmJoin(this.listenAddr, this.advertiseAddr, newValue, this.joinToken);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link SwarmJoin#joinToken() joinToken} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for joinToken
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableSwarmJoin withJoinToken(String value) {
    String newValue = Objects.requireNonNull(value, "joinToken");
    if (this.joinToken.equals(newValue)) return this;
    return new ImmutableSwarmJoin(this.listenAddr, this.advertiseAddr, this.remoteAddrs, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableSwarmJoin} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableSwarmJoin
        && equalsByValue((ImmutableSwarmJoin) another);
  }

  private boolean equalsByValue(ImmutableSwarmJoin another) {
    return listenAddr.equals(another.listenAddr)
        && Objects.equals(advertiseAddr, another.advertiseAddr)
        && remoteAddrs.equals(another.remoteAddrs)
        && joinToken.equals(another.joinToken);
  }

  /**
   * Computes a hash code from attributes: {@code listenAddr}, {@code advertiseAddr}, {@code remoteAddrs}, {@code joinToken}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + listenAddr.hashCode();
    h += (h << 5) + Objects.hashCode(advertiseAddr);
    h += (h << 5) + remoteAddrs.hashCode();
    h += (h << 5) + joinToken.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code SwarmJoin} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "SwarmJoin{"
        + "listenAddr=" + listenAddr
        + ", advertiseAddr=" + advertiseAddr
        + ", remoteAddrs=" + remoteAddrs
        + ", joinToken=" + joinToken
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link SwarmJoin} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable SwarmJoin instance
   */
  public static ImmutableSwarmJoin copyOf(SwarmJoin instance) {
    if (instance instanceof ImmutableSwarmJoin) {
      return (ImmutableSwarmJoin) instance;
    }
    return ImmutableSwarmJoin.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableSwarmJoin ImmutableSwarmJoin}.
   * <pre>
   * ImmutableSwarmJoin.builder()
   *    .listenAddr(String) // required {@link SwarmJoin#listenAddr() listenAddr}
   *    .advertiseAddr(String | null) // nullable {@link SwarmJoin#advertiseAddr() advertiseAddr}
   *    .remoteAddr|addAllRemoteAddrs(String) // {@link SwarmJoin#remoteAddrs() remoteAddrs} elements
   *    .joinToken(String) // required {@link SwarmJoin#joinToken() joinToken}
   *    .build();
   * </pre>
   * @return A new ImmutableSwarmJoin builder
   */
  public static ImmutableSwarmJoin.Builder builder() {
    return new ImmutableSwarmJoin.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableSwarmJoin ImmutableSwarmJoin}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements SwarmJoin.Builder {
    private static final long INIT_BIT_LISTEN_ADDR = 0x1L;
    private static final long INIT_BIT_JOIN_TOKEN = 0x2L;
    private long initBits = 0x3L;

    private String listenAddr;
    private String advertiseAddr;
    private List<String> remoteAddrs = new ArrayList<String>();
    private String joinToken;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code SwarmJoin} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(SwarmJoin instance) {
      Objects.requireNonNull(instance, "instance");
      this.listenAddr(instance.listenAddr());
      @Nullable String advertiseAddrValue = instance.advertiseAddr();
      if (advertiseAddrValue != null) {
        advertiseAddr(advertiseAddrValue);
      }
      addAllRemoteAddrs(instance.remoteAddrs());
      this.joinToken(instance.joinToken());
      return this;
    }

    /**
     * Initializes the value for the {@link SwarmJoin#listenAddr() listenAddr} attribute.
     * @param listenAddr The value for listenAddr 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ListenAddr")
    public final Builder listenAddr(String listenAddr) {
      this.listenAddr = Objects.requireNonNull(listenAddr, "listenAddr");
      initBits &= ~INIT_BIT_LISTEN_ADDR;
      return this;
    }

    /**
     * Initializes the value for the {@link SwarmJoin#advertiseAddr() advertiseAddr} attribute.
     * @param advertiseAddr The value for advertiseAddr (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("AdvertiseAddr")
    public final Builder advertiseAddr(@Nullable String advertiseAddr) {
      this.advertiseAddr = advertiseAddr;
      return this;
    }

    /**
     * Adds one element to {@link SwarmJoin#remoteAddrs() remoteAddrs} list.
     * @param element A remoteAddrs element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder remoteAddr(String element) {
      this.remoteAddrs.add(Objects.requireNonNull(element, "remoteAddrs element"));
      return this;
    }

    /**
     * Adds elements to {@link SwarmJoin#remoteAddrs() remoteAddrs} list.
     * @param elements An array of remoteAddrs elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder remoteAddrs(String... elements) {
      for (String element : elements) {
        this.remoteAddrs.add(Objects.requireNonNull(element, "remoteAddrs element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link SwarmJoin#remoteAddrs() remoteAddrs} list.
     * @param elements An iterable of remoteAddrs elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("RemoteAddrs")
    public final Builder remoteAddrs(Iterable<String> elements) {
      this.remoteAddrs.clear();
      return addAllRemoteAddrs(elements);
    }

    /**
     * Adds elements to {@link SwarmJoin#remoteAddrs() remoteAddrs} list.
     * @param elements An iterable of remoteAddrs elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllRemoteAddrs(Iterable<String> elements) {
      for (String element : elements) {
        this.remoteAddrs.add(Objects.requireNonNull(element, "remoteAddrs element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link SwarmJoin#joinToken() joinToken} attribute.
     * @param joinToken The value for joinToken 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("JoinToken")
    public final Builder joinToken(String joinToken) {
      this.joinToken = Objects.requireNonNull(joinToken, "joinToken");
      initBits &= ~INIT_BIT_JOIN_TOKEN;
      return this;
    }

    /**
     * Builds a new {@link ImmutableSwarmJoin ImmutableSwarmJoin}.
     * @return An immutable instance of SwarmJoin
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableSwarmJoin build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableSwarmJoin(listenAddr, advertiseAddr, createUnmodifiableList(true, remoteAddrs), joinToken);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_LISTEN_ADDR) != 0) attributes.add("listenAddr");
      if ((initBits & INIT_BIT_JOIN_TOKEN) != 0) attributes.add("joinToken");
      return "Cannot build SwarmJoin, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
