//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.spec;

import static org.eclipse.escet.common.java.Strings.fmt;

import java.math.BigInteger;
import java.util.Arrays;
import java.util.stream.Collectors;

import org.eclipse.escet.cif.bdd.conversion.bitvectors.BddBitVector;
import org.eclipse.escet.cif.bdd.conversion.bitvectors.SignedBddBitVector;
import org.eclipse.escet.cif.bdd.conversion.bitvectors.UnsignedBddBitVector;
import org.eclipse.escet.cif.common.CifTextUtils;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.position.metamodel.position.PositionObject;

import com.github.javabdd.BDDVarSet;

/** A CIF/BDD variable. Represents CIF state in a BDD representation. */
public abstract class CifBddVariable {
    /** The end-user readable name of the CIF/BDD variable. */
    public final String readableName;

    /** The name of the CIF/BDD variable, without escaping. */
    public final String rawName;

    /**
     * The 0-based group index number. CIF/BDD variables are in the same group if their {@link #domain domains} are
     * interleaved. Is {@code -1} until actual value is set.
     */
    public int group = -1;

    /** The number of potential values of the CIF variable. Is at least one. */
    public final int count;

    /** The lower bound (minimum value) of the CIF variable. */
    public final int lower;

    /** The upper bound (maximum value) of the CIF variable. Is greater than or equal to {@link #lower}. */
    public final int upper;

    /**
     * The BDD domain of the variable. For updates, this represents the pre/old domain. Is {@code null} until actual
     * domain is set.
     */
    public CifBddDomain domain;

    /**
     * The BDD domain of the variable. For updates, this represents the post/new domain. Is {@code null} until actual
     * domain is set.
     */
    public CifBddDomain domainNew;

    /**
     * The extra BDD domains of the variable, besides the pre/old and post/new domains. Is {@code null} until actual
     * domains are set. May be empty.
     */
    public CifBddDomain[] domainsExtra;

    /**
     * Constructor for the {@link CifBddVariable} class.
     *
     * <p>
     * The range {@code [lower..upper]} must not contain more than {@link Integer#MAX_VALUE} values.
     * </p>
     *
     * @param obj The CIF object that corresponds to this CIF/BDD variable. Must be a {@link CifTextUtils#getName named}
     *     CIF object.
     * @param lower The lower bound (minimum value) of the CIF variable.
     * @param upper The upper bound (maximum value) of the CIF variable. Must be greater than or equal to {@code lower}.
     */
    public CifBddVariable(PositionObject obj, int lower, int upper) {
        Assert.check(lower <= upper);

        long count = (long)upper - (long)lower + 1;
        Assert.check(count > 0);
        Assert.check(count <= Integer.MAX_VALUE);

        this.readableName = CifTextUtils.getAbsName(obj);
        this.rawName = CifTextUtils.getAbsName(obj, false);
        this.count = (int)count;
        this.lower = lower;
        this.upper = upper;
    }

    /**
     * Returns the size of the {@link #domain}, {@link #domainNew} and each of the {@link #domainsExtra} to create for
     * this CIF/BDD variable. This is the not the number of BDD variables, but the number of actual distinct values that
     * can be represented by the BDD variables. For a variable with {@code n} BDD variables, the size is {@code 1 << n}.
     *
     * @return The size of the BDD domains to create for this CIF/BDD variable.
     */
    public BigInteger getBddDomainSize() {
        return BigInteger.ONE.shiftLeft(getBddVarCount());
    }

    /**
     * Returns the number of BDD variables to use to represent this CIF/BDD variable. Is obtained from {@link #domain}
     * if present, and is computed otherwise.
     *
     * @return The number of BDD variables to use to represent this CIF/BDD variable.
     */
    public int getBddVarCount() {
        // Ask the domain, if it is present.
        if (domain != null) {
            return domain.getVarCount();
        }

        // Compute it.
        if (lower < 0) {
            return Math.max(SignedBddBitVector.getMinimumLength(lower), SignedBddBitVector.getMinimumLength(upper));
        } else {
            return UnsignedBddBitVector.getMinimumLength(upper);
        }
    }

    /**
     * Returns the variable support for this CIF/BDD variable, covering its {@link #domain} and {@link #domainNew}
     * domains, and optionally its {@link #domainsExtra} domains.
     *
     * @param includeExtraDomains Whether to include the extra domains.
     * @return The BDD variable support for this CIF/BDD variable.
     */
    public BDDVarSet getSupport(boolean includeExtraDomains) {
        BDDVarSet support = domain.makeVarSet().unionWith(domainNew.makeVarSet());
        if (includeExtraDomains) {
            for (CifBddDomain domainExtra: domainsExtra) {
                support = support.unionWith(domainExtra.makeVarSet());
            }
        }
        return support;
    }

    /**
     * Create a BDD bit vector for this CIF/BDD variable.
     *
     * @param newDomain Whether to create a bit vector for the pre/old domain of the variable ({@code false}) or the
     *     post/new domain of the variable ({@code true}).
     * @return The BDD bit vector.
     */
    public BddBitVector<?, ?> createBitVector(boolean newDomain) {
        CifBddDomain domain = newDomain ? this.domainNew : this.domain;
        if (lower < 0) {
            return SignedBddBitVector.createFromDomain(domain);
        } else {
            return UnsignedBddBitVector.createFromDomain(domain);
        }
    }

    /**
     * Create a BDD bit vector for this CIF/BDD variable.
     *
     * @param extraDomainIdx The 0-based index into {@link #domainsExtra} for which to create the bit vector.
     * @return The BDD bit vector.
     */
    public BddBitVector<?, ?> createBitVectorExtraDomain(int extraDomainIdx) {
        if (lower < 0) {
            return SignedBddBitVector.createFromDomain(domainsExtra[extraDomainIdx]);
        } else {
            return UnsignedBddBitVector.createFromDomain(domainsExtra[extraDomainIdx]);
        }
    }

    @Override
    public String toString() {
        return toString("Variable: ");
    }

    /**
     * Returns a textual representation of the kind of the CIF/BDD variable, relating to the kind of original CIF object
     * it corresponds to.
     *
     * @return The textual representation.
     */
    public abstract String getKindText();

    /**
     * Returns a textual representation of the type of the CIF/BDD variable. Returns {@code null} if the variable has no
     * type.
     *
     * @return The textual representation of the type, or {@code null}.
     */
    public abstract String getTypeText();

    /**
     * Returns a textual representation of the CIF/BDD variable.
     *
     * @param prefix The prefix to use, e.g. {@code "Variable: "} or {@code ""}.
     * @return The textual representation.
     */
    public String toString(String prefix) {
        String domainsExtraTxt = domainsExtra.length == 0 ? "" : ("+" + Arrays.stream(domainsExtra)
                .map(d -> fmt("%,d", d.getIndex())).collect(Collectors.joining("+")));
        return fmt("%s%s (group: %,d, domain: %,d+%,d%s, BDD variables: %,d, CIF/BDD values: %,d/%,d)", prefix,
                toStringInternal(), group, domain.getIndex(), domainNew.getIndex(), domainsExtraTxt,
                domain.getVarCount(), count, getBddDomainSize());
    }

    /**
     * Returns a textual representation of the CIF/BDD variable, to use as part of the output for {@link #toString}.
     *
     * @return The textual representation.
     */
    protected abstract String toStringInternal();
}
