/**
 * Copyright (c) 2016, 2018 Kiel University and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.elk.core.debug.grandom.ui;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.elk.core.debug.grandom.gRandom.Configuration;
import org.eclipse.elk.core.debug.grandom.gRandom.ConstraintType;
import org.eclipse.elk.core.debug.grandom.gRandom.DoubleQuantity;
import org.eclipse.elk.core.debug.grandom.gRandom.Edges;
import org.eclipse.elk.core.debug.grandom.gRandom.Flow;
import org.eclipse.elk.core.debug.grandom.gRandom.FlowType;
import org.eclipse.elk.core.debug.grandom.gRandom.Form;
import org.eclipse.elk.core.debug.grandom.gRandom.Formats;
import org.eclipse.elk.core.debug.grandom.gRandom.Hierarchy;
import org.eclipse.elk.core.debug.grandom.gRandom.Nodes;
import org.eclipse.elk.core.debug.grandom.gRandom.Ports;
import org.eclipse.elk.core.debug.grandom.gRandom.RandGraph;
import org.eclipse.elk.core.debug.grandom.gRandom.Side;
import org.eclipse.elk.core.debug.grandom.gRandom.Size;
import org.eclipse.elk.core.debug.grandom.generators.GeneratorOptions;
import org.eclipse.elk.core.debug.grandom.generators.RandomGraphGenerator;
import org.eclipse.elk.core.debug.grandom.ui.GeneratorOptionsUtil;
import org.eclipse.elk.core.options.PortConstraints;
import org.eclipse.elk.graph.ElkNode;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.MapPropertyHolder;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;

@SuppressWarnings("all")
public class GRandomGraphMaker {
  private final Iterable<Configuration> configs;
  
  private final int DEFAULT_NUM_GRAPHS = 1;
  
  private final String DEFAULT_NAME = "random";
  
  private final String DEFAULT_FORMAT = Formats.ELKT.toString();
  
  public GRandomGraphMaker(final RandGraph rdg) {
    this.configs = rdg.getConfigs();
  }
  
  public void gen(final IProject project) {
    try {
      for (final Configuration config : this.configs) {
        {
          String _xifexpression = null;
          boolean _exists = this.exists(config.getFilename());
          if (_exists) {
            _xifexpression = config.getFilename();
          } else {
            _xifexpression = this.DEFAULT_NAME;
          }
          final String filename = _xifexpression;
          Object _xifexpression_1 = null;
          boolean _exists_1 = this.exists(config.getFormat());
          if (_exists_1) {
            _xifexpression_1 = config.getFormat();
          } else {
            _xifexpression_1 = this.DEFAULT_FORMAT;
          }
          final Object format = ((Object)_xifexpression_1);
          final List<ElkNode> graphs = this.genElkNode(config);
          int fileNum = 0;
          for (int i = 0; (i < graphs.size()); i++) {
            {
              while (project.getFile((((filename + Integer.valueOf(fileNum)) + ".") + format)).exists()) {
                fileNum++;
              }
              final IFile f = project.getFile((((filename + Integer.valueOf(fileNum)) + ".") + format));
              this.serialize(graphs.get(i), f);
            }
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public List<ElkNode> loadGraph() {
    final ArrayList<ElkNode> graphs = new ArrayList<ElkNode>();
    for (final Configuration config : this.configs) {
      graphs.addAll(this.genElkNode(config));
    }
    return graphs;
  }
  
  public List<ElkNode> genElkNode(final Configuration config) {
    final ArrayList<ElkNode> generated = new ArrayList<ElkNode>();
    final int samples = config.getSamples();
    int _xifexpression = (int) 0;
    boolean _exists = this.exists(Integer.valueOf(samples));
    if (_exists) {
      _xifexpression = samples;
    } else {
      _xifexpression = this.DEFAULT_NUM_GRAPHS;
    }
    final int nGraphs = _xifexpression;
    final Random r = this.random(config);
    ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, nGraphs, true);
    for (final Integer i : _doubleDotLessThan) {
      {
        final GeneratorOptions options = this.parseGenOptions(config, r);
        final ElkNode n = new RandomGraphGenerator(r).generate(options);
        generated.add(n);
      }
    }
    return generated;
  }
  
  public GeneratorOptions parseGenOptions(final Configuration config, final Random r) {
    GeneratorOptions _xblockexpression = null;
    {
      final GeneratorOptions genOpt = new GeneratorOptions();
      boolean _exists = this.exists(GeneratorOptionsUtil.getPreferenceStore());
      if (_exists) {
        GeneratorOptionsUtil.loadPreferences(genOpt);
      }
      this.setGraphType(config, genOpt);
      this.nodes(config.getNodes(), r, genOpt);
      this.edges(config, genOpt, r);
      this.hierarchy(config, genOpt, r);
      this.setQuantities(genOpt, config.getFraction(), GeneratorOptions.PARTITION_FRAC);
      this.<Integer>setIfExists(genOpt, config.getMaxWidth(), GeneratorOptions.MAX_WIDTH);
      this.<Integer>setIfExists(genOpt, config.getMaxDegree(), GeneratorOptions.MAX_DEGREE);
      _xblockexpression = genOpt;
    }
    return _xblockexpression;
  }
  
  public MapPropertyHolder hierarchy(final Configuration configuration, final GeneratorOptions options, final Random random) {
    MapPropertyHolder _xblockexpression = null;
    {
      final Hierarchy hierarchy = configuration.getHierarchy();
      MapPropertyHolder _xifexpression = null;
      boolean _exists = this.exists(hierarchy);
      if (_exists) {
        MapPropertyHolder _xblockexpression_1 = null;
        {
          final DoubleQuantity hierarch = hierarchy.getNumHierarchNodes();
          this.setQuantities(options, hierarchy.getCrossHierarchRel(), GeneratorOptions.EXACT_RELATIVE_HIER);
          options.<Boolean>setProperty(GeneratorOptions.SMALL_HIERARCHY, Boolean.valueOf(this.exists(hierarch)));
          this.setQuantities(options, hierarchy.getLevels(), GeneratorOptions.MAX_HIERARCHY_LEVEL);
          this.setQuantities(options, hierarch, GeneratorOptions.NUMBER_HIERARCHICAL_NODES);
          _xblockexpression_1 = this.setQuantities(options, hierarchy.getEdges(), GeneratorOptions.CROSS_HIER);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private Random random(final Configuration g) {
    final Integer seed = g.getSeed();
    if ((seed == null)) {
      return new Random();
    } else {
      return new Random((seed).intValue());
    }
  }
  
  private MapPropertyHolder edges(final Configuration config, final GeneratorOptions genOpt, final Random r) {
    MapPropertyHolder _xblockexpression = null;
    {
      final Edges edges = config.getEdges();
      MapPropertyHolder _xifexpression = null;
      boolean _exists = this.exists(edges);
      if (_exists) {
        MapPropertyHolder _xblockexpression_1 = null;
        {
          boolean _isTotal = edges.isTotal();
          if (_isTotal) {
            genOpt.<GeneratorOptions.EdgeDetermination>setProperty(GeneratorOptions.EDGE_DETERMINATION, GeneratorOptions.EdgeDetermination.ABSOLUTE);
            this.setQuantities(genOpt, edges.getNEdges(), GeneratorOptions.EDGES_ABSOLUTE);
          } else {
            boolean _isDensity = edges.isDensity();
            if (_isDensity) {
              this.setQuantities(genOpt, edges.getNEdges(), GeneratorOptions.DENSITY);
              genOpt.<GeneratorOptions.EdgeDetermination>setProperty(GeneratorOptions.EDGE_DETERMINATION, GeneratorOptions.EdgeDetermination.DENSITY);
            } else {
              boolean _isRelative = edges.isRelative();
              if (_isRelative) {
                genOpt.<GeneratorOptions.EdgeDetermination>setProperty(GeneratorOptions.EDGE_DETERMINATION, GeneratorOptions.EdgeDetermination.RELATIVE);
                this.setQuantities(genOpt, edges.getNEdges(), GeneratorOptions.RELATIVE_EDGES);
              } else {
                genOpt.<GeneratorOptions.EdgeDetermination>setProperty(GeneratorOptions.EDGE_DETERMINATION, GeneratorOptions.EdgeDetermination.OUTGOING);
                this.setQuantities(genOpt, edges.getNEdges(), GeneratorOptions.OUTGOING_EDGES);
              }
            }
          }
          genOpt.<Boolean>setProperty(GeneratorOptions.EDGE_LABELS, Boolean.valueOf(edges.isLabels()));
          _xblockexpression_1 = genOpt.<Boolean>setProperty(GeneratorOptions.SELF_LOOPS, Boolean.valueOf(edges.isSelfLoops()));
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private MapPropertyHolder nodes(final Nodes nodes, final Random r, final GeneratorOptions genOpt) {
    MapPropertyHolder _xifexpression = null;
    boolean _exists = this.exists(nodes);
    if (_exists) {
      MapPropertyHolder _xblockexpression = null;
      {
        this.setQuantities(genOpt, nodes.getNNodes(), GeneratorOptions.NUMBER_OF_NODES);
        genOpt.<Boolean>setProperty(GeneratorOptions.CREATE_NODE_LABELS, Boolean.valueOf(nodes.isLabels()));
        this.ports(nodes, genOpt, r);
        this.size(nodes, r, genOpt);
        boolean _isRemoveIsolated = nodes.isRemoveIsolated();
        boolean _not = (!_isRemoveIsolated);
        _xblockexpression = genOpt.<Boolean>setProperty(GeneratorOptions.ISOLATED_NODES, Boolean.valueOf(_not));
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  private MapPropertyHolder size(final Nodes nodes, final Random r, final GeneratorOptions genOpt) {
    MapPropertyHolder _xblockexpression = null;
    {
      final Size size = nodes.getSize();
      MapPropertyHolder _xifexpression = null;
      boolean _exists = this.exists(size);
      if (_exists) {
        MapPropertyHolder _xblockexpression_1 = null;
        {
          this.setQuantities(genOpt, size.getWidth(), GeneratorOptions.NODE_WIDTH);
          _xblockexpression_1 = this.setQuantities(genOpt, size.getHeight(), GeneratorOptions.NODE_HEIGHT);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private MapPropertyHolder ports(final Nodes nodes, final GeneratorOptions genOpt, final Random r) {
    MapPropertyHolder _xblockexpression = null;
    {
      final Ports ports = nodes.getPorts();
      MapPropertyHolder _xifexpression = null;
      boolean _exists = this.exists(ports);
      if (_exists) {
        MapPropertyHolder _xblockexpression_1 = null;
        {
          genOpt.<Boolean>setProperty(GeneratorOptions.ENABLE_PORTS, Boolean.valueOf(true));
          genOpt.<Boolean>setProperty(GeneratorOptions.CREATE_PORT_LABELS, Boolean.valueOf(ports.isLabels()));
          genOpt.<PortConstraints>setProperty(GeneratorOptions.PORT_CONSTRAINTS, this.getConstraint(ports.getConstraint()));
          this.setQuantities(genOpt, ports.getReUse(), GeneratorOptions.USE_EXISTING_PORTS_CHANCE);
          final EList<Flow> flows = ports.getFlow();
          boolean _exists_1 = this.exists(flows);
          if (_exists_1) {
            for (final Flow f : flows) {
              this.setFlowSide(genOpt, f.getFlowType(), f.getSide(), f.getAmount(), r);
            }
          }
          final Size size = ports.getSize();
          genOpt.<Boolean>setProperty(GeneratorOptions.SET_PORT_SIZE, Boolean.valueOf(true));
          MapPropertyHolder _xifexpression_1 = null;
          boolean _exists_2 = this.exists(size);
          if (_exists_2) {
            MapPropertyHolder _xblockexpression_2 = null;
            {
              this.setQuantities(genOpt, size.getHeight(), GeneratorOptions.PORT_HEIGHT);
              _xblockexpression_2 = this.setQuantities(genOpt, size.getWidth(), GeneratorOptions.PORT_WIDTH);
            }
            _xifexpression_1 = _xblockexpression_2;
          }
          _xblockexpression_1 = _xifexpression_1;
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private MapPropertyHolder setQuantities(final GeneratorOptions genOpt, final DoubleQuantity quant, final IProperty<GeneratorOptions.RandVal> randomValue) {
    MapPropertyHolder _xifexpression = null;
    boolean _exists = this.exists(quant);
    if (_exists) {
      _xifexpression = genOpt.<GeneratorOptions.RandVal>setProperty(randomValue, this.toRandVal(quant));
    }
    return _xifexpression;
  }
  
  public GeneratorOptions.RandVal toRandVal(final DoubleQuantity quant) {
    GeneratorOptions.RandVal _xifexpression = null;
    boolean _isMinMax = quant.isMinMax();
    if (_isMinMax) {
      _xifexpression = GeneratorOptions.RandVal.minMax((quant.getMin()).doubleValue(), (quant.getMax()).doubleValue());
    } else {
      GeneratorOptions.RandVal _xifexpression_1 = null;
      boolean _isGaussian = quant.isGaussian();
      if (_isGaussian) {
        _xifexpression_1 = GeneratorOptions.RandVal.gaussian((quant.getMean()).doubleValue(), (quant.getStddv()).doubleValue());
      } else {
        _xifexpression_1 = GeneratorOptions.RandVal.exact((quant.getQuant()).doubleValue());
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  private MapPropertyHolder setFlowSide(final GeneratorOptions options, final FlowType type, final Side side, final DoubleQuantity quant, final Random r) {
    MapPropertyHolder _xblockexpression = null;
    {
      final int amount = this.toRandVal(quant).intVal(r);
      MapPropertyHolder _switchResult = null;
      if (type != null) {
        switch (type) {
          case INCOMING:
            MapPropertyHolder _switchResult_1 = null;
            if (side != null) {
              switch (side) {
                case EAST:
                  _switchResult_1 = options.<Integer>setProperty(GeneratorOptions.INCOMING_EAST_SIDE, Integer.valueOf(amount));
                  break;
                case NORTH:
                  _switchResult_1 = options.<Integer>setProperty(GeneratorOptions.INCOMING_NORTH_SIDE, Integer.valueOf(amount));
                  break;
                case SOUTH:
                  _switchResult_1 = options.<Integer>setProperty(GeneratorOptions.INCOMING_SOUTH_SIDE, Integer.valueOf(amount));
                  break;
                case WEST:
                  _switchResult_1 = options.<Integer>setProperty(GeneratorOptions.INCOMING_WEST_SIDE, Integer.valueOf(amount));
                  break;
                default:
                  break;
              }
            }
            _switchResult = _switchResult_1;
            break;
          case OUTGOING:
            MapPropertyHolder _switchResult_2 = null;
            if (side != null) {
              switch (side) {
                case EAST:
                  _switchResult_2 = options.<Integer>setProperty(GeneratorOptions.OUTGOING_EAST_SIDE, Integer.valueOf(amount));
                  break;
                case NORTH:
                  _switchResult_2 = options.<Integer>setProperty(GeneratorOptions.OUTGOING_NORTH_SIDE, Integer.valueOf(amount));
                  break;
                case SOUTH:
                  _switchResult_2 = options.<Integer>setProperty(GeneratorOptions.OUTGOING_SOUTH_SIDE, Integer.valueOf(amount));
                  break;
                case WEST:
                  _switchResult_2 = options.<Integer>setProperty(GeneratorOptions.OUTGOING_WEST_SIDE, Integer.valueOf(amount));
                  break;
                default:
                  break;
              }
            }
            _switchResult = _switchResult_2;
            break;
          default:
            break;
        }
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  private PortConstraints getConstraint(final ConstraintType constraint) {
    if (constraint != null) {
      switch (constraint) {
        case FREE:
          return PortConstraints.FREE;
        case ORDER:
          return PortConstraints.FIXED_ORDER;
        case POSITION:
          return PortConstraints.FIXED_POS;
        case SIDE:
          return PortConstraints.FIXED_SIDE;
        case RATIO:
          return PortConstraints.FIXED_RATIO;
        default:
          break;
      }
    }
    return PortConstraints.UNDEFINED;
  }
  
  private void setGraphType(final Configuration configuration, final GeneratorOptions options) {
    Form _form = configuration.getForm();
    if (_form != null) {
      switch (_form) {
        case ACYCLIC:
          options.<GeneratorOptions.GraphType>setProperty(GeneratorOptions.GRAPH_TYPE, GeneratorOptions.GraphType.ACYCLIC_NO_TRANSITIVE_EDGES);
          break;
        case BICONNECTED:
          options.<GeneratorOptions.GraphType>setProperty(GeneratorOptions.GRAPH_TYPE, GeneratorOptions.GraphType.BICONNECTED);
          break;
        case BIPARTITE:
          options.<GeneratorOptions.GraphType>setProperty(GeneratorOptions.GRAPH_TYPE, GeneratorOptions.GraphType.BIPARTITE);
          break;
        case CUSTOM:
          options.<GeneratorOptions.GraphType>setProperty(GeneratorOptions.GRAPH_TYPE, GeneratorOptions.GraphType.CUSTOM);
          break;
        case TREES:
          options.<GeneratorOptions.GraphType>setProperty(GeneratorOptions.GRAPH_TYPE, GeneratorOptions.GraphType.TREE);
          break;
        case TRICONNECTED:
          options.<GeneratorOptions.GraphType>setProperty(GeneratorOptions.GRAPH_TYPE, GeneratorOptions.GraphType.TRICONNECTED);
          break;
        default:
          break;
      }
    }
  }
  
  private void serialize(final ElkNode graph, final IFile file) throws IOException, CoreException {
    final ResourceSetImpl resourceSet = new ResourceSetImpl();
    final Resource resource = resourceSet.createResource(URI.createURI(file.getLocationURI().toString()));
    resource.getContents().add(graph);
    resource.save(Collections.EMPTY_MAP);
    file.refreshLocal(1, null);
  }
  
  private boolean exists(final Object o) {
    return (o != null);
  }
  
  private <T extends Object> boolean setIfExists(final GeneratorOptions genOpt, final T value, final IProperty<T> property) {
    boolean _exists = this.exists(value);
    if (_exists) {
      genOpt.<T>setProperty(property, value);
      return true;
    }
    return false;
  }
}
