!--------------------------------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations                              !
!   Copyright 2000-2020 CP2K developers group <https://cp2k.org>                                   !
!                                                                                                  !
!   SPDX-License-Identifier: GPL-2.0-or-later                                                      !
!--------------------------------------------------------------------------------------------------!

! **************************************************************************************************
!> \brief Routines to calculate the minimax coefficients in order to
!>        approximate 1/x as a sum over exponential functions
!>        1/x ~ SUM_{i}^{K} w_i EXP(-a_i * x) for x belonging to [1:Rc].
!>        This module contains coefficients for minimax approximations with 26, 28, 30, 32, 34 points
!>        that give well-converged GW results
!> \par History
!>      03.2020 created [Jan Wilhelm]
! **************************************************************************************************
MODULE minimax_exp_gw
   USE kinds,                           ONLY: dp
#include "../base/base_uses.f90"

   IMPLICIT NONE

   PRIVATE

   CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'minimax_exp_gw'

   PUBLIC :: get_exp_minimax_coeff_gw

CONTAINS

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param E_range ...
!> \param aw ...
! **************************************************************************************************
   SUBROUTINE get_exp_minimax_coeff_gw(k, E_range, aw)
      INTEGER, INTENT(IN)                                :: k
      REAL(KIND=dp), INTENT(IN)                          :: E_range
      REAL(KIND=dp), DIMENSION(2*k), INTENT(OUT)         :: aw

      CHARACTER(LEN=*), PARAMETER :: routineN = 'get_exp_minimax_coeff_gw'

      INTEGER                                            :: handle
      REAL(KIND=dp)                                      :: E_ratio

      CALL timeset(routineN, handle)

      SELECT CASE (k)
      CASE (26)
         CALL get_coeff_26(k, E_range, aw, E_ratio)
      CASE (28)
         CALL get_coeff_28(k, E_range, aw, E_ratio)
      CASE (30)
         CALL get_coeff_30(k, E_range, aw, E_ratio)
      CASE (32)
         CALL get_coeff_32(k, E_range, aw, E_ratio)
      CASE (34)
         CALL get_coeff_34(k, E_range, aw, E_ratio)
      CASE DEFAULT
         CPABORT("The grid size you choose is not available.")
      END SELECT

      CALL rescale_grid(aw, E_ratio)

      CALL timestop(handle)

   END SUBROUTINE get_exp_minimax_coeff_gw

! **************************************************************************************************
!> \brief ...
!> \param aw ...
!> \param E_ratio ...
! **************************************************************************************************
   PURE SUBROUTINE rescale_grid(aw, E_ratio)
      REAL(KIND=dp), DIMENSION(:), INTENT(INOUT)         :: aw
      REAL(KIND=dp), INTENT(INOUT)                       :: E_ratio

      IF (E_ratio > 1.5_dp) THEN
         E_ratio = E_ratio/1.5_dp
      ELSE
         E_ratio = 1.0_dp
      END IF
      aw(:) = aw(:)*E_ratio
   END SUBROUTINE

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param E_range ...
!> \param aw ...
!> \param E_ratio ...
! **************************************************************************************************
   PURE SUBROUTINE get_coeff_26(k, E_range, aw, E_ratio)
      INTEGER, INTENT(IN)                                :: k
      REAL(KIND=dp), INTENT(IN)                          :: E_range
      REAL(KIND=dp), DIMENSION(2*k), INTENT(OUT)         :: aw
      REAL(KIND=dp), INTENT(OUT)                         :: E_ratio

      E_ratio = 1.0_dp
      IF (E_range < 799.0_dp) THEN
         E_ratio = 799.0_dp/E_range
         aw(:) = (/ &
                 0.00037867004555797054_dp, &
                 0.002019373143542281_dp, &
                 0.005074452374668418_dp, &
                 0.009741339051894624_dp, &
                 0.016351249099763064_dp, &
                 0.025426574118427834_dp, &
                 0.03775841037254848_dp, &
                 0.05450081932003549_dp, &
                 0.0772810957375601_dp, &
                 0.1083363446608876_dp, &
                 0.15069589349866513_dp, &
                 0.20843076717336742_dp, &
                 0.2869911650766578_dp, &
                 0.3936564974164155_dp, &
                 0.5381309695975518_dp, &
                 0.7333307052011886_dp, &
                 0.996427988952117_dp, &
                 1.3502505778649674_dp, &
                 1.8251937089224979_dp, &
                 2.4619239536701665_dp, &
                 3.3154239654767634_dp, &
                 4.461579856601731_dp, &
                 6.009262672635442_dp, &
                 8.12630074937428_dp, &
                 11.10929348730981_dp, &
                 15.660501844751538_dp, &
                 0.0009745069197489692_dp, &
                 0.002323920525926464_dp, &
                 0.003818499918990434_dp, &
                 0.005569525222864092_dp, &
                 0.007736056032946723_dp, &
                 0.01054486610207425_dp, &
                 0.014309337959120037_dp, &
                 0.01944447015531593_dp, &
                 0.026485944451580964_dp, &
                 0.03612533116765179_dp, &
                 0.04926670257709911_dp, &
                 0.06710398765236568_dp, &
                 0.09122022393749045_dp, &
                 0.12371484118454482_dp, &
                 0.16736960998240946_dp, &
                 0.2258689406847251_dp, &
                 0.30409900038431836_dp, &
                 0.40856816694415354_dp, &
                 0.548030961075663_dp, &
                 0.7344894715293502_dp, &
                 0.9849728438705977_dp, &
                 1.3251021306723128_dp, &
                 1.7972852200464182_dp, &
                 2.4831111234607426_dp, &
                 3.582848552118199_dp, &
                 5.8876320817174905_dp/)
      ELSE IF (E_range < 995.0_dp) THEN
         aw(:) = (/ &
                 0.00031416617596844365_dp, &
                 0.0016768070133771967_dp, &
                 0.004220387402167522_dp, &
                 0.008122019548232636_dp, &
                 0.013681535061623657_dp, &
                 0.021375962641606915_dp, &
                 0.03193242611639194_dp, &
                 0.04641604327774391_dp, &
                 0.0663338368572065_dp, &
                 0.09376694799993701_dp, &
                 0.13155072067060036_dp, &
                 0.18352265415559924_dp, &
                 0.2548586347835214_dp, &
                 0.3525227140309385_dp, &
                 0.4858648853075939_dp, &
                 0.667414665262307_dp, &
                 0.9139380450657599_dp, &
                 1.2478576632771592_dp, &
                 1.699195061182984_dp, &
                 2.3083135493129188_dp, &
                 3.13000622971369_dp, &
                 4.240119346476375_dp, &
                 5.747638276053859_dp, &
                 7.820581475731412_dp, &
                 10.755506760817696_dp, &
                 15.2531099085639_dp, &
                 0.0008086633399478686_dp, &
                 0.001931716707111731_dp, &
                 0.0031843549489835176_dp, &
                 0.0046679761695975755_dp, &
                 0.006529332077378777_dp, &
                 0.008979243990088065_dp, &
                 0.012309502604654843_dp, &
                 0.016906748893981475_dp, &
                 0.023272740862285972_dp, &
                 0.03206195264016037_dp, &
                 0.04413963056111703_dp, &
                 0.060659468387992695_dp, &
                 0.08316331669118938_dp, &
                 0.11371010407957528_dp, &
                 0.15504510373245767_dp, &
                 0.21082547573489732_dp, &
                 0.28592663287586667_dp, &
                 0.38687165073350616_dp, &
                 0.5224647206101295_dp, &
                 0.7047999539903935_dp, &
                 0.951040871895601_dp, &
                 1.286968657510818_dp, &
                 1.7551267580277088_dp, &
                 2.437087428172129_dp, &
                 3.5325565753903243_dp, &
                 5.829332987897013_dp/)
      ELSE IF (E_range < 1293.0_dp) THEN
         aw(:) = (/ &
                 0.000251216262630506_dp, &
                 0.0013422657589387847_dp, &
                 0.0033852831378795366_dp, &
                 0.006535679555855439_dp, &
                 0.01105946790435755_dp, &
                 0.017384205344456737_dp, &
                 0.02616631610050564_dp, &
                 0.03837171615696803_dp, &
                 0.05537294469791371_dp, &
                 0.07907682374899021_dp, &
                 0.11210158563747868_dp, &
                 0.15802208108125124_dp, &
                 0.22170328658555793_dp, &
                 0.3097484547351596_dp, &
                 0.43109811316492785_dp, &
                 0.5978298541062674_dp, &
                 0.826228906983008_dp, &
                 1.1382318360717252_dp, &
                 1.5634041077266412_dp, &
                 2.1417299191833186_dp, &
                 2.9277541939220253_dp, &
                 3.9972537939524453_dp, &
                 5.459336924183542_dp, &
                 7.482255655971381_dp, &
                 10.362568049094785_dp, &
                 14.79927594838438_dp, &
                 0.0006467895131307655_dp, &
                 0.0015483789662149572_dp, &
                 0.002562990408244027_dp, &
                 0.00378128864955077_dp, &
                 0.005336413195947385_dp, &
                 0.0074211270754708444_dp, &
                 0.010302498712647285_dp, &
                 0.014334977955021215_dp, &
                 0.01998237358756267_dp, &
                 0.027857715884000284_dp, &
                 0.038782119259036095_dp, &
                 0.05386231670249176_dp, &
                 0.07459076970770032_dp, &
                 0.10297648694866778_dp, &
                 0.1417181535853474_dp, &
                 0.19443583384059848_dp, &
                 0.26598602063677856_dp, &
                 0.36290208001606944_dp, &
                 0.4940399248668749_dp, &
                 0.671597172994337_dp, &
                 0.9128949281590978_dp, &
                 1.2439090969479831_dp, &
                 1.7073582868867894_dp, &
                 2.3848234540233704_dp, &
                 3.4754029453297375_dp, &
                 5.763153512206444_dp/)
      ELSE IF (E_range < 1738.0_dp) THEN
         aw(:) = (/ &
                 0.0001950376405774879_dp, &
                 0.0010434591619187578_dp, &
                 0.002638226324824606_dp, &
                 0.005113292156951265_dp, &
                 0.00870086598207974_dp, &
                 0.013778342596377403_dp, &
                 0.020929407226154117_dp, &
                 0.031017048599119232_dp, &
                 0.04527407877688829_dp, &
                 0.06542623021788685_dp, &
                 0.09386550782858209_dp, &
                 0.1338911807639374_dp, &
                 0.19003887760085253_dp, &
                 0.26852546690556994_dp, &
                 0.3778478282417773_dp, &
                 0.5295878007623821_dp, &
                 0.739496104873784_dp, &
                 1.0289606432203449_dp, &
                 1.427022553455354_dp, &
                 1.9732190338166054_dp, &
                 2.721788596346289_dp, &
                 3.7484019634398744_dp, &
                 5.162286381612833_dp, &
                 7.131961441982249_dp, &
                 9.954041146838616_dp, &
                 14.32582187195048_dp, &
                 0.0005022997712497699_dp, &
                 0.0012056339745159519_dp, &
                 0.0020057022650528562_dp, &
                 0.0029823455742529125_dp, &
                 0.004254672143777001_dp, &
                 0.0059963576674781395_dp, &
                 0.008448134528374854_dp, &
                 0.011930608474038368_dp, &
                 0.016868224194481772_dp, &
                 0.023830776674747995_dp, &
                 0.03359223660591877_dp, &
                 0.04720772204797841_dp, &
                 0.06611395904656978_dp, &
                 0.09226211932358148_dp, &
                 0.12829508657243463_dp, &
                 0.17778587596291995_dp, &
                 0.24556239855995798_dp, &
                 0.3381606409098896_dp, &
                 0.4644850568708377_dp, &
                 0.6368412465900805_dp, &
                 0.8727231844202037_dp, &
                 1.198329166055601_dp, &
                 1.6565902038491545_dp, &
                 2.329130996306286_dp, &
                 3.414441737018883_dp, &
                 5.692646250049292_dp/)
      ELSE IF (E_range < 2238.0_dp) THEN
         aw(:) = (/ &
                 0.0001570075292014777_dp, &
                 0.0008410019818081912_dp, &
                 0.002131222620982682_dp, &
                 0.004145569234358363_dp, &
                 0.007090677510918224_dp, &
                 0.011305441203226727_dp, &
                 0.01731697219552165_dp, &
                 0.02590771238094241_dp, &
                 0.03820108916428952_dp, &
                 0.05578097500511556_dp, &
                 0.08086136666211902_dp, &
                 0.11652297811189155_dp, &
                 0.16703767866605654_dp, &
                 0.23830959694579532_dp, &
                 0.33847251734047956_dp, &
                 0.47869779524697986_dp, &
                 0.6742880172162485_dp, &
                 0.9461646148413084_dp, &
                 1.3229154365464835_dp, &
                 1.8436824667954244_dp, &
                 2.5624225632785795_dp, &
                 3.5546935048363073_dp, &
                 4.929807210724086_dp, &
                 6.856512184015644_dp, &
                 9.631510529999963_dp, &
                 13.950799086937636_dp, &
                 0.0004044673162152666_dp, &
                 0.0009731515874605558_dp, &
                 0.0016264516115000137_dp, &
                 0.002435950210582391_dp, &
                 0.0035098078378292706_dp, &
                 0.005006471432500407_dp, &
                 0.0071456103103081205_dp, &
                 0.010221155980366915_dp, &
                 0.01462678603467543_dp, &
                 0.02089809034158199_dp, &
                 0.02977080559865047_dp, &
                 0.042257136734592034_dp, &
                 0.059746455952240894_dp, &
                 0.08413972088086426_dp, &
                 0.11803003373030124_dp, &
                 0.1649465354635964_dp, &
                 0.22968730801733273_dp, &
                 0.31878354856943114_dp, &
                 0.44117315496449167_dp, &
                 0.6092463295611383_dp, &
                 0.8406397450171417_dp, &
                 1.161742484167732_dp, &
                 1.6156774111835495_dp, &
                 2.2841323787971173_dp, &
                 3.36513821401701_dp, &
                 5.635684641787822_dp/)
      ELSE IF (E_range < 3009.0_dp) THEN
         aw(:) = (/ &
                 0.00012173433282636892_dp, &
                 0.000653044843078349_dp, &
                 0.0016597094198001826_dp, &
                 0.0032432122902587715_dp, &
                 0.005583731099072486_dp, &
                 0.00897974690437299_dp, &
                 0.01389829106398585_dp, &
                 0.021035914001762628_dp, &
                 0.03139940878203811_dp, &
                 0.046421111007702805_dp, &
                 0.06812376578216424_dp, &
                 0.09935124887637425_dp, &
                 0.1440867725430818_dp, &
                 0.20788864079466032_dp, &
                 0.29848482929563186_dp, &
                 0.42658280937440796_dp, &
                 0.6069727081348725_dp, &
                 0.8600354677167223_dp, &
                 1.213825493528422_dp, &
                 1.7070100456167858_dp, &
                 2.393195963336712_dp, &
                 3.3477843441213753_dp, &
                 4.680165709088339_dp, &
                 6.5593534066075065_dp, &
                 9.282192058087176_dp, &
                 13.54332387201712_dp, &
                 0.00031370730222591084_dp, &
                 0.0007570695990484558_dp, &
                 0.0012727251500342805_dp, &
                 0.0019236429401252328_dp, &
                 0.0028062999069873792_dp, &
                 0.004062578033501044_dp, &
                 0.00588929507373728_dp, &
                 0.00855186255429553_dp, &
                 0.012411138722817702_dp, &
                 0.017965594962442332_dp, &
                 0.025908461038187995_dp, &
                 0.03720320806718786_dp, &
                 0.05318444551258281_dp, &
                 0.0756939715032581_dp, &
                 0.10726485539108421_dp, &
                 0.1513713746489183_dp, &
                 0.21277116226058032_dp, &
                 0.297982246389678_dp, &
                 0.41597262011480124_dp, &
                 0.5792223633874778_dp, &
                 0.8055284041136077_dp, &
                 1.1215023110944804_dp, &
                 1.5705011278690364_dp, &
                 2.234313406989394_dp, &
                 3.310497107984999_dp, &
                 5.57261913216792_dp/)
      ELSE IF (E_range < 4377.0_dp) THEN
         aw(:) = (/ &
                 8.814790587980884e-05_dp, &
                 0.0004738555483452858_dp, &
                 0.001209162224838196_dp, &
                 0.0023779784718090265_dp, &
                 0.004131730232456246_dp, &
                 0.006724209366814066_dp, &
                 0.010555208090639503_dp, &
                 0.01622487926307075_dp, &
                 0.02460909648355756_dp, &
                 0.036969426022926015_dp, &
                 0.05511124918766332_dp, &
                 0.08160627715945637_dp, &
                 0.12010201314867164_dp, &
                 0.17574961207193285_dp, &
                 0.2557933011845474_dp, &
                 0.37038042948632577_dp, &
                 0.5336738618256365_dp, &
                 0.7653829566132899_dp, &
                 1.092887558492255_dp, &
                 1.5542408605150586_dp, &
                 2.2025803308040364_dp, &
                 3.113074354623969_dp, &
                 4.395181735531908_dp, &
                 6.218242520601409_dp, &
                 8.87933263258288_dp, &
                 13.071612343557268_dp, &
                 0.0002272629071152242_dp, &
                 0.0005507552663901378_dp, &
                 0.000933441836691016_dp, &
                 0.001428839311231769_dp, &
                 0.002120237386026644_dp, &
                 0.0031304749854564546_dp, &
                 0.0046303259129662575_dp, &
                 0.006853192947474817_dp, &
                 0.01012290436688526_dp, &
                 0.014894995829624644_dp, &
                 0.021812185159477044_dp, &
                 0.03177875816649789_dp, &
                 0.046061554071504564_dp, &
                 0.06642770441212868_dp, &
                 0.09533262717257988_dp, &
                 0.1361769876776064_dp, &
                 0.19366000664329205_dp, &
                 0.27427258699605633_dp, &
                 0.38700765699040135_dp, &
                 0.5444449877078907_dp, &
                 0.7645728576669742_dp, &
                 1.0742808530249828_dp, &
                 1.517233705376216_dp, &
                 2.175383868205045_dp, &
                 3.245782737215517_dp, &
                 5.498017132429984_dp/)
      ELSE IF (E_range < 6256.0_dp) THEN
         aw(:) = (/ &
                 6.476469600717215e-05_dp, &
                 0.00034891783555501116_dp, &
                 0.0008941569546994692_dp, &
                 0.001770490093242039_dp, &
                 0.003106199645499256_dp, &
                 0.0051184335340408015_dp, &
                 0.008151285145167414_dp, &
                 0.012724953439821175_dp, &
                 0.019606591631387022_dp, &
                 0.029915037575018_dp, &
                 0.04527205344500337_dp, &
                 0.06801655695473027_dp, &
                 0.10150518730808944_dp, &
                 0.1505317627030831_dp, &
                 0.22191039986921127_dp, &
                 0.32528375402693044_dp, &
                 0.4742415082162232_dp, &
                 0.6878698417118773_dp, &
                 0.9929114555738491_dp, &
                 1.426826349178114_dp, &
                 2.0422826887022167_dp, &
                 2.914194983288514_dp, &
                 4.152054985305785_dp, &
                 5.925500508772846_dp, &
                 8.531868435284434_dp, &
                 12.663118219514553_dp, &
                 0.00016705896835219716_dp, &
                 0.0004066434009563924_dp, &
                 0.0006951433499719869_dp, &
                 0.0010783855735630424_dp, &
                 0.0016285976932411544_dp, &
                 0.002452428674095891_dp, &
                 0.003698822692463904_dp, &
                 0.005574655476173014_dp, &
                 0.008372611367675033_dp, &
                 0.012511058447330109_dp, &
                 0.01858785984880576_dp, &
                 0.027453700962179424_dp, &
                 0.04031295449132496_dp, &
                 0.058862610090479196_dp, &
                 0.08548341120436852_dp, &
                 0.12350282964756092_dp, &
                 0.17755835622889385_dp, &
                 0.25410556179535176_dp, &
                 0.3621484874152968_dp, &
                 0.5143475290193675_dp, &
                 0.728860394738634_dp, &
                 1.0328353641593397_dp, &
                 1.470237966477004_dp, &
                 2.1232087233584265_dp, &
                 3.1884016755763915_dp, &
                 5.431943610280663_dp/)
      ELSE IF (E_range < 9034.0_dp) THEN
         aw(:) = (/ &
                 4.714267869850108e-05_dp, &
                 0.00025461261481751654_dp, &
                 0.0006556761719344113_dp, &
                 0.001308468824945838_dp, &
                 0.002321150197926946_dp, &
                 0.003878461874698709_dp, &
                 0.006274894154406174_dp, &
                 0.009959418801224819_dp, &
                 0.01560204953263909_dp, &
                 0.02419303582580116_dp, &
                 0.03718681238787033_dp, &
                 0.056707527706112246_dp, &
                 0.08584014778037409_dp, &
                 0.12904063424870618_dp, &
                 0.19271144671607915_dp, &
                 0.2860061442236756_dp, &
                 0.4219516424878554_dp, &
                 0.6190135338939492_dp, &
                 0.903289679923842_dp, &
                 1.3116275192031754_dp, &
                 1.8961959695325519_dp, &
                 2.7316203672239747_dp, &
                 3.927395419802043_dp, &
                 5.653448502577673_dp, &
                 8.207416056488858_dp, &
                 12.280210966173971_dp, &
                 0.00012167157119425256_dp, &
                 0.00029765224752192887_dp, &
                 0.0005138446741296063_dp, &
                 0.0008093181914490766_dp, &
                 0.0012462946084647903_dp, &
                 0.0019166924309280847_dp, &
                 0.002949916160920999_dp, &
                 0.004529176144148134_dp, &
                 0.006918766955709136_dp, &
                 0.010502242217013015_dp, &
                 0.01583456242948638_dp, &
                 0.02371430118668526_dp, &
                 0.03528415663667773_dp, &
                 0.05217068929971722_dp, &
                 0.07667811497799712_dp, &
                 0.11205676769912475_dp, &
                 0.16287592945291882_dp, &
                 0.23554669511627027_dp, &
                 0.3390729578857381_dp, &
                 0.4861840636834821_dp, &
                 0.6951983617349604_dp, &
                 0.9935218371108636_dp, &
                 1.4254333766864864_dp, &
                 2.073293734219426_dp, &
                 3.133425653975384_dp, &
                 5.368705541720747_dp/)
      ELSE IF (E_range < 15564.0_dp) THEN
         aw(:) = (/ &
                 2.9443353786715123e-05_dp, &
                 0.00015968865887835848_dp, &
                 0.0004146519996900374_dp, &
                 0.0008385653229066657_dp, &
                 0.0015154710444555315_dp, &
                 0.002590553782265674_dp, &
                 0.004297458120933096_dp, &
                 0.0069977275055487075_dp, &
                 0.011241365987483699_dp, &
                 0.017857778834567364_dp, &
                 0.02808897944065483_dp, &
                 0.043782322061949784_dp, &
                 0.06766742773244867_dp, &
                 0.1037518951041952_dp, &
                 0.1578839462681274_dp, &
                 0.2385489270378087_dp, &
                 0.3579930679384087_dp, &
                 0.5338068043298525_dp, &
                 0.791161589813274_dp, &
                 1.1660045885058554_dp, &
                 1.7097467191304874_dp, &
                 2.496543051760848_dp, &
                 3.635839789357852_dp, &
                 5.297957374408996_dp, &
                 7.7810148585358885_dp, &
                 11.774676503198675_dp, &
                 7.606243073747058e-05_dp, &
                 0.000187654111092642_dp, &
                 0.00032937906982910266_dp, &
                 0.0005320965908695892_dp, &
                 0.0008454741794288284_dp, &
                 0.0013429974722876687_dp, &
                 0.0021300795419892347_dp, &
                 0.003360638155325681_dp, &
                 0.005262693302642904_dp, &
                 0.008173980176265023_dp, &
                 0.0125917789247931_dp, &
                 0.01924336061080778_dp, &
                 0.029185562811316818_dp, &
                 0.043945018416338716_dp, &
                 0.0657148857554778_dp, &
                 0.09763017054646428_dp, &
                 0.1441532549628787_dp, &
                 0.2116173831358168_dp, &
                 0.3090075514896036_dp, &
                 0.4491309451933545_dp, &
                 0.6505180076674358_dp, &
                 0.9409371471889534_dp, &
                 1.3651313188527026_dp, &
                 2.0058258806612064_dp, &
                 3.0589785756980588_dp, &
                 5.283171742769873_dp/)
      ELSE IF (E_range < 19500.0_dp) THEN
         aw(:) = (/ &
                 2.4221939880648916e-05_dp, &
                 0.00013162518783252502_dp, &
                 0.0003431050538880391_dp, &
                 0.000698189132475488_dp, &
                 0.001272589233863707_dp, &
                 0.0021976433259721703_dp, &
                 0.0036856546266738404_dp, &
                 0.00606744928063478_dp, &
                 0.009850534125738348_dp, &
                 0.015806672884776662_dp, &
                 0.025100760660057786_dp, &
                 0.03947837634367904_dp, &
                 0.06153686837154947_dp, &
                 0.09511493193970912_dp, &
                 0.14584951983758487_dp, &
                 0.22196820786998617_dp, &
                 0.335412339591795_dp, &
                 0.5034260546260962_dp, &
                 0.7508088209241689_dp, &
                 1.1131398816244866_dp, &
                 1.6415127034352714_dp, &
                 2.409875693114849_dp, &
                 3.527638075049888_dp, &
                 5.165269756856301_dp, &
                 7.621098250903887_dp, &
                 11.584360794128232_dp, &
                 6.260081073117763e-05_dp, &
                 0.0001550481237336088_dp, &
                 0.00027425459381644776_dp, &
                 0.0004482075757040794_dp, &
                 0.0007220779147665958_dp, &
                 0.0011628086873198505_dp, &
                 0.0018674017508154028_dp, &
                 0.0029793398493427954_dp, &
                 0.004713349903984844_dp, &
                 0.00739000908391551_dp, &
                 0.011484659947886356_dp, &
                 0.017697084080236986_dp, &
                 0.0270505598375803_dp, &
                 0.04103207563980036_dp, &
                 0.06178995726179555_dp, &
                 0.09241161287658448_dp, &
                 0.13731382654384652_dp, &
                 0.20279428302236951_dp, &
                 0.2978245159544139_dp, &
                 0.43523615879690214_dp, &
                 0.6336386632046422_dp, &
                 0.9209431632237726_dp, &
                 1.3420830429471422_dp, &
                 1.9799449671624767_dp, &
                 3.0303736903126226_dp, &
                 5.250335959421828_dp/)
      ELSE IF (E_range < 22300.0_dp) THEN
         aw(:) = (/ &
                 2.1564828392874976e-05_dp, &
                 0.00011732900716735743_dp, &
                 0.00030658459101179467_dp, &
                 0.0006263105810174703_dp, &
                 0.0011476627771254888_dp, &
                 0.001994368166545555_dp, &
                 0.0033669949117497793_dp, &
                 0.005579455105811666_dp, &
                 0.009115754087433222_dp, &
                 0.014715595238257124_dp, &
                 0.023500724387693545_dp, &
                 0.037159447358704495_dp, &
                 0.058214268280899095_dp, &
                 0.09040780191303678_dp, &
                 0.1392561892112249_dp, &
                 0.21283882379303048_dp, &
                 0.32292082611394995_dp, &
                 0.48654501997143224_dp, &
                 0.728293233827829_dp, &
                 1.0835278065593024_dp, &
                 1.6031531143244464_dp, &
                 2.360992062122655_dp, &
                 3.466427749396279_dp, &
                 5.090015465029667_dp, &
                 7.53020860247837_dp, &
                 11.47601339726731_dp, &
                 5.574871333711635e-05_dp, &
                 0.00013841643320123116_dp, &
                 0.0002460246878681958_dp, &
                 0.00040498060578160683_dp, &
                 0.0006579585426571892_dp, &
                 0.0010682911223034433_dp, &
                 0.0017283542359863323_dp, &
                 0.002775834478515568_dp, &
                 0.0044179824445092115_dp, &
                 0.006965640555090498_dp, &
                 0.010881623599671406_dp, &
                 0.01684992291318925_dp, &
                 0.025874425224763156_dp, &
                 0.03941906165439105_dp, &
                 0.05960588810571241_dp, &
                 0.08949416080199679_dp, &
                 0.13347332726212882_dp, &
                 0.1978191873760846_dp, &
                 0.29149390319797547_dp, &
                 0.4273416663588288_dp, &
                 0.6240165631691874_dp, &
                 0.9095125059692878_dp, &
                 1.3288753869034111_dp, &
                 1.9650901753832044_dp, &
                 3.013944114046813_dp, &
                 5.231485480677545_dp/)
      ELSE IF (E_range < 24783.0_dp) THEN
         aw(:) = (/ &
                 1.968079412594506e-05_dp, &
                 0.00010718493641102105_dp, &
                 0.0002806353581711279_dp, &
                 0.0005751273705017238_dp, &
                 0.0010584284934163331_dp, &
                 0.001848589008502448_dp, &
                 0.003137423022955394_dp, &
                 0.005226209962089254_dp, &
                 0.008581348030605833_dp, &
                 0.013918427654363913_dp, &
                 0.022326614770472653_dp, &
                 0.03545081033521797_dp, &
                 0.0557565940780925_dp, &
                 0.0869132477657862_dp, &
                 0.1343443806186961_dp, &
                 0.20601549407747916_dp, &
                 0.31355578173538573_dp, &
                 0.4738522008187966_dp, &
                 0.7113173707719063_dp, &
                 1.061144148921342_dp, &
                 1.574088149869358_dp, &
                 2.323872276609698_dp, &
                 3.4198568581275923_dp, &
                 5.032662209058464_dp, &
                 7.460841068394957_dp, &
                 11.393228561803523_dp, &
                 5.0889409017427654e-05_dp, &
                 0.00012660472263169967_dp, &
                 0.0002259210980998603_dp, &
                 0.0003740655095418279_dp, &
                 0.0006118384940478349_dp, &
                 0.000999874816429794_dp, &
                 0.0016270976310772938_dp, &
                 0.0026268356436642305_dp, &
                 0.004200673380690335_dp, &
                 0.006652036004218306_dp, &
                 0.010434155159748769_dp, &
                 0.016218897955811638_dp, &
                 0.02499520138410832_dp, &
                 0.038209141684542064_dp, &
                 0.05796233141783732_dp, &
                 0.08729199004516089_dp, &
                 0.13056598592688773_dp, &
                 0.19404256395643912_dp, &
                 0.2866758412418721_dp, &
                 0.4213188646546868_dp, &
                 0.6166595702412992_dp, &
                 0.900755794469128_dp, &
                 1.3187413549017004_dp, &
                 1.9536794891939209_dp, &
                 3.00131680399958_dp, &
                 5.21699996325917_dp/)
      ELSE IF (E_range < 41198.0_dp) THEN
         aw(:) = (/ &
                 1.2674056285680165e-05_dp, &
                 6.938931077849002e-05_dp, &
                 0.0001836100520485683_dp, &
                 0.0003826779493233582_dp, &
                 0.0007202040757498056_dp, &
                 0.0012903838734080182_dp, &
                 0.002248236525495258_dp, &
                 0.00384173613615444_dp, &
                 0.006462366726733287_dp, &
                 0.010722120046078792_dp, &
                 0.017568831387303644_dp, &
                 0.028457372697558436_dp, &
                 0.04560190325370437_dp, &
                 0.07234504966297221_dp, &
                 0.11369479964375559_dp, &
                 0.1771007906182374_dp, &
                 0.2735712154870606_dp, &
                 0.4192744014302692_dp, &
                 0.637834936286395_dp, &
                 0.9636469742972286_dp, &
                 1.4467538592483378_dp, &
                 2.1603857309953622_dp, &
                 3.2137696680006567_dp, &
                 4.777814618678099_dp, &
                 7.151554564591586_dp, &
                 11.0231252332385_dp, &
                 3.2810000033385146e-05_dp, &
                 8.24959243779711e-05_dp, &
                 0.00015031831931530549_dp, &
                 0.0002565196061853303_dp, &
                 0.0004339110517853371_dp, &
                 0.0007317640998323022_dp, &
                 0.0012244457171152918_dp, &
                 0.0020265452915832964_dp, &
                 0.0033148331774103383_dp, &
                 0.005359851622721186_dp, &
                 0.008571929299584299_dp, &
                 0.013568145467107103_dp, &
                 0.021269239209990003_dp, &
                 0.033039016114448296_dp, &
                 0.05088380967787895_dp, &
                 0.07773666161595308_dp, &
                 0.11786137108586371_dp, &
                 0.177428416908905_dp, &
                 0.26534598308886587_dp, &
                 0.3944982542877601_dp, &
                 0.583721306443492_dp, &
                 0.8613658745137791_dp, &
                 1.2729807136647084_dp, &
                 1.9020150846924464_dp, &
                 2.9440730604569967_dp, &
                 5.151372469471086_dp/)
      ELSE IF (E_range < 94407.0_dp) THEN
         aw(:) = (/ &
                 6.186479912580181e-06_dp, &
                 3.4233291323971405e-05_dp, &
                 9.256119767321995e-05_dp, &
                 0.00019951435503347473_dp, &
                 0.000391782808208518_dp, &
                 0.0007348447007241284_dp, &
                 0.0013393430685668998_dp, &
                 0.0023881407715703996_dp, &
                 0.004179359736833124_dp, &
                 0.007193105406189777_dp, &
                 0.012193603475754204_dp, &
                 0.020384081500457202_dp, &
                 0.03363963393153156_dp, &
                 0.054854535910931194_dp, &
                 0.0884563818462964_dp, &
                 0.1411619416118249_dp, &
                 0.22308161106495417_dp, &
                 0.3493255541517698_dp, &
                 0.5423345187791336_dp, &
                 0.835274429372801_dp, &
                 1.277059229828611_dp, &
                 1.9400953490632433_dp, &
                 2.933324383841704_dp, &
                 4.428041894721328_dp, &
                 6.724054874045832_dp, &
                 10.508722000292966_dp, &
                 1.605270034684411e-05_dp, &
                 4.1236777606757607e-05_dp, &
                 7.834646084720584e-05_dp, &
                 0.00014151002746774427_dp, &
                 0.0002536786392750537_dp, &
                 0.00045044627173442046_dp, &
                 0.0007883280940822237_dp, &
                 0.0013578695679215443_dp, &
                 0.0023029240253660617_dp, &
                 0.0038494157006882636_dp, &
                 0.006348322282988488_dp, &
                 0.010339483088961496_dp, &
                 0.016645606665935727_dp, &
                 0.026509781185170438_dp, &
                 0.04179532069895724_dp, &
                 0.06527461060266015_dp, &
                 0.10104501326885583_dp, &
                 0.15512766861607885_dp, &
                 0.23633651295613511_dp, &
                 0.3575717157671407_dp, &
                 0.5378622500972828_dp, &
                 0.8059829678217727_dp, &
                 1.2081212436199635_dp, &
                 1.8283703742955213_dp, &
                 2.862256950464903_dp, &
                 5.057682060420358_dp/)
      ELSE IF (E_range < 189080.0_dp) THEN
         aw(:) = (/ &
                 3.398642214192787e-06_dp, &
                 1.9025312202899538e-05_dp, &
                 5.266161038810626e-05_dp, &
                 0.00011755445811565422_dp, &
                 0.00024054583577387794_dp, &
                 0.00047036774204517446_dp, &
                 0.0008915989199018158_dp, &
                 0.0016480620414409948_dp, &
                 0.00298048536052862_dp, &
                 0.005285920675047294_dp, &
                 0.009210353187127669_dp, &
                 0.01579153884075367_dp, &
                 0.026677173353753962_dp, &
                 0.044455103994883116_dp, &
                 0.07314891087057689_dp, &
                 0.11895589362798491_dp, &
                 0.19133835932619525_dp, &
                 0.3046280564326354_dp, &
                 0.48037697428284243_dp, &
                 0.7508072430736709_dp, &
                 1.1639388014245502_dp, &
                 1.7914938071058262_dp, &
                 2.7421307814095353_dp, &
                 4.18739539739007_dp, &
                 6.427708444690209_dp, &
                 10.150038983021496_dp, &
                 8.840904815568487e-06_dp, &
                 2.3243411245798182e-05_dp, &
                 4.61419948505059e-05_dp, &
                 8.798822647972772e-05_dp, &
                 0.0001658923006259312_dp, &
                 0.0003074688101295367_dp, &
                 0.0005583537180990002_dp, &
                 0.0009937301893068423_dp, &
                 0.001735787847791623_dp, &
                 0.002980424755862107_dp, &
                 0.005037848905051201_dp, &
                 0.008393759041169294_dp, &
                 0.013800737687712429_dp, &
                 0.022413678806554142_dp, &
                 0.03598900168633015_dp, &
                 0.057175813550115986_dp, &
                 0.08993936716796855_dp, &
                 0.1401758115607063_dp, &
                 0.21660925146824841_dp, &
                 0.33212798532910376_dp, &
                 0.5058820683958832_dp, &
                 0.7669510501463104_dp, &
                 1.1620122321700583_dp, &
                 1.775691260253936_dp, &
                 2.8035588004499257_dp, &
                 4.990540570288636_dp/)
      ELSE IF (E_range < 457444.0_dp) THEN
         aw(:) = (/ &
                 1.5924512725077461e-06_dp, &
                 9.091861526358144e-06_dp, &
                 2.617525769293245e-05_dp, &
                 6.171102340282755e-05_dp, &
                 0.00013395463159750724_dp, &
                 0.00027703236128610273_dp, &
                 0.0005524440573740519_dp, &
                 0.0010685815369178362_dp, &
                 0.002012774020332085_dp, &
                 0.0037030176407984823_dp, &
                 0.006670370391832498_dp, &
                 0.011788596543354984_dp, &
                 0.020475865129561125_dp, &
                 0.03500529115221168_dp, &
                 0.058978461638664194_dp, &
                 0.09804109351785203_dp, &
                 0.16095595007242294_dp, &
                 0.26120029165707_dp, &
                 0.41933283009911826_dp, &
                 0.6664993354060413_dp, &
                 1.0496740893941348_dp, &
                 1.6397469396988835_dp, &
                 2.54498795937397_dp, &
                 3.9371777982755236_dp, &
                 6.11745128618483_dp, &
                 9.772519404832513_dp, &
                 4.1599386041678434e-06_dp, &
                 1.1373735340879785e-05_dp, &
                 2.420740702971419e-05_dp, &
                 4.980537853880897e-05_dp, &
                 0.00010018420860309139_dp, &
                 0.00019590630897285626_dp, &
                 0.0003724648960236611_dp, &
                 0.0006901948951647627_dp, &
                 0.0012498546464018527_dp, &
                 0.0022170391172995803_dp, &
                 0.0038600176308915607_dp, &
                 0.006607746433407134_dp, &
                 0.011137935349822067_dp, &
                 0.018509521510997304_dp, &
                 0.030360279138384877_dp, &
                 0.04919938780336098_dp, &
                 0.07883793047658633_dp, &
                 0.12502011000770785_dp, &
                 0.19635093903737422_dp, &
                 0.3056814182538295_dp, &
                 0.47227226126957916_dp, &
                 0.7255282521113733_dp, &
                 1.112683210316189_dp, &
                 1.7190059451835937_dp, &
                 2.7402160914459284_dp, &
                 4.918151105513321_dp/)
      ELSE IF (E_range < 2101965.0_dp) THEN
         aw(:) = (/ &
                 4.376362889407548e-07_dp, &
                 2.643443650482503e-06_dp, &
                 8.433114286038146e-06_dp, &
                 2.2439275175400065e-05_dp, &
                 5.458751916823435e-05_dp, &
                 0.00012469615312218161_dp, &
                 0.00027090154346575427_dp, &
                 0.0005643774226213476_dp, &
                 0.001134437804378847_dp, &
                 0.0022105491534259005_dp, &
                 0.004191467146338867_dp, &
                 0.007757292968144634_dp, &
                 0.014048601297429358_dp, &
                 0.024949196870902246_dp, &
                 0.04352732410856656_dp, &
                 0.07471686858797275_dp, &
                 0.12635898905420878_dp, &
                 0.21078138430975438_dp, &
                 0.3471766909066865_dp, &
                 0.5651735745080563_dp, &
                 0.9102276762862731_dp, &
                 1.451965060215205_dp, &
                 2.2979916853626663_dp, &
                 3.6203271934806356_dp, &
                 5.721128905491445_dp, &
                 9.287036012890901_dp, &
                 1.15698350323011e-06_dp, &
                 3.5239168925237668e-06_dp, &
                 8.790907369106611e-06_dp, &
                 2.0802473841355257e-05_dp, &
                 4.666225571530836e-05_dp, &
                 9.967519249166123e-05_dp, &
                 0.00020419454443829576_dp, &
                 0.0004036329368680413_dp, &
                 0.0007736164897230748_dp, &
                 0.0014432538211834921_dp, &
                 0.002629031662183179_dp, &
                 0.004688130413199208_dp, &
                 0.008201289641287034_dp, &
                 0.014100202125235302_dp, &
                 0.023861417673799547_dp, &
                 0.03979881543108159_dp, &
                 0.06550130495364699_dp, &
                 0.10648414776164583_dp, &
                 0.17115716235174305_dp, &
                 0.27227855433947445_dp, &
                 0.42921718067635106_dp, &
                 0.6717958687439105_dp, &
                 1.048026560686365_dp, &
                 1.6441450980118213_dp, &
                 2.656245656029467_dp, &
                 4.82228579092364_dp/)
      ELSE IF (E_range < 14140999.0_dp) THEN
         aw(:) = (/ &
                 9.23626066730038e-08_dp, &
                 6.508986506667573e-07_dp, &
                 2.548256199033377e-06_dp, &
                 8.121244988671366e-06_dp, &
                 2.280050217208556e-05_dp, &
                 5.838011937544951e-05_dp, &
                 0.00013919237871694767_dp, &
                 0.0003133658603796936_dp, &
                 0.0006727986081208618_dp, &
                 0.0013877632092122146_dp, &
                 0.002765631333703304_dp, &
                 0.005348711634689046_dp, &
                 0.010074592422860395_dp, &
                 0.018535050517138794_dp, &
                 0.03338852029027121_dp, &
                 0.05900918410430352_dp, &
                 0.10249708343303293_dp, &
                 0.17523445903983162_dp, &
                 0.2952639242558341_dp, &
                 0.4909040641368631_dp, &
                 0.8062587841364091_dp, &
                 1.3097798505481377_dp, &
                 2.1083909166084847_dp, &
                 3.3742290423813217_dp, &
                 5.410341564313699_dp, &
                 8.903545764950158_dp, &
                 2.5312239739000636e-07_dp, &
                 9.960996067355009e-07_dp, &
                 3.1628525158730217e-06_dp, &
                 8.836550056217969e-06_dp, &
                 2.2389728532275477e-05_dp, &
                 5.264557648617341e-05_dp, &
                 0.0001166810769940904_dp, &
                 0.00024642731945934143_dp, &
                 0.0004998727804625503_dp, &
                 0.0009797084412907111_dp, &
                 0.0018638236468942935_dp, &
                 0.00345441392005227_dp, &
                 0.0062559619517267165_dp, &
                 0.011097497158462704_dp, &
                 0.019322067389699597_dp, &
                 0.03307729968505806_dp, &
                 0.05575685817980752_dp, &
                 0.09266621527033955_dp, &
                 0.15202316630925258_dp, &
                 0.24647013442575488_dp, &
                 0.39542567933234785_dp, &
                 0.6290352095921129_dp, &
                 0.9959743276132635_dp, &
                 1.583367038071677_dp, &
                 2.5877759334182198_dp, &
                 4.74419254971056_dp/)
      ELSE
         aw(:) = (/ &
                 1.6637944502107968e-08_dp, &
                 1.8303408054343632e-07_dp, &
                 9.687195306133241e-07_dp, &
                 3.7389122528334316e-06_dp, &
                 1.1973447021251308e-05_dp, &
                 3.378096920589386e-05_dp, &
                 8.68338189070593e-05_dp, &
                 0.00020767968945116668_dp, &
                 0.00046874696407475663_dp, &
                 0.001008553647726778_dp, &
                 0.0020841046186256115_dp, &
                 0.0041598936699362606_dp, &
                 0.008056312504669037_dp, &
                 0.015193061067365586_dp, &
                 0.02798246122774749_dp, &
                 0.050456461866690214_dp, &
                 0.08925401954572831_dp, &
                 0.15515900119886547_dp, &
                 0.2654733104360881_dp, &
                 0.44765566677475566_dp, &
                 0.7449030289121367_dp, &
                 1.2248551649261998_dp, &
                 1.9939328236196414_dp, &
                 3.2243037705041826_dp, &
                 5.219599665831332_dp, &
                 8.666863277748455_dp, &
                 5.318821639180232e-08_dp, &
                 3.4945656362875746e-07_dp, &
                 1.4305518660978924e-06_dp, &
                 4.650456118314243e-06_dp, &
                 1.30922605762957e-05_dp, &
                 3.33205562494026e-05_dp, &
                 7.860562991572967e-05_dp, &
                 0.00017467466389322248_dp, &
                 0.0003697061169016621_dp, &
                 0.0007513106176187417_dp, &
                 0.0014748157046539873_dp, &
                 0.0028095645215491876_dp, &
                 0.005213528655212034_dp, &
                 0.009451865748843455_dp, &
                 0.016782910646881463_dp, &
                 0.029246610819691533_dp, &
                 0.05010725392331038_dp, &
                 0.08452724443086411_dp, &
                 0.14058736988255555_dp, &
                 0.23083686098028847_dp, &
                 0.37470521981828925_dp, &
                 0.6025298048228135_dp, &
                 0.9634165096071392_dp, &
                 1.5450976370872966_dp, &
                 2.5445132697805617_dp, &
                 4.694880341120372_dp/)
      END IF
   END SUBROUTINE

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param E_range ...
!> \param aw ...
!> \param E_ratio ...
! **************************************************************************************************
   PURE SUBROUTINE get_coeff_28(k, E_range, aw, E_ratio)
      INTEGER, INTENT(IN)                                :: k
      REAL(KIND=dp), INTENT(IN)                          :: E_range
      REAL(KIND=dp), DIMENSION(2*k), INTENT(OUT)         :: aw
      REAL(KIND=dp), INTENT(OUT)                         :: E_ratio

      E_ratio = 1.0_dp
      IF (E_range < 1545.0_dp) THEN
         E_ratio = 1545.0_dp/E_range
         aw(:) = (/ &
                 0.00019912749139307142_dp, &
                 0.0010623606743332262_dp, &
                 0.002671743519332409_dp, &
                 0.00513532276440788_dp, &
                 0.008635195625198482_dp, &
                 0.013459847519256876_dp, &
                 0.020047630847673463_dp, &
                 0.029039389901087887_dp, &
                 0.04134031090156478_dp, &
                 0.05819756722342521_dp, &
                 0.0813050912035362_dp, &
                 0.11294734291964816_dp, &
                 0.156193889583069_dp, &
                 0.21515890144425773_dp, &
                 0.2953443526600425_dp, &
                 0.4040922882505582_dp, &
                 0.5511803231110346_dp, &
                 0.7496069714265129_dp, &
                 1.016632455219541_dp, &
                 1.3751727396754474_dp, &
                 1.8557042554346754_dp, &
                 2.4989586763196456_dp, &
                 3.3599578911008456_dp, &
                 4.514593597566258_dp, &
                 6.071708534726409_dp, &
                 8.199087984832916_dp, &
                 11.19334614657497_dp, &
                 15.75712024734422_dp, &
                 0.0005125044577801448_dp, &
                 0.0012232237231645984_dp, &
                 0.00201318827111783_dp, &
                 0.0029437832219448537_dp, &
                 0.004103311596300065_dp, &
                 0.005618171589079525_dp, &
                 0.007663188401742024_dp, &
                 0.010469929287815255_dp, &
                 0.01433805703894519_dp, &
                 0.01965640843594298_dp, &
                 0.026936172517646257_dp, &
                 0.036855644579373535_dp, &
                 0.05031748334107659_dp, &
                 0.06852207829804384_dp, &
                 0.09306268185731213_dp, &
                 0.12604985461633972_dp, &
                 0.1702755103243793_dp, &
                 0.22943160937864643_dp, &
                 0.30840755489079047_dp, &
                 0.4137086894086003_dp, &
                 0.5540782272946391_dp, &
                 0.741497238773558_dp, &
                 0.9929644420040254_dp, &
                 1.3340651862992305_dp, &
                 1.8071782497597113_dp, &
                 2.4938998954135005_dp, &
                 3.5946345041711467_dp, &
                 5.901304226915114_dp/)
      ELSE IF (E_range < 2002.0_dp) THEN
         aw(:) = (/ &
                 0.00015937273545708875_dp, &
                 0.0008511031358508971_dp, &
                 0.0021444475784000504_dp, &
                 0.004133806965295614_dp, &
                 0.006979927658415096_dp, &
                 0.010939953042854741_dp, &
                 0.016407375175241108_dp, &
                 0.02396000912675003_dp, &
                 0.034417229669099075_dp, &
                 0.04891406525328216_dp, &
                 0.06900316741446434_dp, &
                 0.09679560984224389_dp, &
                 0.13515199610246875_dp, &
                 0.18793838914519878_dp, &
                 0.2603666777263224_dp, &
                 0.3594458206099125_dp, &
                 0.494579482258333_dp, &
                 0.6783583159834518_dp, &
                 0.927614454599119_dp, &
                 1.2648378420472655_dp, &
                 1.7201130600602677_dp, &
                 2.3338552489887388_dp, &
                 3.1608903963862245_dp, &
                 4.2770721150582895_dp, &
                 5.7913663447139845_dp, &
                 7.871758093638662_dp, &
                 10.814809416909151_dp, &
                 15.321476179970999_dp, &
                 0.00041027809665924954_dp, &
                 0.0009811723452880425_dp, &
                 0.001620915236450962_dp, &
                 0.0023840944111134044_dp, &
                 0.0033503311465523967_dp, &
                 0.004634503191430588_dp, &
                 0.006395709602901231_dp, &
                 0.008844913900028784_dp, &
                 0.012257024616981905_dp, &
                 0.016993101307172367_dp, &
                 0.02353379728419698_dp, &
                 0.03252363542798268_dp, &
                 0.04482787026200198_dp, &
                 0.06160612640371776_dp, &
                 0.08440876245542936_dp, &
                 0.11530375228655552_dp, &
                 0.15704466612137769_dp, &
                 0.21329510792248377_dp, &
                 0.28893382637364007_dp, &
                 0.39048261973383075_dp, &
                 0.5267381633327601_dp, &
                 0.7097795124230211_dp, &
                 0.9567475153369529_dp, &
                 1.2933958092367415_dp, &
                 1.762243934642854_dp, &
                 2.4448656022179347_dp, &
                 3.5410603027618426_dp, &
                 5.839188641520334_dp/)
      ELSE IF (E_range < 2600.0_dp) THEN
         aw(:) = (/ &
                 0.00012724350537371705_dp, &
                 0.0006802420548887551_dp, &
                 0.0017174005833444652_dp, &
                 0.0033210405572595314_dp, &
                 0.005632832416553322_dp, &
                 0.008881578544535886_dp, &
                 0.013419698685145537_dp, &
                 0.019766846961404197_dp, &
                 0.028663060595610065_dp, &
                 0.041139669406250204_dp, &
                 0.058618369068556976_dp, &
                 0.08304861028830621_dp, &
                 0.11709467333563459_dp, &
                 0.16438742365500805_dp, &
                 0.2298611471707931_dp, &
                 0.32020292243210463_dp, &
                 0.44445136041510835_dp, &
                 0.6147946527884036_dp, &
                 0.8476375115705392_dp, &
                 1.1650387509190727_dp, &
                 1.5966797391113152_dp, &
                 2.1826420752053064_dp, &
                 2.9775354931417777_dp, &
                 4.057156308900191_dp, &
                 5.530582551166736_dp, &
                 7.5660060038749855_dp, &
                 10.45997985096501_dp, &
                 14.911921279047414_dp, &
                 0.00032764611790808316_dp, &
                 0.0007852281095230586_dp, &
                 0.001302499732766977_dp, &
                 0.001927928018050743_dp, &
                 0.0027331746862386277_dp, &
                 0.0038223319490506384_dp, &
                 0.005339653436386936_dp, &
                 0.0074768267093066575_dp, &
                 0.010485860861582824_dp, &
                 0.01470204224626301_dp, &
                 0.020577182689456824_dp, &
                 0.028723198501018832_dp, &
                 0.03996852241361688_dp, &
                 0.05543195597157087_dp, &
                 0.07662014166796133_dp, &
                 0.10555670439639143_dp, &
                 0.14495398753342661_dp, &
                 0.1984431170697882_dp, &
                 0.27088687577758624_dp, &
                 0.36881736713632424_dp, &
                 0.501078546468407_dp, &
                 0.6798426309902373_dp, &
                 0.9223911021644032_dp, &
                 1.2546500752337444_dp, &
                 1.719292335657813_dp, &
                 2.3978939243834243_dp, &
                 3.489702010303218_dp, &
                 5.779704886427998_dp/)
      ELSE IF (E_range < 3300.0_dp) THEN
         aw(:) = (/ &
                 0.0001035812061619439_dp, &
                 0.000554312958948722_dp, &
                 0.0014022178351092648_dp, &
                 0.0027199186238307522_dp, &
                 0.004633626332072742_dp, &
                 0.007348901862042884_dp, &
                 0.011184100569752935_dp, &
                 0.016610348266822107_dp, &
                 0.02430141997172798_dp, &
                 0.03520198814732897_dp, &
                 0.05062394356636958_dp, &
                 0.0723803681708885_dp, &
                 0.10296857818003698_dp, &
                 0.14581768397450023_dp, &
                 0.2056217187942231_dp, &
                 0.28878665461351954_dp, &
                 0.4040292970175386_dp, &
                 0.5631795391897031_dp, &
                 0.7822574690472879_dp, &
                 1.0829291748663241_dp, &
                 1.4945032140919154_dp, &
                 2.056746330708999_dp, &
                 2.8240560994099004_dp, &
                 3.87216062398999_dp, &
                 5.31022885971396_dp, &
                 7.306643557351196_dp, &
                 10.157983272563953_dp, &
                 14.562388075641433_dp, &
                 0.000266779348158189_dp, &
                 0.0006406783010995833_dp, &
                 0.0010669475657666657_dp, &
                 0.0015890577230093321_dp, &
                 0.0022720547662622424_dp, &
                 0.003210897076964011_dp, &
                 0.004537212534376952_dp, &
                 0.006426464787961385_dp, &
                 0.009111540381231737_dp, &
                 0.012906037854636046_dp, &
                 0.018237038812244945_dp, &
                 0.025687953276114096_dp, &
                 0.036054499502532374_dp, &
                 0.05041868971863853_dp, &
                 0.070247180509793_dp, &
                 0.0975222869562315_dp, &
                 0.134916933126155_dp, &
                 0.1860296678962944_dp, &
                 0.2557045471811927_dp, &
                 0.3504778528812439_dp, &
                 0.4792309189974202_dp, &
                 0.6542157222577841_dp, &
                 0.8928389893219679_dp, &
                 1.2211855078507017_dp, &
                 1.6820762830637361_dp, &
                 2.357109509558417_dp, &
                 3.4450757129111187_dp, &
                 5.728067480861656_dp/)
      ELSE IF (E_range < 4000.0_dp) THEN
         aw(:) = (/ &
                 8.771437194659891e-05_dp, &
                 0.00046981282949284237_dp, &
                 0.0011904567715385075_dp, &
                 0.0023152670914235655_dp, &
                 0.003959200577898668_dp, &
                 0.006310722793342447_dp, &
                 0.009662905002781183_dp, &
                 0.01445070779560698_dp, &
                 0.021298505075191456_dp, &
                 0.031086296591640365_dp, &
                 0.04504371996352949_dp, &
                 0.06488109904164477_dp, &
                 0.09296905461591826_dp, &
                 0.13258247030343728_dp, &
                 0.1882303453180191_dp, &
                 0.26610050569058297_dp, &
                 0.3746580755427752_dp, &
                 0.5254504247939676_dp, &
                 0.7341916622638109_dp, &
                 1.0222323068269783_dp, &
                 1.418577696146867_dp, &
                 1.9627342158223895_dp, &
                 2.7089187483169885_dp, &
                 3.732793667761796_dp, &
                 5.14359311285564_dp, &
                 7.109854244554064_dp, &
                 9.928196448146211_dp, &
                 14.295810062691444_dp, &
                 0.0002259584596775983_dp, &
                 0.0005436014065931008_dp, &
                 0.0009083522994476501_dp, &
                 0.001360022022949313_dp, &
                 0.0019587330256291995_dp, &
                 0.0027925453027995255_dp, &
                 0.003983542841235872_dp, &
                 0.005695043198214815_dp, &
                 0.008145671980235551_dp, &
                 0.011632693203529134_dp, &
                 0.01656425108725137_dp, &
                 0.023501595060412054_dp, &
                 0.03321470050609412_dp, &
                 0.04675631004359308_dp, &
                 0.06556089291033432_dp, &
                 0.09157703564672734_dp, &
                 0.1274448334217016_dp, &
                 0.17673475828550625_dp, &
                 0.2442731132825866_dp, &
                 0.33659611474367285_dp, &
                 0.4626113339076691_dp, &
                 0.6346313704421029_dp, &
                 0.8701615672163753_dp, &
                 1.1954150913540855_dp, &
                 1.6533375661453558_dp, &
                 2.325557793916785_dp, &
                 3.410528405591019_dp, &
                 5.688122811462863_dp/)
      ELSE IF (E_range < 5000.0_dp) THEN
         aw(:) = (/ &
                 7.231152188478278e-05_dp, &
                 0.00038772726865815517_dp, &
                 0.0009844837786035514_dp, &
                 0.0019209126836657545_dp, &
                 0.003300157154162172_dp, &
                 0.005292557663606624_dp, &
                 0.008164163748376724_dp, &
                 0.012311171926660728_dp, &
                 0.018304991861179568_dp, &
                 0.026956176212698496_dp, &
                 0.03940566382368068_dp, &
                 0.057252298754729467_dp, &
                 0.08272831849932695_dp, &
                 0.11893897414817305_dp, &
                 0.17018831821850053_dp, &
                 0.2424208413642749_dp, &
                 0.34381888188337045_dp, &
                 0.48560994949960345_dp, &
                 0.68315888981334_dp, &
                 0.9574526915358277_dp, &
                 1.3371435383436294_dp, &
                 1.8614300851160115_dp, &
                 2.5843101667934096_dp, &
                 3.5813587287033566_dp, &
                 4.961876490500659_dp, &
                 6.894579341534884_dp, &
                 9.676154843478928_dp, &
                 14.002777296643712_dp, &
                 0.00018632498066864197_dp, &
                 0.00044921855213653585_dp, &
                 0.0007537638296544074_dp, &
                 0.0011359077842782092_dp, &
                 0.0016504930186307747_dp, &
                 0.002378087896208448_dp, &
                 0.0034304274436559386_dp, &
                 0.004957778020056624_dp, &
                 0.007163453510160269_dp, &
                 0.010326973559848835_dp, &
                 0.0148355888498722_dp, &
                 0.02122577871325675_dp, &
                 0.030238442499465622_dp, &
                 0.04289294382786412_dp, &
                 0.06058666241299866_dp, &
                 0.08522882557656473_dp, &
                 0.11942055225842414_dp, &
                 0.16669801861501937_dp, &
                 0.23186426244014152_dp, &
                 0.32145183283272727_dp, &
                 0.44439450297432675_dp, &
                 0.6130709874752484_dp, &
                 0.8450980113616935_dp, &
                 1.1668375144951064_dp, &
                 1.621384532244084_dp, &
                 2.290416624110755_dp, &
                 3.3720270933942214_dp, &
                 5.643641293866133_dp/)
      ELSE IF (E_range < 5800.0_dp) THEN
         aw(:) = (/ &
                 6.358752692633583e-05_dp, &
                 0.0003412046362365837_dp, &
                 0.0008676052314903817_dp, &
                 0.001696724618707933_dp, &
                 0.0029245257586060763_dp, &
                 0.004710233349785592_dp, &
                 0.007303215812339811_dp, &
                 0.011075692416141687_dp, &
                 0.016566304596355927_dp, &
                 0.024542556688053108_dp, &
                 0.03609017098515814_dp, &
                 0.05273820097743335_dp, &
                 0.07663171929303365_dp, &
                 0.11076846568614528_dp, &
                 0.15932179622752599_dp, &
                 0.22808006029257497_dp, &
                 0.32504298834606_dp, &
                 0.4612301705022223_dp, &
                 0.6517778001414668_dp, &
                 0.9174329759430668_dp, &
                 1.2866126278634253_dp, &
                 1.7983078281539955_dp, &
                 2.506365504219328_dp, &
                 3.4862959694100204_dp, &
                 4.847438621720606_dp, &
                 6.7586271477598405_dp, &
                 9.516604972737115_dp, &
                 13.816917061150304_dp, &
                 0.00016387366227471576_dp, &
                 0.0003956833555057065_dp, &
                 0.0006658661861317086_dp, &
                 0.0010080085317879625_dp, &
                 0.0014736812958346364_dp, &
                 0.002138763598196897_dp, &
                 0.00310853235933617_dp, &
                 0.004525172218371909_dp, &
                 0.006582500342505047_dp, &
                 0.0095488999069641_dp, &
                 0.013798331312490912_dp, &
                 0.01985134913295637_dp, &
                 0.028429999006008653_dp, &
                 0.04053182527263095_dp, &
                 0.05752974961085405_dp, &
                 0.08130676932209915_dp, &
                 0.11443765763930915_dp, &
                 0.16043487908983078_dp, &
                 0.2240845367272352_dp, &
                 0.311914711856157_dp, &
                 0.4328741422164402_dp, &
                 0.5993829816920151_dp, &
                 0.8291300891292139_dp, &
                 1.1485759543667498_dp, &
                 1.6009173789914857_dp, &
                 2.2678714811822975_dp, &
                 3.3473105643753676_dp, &
                 5.615101807506323_dp/)
      ELSE IF (E_range < 7000.0_dp) THEN
         aw(:) = (/ &
                 5.40208922943882e-05_dp, &
                 0.0002901574218009231_dp, &
                 0.0007392141971595068_dp, &
                 0.0014500283280455789_dp, &
                 0.002510177021344463_dp, &
                 0.0040657991238261465_dp, &
                 0.006346521670349522_dp, &
                 0.009696151598289092_dp, &
                 0.01461448675071523_dp, &
                 0.021817888400210955_dp, &
                 0.03232620376087572_dp, &
                 0.047584791601727995_dp, &
                 0.06963362235719665_dp, &
                 0.10134008926088696_dp, &
                 0.14671822700211287_dp, &
                 0.21136500303928102_dp, &
                 0.30305507802937337_dp, &
                 0.43255028902361986_dp, &
                 0.6147016140691686_dp, &
                 0.8699548471471977_dp, &
                 1.2264290200274872_dp, &
                 1.7228495271474296_dp, &
                 2.412866754832892_dp, &
                 3.371902180409746_dp, &
                 4.7093379807234985_dp, &
                 6.594155669520039_dp, &
                 9.323180769772668_dp, &
                 13.591212022178699_dp, &
                 0.00013925034927611353_dp, &
                 0.00033689758020930645_dp, &
                 0.0005691288628998808_dp, &
                 0.000866760965686723_dp, &
                 0.0012774771286669078_dp, &
                 0.0018715412342993915_dp, &
                 0.0027465275546555167_dp, &
                 0.004035039597245034_dp, &
                 0.0059195955044671665_dp, &
                 0.0086551765726244_dp, &
                 0.012599568036048066_dp, &
                 0.018253775832994857_dp, &
                 0.02631653618635365_dp, &
                 0.03775822887923194_dp, &
                 0.053921093309633066_dp, &
                 0.07665492497469895_dp, &
                 0.10850075471633389_dp, &
                 0.1529401192419044_dp, &
                 0.2147361484742657_dp, &
                 0.30040908293128193_dp, &
                 0.41892390451696465_dp, &
                 0.5827503975591537_dp, &
                 0.8096664957210812_dp, &
                 1.1262568684972982_dp, &
                 1.5758496574356475_dp, &
                 2.240219834134439_dp, &
                 3.316979586478506_dp, &
                 5.5800996039424104_dp/)
      ELSE IF (E_range < 8500.0_dp) THEN
         aw(:) = (/ &
                 4.564555265477108e-05_dp, &
                 0.0002454346685212948_dp, &
                 0.0006265784407134182_dp, &
                 0.001233158622447077_dp, &
                 0.0021448652153409154_dp, &
                 0.0034954182642228226_dp, &
                 0.005495608826652996_dp, &
                 0.00846210660098653_dp, &
                 0.012857585326682552_dp, &
                 0.01934938119006627_dp, &
                 0.028893873468546265_dp, &
                 0.04285529874482882_dp, &
                 0.06317112153384097_dp, &
                 0.0925808252186819_dp, &
                 0.13494115158016368_dp, &
                 0.19565897567855495_dp, &
                 0.2822840227867385_dp, &
                 0.405318874954234_dp, &
                 0.5793256641233082_dp, &
                 0.8244427232040404_dp, &
                 1.168482383587411_dp, &
                 1.6498933305311403_dp, &
                 2.322117411253598_dp, &
                 3.2604766389691147_dp, &
                 4.574390024972901_dp, &
                 6.432989341834178_dp, &
                 9.133196407474784_dp, &
                 13.369095389855303_dp, &
                 0.00011768969816977658_dp, &
                 0.00028534927032496446_dp, &
                 0.0004840726346740154_dp, &
                 0.0007420579710874475_dp, &
                 0.0011032591807000774_dp, &
                 0.0016325138217318865_dp, &
                 0.002419994753070502_dp, &
                 0.0035891515886961726_dp, &
                 0.005311642196826666_dp, &
                 0.007829394245850389_dp, &
                 0.011484246513613412_dp, &
                 0.016757755663488007_dp, &
                 0.024325289635545783_dp, &
                 0.03512979233715042_dp, &
                 0.05048226189921966_dp, &
                 0.07219835845965666_dp, &
                 0.1027839824530896_dp, &
                 0.14568785531725492_dp, &
                 0.20564777829998312_dp, &
                 0.2891735001868529_dp, &
                 0.4052436947742419_dp, &
                 0.5663760127138803_dp, &
                 0.7904374777760682_dp, &
                 1.1041398360339376_dp, &
                 1.5509491079980848_dp, &
                 2.21270825293388_dp, &
                 3.2867828515308894_dp, &
                 5.5452724054483005_dp/)
      ELSE IF (E_range < 11000.0_dp) THEN
         aw(:) = (/ &
                 3.6488006929306264e-05_dp, &
                 0.00019649151810584802_dp, &
                 0.000503107877439482_dp, &
                 0.000994818980265892_dp, &
                 0.0017419330458111762_dp, &
                 0.00286324584306399_dp, &
                 0.004546790823266163_dp, &
                 0.0070764254736432365_dp, &
                 0.01086988103008825_dp, &
                 0.01653489379720316_dp, &
                 0.024950193189557313_dp, &
                 0.0373800572984858_dp, &
                 0.05563473102373159_dp, &
                 0.08229382069514388_dp, &
                 0.12101608504744633_dp, &
                 0.1769674560118736_dp, &
                 0.25741052665895353_dp, &
                 0.3725145015640989_dp, &
                 0.5364671734722614_dp, &
                 0.7690049629152851_dp, &
                 1.0975352324996732_dp, &
                 1.5601365551801154_dp, &
                 2.209966148011392_dp, &
                 3.1222026085468673_dp, &
                 4.406302512891578_dp, &
                 6.231592566777427_dp, &
                 8.895138025566414_dp, &
                 13.09015522563044_dp, &
                 9.411058161764564e-05_dp, &
                 0.00022887458151702356_dp, &
                 0.00039057680221556677_dp, &
                 0.0006042814819193009_dp, &
                 0.0009094035245976271_dp, &
                 0.0013641332935019103_dp, &
                 0.002049648033435726_dp, &
                 0.003078322757949694_dp, &
                 0.004608548064372825_dp, &
                 0.006866051954909783_dp, &
                 0.010172626387128262_dp, &
                 0.014985141010389004_dp, &
                 0.021949052283394388_dp, &
                 0.03197188778843743_dp, &
                 0.046323942696241904_dp, &
                 0.0667759221190767_dp, &
                 0.09578682159309504_dp, &
                 0.13676065620689198_dp, &
                 0.19439935510505243_dp, &
                 0.27519525733539296_dp, &
                 0.3881406112452913_dp, &
                 0.5458114344320582_dp, &
                 0.7661884133896568_dp, &
                 1.0761497928442625_dp, &
                 1.519347409242135_dp, &
                 2.177726198492106_dp, &
                 3.2483564776686458_dp, &
                 5.5009815780981555_dp/)
      ELSE IF (E_range < 14000.0_dp) THEN
         aw(:) = (/ &
                 2.9588446308604547e-05_dp, &
                 0.00015957764714927812_dp, &
                 0.00040980061859442377_dp, &
                 0.0008141581268268158_dp, &
                 0.0014351942867949477_dp, &
                 0.002379218143911262_dp, &
                 0.003815130228038144_dp, &
                 0.0059991869926145835_dp, &
                 0.009311238414156518_dp, &
                 0.014308514375164259_dp, &
                 0.021803455514145805_dp, &
                 0.03297432754720971_dp, &
                 0.04952104060181237_dp, &
                 0.07388347387186651_dp, &
                 0.1095460460641794_dp, &
                 0.16146092263899975_dp, &
                 0.23663401107846255_dp, &
                 0.3449341441338246_dp, &
                 0.5002090250697223_dp, &
                 0.7218265976855273_dp, &
                 1.036819021323394_dp, &
                 1.4829173538285452_dp, &
                 2.113006031585928_dp, &
                 3.002118887270464_dp, &
                 4.25973634734891_dp, &
                 6.055362139002693_dp, &
                 8.68620971747294_dp, &
                 12.84476212055696_dp, &
                 7.634110961434247e-05_dp, &
                 0.00018622528846790902_dp, &
                 0.00031969127466536623_dp, &
                 0.0004991918529001643_dp, &
                 0.0007602900708133781_dp, &
                 0.0011555100382159716_dp, &
                 0.0017584338105163752_dp, &
                 0.0026721061529931525_dp, &
                 0.004043591457129711_dp, &
                 0.00608455281208252_dp, &
                 0.009099150682157118_dp, &
                 0.013522328657324851_dp, &
                 0.01997274372862434_dp, &
                 0.029325909214924503_dp, &
                 0.04281497070968849_dp, &
                 0.06216914194983893_dp, &
                 0.08980351557167271_dp, &
                 0.12907941549114965_dp, &
                 0.184663246594884_dp, &
                 0.2630278258762335_dp, &
                 0.3731736346682672_dp, &
                 0.5277260380561264_dp, &
                 0.7447670823860634_dp, &
                 1.051327938626868_dp, &
                 1.4912362410625788_dp, &
                 2.1465432019415753_dp, &
                 3.214074319529515_dp, &
                 5.461496007193266_dp/)
      ELSE IF (E_range < 18000.0_dp) THEN
         aw(:) = (/ &
                 2.3779414127343956e-05_dp, &
                 0.00012846422005240743_dp, &
                 0.0003309928831073909_dp, &
                 0.0006610841016675125_dp, &
                 0.001174111517851935_dp, &
                 0.001964735642046167_dp, &
                 0.0031839437046973372_dp, &
                 0.005062138452978201_dp, &
                 0.007943587365676456_dp, &
                 0.012337793711020667_dp, &
                 0.01899408520862755_dp, &
                 0.02900819673889348_dp, &
                 0.0439733623874622_dp, &
                 0.06619336766731646_dp, &
                 0.09898159274083276_dp, &
                 0.1470789732012216_dp, &
                 0.2172359337388872_dp, &
                 0.31902010927146557_dp, &
                 0.465935489961852_dp, &
                 0.676974407399601_dp, &
                 0.9787827564627841_dp, &
                 1.4087293747222094_dp, &
                 2.0194090526711066_dp, &
                 2.8856949554637783_dp, &
                 4.117081205191889_dp, &
                 5.883249820883165_dp, &
                 8.481580484406694_dp, &
                 12.603865500485162_dp, &
                 6.137647450224398e-05_dp, &
                 0.00015022918522230813_dp, &
                 0.00025961689799477495_dp, &
                 0.00040956404897364297_dp, &
                 0.0006319903102840593_dp, &
                 0.0009740518614949558_dp, &
                 0.0015022032585384295_dp, &
                 0.002310722155775527_dp, &
                 0.0035358702066416604_dp, &
                 0.005375683414999454_dp, &
                 0.008117044767788793_dp, &
                 0.012173222749539778_dp, &
                 0.018136165890969762_dp, &
                 0.026849214184554587_dp, &
                 0.03950783752538176_dp, &
                 0.057798731689314514_dp, &
                 0.08409142368704033_dp, &
                 0.1217021378276602_dp, &
                 0.17525857508663245_dp, &
                 0.2512102337617962_dp, &
                 0.3585618870750593_dp, &
                 0.5099851018896473_dp, &
                 0.7236623896941952_dp, &
                 1.0267809590048702_dp, &
                 1.4633527890610423_dp, &
                 2.1155495825866195_dp, &
                 3.1799713652899073_dp, &
                 5.422242322222096_dp/)
      ELSE IF (E_range < 22000.0_dp) THEN
         aw(:) = (/ &
                 1.996975719114667e-05_dp, &
                 0.00010803766024790043_dp, &
                 0.00027914946672295566_dp, &
                 0.0005600685201304379_dp, &
                 0.0010010455833094946_dp, &
                 0.0016883469263468778_dp, &
                 0.0027600190175514337_dp, &
                 0.004427777354772284_dp, &
                 0.0070100858816421925_dp, &
                 0.010981630036803518_dp, &
                 0.017045370162921285_dp, &
                 0.026235975775831418_dp, &
                 0.04006720477930825_dp, &
                 0.060740772707250645_dp, &
                 0.09144094582526643_dp, &
                 0.13674817658247623_dp, &
                 0.20321752055734807_dp, &
                 0.3001847449852204_dp, &
                 0.44088738725627163_dp, &
                 0.6440243823419953_dp, &
                 0.9359372859467849_dp, &
                 1.3537064463445627_dp, &
                 1.9496922567626198_dp, &
                 2.798633041764183_dp, &
                 4.010025863301125_dp, &
                 5.753689968035089_dp, &
                 8.327144493740365_dp, &
                 12.421679016019592_dp, &
                 5.156001410295644e-05_dp, &
                 0.00012656590579908283_dp, &
                 0.00021996531665695752_dp, &
                 0.00035003655094498806_dp, &
                 0.0005460402965281552_dp, &
                 0.0008512186113124776_dp, &
                 0.0013268718440107125_dp, &
                 0.0020609116400843674_dp, &
                 0.0031816393309671174_dp, &
                 0.004876914587104125_dp, &
                 0.007420595371219653_dp, &
                 0.011209484450019506_dp, &
                 0.016815094297439877_dp, &
                 0.02505596575962014_dp, &
                 0.037098296020097635_dp, &
                 0.05459544297770877_dp, &
                 0.07988080804958117_dp, &
                 0.11623432159124078_dp, &
                 0.16825175472580994_dp, &
                 0.2423620424395829_dp, &
                 0.3475704605516449_dp, &
                 0.4965817576093905_dp, &
                 0.7076548141522077_dp, &
                 1.0080986813480735_dp, &
                 1.4420727665642223_dp, &
                 2.09185113214404_dp, &
                 3.1538737345296277_dp, &
                 5.392217904025483_dp/)
      ELSE IF (E_range < 30000.0_dp) THEN
         aw(:) = (/ &
                 1.5245472100141073e-05_dp, &
                 8.267520676446116e-05_dp, &
                 0.0002146253100574728_dp, &
                 0.00043387740900171205_dp, &
                 0.0007836944027554964_dp, &
                 0.0013387930663292789_dp, &
                 0.0022193817580203824_dp, &
                 0.0036113909961874636_dp, &
                 0.005797519906650395_dp, &
                 0.009203847862920226_dp, &
                 0.01446808989009802_dp, &
                 0.022538313347044602_dp, &
                 0.034814713569521594_dp, &
                 0.05335209638799308_dp, &
                 0.0811475703355085_dp, &
                 0.12254732266532327_dp, &
                 0.18381921201396537_dp, &
                 0.27395570440350625_dp, &
                 0.40579685475434923_dp, &
                 0.5976003453364507_dp, &
                 0.875245772613848_dp, &
                 1.2753705825844541_dp, &
                 1.8499693672606738_dp, &
                 2.6735624079289133_dp, &
                 3.855638524201632_dp, &
                 5.566220151857392_dp, &
                 8.103052167483417_dp, &
                 12.156727650422702_dp, &
                 3.9383299458817714e-05_dp, &
                 9.713918675705757e-05_dp, &
                 0.000170421510611186_dp, &
                 0.0002751089399893439_dp, &
                 0.0004367526322332042_dp, &
                 0.0006931539398019088_dp, &
                 0.0010985049239842675_dp, &
                 0.0017318671480729233_dp, &
                 0.0027102869498213354_dp, &
                 0.004207039027136905_dp, &
                 0.006477134639756053_dp, &
                 0.009893360814081634_dp, &
                 0.014997191697894255_dp, &
                 0.02257042180249529_dp, &
                 0.03373549826844379_dp, &
                 0.05009547634855471_dp, &
                 0.07392862462849442_dp, &
                 0.10845863055245018_dp, &
                 0.15823054590725322_dp, &
                 0.22963859190692054_dp, &
                 0.33168416368000975_dp, &
                 0.47711716376192037_dp, &
                 0.6843078702870563_dp, &
                 0.9807484854746213_dp, &
                 1.4108257998358733_dp, &
                 2.0569814817847933_dp, &
                 3.1154409728311463_dp, &
                 5.348031605731007_dp/)
      ELSE IF (E_range < 40000.0_dp) THEN
         aw(:) = (/ &
                 1.1868158588050793e-05_dp, &
                 6.451496656973e-05_dp, &
                 0.00016828367389327262_dp, &
                 0.0003428138578813575_dp, &
                 0.0006257728657443166_dp, &
                 0.001082546664145262_dp, &
                 0.0018189122009131555_dp, &
                 0.002999899699939122_dp, &
                 0.004879035508307028_dp, &
                 0.007842398392055541_dp, &
                 0.012473504017331851_dp, &
                 0.019647823262643315_dp, &
                 0.030669526100067415_dp, &
                 0.04746815571454119_dp, &
                 0.07287994053111492_dp, &
                 0.11104807754173227_dp, &
                 0.16798957023910677_dp, &
                 0.2523945702323838_dp, &
                 0.37675011948627407_dp, &
                 0.5589184164507482_dp, &
                 0.8243607282457994_dp, &
                 1.209308359311203_dp, &
                 1.7654144169432378_dp, &
                 2.566987933665634_dp, &
                 3.7234969430883793_dp, &
                 5.405141983044582_dp, &
                 7.9098851591902335_dp, &
                 11.927751732780631_dp, &
                 3.067518035880976e-05_dp, &
                 7.602736152767003e-05_dp, &
                 0.00013466099350051709_dp, &
                 0.0002205156203375216_dp, &
                 0.00035609796830194946_dp, &
                 0.0005747749473125263_dp, &
                 0.0009249835582363322_dp, &
                 0.0014785254567841132_dp, &
                 0.0023430318256698933_dp, &
                 0.003679400712048183_dp, &
                 0.005726484701954953_dp, &
                 0.008836295353538897_dp, &
                 0.013524092658912494_dp, &
                 0.02053931124436599_dp, &
                 0.030965498069383788_dp, &
                 0.04636050178159656_dp, &
                 0.06895241489824191_dp, &
                 0.1019128924749107_dp, &
                 0.14973885102211165_dp, &
                 0.2187896005944566_dp, &
                 0.31805835198625043_dp, &
                 0.4603305714250192_dp, &
                 0.6640726964679041_dp, &
                 0.9569407810619277_dp, &
                 1.3835309605116226_dp, &
                 2.026448697565121_dp, &
                 3.0817527233540427_dp, &
                 5.309325429687417_dp/)
      ELSE IF (E_range < 55000.0_dp) THEN
         aw(:) = (/ &
                 8.994970387322457e-06_dp, &
                 4.9038651219381535e-05_dp, &
                 0.00012866021417541092_dp, &
                 0.00026454506965524195_dp, &
                 0.0004890218504779385_dp, &
                 0.0008585083873106919_dp, &
                 0.0014649004957659442_dp, &
                 0.002453066714438582_dp, &
                 0.00404818477662515_dp, &
                 0.006597100721999991_dp, &
                 0.010629669867507806_dp, &
                 0.016948840913414677_dp, &
                 0.026762034057250313_dp, &
                 0.041871547214080776_dp, &
                 0.06494887934178976_dp, &
                 0.09992774926372437_dp, &
                 0.15256426013426896_dp, &
                 0.2312316469143259_dp, &
                 0.3480438390880272_dp, &
                 0.520441341810499_dp, &
                 0.7734349455842019_dp, &
                 1.1428133579373827_dp, &
                 1.6798513048817534_dp, &
                 2.4586158447712725_dp, &
                 3.5885384348523433_dp, &
                 5.240004673003719_dp, &
                 7.7112226946250795_dp, &
                 11.691667476075265_dp, &
                 2.3263971840934205e-05_dp, &
                 5.799733961007195e-05_dp, &
                 0.00010391874644151258_dp, &
                 0.0001730995125618299_dp, &
                 0.0002850761818642179_dp, &
                 0.00046892999174830916_dp, &
                 0.0007675526539416234_dp, &
                 0.0012456339379473573_dp, &
                 0.0020014080059075953_dp, &
                 0.0031832610399331806_dp, &
                 0.005013553854404307_dp, &
                 0.007822909693482165_dp, &
                 0.012099368861470802_dp, &
                 0.018558470273632704_dp, &
                 0.02824263531163439_dp, &
                 0.04266142900862469_dp, &
                 0.06398871559990858_dp, &
                 0.09533906369189366_dp, &
                 0.14115536539598936_dp, &
                 0.20775581572804178_dp, &
                 0.3041200099484807_dp, &
                 0.4430661693538553_dp, &
                 0.6431592914657807_dp, &
                 0.9322296232686789_dp, &
                 1.355102086814327_dp, &
                 1.9945706936958254_dp, &
                 3.046542150566686_dp, &
                 5.268893998445498_dp/)
      ELSE IF (E_range < 75000.0_dp) THEN
         aw(:) = (/ &
                 6.8673479673260504e-06_dp, &
                 3.755590753670896e-05_dp, &
                 9.915139981458229e-05_dp, &
                 0.00020590986120619216_dp, &
                 0.00038570411969655687_dp, &
                 0.0006874225835250557_dp, &
                 0.0011912995282688835_dp, &
                 0.0020251938479038355_dp, &
                 0.0033901629916074914_dp, &
                 0.005599342672133319_dp, &
                 0.009136035642934201_dp, &
                 0.0147397005849582_dp, &
                 0.023532325447503034_dp, &
                 0.03720291582111918_dp, &
                 0.05827512485526659_dp, &
                 0.09049318420974399_dp, &
                 0.13937535368635193_dp, &
                 0.21300368025384273_dp, &
                 0.3231464746438852_dp, &
                 0.4868502227776408_dp, &
                 0.7287007191729763_dp, &
                 1.0840642579491595_dp, &
                 1.6038486600019533_dp, &
                 2.3618790984089095_dp, &
                 3.467539083950752_dp, &
                 5.09138219825452_dp, &
                 7.53185924341038_dp, &
                 11.47798005469671_dp, &
                 1.7773461657358013e-05_dp, &
                 4.458780785431919e-05_dp, &
                 8.088403265892346e-05_dp, &
                 0.00013715743270713957_dp, &
                 0.00023041539243485957_dp, &
                 0.00038613012816423253_dp, &
                 0.0006425163485643913_dp, &
                 0.001058147634119284_dp, &
                 0.0017230321030210073_dp, &
                 0.002774473725192771_dp, &
                 0.004420090523231148_dp, &
                 0.006971221027705693_dp, &
                 0.010891151319944746_dp, &
                 0.01686432115872218_dp, &
                 0.025895072083461815_dp, &
                 0.03944781877116618_dp, &
                 0.059645137430023996_dp, &
                 0.0895468281752171_dp, &
                 0.13354285706817015_dp, &
                 0.1979094322641605_dp, &
                 0.291608890942118_dp, &
                 0.42748519623553954_dp, &
                 0.6241916121496675_dp, &
                 0.9097205132707596_dp, &
                 1.3291156770565788_dp, &
                 1.965360170512719_dp, &
                 3.0142420912237333_dp, &
                 5.231825921337635_dp/)
      ELSE IF (E_range < 100000.0_dp) THEN
         aw(:) = (/ &
                 5.34722652436438e-06_dp, &
                 2.9335345788005576e-05_dp, &
                 7.794420445994551e-05_dp, &
                 0.0001635102977617264_dp, &
                 0.0003103380666014091_dp, &
                 0.0005612614423847721_dp, &
                 0.0009871305012123896_dp, &
                 0.0017020454525922442_dp, &
                 0.002887383643586052_dp, &
                 0.004828517011721044_dp, &
                 0.007970062065688042_dp, &
                 0.012998268728170836_dp, &
                 0.02096295289681554_dp, &
                 0.03345668031415268_dp, &
                 0.052876323139595754_dp, &
                 0.08280245241436206_dp, &
                 0.1285464334202786_dp, &
                 0.19793519202552176_dp, &
                 0.30243198793781245_dp, &
                 0.4587328138519049_dp, &
                 0.6910421452320579_dp, &
                 1.0343433822030452_dp, &
                 1.5392071335504423_dp, &
                 2.279230497059965_dp, &
                 3.3637429883435237_dp, &
                 4.96344347806973_dp, &
                 7.377008968134632_dp, &
                 11.293074733218035_dp, &
                 1.3848866712786266e-05_dp, &
                 3.496420645730977e-05_dp, &
                 6.422528074399328e-05_dp, &
                 0.00011085156828305275_dp, &
                 0.00018979249819623976_dp, &
                 0.0003236153897281577_dp, &
                 0.0005467478277421638_dp, &
                 0.000912713310709869_dp, &
                 0.001504624320381209_dp, &
                 0.0024504009658675406_dp, &
                 0.003945073258731463_dp, &
                 0.006283381657271798_dp, &
                 0.009907137885517564_dp, &
                 0.015473588963650402_dp, &
                 0.023953503710281564_dp, &
                 0.03677113226590553_dp, &
                 0.056002962059688105_dp, &
                 0.08465896050001478_dp, &
                 0.1270800799647972_dp, &
                 0.18950233376937137_dp, &
                 0.2808690547904898_dp, &
                 0.4140431391215438_dp, &
                 0.6077531842367035_dp, &
                 0.8901351666158656_dp, &
                 1.3064316369806335_dp, &
                 1.9398043844798791_dp, &
                 2.9859550412644587_dp, &
                 5.199382405454911_dp/)
      ELSE IF (E_range < 140000.0_dp) THEN
         aw(:) = (/ &
                 3.991479954961762e-06_dp, &
                 2.198720501971829e-05_dp, &
                 5.890526173006801e-05_dp, &
                 0.00012518031616235489_dp, &
                 0.00024153524567753135_dp, &
                 0.00044471016635781095_dp, &
                 0.0007960973675010948_dp, &
                 0.0013958401833713643_dp, &
                 0.0024051630337238337_dp, &
                 0.004080730226383008_dp, &
                 0.0068267899198047955_dp, &
                 0.011273593700680132_dp, &
                 0.018394388130399692_dp, &
                 0.029678633788751282_dp, &
                 0.04738664952634921_dp, &
                 0.07492145227987683_dp, &
                 0.11736834467836472_dp, &
                 0.18227349630314565_dp, &
                 0.2807619982420454_dp, &
                 0.42913828874392707_dp, &
                 0.65117720616762_dp, &
                 0.9814263687158781_dp, &
                 1.4700673778736164_dp, &
                 2.190427750165236_dp, &
                 3.2517634440541547_dp, &
                 4.824930255638227_dp, &
                 7.208868255000471_dp, &
                 11.091834807498854_dp, &
                 1.0346862342302347e-05_dp, &
                 2.6338249637987535e-05_dp, &
                 4.916412854301245e-05_dp, &
                 8.674834277534896e-05_dp, &
                 0.00015194758780428323_dp, &
                 0.00026440532385478294_dp, &
                 0.00045469618333409404_dp, &
                 0.0007711009718467_dp, &
                 0.0012894728448137978_dp, &
                 0.002127757911813279_dp, &
                 0.0034674947327675617_dp, &
                 0.005585492420042323_dp, &
                 0.008900179331682422_dp, &
                 0.014038952538275588_dp, &
                 0.02193542166778625_dp, &
                 0.033968992230147974_dp, &
                 0.05216418521519709_dp, &
                 0.07947410573244414_dp, &
                 0.12018284604662782_dp, &
                 0.18047837042907597_dp, &
                 0.26927867836352665_dp, &
                 0.3994632895161757_dp, &
                 0.5898412436801937_dp, &
                 0.8687080586761889_dp, &
                 1.2815327235597582_dp, &
                 1.91168824489995_dp, &
                 2.954800144592791_dp, &
                 5.163666162590388_dp/)
      ELSE IF (E_range < 200000.0_dp) THEN
         aw(:) = (/ &
                 2.9287097241919463e-06_dp, &
                 1.6211294489452685e-05_dp, &
                 4.3860626639419625e-05_dp, &
                 9.463310124321836e-05_dp, &
                 0.00018604994609815954_dp, &
                 0.0003493943765511619_dp, &
                 0.0006375682448360294_dp, &
                 0.0011380795201948043_dp, &
                 0.001993709375951546_dp, &
                 0.00343456904920158_dp, &
                 0.005827197217933671_dp, &
                 0.009749051421957427_dp, &
                 0.016100560249175083_dp, &
                 0.026272317263678777_dp, &
                 0.042392659447799845_dp, &
                 0.06769166630053426_dp, &
                 0.10703277867836304_dp, &
                 0.16768452950177593_dp, &
                 0.26043499640668427_dp, &
                 0.4011952042193507_dp, &
                 0.6133044483273886_dp, &
                 0.9308639628203111_dp, &
                 1.4036513854446626_dp, &
                 2.1047075037398466_dp, &
                 3.143200044406983_dp, &
                 4.6901364769963045_dp, &
                 7.044731962208944_dp, &
                 10.894906305627384_dp, &
                 7.599945789593838e-06_dp, &
                 1.9535340209099297e-05_dp, &
                 3.7160656136204285e-05_dp, &
                 6.722566694134913e-05_dp, &
                 0.00012069652213103665_dp, &
                 0.00021459840053591846_dp, &
                 0.0003759981286012543_dp, &
                 0.0006483001768113125_dp, &
                 0.0011005125730724488_dp, &
                 0.0018410833738741884_dp, &
                 0.0030385893339244744_dp, &
                 0.004952462825414513_dp, &
                 0.007978278215654642_dp, &
                 0.012714011523554197_dp, &
                 0.02005632833363888_dp, &
                 0.03133963591517064_dp, &
                 0.04853577656951428_dp, &
                 0.07453950556949848_dp, &
                 0.11357558363229604_dp, &
                 0.17178036896999477_dp, &
                 0.2580420507206907_dp, &
                 0.3852520170265286_dp, &
                 0.5722961014253114_dp, &
                 0.847629061135607_dp, &
                 1.2569519792288701_dp, &
                 1.8838624046566568_dp, &
                 2.923931086168498_dp, &
                 5.1282957309537895_dp/)
      ELSE IF (E_range < 280000.0_dp) THEN
         aw(:) = (/ &
                 2.187887975241367e-06_dp, &
                 1.217302105778866e-05_dp, &
                 3.3280277384650595e-05_dp, &
                 7.294651585196924e-05_dp, &
                 0.0001461477506435841_dp, &
                 0.0002798259055981046_dp, &
                 0.0005201025046636251_dp, &
                 0.0009442991970046586_dp, &
                 0.0016801646722205163_dp, &
                 0.0029359404461932336_dp, &
                 0.005046805926292481_dp, &
                 0.008545920293678393_dp, &
                 0.014272096616783494_dp, &
                 0.023531600046055763_dp, &
                 0.038339327744621156_dp, &
                 0.061775614958965434_dp, &
                 0.09851040024837446_dp, &
                 0.15556831587766237_dp, &
                 0.243439200558_dp, &
                 0.3776832590482275_dp, &
                 0.5812481524315448_dp, &
                 0.8878298248851654_dp, &
                 1.3468344739190594_dp, &
                 2.031033514764075_dp, &
                 3.0495035108172392_dp, &
                 4.573381799063933_dp, &
                 6.902137854735053_dp, &
                 10.723425645457905_dp, &
                 5.683864101635673e-06_dp, &
                 1.4761618602067145e-05_dp, &
                 2.863922084439926e-05_dp, &
                 5.31200028912619e-05_dp, &
                 9.765726111159933e-05_dp, &
                 0.00017719091014345152_dp, &
                 0.00031593721698955905_dp, &
                 0.0005532594185059319_dp, &
                 0.000952423486460829_dp, &
                 0.001613835283450162_dp, &
                 0.002695001718544619_dp, &
                 0.004440391442639413_dp, &
                 0.007225736946277607_dp, &
                 0.01162326005144731_dp, &
                 0.018497020133015078_dp, &
                 0.02914136949940742_dp, &
                 0.04548081139069081_dp, &
                 0.07035707490413387_dp, &
                 0.10794019701910468_dp, &
                 0.1643177452299961_dp, &
                 0.24834763708314758_dp, &
                 0.37292767817567907_dp, &
                 0.5570086406904123_dp, &
                 0.829186245882319_dp, &
                 1.2353723738001468_dp, &
                 1.8593753882752646_dp, &
                 2.896735710900212_dp, &
                 5.0971507815821235_dp/)
      ELSE IF (E_range < 400000.0_dp) THEN
         aw(:) = (/ &
                 1.6070031867276192e-06_dp, &
                 8.996223208889756e-06_dp, &
                 2.4903330256736718e-05_dp, &
                 5.5597271413213874e-05_dp, &
                 0.00011378123225419507_dp, &
                 0.00022251943620373886_dp, &
                 0.0004218452833351526_dp, &
                 0.0007798397462569304_dp, &
                 0.001410464737003683_dp, &
                 0.0025017001910060442_dp, &
                 0.0043594012767892145_dp, &
                 0.007474944527557248_dp, &
                 0.012628553354571935_dp, &
                 0.021045689348535686_dp, &
                 0.03463175327773065_dp, &
                 0.05632151430959264_dp, &
                 0.09059550579868533_dp, &
                 0.1442379871996754_dp, &
                 0.2274428250359289_dp, &
                 0.35541953739182136_dp, &
                 0.5507212109659467_dp, &
                 0.8466317459799636_dp, &
                 1.2921752687688919_dp, &
                 1.9598406408284006_dp, &
                 2.9586009577769_dp, &
                 4.459717588193107_dp, &
                 6.762921787948558_dp, &
                 10.555632767123296_dp, &
                 4.180340751600858e-06_dp, &
                 1.099130280826382e-05_dp, &
                 2.1822771688601605e-05_dp, &
                 4.1621175974422805e-05_dp, &
                 7.848461568768336e-05_dp, &
                 0.00014548449813908576_dp, &
                 0.00026422505480514413_dp, &
                 0.0004703011644186479_dp, &
                 0.0008215666081596167_dp, &
                 0.0014107803843716492_dp, &
                 0.002384836263299515_dp, &
                 0.0039737397649737246_dp, &
                 0.006533894710819361_dp, &
                 0.010612242095881175_dp, &
                 0.017040581368346817_dp, &
                 0.027073311442312262_dp, &
                 0.04258730399078407_dp, &
                 0.06637038094501468_dp, &
                 0.10253618298821561_dp, &
                 0.15712087871604702_dp, &
                 0.23894860519216926_dp, &
                 0.36091957483420634_dp, &
                 0.5420459855749467_dp, &
                 0.8110632955302136_dp, &
                 1.2140975460551333_dp, &
                 1.8351776848641823_dp, &
                 2.8698308488017825_dp, &
                 5.066349620899207_dp/)
      ELSE IF (E_range < 700000.0_dp) THEN
         aw(:) = (/ &
                 9.917609137977847e-07_dp, &
                 5.61571565668021e-06_dp, &
                 1.5905578931798044e-05_dp, &
                 3.668045117366575e-05_dp, &
                 7.779911321491552e-05_dp, &
                 0.0001574703243782781_dp, &
                 0.0003080352227521647_dp, &
                 0.0005857339949829306_dp, &
                 0.0010866136806263423_dp, &
                 0.0019719871993643633_dp, &
                 0.0035086606810778888_dp, &
                 0.006131766091278258_dp, &
                 0.010541856280946737_dp, &
                 0.017853461412239566_dp, &
                 0.029820286970471933_dp, &
                 0.049173663410606715_dp, &
                 0.08012714696416229_dp, &
                 0.1291233103698595_dp, &
                 0.20593171918618178_dp, &
                 0.32525471926745003_dp, &
                 0.5090693312937355_dp, &
                 0.7900513389150534_dp, &
                 1.216653152188614_dp, &
                 1.8609312710223198_dp, &
                 2.831687087034342_dp, &
                 4.300351181611193_dp, &
                 6.567047622277454_dp, &
                 10.318912411219488_dp, &
                 2.5862090723245625e-06_dp, &
                 6.95626109831625e-06_dp, &
                 1.4392071614748994e-05_dp, &
                 2.874925884264212e-05_dp, &
                 5.642970465568392e-05_dp, &
                 0.00010814505912642593_dp, &
                 0.0002021008939835649_dp, &
                 0.0003688922773676731_dp, &
                 0.0006591003182679095_dp, &
                 0.0011550942822986209_dp, &
                 0.0019891964272602783_dp, &
                 0.0033713558748037233_dp, &
                 0.0056309115068987_dp, &
                 0.00927906547653282_dp, &
                 0.015101574633332329_dp, &
                 0.024295245067122773_dp, &
                 0.03866754865528627_dp, &
                 0.060926803424727255_dp, &
                 0.09510216196058996_dp, &
                 0.14715087305959354_dp, &
                 0.22584200604915872_dp, &
                 0.34407208794479094_dp, &
                 0.5209358458550991_dp, &
                 0.7853686557167923_dp, &
                 1.1838124863793347_dp, &
                 1.8006332661854303_dp, &
                 2.8313699702065134_dp, &
                 5.0223450191392205_dp/)
      ELSE IF (E_range < 1200000.0_dp) THEN
         aw(:) = (/ &
                 6.243870742444705e-07_dp, &
                 3.5837547513507616e-06_dp, &
                 1.0424530634626105e-05_dp, &
                 2.4910967538724005e-05_dp, &
                 5.48216536959138e-05_dp, &
                 0.0001148065303999889_dp, &
                 0.000231490488679646_dp, &
                 0.00045215860524583124_dp, &
                 0.0008590837338469464_dp, &
                 0.0015927682432411259_dp, &
                 0.0028891299956185297_dp, &
                 0.005138253059658019_dp, &
                 0.008976135083193377_dp, &
                 0.015426433468499803_dp, &
                 0.026117293389759495_dp, &
                 0.043609983170991296_dp, &
                 0.0718927383781267_dp, &
                 0.11711702885780055_dp, &
                 0.18868749571084845_dp, &
                 0.3008660121451099_dp, &
                 0.47512403493520045_dp, &
                 0.7435972663838382_dp, &
                 1.154222692676153_dp, &
                 1.7786576652841575_dp, &
                 2.7255320837393326_dp, &
                 4.166412460588324_dp, &
                 6.401776404663582_dp, &
                 10.11856522638141_dp, &
                 1.6329106731735064e-06_dp, &
                 4.511304110662316e-06_dp, &
                 9.770509823422929e-06_dp, &
                 2.0454024246766166e-05_dp, &
                 4.172679855059072e-05_dp, &
                 8.25404415798886e-05_dp, &
                 0.00015847711044459983_dp, &
                 0.00029619555479324744_dp, &
                 0.0005404696680087612_dp, &
                 0.000965266351597196_dp, &
                 0.0016909815488903273_dp, &
                 0.0029109600865339395_dp, &
                 0.004931885217935304_dp, &
                 0.008234716202420396_dp, &
                 0.01356583086594651_dp, &
                 0.022072242618732946_dp, &
                 0.035500720158786525_dp, &
                 0.05648910800327906_dp, &
                 0.08899040639475064_dp, &
                 0.13888891315246274_dp, &
                 0.21489980129365738_dp, &
                 0.32990928766337146_dp, &
                 0.5030773832314516_dp, &
                 0.7635106326257592_dp, &
                 1.157931116384902_dp, &
                 1.771014718506682_dp, &
                 2.798340277794168_dp, &
                 4.984574132350961_dp/)
      ELSE IF (E_range < 2000000.0_dp) THEN
         aw(:) = (/ &
                 4.0370316213559476e-07_dp, &
                 2.3541711422913717e-06_dp, &
                 7.057573612279361e-06_dp, &
                 1.751162431119145e-05_dp, &
                 3.998197703771905e-05_dp, &
                 8.651758104084795e-05_dp, &
                 0.00017950101241220228_dp, &
                 0.00035945947508640393_dp, &
                 0.0006981140824218166_dp, &
                 0.001319810424458466_dp, &
                 0.0024361923309759396_dp, &
                 0.004401544013890147_dp, &
                 0.007800025204094134_dp, &
                 0.013581611201940205_dp, &
                 0.023271704375636543_dp, &
                 0.03929119474223725_dp, &
                 0.06544075124204898_dp, &
                 0.10762751156023793_dp, &
                 0.17494730054790258_dp, &
                 0.2812860986936203_dp, &
                 0.44768004577537496_dp, &
                 0.705795087398322_dp, &
                 1.1031142167130359_dp, &
                 1.7109362858917831_dp, &
                 2.6377277178978358_dp, &
                 4.055162275207615_dp, &
                 6.264028054745519_dp, &
                 9.951137323627291_dp, &
                 1.0593262130160056e-06_dp, &
                 3.0185707613834373e-06_dp, &
                 6.8658831794879865e-06_dp, &
                 1.5046778554922312e-05_dp, &
                 3.182893187575826e-05_dp, &
                 6.484710011709137e-05_dp, &
                 0.00012766407757989652_dp, &
                 0.000243860437909589_dp, &
                 0.00045361234593234906_dp, &
                 0.0008241579727338823_dp, &
                 0.001466238321223863_dp, &
                 0.002559617356616681_dp, &
                 0.004392266266516603_dp, &
                 0.0074199303035661236_dp, &
                 0.012355848225194005_dp, &
                 0.02030472924789548_dp, &
                 0.0329612503015901_dp, &
                 0.05290210344877982_dp, &
                 0.08401329609895744_dp, &
                 0.13211363792116823_dp, &
                 0.20586780977743016_dp, &
                 0.3181478896142563_dp, &
                 0.48816468770340044_dp, &
                 0.7451687687530838_dp, &
                 1.1361253599700833_dp, &
                 1.7459877749472628_dp, &
                 2.7703909434425067_dp, &
                 4.952626774509824_dp/)
      ELSE IF (E_range < 4500000.0_dp) THEN
         aw(:) = (/ &
                 2.0330900665342417e-07_dp, &
                 1.2259721995909777e-06_dp, &
                 3.9000493864165235e-06_dp, &
                 1.0346548674455454e-05_dp, &
                 2.5104573254227977e-05_dp, &
                 5.722321695341548e-05_dp, &
                 0.0001240915794841191_dp, &
                 0.00025812480274594484_dp, &
                 0.0005181574776824829_dp, &
                 0.0010084979886612006_dp, &
                 0.0019102647242876139_dp, &
                 0.003532146072512572_dp, &
                 0.0063914997207670885_dp, &
                 0.011342304908756772_dp, &
                 0.019774721642387954_dp, &
                 0.03392304253932974_dp, &
                 0.05733626485492922_dp, &
                 0.0955906794116426_dp, &
                 0.15736013123361753_dp, &
                 0.2560121769552906_dp, &
                 0.4119766499624099_dp, &
                 0.6562580036536524_dp, &
                 1.0356912360288544_dp, &
                 1.6210528978513052_dp, &
                 2.5205569278227222_dp, &
                 3.906010837038453_dp, &
                 6.07864411392436_dp, &
                 9.725147433824423_dp, &
                 5.372960626792664e-07_dp, &
                 1.6313777234005112e-06_dp, &
                 4.053582188809558e-06_dp, &
                 9.562645330678188e-06_dp, &
                 2.140029237042944e-05_dp, &
                 4.5628018410594356e-05_dp, &
                 9.332815320834305e-05_dp, &
                 0.00018423715674012751_dp, &
                 0.0003527071740962291_dp, &
                 0.0006573395283833412_dp, &
                 0.0011963286460791483_dp, &
                 0.0021315852595548933_dp, &
                 0.003726211411404738_dp, &
                 0.006402085885163769_dp, &
                 0.010827471574921069_dp, &
                 0.018049062427338197_dp, &
                 0.029689317308315376_dp, &
                 0.048239120452325215_dp, &
                 0.0774890829270111_dp, &
                 0.12316278045588097_dp, &
                 0.1938483490800667_dp, &
                 0.30239008881049667_dp, &
                 0.4680611192958136_dp, &
                 0.7203071572907666_dp, &
                 1.1064348627930074_dp, &
                 1.7118002420359797_dp, &
                 2.732149945273555_dp, &
                 4.908937928661797_dp/)
      ELSE IF (E_range < 10000000.0_dp) THEN
         aw(:) = (/ &
                 1.0453117858944355e-07_dp, &
                 6.608617412970565e-07_dp, &
                 2.263239779628554e-06_dp, &
                 6.456580799232289e-06_dp, &
                 1.6648773232985477e-05_dp, &
                 3.988538010811069e-05_dp, &
                 9.013431830037881e-05_dp, &
                 0.00019412595183632066_dp, &
                 0.00040148594866505004_dp, &
                 0.000801953768960446_dp, &
                 0.0015541047607406542_dp, &
                 0.002932474750123789_dp, &
                 0.0054037297272779_dp, &
                 0.0097481213186169_dp, &
                 0.01725076571642057_dp, &
                 0.029999477630049398_dp, &
                 0.05134367891607686_dp, &
                 0.08659470336954253_dp, &
                 0.14408515309896788_dp, &
                 0.2367592019652705_dp, &
                 0.3845463783654337_dp, &
                 0.6178989765753549_dp, &
                 0.9831034515051852_dp, &
                 1.5504849076462728_dp, &
                 2.4280266299570483_dp, &
                 3.7876319869184405_dp, &
                 5.930901453777237_dp, &
                 9.544473553973464_dp, &
                 2.7911359387760054e-07_dp, &
                 9.225699169205882e-07_dp, &
                 2.525970869540034e-06_dp, &
                 6.400448408750558e-06_dp, &
                 1.5105857774384374e-05_dp, &
                 3.360200325419909e-05_dp, &
                 7.118848251951222e-05_dp, &
                 0.00014478894137667428_dp, &
                 0.0002844248664965114_dp, &
                 0.0005421792095475284_dp, &
                 0.001006646872918385_dp, &
                 0.0018259034247645597_dp, &
                 0.003243545770723624_dp, &
                 0.0056545862514208655_dp, &
                 0.009691209964763319_dp, &
                 0.016353052767086353_dp, &
                 0.027203315594152448_dp, &
                 0.044661579803193296_dp, &
                 0.07243798977464_dp, &
                 0.11617410461859103_dp, &
                 0.18438950337219645_dp, &
                 0.2898984327270837_dp, &
                 0.45201785956261764_dp, &
                 0.7003495143727266_dp, &
                 1.0824839695902242_dp, &
                 1.6841236033891487_dp, &
                 2.701136203287264_dp, &
                 4.87352184807004_dp/)
      ELSE IF (E_range < 50000000.0_dp) THEN
         aw(:) = (/ &
                 2.883608355013751e-08_dp, &
                 2.1533521290478e-07_dp, &
                 8.922309745530507e-07_dp, &
                 2.9625449220607536e-06_dp, &
                 8.567451943237967e-06_dp, &
                 2.243432347635065e-05_dp, &
                 5.4443793898855104e-05_dp, &
                 0.00012434900771925045_dp, &
                 0.0002702071667839422_dp, &
                 0.0005630773934606482_dp, &
                 0.001132097168889575_dp, &
                 0.0022064607569192898_dp, &
                 0.004184511435918261_dp, &
                 0.007745737975074526_dp, &
                 0.014029823575166972_dp, &
                 0.024919304936911416_dp, &
                 0.04348066531140458_dp, &
                 0.07464540686704813_dp, &
                 0.12625156513534663_dp, &
                 0.2106228965724005_dp, &
                 0.34694729093744464_dp, &
                 0.5648480693819434_dp, &
                 0.9097754411882123_dp, &
                 1.4513508415022998_dp, &
                 2.297177622667234_dp, &
                 3.619275978541861_dp, &
                 5.719806682040045_dp, &
                 9.28540894538464_dp, &
                 8.026935386394373e-08_dp, &
                 3.431660125547769e-07_dp, &
                 1.1507249938591508e-06_dp, &
                 3.3255150480524145e-06_dp, &
                 8.631544858985621e-06_dp, &
                 2.0673889190163757e-05_dp, &
                 4.650257994599218e-05_dp, &
                 9.941521329490123e-05_dp, &
                 0.0002037413884778986_dp, &
                 0.00040284592225487954_dp, &
                 0.0007722776557802875_dp, &
                 0.0014410293829550617_dp, &
                 0.0026254207291881958_dp, &
                 0.0046823958218659345_dp, &
                 0.008192369379080482_dp, &
                 0.014086602667824237_dp, &
                 0.023841088169455484_dp, &
                 0.039769020002557186_dp, &
                 0.06545851893082533_dp, &
                 0.1064240118543133_dp, &
                 0.1710745747479101_dp, &
                 0.2721680114354786_dp, &
                 0.42907347105223687_dp, &
                 0.671615155080402_dp, &
                 1.0478076752969256_dp, &
                 1.6438903234547084_dp, &
                 2.6559588426369642_dp, &
                 4.821957836337722_dp/)
      ELSE
         aw(:) = (/ &
                 7.2246193648183315e-09_dp, &
                 7.947805205434051e-08_dp, &
                 4.2064242178143404e-07_dp, &
                 1.6235298675242196e-06_dp, &
                 5.1991703168705275e-06_dp, &
                 1.4668546651111404e-05_dp, &
                 3.7705424076918065e-05_dp, &
                 9.01796708136201e-05_dp, &
                 0.0002035415234868656_dp, &
                 0.0004379390624442907_dp, &
                 0.0009049701100068852_dp, &
                 0.0018063291005268592_dp, &
                 0.0034982504743316482_dp, &
                 0.006597202794116675_dp, &
                 0.012150671665557649_dp, &
                 0.021909404675932484_dp, &
                 0.03875606413936719_dp, &
                 0.06737255773930659_dp, &
                 0.11526829896699824_dp, &
                 0.1943501837935314_dp, &
                 0.3233020082512726_dp, &
                 0.5311764928093146_dp, &
                 0.8628395371729328_dp, &
                 1.3874103242942566_dp, &
                 2.212207252365476_dp, &
                 3.5093118591573305_dp, &
                 5.581272308767191_dp, &
                 9.114780312141413_dp, &
                 2.3095697901478273e-08_dp, &
                 1.5174284298730647e-07_dp, &
                 6.211817019789001e-07_dp, &
                 2.0193446034719265e-06_dp, &
                 5.684986670359784e-06_dp, &
                 1.4468629362219993e-05_dp, &
                 3.413251437719442e-05_dp, &
                 7.584802072328596e-05_dp, &
                 0.00016053557840876553_dp, &
                 0.0003262378591667722_dp, &
                 0.0006404016188164239_dp, &
                 0.0012199823490146536_dp, &
                 0.0022638430447923645_dp, &
                 0.0041042325427612_dp, &
                 0.007287543243404729_dp, &
                 0.012699528275383098_dp, &
                 0.021757390677572788_dp, &
                 0.03670161910421734_dp, &
                 0.061035614898498174_dp, &
                 0.10018369236083766_dp, &
                 0.16247380492926894_dp, &
                 0.26061721476056016_dp, &
                 0.41400956019249524_dp, &
                 0.6526199011491635_dp, &
                 1.0247530046294373_dp, &
                 1.617029280212683_dp, &
                 2.625732716141634_dp, &
                 4.78747600908045_dp/)
      END IF
   END SUBROUTINE

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param E_range ...
!> \param aw ...
!> \param E_ratio ...
! **************************************************************************************************
   PURE SUBROUTINE get_coeff_30(k, E_range, aw, E_ratio)
      INTEGER, INTENT(IN)                                :: k
      REAL(KIND=dp), INTENT(IN)                          :: E_range
      REAL(KIND=dp), DIMENSION(2*k), INTENT(OUT)         :: aw
      REAL(KIND=dp), INTENT(OUT)                         :: E_ratio

      E_ratio = 1.0_dp

      IF (E_range < 2906.0_dp) THEN
         E_ratio = 2906.0_dp/E_range
         aw(:) = (/ &
                 0.00010719782131833758_dp, &
                 0.0005720967766785331_dp, &
                 0.001439669174001423_dp, &
                 0.002769849121659816_dp, &
                 0.004664000049778321_dp, &
                 0.007283241780546586_dp, &
                 0.010873000028774003_dp, &
                 0.015792629918665993_dp, &
                 0.02255038608921401_dp, &
                 0.031847782299297675_dp, &
                 0.04463988143266245_dp, &
                 0.06221821662158963_dp, &
                 0.08632310206096853_dp, &
                 0.11929354905847961_dp, &
                 0.16426574047985984_dp, &
                 0.2254347024266385_dp, &
                 0.30839852293765396_dp, &
                 0.42061064113062563_dp, &
                 0.5719741696129382_dp, &
                 0.7756243966360713_dp, &
                 1.0489645297600516_dp, &
                 1.415051852321483_dp, &
                 1.9044914330966105_dp, &
                 2.558117126903484_dp, &
                 3.4310117702125913_dp, &
                 4.5990752550643395_dp, &
                 6.171106184489477_dp, &
                 8.314824292369494_dp, &
                 11.326871889957468_dp, &
                 15.91048796379593_dp, &
                 0.0002759212187476099_dp, &
                 0.0006589920695996667_dp, &
                 0.001085937188673809_dp, &
                 0.0015910122106733914_dp, &
                 0.0022237295869651035_dp, &
                 0.003055163774041633_dp, &
                 0.004183672861951183_dp, &
                 0.005739573533927359_dp, &
                 0.007891868776677179_dp, &
                 0.010860734164881057_dp, &
                 0.014936866838559482_dp, &
                 0.020507383044696917_dp, &
                 0.02808894961545452_dp, &
                 0.038370325842103176_dp, &
                 0.052267550952526234_dp, &
                 0.07099592893538823_dp, &
                 0.09616415398590554_dp, &
                 0.1298977096485147_dp, &
                 0.17500152116499199_dp, &
                 0.235176717264172_dp, &
                 0.3153154828400979_dp, &
                 0.4219165107665381_dp, &
                 0.5637038218742615_dp, &
                 0.7526245591453196_dp, &
                 1.005629375577748_dp, &
                 1.3482480324070423_dp, &
                 1.8228149068796895_dp, &
                 2.510939848106289_dp, &
                 3.6132439344998795_dp, &
                 5.9228966909439205_dp/)
      ELSE IF (E_range < 3236.0_dp) THEN
         aw(:) = (/ &
                 9.76676051737233e-05_dp, &
                 0.000521439839796676_dp, &
                 0.0013131673080453233_dp, &
                 0.0025293903535565823_dp, &
                 0.0042661250059111795_dp, &
                 0.0066765801036800295_dp, &
                 0.009994783165295324_dp, &
                 0.014564000812497294_dp, &
                 0.020870492553693578_dp, &
                 0.02958685422269625_dp, &
                 0.04163138568046643_dp, &
                 0.058249942771901375_dp, &
                 0.08112692839971898_dp, &
                 0.11253369957795559_dp, &
                 0.15552549205786806_dp, &
                 0.2142017013001525_dp, &
                 0.29404913356355256_dp, &
                 0.4023941070276559_dp, &
                 0.5489978387767454_dp, &
                 0.7468418586806641_dp, &
                 1.0131691905482183_dp, &
                 1.3708791149003934_dp, &
                 1.8504330035916199_dp, &
                 2.4925505474173923_dp, &
                 3.3522463174174093_dp, &
                 4.505410618668513_dp, &
                 6.060890691327169_dp, &
                 8.186478838726869_dp, &
                 11.178786177384513_dp, &
                 15.74038405881577_dp, &
                 0.00025141352623809566_dp, &
                 0.0006009323005964832_dp, &
                 0.0009917485369505094_dp, &
                 0.0014564055840763494_dp, &
                 0.002042200574311218_dp, &
                 0.0028172406584227633_dp, &
                 0.003875825130607345_dp, &
                 0.005342947480157929_dp, &
                 0.00738116098048549_dp, &
                 0.010203263291670922_dp, &
                 0.014091611662835478_dp, &
                 0.019423788992341998_dp, &
                 0.026705494627716198_dp, &
                 0.03661296179988794_dp, &
                 0.05004819563708869_dp, &
                 0.06821123613467164_dp, &
                 0.09269486715309114_dp, &
                 0.12560901090026905_dp, &
                 0.16974491105273157_dp, &
                 0.22879407342136257_dp, &
                 0.30764598759629597_dp, &
                 0.4128070021079015_dp, &
                 0.5530226357541534_dp, &
                 0.7402778334602197_dp, &
                 0.9915767219176012_dp, &
                 1.332510891997349_dp, &
                 1.8054641559432192_dp, &
                 2.49203146209568_dp, &
                 3.5925935804922893_dp, &
                 5.89893592679766_dp/)
      ELSE IF (E_range < 3810.0_dp) THEN
         aw(:) = (/ &
                 8.47819458078811e-05_dp, &
                 0.00045292247232938765_dp, &
                 0.0011419488590203569_dp, &
                 0.0022036034544386665_dp, &
                 0.0037263081029974285_dp, &
                 0.005851972143509195_dp, &
                 0.008798246612654406_dp, &
                 0.012885194918659285_dp, &
                 0.018567320347579703_dp, &
                 0.026475458982841895_dp, &
                 0.03747475660944132_dp, &
                 0.05274484615883288_dp, &
                 0.07388876569982615_dp, &
                 0.103079000554941_dp, &
                 0.14325196624460337_dp, &
                 0.19836605747691327_dp, &
                 0.27374326191660053_dp, &
                 0.3765207493043709_dp, &
                 0.5162475872690495_dp, &
                 0.7056742360429414_dp, &
                 0.9618016216492792_dp, &
                 1.3072886416997354_dp, &
                 1.772376230882795_dp, &
                 2.39760681930879_dp, &
                 3.237886311434963_dp, &
                 4.369085529708981_dp, &
                 5.900122672480505_dp, &
                 7.998902446784312_dp, &
                 10.962002095892634_dp, &
                 15.491026703179935_dp, &
                 0.00021827426323305446_dp, &
                 0.0005223668404015434_dp, &
                 0.0008641224881204865_dp, &
                 0.0012736436118541284_dp, &
                 0.001795042273807457_dp, &
                 0.0024921178119473955_dp, &
                 0.00345325315447348_dp, &
                 0.004795711504484412_dp, &
                 0.006672726568911133_dp, &
                 0.009286420295701826_dp, &
                 0.012906984705977642_dp, &
                 0.017897959998054307_dp, &
                 0.02474875419566456_dp, &
                 0.0341168643096291_dp, &
                 0.046883167956700916_dp, &
                 0.06422456826044022_dp, &
                 0.08770953467687731_dp, &
                 0.11942394754852313_dp, &
                 0.16213754631449626_dp, &
                 0.21952613567742027_dp, &
                 0.2964736644552558_dp, &
                 0.39949638395652903_dp, &
                 0.5373706722452656_dp, &
                 0.7221373998987525_dp, &
                 0.9708812677144617_dp, &
                 1.309288667276434_dp, &
                 1.7798217912856973_dp, &
                 2.464060270734846_dp, &
                 3.5620371023687465_dp, &
                 5.8635023001897295_dp/)
      ELSE IF (E_range < 4405.0_dp) THEN
         aw(:) = (/ &
                 7.475415175359787e-05_dp, &
                 0.0003995782940364109_dp, &
                 0.0010085401523130205_dp, &
                 0.0019494530926649103_dp, &
                 0.003304487334168418_dp, &
                 0.005206188391856581_dp, &
                 0.007858547072014874_dp, &
                 0.011562196417307553_dp, &
                 0.01674502130839572_dp, &
                 0.02400284151062447_dp, &
                 0.03415615402045192_dp, &
                 0.04832877588615427_dp, &
                 0.06805486346691048_dp, &
                 0.09542279350363236_dp, &
                 0.13326739942725138_dp, &
                 0.18542592486136436_dp, &
                 0.25707802147355546_dp, &
                 0.3551966628680496_dp, &
                 0.48914575629830964_dp, &
                 0.6714729217979144_dp, &
                 0.9189652081581203_dp, &
                 1.2540675821639358_dp, &
                 1.706822983665265_dp, &
                 2.3176124905704714_dp, &
                 3.1412403932005213_dp, &
                 4.253555048851896_dp, &
                 5.763534315696107_dp, &
                 7.839184077942836_dp, &
                 10.777063260156016_dp, &
                 15.27796168145106_dp, &
                 0.00019248226458298544_dp, &
                 0.0004611671410305988_dp, &
                 0.0007645470771240003_dp, &
                 0.0011307064674945877_dp, &
                 0.001601097912084883_dp, &
                 0.002235885143791768_dp, &
                 0.003118439955104845_dp, &
                 0.00435950977219228_dp, &
                 0.006104510985566756_dp, &
                 0.008546583020796826_dp, &
                 0.011945581030119579_dp, &
                 0.016652984461283625_dp, &
                 0.02314408570724871_dp, &
                 0.03206002850298206_dp, &
                 0.044263120398424796_dp, &
                 0.06090977430142275_dp, &
                 0.08354673398267808_dp, &
                 0.11423814381228614_dp, &
                 0.15573394602932067_dp, &
                 0.2116949419787466_dp, &
                 0.28699873776286255_dp, &
                 0.38816866941911854_dp, &
                 0.5240067210209787_dp, &
                 0.7066019007188943_dp, &
                 0.9531097377337013_dp, &
                 1.2893015561438181_dp, &
                 1.7577121859945424_dp, &
                 2.439914418694296_dp, &
                 3.5356481389173076_dp, &
                 5.832916244207001_dp/)
      ELSE IF (E_range < 5400.0_dp) THEN
         aw(:) = (/ &
                 6.263685476987877e-05_dp, &
                 0.0003350864107617077_dp, &
                 0.0008471021406263553_dp, &
                 0.0016414732496533968_dp, &
                 0.0027923267352349805_dp, &
                 0.004420069092229887_dp, &
                 0.006710856247430801_dp, &
                 0.009939856329159545_dp, &
                 0.014500042255919464_dp, &
                 0.02094127936473627_dp, &
                 0.030025328730095918_dp, &
                 0.042802276529506474_dp, &
                 0.06071482338334902_dp, &
                 0.08573906273600417_dp, &
                 0.1205734666631485_dp, &
                 0.168891746293846_dp, &
                 0.23568035070007362_dp, &
                 0.3276881055094081_dp, &
                 0.4540246498488577_dp, &
                 0.6269572963379791_dp, &
                 0.8629754834557104_dp, &
                 1.1842241235652444_dp, &
                 1.6204657246854692_dp, &
                 2.2118496417292866_dp, &
                 3.0130302596964915_dp, &
                 4.099816741357715_dp, &
                 5.581265514191353_dp, &
                 7.625526452879397_dp, &
                 10.52915117245689_dp, &
                 14.99185357007055_dp, &
                 0.00016131234676709978_dp, &
                 0.0003871323803883455_dp, &
                 0.0006438639425224474_dp, &
                 0.0009569827684469891_dp, &
                 0.0013644640382973136_dp, &
                 0.0019216628235682472_dp, &
                 0.002705305661659533_dp, &
                 0.003817552287654078_dp, &
                 0.005393559577915718_dp, &
                 0.007614615789728071_dp, &
                 0.0107267793784036_dp, &
                 0.01506524759897237_dp, &
                 0.02108608211163133_dp, &
                 0.029407976403737046_dp, &
                 0.040867556214958384_dp, &
                 0.05659266977268607_dp, &
                 0.07809948510740791_dp, &
                 0.10742117687653448_dp, &
                 0.1472789549319935_dp, &
                 0.20131104861420793_dp, &
                 0.27438405593923587_dp, &
                 0.3730286538370253_dp, &
                 0.50607994537212_dp, &
                 0.6856919919465865_dp, &
                 0.9291183488974969_dp, &
                 1.2622503775094018_dp, &
                 1.7277292939880777_dp, &
                 2.407128771305674_dp, &
                 3.4998023815666013_dp, &
                 5.791398263742947_dp/)
      ELSE IF (E_range < 6800.0_dp) THEN
         aw(:) = (/ &
                 5.1260796624319686e-05_dp, &
                 0.00027449998212479283_dp, &
                 0.0006952559402364613_dp, &
                 0.0013512581469068891_dp, &
                 0.0023084695955214183_dp, &
                 0.0036748571064951344_dp, &
                 0.005618146035815099_dp, &
                 0.008387084892047715_dp, &
                 0.012338401496919359_dp, &
                 0.01797419704821914_dp, &
                 0.025994966614220735_dp, &
                 0.03737346294738748_dp, &
                 0.05345583818085436_dp, &
                 0.0760988446106003_dp, &
                 0.10785503448538585_dp, &
                 0.15222193676863333_dp, &
                 0.2139764359272559_dp, &
                 0.29962253812775563_dp, &
                 0.4179901608236006_dp, &
                 0.5810358869798488_dp, &
                 0.8049164836351872_dp, &
                 1.11143825413834_dp, &
                 1.5300435290071273_dp, &
                 2.1006127473331047_dp, &
                 2.8776209315806276_dp, &
                 3.93682226153259_dp, &
                 5.387354342884511_dp, &
                 7.397531351751853_dp, &
                 10.263919356557402_dp, &
                 14.685102820756857_dp, &
                 0.00013204473011131683_dp, &
                 0.0003175248324260899_dp, &
                 0.0005301211290638112_dp, &
                 0.000792646090309341_dp, &
                 0.0011394726365449568_dp, &
                 0.0016209075495792747_dp, &
                 0.0023066917116162182_dp, &
                 0.0032900298122596973_dp, &
                 0.004695415042196518_dp, &
                 0.0066917090579473775_dp, &
                 0.009510290459434545_dp, &
                 0.013468802311717997_dp, &
                 0.019002357781084556_dp, &
                 0.02670497738030662_dp, &
                 0.037384817040713114_dp, &
                 0.05213776033026327_dp, &
                 0.072445378072951_dp, &
                 0.10030529745315488_dp, &
                 0.13840505606579703_dp, &
                 0.19035539860923475_dp, &
                 0.2610076849435031_dp, &
                 0.3568973694825793_dp, &
                 0.48689310015691706_dp, &
                 0.6632189222745253_dp, &
                 0.9032371034193576_dp, &
                 1.2329755271865048_dp, &
                 1.695201274879912_dp, &
                 2.3715025399546223_dp, &
                 3.4608285109987857_dp, &
                 5.746290251951392_dp/)
      ELSE IF (E_range < 8400.0_dp) THEN
         aw(:) = (/ &
                 4.264827171158114e-05_dp, &
                 0.00022859921129368268_dp, &
                 0.0005800651655637003_dp, &
                 0.0011306618266057897_dp, &
                 0.001939659276932296_dp, &
                 0.0031047255791507274_dp, &
                 0.004778205945319036_dp, &
                 0.007186731964418128_dp, &
                 0.010656673525706605_dp, &
                 0.01565005735652159_dp, &
                 0.02281575734702256_dp, &
                 0.03306097534037487_dp, &
                 0.04764947470167637_dp, &
                 0.0683354702649706_dp, &
                 0.09754528626464402_dp, &
                 0.13862301432535307_dp, &
                 0.19616179200890152_dp, &
                 0.27644949046799433_dp, &
                 0.3880673249123362_dp, &
                 0.5426935306072133_dp, &
                 0.7561844283103211_dp, &
                 1.0500376608384598_dp, &
                 1.4534003880410857_dp, &
                 2.005901250325418_dp, &
                 2.7618425195091647_dp, &
                 3.796918364528742_dp, &
                 5.220333260426152_dp, &
                 7.200552079960992_dp, &
                 10.034172866297915_dp, &
                 14.41882124837872_dp, &
                 0.0001098833579123983_dp, &
                 0.0002647437855875365_dp, &
                 0.0004436472669265954_dp, &
                 0.0006672101026873376_dp, &
                 0.0009667895306084336_dp, &
                 0.0013884135997529708_dp, &
                 0.0019959070037335724_dp, &
                 0.0028749526284055213_dp, &
                 0.004141093118158889_dp, &
                 0.005952643864305207_dp, &
                 0.008528340123693306_dp, &
                 0.01217052007837369_dp, &
                 0.017295858304956554_dp, &
                 0.024476476682491477_dp, &
                 0.03449505017711748_dp, &
                 0.04841859097574648_dp, &
                 0.06769708781585487_dp, &
                 0.09429528482789237_dp, &
                 0.13086897935334732_dp, &
                 0.1810021324793278_dp, &
                 0.24952974948475865_dp, &
                 0.3429885311920854_dp, &
                 0.4702742651357602_dp, &
                 0.6436716531183656_dp, &
                 0.880640225748484_dp, &
                 1.2073331130934455_dp, &
                 1.6666371474517014_dp, &
                 2.3401654531537157_dp, &
                 3.4265253480645885_dp, &
                 5.706615197039722_dp/)
      ELSE IF (E_range < 10000.0_dp) THEN
         aw(:) = (/ &
                 3.6638445936143444e-05_dp, &
                 0.00019654898584533858_dp, &
                 0.0004995365456010348_dp, &
                 0.0009761629469610179_dp, &
                 0.001680692380508944_dp, &
                 0.0027030234477272513_dp, &
                 0.004183824314963024_dp, &
                 0.006332898868769434_dp, &
                 0.009453508422633638_dp, &
                 0.013977114132271049_dp, &
                 0.020513053005428484_dp, &
                 0.02991804681839935_dp, &
                 0.043392038467264346_dp, &
                 0.06260934488416951_dp, &
                 0.089897358898568_dp, &
                 0.12847922117893293_dp, &
                 0.18280238941596594_dp, &
                 0.25898237331386087_dp, &
                 0.36540085871444006_dp, &
                 0.5135113514430796_dp, &
                 0.718925939367594_dp, &
                 1.0028894169830442_dp, &
                 1.3943048957706679_dp, &
                 1.9325902027814568_dp, &
                 2.6718997828791795_dp, &
                 3.6878719059499683_dp, &
                 5.089761008341865_dp, &
                 7.046156150294659_dp, &
                 9.853694451194254_dp, &
                 14.209263474504755_dp, &
                 9.441684068693651e-05_dp, &
                 0.00022786014262271372_dp, &
                 0.00038307351711976584_dp, &
                 0.0005790223101422384_dp, &
                 0.0008447661561374952_dp, &
                 0.0012230417856923205_dp, &
                 0.0017731334822002557_dp, &
                 0.002574990545123123_dp, &
                 0.003737321923840727_dp, &
                 0.005410302324390409_dp, &
                 0.007802786243657_dp, &
                 0.011205034742082835_dp, &
                 0.016019061597614896_dp, &
                 0.02279945700439318_dp, &
                 0.032308353668577205_dp, &
                 0.04558931015753348_dp, &
                 0.06406643328220389_dp, &
                 0.08967722561429306_dp, &
                 0.1250507942222136_dp, &
                 0.17374800817672645_dp, &
                 0.24058882427995695_dp, &
                 0.3321088613980677_dp, &
                 0.4572237934265981_dp, &
                 0.6282658627647734_dp, &
                 0.8627730246821379_dp, &
                 1.187001575527606_dp, &
                 1.6439399241381616_dp, &
                 2.3152296444802993_dp, &
                 3.3992160109205325_dp, &
                 5.6750508010285134_dp/)
      ELSE IF (E_range < 12000.0_dp) THEN
         aw(:) = (/ &
                 3.125352048295285e-05_dp, &
                 0.00016781351172929327_dp, &
                 0.00042725280780375096_dp, &
                 0.0008372368297287864_dp, &
                 0.0014472482996344947_dp, &
                 0.0023397061489668085_dp, &
                 0.00364397706459525_dp, &
                 0.005553554215846271_dp, &
                 0.008349288769114196_dp, &
                 0.012432933353972381_dp, &
                 0.018375235671074245_dp, &
                 0.0269833920064158_dp, &
                 0.03939439104594155_dp, &
                 0.057203310896803944_dp, &
                 0.08263890933961444_dp, &
                 0.118803108858221_dp, &
                 0.16999660677619183_dp, &
                 0.24216036024367976_dp, &
                 0.3434728927615001_dp, &
                 0.4851575786991668_dp, &
                 0.6825748514741242_dp, &
                 0.956707159367906_dp, &
                 1.3362022902301018_dp, &
                 1.8602550523674573_dp, &
                 2.582860494442485_dp, &
                 3.5795923551771076_dp, &
                 4.959752013207221_dp, &
                 6.892057403111867_dp, &
                 9.673196873736643_dp, &
                 13.999332658018544_dp, &
                 8.055654719138439e-05_dp, &
                 0.0001947658525515697_dp, &
                 0.00032859686532794565_dp, &
                 0.0004994306825817649_dp, &
                 0.0007340947583837473_dp, &
                 0.0010721035282296792_dp, &
                 0.001568310846736925_dp, &
                 0.002297102568888374_dp, &
                 0.0033605346972129828_dp, &
                 0.00490077013024238_dp, &
                 0.0071168327918336724_dp, &
                 0.010286868707377024_dp, &
                 0.01479808821232481_dp, &
                 0.021187271898748812_dp, &
                 0.03019555503700218_dp, &
                 0.042842362400043486_dp, &
                 0.06052496436664667_dp, &
                 0.08515234916902564_dp, &
                 0.11932532276255173_dp, &
                 0.16657974696296404_dp, &
                 0.23171843775477596_dp, &
                 0.3212739304330216_dp, &
                 0.4441803236632917_dp, &
                 0.6128171239515162_dp, &
                 0.8448023886795951_dp, &
                 1.1664998500858539_dp, &
                 1.6210063652060114_dp, &
                 2.2900001311037035_dp, &
                 3.371570229987856_dp, &
                 5.643112922874634_dp/)
      ELSE IF (E_range < 15000.0_dp) THEN
         aw(:) = (/ &
                 2.572460674430881e-05_dp, &
                 0.00013828809639808773_dp, &
                 0.0003528803185321503_dp, &
                 0.0006939963784805861_dp, &
                 0.0012058419667301263_dp, &
                 0.00196250811153749_dp, &
                 0.003080710120197529_dp, &
                 0.004735659001322314_dp, &
                 0.007183065998906602_dp, &
                 0.010791253848864667_dp, &
                 0.016087325873200328_dp, &
                 0.02382213056319325_dp, &
                 0.03506059129972535_dp, &
                 0.051306532736392435_dp, &
                 0.07467445268948096_dp, &
                 0.10812503757927873_dp, &
                 0.15578699671299615_dp, &
                 0.2233955307214805_dp, &
                 0.31888824542436384_dp, &
                 0.45321390622010965_dp, &
                 0.6414306299530386_dp, &
                 0.9042033223791973_dp, &
                 1.2698679478183343_dp, &
                 1.7773436933847355_dp, &
                 2.4804246798636003_dp, &
                 3.4545978455659587_dp, &
                 4.80921488681201_dp, &
                 6.713149815396915_dp, &
                 9.463167209838238_dp, &
                 13.75460415331797_dp, &
                 6.632324623333035e-05_dp, &
                 0.00016073115506706547_dp, &
                 0.0002724186286744679_dp, &
                 0.00041700914328903177_dp, &
                 0.0006188187442380069_dp, &
                 0.000913709771883424_dp, &
                 0.0013515392555132155_dp, &
                 0.002000450746020119_dp, &
                 0.0029549917404197673_dp, &
                 0.004348167160940399_dp, &
                 0.006367634264735562_dp, &
                 0.009277411142785488_dp, &
                 0.013447324345434384_dp, &
                 0.01939310256171694_dp, &
                 0.027830892296757826_dp, &
                 0.03975118351588367_dp, &
                 0.056518799068157294_dp, &
                 0.08000790205760616_dp, &
                 0.11278427119956595_dp, &
                 0.15835215912014322_dp, &
                 0.2214916578911036_dp, &
                 0.3087289982044895_dp, &
                 0.42901765162073763_dp, &
                 0.5947915793660636_dp, &
                 0.8237640916060611_dp, &
                 1.142429482918768_dp, &
                 1.5940199744065582_dp, &
                 2.2602675888442345_dp, &
                 3.3389719364130626_dp, &
                 5.605477212236766_dp/)
      ELSE IF (E_range < 20000.0_dp) THEN
         aw(:) = (/ &
                 2.0010353507234397e-05_dp, &
                 0.0001077433422458143_dp, &
                 0.00027580023006758454_dp, &
                 0.0005451258515021734_dp, &
                 0.0009539529098224693_dp, &
                 0.0015668441726082855_dp, &
                 0.0024859562396115408_dp, &
                 0.003865438814677545_dp, &
                 0.005931997279200694_dp, &
                 0.009015206545406032_dp, &
                 0.013591237471033118_dp, &
                 0.020344680291656994_dp, &
                 0.03025506904355767_dp, &
                 0.04471729454363523_dp, &
                 0.0657084625235162_dp, &
                 0.09601821543882148_dp, &
                 0.13956549956219141_dp, &
                 0.2018327748807436_dp, &
                 0.2904595479070126_dp, &
                 0.41605219017177664_dp, &
                 0.5932887745998796_dp, &
                 0.8424313672051914_dp, &
                 1.1914160744119529_dp, &
                 1.678803642976038_dp, &
                 2.3581208348346263_dp, &
                 3.3047307785320212_dp, &
                 4.6280385004171025_dp, &
                 6.497115386386612_dp, &
                 9.208843054933286_dp, &
                 13.457588010635305_dp, &
                 5.1609539489171625e-05_dp, &
                 0.00012547938250585046_dp, &
                 0.00021401951207906163_dp, &
                 0.0003308508348857692_dp, &
                 0.0004973777574979467_dp, &
                 0.0007451991801870302_dp, &
                 0.0011183825299403462_dp, &
                 0.0016778698024847744_dp, &
                 0.0025094583225573333_dp, &
                 0.0037353071207149595_dp, &
                 0.005529432603001048_dp, &
                 0.008138731296629963_dp, &
                 0.011911786975259289_dp, &
                 0.0173383968709882_dp, &
                 0.02510367274551181_dp, &
                 0.03616185208550007_dp, &
                 0.05183670042056887_dp, &
                 0.07395778747131394_dp, &
                 0.10504533244757883_dp, &
                 0.14856147943385148_dp, &
                 0.20925449434054086_dp, &
                 0.2936386766166653_dp, &
                 0.41068755123213285_dp, &
                 0.5728998438821491_dp, &
                 0.7981069410955339_dp, &
                 1.1129693992413612_dp, &
                 1.5608972768887142_dp, &
                 2.223705100788062_dp, &
                 3.298855550400102_dp, &
                 5.559194240832182_dp/)
      ELSE IF (E_range < 28000.0_dp) THEN
         aw(:) = (/ &
                 1.4913158562437485e-05_dp, &
                 8.046249863674396e-05_dp, &
                 0.00020679269836799646_dp, &
                 0.00041135606175085387_dp, &
                 0.000726429457279576_dp, &
                 0.001206954515218603_dp, &
                 0.0019403041016435166_dp, &
                 0.0030592205520702234_dp, &
                 0.004760844972277948_dp, &
                 0.007334997531745175_dp, &
                 0.011205132165430936_dp, &
                 0.016986616421301493_dp, &
                 0.025568952229820977_dp, &
                 0.03823114994605764_dp, &
                 0.056802896237992014_dp, &
                 0.08388874591064059_dp, &
                 0.12317874197383723_dp, &
                 0.17987720132408636_dp, &
                 0.26129273365588884_dp, &
                 0.3776482327688284_dp, &
                 0.5431920469707082_dp, &
                 0.7777259064649447_dp, &
                 1.1087233079248426_dp, &
                 1.5743236370665366_dp, &
                 2.2277313150621567_dp, &
                 3.144149304477041_dp, &
                 4.433028932012796_dp, &
                 6.263665410565493_dp, &
                 8.933099495687644_dp, &
                 13.134684015624826_dp, &
                 3.848089108885685e-05_dp, &
                 9.39454195069688e-05_dp, &
                 0.00016153027576924122_dp, &
                 0.0002528432683507074_dp, &
                 0.0003863006748200446_dp, &
                 0.000589103857593431_dp, &
                 0.0008994092337673156_dp, &
                 0.0013708168552981645_dp, &
                 0.0020800635852347526_dp, &
                 0.0031378613819817206_dp, &
                 0.004703620407911122_dp, &
                 0.007005705166384156_dp, &
                 0.010369489117192241_dp, &
                 0.015256172448531693_dp, &
                 0.02231631000951314_dp, &
                 0.03246335096484727_dp, &
                 0.04697433098325742_dp, &
                 0.06762737851039459_dp, &
                 0.09688924910721004_dp, &
                 0.13817140362709024_dp, &
                 0.19618184162864452_dp, &
                 0.2774160356134996_dp, &
                 0.39086437274546876_dp, &
                 0.5490937144681445_dp, &
                 0.7700664939132916_dp, &
                 1.0806338759241187_dp, &
                 1.5244170903885788_dp, &
                 2.183343539374224_dp, &
                 3.254529601017961_dp, &
                 5.508095437648589_dp/)
      ELSE IF (E_range < 38000.0_dp) THEN
         aw(:) = (/ &
                 1.1418872198921082e-05_dp, &
                 6.173413750253069e-05_dp, &
                 0.00015929220328335313_dp, &
                 0.00031889503854739523_dp, &
                 0.0005682342649483396_dp, &
                 0.0009547547779672074_dp, &
                 0.0015542608919948567_dp, &
                 0.0024827272879236525_dp, &
                 0.003914044995780404_dp, &
                 0.006106526724782397_dp, &
                 0.009441451620849054_dp, &
                 0.01447827421468276_dp, &
                 0.022033096235727034_dp, &
                 0.03328960553532699_dp, &
                 0.04995516756713029_dp, &
                 0.07447945346316431_dp, &
                 0.11035934551634245_dp, &
                 0.16256247169247554_dp, &
                 0.23811344985212937_dp, &
                 0.3469031346310937_dp, &
                 0.5028042883523934_dp, &
                 0.7252121362263632_dp, &
                 1.0411867545570728_dp, &
                 1.4884852069632906_dp, &
                 2.120012518561155_dp, &
                 3.0108136851550755_dp, &
                 4.2703677397197355_dp, &
                 6.068165404997884_dp, &
                 8.701408700808228_dp, &
                 12.8626331750244_dp, &
                 2.947787098787719e-05_dp, &
                 7.225954134028555e-05_dp, &
                 0.0001252402865134724_dp, &
                 0.00019846473664424758_dp, &
                 0.00030798161384563543_dp, &
                 0.0004775038459907116_dp, &
                 0.0007405481153276563_dp, &
                 0.0011449321771295313_dp, &
                 0.0017601237436588445_dp, &
                 0.0026874815648166833_dp, &
                 0.004074324357547685_dp, &
                 0.0061335185303593494_dp, &
                 0.009170846038462778_dp, &
                 0.013623145317805922_dp, &
                 0.020111251332522298_dp, &
                 0.029513178507526473_dp, &
                 0.043064913823154624_dp, &
                 0.062498815014491525_dp, &
                 0.09023329634576005_dp, &
                 0.12963292767495377_dp, &
                 0.18536685282870904_dp, &
                 0.26390946533406945_dp, &
                 0.3742607782294386_dp, &
                 0.5290426355523791_dp, &
                 0.7463296721416969_dp, &
                 1.0531417380773798_dp, &
                 1.493293292834725_dp, &
                 2.1488272848196677_dp, &
                 3.216586556758033_dp, &
                 5.464388958502095_dp/)
      ELSE IF (E_range < 50000.0_dp) THEN
         aw(:) = (/ &
                 8.982546693806235e-06_dp, &
                 4.865818678309805e-05_dp, &
                 0.00012604148335636865_dp, &
                 0.00025390861901678955_dp, &
                 0.0004563995854196066_dp, &
                 0.0007750952045840026_dp, &
                 0.0012767265489376275_dp, &
                 0.002064101925548432_dp, &
                 0.0032927700122513484_dp, &
                 0.005195994959981289_dp, &
                 0.008121235122746024_dp, &
                 0.012582721029292343_dp, &
                 0.019336695756878207_dp, &
                 0.029488494866509965_dp, &
                 0.044644170175709534_dp, &
                 0.06712415308115761_dp, &
                 0.10026296333569998_dp, &
                 0.14882782019535817_dp, &
                 0.2196010954609523_dp, &
                 0.3221882442730784_dp, &
                 0.47013658664261176_dp, &
                 0.6824859983381489_dp, &
                 0.9859314820504801_dp, &
                 1.417888163895422_dp, &
                 2.030988118342932_dp, &
                 2.900125529064684_dp, &
                 4.134793295219112_dp, &
                 5.904650857570968_dp, &
                 8.50705581410213_dp, &
                 12.633884453832914_dp, &
                 2.3198709666506916e-05_dp, &
                 5.709307522017352e-05_dp, &
                 9.972792415428492e-05_dp, &
                 0.0001599273928986242_dp, &
                 0.0002518605559821019_dp, &
                 0.00039647689416403176_dp, &
                 0.0006236485188629505_dp, &
                 0.0009766180462365724_dp, &
                 0.0015189982582493745_dp, &
                 0.0023445004429086033_dp, &
                 0.0035904524574293063_dp, &
                 0.005456804956339633_dp, &
                 0.008232881015796438_dp, &
                 0.012334887697009843_dp, &
                 0.01835828332102448_dp, &
                 0.027150564163334993_dp, &
                 0.03991202944665125_dp, &
                 0.05833481297012722_dp, &
                 0.08479428042701978_dp, &
                 0.12261246982859136_dp, &
                 0.17642213365813839_dp, &
                 0.2526759210340022_dp, &
                 0.3603782883628389_dp, &
                 0.5121951678672628_dp, &
                 0.7262964913958404_dp, &
                 1.0298496814196687_dp, &
                 1.4668430564566999_dp, &
                 2.119432331068027_dp, &
                 3.1842446525056274_dp, &
                 5.427158434113688_dp/)
      ELSE IF (E_range < 64000.0_dp) THEN
         aw(:) = (/ &
                 7.238349702296008e-06_dp, &
                 3.928458373506955e-05_dp, &
                 0.00010214563688496669_dp, &
                 0.00020702207021802802_dp, &
                 0.0003752580104799391_dp, &
                 0.0006437818300993232_dp, &
                 0.001072114382774453_dp, &
                 0.0017525850080720873_dp, &
                 0.002826070350614258_dp, &
                 0.004505651346498813_dp, &
                 0.007111320652225213_dp, &
                 0.011120302372900251_dp, &
                 0.017239502228764955_dp, &
                 0.026509227663294707_dp, &
                 0.04045088318380033_dp, &
                 0.06127621597555976_dp, &
                 0.09218233054747209_dp, &
                 0.13776574927878416_dp, &
                 0.2046011860485274_dp, &
                 0.3020478300836994_dp, &
                 0.443370229666864_dp, &
                 0.6472971651730427_dp, &
                 0.9402012658200933_dp, &
                 1.3591924682162404_dp, &
                 1.956655369037565_dp, &
                 2.8073424394429862_dp, &
                 4.020750844752612_dp, &
                 5.766686098521229_dp, &
                 8.342653009498491_dp, &
                 12.439991369032668_dp, &
                 1.8702020468622055e-05_dp, &
                 4.6203219782190167e-05_dp, &
                 8.131762180666392e-05_dp, &
                 0.00013190143534228278_dp, &
                 0.00021061299324796057_dp, &
                 0.0003361903026363539_dp, &
                 0.000535605997735673_dp, &
                 0.0008484276122519987_dp, &
                 0.0013334887847408908_dp, &
                 0.0020781819161196638_dp, &
                 0.003211507934451115_dp, &
                 0.004922574199422692_dp, &
                 0.007486789650255751_dp, &
                 0.011302797934300408_dp, &
                 0.016944298667165195_dp, &
                 0.02523242357196762_dp, &
                 0.03733639323117268_dp, &
                 0.05491299477721061_dp, &
                 0.08029934786534572_dp, &
                 0.11677913471088484_dp, &
                 0.16895145657274943_dp, &
                 0.24324744526263437_dp, &
                 0.3486724489732179_dp, &
                 0.49792796808610074_dp, &
                 0.7092652162431419_dp, &
                 1.009980880755789_dp, &
                 1.4442192775687972_dp, &
                 2.094243792411222_dp, &
                 3.1565101709731045_dp, &
                 5.395251764843703_dp/)
      ELSE IF (E_range < 84000.0_dp) THEN
         aw(:) = (/ &
                 5.706429844293361e-06_dp, &
                 3.1040364829701505e-05_dp, &
                 8.107328661660111e-05_dp, &
                 0.00016550326188419503_dp, &
                 0.0003029763316477536_dp, &
                 0.0005259009041296794_dp, &
                 0.0008867852843333584_dp, &
                 0.0014677450288040894_dp, &
                 0.002395273728414361_dp, &
                 0.003862517134876226_dp, &
                 0.006162129729070569_dp, &
                 0.00973422779264423_dp, &
                 0.015235895683403678_dp, &
                 0.023641332048275565_dp, &
                 0.03638532798309582_dp, &
                 0.055567718600602306_dp, &
                 0.08424323176407787_dp, &
                 0.12683043751734444_dp, &
                 0.18968622248619452_dp, &
                 0.2819098113026998_dp, &
                 0.4164652673968472_dp, &
                 0.6117484077902107_dp, &
                 0.893783996100621_dp, &
                 1.2993493017998998_dp, &
                 1.8805551654919748_dp, &
                 2.7119921346520504_dp, &
                 3.9031533013982855_dp, &
                 5.623997981838861_dp, &
                 8.172198452849088_dp, &
                 12.238558563265647_dp, &
                 1.4751349028333799e-05_dp, &
                 3.660909835718544e-05_dp, &
                 6.501211560655282e-05_dp, &
                 0.00010687536539993863_dp, &
                 0.00017337123549154558_dp, &
                 0.00028107588099189824_dp, &
                 0.000454137569267576_dp, &
                 0.0007285000756826098_dp, &
                 0.001158211238344495_dp, &
                 0.0018242672088928367_dp, &
                 0.0028471735084972696_dp, &
                 0.004404897269066725_dp, &
                 0.006758454939132606_dp, &
                 0.01028819929336629_dp, &
                 0.01554501319618277_dp, &
                 0.023322182186959424_dp, &
                 0.03475584758118297_dp, &
                 0.05146485198022031_dp, &
                 0.07574483876932264_dp, &
                 0.11083732616584702_dp, &
                 0.16130359908425193_dp, &
                 0.23354923720333218_dp, &
                 0.3365774511982118_dp, &
                 0.48312462993088157_dp, &
                 0.6915266416555069_dp, &
                 0.9892183849743513_dp, &
                 1.420514754404278_dp, &
                 2.0678032116600904_dp, &
                 3.127373080182709_dp, &
                 5.361746811812883_dp/)
      ELSE IF (E_range < 110000.0_dp) THEN
         aw(:) = (/ &
                 4.507925303711096e-06_dp, &
                 2.4580451494530444e-05_dp, &
                 6.451263281768541e-05_dp, &
                 0.00013272058871030806_dp, &
                 0.0002455197572529811_dp, &
                 0.0004313910776655633_dp, &
                 0.0007367445062259567_dp, &
                 0.001234786889942828_dp, &
                 0.002039383465959761_dp, &
                 0.00332603532508658_dp, &
                 0.005363004530984576_dp, &
                 0.008557044500582709_dp, &
                 0.01352012174944254_dp, &
                 0.021166153619467505_dp, &
                 0.03285042829603399_dp, &
                 0.05056939685207151_dp, &
                 0.07724543673788668_dp, &
                 0.11713067199112848_dp, &
                 0.17637697527453522_dp, &
                 0.263837336282913_dp, &
                 0.39218930488595755_dp, &
                 0.5795089364205666_dp, &
                 0.8514842120807159_dp, &
                 1.2445671334250805_dp, &
                 1.8105970655011243_dp, &
                 2.6239991397280553_dp, &
                 3.7942542214271437_dp, &
                 5.491467032685958_dp, &
                 8.013480504379215_dp, &
                 12.05062051565182_dp, &
                 1.165943020924494e-05_dp, &
                 2.907711562944157e-05_dp, &
                 5.2135381292650646e-05_dp, &
                 8.692931914146652e-05_dp, &
                 0.00014332433921579002_dp, &
                 0.00023600989072825049_dp, &
                 0.0003866717964299752_dp, &
                 0.0006280479343018203_dp, &
                 0.0010098870748762492_dp, &
                 0.001607379741603303_dp, &
                 0.00253325997236095_dp, &
                 0.003955231709080428_dp, &
                 0.006120960061358173_dp, &
                 0.009393712743103142_dp, &
                 0.014302912174317341_dp, &
                 0.02161546778139668_dp, &
                 0.03243595167457546_dp, &
                 0.04834668628642002_dp, &
                 0.07160299218511915_dp, &
                 0.10540485530435402_dp, &
                 0.15427553890661685_dp, &
                 0.22459367842315606_dp, &
                 0.3253574621758251_dp, &
                 0.46933370101682803_dp, &
                 0.6749372715340488_dp, &
                 0.9697356715189667_dp, &
                 1.3982111886845372_dp, &
                 2.0428791235793997_dp, &
                 3.099885407679866_dp, &
                 5.330156088439341_dp/)
      ELSE IF (E_range < 160000.0_dp) THEN
         aw(:) = (/ &
                 3.2492504706355543e-06_dp, &
                 1.7782935938647487e-05_dp, &
                 4.702094527107548e-05_dp, &
                 9.788765130060788e-05_dp, &
                 0.00018394800566011574_dp, &
                 0.0003290226690843259_dp, &
                 0.0005722787919864911_dp, &
                 0.0009762934532250238_dp, &
                 0.0016397346725346576_dp, &
                 0.002716663917457915_dp, &
                 0.004445425742834286_dp, &
                 0.007191491449190322_dp, &
                 0.011510534331810433_dp, &
                 0.01824066242763528_dp, &
                 0.02863642038498904_dp, &
                 0.044562287024877995_dp, &
                 0.06877046943314344_dp, &
                 0.10529755457345026_dp, &
                 0.16002804271088766_dp, &
                 0.24149146592791995_dp, &
                 0.361986165199179_dp, &
                 0.5391615439925698_dp, &
                 0.7982521139849742_dp, &
                 1.1752670593405194_dp, &
                 1.7216703901842698_dp, &
                 2.511651208522495_dp, &
                 3.654660979891775_dp, &
                 5.320994285506342_dp, &
                 7.808735565003434_dp, &
                 11.80762543615471_dp, &
                 8.410837332223653e-06_dp, &
                 2.113245841372446e-05_dp, &
                 3.845111058827689e-05_dp, &
                 6.548419810172488e-05_dp, &
                 0.00011052592304298797_dp, &
                 0.00018602054544749434_dp, &
                 0.0003107143750285354_dp, &
                 0.0005134448260073493_dp, &
                 0.0008386432327700857_dp, &
                 0.0013542399463937198_dp, &
                 0.002163165180703502_dp, &
                 0.0034200707057045_dp, &
                 0.0053555078104983965_dp, &
                 0.008310672199526598_dp, &
                 0.012787041135722151_dp, &
                 0.019516894799063707_dp, &
                 0.029562998187469335_dp, &
                 0.044458853860523784_dp, &
                 0.0664052872217014_dp, &
                 0.09854535532252624_dp, &
                 0.14534905940024856_dp, &
                 0.2131554574377401_dp, &
                 0.3109515151654752_dp, &
                 0.4515398978356489_dp, &
                 0.653437304234083_dp, &
                 0.9443877723109725_dp, &
                 1.3691021780052584_dp, &
                 2.010279336423766_dp, &
                 3.063897971847719_dp, &
                 5.288820184644417_dp/)
      ELSE IF (E_range < 220000.0_dp) THEN
         aw(:) = (/ &
                 2.460537860443357e-06_dp, &
                 1.3513517677623214e-05_dp, &
                 3.598515389781885e-05_dp, &
                 7.575261086818076e-05_dp, &
                 0.0001444216294004159_dp, &
                 0.00026247903310060725_dp, &
                 0.0004639039840208787_dp, &
                 0.0008036114726657992_dp, &
                 0.001369205915452374_dp, &
                 0.0022989755449385045_dp, &
                 0.0038090333768977967_dp, &
                 0.006233886266199457_dp, &
                 0.01008661073890614_dp, &
                 0.016147469119526157_dp, &
                 0.025593516684082565_dp, &
                 0.04018693519947543_dp, &
                 0.06254703089827915_dp, &
                 0.0965408099871576_dp, &
                 0.14784085162679408_dp, &
                 0.22471839016516207_dp, &
                 0.33916659817804756_dp, &
                 0.5084886871639984_dp, &
                 0.7575478060875892_dp, &
                 1.1219864437669036_dp, &
                 1.6529529420726783_dp, &
                 2.4244318079577103_dp, &
                 3.5458393070593472_dp, &
                 5.187620119758077_dp, &
                 7.648065546110419_dp, &
                 11.61648316806262_dp, &
                 6.374120402922641e-06_dp, &
                 1.6128252577743594e-05_dp, &
                 2.9754113670763418e-05_dp, &
                 5.166430893732005e-05_dp, &
                 8.901502625196717e-05_dp, &
                 0.00015264214630912148_dp, &
                 0.0002591687585053359_dp, &
                 0.00043455383137211705_dp, &
                 0.000719240735481554_dp, &
                 0.0011756556233951615_dp, &
                 0.0018992288700902785_dp, &
                 0.003034542038483531_dp, &
                 0.004798831948386736_dp, &
                 0.007515980097531093_dp, &
                 0.011665360331427038_dp, &
                 0.017951632069288924_dp, &
                 0.027403921176393425_dp, &
                 0.041516070343442624_dp, &
                 0.06244414300326091_dp, &
                 0.0932837802259788_dp, &
                 0.13845969920620585_dp, &
                 0.20427585444540086_dp, &
                 0.2997063234922189_dp, &
                 0.43757882986144675_dp, &
                 0.6364895482287001_dp, &
                 0.9243252918542686_dp, &
                 1.3459866886820373_dp, &
                 1.9843322156142236_dp, &
                 3.035224685946777_dp, &
                 5.255903429150008_dp/)
      ELSE IF (E_range < 370000.0_dp) THEN
         aw(:) = (/ &
                 1.5637740408395342e-06_dp, &
                 8.644581438556428e-06_dp, &
                 2.3326026659580105e-05_dp, &
                 5.012244143843333e-05_dp, &
                 9.804997776691128e-05_dp, &
                 0.000183175648888439_dp, &
                 0.0003325825937550191_dp, &
                 0.0005909069030179598_dp, &
                 0.0010307262850542825_dp, &
                 0.001768639145774748_dp, &
                 0.0029898323691103764_dp, &
                 0.004985271727949821_dp, &
                 0.008207526024267482_dp, &
                 0.013353895620644004_dp, &
                 0.0214892751837372_dp, &
                 0.034226463130117915_dp, &
                 0.053989018153179254_dp, &
                 0.08439206340451581_dp, &
                 0.1307907673641504_dp, &
                 0.20106621537104283_dp, &
                 0.3067466011130968_dp, &
                 0.46460274135109153_dp, &
                 0.698920779817791_dp, &
                 1.0447664262134877_dp, &
                 1.5527831264532448_dp, &
                 2.296617666855539_dp, &
                 3.3856119528633637_dp, &
                 4.990434105875191_dp, &
                 7.4097119078577_dp, &
                 11.33215729315795_dp, &
                 4.056801137252665e-06_dp, &
                 1.0400304864325584e-05_dp, &
                 1.968307964805301e-05_dp, &
                 3.537211676997081e-05_dp, &
                 6.309758174077909e-05_dp, &
                 0.00011156144898932593_dp, &
                 0.0001945228908869725_dp, &
                 0.0003339605532184735_dp, &
                 0.000564711446670201_dp, &
                 0.0009413729224733118_dp, &
                 0.0015485976328911168_dp, &
                 0.0025163527439127716_dp, &
                 0.004042366217657837_dp, &
                 0.006424908889055276_dp, &
                 0.010110351754549359_dp, &
                 0.01576172006986062_dp, &
                 0.02435693199744164_dp, &
                 0.03732881710193292_dp, &
                 0.05676374129839576_dp, &
                 0.08568239505343193_dp, &
                 0.12843634401072082_dp, &
                 0.19127041101280928_dp, &
                 0.28313227875730185_dp, &
                 0.41688110772833514_dp, &
                 0.6112296495300013_dp, &
                 0.8942832614684186_dp, &
                 1.3112417204467246_dp, &
                 1.9452277892500305_dp, &
                 2.9919599852080703_dp, &
                 5.206267501211972_dp/)
      ELSE IF (E_range < 520000.0_dp) THEN
         aw(:) = (/ &
                 1.162904999603526e-06_dp, &
                 6.460220717190712e-06_dp, &
                 1.7606607480068412e-05_dp, &
                 3.8410731277459235e-05_dp, &
                 7.652869912290393e-05_dp, &
                 0.00014570244755436322_dp, &
                 0.00026937164379552174_dp, &
                 0.0004866802067040396_dp, &
                 0.0008620687596415197_dp, &
                 0.001500239074857704_dp, &
                 0.0025692139442485516_dp, &
                 0.004335527408651135_dp, &
                 0.00721744537817494_dp, &
                 0.011864777494471953_dp, &
                 0.01927761941951377_dp, &
                 0.030981700048419165_dp, &
                 0.04928550606403292_dp, &
                 0.07765484694198317_dp, &
                 0.12125517110926898_dp, &
                 0.18773242857967326_dp, &
                 0.2883321960866146_dp, &
                 0.43949878526491704_dp, &
                 0.6651609536255874_dp, &
                 1.0000229662476519_dp, &
                 1.4944068851808134_dp, &
                 2.2217383426459816_dp, &
                 3.2913012165448343_dp, &
                 4.873895944019108_dp, &
                 7.268367413333011_dp, &
                 11.163103194460474_dp, &
                 3.0200745063435063e-06_dp, &
                 7.819224105572315e-06_dp, &
                 1.5081156733800399e-05_dp, &
                 2.776801766506454e-05_dp, &
                 5.0700455074630125e-05_dp, &
                 9.145615638049602e-05_dp, &
                 0.0001622504928743556_dp, &
                 0.00028286367265657205_dp, &
                 0.0004849887879932366_dp, &
                 0.0008187828750348566_dp, &
                 0.0013627221411797283_dp, &
                 0.002238314348285273_dp, &
                 0.003631885398877431_dp, &
                 0.005826597760169261_dp, &
                 0.009249172128608455_dp, &
                 0.014537611686674936_dp, &
                 0.0226387619956523_dp, &
                 0.03494804598497715_dp, &
                 0.053508604450163914_dp, &
                 0.0812939971568708_dp, &
                 0.12260888215529167_dp, &
                 0.18365876903663883_dp, &
                 0.27337120427563866_dp, &
                 0.4046203209300098_dp, &
                 0.5961868839329513_dp, &
                 0.8763095433545912_dp, &
                 1.2903758814089559_dp, &
                 1.9216820534456363_dp, &
                 2.9658783009061263_dp, &
                 5.176364379511127_dp/)
      ELSE IF (E_range < 700000.0_dp) THEN
         aw(:) = (/ &
                 8.982200709109713e-07_dp, &
                 5.013746520038623e-06_dp, &
                 1.3797556086375253e-05_dp, &
                 3.053886905121258e-05_dp, &
                 6.188411270912267e-05_dp, &
                 0.00011984809712211666_dp, &
                 0.00022515174308936954_dp, &
                 0.00041280335546669807_dp, &
                 0.0007410557365297971_dp, &
                 0.0013054805064640572_dp, &
                 0.0022608163607798406_dp, &
                 0.003854540538433318_dp, &
                 0.006477967304857807_dp, &
                 0.010743336152506139_dp, &
                 0.01759914400488243_dp, &
                 0.028501352515678965_dp, &
                 0.04566568530951012_dp, &
                 0.07243687914225057_dp, &
                 0.11382566141804518_dp, &
                 0.17728511649027562_dp, &
                 0.2738276317065499_dp, &
                 0.41962643246873177_dp, &
                 0.6383115216108352_dp, &
                 0.9642825901967439_dp, &
                 1.4475879782153047_dp, &
                 2.1614613507414213_dp, &
                 3.2151308325865324_dp, &
                 4.779503431415253_dp, &
                 7.153609650797204_dp, &
                 11.025589359258367_dp, &
                 2.33509910395076e-06_dp, &
                 6.10398857813977e-06_dp, &
                 1.1988268517456785e-05_dp, &
                 2.2570731247056643e-05_dp, &
                 4.206867163196105e-05_dp, &
                 7.722227756857685e-05_dp, &
                 0.00013907449507452585_dp, &
                 0.0002457091386543816_dp, &
                 0.00042636927263886435_dp, &
                 0.0007277240308901886_dp, &
                 0.0012233619517539264_dp, &
                 0.0020280492037779985_dp, &
                 0.0033189631613667883_dp, &
                 0.005367062740884365_dp, &
                 0.008583092022062843_dp, &
                 0.0135845766573915_dp, &
                 0.021292762750234086_dp, &
                 0.033072042665578866_dp, &
                 0.05092942142345538_dp, &
                 0.07779867772338114_dp, &
                 0.11794435500606822_dp, &
                 0.17753757154997848_dp, &
                 0.26548687080483935_dp, &
                 0.3946762763128839_dp, &
                 0.5839408961339644_dp, &
                 0.8616294721408878_dp, &
                 1.27328786626255_dp, &
                 1.902362545255473_dp, &
                 2.944458259522062_dp, &
                 5.1518134931437265_dp/)
      ELSE IF (E_range < 1100000.0_dp) THEN
         aw(:) = (/ &
                 6.070765776691511e-07_dp, &
                 3.416871225040894e-06_dp, &
                 9.561919657171797e-06_dp, &
                 2.1683051083242104e-05_dp, &
                 4.515689173365717e-05_dp, &
                 8.982491421280511e-05_dp, &
                 0.00017296367691311632_dp, &
                 0.00032428348083868487_dp, &
                 0.0005940202656656541_dp, &
                 0.001065793876730367_dp, &
                 0.0018767883279394732_dp, &
                 0.0032490764599530236_dp, &
                 0.0055377486897541_dp, &
                 0.009304158830434279_dp, &
                 0.015426407047504075_dp, &
                 0.02526460961553787_dp, &
                 0.0409062041812229_dp, &
                 0.06552740429734295_dp, &
                 0.10392219781712_dp, &
                 0.16327176797255086_dp, &
                 0.25425762049527484_dp, &
                 0.3926657116824371_dp, &
                 0.6016960744289409_dp, &
                 0.9153062522487351_dp, &
                 1.3831427017478757_dp, &
                 2.0781516959653166_dp, &
                 3.1094698633638336_dp, &
                 4.64815163738497_dp, &
                 6.99350202546865_dp, &
                 10.833342615226073_dp, &
                 1.5810344568110277e-06_dp, &
                 4.201983132608034e-06_dp, &
                 8.509301559244783e-06_dp, &
                 1.660217499192091e-05_dp, &
                 3.1937948250709083e-05_dp, &
                 6.0196622838881555e-05_dp, &
                 0.00011090164490251824_dp, &
                 0.00019990225033881748_dp, &
                 0.0003531805009952824_dp, &
                 0.000612721804071078_dp, &
                 0.0010454964371682956_dp, &
                 0.0017570690249359784_dp, &
                 0.0029120339233426416_dp, &
                 0.004764440773172712_dp, &
                 0.007702737556158165_dp, &
                 0.012315669576916223_dp, &
                 0.019488241944529968_dp, &
                 0.03054057337997689_dp, &
                 0.0474276677063864_dp, &
                 0.0730254826754342_dp, &
                 0.11153946595828376_dp, &
                 0.1690888848289202_dp, &
                 0.2545515426791882_dp, &
                 0.3808215765752087_dp, &
                 0.5668083660036856_dp, &
                 0.8410170454494971_dp, &
                 1.2492234533619844_dp, &
                 1.8750991975085831_dp, &
                 2.91420232217261_dp, &
                 5.117153113908438_dp/)
      ELSE IF (E_range < 1800000.0_dp) THEN
         aw(:) = (/ &
                 3.9676424024146875e-07_dp, &
                 2.2573188915559954e-06_dp, &
                 6.453978574008018e-06_dp, &
                 1.5076014172691726e-05_dp, &
                 3.241300804980766e-05_dp, &
                 6.644316731632574e-05_dp, &
                 0.00013145859105193587_dp, &
                 0.0002525113816701914_dp, &
                 0.0004726891479758136_dp, &
                 0.0008648183450896165_dp, &
                 0.0015500448519401517_dp, &
                 0.00272698303228361_dp, &
                 0.004716941205527944_dp, &
                 0.008033374074156422_dp, &
                 0.013487530814882825_dp, &
                 0.022347744374551653_dp, &
                 0.03657761361377878_dp, &
                 0.0591894063537043_dp, &
                 0.09476465831169727_dp, &
                 0.15021601736547668_dp, &
                 0.23589569241536856_dp, &
                 0.3672011309219512_dp, &
                 0.5668972518738795_dp, &
                 0.8684896297613345_dp, &
                 1.3212084913196627_dp, &
                 1.9976957850654782_dp, &
                 3.00698161245029_dp, &
                 4.520261622699826_dp, &
                 6.837125864760079_dp, &
                 10.645115539809895_dp, &
                 1.0356874216884052e-06_dp, &
                 2.812088667709278e-06_dp, &
                 5.914445821533368e-06_dp, &
                 1.2021336664547268e-05_dp, &
                 2.3940050145157427e-05_dp, &
                 4.6430499419310365e-05_dp, &
                 8.765815047744608e-05_dp, &
                 0.00016143932217372545_dp, &
                 0.0002907528424437285_dp, &
                 0.0005132235070598931_dp, &
                 0.0008895970920506905_dp, &
                 0.0015167009659352298_dp, &
                 0.0025470709958953243_dp, &
                 0.004218406965056695_dp, &
                 0.006897410357753999_dp, &
                 0.011144514451186464_dp, &
                 0.017808764773569234_dp, &
                 0.028165958189904933_dp, &
                 0.044118529654135305_dp, &
                 0.0684833014292294_dp, &
                 0.1054043456969387_dp, &
                 0.16094570015015752_dp, &
                 0.24395004441521287_dp, &
                 0.3673168074143_dp, &
                 0.550025709656669_dp, &
                 0.8207374761240367_dp, &
                 1.2254629611475691_dp, &
                 1.8481116761650778_dp, &
                 2.884215749911384_dp, &
                 5.082816358644647_dp/)
      ELSE IF (E_range < 3300000.0_dp) THEN
         aw(:) = (/ &
                 2.3571329342099874e-07_dp, &
                 1.3632343165414816e-06_dp, &
                 4.023756596461991e-06_dp, &
                 9.795394424152668e-06_dp, &
                 2.195714519062327e-05_dp, &
                 4.674841254419276e-05_dp, &
                 9.563512339593217e-05_dp, &
                 0.00018918230274547437_dp, &
                 0.00036348321259617266_dp, &
                 0.0006806544569921736_dp, &
                 0.001245724390748882_dp, &
                 0.002233454990133142_dp, &
                 0.00393042938989855_dp, &
                 0.0068003647335751725_dp, &
                 0.011584430850915652_dp, &
                 0.019453873577319414_dp, &
                 0.032240162866761804_dp, &
                 0.052779185694875504_dp, &
                 0.08542204924536101_dp, &
                 0.136787796738067_dp, &
                 0.2168656674798965_dp, &
                 0.3406212481786099_dp, &
                 0.5303313641534245_dp, &
                 0.8189892522205988_dp, &
                 1.2553469341161432_dp, &
                 1.9116889104572583_dp, &
                 2.896909089589307_dp, &
                 4.382351904816623_dp, &
                 6.667935664555373_dp, &
                 10.44093549828013_dp, &
                 6.17436261768095e-07_dp, &
                 1.7313881986083482e-06_dp, &
                 3.841234035608705e-06_dp, &
                 8.228727173271725e-06_dp, &
                 1.709820341911912e-05_dp, &
                 3.4332974259821285e-05_dp, &
                 6.676429369632049e-05_dp, &
                 0.00012617607786624823_dp, &
                 0.0002325052400773221_dp, &
                 0.0004189073145508487_dp, &
                 0.000739676578430109_dp, &
                 0.0012824873185346604_dp, &
                 0.0021871159459730906_dp, &
                 0.0036737980596718025_dp, &
                 0.006085791120718465_dp, &
                 0.009952725340223944_dp, &
                 0.01608416328295351_dp, &
                 0.02570677247015947_dp, &
                 0.04066411629576856_dp, &
                 0.06370600198386596_dp, &
                 0.09890586632566792_dp, &
                 0.15226260044097462_dp, &
                 0.23257478024275838_dp, &
                 0.35274192070832605_dp, &
                 0.5318168800175598_dp, &
                 0.7986316416317844_dp, &
                 1.1994633292908101_dp, &
                 1.8185003473260537_dp, &
                 2.851271019221645_dp, &
                 5.045111765749473_dp/)
      ELSE IF (E_range < 6000000.0_dp) THEN
         aw(:) = (/ &
                 1.4156108142769783e-07_dp, &
                 8.358408090279267e-07_dp, &
                 2.563425668333653e-06_dp, &
                 6.532496957198105e-06_dp, &
                 1.529085429435563e-05_dp, &
                 3.381004239065925e-05_dp, &
                 7.145759572627074e-05_dp, &
                 0.0001454048583206386_dp, &
                 0.0002863676783658706_dp, &
                 0.0005481076173216848_dp, &
                 0.0010229120533241523_dp, &
                 0.0018664583231752616_dp, &
                 0.0033372345185815654_dp, &
                 0.005858297289124_dp, &
                 0.010112965971934416_dp, &
                 0.017191603273076256_dp, &
                 0.028814641239522522_dp, &
                 0.04766849746482397_dp, &
                 0.07790741829272711_dp, &
                 0.12589760019317675_dp, &
                 0.2013131761655887_dp, &
                 0.3187416881227427_dp, &
                 0.5000288557106496_dp, &
                 0.7777107650960449_dp, &
                 1.200107192058866_dp, &
                 1.8391727503056268_dp, &
                 2.803666178964093_dp, &
                 4.265054523983134_dp, &
                 6.5235529232359735_dp, &
                 10.266242055151874_dp, &
                 3.724391604535185e-07_dp, &
                 1.0869250980985111e-06_dp, &
                 2.5605516562597e-06_dp, &
                 5.7850356623393795e-06_dp, &
                 1.2528619638131183e-05_dp, &
                 2.6016291223482876e-05_dp, &
                 5.204832564381964e-05_dp, &
                 0.00010081228246405614_dp, &
                 0.00018982349811898816_dp, &
                 0.0003486351977763963_dp, &
                 0.000626280938172421_dp, &
                 0.0011028905979224082_dp, &
                 0.0019076162117445236_dp, &
                 0.0032460080919710396_dp, &
                 0.005441424633595424_dp, &
                 0.00899711363781583_dp, &
                 0.01468851247521531_dp, &
                 0.023699427657021866_dp, &
                 0.03782154672437762_dp, &
                 0.059744945747629535_dp, &
                 0.09347917704011861_dp, &
                 0.1449628687820927_dp, &
                 0.2229515840638538_dp, &
                 0.34033980700933175_dp, &
                 0.5162398701965166_dp, &
                 0.7796319970356158_dp, &
                 1.1770306615230532_dp, &
                 1.7928810285625805_dp, &
                 2.8227297861916565_dp, &
                 5.012462806749877_dp/)
      ELSE IF (E_range < 18000000.0_dp) THEN
         aw(:) = (/ &
                 5.619543540742037e-08_dp, &
                 3.5072928313745456e-07_dp, &
                 1.1785442886447028e-06_dp, &
                 3.3029924055616623e-06_dp, &
                 8.394882189434066e-06_dp, &
                 1.987896075698813e-05_dp, &
                 4.449626404673206e-05_dp, &
                 9.506973724339386e-05_dp, &
                 0.00019528307697601834_dp, &
                 0.0003877720465039687_dp, &
                 0.0007475784942017128_dp, &
                 0.001404160723388905_dp, &
                 0.002576876821658531_dp, &
                 0.004631430044514_dp, &
                 0.008168570072664031_dp, &
                 0.014161927673231893_dp, &
                 0.024169988544510326_dp, &
                 0.04065896918886588_dp, &
                 0.06749026098429482_dp, &
                 0.1106503055340562_dp, &
                 0.17933541972048098_dp, &
                 0.28755422315274737_dp, &
                 0.45648527437792397_dp, &
                 0.7179486868235401_dp, &
                 1.119577032884951_dp, &
                 1.7327879813628193_dp, &
                 2.6661030665007965_dp, &
                 4.091161983842176_dp, &
                 6.3086506803163065_dp, &
                 10.005419833797736_dp, &
                 1.4962249041277701e-07_dp, &
                 4.835223883864694e-07_dp, &
                 1.2927723889950915e-06_dp, &
                 3.221113081192073e-06_dp, &
                 7.5085382788298944e-06_dp, &
                 1.653889597388599e-05_dp, &
                 3.475621269796358e-05_dp, &
                 7.020789074538884e-05_dp, &
                 0.000137107896070371_dp, &
                 0.0002600215543190807_dp, &
                 0.00048059369103468994_dp, &
                 0.0008682158907070847_dp, &
                 0.0015367276921454673_dp, &
                 0.0026702645844449866_dp, &
                 0.004562839898703722_dp, &
                 0.007678366608423676_dp, &
                 0.012740848501038493_dp, &
                 0.02086877544772841_dp, &
                 0.03377384609132226_dp, &
                 0.05405280484981297_dp, &
                 0.08561371875294983_dp, &
                 0.13429708936338725_dp, &
                 0.20878452860262342_dp, &
                 0.32195328331951945_dp, &
                 0.4929981089833493_dp, &
                 0.7511227960385115_dp, &
                 1.1432128213684527_dp, &
                 1.7541296769802968_dp, &
                 2.7794877584199225_dp, &
                 4.963023556769552_dp/)
      ELSE IF (E_range < 50000000.0_dp) THEN
         aw(:) = (/ &
                 2.4343774571442308e-08_dp, &
                 1.6513085830933388e-07_dp, &
                 6.196802532079424e-07_dp, &
                 1.9114039978346155e-06_dp, &
                 5.2375690334248195e-06_dp, &
                 1.316277905469794e-05_dp, &
                 3.092030271194152e-05_dp, &
                 6.876819455159673e-05_dp, &
                 0.0001461441409773963_dp, &
                 0.0002988264714800754_dp, &
                 0.0005910302376020617_dp, &
                 0.0011354771724870155_dp, &
                 0.0021261659208679307_dp, &
                 0.0038911087717360972_dp, &
                 0.0069760886290369745_dp, &
                 0.012276085772085119_dp, &
                 0.021239226973771258_dp, &
                 0.03618004618429005_dp, &
                 0.060756103616981624_dp, &
                 0.10068683961443668_dp, &
                 0.16482922554469034_dp, &
                 0.2667764798918916_dp, &
                 0.4272231990996556_dp, &
                 0.6774637303634756_dp, &
                 1.064618278549332_dp, &
                 1.6596947503579922_dp, &
                 2.5710208535654164_dp, &
                 3.9703478472317504_dp, &
                 6.158711044835771_dp, &
                 9.822845740299956_dp, &
                 6.607779435757453e-08_dp, &
                 2.4531316662915766e-07_dp, &
                 7.456766806787507e-07_dp, &
                 2.026328956129065e-06_dp, &
                 5.0327732385235214e-06_dp, &
                 1.1652295901214502e-05_dp, &
                 2.5505955301673674e-05_dp, &
                 5.331453501898778e-05_dp, &
                 0.0001072058234797249_dp, &
                 0.00020853777085767683_dp, &
                 0.0003941273627591354_dp, &
                 0.0007262462873395727_dp, &
                 0.0013084426047463612_dp, &
                 0.00231027419159065_dp, &
                 0.004005535279633006_dp, &
                 0.006830718203653046_dp, &
                 0.011473559821092195_dp, &
                 0.019005957730023747_dp, &
                 0.031081856143062504_dp, &
                 0.0502296954993904_dp, &
                 0.08028203266369409_dp, &
                 0.12700459445500437_dp, &
                 0.19901982937944282_dp, &
                 0.3091853239783209_dp, &
                 0.4767481905112323_dp, &
                 0.7310696923220488_dp, &
                 1.1193069602208368_dp, &
                 1.72663775937696_dp, &
                 2.74875508795335_dp, &
                 4.927904659791831_dp/)
      ELSE
         aw(:) = (/ &
                 3.3134286206436378e-09_dp, &
                 3.608476731974466e-08_dp, &
                 1.9011771017402648e-07_dp, &
                 7.319641454121685e-07_dp, &
                 2.3402664590560514e-06_dp, &
                 6.59508695658557e-06_dp, &
                 1.6937909149354525e-05_dp, &
                 4.048237319663682e-05_dp, &
                 9.132011445721631e-05_dp, &
                 0.00019639172259169963_dp, &
                 0.0004056673541505352_dp, &
                 0.0008094360939409007_dp, &
                 0.001567130287522824_dp, &
                 0.0029545924806747004_dp, &
                 0.005440438931022152_dp, &
                 0.009807773172795757_dp, &
                 0.017345795847210112_dp, &
                 0.030148043017114434_dp, &
                 0.0515717640546498_dp, &
                 0.08693872933672857_dp, &
                 0.14459504048117927_dp, &
                 0.23750169179013617_dp, &
                 0.3856081802743787_dp, &
                 0.6193889538319287_dp, &
                 0.9851526108818242_dp, &
                 1.5532426287786882_dp, &
                 2.431651849778814_dp, &
                 3.7922799603529507_dp, &
                 5.93671227399739_dp, &
                 9.551588089060862_dp, &
                 1.0546633313159667e-08_dp, &
                 6.866238130433312e-08_dp, &
                 2.8014791592507246e-07_dp, &
                 9.089927672975243e-07_dp, &
                 2.5558210419120903e-06_dp, &
                 6.498693707400478e-06_dp, &
                 1.531991208406891e-05_dp, &
                 3.4023779895267246e-05_dp, &
                 7.197856727334818e-05_dp, &
                 0.00014621506144801751_dp, &
                 0.0002869199340733901_dp, &
                 0.0005464264697957605_dp, &
                 0.0010137006051227095_dp, &
                 0.0018373548659257801_dp, &
                 0.0032617470785987366_dp, &
                 0.005682944230668296_dp, &
                 0.009734554219352109_dp, &
                 0.016418075032354977_dp, &
                 0.02729906435610205_dp, &
                 0.04479995778261982_dp, &
                 0.07263414004010217_dp, &
                 0.11644650227746163_dp, &
                 0.1847594577631817_dp, &
                 0.2903885825624092_dp, &
                 0.4526492240843357_dp, &
                 0.7011369514153092_dp, &
                 1.0834309171758731_dp, &
                 1.6852193328076974_dp, &
                 2.7023644845108916_dp, &
                 4.874923027763007_dp/)
      END IF
   END SUBROUTINE

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param E_range ...
!> \param aw ...
!> \param E_ratio ...
! **************************************************************************************************
   PURE SUBROUTINE get_coeff_32(k, E_range, aw, E_ratio)
      INTEGER, INTENT(IN)                                :: k
      REAL(KIND=dp), INTENT(IN)                          :: E_range
      REAL(KIND=dp), DIMENSION(2*k), INTENT(OUT)         :: aw
      REAL(KIND=dp), INTENT(OUT)                         :: E_ratio

      E_ratio = 1.0_dp
      IF (E_range < 4862.0_dp) THEN
         E_ratio = 4862.0_dp/E_range
         aw(:) = (/ &
                 6.390102821995496e-05_dp, &
                 0.0003410048490902184_dp, &
                 0.000858015893339577_dp, &
                 0.0016504328288047757_dp, &
                 0.002778247721262644_dp, &
                 0.004336751679872466_dp, &
                 0.0064710209548362084_dp, &
                 0.009393407912277671_dp, &
                 0.01340416788754177_dp, &
                 0.018917565422190717_dp, &
                 0.026497301468281335_dp, &
                 0.036905213045864395_dp, &
                 0.0511672148624311_dp, &
                 0.07066128158960995_dp, &
                 0.09723384839331731_dp, &
                 0.13335312533591778_dp, &
                 0.18231048817210543_dp, &
                 0.24848450523394644_dp, &
                 0.3376865863344385_dp, &
                 0.4576131552357413_dp, &
                 0.6184374387625521_dp, &
                 0.8335859250026539_dp, &
                 1.1207633263472254_dp, &
                 1.5033220996389998_dp, &
                 2.0121331479703484_dp, &
                 2.6882384204905283_dp, &
                 3.5868412256539663_dp, &
                 4.783852020373796_dp, &
                 6.387973693919086_dp, &
                 8.566792744620582_dp, &
                 11.61702931918549_dp, &
                 16.243244163196902_dp, &
                 0.0001644750107629364_dp, &
                 0.0003927652580734362_dp, &
                 0.0006470526340880139_dp, &
                 0.0009476004809462573_dp, &
                 0.0013236672585754251_dp, &
                 0.00181722804655121_dp, &
                 0.0024863648293148356_dp, &
                 0.0034080264366486593_dp, &
                 0.004681956510169319_dp, &
                 0.006437986099490384_dp, &
                 0.00884737093390095_dp, &
                 0.01213798662642807_dp, &
                 0.01661375946425665_dp, &
                 0.022679584239104586_dp, &
                 0.030873595466529645_dp, &
                 0.0419091652309378_dp, &
                 0.05672961944641861_dp, &
                 0.07657952416894345_dp, &
                 0.10309761695759492_dp, &
                 0.13843826485639682_dp, &
                 0.18543118617833296_dp, &
                 0.24779409719067566_dp, &
                 0.3304222201755898_dp, &
                 0.43979746242641576_dp, &
                 0.5846010059990558_dp, &
                 0.7767070228661042_dp, &
                 1.0329650164221098_dp, &
                 1.3787901136203442_dp, &
                 1.8564287417380754_dp, &
                 2.547529279131862_dp, &
                 3.6531886321446416_dp, &
                 5.969270620351697_dp/)
      ELSE IF (E_range < 5846.0_dp) THEN
         aw(:) = (/ &
                 5.442349002883837e-05_dp, &
                 0.0002906163628933693_dp, &
                 0.0007321293214401846_dp, &
                 0.0014109784539713872_dp, &
                 0.0023816365307190094_dp, &
                 0.0037311772208214787_dp, &
                 0.005592773584012946_dp, &
                 0.0081619152463338_dp, &
                 0.011715729132325964_dp, &
                 0.016637929524033673_dp, &
                 0.023453105417011018_dp, &
                 0.03287403908552556_dp, &
                 0.045865900699847945_dp, &
                 0.06373214396298486_dp, &
                 0.08822857905561622_dp, &
                 0.12171425464017217_dp, &
                 0.16735049859917261_dp, &
                 0.22936294868166074_dp, &
                 0.3133859506800179_dp, &
                 0.4269147709487938_dp, &
                 0.5798994400836499_dp, &
                 0.7855261781163106_dp, &
                 1.0612512378052221_dp, &
                 1.4301841263166695_dp, &
                 1.9229772371853768_dp, &
                 2.580501782235826_dp, &
                 3.4578625966778955_dp, &
                 4.630962088019041_dp, &
                 6.208582258228505_dp, &
                 8.358418878620247_dp, &
                 11.37712612667173_dp, &
                 15.968170544511102_dp, &
                 0.0001401015051695221_dp, &
                 0.00033499665110023704_dp, &
                 0.0005532522530987489_dp, &
                 0.0008133560062402562_dp, &
                 0.0011422450286066607_dp, &
                 0.0015787618621387204_dp, &
                 0.002176697085238142_dp, &
                 0.0030073676766694266_dp, &
                 0.004163649639092577_dp, &
                 0.005767402761174793_dp, &
                 0.007980687395399007_dp, &
                 0.011020625281672214_dp, &
                 0.015178459352483943_dp, &
                 0.020844174229750198_dp, &
                 0.028538588286649656_dp, &
                 0.03895532923923562_dp, &
                 0.05301575138551127_dp, &
                 0.07194074865569469_dp, &
                 0.09734467595148226_dp, &
                 0.1313584288969531_dp, &
                 0.17679160298883081_dp, &
                 0.23734854692003116_dp, &
                 0.31792227850976007_dp, &
                 0.42500882037583165_dp, &
                 0.5673248919963556_dp, &
                 0.7568049338284129_dp, &
                 1.0103817704846647_dp, &
                 1.3535647228420928_dp, &
                 1.8286721197765015_dp, &
                 2.517319618502594_dp, &
                 3.6202102280059703_dp, &
                 5.930981680526662_dp/)
      ELSE IF (E_range < 6665.0_dp) THEN
         aw(:) = (/ &
                 4.8544808464658475e-05_dp, &
                 0.0002593485712181486_dp, &
                 0.0006539518705617748_dp, &
                 0.0012620998863735048_dp, &
                 0.002134648029009164_dp, &
                 0.003353250117842424_dp, &
                 0.005043178699940835_dp, &
                 0.007388686080392843_dp, &
                 0.010651467508498257_dp, &
                 0.01519484467183565_dp, &
                 0.02151725732341947_dp, &
                 0.03029858754646109_dp, &
                 0.04246309490817034_dp, &
                 0.05926381239480956_dp, &
                 0.08239494336727046_dp, &
                 0.11414097839482948_dp, &
                 0.15757400511758252_dp, &
                 0.21681422548727597_dp, &
                 0.29737332757633034_dp, &
                 0.4066065418252456_dp, &
                 0.5543077141349219_dp, &
                 0.7534939940706474_dp, &
                 1.0214457127670413_dp, &
                 1.3810991072747199_dp, &
                 1.8629492914867638_dp, &
                 2.5077428833752067_dp, &
                 3.370510682262872_dp, &
                 4.52714545796296_dp, &
                 6.086483167699423_dp, &
                 8.216299008759131_dp, &
                 11.213210923746692_dp, &
                 15.779945298966286_dp, &
                 0.00012498176375406726_dp, &
                 0.000299130758824733_dp, &
                 0.0004949254877793316_dp, &
                 0.0007296849915859161_dp, &
                 0.0010288042473983914_dp, &
                 0.0014290235014882077_dp, &
                 0.0019812415860893163_dp, &
                 0.0027529950493393966_dp, &
                 0.00383257137438519_dp, &
                 0.005336492795437645_dp, &
                 0.007420608589016774_dp, &
                 0.01029470391501449_dp, &
                 0.014241296894904196_dp, &
                 0.01964004703493962_dp, &
                 0.02699971016577413_dp, &
                 0.03700007872604122_dp, &
                 0.050547012803328586_dp, &
                 0.06884458868981981_dp, &
                 0.09348967784530542_dp, &
                 0.12659613160197555_dp, &
                 0.17095862941207848_dp, &
                 0.2302711267303353_dp, &
                 0.3094239152326776_dp, &
                 0.4149216994326361_dp, &
                 0.555505084520509_dp, &
                 0.7431502415960526_dp, &
                 0.9948487920453783_dp, &
                 1.3361777826685788_dp, &
                 1.809509286088653_dp, &
                 2.496441460806496_dp, &
                 3.597410863544938_dp, &
                 5.9045255378523525_dp/)
      ELSE IF (E_range < 7800.0_dp) THEN
         aw(:) = (/ &
                 4.2319791684398945e-05_dp, &
                 0.0002262256833340046_dp, &
                 0.0005710759339812151_dp, &
                 0.0011041000499916853_dp, &
                 0.0018721273080030145_dp, &
                 0.002950744254164121_dp, &
                 0.004456329948077625_dp, &
                 0.006560445775378507_dp, &
                 0.009507343812482433_dp, &
                 0.013637283047013359_dp, &
                 0.01941907839765781_dp, &
                 0.027495223047590025_dp, &
                 0.03874330338636378_dp, &
                 0.054358588747561776_dp, &
                 0.07596440384363516_dp, &
                 0.10575909660902735_dp, &
                 0.14671121294566578_dp, &
                 0.20281810337064493_dp, &
                 0.27944792395127677_dp, &
                 0.38379131316561327_dp, &
                 0.525457692252208_dp, &
                 0.7172635733637082_dp, &
                 0.9762793617085143_dp, &
                 1.3252331952635001_dp, &
                 1.7944297509586076_dp, &
                 2.4244629711653745_dp, &
                 3.270270626899914_dp, &
                 4.407729666691881_dp, &
                 5.945738068832148_dp, &
                 8.052167730865541_dp, &
                 11.023603896745234_dp, &
                 15.561924493694205_dp, &
                 0.00010896980958811763_dp, &
                 0.0002611185815588283_dp, &
                 0.00043301817888623847_dp, &
                 0.0006406820460768114_dp, &
                 0.0009077674409158128_dp, &
                 0.001268624601649587_dp, &
                 0.0017708556976942175_dp, &
                 0.002477704567241762_dp, &
                 0.0034722665030748113_dp, &
                 0.004865004260125733_dp, &
                 0.006804656478190395_dp, &
                 0.009492536281842186_dp, &
                 0.013201017163803366_dp, &
                 0.01829768554187978_dp, &
                 0.025277118873799437_dp, &
                 0.034802761118126926_dp, &
                 0.04776205831199323_dp, &
                 0.06533896907314592_dp, &
                 0.08910928388293891_dp, &
                 0.1211660799438405_dp, &
                 0.16428554188642505_dp, &
                 0.22214824519628396_dp, &
                 0.29963998828115096_dp, &
                 0.40327442471190905_dp, &
                 0.5418192779326029_dp, &
                 0.7272994796496782_dp, &
                 0.9767765646718063_dp, &
                 1.315909458684106_dp, &
                 1.7871373292268897_dp, &
                 2.4720433230436907_dp, &
                 3.5707586649853615_dp, &
                 5.873612522584144_dp/)
      ELSE IF (E_range < 10044.0_dp) THEN
         aw(:) = (/ &
                 3.393255390660133e-05_dp, &
                 0.00018157221465841256_dp, &
                 0.0004592307676320871_dp, &
                 0.0008905276367436023_dp, &
                 0.0015164746316516245_dp, &
                 0.0024038182425049908_dp, &
                 0.003655880996707775_dp, &
                 0.0054255179504103905_dp, &
                 0.007931245931576318_dp, &
                 0.0114792650369607_dp, &
                 0.01649451279684121_dp, &
                 0.023563843280162002_dp, &
                 0.033494992691347265_dp, &
                 0.047396265575400595_dp, &
                 0.06678363974811272_dp, &
                 0.093724229945987_dp, &
                 0.13102792038715239_dp, &
                 0.18250270781166214_dp, &
                 0.2532942033927338_dp, &
                 0.3503362853107127_dp, &
                 0.4829488331079465_dp, &
                 0.6636312054412428_dp, &
                 0.9091194626135006_dp, &
                 1.241807411536441_dp, &
                 1.6916904363164948_dp, &
                 2.299110603722669_dp, &
                 3.1188473050465815_dp, &
                 4.226742880544712_dp, &
                 5.73178901479017_dp, &
                 7.802015996740442_dp, &
                 10.73398018079764_dp, &
                 15.228283328315147_dp, &
                 8.739335036060486e-05_dp, &
                 0.00020983760313852035_dp, &
                 0.0003493229451058535_dp, &
                 0.0005199664146008647_dp, &
                 0.0007428694438573918_dp, &
                 0.001048824687194698_dp, &
                 0.0014805150854819704_dp, &
                 0.0020948189118065326_dp, &
                 0.0029671598393268173_dp, &
                 0.004198990073584458_dp, &
                 0.005928343272370908_dp, &
                 0.008343628060712927_dp, &
                 0.011701624144711492_dp, &
                 0.0163512213147089_dp, &
                 0.022764888824347185_dp, &
                 0.03158039794694182_dp, &
                 0.04365605404949107_dp, &
                 0.06014369592103612_dp, &
                 0.08258509281019695_dp, &
                 0.11303931699899293_dp, &
                 0.1542516135087157_dp, &
                 0.20987914795565896_dp, &
                 0.28479788380848736_dp, &
                 0.38553266378218126_dp, &
                 0.5208913754190811_dp, &
                 0.7029743313415381_dp, &
                 0.9489538722825864_dp, &
                 1.2846216880545815_dp, &
                 1.7525303625505235_dp, &
                 2.4342520367725067_dp, &
                 3.5294590063092537_dp, &
                 5.82574638448254_dp/)
      ELSE IF (E_range < 14058.0_dp) THEN
         aw(:) = (/ &
                 2.5286922170354894e-05_dp, &
                 0.00013550300515152938_dp, &
                 0.00034365254863389185_dp, &
                 0.0006692829830793605_dp, &
                 0.0011467768099562_dp, &
                 0.0018326842825994733_dp, &
                 0.0028151146148173586_dp, &
                 0.004225026003742279_dp, &
                 0.006250780019705354_dp, &
                 0.009158586718943748_dp, &
                 0.013321604127882514_dp, &
                 0.01926054419553751_dp, &
                 0.02769941545264008_dp, &
                 0.03964139074601272_dp, &
                 0.056471580120530336_dp, &
                 0.08009577889831763_dp, &
                 0.11312722824816804_dp, &
                 0.15913731138320336_dp, &
                 0.22299123810302657_dp, &
                 0.3112966160076166_dp, &
                 0.43300212495019025_dp, &
                 0.6001966661049284_dp, &
                 0.8291790761739616_dp, &
                 1.141900704443846_dp, &
                 1.5679415214227161_dp, &
                 2.147298128643314_dp, &
                 2.934523705189343_dp, &
                 4.00539737112386_dp, &
                 5.469022915149277_dp, &
                 7.493643904207673_dp, &
                 10.375817412384437_dp, &
                 14.814601510443076_dp, &
                 6.514769920993364e-05_dp, &
                 0.00015687397983623111_dp, &
                 0.00026260059811101074_dp, &
                 0.000394267458907849_dp, &
                 0.000569987485627106_dp, &
                 0.0008163276402528582_dp, &
                 0.0011701237352853642_dp, &
                 0.001680758396099781_dp, &
                 0.002414639429717159_dp, &
                 0.0034624986342827427_dp, &
                 0.004949404187288667_dp, &
                 0.00704789552613599_dp, &
                 0.00999535091125554_dp, &
                 0.014117167183015276_dp, &
                 0.019857768872086755_dp, &
                 0.02782204007098351_dp, &
                 0.03883055990410155_dp, &
                 0.05399308565310966_dp, &
                 0.07480617533828689_dp, &
                 0.1032828703661818_dp, &
                 0.14212537017885443_dp, &
                 0.19495650401901887_dp, &
                 0.2666345532300576_dp, &
                 0.3636933937165475_dp, &
                 0.4949878042793592_dp, &
                 0.6727122940047311_dp, &
                 0.91418280277028_dp, &
                 1.2453685853158643_dp, &
                 1.7089820529103965_dp, &
                 2.3866034784779506_dp, &
                 3.4773514427582293_dp, &
                 5.765409584335823_dp/)
      ELSE IF (E_range < 19114.0_dp) THEN
         aw(:) = (/ &
                 1.932135319864962e-05_dp, &
                 0.00010368197789193328_dp, &
                 0.0002636661772942731_dp, &
                 0.0005157191470705282_dp, &
                 0.0008891130218806238_dp, &
                 0.0014324323017507073_dp, &
                 0.002221781063255329_dp, &
                 0.003370774273254239_dp, &
                 0.005043864110016273_dp, &
                 0.00747544974094905_dp, &
                 0.010997228176632299_dp, &
                 0.01607649385487372_dp, &
                 0.023369007932130658_dp, &
                 0.033791453456649585_dp, &
                 0.04862029962984213_dp, &
                 0.06962623074409648_dp, &
                 0.09925634818539467_dp, &
                 0.14088037853305638_dp, &
                 0.199122452009721_dp, &
                 0.2803071363072363_dp, &
                 0.39305808457961094_dp, &
                 0.549101229063986_dp, &
                 0.7643445772055675_dp, &
                 1.0603390883163382_dp, &
                 1.466283147413628_dp, &
                 2.0218493641999524_dp, &
                 2.781370504056042_dp, &
                 3.8205517999540817_dp, &
                 5.248586694740235_dp, &
                 7.23391404025796_dp, &
                 10.073125286074314_dp, &
                 14.464007598891323_dp, &
                 4.9794334064441845e-05_dp, &
                 0.0001202442466871008_dp, &
                 0.00020239142207391234_dp, &
                 0.00030648388928353133_dp, &
                 0.00044826394218884584_dp, &
                 0.0006508962458350465_dp, &
                 0.0009465213181019036_dp, &
                 0.001378560941846276_dp, &
                 0.002006235556189542_dp, &
                 0.002911588665713435_dp, &
                 0.004208981250342487_dp, &
                 0.006057652087479506_dp, &
                 0.008678530354123182_dp, &
                 0.012376893766972177_dp, &
                 0.01757291406967402_dp, &
                 0.02484274557305271_dp, &
                 0.034973648086824426_dp, &
                 0.04903775694722466_dp, &
                 0.0684906283923225_dp, &
                 0.09530279631811456_dp, &
                 0.1321356654214213_dp, &
                 0.1825779770666456_dp, &
                 0.2514677535724369_dp, &
                 0.34534170861863805_dp, &
                 0.473091186246975_dp, &
                 0.6469906291688993_dp, &
                 0.8844828931716822_dp, &
                 1.2116994219481383_dp, &
                 1.6715060390575855_dp, &
                 2.3455108368533817_dp, &
                 3.432378601360354_dp, &
                 5.713384147526647_dp/)
      ELSE IF (E_range < 25870.0_dp) THEN
         aw(:) = (/ &
                 1.4818639796865528e-05_dp, &
                 7.963886363515554e-05_dp, &
                 0.00020311223580610354_dp, &
                 0.00039911404589973475_dp, &
                 0.0006926333595966217_dp, &
                 0.0011254943882583104_dp, &
                 0.0017635291623248677_dp, &
                 0.0027054780460561163_dp, &
                 0.004095260696633318_dp, &
                 0.006139843660554759_dp, &
                 0.009134914466544177_dp, &
                 0.013500979005727987_dp, &
                 0.019833495884515773_dp, &
                 0.02897206838249403_dp, &
                 0.042095533039198385_dp, &
                 0.06085215831868936_dp, &
                 0.08753730146299196_dp, &
                 0.1253350272549887_dp, &
                 0.17864571781396288_dp, &
                 0.2535290933953418_dp, &
                 0.35830209305587557_dp, &
                 0.5043450664128841_dp, &
                 0.7071902955364795_dp, &
                 0.9879995840794562_dp, &
                 1.3755955013642696_dp, &
                 1.9093257812455384_dp, &
                 2.6432950751333295_dp, &
                 3.6531218956627853_dp, &
                 5.048075954994877_dp, &
                 6.99678689342338_dp, &
                 9.795906582751762_dp, &
                 14.142088288126018_dp, &
                 3.820307739174523e-05_dp, &
                 9.253211423715926e-05_dp, &
                 0.00015666144789002137_dp, &
                 0.00023941035175647392_dp, &
                 0.0003544795957843895_dp, &
                 0.0005220706038556393_dp, &
                 0.0007702567427026573_dp, &
                 0.0011373348215882129_dp, &
                 0.001676302988186351_dp, &
                 0.002461551034244103_dp, &
                 0.003597851119996275_dp, &
                 0.005232383925906072_dp, &
                 0.007571020540645877_dp, &
                 0.01090045623553385_dp, &
                 0.01561826230147036_dp, &
                 0.022273572546892197_dp, &
                 0.03162200007875945_dp, &
                 0.04469955727258082_dp, &
                 0.06292193738822648_dp, &
                 0.08821770562181855_dp, &
                 0.12320712191071415_dp, &
                 0.17144328423198826_dp, &
                 0.23774090556227068_dp, &
                 0.3286348439935499_dp, &
                 0.45304683721268363_dp, &
                 0.6233242933376546_dp, &
                 0.8570306244015214_dp, &
                 1.1804560303362448_dp, &
                 1.6366228282940412_dp, &
                 2.3071833124934_dp, &
                 3.3903998854623985_dp, &
                 5.664862602289436_dp/)
      ELSE IF (E_range < 35180.0_dp) THEN
         aw(:) = (/ &
                 1.1315989220620849e-05_dp, &
                 6.091484824470965e-05_dp, &
                 0.0001558556348063919_dp, &
                 0.0003078203984115187_dp, &
                 0.0005380983662976303_dp, &
                 0.0008826023610964867_dp, &
                 0.0013981206622491853_dp, &
                 0.0021702768627684973_dp, &
                 0.0033248504851841115_dp, &
                 0.005044444967959304_dp, &
                 0.007592519217726472_dp, &
                 0.011347339662602948_dp, &
                 0.016849450868861544_dp, &
                 0.024867669342019266_dp, &
                 0.03649043474384647_dp, &
                 0.05325176815491175_dp, &
                 0.07730430050505666_dp, &
                 0.1116561161775443_dp, &
                 0.16049387125009024_dp, &
                 0.22962230812156062_dp, &
                 0.3270606812998151_dp, &
                 0.46385106272587334_dp, &
                 0.6551545548992531_dp, &
                 0.9217445361800276_dp, &
                 1.2920638357850092_dp, &
                 1.8051264992628424_dp, &
                 2.514796276521399_dp, &
                 3.496590833178514_dp, &
                 4.85984548686659_dp, &
                 6.773380959237685_dp, &
                 9.533934166411905_dp, &
                 13.837118014874001_dp, &
                 2.918394667912216e-05_dp, &
                 7.092102053120252e-05_dp, &
                 0.0001208490607819795_dp, &
                 0.00018654379226223144_dp, &
                 0.00027989427379719806_dp, &
                 0.0004184502969887145_dp, &
                 0.000626673050017551_dp, &
                 0.0009383293128013663_dp, &
                 0.0014008572107836192_dp, &
                 0.002081684144431634_dp, &
                 0.0030767293366374213_dp, &
                 0.004521918459768059_dp, &
                 0.006608939226821494_dp, &
                 0.009606834282132472_dp, &
                 0.013891522205312699_dp, &
                 0.0199860217642558_dp, &
                 0.028615075699006714_dp, &
                 0.04077910134384887_dp, &
                 0.057854053870927825_dp, &
                 0.08172606721705969_dp, &
                 0.11497300539733377_dp, &
                 0.16111009858027334_dp, &
                 0.22492545113901904_dp, &
                 0.3129477675464925_dp, &
                 0.43412426576959084_dp, &
                 0.6008706467915673_dp, &
                 0.8308678912066612_dp, &
                 1.1505656278854524_dp, &
                 1.6031493439397482_dp, &
                 2.270331555451258_dp, &
                 3.3500083706514476_dp, &
                 5.6182166490122984_dp/)
      ELSE IF (E_range < 58986.0_dp) THEN
         aw(:) = (/ &
                 7.188720818598015e-06_dp, &
                 3.881661788798954e-05_dp, &
                 9.991514616563082e-05_dp, &
                 0.00019924670807668527_dp, &
                 0.0003530921393375012_dp, &
                 0.0005892390933533132_dp, &
                 0.0009519594342766803_dp, &
                 0.0015086954155209336_dp, &
                 0.0023599840025008696_dp, &
                 0.003654238190455114_dp, &
                 0.005609186735480063_dp, &
                 0.008542461005020551_dp, &
                 0.01291487453690728_dp, &
                 0.01939133485309362_dp, &
                 0.028926175238232574_dp, &
                 0.04288217331115784_dp, &
                 0.06319585740381348_dp, &
                 0.09260618135064007_dp, &
                 0.13496966452445772_dp, &
                 0.19569319485572675_dp, &
                 0.2823266941469131_dp, &
                 0.40537308432315655_dp, &
                 0.5793949454394274_dp, &
                 0.8245311284762115_dp, &
                 1.1685945009236132_dp, &
                 1.6500342331912714_dp, &
                 2.322292524496194_dp, &
                 3.260691519334744_dp, &
                 4.574650070688025_dp, &
                 6.433299519631953_dp, &
                 9.133561259222846_dp, &
                 13.369520380969645_dp, &
                 1.8552570543955455e-05_dp, &
                 4.5365280395614445e-05_dp, &
                 7.824430449992235e-05_dp, &
                 0.000123064841905601_dp, &
                 0.00018917614532398957_dp, &
                 0.00029038908164917863_dp, &
                 0.0004461302902503803_dp, &
                 0.0006838555287975979_dp, &
                 0.001043105615047013_dp, &
                 0.0015811894241096348_dp, &
                 0.0023809364846653818_dp, &
                 0.003561415027894291_dp, &
                 0.005292829489029218_dp, &
                 0.007817192602841475_dp, &
                 0.011476902733337355_dp, &
                 0.01675409107074492_dp, &
                 0.024324590249295588_dp, &
                 0.035131696353015066_dp, &
                 0.050486674490227534_dp, &
                 0.07220539175013203_dp, &
                 0.10279391074071534_dp, &
                 0.14570107833907187_dp, &
                 0.20566478637597496_dp, &
                 0.28919483053487227_dp, &
                 0.40526987079618043_dp, &
                 0.5664074675124494_dp, &
                 0.7904744673132619_dp, &
                 1.1041823544223337_dp, &
                 1.5509968448017435_dp, &
                 2.2127606942845244_dp, &
                 3.286839828729209_dp, &
                 5.545336982828468_dp/)
      ELSE IF (E_range < 85052.0_dp) THEN
         aw(:) = (/ &
                 5.212842853721373e-06_dp, &
                 2.8216485728556984e-05_dp, &
                 7.298119875698969e-05_dp, &
                 0.00014666687860403872_dp, &
                 0.00026275225618308033_dp, &
                 0.0004444091505151155_dp, &
                 0.0007287615881022521_dp, &
                 0.0011728502574399147_dp, &
                 0.0018626816535886259_dp, &
                 0.0029267741621968922_dp, &
                 0.004555909601317652_dp, &
                 0.007031533755260647_dp, &
                 0.01076629475072859_dp, &
                 0.016361592669832718_dp, &
                 0.02468887356341755_dp, &
                 0.03700391916268461_dp, &
                 0.055106793600662854_dp, &
                 0.08156471559554047_dp, &
                 0.12002134058150862_dp, &
                 0.17562433753378992_dp, &
                 0.2556145691836236_dp, &
                 0.370135979817623_dp, &
                 0.5333479158700739_dp, &
                 0.7649561302861829_dp, &
                 1.092336920188996_dp, &
                 1.5535402649553307_dp, &
                 2.201701148128128_dp, &
                 3.111986649331865_dp, &
                 4.39385578004782_dp, &
                 6.216650142887412_dp, &
                 8.877446914810541_dp, &
                 13.069399675595056_dp, &
                 1.3460633699481115e-05_dp, &
                 3.307685267989567e-05_dp, &
                 5.7604518102217696e-05_dp, &
                 9.195641910083116e-05_dp, &
                 0.00014400755053456355_dp, &
                 0.00022539775962319305_dp, &
                 0.0003526662335951264_dp, &
                 0.0005496236389883362_dp, &
                 0.0008511354053799554_dp, &
                 0.0013083814172165737_dp, &
                 0.0019961328695681448_dp, &
                 0.0030229467394267007_dp, &
                 0.00454547537639009_dp, &
                 0.00678848631990991_dp, &
                 0.010072747132319853_dp, &
                 0.014853695284454384_dp, &
                 0.021774843424492554_dp, &
                 0.03174125164120884_dp, &
                 0.04602026166587432_dp, &
                 0.06637923482681696_dp, &
                 0.0952736156331887_dp, &
                 0.13610397708422503_dp, &
                 0.19356942137149777_dp, &
                 0.27416081565736494_dp, &
                 0.3868712620397728_dp, &
                 0.5442810461868627_dp, &
                 0.7643794009427055_dp, &
                 1.0740573000929907_dp, &
                 1.5169810386781961_dp, &
                 2.1751039860564134_dp, &
                 3.2454752973208514_dp, &
                 5.497663131109453_dp/)
      ELSE IF (E_range < 126612.0_dp) THEN
         aw(:) = (/ &
                 3.6756703015465298e-06_dp, &
                 1.9954750677579616e-05_dp, &
                 5.191549207164041e-05_dp, &
                 0.00010531730513004429_dp, &
                 0.00019114894568566566_dp, &
                 0.0003284345206580502_dp, &
                 0.0005478574351058358_dp, &
                 0.0008970642726998205_dp, &
                 0.001448846563954588_dp, &
                 0.0023134440610659145_dp, &
                 0.00365660754627558_dp, &
                 0.005725809627536767_dp, &
                 0.008888016831791598_dp, &
                 0.01368381089343313_dp, &
                 0.020904518142872384_dp, &
                 0.031701561841387796_dp, &
                 0.04774073523681854_dp, &
                 0.07141882485947987_dp, &
                 0.10616643258939448_dp, &
                 0.15686955518574938_dp, &
                 0.23045435097627895_dp, &
                 0.33669592809026266_dp, &
                 0.48933536087816165_dp, &
                 0.7076244370403264_dp, &
                 1.0184762856769092_dp, &
                 1.4595108245228903_dp, &
                 2.0835241856266555_dp, &
                 2.965501956804563_dp, &
                 4.214929989807013_dp, &
                 6.001367447071038_dp, &
                 8.622077664029446_dp, &
                 12.76932426608392_dp, &
                 9.497600314818484e-06_dp, &
                 2.347760247868802e-05_dp, &
                 4.136863276834202e-05_dp, &
                 6.721947207800727e-05_dp, &
                 0.00010755652914793868_dp, &
                 0.00017204310109274583_dp, &
                 0.00027461020521950513_dp, &
                 0.00043573410701580053_dp, &
                 0.0006859083122114215_dp, &
                 0.0010704829571126738_dp, &
                 0.0016564711186942442_dp, &
                 0.0025422076964986724_dp, &
                 0.003871039755848531_dp, &
                 0.005850643295798488_dp, &
                 0.008780144703142966_dp, &
                 0.013088020403816946_dp, &
                 0.01938482007375741_dp, &
                 0.02853620387815694_dp, &
                 0.04176374049584569_dp, &
                 0.06078358076425594_dp, &
                 0.087996854859105_dp, &
                 0.12675113895160917_dp, &
                 0.18170115882627896_dp, &
                 0.2593129035941683_dp, &
                 0.3685886307380454_dp, &
                 0.5221684374385586_dp, &
                 0.7381657526850111_dp, &
                 1.0436599990100175_dp, &
                 1.4825352832265704_dp, &
                 2.136878793265541_dp, &
                 3.2034437811357703_dp, &
                 5.449257489548676_dp/)
      ELSE IF (E_range < 247709.0_dp) THEN
         aw(:) = (/ &
                 2.0394837579119305e-06_dp, &
                 1.1136428689231387e-05_dp, &
                 2.9311026954581983e-05_dp, &
                 6.057395938117077e-05_dp, &
                 0.00011273361636596131_dp, &
                 0.00019945262227642557_dp, &
                 0.00034307020413915784_dp, &
                 0.0005789986244949678_dp, &
                 0.0009626136389075716_dp, &
                 0.0015797047702976012_dp, &
                 0.0025620328180390273_dp, &
                 0.00411027194192014_dp, &
                 0.006527602589322277_dp, &
                 0.010268581467814463_dp, &
                 0.016009798955308914_dp, &
                 0.02475143770962476_dp, &
                 0.03796243042890685_dp, &
                 0.057786831913653906_dp, &
                 0.0873357482301367_dp, &
                 0.1310983577206706_dp, &
                 0.19551814727621097_dp, &
                 0.2897978980472306_dp, &
                 0.4270216153720778_dp, &
                 0.6257183147230295_dp, &
                 0.9120522680792509_dp, &
                 1.3229345707400562_dp, &
                 1.9105866829861018_dp, &
                 2.7496652071687744_dp, &
                 3.9496659363104385_dp, &
                 5.680486965184195_dp, &
                 8.239732353995374_dp, &
                 12.318415719148392_dp, &
                 5.276628830314947e-06_dp, &
                 1.3196668481729192e-05_dp, &
                 2.379458469172375e-05_dp, &
                 3.999888381841533e-05_dp, &
                 6.655277116013134e-05_dp, &
                 0.00011053727387478837_dp, &
                 0.00018248747181848964_dp, &
                 0.00029842809341232193_dp, &
                 0.0004828462223872788_dp, &
                 0.0007729214376067976_dp, &
                 0.001224631278196218_dp, &
                 0.001921577765363735_dp, &
                 0.0029876866735399437_dp, &
                 0.0046053732161687655_dp, &
                 0.007041380807008615_dp, &
                 0.010683336283217822_dp, &
                 0.016091204665789092_dp, &
                 0.02406937359680715_dp, &
                 0.03576720002987578_dp, &
                 0.05281872257231177_dp, &
                 0.07753624384014632_dp, &
                 0.11317828438072734_dp, &
                 0.16432147566237754_dp, &
                 0.2373819281103962_dp, &
                 0.34136408980953675_dp, &
                 0.48899079151871744_dp, &
                 0.6985643332266451_dp, &
                 0.9974643386084848_dp, &
                 1.4299370040199422_dp, &
                 2.0783190576822954_dp, &
                 3.138964327630411_dp, &
                 5.375073752754771_dp/)
      ELSE IF (E_range < 452410.0_dp) THEN
         aw(:) = (/ &
                 1.2029417634752254e-06_dp, &
                 6.609895643735541e-06_dp, &
                 1.761886326860991e-05_dp, &
                 3.714701435066809e-05_dp, &
                 7.09606454750247e-05_dp, &
                 0.00012924586800944736_dp, &
                 0.0002289177620104591_dp, &
                 0.00039735575327091586_dp, &
                 0.0006783049971089168_dp, &
                 0.001140918627613451_dp, &
                 0.0018934100817969098_dp, &
                 0.0031034810098270562_dp, &
                 0.0050286516433828725_dp, &
                 0.008060966671683137_dp, &
                 0.012792436500279173_dp, &
                 0.020110206423758527_dp, &
                 0.03133410211483852_dp, &
                 0.04841425570428367_dp, &
                 0.07421348496567018_dp, &
                 0.11290867950409522_dp, &
                 0.1705586323733128_dp, &
                 0.2559040285419376_dp, &
                 0.38149111837813177_dp, &
                 0.5652486706311622_dp, &
                 0.8327086526705059_dp, &
                 1.2201713489652732_dp, &
                 1.7793482729282073_dp, &
                 2.5845851027859807_dp, &
                 3.745354303892851_dp, &
                 5.4318267387256425_dp, &
                 7.941927006849686_dp, &
                 11.965772395891321_dp, &
                 3.1166015808204985e-06_dp, &
                 7.893584647537054e-06_dp, &
                 1.4590421523143668e-05_dp, &
                 2.540166835137321e-05_dp, &
                 4.38866273658644e-05_dp, &
                 7.544235506820306e-05_dp, &
                 0.00012836853819339647_dp, &
                 0.0002156510284757784_dp, &
                 0.00035755049600582473_dp, &
                 0.0005853791673295657_dp, &
                 0.0009470589489445969_dp, &
                 0.0015152726995104847_dp, &
                 0.0023993396733646157_dp, &
                 0.0037624022005128713_dp, &
                 0.005846143988201738_dp, &
                 0.009006130762738474_dp, &
                 0.01376205648174844_dp, &
                 0.02086880821287917_dp, &
                 0.03141648831640661_dp, &
                 0.04697057987136505_dp, &
                 0.06976767926454802_dp, &
                 0.10298830876584288_dp, &
                 0.1511376630263013_dp, &
                 0.2205811788480397_dp, &
                 0.320313759532162_dp, &
                 0.4631152216015924_dp, &
                 0.6674360776688134_dp, &
                 0.9609048344557064_dp, &
                 1.3880820200304151_dp, &
                 2.031544618845653_dp, &
                 3.087377771089118_dp, &
                 5.31578682998986_dp/)
      ELSE IF (E_range < 1104308.0_dp) THEN
         aw(:) = (/ &
                 5.514768391533348e-07_dp, &
                 3.0661463050989084e-06_dp, &
                 8.370607042832422e-06_dp, &
                 1.830793322995512e-05_dp, &
                 3.65866489044243e-05_dp, &
                 6.987140555359488e-05_dp, &
                 0.00012955245746241426_dp, &
                 0.00023469156977383316_dp, &
                 0.00041673066301572737_dp, &
                 0.0007268469188482661_dp, &
                 0.0012473021552990137_dp, &
                 0.002108784237240361_dp, &
                 0.0035166643543472435_dp, &
                 0.005790416868049286_dp, &
                 0.009422327737378779_dp, &
                 0.015164272167993003_dp, &
                 0.02415507435487594_dp, &
                 0.038106181361843135_dp, &
                 0.05957064947730835_dp, &
                 0.09233052534676135_dp, &
                 0.14195168557080626_dp, &
                 0.2165746550862792_dp, &
                 0.3280373760158957_dp, &
                 0.493465990037226_dp, &
                 0.7375325370826453_dp, &
                 1.0956894156446793_dp, &
                 1.6189196862917987_dp, &
                 2.381098691427177_dp, &
                 3.4916207819996172_dp, &
                 5.1210060943387665_dp, &
                 7.5676555925110085_dp, &
                 11.520669806145102_dp, &
                 1.4324492833187423e-06_dp, &
                 3.714945365379106e-06_dp, &
                 7.187993199450151e-06_dp, &
                 1.3287667789384625e-05_dp, &
                 2.435221365340668e-05_dp, &
                 4.406786544877652e-05_dp, &
                 7.839477316844559e-05_dp, &
                 0.00013700432533564813_dp, &
                 0.00023541888373746023_dp, &
                 0.0003982417143724247_dp, &
                 0.0006640223423546365_dp, &
                 0.0010925276402432504_dp, &
                 0.0017755203327472222_dp, &
                 0.0028526152739405167_dp, &
                 0.0045344377578051915_dp, &
                 0.007136218994535957_dp, &
                 0.011126225466246156_dp, &
                 0.01719515628222213_dp, &
                 0.026355033830470446_dp, &
                 0.04007940751399853_dp, &
                 0.06050127009572156_dp, &
                 0.09069160330701329_dp, &
                 0.13505126572297815_dp, &
                 0.19986523979796597_dp, &
                 0.2940997147198093_dp, &
                 0.4305938358212778_dp, &
                 0.6279833290723216_dp, &
                 0.9142278655051918_dp, &
                 1.3343266294937697_dp, &
                 1.971223529289826_dp, &
                 3.0207289388880563_dp, &
                 5.239269717116346_dp/)
      ELSE IF (E_range < 2582180.0_dp) THEN
         aw(:) = (/ &
                 2.6346628583772034e-07_dp, &
                 1.4870682611802608e-06_dp, &
                 4.18499777462391e-06_dp, &
                 9.565754079998203e-06_dp, &
                 2.0095438915823317e-05_dp, &
                 4.030712638205273e-05_dp, &
                 7.820310873891258e-05_dp, &
                 0.0001476189063429688_dp, &
                 0.00027206125557350966_dp, &
                 0.0004908276742423054_dp, &
                 0.0008686422030538531_dp, &
                 0.0015106576888148007_dp, &
                 0.0025855700579017434_dp, &
                 0.0043608859416256075_dp, &
                 0.007256247781115456_dp, &
                 0.011923381221441286_dp, &
                 0.01936500732097008_dp, &
                 0.031110395247887928_dp, &
                 0.049472724953768164_dp, &
                 0.07792391423249093_dp, &
                 0.12163719777397036_dp, &
                 0.18826820696011917_dp, &
                 0.2890741916661454_dp, &
                 0.44051299024821017_dp, &
                 0.6665282145336267_dp, &
                 1.0018392157983365_dp, &
                 1.4967815518731842_dp, &
                 2.2247902473537535_dp, &
                 3.295151802440413_dp, &
                 4.87866127574686_dp, &
                 7.274154425483326_dp, &
                 11.170031678415262_dp, &
                 6.865691087424058e-07_dp, &
                 1.8349687179642652e-06_dp, &
                 3.75363535847515e-06_dp, &
                 7.406302503889748e-06_dp, &
                 1.4385998863803715e-05_dp, &
                 2.7332856632656454e-05_dp, &
                 5.070293872960844e-05_dp, &
                 9.194731882902252e-05_dp, &
                 0.00016333066755487376_dp, &
                 0.0002847458523672339_dp, &
                 0.0004880328440654819_dp, &
                 0.0008235400684252141_dp, &
                 0.001370000313997599_dp, &
                 0.002249268185463258_dp, &
                 0.003648135807576948_dp, &
                 0.005850384133560756_dp, &
                 0.009283540163912576_dp, &
                 0.01458663604211278_dp, &
                 0.022707798579110823_dp, &
                 0.03504400012572891_dp, &
                 0.05364018073213217_dp, &
                 0.08147187176803067_dp, &
                 0.12284570471579664_dp, &
                 0.18396885882280342_dp, &
                 0.27376977104743744_dp, &
                 0.4051220237008429_dp, &
                 0.5968036286827566_dp, &
                 0.8770477402799991_dp, &
                 1.2912341021707199_dp, &
                 1.9226515113345246_dp, &
                 2.9669527288987725_dp, &
                 5.177596037459169_dp/)
      ELSE IF (E_range < 10786426.0_dp) THEN
         aw(:) = (/ &
                 7.696764691509552e-08_dp, &
                 4.5183075501644766e-07_dp, &
                 1.3712733197245156e-06_dp, &
                 3.452398063116915e-06_dp, &
                 7.99119238288884e-06_dp, &
                 1.7499890158507848e-05_dp, &
                 3.668225172908821e-05_dp, &
                 7.411367023252019e-05_dp, &
                 0.0001450609740104884_dp, &
                 0.0002761333471939282_dp, &
                 0.0005128366382617873_dp, &
                 0.0009316733536716902_dp, &
                 0.0016592776527650684_dp, &
                 0.0029023227583525545_dp, &
                 0.004993762465329047_dp, &
                 0.008463615785356367_dp, &
                 0.014146318917564842_dp, &
                 0.02334212828021178_dp, &
                 0.03805782856888779_dp, &
                 0.0613630042242151_dp, &
                 0.0979136503362365_dp, &
                 0.15471676332581316_dp, &
                 0.24224053787692895_dp, &
                 0.3760196167953944_dp, &
                 0.5789730031642062_dp, &
                 0.8847668378302425_dp, &
                 1.3427798213464026_dp, &
                 2.0257631969638736_dp, &
                 3.042786371879784_dp, &
                 4.564995894383141_dp, &
                 6.891880055654338_dp, &
                 10.711074682469496_dp, &
                 2.0224960490304485e-07_dp, &
                 5.837519191389681e-07_dp, &
                 1.3534097826701593e-06_dp, &
                 3.0163324999016133e-06_dp, &
                 6.464297707355912e-06_dp, &
                 1.331006273228438e-05_dp, &
                 2.6438493784462622e-05_dp, &
                 5.089377871159909e-05_dp, &
                 9.53135908336066e-05_dp, &
                 0.00017421956824809404_dp, &
                 0.0003116271902197154_dp, &
                 0.0005466663994455823_dp, &
                 0.0009422361350401303_dp, &
                 0.0015981946213501235_dp, &
                 0.0026712728244169207_dp, &
                 0.004404875333973468_dp, &
                 0.00717331310959028_dp, &
                 0.011546952692401411_dp, &
                 0.018387490277830258_dp, &
                 0.028986360940377882_dp, &
                 0.04526460807725196_dp, &
                 0.07006006488175816_dp, &
                 0.10753871373789183_dp, &
                 0.16378446149863707_dp, &
                 0.24765288245177752_dp, &
                 0.3720420890669611_dp, &
                 0.5559074354965244_dp, &
                 0.8278548563568215_dp, &
                 1.2338117262467065_dp, &
                 1.8576021658437778_dp, &
                 2.8947650724141107_dp, &
                 5.094894347251689_dp/)
      ELSE IF (E_range < 72565710.0_dp) THEN
         aw(:) = (/ &
                 1.5580960122828055e-08_dp, &
                 1.0192299261551187e-07_dp, &
                 3.657511654714275e-07_dp, &
                 1.0874230387694592e-06_dp, &
                 2.8971440093678986e-06_dp, &
                 7.1231112287328e-06_dp, &
                 1.6440376252868036e-05_dp, &
                 3.6035911594207885e-05_dp, &
                 7.564822282610485e-05_dp, &
                 0.0001530604093750471_dp, &
                 0.0002999674552596099_dp, &
                 0.0005716626169231291_dp, &
                 0.0010627819854068627_dp, &
                 0.0019325381417571955_dp, &
                 0.0034446462658281212_dp, &
                 0.006029758821968773_dp, &
                 0.010382043759726605_dp, &
                 0.017607085158632928_dp, &
                 0.02944628039743525_dp, &
                 0.04861435988540585_dp, &
                 0.07930297685700823_dp, &
                 0.12792652074627453_dp, &
                 0.20421936908257549_dp, &
                 0.3228415686086713_dp, &
                 0.5057217947095013_dp, &
                 0.785484422991801_dp, &
                 1.2105330871831885_dp, &
                 1.852886865611021_dp, &
                 2.8213315422092657_dp, &
                 4.2873111269979765_dp, &
                 6.550982976987269_dp, &
                 10.299461972660755_dp, &
                 4.192785461816036e-08_dp, &
                 1.4682586910958697e-07_dp, &
                 4.248767433871547e-07_dp, &
                 1.117972412187448e-06_dp, &
                 2.71251708562224e-06_dp, &
                 6.166664608205133e-06_dp, &
                 1.3299698724334126e-05_dp, &
                 2.7458360437668858e-05_dp, &
                 5.463565964132874e-05_dp, &
                 0.00010531494807998692_dp, &
                 0.0001974602213268062_dp, &
                 0.0003612942851684024_dp, &
                 0.0006468325897487988_dp, &
                 0.001135618706607295_dp, &
                 0.0019588032650003236_dp, &
                 0.003324710089253406_dp, &
                 0.005560469162003084_dp, &
                 0.009174344598623402_dp, &
                 0.014948285309521006_dp, &
                 0.024074306780617383_dp, &
                 0.03835406948576763_dp, &
                 0.060489181968959096_dp, &
                 0.0945015966761899_dp, &
                 0.1463417320393485_dp, &
                 0.22477371348415012_dp, &
                 0.34269334477792374_dp, &
                 0.5192018690921564_dp, &
                 0.7832512032052583_dp, &
                 1.1813099794431254_dp, &
                 1.7977731364021527_dp, &
                 2.828182155100705_dp, &
                 5.018697893750251_dp/)
      ELSE
         aw(:) = (/ &
                 1.4796558719757977e-09_dp, &
                 1.6277762282636202e-08_dp, &
                 8.615102100471783e-08_dp, &
                 3.3251166768869524e-07_dp, &
                 1.064831343891703e-06_dp, &
                 3.0042406722730696e-06_dp, &
                 7.722404719947578e-06_dp, &
                 1.8469628120696557e-05_dp, &
                 4.168720676495336e-05_dp, &
                 8.969407716668371e-05_dp, &
                 0.00018534670944540425_dp, &
                 0.00036995434452620024_dp, &
                 0.0007164776276992899_dp, &
                 0.0013511756947926401_dp, &
                 0.0024885840769342527_dp, &
                 0.004487272639985271_dp, &
                 0.007937634518341114_dp, &
                 0.013798542180092185_dp, &
                 0.023607852053926744_dp, &
                 0.03980353161167122_dp, &
                 0.06620914631345257_dp, &
                 0.10876176019481658_dp, &
                 0.17659514989891748_dp, &
                 0.28364164912494744_dp, &
                 0.45099127302133024_dp, &
                 0.7103683461225575_dp, &
                 1.109312567732835_dp, &
                 1.7191679096665389_dp, &
                 2.648421848786194_dp, &
                 4.06873544538514_dp, &
                 6.280858212318488_dp, &
                 9.971616727521422_dp, &
                 4.7301698039073534e-09_dp, &
                 3.1078236876760897e-08_dp, &
                 1.2722293347158736e-07_dp, &
                 4.135773629354279e-07_dp, &
                 1.1643319139692173e-06_dp, &
                 2.9633030798047164e-06_dp, &
                 6.990659700961397e-06_dp, &
                 1.5534394085699292e-05_dp, &
                 3.28792052583233e-05_dp, &
                 6.681670495514343e-05_dp, &
                 0.00013116076656301764_dp, &
                 0.0002498651223560153_dp, &
                 0.0004636589258144925_dp, &
                 0.0008405899279454757_dp, &
                 0.0014925644239063773_dp, &
                 0.0026009920182698686_dp, &
                 0.004456120959091793_dp, &
                 0.00751677674389606_dp, &
                 0.012500261250050913_dp, &
                 0.020516489982978243_dp, &
                 0.03326657617193578_dp, &
                 0.053334802223457274_dp, &
                 0.08461553683522265_dp, &
                 0.13293582624456382_dp, &
                 0.20696680596446562_dp, &
                 0.31958257452387945_dp, &
                 0.4899879264178536_dp, &
                 0.7474157763110646_dp, &
                 1.1388011986618074_dp, &
                 1.7490627041810483_dp, &
                 2.7738272660229994_dp, &
                 4.9565545529977175_dp/)
      END IF
   END SUBROUTINE

! **************************************************************************************************
!> \brief ...
!> \param k ...
!> \param E_range ...
!> \param aw ...
!> \param E_ratio ...
! **************************************************************************************************
   PURE SUBROUTINE get_coeff_34(k, E_range, aw, E_ratio)
      INTEGER, INTENT(IN)                                :: k
      REAL(KIND=dp), INTENT(IN)                          :: E_range
      REAL(KIND=dp), DIMENSION(2*k), INTENT(OUT)         :: aw
      REAL(KIND=dp), INTENT(OUT)                         :: E_ratio

      E_ratio = 1.0_dp
      IF (E_range < 9649.0_dp) THEN
         E_ratio = 9649.0_dp/E_range
         aw(:) = (/ &
                 3.284029233271286e-05_dp, &
                 0.0001753443602861097_dp, &
                 0.0004416382833363001_dp, &
                 0.0008508530310865113_dp, &
                 0.001435500605855746_dp, &
                 0.0022474948219129297_dp, &
                 0.003366173371324329_dp, &
                 0.004907942966760921_dp, &
                 0.007037748206337417_dp, &
                 0.009983831117905245_dp, &
                 0.014057973412231587_dp, &
                 0.019683414844193538_dp, &
                 0.02743271515183254_dp, &
                 0.03807838736551588_dp, &
                 0.052660091008446035_dp, &
                 0.07257342975355677_dp, &
                 0.09968697353631621_dp, &
                 0.13649612296545818_dp, &
                 0.18632500099011615_dp, &
                 0.2535908886542759_dp, &
                 0.3441500984275648_dp, &
                 0.46575004766668_dp, &
                 0.6286204407593304_dp, &
                 0.8462483877785438_dp, &
                 1.1364010501626234_dp, &
                 1.5224916494827863_dp, &
                 2.035445381255887_dp, &
                 2.716346186249304_dp, &
                 3.620421040738206_dp, &
                 4.823581438788581_dp, &
                 6.434509907627798_dp, &
                 8.620766164346367_dp, &
                 11.679090118639257_dp, &
                 16.31432816585662_dp, &
                 8.4538052355896e-05_dp, &
                 0.00020209318539166847_dp, &
                 0.00033361605486797254_dp, &
                 0.0004901319572083779_dp, &
                 0.0006876803644732437_dp, &
                 0.0009493703556928221_dp, &
                 0.0013072005549165328_dp, &
                 0.0018035890085942577_dp, &
                 0.0024937295684409103_dp, &
                 0.0034499397324379713_dp, &
                 0.00476826036404788_dp, &
                 0.0065772141488150675_dp, &
                 0.009049030640506878_dp, &
                 0.012414126010019873_dp, &
                 0.016979951815415794_dp, &
                 0.02315561592334051_dp, &
                 0.031484041042937_dp, &
                 0.04268391545844468_dp, &
                 0.05770434764295233_dp, &
                 0.07779602216248598_dp, &
                 0.10460388481675117_dp, &
                 0.14028819232415718_dp, &
                 0.18768361957484037_dp, &
                 0.2505110517811862_dp, &
                 0.33366599536844854_dp, &
                 0.4436264870131247_dp, &
                 0.5890644515470643_dp, &
                 0.7818386670916874_dp, &
                 1.0387776399865274_dp, &
                 1.3852730956246349_dp, &
                 1.8635541689521873_dp, &
                 2.5552790973254864_dp, &
                 3.661647560244881_dp, &
                 5.979097365202448_dp/)
      ELSE IF (E_range < 15161.0_dp) THEN
         aw(:) = (/ &
                 2.2094687262226044e-05_dp, &
                 0.00011816317101145329_dp, &
                 0.00029854362986100777_dp, &
                 0.0005779739014899124_dp, &
                 0.0009819112670486493_dp, &
                 0.0015516034173501654_dp, &
                 0.0023507187818878183_dp, &
                 0.0034732685500535195_dp, &
                 0.00505334279364789_dp, &
                 0.007278229326144087_dp, &
                 0.010406819589614869_dp, &
                 0.014795161155330522_dp, &
                 0.020931286923915934_dp, &
                 0.029482154546951286_dp, &
                 0.04135653654321724_dp, &
                 0.05778898092867048_dp, &
                 0.08045158704880996_dp, &
                 0.11160243319073283_dp, &
                 0.15428220578421106_dp, &
                 0.2125741178044686_dp, &
                 0.2919468573365305_dp, &
                 0.3997065271731107_dp, &
                 0.545592087818525_dp, &
                 0.7425611332686428_dp, &
                 1.0078318394632224_dp, &
                 1.3642789958888877_dp, &
                 1.8423416665523202_dp, &
                 2.48272174251732_dp, &
                 3.34042303033154_dp, &
                 4.49133411535853_dp, &
                 6.044309582675748_dp, &
                 8.167153038233048_dp, &
                 11.156471603344267_dp, &
                 15.714736908294974_dp, &
                 5.689777248462453e-05_dp, &
                 0.0001364647878286462_dp, &
                 0.0002266959149215116_dp, &
                 0.0003363227347021138_dp, &
                 0.0004783149170860289_dp, &
                 0.0006715498663212837_dp, &
                 0.0009421913049362618_dp, &
                 0.0013250426248297155_dp, &
                 0.0018660179676916586_dp, &
                 0.00262648268480602_dp, &
                 0.003689465255982615_dp, &
                 0.005167783411766887_dp, &
                 0.007214591621086576_dp, &
                 0.010037222243044243_dp, &
                 0.013915457617770334_dp, &
                 0.01922566776214666_dp, &
                 0.02647264919678102_dp, &
                 0.03633153858613665_dp, &
                 0.049702887216291285_dp, &
                 0.06778493328431712_dp, &
                 0.09216841458681874_dp, &
                 0.12496114099205212_dp, &
                 0.1689524368948172_dp, &
                 0.22783243637559455_dp, &
                 0.3064902609254688_dp, &
                 0.411433435566918_dp, &
                 0.5514107558972641_dp, &
                 0.7384129467740507_dp, &
                 0.9894523508058952_dp, &
                 1.3301301240547123_dp, &
                 1.8028378573543902_dp, &
                 2.489168614889156_dp, &
                 3.5894672490279578_dp, &
                 5.895310588871672_dp/)
      ELSE IF (E_range < 29986.0_dp) THEN
         aw(:) = (/ &
                 1.2132019239369146e-05_dp, &
                 6.506554392462203e-05_dp, &
                 0.00016528151320033429_dp, &
                 0.00032271987439700723_dp, &
                 0.0005549904907647636_dp, &
                 0.0008912186380916402_dp, &
                 0.0013769333304078336_dp, &
                 0.00207998381188429_dp, &
                 0.0030983248733590404_dp, &
                 0.004571097441508312_dp, &
                 0.006694456763746173_dp, &
                 0.009743704668641925_dp, &
                 0.0141037713916676_dp, &
                 0.02031087384545962_dp, &
                 0.029109193553804015_dp, &
                 0.04152772200656733_dp, &
                 0.05898411878582171_dp, &
                 0.08342465247911345_dp, &
                 0.11751221427114543_dp, &
                 0.1648782324800013_dp, &
                 0.23045938623741916_dp, &
                 0.3209467823295079_dp, &
                 0.44538447707509593_dp, &
                 0.6159672665711226_dp, &
                 0.8491072823533072_dp, &
                 1.1668710896432313_dp, &
                 1.5989474740844776_dp, &
                 2.1854242653839444_dp, &
                 2.9809152725739025_dp, &
                 4.061217864274067_dp, &
                 5.535407811609172_dp, &
                 7.571672639611632_dp, &
                 10.466564989305379_dp, &
                 14.91952959384994_dp, &
                 3.126217597574477e-05_dp, &
                 7.540587737410632e-05_dp, &
                 0.00012663926218204873_dp, &
                 0.00019110769108438634_dp, &
                 0.00027820581560489743_dp, &
                 0.000401742249839652_dp, &
                 0.0005808693019056328_dp, &
                 0.0008413761481210849_dp, &
                 0.0012182358494751703_dp, &
                 0.001759637722089599_dp, &
                 0.00253246317609668_dp, &
                 0.003629502407336348_dp, &
                 0.00517906799440672_dp, &
                 0.0073579025027302025_dp, &
                 0.01040852585005309_dp, &
                 0.014662503493069159_dp, &
                 0.020571571811010672_dp, &
                 0.028749161300949972_dp, &
                 0.04002565337439286_dp, &
                 0.05552176300115481_dp, &
                 0.07674587007131725_dp, &
                 0.10572312939601615_dp, &
                 0.14516718239506418_dp, &
                 0.19871016099008434_dp, &
                 0.2712154487579484_dp, &
                 0.3692151806287191_dp, &
                 0.5015525805409441_dp, &
                 0.6803982020135383_dp, &
                 0.9230308920142081_dp, &
                 1.2553734442165063_dp, &
                 1.7200955901449364_dp, &
                 2.3987730554757754_dp, &
                 3.4906629653715_dp, &
                 5.780815874478559_dp/)
      ELSE IF (E_range < 49196.0_dp) THEN
         aw(:) = (/ &
                 7.845531705064765e-06_dp, &
                 4.217579566703759e-05_dp, &
                 0.0001076255406955941_dp, &
                 0.0002116704074840506_dp, &
                 0.0003678026352786526_dp, &
                 0.000598637239972256_dp, &
                 0.000939800791185946_dp, &
                 0.0014447771966513766_dp, &
                 0.002191627535252642_dp, &
                 0.0032927972200630018_dp, &
                 0.004909216880554122_dp, &
                 0.007270144108734435_dp, &
                 0.010700753084513917_dp, &
                 0.015660261119126325_dp, &
                 0.02279439226673758_dp, &
                 0.033007300151380946_dp, &
                 0.04755981744820865_dp, &
                 0.06820320985515761_dp, &
                 0.09736066855556012_dp, &
                 0.13837281589809153_dp, &
                 0.19582885861759528_dp, &
                 0.2760121868265214_dp, &
                 0.3874989458973457_dp, &
                 0.5419617440352286_dp, &
                 0.7552508475941064_dp, &
                 1.048857678610048_dp, &
                 1.4519234318259462_dp, &
                 2.004071631268318_dp, &
                 2.759600998924181_dp, &
                 3.7942044073242056_dp, &
                 5.217087596451319_dp, &
                 7.196718451225344_dp, &
                 10.029695823161129_dp, &
                 14.413626849760721_dp, &
                 2.022742080929783e-05_dp, &
                 4.902127023830298e-05_dp, &
                 8.308838409631613e-05_dp, &
                 0.00012719755797047105_dp, &
                 0.00018877141608928316_dp, &
                 0.00027875729779708194_dp, &
                 0.00041237481957666014_dp, &
                 0.0006104274314639044_dp, &
                 0.0009017824485847016_dp, &
                 0.0013270526990110587_dp, &
                 0.0019435429478184875_dp, &
                 0.0028318806888766317_dp, &
                 0.004105011117474572_dp, &
                 0.0059204430431561185_dp, &
                 0.008496895591229079_dp, &
                 0.012136858469163465_dp, &
                 0.017257067102487582_dp, &
                 0.02442954195917895_dp, &
                 0.03443669419909579_dp, &
                 0.04834513188831433_dp, &
                 0.06760433082481605_dp, &
                 0.09417845588835595_dp, &
                 0.13072271761835513_dp, &
                 0.18082056576074482_dp, &
                 0.2493066841224739_dp, &
                 0.3427177893721846_dp, &
                 0.46995019394607707_dp, &
                 0.6432897947695325_dp, &
                 0.8801980582788952_dp, &
                 1.2068306280775711_dp, &
                 1.6660767737909483_dp, &
                 2.3395502248176463_dp, &
                 3.4258517245052684_dp, &
                 5.7058363778559205_dp/)
      ELSE IF (E_range < 109833.0_dp) THEN
         aw(:) = (/ &
                 3.865603619912879e-06_dp, &
                 2.087430645188353e-05_dp, &
                 5.3738065099243575e-05_dp, &
                 0.00010718454630525007_dp, &
                 0.0001900012319697737_dp, &
                 0.00031718955789873134_dp, &
                 0.0005126537329759226_dp, &
                 0.0008128168395638415_dp, &
                 0.0012719920833382853_dp, &
                 0.001970384125983329_dp, &
                 0.0030256963224515113_dp, &
                 0.00460969356562721_dp, &
                 0.006971643847773047_dp, &
                 0.010471317877795917_dp, &
                 0.01562522911317097_dp, &
                 0.023171141484273704_dp, &
                 0.034157679556524934_dp, &
                 0.05006830174859498_dp, &
                 0.0729921440414609_dp, &
                 0.1058585802344639_dp, &
                 0.1527581466742932_dp, &
                 0.2193802673426605_dp, &
                 0.31360877991591823_dp, &
                 0.4463309353744743_dp, &
                 0.6325368438825444_dp, &
                 0.8928196332263253_dp, &
                 1.2554443726415525_dp, &
                 1.759267106284206_dp, &
                 2.458035348630705_dp, &
                 3.427215114290294_dp, &
                 4.77616847618911_dp, &
                 6.673804689443379_dp, &
                 9.416907604833742_dp, &
                 13.70063546514659_dp, &
                 9.976454765718753e-06_dp, &
                 2.4397956666150202e-05_dp, &
                 4.2091633541468636e-05_dp, &
                 6.622948123983615e-05_dp, &
                 0.00010186030376456873_dp, &
                 0.00015644233845744574_dp, &
                 0.0002404690130604266_dp, &
                 0.0003687782406776715_dp, &
                 0.0005627493820533361_dp, &
                 0.0008533803933831028_dp, &
                 0.0012854870718438482_dp, &
                 0.0019235107198793406_dp, &
                 0.002859592750775737_dp, &
                 0.004224781787747757_dp, &
                 0.0062045295634641385_dp, &
                 0.009060029072853486_dp, &
                 0.0131574822853581_dp, &
                 0.019008095244250185_dp, &
                 0.02732254096178089_dp, &
                 0.03908488812876366_dp, &
                 0.055652686745107795_dp, &
                 0.07889222315477536_dp, &
                 0.11136127365662019_dp, &
                 0.15655676737113142_dp, &
                 0.21925337945032786_dp, &
                 0.30597553969409247_dp, &
                 0.42568066454670456_dp, &
                 0.5908146138910935_dp, &
                 0.8191119697759165_dp, &
                 1.137096600179962_dp, &
                 1.5880319126786595_dp, &
                 2.2536634913094686_dp, &
                 3.331728541732483_dp, &
                 5.597118014202835_dp/)
      ELSE IF (E_range < 276208.0_dp) THEN
         aw(:) = (/ &
                 1.7147362511403065e-06_dp, &
                 9.319802731855972e-06_dp, &
                 2.4302616807482706e-05_dp, &
                 4.948261860530359e-05_dp, &
                 9.026404229610094e-05_dp, &
                 0.0001560250715046401_dp, &
                 0.00026193159643869907_dp, &
                 0.0004316317370034808_dp, &
                 0.0007014310228413562_dp, &
                 0.0011265811323026616_dp, &
                 0.001790538099368303_dp, &
                 0.002818442710487939_dp, &
                 0.004396613985562227_dp, &
                 0.006800572618267542_dp, &
                 0.01043510374307695_dp, &
                 0.015891226467704685_dp, &
                 0.02402679235817231_dp, &
                 0.03607995947798214_dp, &
                 0.05382821091103021_dp, &
                 0.07981021388476216_dp, &
                 0.11763406509273454_dp, &
                 0.17240391938168603_dp, &
                 0.25130849721857534_dp, &
                 0.36443086121519086_dp, &
                 0.525861591666699_dp, &
                 0.7552321345423689_dp, &
                 1.0798433566303969_dp, &
                 1.5376756421659512_dp, &
                 2.1818096583688225_dp, &
                 3.087384050821325_dp, &
                 4.363863206811109_dp, &
                 6.180624421988004_dp, &
                 8.83477361053772_dp, &
                 13.019312520088521_dp, &
                 4.43185996007463e-06_dp, &
                 1.0980758773489005e-05_dp, &
                 1.943725634861249e-05_dp, &
                 3.1800270744211854e-05_dp, &
                 5.129667367655266e-05_dp, &
                 8.27095880626353e-05_dp, &
                 0.0001329766243411858_dp, &
                 0.00021237174266944332_dp, &
                 0.0003362840713927107_dp, &
                 0.0005277050042553842_dp, &
                 0.0008207467241987074_dp, &
                 0.001265656823748127_dp, &
                 0.001935949524221289_dp, &
                 0.002938498181312597_dp, &
                 0.004427745609890722_dp, &
                 0.00662561356009038_dp, &
                 0.009849268137973679_dp, &
                 0.01454967083492188_dp, &
                 0.021364880820575197_dp, &
                 0.031193465982641635_dp, &
                 0.04529525929852885_dp, &
                 0.06542926430703719_dp, &
                 0.09404211859223464_dp, &
                 0.13452588222462222_dp, &
                 0.19157264817324873_dp, &
                 0.27166955823017896_dp, &
                 0.3838116245690371_dp, &
                 0.5405893273969905_dp, &
                 0.7600124959108089_dp, &
                 1.0690028998164918_dp, &
                 1.5112619233396074_dp, &
                 2.168763561959426_dp, &
                 3.238506003394989_dp, &
                 5.489633460260332_dp/)
      ELSE IF (E_range < 852991.0_dp) THEN
         aw(:) = (/ &
                 6.35749973501056e-07_dp, &
                 3.492169089246581e-06_dp, &
                 9.302356973224121e-06_dp, &
                 1.9592566972831844e-05_dp, &
                 3.7377729183938204e-05_dp, &
                 6.798117415794394e-05_dp, &
                 0.00012023568563760358_dp, &
                 0.0002084236297872551_dp, &
                 0.00035534106272681936_dp, &
                 0.0005969900768905776_dp, &
                 0.0009896587375544644_dp, &
                 0.001620507511517086_dp, &
                 0.0026232770839453665_dp, &
                 0.004201426984363522_dp, &
                 0.006661986770200809_dp, &
                 0.010464757550535931_dp, &
                 0.01629338409394917_dp, &
                 0.025157418466823473_dp, &
                 0.038538075135755495_dp, &
                 0.05859528378548497_dp, &
                 0.088460352930584_dp, &
                 0.13264771717146925_dp, &
                 0.19763178422753508_dp, &
                 0.2926522443111407_dp, &
                 0.4308357751366649_dp, &
                 0.6307586891808089_dp, &
                 0.9186346830441839_dp, &
                 1.3314221217344053_dp, &
                 1.9213813610912545_dp, &
                 2.76319200354934_dp, &
                 3.9663504041897766_dp, &
                 5.700732751780247_dp, &
                 8.263919165830904_dp, &
                 12.346999008174697_dp, &
                 1.646994148539095e-06_dp, &
                 4.168704436960474e-06_dp, &
                 7.695536879782202e-06_dp, &
                 1.3374173961844378e-05_dp, &
                 2.3064311988245763e-05_dp, &
                 3.958321376425686e-05_dp, &
                 6.725628425988863e-05_dp, &
                 0.00011284251346096221_dp, &
                 0.0001868773135187132_dp, &
                 0.0003056293856662244_dp, &
                 0.0004939782736171523_dp, &
                 0.0007896314087630558_dp, &
                 0.0012492643080895956_dp, &
                 0.001957402304692391_dp, &
                 0.0030391875844591482_dp, &
                 0.004678623430285177_dp, &
                 0.007144500300310407_dp, &
                 0.010827043576140467_dp, &
                 0.016289458703073287_dp, &
                 0.024340088611569204_dp, &
                 0.03613298947254673_dp, &
                 0.05330759009262987_dp, &
                 0.07818208570144934_dp, &
                 0.11402099109647866_dp, &
                 0.16540631928634394_dp, &
                 0.23875782212611196_dp, &
                 0.3430802599354601_dp, &
                 0.49109150628018977_dp, &
                 0.7010818269759505_dp, &
                 1.0004112023843146_dp, &
                 1.4333015739126787_dp, &
                 2.0820719815510036_dp, &
                 3.1430997613534135_dp, &
                 5.379828456565977_dp/)
      ELSE
         aw(:) = (/ &
                 1.117489994479298e-09_dp, &
                 1.0437767934573854e-08_dp, &
                 5.084528505414382e-08_dp, &
                 1.872213850467172e-07_dp, &
                 5.813350591395648e-07_dp, &
                 1.6041946888512065e-06_dp, &
                 4.054564459005386e-06_dp, &
                 9.568204612679587e-06_dp, &
                 2.1360561665920905e-05_dp, &
                 4.553900601252074e-05_dp, &
                 9.336828090817695e-05_dp, &
                 0.00018510270750313838_dp, &
                 0.0003563527131882449_dp, &
                 0.0006684922084643366_dp, &
                 0.0012254246986170441_dp, &
                 0.002200236453707954_dp, &
                 0.0038770557829877106_dp, &
                 0.006716063190006149_dp, &
                 0.011453417554768287_dp, &
                 0.019253386304452922_dp, &
                 0.0319378960087355_dp, &
                 0.052330033425615166_dp, &
                 0.0847641054822857_dp, &
                 0.13583764990140365_dp, &
                 0.21551319196591645_dp, &
                 0.3387243798072503_dp, &
                 0.527711765290698_dp, &
                 0.8154303015314811_dp, &
                 1.2505960123470905_dp, &
                 1.9054661547111273_dp, &
                 2.8889237961629872_dp, &
                 4.37232400643935_dp, &
                 6.65560984092506_dp, &
                 10.426038695834771_dp, &
                 3.3457340053937633e-09_dp, &
                 1.873386519501546e-08_dp, &
                 7.20396391424945e-08_dp, &
                 2.259124367219339e-07_dp, &
                 6.20700398368113e-07_dp, &
                 1.5516519034576341e-06_dp, &
                 3.6098773524770243e-06_dp, &
                 7.932321947059267e-06_dp, &
                 1.6633942623129364e-05_dp, &
                 3.353876533808818e-05_dp, &
                 6.539287755778778e-05_dp, &
                 0.00012384295496778607_dp, &
                 0.00022861478182158466_dp, &
                 0.0004125495803926801_dp, &
                 0.0007294839100515848_dp, &
                 0.0012664381661580966_dp, &
                 0.002162272032202245_dp, &
                 0.003635959152010226_dp, &
                 0.006029053426126615_dp, &
                 0.009868936147464018_dp, &
                 0.015962272528537477_dp, &
                 0.025532104472748035_dp, &
                 0.040417626126145245_dp, &
                 0.06336363493503006_dp, &
                 0.09843823938908386_dp, &
                 0.1516353587304579_dp, &
                 0.23175010955983177_dp, &
                 0.35168177852781385_dp, &
                 0.5304884039781542_dp, &
                 0.7970145382697631_dp, &
                 1.197557199061726_dp, &
                 1.8163260289376923_dp, &
                 2.848850144582029_dp, &
                 5.042342061181277_dp/)
      END IF
   END SUBROUTINE

END MODULE minimax_exp_gw
