## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  warning = FALSE,
  message = FALSE
)

library(cowfootR)
library(knitr)

## ----echo=FALSE---------------------------------------------------------------
area_breakdown_options <- data.frame(
  Name = c("pasture_permanent", "pasture_temporary", "crops_feed", "crops_cash", 
           "infrastructure", "woodland", "wetlands", "other"),
  Description = c("Permanent grassland", "Rotational/temporary pasture", "Feed crop production",
                  "Cash crop production", "Buildings, roads, facilities", "Forest/trees",
                  "Water bodies, wetlands", "Other non-productive areas"),
  Typical_Range = c("40-80%", "5-20%", "5-15%", "0-10%", "2-5%", "0-10%", "0-5%", "0-5%")
)

kable(area_breakdown_options, caption = "Valid area_breakdown Names and Descriptions")

## ----echo=FALSE---------------------------------------------------------------
template_structure <- data.frame(
  Column_Group = c(
    rep("Identification", 2),
    rep("Production", 4),
    rep("Herd_Composition", 5),
    rep("Animal_Weights", 4),
    rep("Feed_Management", 5),
    rep("Land_Use", 8),
    rep("Inputs", 3),
    rep("Energy", 6),
    rep("Management", 1)
  ),
  Column_Name = c(
    "FarmID", "Year",
    "Milk_litres", "Fat_percent", "Protein_percent", "Milk_density", 
    "Cows_milking", "Cows_dry", "Heifers_total", "Calves_total", "Bulls_total",
    "Body_weight_cows_kg", "Body_weight_heifers_kg", "Body_weight_calves_kg", "Body_weight_bulls_kg",
    "MS_intake_cows_kg_day", "MS_intake_heifers_kg_day", "MS_intake_calves_kg_day", 
    "MS_intake_bulls_kg_day", "Ym_percent",
    "Area_total_ha", "Area_productive_ha", "Pasture_permanent_ha", "Pasture_temporary_ha",
    "Crops_feed_ha", "Crops_cash_ha", "Infrastructure_ha", "Woodland_ha",
    "N_fertilizer_kg", "Concentrate_feed_kg", "Plastic_kg",
    "Diesel_litres", "Petrol_litres", "Electricity_kWh", "LPG_kg", "Natural_gas_m3", "Country",
    "Manure_system"
  ),
  Data_Type = c(
    "character", "character",
    "numeric", "numeric", "numeric", "numeric",
    "numeric", "numeric", "numeric", "numeric", "numeric", 
    "numeric", "numeric", "numeric", "numeric",
    "numeric", "numeric", "numeric", "numeric", "numeric",
    "numeric", "numeric", "numeric", "numeric", "numeric", "numeric", "numeric", "numeric",
    "numeric", "numeric", "numeric",
    "numeric", "numeric", "numeric", "numeric", "numeric", "character",
    "character"
  ),
  Required = c(
    "Yes", "No",
    "Yes", "No", "No", "No",
    "Yes", "No", "No", "No", "No",
    "No", "No", "No", "No",
    "Tier 2", "Tier 2", "Tier 2", "Tier 2", "Tier 2",
    "Yes", "No", "No", "No", "No", "No", "No", "No",
    "No", "No", "No",
    "No", "No", "No", "No", "No", "No",
    "No"
  )
)

kable(head(template_structure, 15), caption = "Template Structure (First 15 columns)")

## ----echo=FALSE---------------------------------------------------------------
validation_checks <- data.frame(
  Parameter_Type = c("Animal Numbers", "Production Metrics", "Area Data", "Input Quantities", "Ratios"),
  Validation_Rules = c(
    "Must be positive integers",
    "Milk yield 1000-15000 kg/cow/year",
    "Area breakdown must sum to total (if validate=TRUE)",
    "All quantities ≥ 0", 
    "Stocking rate 0.1-3.0 cows/ha"
  ),
  Error_Actions = c(
    "Stop execution with error message",
    "Warning with guidance on typical ranges",
    "Stop or warn based on validate_area_sum setting",
    "Stop with error message",
    "Warning about unusual values"
  ),
  User_Guidance = c(
    "Check data entry and farm records",
    "Verify annual vs daily units",
    "Review land use classification",
    "Check for data entry errors",
    "Confirm farm characteristics"
  )
)

kable(validation_checks, caption = "Built-in Validation Rules")

## ----echo=FALSE---------------------------------------------------------------
quality_indicators <- data.frame(
  Indicator = c("Milk yield per cow", "Stocking rate", "Feed conversion", "Energy intensity"),
  Formula = c("Milk_litres / Cows_milking / 1000", "Cows_milking / Area_total_ha", 
              "Milk_litres / Concentrate_feed_kg", "Electricity_kWh / Milk_litres"),
  Excellent_Range = c("7000-9000", "1.2-1.8", "3.0-5.0", "0.04-0.06"),
  Good_Range = c("6000-7000", "0.8-1.2", "2.0-3.0", "0.06-0.08"),
  Poor_Range = c("<5000 or >10000", "<0.5 or >2.5", "<1.5 or >6.0", ">0.10"),
  Unit = c("kg/cow/year", "cows/ha", "L milk/kg conc", "kWh/L milk")
)

kable(quality_indicators, caption = "Data Quality Assessment Indicators")

## ----echo=FALSE---------------------------------------------------------------
missing_data_guide <- data.frame(
  Missing_Parameter = c("Body weights", "DM intake", "Feed breakdown", "Area breakdown", "Ym factor"),
  Default_Used = c("Species-specific defaults", "Calculated from body weight", "Concentrate only", "Total area only", "6.5%"),
  Accuracy_Impact = c("Low", "Medium", "High", "Medium", "Medium"),
  Recommended_Action = c(
    "Use literature values for breed/region",
    "Estimate from feeding standards", 
    "Collect detailed feed records",
    "Survey farm land use patterns",
    "Use regional studies or 6.0-6.8 range"
  )
)

kable(missing_data_guide, caption = "Handling Missing Parameters")

## ----echo=FALSE---------------------------------------------------------------
conversion_guide <- data.frame(
  Parameter = c("Milk production", "Feed amounts", "Fertilizer", "Body weight", "Area"),
  Common_Units = c("L, kg", "kg fresh, kg DM, tons", "kg product, kg N", "kg, lbs", "ha, acres"),
  cowfootR_Unit = c("L/year", "kg DM/year", "kg N/year", "kg", "hectares"),
  Conversion_Factor = c("kg = L × density", "DM = fresh × (1 - moisture%)", 
                       "kg N = kg product × N%", "kg = lbs ÷ 2.205", "ha = acres × 0.405"),
  Typical_Values = c("1.03 kg/L", "35% DM corn silage", "46% N in urea", "580 kg dairy cow", "0.405 ha/acre")
)

kable(conversion_guide, caption = "Unit Conversion Reference")

## ----echo=FALSE---------------------------------------------------------------
regional_adjustments <- data.frame(
  Region = c("EU", "US", "Brazil", "Argentina", "Australia", "Global"),
  Soy_EF_Range = c("2.1-3.2", "1.2-2.2", "0.9-1.6", "0.8-1.5", "1.8-3.0", "1.5-2.8"),
  Fertilizer_EF = c("5.8-7.9", "5.3-7.6", "6.0-8.3", "5.8-8.1", "5.4-7.7", "5.5-7.8"),
  Key_Differences = c(
    "High soy transport costs",
    "Domestic grain production", 
    "Local soy, high N fertilizer",
    "Local grain/soy production",
    "High transport distances",
    "Average of all regions"
  ),
  Use_When = c(
    "European farms", "US/Canadian farms", "Brazilian operations", 
    "Argentinian farms", "Australian/NZ farms", "Unknown/mixed sourcing"
  )
)

kable(regional_adjustments, caption = "Regional Emission Factor Variations")

## ----echo=FALSE---------------------------------------------------------------
high_impact <- data.frame(
  Parameter = c("n_animals", "milk_litres", "conc_kg", "ym_percent", "avg_body_weight"),
  Function = c("enteric", "intensity", "inputs", "enteric", "enteric"),
  Impact_Direction = c("Linear", "Inverse", "Linear", "Linear", "Linear"),
  Typical_Variation = c("±20%", "±25%", "±30%", "±15%", "±10%"),
  Result_Sensitivity = c("±20%", "±25%", "±25%", "±15%", "±8%"),
  Data_Priority = c("High", "High", "High", "Medium", "Medium")
)

kable(high_impact, caption = "High Impact Parameters (Priority for Accurate Data)")

## ----echo=FALSE---------------------------------------------------------------
medium_impact <- data.frame(
  Parameter = c("n_fertilizer_kg", "diet_digestibility", "area_total_ha", "manure_system", "region"),
  Impact_Range = c("5-12%", "8-15%", "Area metrics only", "10-25% manure", "5-20% inputs"),
  Collection_Difficulty = c("Easy", "Medium", "Easy", "Easy", "Easy"),
  Recommendation = c("Get purchase records", "Estimate from feed quality", 
                    "Survey or property records", "Observe system", "Select best match")
)

kable(medium_impact, caption = "Medium Impact Parameters")

## ----echo=FALSE---------------------------------------------------------------
low_impact <- data.frame(
  Parameter = c("plastic_kg", "lpg_kg", "gwp values", "milk_density", "transport_km"),
  Impact_Range = c("<2%", "<3%", "<5%", "<2%", "<5%"),
  Default_Approach = c("Estimate broadly", "Estimate or ignore", "Use package defaults", 
                      "Use 1.03", "Estimate 100-200 km"),
  Notes = c("Small contribution unless very large", "Often minimal in dairy",
           "IPCC AR6 values recommended", "Varies little", "Affects feed emissions only")
)

kable(low_impact, caption = "Low Impact Parameters (Can Use Estimates)")

## ----echo=FALSE---------------------------------------------------------------
error_solutions <- data.frame(
  Error_Type = c("Invalid region", "Negative values", "Area sum mismatch", "Missing required data", "Unrealistic results"),
  Common_Cause = c(
    "Typo in region name",
    "Data entry error or wrong units",
    "Land use breakdown doesn't add up",
    "Empty cells in required columns",
    "Wrong units or extreme outliers"
  ),
  Solution = c(
    "Check spelling: 'EU', 'US', 'Brazil', 'Argentina', 'Australia'",
    "Verify all quantities ≥ 0 and units are correct",
    "Review area_breakdown list or set validate_area_sum = FALSE",
    "Fill required columns or use defaults",
    "Check units, outliers, and parameter ranges"
  ),
  Prevention = c(
    "Use template dropdown lists",
    "Implement data validation in Excel",
    "Use GIS or survey data for areas",
    "Document data requirements clearly",
    "Compare results with similar farms"
  )
)

kable(error_solutions, caption = "Common Error Messages and Solutions")

## ----echo=FALSE---------------------------------------------------------------
optimization_tips <- data.frame(
  Aspect = c("Data Preparation", "Processing Speed", "Memory Management", "Error Handling", "Result Storage"),
  Recommendation = c(
    "Pre-validate data, use consistent formats, remove empty rows",
    "Process in chunks of 50-100 farms, use tier 1 for screening",
    "Set save_detailed_objects = FALSE for large batches",
    "Implement robust error logging and recovery mechanisms",
    "Export results incrementally, use database for >1000 farms"
  ),
  Performance_Gain = c("50-70%", "30-50%", "60-80%", "Prevents crashes", "Scalable"),
  Implementation_Effort = c("Low", "Medium", "Low", "High", "High")
)

kable(optimization_tips, caption = "Performance Optimization for Large Datasets")

## ----echo=FALSE---------------------------------------------------------------
optimal_combinations <- data.frame(
  System_Type = c("Intensive Dairy", "Extensive Grazing", "Mixed System", "Organic System"),
  Key_Parameters = c(
    "High DM intake, concentrate feeds, precise body weights",
    "Pasture N excretion, extensive manure system, lower Ym",
    "Balanced feed inputs, moderate intensities",
    "Organic fertilizers, lower input emissions, pasture focus"
  ),
  Critical_Measurements = c(
    "Feed composition, milk yield, system temperature",
    "Grazing management, soil conditions, climate data",
    "Feed efficiency ratios, land use breakdown",
    "Organic input quantities, certification requirements"
  ),
  Typical_Accuracy = c("±10-15%", "±15-25%", "±12-20%", "±15-30%")
)

kable(optimal_combinations, caption = "Optimal Parameter Combinations by System Type")

## ----echo=FALSE---------------------------------------------------------------
interaction_effects <- data.frame(
  Parameter_Pair = c("Body weight + DM intake", "Ym% + Feed quality", "Climate + Soil type", 
                    "Region + Feed sources", "Manure system + Retention time"),
  Interaction_Type = c("Multiplicative", "Exponential", "Additive", "Complex", "Threshold"),
  Effect_Magnitude = c("Medium", "High", "Medium", "High", "Variable"),
  Management_Implication = c(
    "Heavier cows need proportionally more feed",
    "Poor quality diets increase methane conversion",
    "Tropical poorly-drained soils have highest N2O",
    "Local feed sourcing reduces transport emissions",
    "Short retention (<30 days) limits CH4 conversion"
  )
)

kable(interaction_effects, caption = "Important Parameter Interactions")

## ----echo=FALSE---------------------------------------------------------------
data_requirements <- data.frame(
  Assessment_Goal = c("Screening Assessment", "Management Planning", "Carbon Trading", "Research Study"),
  Essential_Data = c(
    "Animal numbers, milk production, basic inputs",
    "Detailed feeds, precise areas, management practices",  
    "Verified production, third-party validated inputs",
    "Complete parameter set, uncertainty quantification"
  ),
  Time_Investment = c("2-4 hours", "1-2 days", "3-5 days", "1-2 weeks"),
  Accuracy_Target = c("±30%", "±15%", "±10%", "±5%"),
  Tier_Recommendation = c("Tier 1", "Tier 2", "Tier 2 + validation", "Tier 2 + uncertainty")
)

kable(data_requirements, caption = "Data Requirements by Assessment Objective")

## ----echo=FALSE---------------------------------------------------------------
collection_schedule <- data.frame(
  Data_Category = c("Production Records", "Feed Purchases", "Energy Consumption", 
                   "Land Management", "Animal Characteristics"),
  Collection_Frequency = c("Monthly", "Each delivery", "Monthly", "Seasonal", "Annual"),
  Storage_Location = c("Farm office", "Purchase invoices", "Utility bills", 
                      "Management records", "Herd records"),
  Quality_Control = c("Cross-check with processor", "Verify units and quantities", 
                     "Monitor seasonal patterns", "Update land use changes", 
                     "Weigh representative sample")
)

kable(collection_schedule, caption = "Recommended Data Collection Schedule")

## ----echo=FALSE---------------------------------------------------------------
validation_hierarchy <- data.frame(
  Level = c("Level 1: Range Checks", "Level 2: Consistency Checks", 
            "Level 3: Benchmark Comparison", "Level 4: Expert Review"),
  Validation_Type = c("Automatic", "Automatic", "Semi-automatic", "Manual"),
  Examples = c(
    "Values within expected ranges, correct units",
    "Milk yield vs feed intake, stocking rate vs area",
    "Results vs regional averages, peer farm comparison",
    "Technical review by LCA specialist"
  ),
  Error_Detection = c("90%", "70%", "50%", "95%"),
  Implementation = c("Built-in cowfootR", "Built-in cowfootR", "User comparison", "External expert")
)

kable(validation_hierarchy, caption = "Quality Assurance Validation Levels")

## ----echo=FALSE---------------------------------------------------------------
red_flags <- data.frame(
  Indicator = c("Milk intensity", "Feed efficiency", "Energy use", "Emission ratios", "System consistency"),
  Warning_Threshold = c(">2.5 kg CO2eq/kg FPCM", "<1.0 L milk/kg concentrate", 
                       ">0.15 kWh/L milk", "Enteric <30% of total", "Intensive system + low inputs"),
  Likely_Cause = c(
    "Poor productivity or data errors",
    "Overestimated feed use or underestimated milk",
    "Energy-intensive processes or errors",
    "Missing emission sources or calculation errors", 
    "Inconsistent system classification"
  ),
  Investigation_Priority = c("High", "High", "Medium", "High", "Medium")
)

kable(red_flags, caption = "Data Quality Red Flag Indicators")

