// Weibull proportional hazards model (event time)
// Hazard-scale parameterization
data{
  int N;       // number of patients (M(T1))
  real<lower=0> t[N];                    // observed time (t^*_{ei} in the paper)
  int<lower=0, upper=1> status[N];       // event indicator (delta_{ei} in the paper: 1 = event, 0 = right censored)
  int<lower=0, upper=1> treatment[N];     // treatment assignment (x_i in the paper: 1=experiment, 0=control)
  real<lower=0> a;              // 1st parameter of the gamma prior (shape) for lambda_0e (=alpha_e in the paper)
  real<lower=0> b;              // 2nd parameter of the gamma prior (rate / inverse scale) for lambda_0e (=beta_e in the paper)
  real<lower=0> c;              // 1st parameter of the gamma prior (shape) for rho_e (=alpha_re in the paper)
  real<lower=0> d;              // 2nd parameter of the gamma prior (rate / inverse scale) for rho_e (=beta_re in the paper)
  real mean_c;                  // 1st (mean) parameter of the normal prior for eta
  real<lower=0> se_c;           // 2nd (sd) parameter of the normal prior for eta (=sigma_eta in the paper)
  }

parameters{
  real<lower=0> rho;            // shape of weibull: rho_e
  real<lower=0> lambda;     // hazard scale of Weibull (lambda_0e)-- warning: this is not the time scale sigma for weibull used in stan
  real eta;                         // treatment effect (eta)
 }

model{
 rho ~ gamma(c,d);  // rho_e
 lambda ~ gamma(a,b); // lambda_0e
 eta ~ normal(mean_c,se_c);   // eta
    for(i in 1:N){
      real hazard_scale = lambda*exp(eta * treatment[i]); // the hazard scale lambda_{ei}(x_i,Z_i) as in equation (2) in the paper (no covariates)
      real time_scale   = pow(hazard_scale, -1.0 / rho);             // (hazard_scale)^(-1/rho)

    if(status[i]==1){ // δ_ei = 1 (event)
      target += weibull_lpdf(t[i] | rho, time_scale);
    }
    else{ // δ_ei=0 (right censored)
      target += weibull_lccdf(t[i] | rho, time_scale);
    }
    }
}
