from typing import Optional, List

from pydantic import BaseModel
from sqlalchemy import Column, Integer, ForeignKey, String, Double, Boolean, BigInteger
from sqlalchemy.orm import relationship

from .thread_callstacks import ThreadCallstackCreate, ThreadCallstackInDB, ThreadCallstack
from .thread_hotspots import ThreadHotspotCollection, ThreadHotspotCollectionInDB, ThreadHotspotCollectionCreate
from ..base import Base


class RegionThread(Base):
    __tablename__ = 'region_thread'
    id = Column(Integer, primary_key=True, index=True)
    name = Column(String)
    thread_id = Column(BigInteger)
    time_ms = Column(Double)
    time_pcore_ms = Column(Double, nullable=True)
    time_ecore_ms = Column(Double, nullable=True)
    utilization_pct = Column(Double)
    utilization_pcore_pct = Column(Double, nullable=True)
    utilization_ecore_pct = Column(Double, nullable=True)

    has_info = Column(Boolean)

    region_id = Column(Integer, ForeignKey('region.id'))
    region = relationship('Region', back_populates='region_threads')

    thread_callstack = relationship(ThreadCallstack, back_populates='region_thread')
    thread_hotspot_collections = relationship(ThreadHotspotCollection, back_populates='region_thread')


class RegionThreadShortInDB(BaseModel):
    id: int
    name: str
    region_id: int
    thread_id: int
    time_ms: float
    time_pcore_ms: Optional[float]
    time_ecore_ms: Optional[float]
    utilization_pct: float
    utilization_pcore_pct: Optional[float]
    utilization_ecore_pct: Optional[float]
    has_info: bool

    class Config:
        from_attributes = True


class RegionThreadInDB(RegionThreadShortInDB):
    thread_callstack: List[ThreadCallstackInDB]
    thread_hotspot_collections: List[ThreadHotspotCollectionInDB]

    class Config:
        from_attributes = True


class RegionThreadSearchQuery(BaseModel):
    hotspot_analysis_id: Optional[int] = None
    region_id: Optional[int] = None
    thread_id: Optional[int] = None
    limit: Optional[int] = 100
    offset: Optional[int] = 0

    class Config:
        from_attributes = True


class RegionThreadSearchInDB(BaseModel):
    results: List[RegionThreadInDB]
    count: int

    class Config:
        from_attributes = True


class RegionThreadBaseCreate(BaseModel):
    region_id: int
    name: str
    thread_id: int
    time_ms: float
    time_pcore_ms: Optional[float]
    time_ecore_ms: Optional[float]
    utilization_pct: float
    utilization_pcore_pct: Optional[float]
    utilization_ecore_pct: Optional[float]
    has_info: bool

    class Config:
        from_attributes = True


class RegionThreadCreate(RegionThreadBaseCreate):
    thread_callstack: List[ThreadCallstackCreate]
    thread_hotspot_collections: List[ThreadHotspotCollectionCreate]

    class Config:
        from_attributes = True
