﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 */
class DescribeDBShardGroupsRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeDBShardGroupsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeDBShardGroups"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The user-supplied DB shard group identifier. If this parameter is specified,
   * information for only the specific DB shard group is returned. This parameter
   * isn't case-sensitive.</p> <p>Constraints:</p> <ul> <li> <p>If supplied, must
   * match an existing DB shard group identifier.</p> </li> </ul>
   */
  inline const Aws::String& GetDBShardGroupIdentifier() const { return m_dBShardGroupIdentifier; }
  inline bool DBShardGroupIdentifierHasBeenSet() const { return m_dBShardGroupIdentifierHasBeenSet; }
  template <typename DBShardGroupIdentifierT = Aws::String>
  void SetDBShardGroupIdentifier(DBShardGroupIdentifierT&& value) {
    m_dBShardGroupIdentifierHasBeenSet = true;
    m_dBShardGroupIdentifier = std::forward<DBShardGroupIdentifierT>(value);
  }
  template <typename DBShardGroupIdentifierT = Aws::String>
  DescribeDBShardGroupsRequest& WithDBShardGroupIdentifier(DBShardGroupIdentifierT&& value) {
    SetDBShardGroupIdentifier(std::forward<DBShardGroupIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that specifies one or more DB shard groups to describe.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeDBShardGroupsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeDBShardGroupsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous
   * <code>DescribeDBShardGroups</code> request. If this parameter is specified, the
   * response includes only records beyond the marker, up to the value specified by
   * <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeDBShardGroupsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so you can retrieve the remaining
   * results.</p> <p>Default: 100</p> <p>Constraints: Minimum 20, maximum 100</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeDBShardGroupsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBShardGroupIdentifier;

  Aws::Vector<Filter> m_filters;

  Aws::String m_marker;

  int m_maxRecords{0};
  bool m_dBShardGroupIdentifierHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
