﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/omics/OmicsRequest.h>
#include <aws/omics/Omics_EXPORTS.h>
#include <aws/omics/model/CompleteReadSetUploadPartListItem.h>

#include <utility>

namespace Aws {
namespace Omics {
namespace Model {

/**
 */
class CompleteMultipartReadSetUploadRequest : public OmicsRequest {
 public:
  AWS_OMICS_API CompleteMultipartReadSetUploadRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CompleteMultipartReadSetUpload"; }

  AWS_OMICS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The sequence store ID for the store involved in the multipart upload.</p>
   */
  inline const Aws::String& GetSequenceStoreId() const { return m_sequenceStoreId; }
  inline bool SequenceStoreIdHasBeenSet() const { return m_sequenceStoreIdHasBeenSet; }
  template <typename SequenceStoreIdT = Aws::String>
  void SetSequenceStoreId(SequenceStoreIdT&& value) {
    m_sequenceStoreIdHasBeenSet = true;
    m_sequenceStoreId = std::forward<SequenceStoreIdT>(value);
  }
  template <typename SequenceStoreIdT = Aws::String>
  CompleteMultipartReadSetUploadRequest& WithSequenceStoreId(SequenceStoreIdT&& value) {
    SetSequenceStoreId(std::forward<SequenceStoreIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID for the multipart upload.</p>
   */
  inline const Aws::String& GetUploadId() const { return m_uploadId; }
  inline bool UploadIdHasBeenSet() const { return m_uploadIdHasBeenSet; }
  template <typename UploadIdT = Aws::String>
  void SetUploadId(UploadIdT&& value) {
    m_uploadIdHasBeenSet = true;
    m_uploadId = std::forward<UploadIdT>(value);
  }
  template <typename UploadIdT = Aws::String>
  CompleteMultipartReadSetUploadRequest& WithUploadId(UploadIdT&& value) {
    SetUploadId(std::forward<UploadIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The individual uploads or parts of a multipart upload.</p>
   */
  inline const Aws::Vector<CompleteReadSetUploadPartListItem>& GetParts() const { return m_parts; }
  inline bool PartsHasBeenSet() const { return m_partsHasBeenSet; }
  template <typename PartsT = Aws::Vector<CompleteReadSetUploadPartListItem>>
  void SetParts(PartsT&& value) {
    m_partsHasBeenSet = true;
    m_parts = std::forward<PartsT>(value);
  }
  template <typename PartsT = Aws::Vector<CompleteReadSetUploadPartListItem>>
  CompleteMultipartReadSetUploadRequest& WithParts(PartsT&& value) {
    SetParts(std::forward<PartsT>(value));
    return *this;
  }
  template <typename PartsT = CompleteReadSetUploadPartListItem>
  CompleteMultipartReadSetUploadRequest& AddParts(PartsT&& value) {
    m_partsHasBeenSet = true;
    m_parts.emplace_back(std::forward<PartsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sequenceStoreId;

  Aws::String m_uploadId;

  Aws::Vector<CompleteReadSetUploadPartListItem> m_parts;
  bool m_sequenceStoreIdHasBeenSet = false;
  bool m_uploadIdHasBeenSet = false;
  bool m_partsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Omics
}  // namespace Aws
