﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mgn/MgnRequest.h>
#include <aws/mgn/Mgn_EXPORTS.h>
#include <aws/mgn/model/BootMode.h>
#include <aws/mgn/model/LaunchDisposition.h>
#include <aws/mgn/model/Licensing.h>
#include <aws/mgn/model/PostLaunchActions.h>
#include <aws/mgn/model/TargetInstanceTypeRightSizingMethod.h>

#include <utility>

namespace Aws {
namespace mgn {
namespace Model {

/**
 */
class UpdateLaunchConfigurationRequest : public MgnRequest {
 public:
  AWS_MGN_API UpdateLaunchConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateLaunchConfiguration"; }

  AWS_MGN_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Update Launch configuration by Source Server ID request.</p>
   */
  inline const Aws::String& GetSourceServerID() const { return m_sourceServerID; }
  inline bool SourceServerIDHasBeenSet() const { return m_sourceServerIDHasBeenSet; }
  template <typename SourceServerIDT = Aws::String>
  void SetSourceServerID(SourceServerIDT&& value) {
    m_sourceServerIDHasBeenSet = true;
    m_sourceServerID = std::forward<SourceServerIDT>(value);
  }
  template <typename SourceServerIDT = Aws::String>
  UpdateLaunchConfigurationRequest& WithSourceServerID(SourceServerIDT&& value) {
    SetSourceServerID(std::forward<SourceServerIDT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update Launch configuration name request.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateLaunchConfigurationRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update Launch configuration launch disposition request.</p>
   */
  inline LaunchDisposition GetLaunchDisposition() const { return m_launchDisposition; }
  inline bool LaunchDispositionHasBeenSet() const { return m_launchDispositionHasBeenSet; }
  inline void SetLaunchDisposition(LaunchDisposition value) {
    m_launchDispositionHasBeenSet = true;
    m_launchDisposition = value;
  }
  inline UpdateLaunchConfigurationRequest& WithLaunchDisposition(LaunchDisposition value) {
    SetLaunchDisposition(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update Launch configuration Target instance right sizing request.</p>
   */
  inline TargetInstanceTypeRightSizingMethod GetTargetInstanceTypeRightSizingMethod() const {
    return m_targetInstanceTypeRightSizingMethod;
  }
  inline bool TargetInstanceTypeRightSizingMethodHasBeenSet() const { return m_targetInstanceTypeRightSizingMethodHasBeenSet; }
  inline void SetTargetInstanceTypeRightSizingMethod(TargetInstanceTypeRightSizingMethod value) {
    m_targetInstanceTypeRightSizingMethodHasBeenSet = true;
    m_targetInstanceTypeRightSizingMethod = value;
  }
  inline UpdateLaunchConfigurationRequest& WithTargetInstanceTypeRightSizingMethod(TargetInstanceTypeRightSizingMethod value) {
    SetTargetInstanceTypeRightSizingMethod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update Launch configuration copy Private IP request.</p>
   */
  inline bool GetCopyPrivateIp() const { return m_copyPrivateIp; }
  inline bool CopyPrivateIpHasBeenSet() const { return m_copyPrivateIpHasBeenSet; }
  inline void SetCopyPrivateIp(bool value) {
    m_copyPrivateIpHasBeenSet = true;
    m_copyPrivateIp = value;
  }
  inline UpdateLaunchConfigurationRequest& WithCopyPrivateIp(bool value) {
    SetCopyPrivateIp(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update Launch configuration copy Tags request.</p>
   */
  inline bool GetCopyTags() const { return m_copyTags; }
  inline bool CopyTagsHasBeenSet() const { return m_copyTagsHasBeenSet; }
  inline void SetCopyTags(bool value) {
    m_copyTagsHasBeenSet = true;
    m_copyTags = value;
  }
  inline UpdateLaunchConfigurationRequest& WithCopyTags(bool value) {
    SetCopyTags(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update Launch configuration licensing request.</p>
   */
  inline const Licensing& GetLicensing() const { return m_licensing; }
  inline bool LicensingHasBeenSet() const { return m_licensingHasBeenSet; }
  template <typename LicensingT = Licensing>
  void SetLicensing(LicensingT&& value) {
    m_licensingHasBeenSet = true;
    m_licensing = std::forward<LicensingT>(value);
  }
  template <typename LicensingT = Licensing>
  UpdateLaunchConfigurationRequest& WithLicensing(LicensingT&& value) {
    SetLicensing(std::forward<LicensingT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update Launch configuration boot mode request.</p>
   */
  inline BootMode GetBootMode() const { return m_bootMode; }
  inline bool BootModeHasBeenSet() const { return m_bootModeHasBeenSet; }
  inline void SetBootMode(BootMode value) {
    m_bootModeHasBeenSet = true;
    m_bootMode = value;
  }
  inline UpdateLaunchConfigurationRequest& WithBootMode(BootMode value) {
    SetBootMode(value);
    return *this;
  }
  ///@}

  ///@{

  inline const PostLaunchActions& GetPostLaunchActions() const { return m_postLaunchActions; }
  inline bool PostLaunchActionsHasBeenSet() const { return m_postLaunchActionsHasBeenSet; }
  template <typename PostLaunchActionsT = PostLaunchActions>
  void SetPostLaunchActions(PostLaunchActionsT&& value) {
    m_postLaunchActionsHasBeenSet = true;
    m_postLaunchActions = std::forward<PostLaunchActionsT>(value);
  }
  template <typename PostLaunchActionsT = PostLaunchActions>
  UpdateLaunchConfigurationRequest& WithPostLaunchActions(PostLaunchActionsT&& value) {
    SetPostLaunchActions(std::forward<PostLaunchActionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Enable map auto tagging.</p>
   */
  inline bool GetEnableMapAutoTagging() const { return m_enableMapAutoTagging; }
  inline bool EnableMapAutoTaggingHasBeenSet() const { return m_enableMapAutoTaggingHasBeenSet; }
  inline void SetEnableMapAutoTagging(bool value) {
    m_enableMapAutoTaggingHasBeenSet = true;
    m_enableMapAutoTagging = value;
  }
  inline UpdateLaunchConfigurationRequest& WithEnableMapAutoTagging(bool value) {
    SetEnableMapAutoTagging(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Launch configuration map auto tagging MPE ID.</p>
   */
  inline const Aws::String& GetMapAutoTaggingMpeID() const { return m_mapAutoTaggingMpeID; }
  inline bool MapAutoTaggingMpeIDHasBeenSet() const { return m_mapAutoTaggingMpeIDHasBeenSet; }
  template <typename MapAutoTaggingMpeIDT = Aws::String>
  void SetMapAutoTaggingMpeID(MapAutoTaggingMpeIDT&& value) {
    m_mapAutoTaggingMpeIDHasBeenSet = true;
    m_mapAutoTaggingMpeID = std::forward<MapAutoTaggingMpeIDT>(value);
  }
  template <typename MapAutoTaggingMpeIDT = Aws::String>
  UpdateLaunchConfigurationRequest& WithMapAutoTaggingMpeID(MapAutoTaggingMpeIDT&& value) {
    SetMapAutoTaggingMpeID(std::forward<MapAutoTaggingMpeIDT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update Launch configuration Account ID.</p>
   */
  inline const Aws::String& GetAccountID() const { return m_accountID; }
  inline bool AccountIDHasBeenSet() const { return m_accountIDHasBeenSet; }
  template <typename AccountIDT = Aws::String>
  void SetAccountID(AccountIDT&& value) {
    m_accountIDHasBeenSet = true;
    m_accountID = std::forward<AccountIDT>(value);
  }
  template <typename AccountIDT = Aws::String>
  UpdateLaunchConfigurationRequest& WithAccountID(AccountIDT&& value) {
    SetAccountID(std::forward<AccountIDT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sourceServerID;

  Aws::String m_name;

  LaunchDisposition m_launchDisposition{LaunchDisposition::NOT_SET};

  TargetInstanceTypeRightSizingMethod m_targetInstanceTypeRightSizingMethod{TargetInstanceTypeRightSizingMethod::NOT_SET};

  bool m_copyPrivateIp{false};

  bool m_copyTags{false};

  Licensing m_licensing;

  BootMode m_bootMode{BootMode::NOT_SET};

  PostLaunchActions m_postLaunchActions;

  bool m_enableMapAutoTagging{false};

  Aws::String m_mapAutoTaggingMpeID;

  Aws::String m_accountID;
  bool m_sourceServerIDHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_launchDispositionHasBeenSet = false;
  bool m_targetInstanceTypeRightSizingMethodHasBeenSet = false;
  bool m_copyPrivateIpHasBeenSet = false;
  bool m_copyTagsHasBeenSet = false;
  bool m_licensingHasBeenSet = false;
  bool m_bootModeHasBeenSet = false;
  bool m_postLaunchActionsHasBeenSet = false;
  bool m_enableMapAutoTaggingHasBeenSet = false;
  bool m_mapAutoTaggingMpeIDHasBeenSet = false;
  bool m_accountIDHasBeenSet = false;
};

}  // namespace Model
}  // namespace mgn
}  // namespace Aws
