/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.storage.managers;

import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.stream.Stream;
import org.infinispan.protostream.annotations.ProtoField;
import org.infinispan.protostream.annotations.ProtoTypeId;
import org.jboss.logging.Logger;
import org.keycloak.cluster.ClusterEvent;
import org.keycloak.cluster.ClusterListener;
import org.keycloak.cluster.ClusterProvider;
import org.keycloak.cluster.ExecutionResult;
import org.keycloak.common.util.Time;
import org.keycloak.component.ComponentModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.KeycloakSessionTask;
import org.keycloak.models.RealmModel;
import org.keycloak.models.StorageProviderRealmModel;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.storage.UserStorageProvider;
import org.keycloak.storage.UserStorageProviderFactory;
import org.keycloak.storage.UserStorageProviderModel;
import org.keycloak.storage.user.ImportSynchronization;
import org.keycloak.storage.user.SynchronizationResult;
import org.keycloak.timer.TimerProvider;

public class UserStorageSyncManager {
    private static final String USER_STORAGE_TASK_KEY = "user-storage";
    private static final Logger logger = Logger.getLogger(UserStorageSyncManager.class);

    public static void bootstrapPeriodic(final KeycloakSessionFactory sessionFactory, final TimerProvider timer) {
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                Stream realms = session.realms().getRealmsWithProviderTypeStream(UserStorageProvider.class);
                realms.forEach(realm -> {
                    Stream providers = ((StorageProviderRealmModel)realm).getUserStorageProvidersStream();
                    providers.forEachOrdered(provider -> {
                        UserStorageProviderFactory factory = (UserStorageProviderFactory)session.getKeycloakSessionFactory().getProviderFactory(UserStorageProvider.class, provider.getProviderId());
                        if (factory instanceof ImportSynchronization && provider.isImportEnabled()) {
                            UserStorageSyncManager.refreshPeriodicSyncForProvider(sessionFactory, timer, provider, realm);
                        }
                    });
                });
                ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
                clusterProvider.registerListener(UserStorageSyncManager.USER_STORAGE_TASK_KEY, (ClusterListener)new UserStorageClusterListener(sessionFactory));
            }
        });
    }

    public static SynchronizationResult syncAllUsers(final KeycloakSessionFactory sessionFactory, final String realmId, final UserStorageProviderModel provider) {
        final UserStorageProviderFactory factory = (UserStorageProviderFactory)sessionFactory.getProviderFactory(UserStorageProvider.class, provider.getProviderId());
        if (!(factory instanceof ImportSynchronization && provider.isImportEnabled() && provider.isEnabled())) {
            return SynchronizationResult.ignored();
        }
        final Holder holder = new Holder();
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
                String taskKey = provider.getId() + "::sync";
                int timeout = Math.max(30, provider.getFullSyncPeriod());
                holder.result = clusterProvider.executeIfNotExecuted(taskKey, timeout, (Callable)new Callable<SynchronizationResult>(){

                    @Override
                    public SynchronizationResult call() throws Exception {
                        int lastSync = Time.currentTime();
                        SynchronizationResult result = ((ImportSynchronization)factory).sync(sessionFactory, realmId, provider);
                        if (!result.isIgnored()) {
                            UserStorageSyncManager.updateLastSyncInterval(sessionFactory, provider, realmId, lastSync);
                        }
                        return result;
                    }
                });
            }
        });
        if (holder.result == null || !holder.result.isExecuted()) {
            logger.debugf("syncAllUsers for federation provider %s was ignored as it's already in progress", (Object)provider.getName());
            return SynchronizationResult.ignored();
        }
        return (SynchronizationResult)holder.result.getResult();
    }

    public static SynchronizationResult syncChangedUsers(final KeycloakSessionFactory sessionFactory, final String realmId, final UserStorageProviderModel provider) {
        final UserStorageProviderFactory factory = (UserStorageProviderFactory)sessionFactory.getProviderFactory(UserStorageProvider.class, provider.getProviderId());
        if (!(factory instanceof ImportSynchronization && provider.isImportEnabled() && provider.isEnabled())) {
            return SynchronizationResult.ignored();
        }
        final Holder holder = new Holder();
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
                String taskKey = provider.getId() + "::sync";
                int timeout = Math.max(30, provider.getChangedSyncPeriod());
                holder.result = clusterProvider.executeIfNotExecuted(taskKey, timeout, (Callable)new Callable<SynchronizationResult>(){

                    @Override
                    public SynchronizationResult call() throws Exception {
                        int oldLastSync = provider.getLastSync();
                        int lastSync = Time.currentTime();
                        SynchronizationResult result = ((ImportSynchronization)factory).syncSince(Time.toDate((int)oldLastSync), sessionFactory, realmId, provider);
                        if (!result.isIgnored()) {
                            UserStorageSyncManager.updateLastSyncInterval(sessionFactory, provider, realmId, lastSync);
                        }
                        return result;
                    }
                });
            }
        });
        if (holder.result == null || !holder.result.isExecuted()) {
            logger.debugf("syncChangedUsers for federation provider %s was ignored as it's already in progress", (Object)provider.getName());
            return SynchronizationResult.ignored();
        }
        return (SynchronizationResult)holder.result.getResult();
    }

    public static void notifyToRefreshPeriodicSyncAll(KeycloakSession session, RealmModel realm, boolean removed) {
        ((StorageProviderRealmModel)realm).getUserStorageProvidersStream().forEachOrdered(fedProvider -> UserStorageSyncManager.notifyToRefreshPeriodicSync(session, realm, fedProvider, removed));
    }

    public static void notifyToRefreshPeriodicSyncSingle(KeycloakSession session, RealmModel realm, ComponentModel component, boolean removed) {
        UserStorageSyncManager.notifyToRefreshPeriodicSync(session, realm, new UserStorageProviderModel(component), removed);
    }

    public static void notifyToRefreshPeriodicSync(KeycloakSession session, RealmModel realm, UserStorageProviderModel provider, boolean removed) {
        UserStorageProviderFactory factory = (UserStorageProviderFactory)session.getKeycloakSessionFactory().getProviderFactory(UserStorageProvider.class, provider.getProviderId());
        if (!(factory instanceof ImportSynchronization) || !provider.isImportEnabled()) {
            return;
        }
        ClusterProvider cp = (ClusterProvider)session.getProvider(ClusterProvider.class);
        if (cp != null) {
            UserStorageProviderClusterEvent event = UserStorageProviderClusterEvent.createEvent(removed, realm.getId(), provider);
            cp.notify(USER_STORAGE_TASK_KEY, (ClusterEvent)event, false);
        }
    }

    protected static void refreshPeriodicSyncForProvider(KeycloakSessionFactory sessionFactory, TimerProvider timer, UserStorageProviderModel provider, RealmModel realm) {
        logger.debugf("Going to refresh periodic sync settings for provider '%s' in realm '%s' with realmId '%s'. Full sync period: %d , changed users sync period: %d", new Object[]{provider.getName(), realm.getName(), realm.getId(), provider.getFullSyncPeriod(), provider.getChangedSyncPeriod()});
        String fullSyncTaskName = UserStorageSyncManager.createSyncTaskName(provider, UserStorageSyncTask.SyncMode.FULL);
        if (provider.getFullSyncPeriod() > 0) {
            UserStorageSyncTask task = new UserStorageSyncTask(provider, realm, sessionFactory, UserStorageSyncTask.SyncMode.FULL);
            timer.schedule((Runnable)task, (long)provider.getFullSyncPeriod() * 1000L, fullSyncTaskName);
        } else {
            timer.cancelTask(fullSyncTaskName);
        }
        String changedSyncTaskName = UserStorageSyncManager.createSyncTaskName(provider, UserStorageSyncTask.SyncMode.CHANGED);
        if (provider.getChangedSyncPeriod() > 0) {
            UserStorageSyncTask task = new UserStorageSyncTask(provider, realm, sessionFactory, UserStorageSyncTask.SyncMode.CHANGED);
            timer.schedule((Runnable)task, (long)provider.getChangedSyncPeriod() * 1000L, changedSyncTaskName);
        } else {
            timer.cancelTask(changedSyncTaskName);
        }
    }

    public static String createSyncTaskName(UserStorageProviderModel model, UserStorageSyncTask.SyncMode syncMode) {
        return UserStorageSyncTask.class.getSimpleName() + "-" + model.getId() + "-" + String.valueOf((Object)syncMode);
    }

    private static boolean shouldPerformNewPeriodicSync(int lastSyncTime, int period) {
        if (lastSyncTime <= 0) {
            return true;
        }
        int currentTime = Time.currentTime();
        int timeSinceLastSync = currentTime - lastSyncTime;
        return timeSinceLastSync * 2 > period;
    }

    protected static void removePeriodicSyncForProvider(TimerProvider timer, UserStorageProviderModel fedProvider) {
        UserStorageSyncManager.cancelPeriodicSyncForProviderIfPresent(timer, fedProvider, UserStorageSyncTask.SyncMode.FULL);
        UserStorageSyncManager.cancelPeriodicSyncForProviderIfPresent(timer, fedProvider, UserStorageSyncTask.SyncMode.CHANGED);
    }

    protected static void cancelPeriodicSyncForProviderIfPresent(TimerProvider timer, UserStorageProviderModel providerModel, UserStorageSyncTask.SyncMode syncMode) {
        String taskName = UserStorageSyncManager.createSyncTaskName(providerModel, syncMode);
        TimerProvider.TimerTaskContext existingTask = timer.cancelTask(taskName);
        if (existingTask != null) {
            logger.debugf("Cancelled periodic sync task with task-name '%s' for provider with id '%s' and name '%s'", (Object)taskName, (Object)providerModel.getId(), (Object)providerModel.getName());
        }
    }

    private static void updateLastSyncInterval(KeycloakSessionFactory sessionFactory, final UserStorageProviderModel provider, final String realmId, final int lastSync) {
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                RealmModel persistentRealm = session.realms().getRealm(realmId);
                ((StorageProviderRealmModel)persistentRealm).getUserStorageProvidersStream().filter(persistentFedProvider -> Objects.equals(provider.getId(), persistentFedProvider.getId())).forEachOrdered(persistentFedProvider -> {
                    persistentFedProvider.setLastSync(lastSync);
                    persistentRealm.updateComponent((ComponentModel)persistentFedProvider);
                    provider.setLastSync(lastSync);
                });
            }
        });
    }

    private static class Holder {
        ExecutionResult<SynchronizationResult> result;

        private Holder() {
        }
    }

    @ProtoTypeId(value=65540)
    public static class UserStorageProviderClusterEvent
    implements ClusterEvent {
        private boolean removed;
        private String realmId;
        private UserStorageProviderModel storageProvider;

        @ProtoField(value=1)
        public boolean isRemoved() {
            return this.removed;
        }

        public void setRemoved(boolean removed) {
            this.removed = removed;
        }

        @ProtoField(value=2)
        public String getRealmId() {
            return this.realmId;
        }

        public void setRealmId(String realmId) {
            this.realmId = realmId;
        }

        @ProtoField(value=3)
        public UserStorageProviderModel getStorageProvider() {
            return this.storageProvider;
        }

        public void setStorageProvider(UserStorageProviderModel federationProvider) {
            this.storageProvider = federationProvider;
        }

        public static UserStorageProviderClusterEvent createEvent(boolean removed, String realmId, UserStorageProviderModel provider) {
            UserStorageProviderClusterEvent notification = new UserStorageProviderClusterEvent();
            notification.setRemoved(removed);
            notification.setRealmId(realmId);
            notification.setStorageProvider(provider);
            return notification;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            UserStorageProviderClusterEvent that = (UserStorageProviderClusterEvent)o;
            return this.removed == that.removed && Objects.equals(this.realmId, that.realmId) && Objects.equals(this.storageProvider.getId(), that.storageProvider.getId());
        }

        public int hashCode() {
            return Objects.hash(this.removed, this.realmId, this.storageProvider.getId());
        }
    }

    public static class UserStorageSyncTask
    implements Runnable {
        private final UserStorageProviderModel provider;
        private final RealmModel realm;
        private final KeycloakSessionFactory sessionFactory;
        private final SyncMode syncMode;

        public UserStorageSyncTask(UserStorageProviderModel provider, RealmModel realm, KeycloakSessionFactory sessionFactory, SyncMode syncMode) {
            this.provider = provider;
            this.realm = realm;
            this.sessionFactory = sessionFactory;
            this.syncMode = syncMode;
        }

        @Override
        public void run() {
            try {
                boolean shouldPerformSync = UserStorageSyncManager.shouldPerformNewPeriodicSync(this.provider.getLastSync(), this.provider.getChangedSyncPeriod());
                if (!shouldPerformSync) {
                    logger.debugf("Ignored periodic %s users-sync with storage provider %s due small time since last sync in realm %s", (Object)this.syncMode, (Object)this.provider.getName(), (Object)this.realm.getName());
                    return;
                }
                switch (this.syncMode.ordinal()) {
                    case 0: {
                        UserStorageSyncManager.syncAllUsers(this.sessionFactory, this.realm.getId(), this.provider);
                        break;
                    }
                    case 1: {
                        UserStorageSyncManager.syncChangedUsers(this.sessionFactory, this.realm.getId(), this.provider);
                    }
                }
            }
            catch (Throwable t) {
                logger.errorf(t, "Error occurred during %s users-sync in realm %s", (Object)this.syncMode, (Object)this.realm.getName());
            }
        }

        public static enum SyncMode {
            FULL,
            CHANGED;

        }
    }

    private static class UserStorageClusterListener
    implements ClusterListener {
        private final KeycloakSessionFactory sessionFactory;

        public UserStorageClusterListener(KeycloakSessionFactory sessionFactory) {
            this.sessionFactory = sessionFactory;
        }

        public void eventReceived(ClusterEvent event) {
            final UserStorageProviderClusterEvent fedEvent = (UserStorageProviderClusterEvent)event;
            KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)this.sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

                public void run(KeycloakSession session) {
                    TimerProvider timer = (TimerProvider)session.getProvider(TimerProvider.class);
                    if (fedEvent.isRemoved()) {
                        UserStorageSyncManager.removePeriodicSyncForProvider(timer, fedEvent.getStorageProvider());
                    } else {
                        RealmModel realm = session.realms().getRealm(fedEvent.getRealmId());
                        UserStorageSyncManager.refreshPeriodicSyncForProvider(sessionFactory, timer, fedEvent.getStorageProvider(), realm);
                    }
                }
            });
        }
    }
}

