/*******************************************************************************
* Copyright 2014-2020 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

//@HEADER
// ***************************************************
//
// HPCG: High Performance Conjugate Gradient Benchmark
//
// Contact:
// Michael A. Heroux ( maherou@sandia.gov)
// Jack Dongarra     (dongarra@eecs.utk.edu)
// Piotr Luszczek    (luszczek@eecs.utk.edu)
//
// ***************************************************
//@HEADER

/*!
 @file Vector.hpp

 HPCG data structures for dense vectors
 */

#ifndef VECTOR_HPP
#define VECTOR_HPP
#include <cassert>
#include <cstdlib>
#include "Geometry.hpp"

//SYCL includes
#include <sycl/sycl.hpp>


struct Vector_STRUCT {
  local_int_t localLength = 0;  //!< length of local portion of the vector
  double * values = nullptr;          //!< array of values
  /*!
   This is for storing optimized data structures created in OptimizeProblem and
   used inside optimized ComputeSPMV().
   */
  void * optimizationData = nullptr;

};
typedef struct Vector_STRUCT Vector;

/*!
  Initializes input vector.

  @param[in] v
  @param[in] localLength Length of local portion of input vector
 */
inline void InitializeVectorHost(Vector & v, local_int_t localLength, sycl::queue & main_queue) {
  v.localLength = localLength;
  v.values = (double*) sycl::malloc_host( sizeof(double)*localLength, main_queue);
  v.optimizationData = nullptr;

  return;
}

inline void InitializeVectorShared(Vector & v, local_int_t localLength, sycl::queue & main_queue) {
  v.localLength = localLength;
  v.values = (double*) sycl::malloc_shared(
      sizeof(double)*localLength, main_queue.get_device(), main_queue.get_context());
  v.optimizationData = nullptr;

  return;
}

inline void InitializeVectorDevice(Vector & v, local_int_t localLength, sycl::queue & main_queue) {
  v.localLength = localLength;
  v.values = (double*) sycl::malloc_device(
      sizeof(double)*localLength, main_queue.get_device(), main_queue.get_context());
  v.optimizationData = nullptr;

  return;
}

inline void * hpcg_aligned_alloc(size_t size, size_t alignment) {
    if (size < 1) return nullptr;
    size_t actual_size = (((size - 1) / alignment) + 1) * alignment;
    return std::aligned_alloc(alignment, size);
}

inline void InitializeVector(Vector & v, local_int_t localLength) {
  v.localLength = localLength;
  v.values = (double*) hpcg_aligned_alloc(sizeof(double)*localLength, 512); //new double[localLength];
  v.optimizationData = nullptr;

  return;
}

// see also UshUtil.hpp ZeroVector()
inline void ZeroVector(Vector & v) {
  local_int_t localLength = v.localLength;
  double * vv = v.values;
  for (int i=0; i<localLength; ++i) vv[i] = 0.0;

  return;
}

/*!
  Multiply (scale) a specific vector entry by a given value.

  @param[inout] v Vector to be modified
  @param[in] index Local index of entry to scale
  @param[in] value Value to scale by
 */
inline void ScaleVectorValue(Vector & v, local_int_t index, double value) {
  assert(index>=0 && index < v.localLength);
  double * vv = v.values;
  vv[index] *= value;
  return;
}

/*!
  Fill the input vector with pseudo-random values.

  @param[in] v
 */
inline void FillRandomVector(Vector & v) {
  local_int_t localLength = v.localLength;
  double * vv = v.values;
  for (int i=0; i<localLength; ++i) vv[i] = rand() / (double)(RAND_MAX) + 1.0;
  return;
}

/*!
  Copy input vector to output vector.

  @param[in] v Input vector
  @param[in] w Output vector
 */
inline void CopyVector(const Vector & v, Vector & w) {
  local_int_t localLength = v.localLength;
  assert(w.localLength >= localLength);
  double * vv = v.values;
  double * wv = w.values;
  for (int i=0; i<localLength; ++i) wv[i] = vv[i];
  return;
}

/*!
  Deallocates the members of the data structure of the known system matrix provided they are not 0.

  @param[in] A the known system matrix
 */
inline void DeleteVector(Vector & v) {

  if (v.values) delete [] v.values;
  v.localLength = 0;
  return;
}

inline void DeleteVector(Vector & v, sycl::queue & main_queue) {

  if (v.values) sycl::free(v.values, main_queue);
  v.localLength = 0;
  return;
}

#endif // VECTOR_HPP
