﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DeleteInstanceEventWindowRequest : public EC2Request {
 public:
  AWS_EC2_API DeleteInstanceEventWindowRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteInstanceEventWindow"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DeleteInstanceEventWindowRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify <code>true</code> to force delete the event window. Use the force
   * delete parameter if the event window is currently associated with targets.</p>
   */
  inline bool GetForceDelete() const { return m_forceDelete; }
  inline bool ForceDeleteHasBeenSet() const { return m_forceDeleteHasBeenSet; }
  inline void SetForceDelete(bool value) {
    m_forceDeleteHasBeenSet = true;
    m_forceDelete = value;
  }
  inline DeleteInstanceEventWindowRequest& WithForceDelete(bool value) {
    SetForceDelete(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the event window.</p>
   */
  inline const Aws::String& GetInstanceEventWindowId() const { return m_instanceEventWindowId; }
  inline bool InstanceEventWindowIdHasBeenSet() const { return m_instanceEventWindowIdHasBeenSet; }
  template <typename InstanceEventWindowIdT = Aws::String>
  void SetInstanceEventWindowId(InstanceEventWindowIdT&& value) {
    m_instanceEventWindowIdHasBeenSet = true;
    m_instanceEventWindowId = std::forward<InstanceEventWindowIdT>(value);
  }
  template <typename InstanceEventWindowIdT = Aws::String>
  DeleteInstanceEventWindowRequest& WithInstanceEventWindowId(InstanceEventWindowIdT&& value) {
    SetInstanceEventWindowId(std::forward<InstanceEventWindowIdT>(value));
    return *this;
  }
  ///@}
 private:
  bool m_dryRun{false};

  bool m_forceDelete{false};

  Aws::String m_instanceEventWindowId;
  bool m_dryRunHasBeenSet = false;
  bool m_forceDeleteHasBeenSet = false;
  bool m_instanceEventWindowIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
