﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/rbin/RecycleBin_EXPORTS.h>
#include <aws/rbin/model/LockState.h>
#include <aws/rbin/model/RetentionPeriod.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace RecycleBin {
namespace Model {

/**
 * <p>Information about a Recycle Bin retention rule.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rbin-2021-06-15/RuleSummary">AWS
 * API Reference</a></p>
 */
class RuleSummary {
 public:
  AWS_RECYCLEBIN_API RuleSummary() = default;
  AWS_RECYCLEBIN_API RuleSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_RECYCLEBIN_API RuleSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_RECYCLEBIN_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The unique ID of the retention rule.</p>
   */
  inline const Aws::String& GetIdentifier() const { return m_identifier; }
  inline bool IdentifierHasBeenSet() const { return m_identifierHasBeenSet; }
  template <typename IdentifierT = Aws::String>
  void SetIdentifier(IdentifierT&& value) {
    m_identifierHasBeenSet = true;
    m_identifier = std::forward<IdentifierT>(value);
  }
  template <typename IdentifierT = Aws::String>
  RuleSummary& WithIdentifier(IdentifierT&& value) {
    SetIdentifier(std::forward<IdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The retention rule description.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  RuleSummary& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the retention period for which the retention rule is to
   * retain resources.</p>
   */
  inline const RetentionPeriod& GetRetentionPeriod() const { return m_retentionPeriod; }
  inline bool RetentionPeriodHasBeenSet() const { return m_retentionPeriodHasBeenSet; }
  template <typename RetentionPeriodT = RetentionPeriod>
  void SetRetentionPeriod(RetentionPeriodT&& value) {
    m_retentionPeriodHasBeenSet = true;
    m_retentionPeriod = std::forward<RetentionPeriodT>(value);
  }
  template <typename RetentionPeriodT = RetentionPeriod>
  RuleSummary& WithRetentionPeriod(RetentionPeriodT&& value) {
    SetRetentionPeriod(std::forward<RetentionPeriodT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>[Region-level retention rules only] The lock state for the retention
   * rule.</p> <ul> <li> <p> <code>locked</code> - The retention rule is locked and
   * can't be modified or deleted.</p> </li> <li> <p> <code>pending_unlock</code> -
   * The retention rule has been unlocked but it is still within the unlock delay
   * period. The retention rule can be modified or deleted only after the unlock
   * delay period has expired.</p> </li> <li> <p> <code>unlocked</code> - The
   * retention rule is unlocked and it can be modified or deleted by any user with
   * the required permissions.</p> </li> <li> <p> <code>null</code> - The retention
   * rule has never been locked. Once a retention rule has been locked, it can
   * transition between the <code>locked</code> and <code>unlocked</code> states
   * only; it can never transition back to <code>null</code>.</p> </li> </ul>
   */
  inline LockState GetLockState() const { return m_lockState; }
  inline bool LockStateHasBeenSet() const { return m_lockStateHasBeenSet; }
  inline void SetLockState(LockState value) {
    m_lockStateHasBeenSet = true;
    m_lockState = value;
  }
  inline RuleSummary& WithLockState(LockState value) {
    SetLockState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the retention rule.</p>
   */
  inline const Aws::String& GetRuleArn() const { return m_ruleArn; }
  inline bool RuleArnHasBeenSet() const { return m_ruleArnHasBeenSet; }
  template <typename RuleArnT = Aws::String>
  void SetRuleArn(RuleArnT&& value) {
    m_ruleArnHasBeenSet = true;
    m_ruleArn = std::forward<RuleArnT>(value);
  }
  template <typename RuleArnT = Aws::String>
  RuleSummary& WithRuleArn(RuleArnT&& value) {
    SetRuleArn(std::forward<RuleArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_identifier;

  Aws::String m_description;

  RetentionPeriod m_retentionPeriod;

  LockState m_lockState{LockState::NOT_SET};

  Aws::String m_ruleArn;
  bool m_identifierHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_retentionPeriodHasBeenSet = false;
  bool m_lockStateHasBeenSet = false;
  bool m_ruleArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace RecycleBin
}  // namespace Aws
