﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/OpsItemRelatedItemsFilter.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 */
class ListOpsItemRelatedItemsRequest : public SSMRequest {
 public:
  AWS_SSM_API ListOpsItemRelatedItemsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListOpsItemRelatedItems"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the OpsItem for which you want to list all related-item
   * resources.</p>
   */
  inline const Aws::String& GetOpsItemId() const { return m_opsItemId; }
  inline bool OpsItemIdHasBeenSet() const { return m_opsItemIdHasBeenSet; }
  template <typename OpsItemIdT = Aws::String>
  void SetOpsItemId(OpsItemIdT&& value) {
    m_opsItemIdHasBeenSet = true;
    m_opsItemId = std::forward<OpsItemIdT>(value);
  }
  template <typename OpsItemIdT = Aws::String>
  ListOpsItemRelatedItemsRequest& WithOpsItemId(OpsItemIdT&& value) {
    SetOpsItemId(std::forward<OpsItemIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more OpsItem filters. Use a filter to return a more specific list of
   * results. </p>
   */
  inline const Aws::Vector<OpsItemRelatedItemsFilter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<OpsItemRelatedItemsFilter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<OpsItemRelatedItemsFilter>>
  ListOpsItemRelatedItemsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = OpsItemRelatedItemsFilter>
  ListOpsItemRelatedItemsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to return for this call. The call also returns a
   * token that you can specify in a subsequent call to get the next set of
   * results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListOpsItemRelatedItemsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of items to return. (You received this token from
   * a previous call.)</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListOpsItemRelatedItemsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_opsItemId;

  Aws::Vector<OpsItemRelatedItemsFilter> m_filters;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_opsItemIdHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
