﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/managedblockchain/ManagedBlockchain_EXPORTS.h>
#include <aws/managedblockchain/model/Framework.h>
#include <aws/managedblockchain/model/NetworkFrameworkAttributes.h>
#include <aws/managedblockchain/model/NetworkStatus.h>
#include <aws/managedblockchain/model/VotingPolicy.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ManagedBlockchain {
namespace Model {

/**
 * <p>Network configuration properties.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/managedblockchain-2018-09-24/Network">AWS
 * API Reference</a></p>
 */
class Network {
 public:
  AWS_MANAGEDBLOCKCHAIN_API Network() = default;
  AWS_MANAGEDBLOCKCHAIN_API Network(Aws::Utils::Json::JsonView jsonValue);
  AWS_MANAGEDBLOCKCHAIN_API Network& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_MANAGEDBLOCKCHAIN_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The unique identifier of the network.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  Network& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the network.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Network& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Attributes of the blockchain framework for the network.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Network& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The blockchain framework that the network uses.</p>
   */
  inline Framework GetFramework() const { return m_framework; }
  inline bool FrameworkHasBeenSet() const { return m_frameworkHasBeenSet; }
  inline void SetFramework(Framework value) {
    m_frameworkHasBeenSet = true;
    m_framework = value;
  }
  inline Network& WithFramework(Framework value) {
    SetFramework(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the blockchain framework that the network uses.</p>
   */
  inline const Aws::String& GetFrameworkVersion() const { return m_frameworkVersion; }
  inline bool FrameworkVersionHasBeenSet() const { return m_frameworkVersionHasBeenSet; }
  template <typename FrameworkVersionT = Aws::String>
  void SetFrameworkVersion(FrameworkVersionT&& value) {
    m_frameworkVersionHasBeenSet = true;
    m_frameworkVersion = std::forward<FrameworkVersionT>(value);
  }
  template <typename FrameworkVersionT = Aws::String>
  Network& WithFrameworkVersion(FrameworkVersionT&& value) {
    SetFrameworkVersion(std::forward<FrameworkVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Attributes of the blockchain framework that the network uses.</p>
   */
  inline const NetworkFrameworkAttributes& GetFrameworkAttributes() const { return m_frameworkAttributes; }
  inline bool FrameworkAttributesHasBeenSet() const { return m_frameworkAttributesHasBeenSet; }
  template <typename FrameworkAttributesT = NetworkFrameworkAttributes>
  void SetFrameworkAttributes(FrameworkAttributesT&& value) {
    m_frameworkAttributesHasBeenSet = true;
    m_frameworkAttributes = std::forward<FrameworkAttributesT>(value);
  }
  template <typename FrameworkAttributesT = NetworkFrameworkAttributes>
  Network& WithFrameworkAttributes(FrameworkAttributesT&& value) {
    SetFrameworkAttributes(std::forward<FrameworkAttributesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The VPC endpoint service name of the VPC endpoint service of the network.
   * Members use the VPC endpoint service name to create a VPC endpoint to access
   * network resources.</p>
   */
  inline const Aws::String& GetVpcEndpointServiceName() const { return m_vpcEndpointServiceName; }
  inline bool VpcEndpointServiceNameHasBeenSet() const { return m_vpcEndpointServiceNameHasBeenSet; }
  template <typename VpcEndpointServiceNameT = Aws::String>
  void SetVpcEndpointServiceName(VpcEndpointServiceNameT&& value) {
    m_vpcEndpointServiceNameHasBeenSet = true;
    m_vpcEndpointServiceName = std::forward<VpcEndpointServiceNameT>(value);
  }
  template <typename VpcEndpointServiceNameT = Aws::String>
  Network& WithVpcEndpointServiceName(VpcEndpointServiceNameT&& value) {
    SetVpcEndpointServiceName(std::forward<VpcEndpointServiceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The voting rules that the network uses to decide if a proposal is
   * accepted.</p>
   */
  inline const VotingPolicy& GetVotingPolicy() const { return m_votingPolicy; }
  inline bool VotingPolicyHasBeenSet() const { return m_votingPolicyHasBeenSet; }
  template <typename VotingPolicyT = VotingPolicy>
  void SetVotingPolicy(VotingPolicyT&& value) {
    m_votingPolicyHasBeenSet = true;
    m_votingPolicy = std::forward<VotingPolicyT>(value);
  }
  template <typename VotingPolicyT = VotingPolicy>
  Network& WithVotingPolicy(VotingPolicyT&& value) {
    SetVotingPolicy(std::forward<VotingPolicyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current status of the network.</p>
   */
  inline NetworkStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(NetworkStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline Network& WithStatus(NetworkStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the network was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDate() const { return m_creationDate; }
  inline bool CreationDateHasBeenSet() const { return m_creationDateHasBeenSet; }
  template <typename CreationDateT = Aws::Utils::DateTime>
  void SetCreationDate(CreationDateT&& value) {
    m_creationDateHasBeenSet = true;
    m_creationDate = std::forward<CreationDateT>(value);
  }
  template <typename CreationDateT = Aws::Utils::DateTime>
  Network& WithCreationDate(CreationDateT&& value) {
    SetCreationDate(std::forward<CreationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags assigned to the network. Each tag consists of a key and optional
   * value.</p> <p>For more information about tags, see <a
   * href="https://docs.aws.amazon.com/managed-blockchain/latest/ethereum-dev/tagging-resources.html">Tagging
   * Resources</a> in the <i>Amazon Managed Blockchain Ethereum Developer Guide</i>,
   * or <a
   * href="https://docs.aws.amazon.com/managed-blockchain/latest/hyperledger-fabric-dev/tagging-resources.html">Tagging
   * Resources</a> in the <i>Amazon Managed Blockchain Hyperledger Fabric Developer
   * Guide</i>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  Network& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  Network& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the network. For more information about
   * ARNs and their format, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i>.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Network& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::String m_name;

  Aws::String m_description;

  Framework m_framework{Framework::NOT_SET};

  Aws::String m_frameworkVersion;

  NetworkFrameworkAttributes m_frameworkAttributes;

  Aws::String m_vpcEndpointServiceName;

  VotingPolicy m_votingPolicy;

  NetworkStatus m_status{NetworkStatus::NOT_SET};

  Aws::Utils::DateTime m_creationDate{};

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_arn;
  bool m_idHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_frameworkHasBeenSet = false;
  bool m_frameworkVersionHasBeenSet = false;
  bool m_frameworkAttributesHasBeenSet = false;
  bool m_vpcEndpointServiceNameHasBeenSet = false;
  bool m_votingPolicyHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_creationDateHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_arnHasBeenSet = false;
};

}  // namespace Model
}  // namespace ManagedBlockchain
}  // namespace Aws
