import gi

gi.require_version("GtkSource", "5")
from gi.repository import Gdk, Gio, GLib, GObject, Gtk, GtkSource

from gtkspellcheck import SpellChecker, NoDictionariesFound

import locale
import logging
from typing import Optional

from iotas.config_manager import ConfigManager
import iotas.const as const
from iotas import list_formatter


class EditorTextView(GtkSource.View):
    __gtype_name__ = "EditorTextView"

    __gsignals__ = {
        "margins-updated": (GObject.SignalFlags.RUN_FIRST, None, ()),
    }

    SOURCEVIEW_NO_SPELLCHECK_TAG = "gtksourceview:context-classes:no-spell-check"

    STANDARD_MARGIN = 36
    MINIMUM_MARGIN = 10

    def __init__(self) -> None:
        super().__init__()

        self.__css_provider: Optional[Gtk.CssProvider] = None
        self.__config_manager = ConfigManager.get_default()
        self.__line_length = -1
        self.__longpress_popover: Gtk.PopoverMenu
        self.__font_family = "monospace"
        self.__width_margins_set_for = None

        self.__config_manager.connect_changed(ConfigManager.FONT_SIZE, self.__update_font)

        self.__config_manager.connect_changed(
            ConfigManager.EDITOR_HEADER_BAR_VISIBILTY,
            lambda: self.__update_margins(self.get_width()),
        )

        controller = Gtk.EventControllerKey()
        controller.connect("key-pressed", self.__on_key_pressed)
        self.add_controller(controller)

        self.__spellchecker = None
        if self.__config_manager.spelling_enabled:
            self.__init_spellchecker()

        self.__config_manager.connect_changed(
            ConfigManager.SPELLING_ENABLED,
            self.__on_spelling_toggled,
        )

        # TODO Temporary hack to provide access to spelling on mobile
        self.__setup_longpress_touch_menu()

    def setup(self) -> None:
        """Initialisation after window creation."""
        window = self.get_root()
        window.connect(
            "notify::fullscreened", lambda _o, _v: self.__update_margins(self.get_width())
        )

    def do_size_allocate(self, width: int, height: int, baseline: int) -> None:
        """Allocates widget with a transformation that translates the origin to the position in
        allocation.

        :param int width: Width of the allocation
        :param int height: Height of the allocation
        :param int baseline: The baseline of the child
        """
        GtkSource.View.do_size_allocate(self, width, height, baseline)
        if width != self.__width_margins_set_for:
            self.__update_margins(width)

    def jump_to_insertion_point(self) -> None:
        """Jump to the insertion point.

        This jump is used to avoid issues with height allocation in `scroll_to_mark` when the
        buffer contains lines of varying height.
        """
        buffer = self.get_buffer()
        insert_iter = buffer.get_iter_at_mark(buffer.get_insert())
        location = self.get_iter_location(insert_iter)
        adjustment = self.get_vadjustment()
        adjustment.set_value(location.y)

    def update_font_family(self, font_family: str) -> None:
        """Update font family from system preference.

        :param str font_family: The font family
        """
        self.__font_family = font_family
        self.__update_font()

    @GObject.Property(type=bool, default=False)
    def spellchecker_enabled(self) -> bool:
        if self.__spellchecker is not None:
            return self.__spellchecker.enabled
        else:
            return False

    @spellchecker_enabled.setter
    def set_spellchecker_enabled(self, value: bool) -> None:
        if self.__spellchecker is not None:
            self.__spellchecker.enabled = value

    @GObject.Property(type=int, default=-1)
    def line_length(self) -> int:
        return self.__line_length

    @line_length.setter
    def set_line_length(self, value: int) -> None:
        self.__line_length = value
        # Ensure margin resizing works during resize
        self.__update_margins(self.get_width())

    # TODO Part of temporary hack to provide access to spelling menu on mobile
    def __on_longpress(self, _gesture: Gtk.GestureLongPress, x: float, y: float) -> None:
        # Long press menu is only currently used for spelling on mobile, don't show many if
        # spelling disabled
        if self.__spellchecker is None or not self.__spellchecker.enabled:
            return
        buffer_x, buffer_y = self.window_to_buffer_coords(Gtk.TextWindowType.TEXT, int(x), int(y))
        iter = self.get_iter_at_location(buffer_x, buffer_y)[1]
        self.__spellchecker.move_click_mark(iter)
        self.__spellchecker.populate_menu(self.__longpress_popover.get_menu_model())
        rect = Gdk.Rectangle()
        rect.x = x
        rect.y = y
        rect.width = rect.height = 1
        self.__longpress_popover.set_pointing_to(rect)
        self.__longpress_popover.popup()

    def __on_spelling_toggled(self) -> None:
        if self.__config_manager.spelling_enabled:
            self.__init_spellchecker()
            self.spellchecker_enabled = True
        else:
            self.spellchecker_enabled = False

    def __on_key_pressed(
        self,
        controller: Gtk.EventControllerKey,
        keyval: int,
        keycode: int,
        state: Gdk.ModifierType,
    ) -> bool:
        buffer = self.get_buffer()
        if keyval in (Gdk.KEY_Return, Gdk.KEY_KP_Enter, Gdk.KEY_ISO_Enter):
            if list_formatter.check_and_extend_list(buffer):
                # Attempt to track cursor in viewport
                mark = buffer.get_insert()
                insert_iter = buffer.get_iter_at_mark(mark)
                buffer_coords = self.get_iter_location(insert_iter)
                visible_rect = self.get_visible_rect()
                if buffer_coords.y + buffer_coords.height >= visible_rect.y + visible_rect.height:
                    # Without adding as an idle task we sometimes end up with an insufficient scroll
                    GLib.idle_add(self.emit, "move-viewport", Gtk.ScrollStep.STEPS, 1)

                return Gdk.EVENT_STOP
        elif keyval in (Gdk.KEY_Tab, Gdk.KEY_ISO_Left_Tab):
            if state == Gdk.ModifierType.SHIFT_MASK:
                list_formatter.decrease_indentation(buffer)
            else:
                list_formatter.increase_indentation(buffer)
            return Gdk.EVENT_STOP

        return Gdk.EVENT_PROPAGATE

    def __update_margins(self, width) -> None:
        self.__width_margins_set_for = width

        # Update top and bottom margin
        min_width_for_y_margin = 400
        max_width_for_y_margin = self.__line_length if self.__line_length > 500 else 800
        if width <= min_width_for_y_margin:
            y_margin_base = self.MINIMUM_MARGIN
        elif width < max_width_for_y_margin:
            value_range = max_width_for_y_margin - min_width_for_y_margin
            y_range = float(self.STANDARD_MARGIN - self.MINIMUM_MARGIN)
            in_range = width - min_width_for_y_margin
            y_margin_base = self.MINIMUM_MARGIN + int(in_range / value_range * y_range)
        else:
            y_margin_base = self.STANDARD_MARGIN

        # If the header bar is non hiding (set visible for the fullscreen state) there will be
        # padding elsewhere to ensure the text avoids the headerbar. Otherwise we add it here.
        if self.__config_manager.editor_header_bar_visible_for_window_state:
            y_margin = y_margin_base
        else:
            y_margin = y_margin_base + const.HEADER_BAR_HEIGHT

        if self.props.top_margin != y_margin:
            self.set_top_margin(y_margin)

        # Update side margins
        if width < 1:
            return
        if self.__line_length > 0 and width - 2 * self.STANDARD_MARGIN > self.__line_length:
            x_margin = (width - self.__line_length) / 2
        else:
            x_margin = y_margin_base
        if self.get_left_margin() != x_margin:
            self.set_left_margin(x_margin)
            self.set_right_margin(x_margin)

        self.emit("margins-updated")

    def __update_font(self) -> None:
        if not self.__css_provider:
            self.__css_provider = Gtk.CssProvider()
            self.get_style_context().add_provider(
                self.__css_provider, Gtk.STYLE_PROVIDER_PRIORITY_USER
            )

        size = self.__config_manager.font_size
        style = f"""
        .editor-textview {{
            font-size: {size}pt;
            font-family: {self.__font_family}, monospace;
        }}"""
        self.__css_provider.load_from_data(style, -1)

    def __init_spellchecker(self) -> None:
        if self.__spellchecker is not None:
            return

        pref_language = self.__config_manager.spelling_language
        if pref_language is not None:
            language = pref_language
            logging.debug(f'Attempting to use spelling language from preference "{pref_language}"')
        else:
            default_language, _encoding = locale.getdefaultlocale()
            if default_language:
                language = default_language
                logging.debug(f'Attempting to use locale default spelling language "{language}"')
            else:
                logging.warning("Couldn't determine locale default language")

        try:
            self.__spellchecker = SpellChecker(self, language, collapse=False)
        except NoDictionariesFound:
            # Not communicating via UI for now as systems without any dictionaries are likely to
            # be corner cases, bespoke systems eg. Arch etc
            logging.warning("Disabling spell checker as no dictionaries were found")
            return

        assert self.__spellchecker  # mypy

        self.spellchecker_enabled = False
        self.__spellchecker.batched_rechecking = True
        if pref_language is not None:
            self.__verify_preferred_language_in_use(pref_language)
        self.__spellchecker.connect(
            "notify::language", lambda _o, _v: self.__spelling_language_changed()
        )

        buffer = self.get_buffer()
        table = buffer.get_tag_table()

        def process_added_tag(tag):
            if tag.get_property("name") == self.SOURCEVIEW_NO_SPELLCHECK_TAG:
                self.__spellchecker.append_ignore_tag(self.SOURCEVIEW_NO_SPELLCHECK_TAG)

        def tag_added(tag, *args):
            if isinstance(tag, Gtk.TextTag):
                process_added_tag(tag)
            elif len(args) > 0 and isinstance(args[0], Gtk.TextTag):
                process_added_tag(args[0])

        table.connect("tag-added", tag_added)

    def __verify_preferred_language_in_use(self, pref_language: str) -> None:
        assert self.__spellchecker
        language_in_use = self.__spellchecker.language
        if language_in_use != pref_language:
            logging.warning(
                f'Spelling language from preference "{pref_language}" not found, clearing'
                " preference"
            )

            logging.info("Available languages:")
            for code, name in self.__spellchecker.languages:
                logging.info(" - %s (%5s)" % (name, code))

            self.__config_manager.spelling_language = ""

    def __spelling_language_changed(self) -> None:
        assert self.__spellchecker
        language = self.__spellchecker.language
        logging.info(f'New spelling language "{language}"')
        self.__config_manager.spelling_language = language

    # TODO Part of temporary hack to provide access to spelling menu on mobile
    def __setup_longpress_touch_menu(self) -> None:
        gesture = Gtk.GestureLongPress.new()
        gesture.set_touch_only(True)

        gesture.connect("pressed", self.__on_longpress)

        self.add_controller(gesture)

        self.__longpress_popover = Gtk.PopoverMenu.new_from_model(Gio.Menu())
        self.__longpress_popover.set_parent(self)
