/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sling.auth.saml2.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TokenStore {
    private static final char[] TOHEX = "0123456789abcdef".toCharArray();
    private static final String SHA1PRNG = "SHA1PRNG";
    private static final String ALGORITHM = "HmacSHA256";
    private static final String UTF_8 = "UTF-8";
    private static final int TOKEN_BUFFER_SIZE = 5;
    public final Logger log = LoggerFactory.getLogger(TokenStore.class);
    private final long ttl;
    private long nextUpdate = System.currentTimeMillis();
    private volatile int currentToken = 0;
    private SecretKey[] currentTokens;
    private SecureRandom random;
    private File tokenFile;
    private File tmpTokenFile;

    TokenStore(File tokenFile, long sessionTimeout, boolean fastSeed) throws NoSuchAlgorithmException, InvalidKeyException, IllegalStateException {
        if (tokenFile == null) {
            throw new NullPointerException("tokenfile");
        }
        this.random = SecureRandom.getInstance(SHA1PRNG);
        this.ttl = sessionTimeout;
        this.tokenFile = tokenFile;
        this.tmpTokenFile = new File(tokenFile + ".tmp");
        this.loadTokens();
        if (fastSeed) {
            this.random.setSeed(TokenStore.getFastEntropy());
        } else {
            this.log.info("Seeding the secure random number generator can take up to several minutes on some operating systems depending upon environment factors. If this is a problem for you, set the system property 'java.security.egd' to 'file:/dev/./urandom' or enable the Fast Seed Generator in the Web Console");
        }
        byte[] b = new byte[20];
        this.random.nextBytes(b);
        SecretKeySpec secretKey = new SecretKeySpec(b, ALGORITHM);
        Mac m = Mac.getInstance(ALGORITHM);
        m.init(secretKey);
        m.update(UTF_8.getBytes(StandardCharsets.UTF_8));
        m.doFinal();
    }

    String encode(long expires, String userId) throws IllegalStateException, UnsupportedEncodingException, NoSuchAlgorithmException, InvalidKeyException {
        int token = this.getActiveToken();
        SecretKey key = this.currentTokens[token];
        return this.encode(expires, userId, token, key);
    }

    private String encode(long expires, String userId, int token, SecretKey key) throws IllegalStateException, UnsupportedEncodingException, NoSuchAlgorithmException, InvalidKeyException {
        String cookiePayload = "" + token + expires + "@" + userId;
        Mac m = Mac.getInstance(ALGORITHM);
        m.init(key);
        m.update(cookiePayload.getBytes(StandardCharsets.UTF_8));
        String cookieValue = this.byteToHex(m.doFinal());
        return cookieValue + "@" + cookiePayload;
    }

    static String[] split(String authData) {
        String[] parts = StringUtils.split((String)authData, (String)"@", (int)3);
        if (parts != null && parts.length == 3) {
            return parts;
        }
        return null;
    }

    boolean isValid(String value) {
        String[] parts = TokenStore.split(value);
        if (parts != null) {
            int tokenNumber = parts[1].charAt(0) - 48;
            if (tokenNumber >= 0 && tokenNumber < this.currentTokens.length) {
                long cookieTime = Long.parseLong(parts[1].substring(1));
                if (System.currentTimeMillis() < cookieTime) {
                    try {
                        SecretKey secretKey = this.currentTokens[tokenNumber];
                        String hmac = this.encode(cookieTime, parts[2], tokenNumber, secretKey);
                        return value.equals(hmac);
                    }
                    catch (UnsupportedEncodingException | ArrayIndexOutOfBoundsException | IllegalStateException | InvalidKeyException | NoSuchAlgorithmException e) {
                        this.log.error(e.getMessage(), (Throwable)e);
                        this.log.error("AuthNCookie value '{}' is invalid", (Object)value);
                    }
                } else {
                    this.log.error("AuthNCookie value '{}' has expired {}ms ago", (Object)value, (Object)(System.currentTimeMillis() - cookieTime));
                }
            } else {
                this.log.error("AuthNCookie value '{}' is invalid: refers to an invalid token number, {}", (Object)value, (Object)tokenNumber);
            }
        } else {
            this.log.error("AuthNCookie value '{}' has invalid format", (Object)value);
        }
        return false;
    }

    synchronized int getActiveToken() {
        if (System.currentTimeMillis() > this.nextUpdate || this.currentTokens[this.currentToken] == null) {
            this.nextUpdate = System.currentTimeMillis() + this.ttl / (long)(this.currentTokens.length - 1);
            byte[] b = new byte[20];
            this.random.nextBytes(b);
            SecretKeySpec newToken = new SecretKeySpec(b, ALGORITHM);
            int nextToken = this.currentToken + 1;
            if (nextToken == this.currentTokens.length) {
                nextToken = 0;
            }
            this.currentTokens[nextToken] = newToken;
            this.currentToken = nextToken;
            this.saveTokens();
        }
        return this.currentToken;
    }

    void saveTokens() {
        try (FileOutputStream fout = new FileOutputStream(this.tmpTokenFile);
             DataOutputStream keyOutputStream = new DataOutputStream(fout);){
            File parent = this.tokenFile.getAbsoluteFile().getParentFile();
            this.log.info("Token File {} parent {} ", (Object)this.tokenFile, (Object)parent);
            if (!parent.exists()) {
                parent.mkdirs();
            }
            keyOutputStream.writeInt(this.currentToken);
            keyOutputStream.writeLong(this.nextUpdate);
            for (int i = 0; i < this.currentTokens.length; ++i) {
                if (this.currentTokens[i] == null) {
                    keyOutputStream.writeInt(0);
                    continue;
                }
                keyOutputStream.writeInt(1);
                byte[] b = this.currentTokens[i].getEncoded();
                keyOutputStream.writeInt(b.length);
                keyOutputStream.write(b);
            }
            boolean renameWorked = this.tmpTokenFile.renameTo(this.tokenFile);
            if (!renameWorked) {
                this.log.error("renaming token file failed");
            }
        }
        catch (IOException e) {
            this.log.error("Failed to save cookie keys {}", (Object)e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadTokens() {
        if (this.tokenFile.isFile() && this.tokenFile.canRead()) {
            FilterInputStream keyInputStream = null;
            try (FileInputStream fin = new FileInputStream(this.tokenFile);){
                keyInputStream = new DataInputStream(fin);
                int newCurrentToken = ((DataInputStream)keyInputStream).readInt();
                long newNextUpdate = ((DataInputStream)keyInputStream).readLong();
                SecretKey[] newKeys = new SecretKey[5];
                for (int i = 0; i < newKeys.length; ++i) {
                    int isNull = ((DataInputStream)keyInputStream).readInt();
                    if (isNull == 1) {
                        int l = ((DataInputStream)keyInputStream).readInt();
                        byte[] b = new byte[l];
                        int readBytes = ((DataInputStream)keyInputStream).read(b);
                        if (readBytes != l) {
                            throw new IOException("could not confirm bytes read");
                        }
                        newKeys[i] = new SecretKeySpec(b, ALGORITHM);
                        continue;
                    }
                    newKeys[i] = null;
                }
                this.nextUpdate = newNextUpdate;
                this.currentToken = newCurrentToken;
                this.currentTokens = newKeys;
            }
            catch (IOException e) {
                this.log.error("Failed to load cookie keys {}", (Object)e.getMessage());
            }
            finally {
                if (keyInputStream != null) {
                    try {
                        keyInputStream.close();
                    }
                    catch (IOException e) {
                        this.log.warn("failed to close steam {}", (Object)e.getMessage());
                    }
                }
            }
        }
        if (this.currentTokens == null) {
            this.currentTokens = new SecretKey[5];
            this.nextUpdate = System.currentTimeMillis();
            this.currentToken = 0;
        }
    }

    private String byteToHex(byte[] base) {
        char[] c = new char[base.length * 2];
        int i = 0;
        byte[] byArray = base;
        int n = byArray.length;
        for (int j = 0; j < n; ++j) {
            int b;
            int j2 = b = byArray[j];
            c[i++] = TOHEX[(j2 += 128) / 16];
            c[i++] = TOHEX[j2 % 16];
        }
        return new String(c);
    }

    private static byte[] getFastEntropy() {
        MessageDigest md;
        try {
            md = MessageDigest.getInstance("SHA-256");
        }
        catch (NoSuchAlgorithmException nsae) {
            throw new InternalError("internal error: SHA-1 not available.");
        }
        TokenStore.update(md, System.currentTimeMillis());
        TokenStore.update(md, System.nanoTime());
        File file = new File(System.getProperty("java.io.tmpdir"));
        File[] entries = file.listFiles();
        if (entries != null) {
            for (File entry : entries) {
                md.update(entry.getName().getBytes());
                TokenStore.update(md, entry.lastModified());
                TokenStore.update(md, entry.length());
            }
        }
        return md.digest();
    }

    private static void update(MessageDigest md, long value) {
        value ^= value << 21;
        value ^= value >>> 35;
        value ^= value << 4;
        for (int i = 0; i < 8; ++i) {
            md.update((byte)value);
            value >>= 8;
        }
    }
}

