﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/migrationhuborchestrator/MigrationHubOrchestratorRequest.h>
#include <aws/migrationhuborchestrator/MigrationHubOrchestrator_EXPORTS.h>
#include <aws/migrationhuborchestrator/model/MigrationWorkflowStatusEnum.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace MigrationHubOrchestrator {
namespace Model {

/**
 */
class ListWorkflowsRequest : public MigrationHubOrchestratorRequest {
 public:
  AWS_MIGRATIONHUBORCHESTRATOR_API ListWorkflowsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListWorkflows"; }

  AWS_MIGRATIONHUBORCHESTRATOR_API Aws::String SerializePayload() const override;

  AWS_MIGRATIONHUBORCHESTRATOR_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The maximum number of results that can be returned.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListWorkflowsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListWorkflowsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the template.</p>
   */
  inline const Aws::String& GetTemplateId() const { return m_templateId; }
  inline bool TemplateIdHasBeenSet() const { return m_templateIdHasBeenSet; }
  template <typename TemplateIdT = Aws::String>
  void SetTemplateId(TemplateIdT&& value) {
    m_templateIdHasBeenSet = true;
    m_templateId = std::forward<TemplateIdT>(value);
  }
  template <typename TemplateIdT = Aws::String>
  ListWorkflowsRequest& WithTemplateId(TemplateIdT&& value) {
    SetTemplateId(std::forward<TemplateIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the application configured in Application Discovery Service.</p>
   */
  inline const Aws::String& GetAdsApplicationConfigurationName() const { return m_adsApplicationConfigurationName; }
  inline bool AdsApplicationConfigurationNameHasBeenSet() const { return m_adsApplicationConfigurationNameHasBeenSet; }
  template <typename AdsApplicationConfigurationNameT = Aws::String>
  void SetAdsApplicationConfigurationName(AdsApplicationConfigurationNameT&& value) {
    m_adsApplicationConfigurationNameHasBeenSet = true;
    m_adsApplicationConfigurationName = std::forward<AdsApplicationConfigurationNameT>(value);
  }
  template <typename AdsApplicationConfigurationNameT = Aws::String>
  ListWorkflowsRequest& WithAdsApplicationConfigurationName(AdsApplicationConfigurationNameT&& value) {
    SetAdsApplicationConfigurationName(std::forward<AdsApplicationConfigurationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the migration workflow.</p>
   */
  inline MigrationWorkflowStatusEnum GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(MigrationWorkflowStatusEnum value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline ListWorkflowsRequest& WithStatus(MigrationWorkflowStatusEnum value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the migration workflow.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  ListWorkflowsRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}
 private:
  int m_maxResults{0};

  Aws::String m_nextToken;

  Aws::String m_templateId;

  Aws::String m_adsApplicationConfigurationName;

  MigrationWorkflowStatusEnum m_status{MigrationWorkflowStatusEnum::NOT_SET};

  Aws::String m_name;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_templateIdHasBeenSet = false;
  bool m_adsApplicationConfigurationNameHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_nameHasBeenSet = false;
};

}  // namespace Model
}  // namespace MigrationHubOrchestrator
}  // namespace Aws
