
=pod

=head1 COPYRIGHT

# (c) 1992-2024 Intel Corporation.                                              
# Intel, the Intel logo, Intel, MegaCore, NIOS II, Quartus and TalkBack         
# words and logos are trademarks of Intel Corporation or its                    
# subsidiaries in the U.S. and/or other countries. Other marks and              
# brands may be claimed as the property of others.                              
# See Trademarks on intel.com for full list of Intel trademarks or the          
# Trademarks & Brands Names Database (if Intel)                                 
# or See www.Intel.com/legal (if Altera)                                        
# Your use of Intel Corporation's design tools, logic functions and             
# other software and tools, and its AMPP partner logic functions, and           
# any output files any of the foregoing (including device programming           
# or simulation files), and any associated documentation or information         
# are expressly subject to the terms and conditions of the Altera               
# Program License Subscription Agreement, Intel MegaCore Function               
# License Agreement, or other applicable license agreement, including,          
# without limitation, that your use is for the sole purpose of                  
# programming logic devices manufactured by Intel and sold by Intel or          
# its authorized distributors.                                                  
# Please refer to the applicable agreement for further details.                 


=cut

package acl::Report;
require acl::Env;
require acl::File;
require acl::Common;
require acl::Report_reader;
require Exporter;
require JSON::PP;
use File::Spec;

@ISA        = qw(Exporter);
@EXPORT     = qw( escape_string
                  create_reporting_tool
                  get_file_list_from_dependency_files
                  parse_acl_quartus_report
                  print_area_estimate_or_ii_message
                  create_info_json_transfer_file
                  create_warning_transfer_file
                  get_standalone_exe
                  create_kernel_list_file
                  create_full_quartus_json
                );

use strict;

my $module = 'acl::Report';

sub log_string { }  # Dummy

=head1 NAME

acl::Report - Reporting utilities

=head1 VERSION

$Header$

=head1 SYNOPSIS


=head1 DESCRIPTION

This module provides utilities for the HLD Reports.

All methods names may optionally be imported, e.g.:

   use acl::Report qw( create_json_file );

=cut

use constant KERNEL_FILELIST_NAME => "kernel_list_file.txt";

=head2 get_standalone_exe()

Returns the path to the standalone exe, or an empty string if can't find the exe.

=cut

sub get_standalone_exe {
  my $standalone_exe_path = acl::Env::sdk_root()."/share/lib/reports/standalone_report/report_helper";
  if (acl::Env::is_windows()) {
    $standalone_exe_path = $standalone_exe_path.".exe";
  }

  if (-e $standalone_exe_path && -x $standalone_exe_path) {
    return $standalone_exe_path;
  }
  return "";
}

=head2 escape_string($string, $escape_quotes_and_newline, $is_source_code)

Given a $string, replace all control characters with their octal equivalent or
escape them appropriately.  Optionally escape quotes and newlines.
if $is_source_code is enabled, then we always escape quotes and newlines.

=cut

sub escape_string {
  my $string = shift;
  my $escape_quotes_and_newline = @_ ? shift : 1;
  my $is_source_code = @_ ? shift : 0;

  if (not $escape_quotes_and_newline and $is_source_code) {
    $escape_quotes_and_newline = 1;
  }

  if ($is_source_code) {
    $string =~ s/\\/\\\\/g; # adds escape for \
    $string =~ s/"/\\"/g;  # adds escape for quotes
    $string =~ s/	/\\t/g;  # adds escape for tabs
  } else {
    # Add escape string to a real escape string, i.e. not \n, \t, \f, \b, \r, \", \\
    $string =~ s/[^\\](\\[^ntfbr"\\])/\\$1/g;

    # corner cases
    $string =~ s/^(\\[^ntfbr"\\])/\\$1/g;  # add \\ to the start of the escape character at the start of the string
    $string =~ s/([^\\]\\)$/$1\\/g;  # add \\ to the start of the escape character at the end of the string

    $string =~ s/(?<!\\)\\n/\\\\n/g; # escaping all newline characters ("\n" becomes "\\n")
    $string =~ s/(?<!\\)\\t/\\\\t/g; # escaping all tab characters ("\t" becomes "\\t")
    $string =~ s/(?<!\\)\\f/\\\\f/g; # escaping all form feed characters ("\f" becomes "\\f")
    $string =~ s/(?<!\\)\\b/\\\\b/g; # escaping all backspace characters ("\b" becomes "\\b")
    $string =~ s/(?<!\\)\\r/\\\\r/g; # escaping all carriage returns ("\r" becomes "\\r")

    # Now escape the quotes and/or escape with quotes
    if ($escape_quotes_and_newline) {
      # decide if it's just " or \". We do this by first check if \"
      # "  -> \"
      # \" -> \\\"
      $string =~ s/(?<!\\)\\(")/\\\\\"/g;  # adds the escape after escape
      $string =~ s/\"/\\"/g;  # adds escape for quotes
    }
  }
  # replacing octal line feed (newline) characters with unicode equivalents
  $string =~ s/(\012|\015\012|\015\015\012?)/\\u000A/g if $escape_quotes_and_newline;
  $string =~ s/\015/\\u000A/g if $escape_quotes_and_newline;

  return $string;
}

=head2 create_info_json_transfer_file($work_dir, $progCmd, $proj_name, $target_family, $target_device, $quartus_version, $command, $board, [$acl_version, $build_num])

Create a file that stores important compiler information to be used for
info.json creation.

=cut

sub create_info_json_transfer_file {
  my ($work_dir, $progCmd, $proj_name, $target_family, $target_device, $quartus_version, $command, $board_variant, $acl_version, $build_num) = @_;
  
  # as to not mess with any whitespace within the info being passed,
  # break lines and info with these internal separators
  my $between_line_separator = q{__INTEL_INTERNAL_INFO_LIST_SEPARATOR__};
  my $within_line_separator = q{__INTEL_INTERNAL_INFO_LINE_SEPARATOR__};
  my @info_list = ();

  # build_num and acl_version are useful for testing, but never used in the full flow
  my $version = (defined($acl_version) && defined($build_num) && $acl_version && $build_num) ? $acl_version." Build ".$build_num :
                                                                       qq{2024.1.0 Build 1097aa05bafa5450f9d5ca1475d35f5295a7b2fb};

  my $board_string = "";
  if ($progCmd eq 'AOC' || $progCmd eq 'SYCL') {
    # board_name() returns an list with element 0 being the board name
    my ($board_name) = acl::Env::board_name();
    $board_string = "$board_name:$board_variant";
  }

  push @info_list, "name".$within_line_separator.$proj_name;
  push @info_list, "version".$within_line_separator.$version;
  push @info_list, "command".$within_line_separator.$command;
  push @info_list, "quartus".$within_line_separator.$quartus_version;
  push @info_list, "target_family".$within_line_separator.$target_family;
  push @info_list, "target_device".$within_line_separator.$target_device;

  if (!($board_string eq "")) {
    push @info_list, "board".$within_line_separator.$board_string;
  }
  my $info_string = join($between_line_separator, @info_list);

  my $temporary_info_file = $work_dir."/info_from_compiler.tmp";
  (open my $info_transfer_file, '>', $temporary_info_file) or die("Unable to make a temporary info transfer file");
  print $info_transfer_file $info_string;
  close($info_transfer_file);
  # return the name of the file to be passed to fpga_report
  return acl::File::abs_path($temporary_info_file);
}

=head2 create_warning_transfer_file($filename, $warnings_filelist)

Go through all the files in the warning filelist, and combine all of their 
lines into a single warnings file that contains all the warnings from the compile.

=cut

sub create_warning_transfer_file {
  my ($filename, $warnings_filelist) = @_;
  (open my $all_warnings_file, '>', $filename) or die("Unable to make a temporary warning file");
  foreach my $error_file_name (@{$warnings_filelist}) {
    if (open my $file, '<', $error_file_name) {
      while (my $line = <$file>) {
        print $all_warnings_file $line;
      }
    }
  }
  close( $all_warnings_file );
  return;
}

=head2 create_reporting_tool()
Create the whole report folder, populate with:
Compile info: command, target device, pre-Quartus compile instruction, source files
Compiler statistics: loops optimization result, estimated resource utilization, scheduler data
# Types of exception handling:
#  err - Error, i.e. missing JSON, incomplete/ missing attribute
#  NA  - Not available, i.e. aocr to aocx compiling quartus, running simulation
#  NA2 - Not applicable, data may exist, but front-end will not show even it has data,
#        i.e. i++ && incr
=cut

sub create_reporting_tool {
  # manditory arguments, no matter the flow
  my $progCmd = shift;            # accepted values are: "AOC": OpenCL, "i++": HLS, or "SYCL": oneAPI SYCL
  my $work_dir = shift;           # path of working directory
  my $project_name = shift;
  my $reports_creation_type = shift; # <full, obj>
  my $devicefamily = shift;
  my $devicemodel = shift;
  my $quartus_version_str = shift;
  my $compiler_args = shift;
  my $board_variant = shift;
  # up till here is necessary for both modes (full, obj)
  # optional if mode is obj - these will be undefined
  my $warnings_filelist = shift;   # array of files containing compiler warnings
  my $dep_files_list_filename = shift; # a filename for a file listing deps files
  my $disabled_lmem_repl = shift;  # is local memory replication disabled? print a warning if yes
  my $debug_symbols_enabled = shift; # are debug symbols enabled? default to yes

  my $verbose = acl::Common::get_verbose();
  my @cleanup_list = [];
  my $temporary_warning_file = acl::File::abs_path($work_dir."/all_warnings.tmp");
  if ($reports_creation_type eq "full") {
    create_warning_transfer_file($temporary_warning_file, $warnings_filelist);
    push @cleanup_list, $temporary_warning_file;
    # in the full flow, there is also a dependency list file - clean it too
    push @cleanup_list, $dep_files_list_filename;
  }

  my $temporary_info_file = create_info_json_transfer_file($work_dir, $progCmd, $project_name, $devicefamily, $devicemodel, $quartus_version_str, $compiler_args, $board_variant);
  push @cleanup_list, $temporary_info_file;

  my $standalone_exe_path = get_standalone_exe();
  if ( ! $standalone_exe_path eq "" ) {
    my @create_reporting_tool_command = (
        $standalone_exe_path,
        'create_reporting_tool',
        '--sdk-root=' . acl::Env::sdk_root(),
        '--project-name=' . $project_name,
        '--compiler-program=' . $progCmd,
        '--work-directory=' . acl::File::abs_path($work_dir),
        '--info-filename=' . $temporary_info_file
      );
    push @create_reporting_tool_command, '--save-temps' if acl::Common::get_save_temps();
    if ($reports_creation_type eq "full") {
      push @create_reporting_tool_command, '--warning-filename=' . $temporary_warning_file;
      push @create_reporting_tool_command, '--lmem-disabled' if ($disabled_lmem_repl);
      push @create_reporting_tool_command, '--dependency-file-list-filename=' . $dep_files_list_filename;
      push @create_reporting_tool_command, '--debug-symbols-disabled' if (!$debug_symbols_enabled);
    }

    my $rc = acl::Common::mysystem_full({
      'stdout' => 'static_reports_creation_log.temp',
      'stderr' => 'static_reports_creation_log.temp'
      },
      @create_reporting_tool_command);
    if ($rc != 0) {
      print STDERR qq{Error: unable to fully generate static reports for this compile!};
    } else {
      push @cleanup_list, 'static_reports_creation_log.temp';
    }
  }

  # clean up the temporary info-transferring files
  if (! acl::Common::get_save_temps()) {
    foreach my $cleanup_list_item (@cleanup_list) {
      unlink $cleanup_list_item;
    }
  }

  return;
}

=head2 create_kernel_list_file()

Create a file which contains all of the kernels in the design.
If component list is passed in, use that, and otherwise try to get
the kernel list from the compiler 

=cut
sub create_kernel_list_file {
  my ($base, $progCmd, $work_dir, $component_list) = @_;
  my @final_list;
  if ($component_list) {
    @final_list = @$component_list;
  } else {
    my @kernel_list = acl::AOCDriverCommon::get_kernel_list($base);
    if ($progCmd eq "SYCL-IPA") {
      # IPA flow doesn't have a "kernel system"
      # but it does have a "device image" (or di) for the whole design
      # that the users should be using within their system
      # Note that the USE_ME_VERBATIM prefix tells the javascript part to
      # look for this exact entity name (nothing before or after) in the
      # quartus report (can be used for top level entities or for other
      # non-kernel entities we want to parse out of the quartus fit report)
      push @final_list, "USE_ME_VERBATIM:${base}_di";
    } else {
      # in full flow, it's called "kernel_system"
      push @final_list, "USE_ME_VERBATIM:kernel_system";
    }
    foreach (@kernel_list) {
      # The entries in the list are actually: kernel_name,counter
      # just want the kernel name here
      my @split_kernel_entry = split(',', $_);
      push @final_list, $split_kernel_entry[0];
    }
  }
  my $kernel_list_filename = acl::File::abs_path($work_dir) . '/' . KERNEL_FILELIST_NAME;
  open( my $kernel_list_fh, "> $kernel_list_filename" ) or print "Warning: could not create internal file needed to process the resource values from quartus. Resource values will be unavailable in the quartus summary!";
  foreach (@final_list) {
    print $kernel_list_fh "$_\n";
  }
  close $kernel_list_fh;
  return $kernel_list_filename;
}

=head2 create_full_quartus_json

Create the quartus.json file after finishing a quartus compile.
Package this file into the reports.zip, and adjust the quartus_data.js file.
If dry_run mode is set, do not execute, just return the fpga_report command
that will be run (for testing purposes).

=cut

sub create_full_quartus_json {
  my ($work_dir, $progCmd, $do_database_call, $base, 
    $quartus_version, $acl_quartus_report_path, $project_name,
    $project_rev, $quartus_is_std, $quartus_work_dir,
    $component_list, $dry_run) = @_;

  my $output_zip_name = "${base}_report.zip";
  my @cleanup_list = ();

  my $standalone_exe_path = get_standalone_exe();
  if (-e $standalone_exe_path && -x $standalone_exe_path) {
    my @create_command = (
        $standalone_exe_path,
        'create_quartus_json',
        '--work-directory='   . acl::File::abs_path($work_dir),
        '--compiler-program=' . $progCmd,
        '--output-zip-name='  . $output_zip_name,
      );
    if ($do_database_call) {
      # First check if this file already exists in the directory we're working in
      # This file is actually also made during the image/early part of the compile
      # The file needs to be made then because if the compile is broken up into stages,
      # the quartus compile stage can have a different name from the image/early compile part,
      # so by this point in time (at the quartus compile stage) we don't know the name of 
      # the .bc.xml file that has the name of all the kernels.
      # However if that file doens't exist yet for whatever reason (running in HLS is one of them),
      # try to make the file now.
      my $kernel_list_filename = KERNEL_FILELIST_NAME;
      if ( !$dry_run && ! -f $kernel_list_filename ) {
        $kernel_list_filename = create_kernel_list_file($base, $progCmd, $work_dir, $component_list);
      }
      push @cleanup_list, $kernel_list_filename;
      push @create_command, '--kernel-list-file=' . $kernel_list_filename;

      if ($project_name ne "") {
        push @create_command, '--quartus-project-name=' . $project_name;
      }
      if ($project_rev ne "") {
        push @create_command, '--quartus-project-revision=' . $project_rev;
      }
      if ($quartus_work_dir ne "") {
        push @create_command, '--quartus-work-directory=' . $quartus_work_dir;
      }
    } else {
      push @create_command, '--dont-do-database-call';
    }
    if (($progCmd eq "SYCL") && $acl_quartus_report_path ne "") {
      push @create_command, '--acl-quartus-report-path=' . $acl_quartus_report_path;
    }
    if ($quartus_version) {
      push @create_command, '--quartus-version=' . $quartus_version;
    }
    if ($quartus_is_std) {
      push @create_command, '--quartus-is-std';
    }

    if ($dry_run) {
      return join(" ", @create_command);
    }
    # don't die() here on failure, but do warn that the static reports couldn't gather quartus data
    my $rc = acl::Common::mysystem_full({
      'stdout' => 'quartus_json_creation_log.temp',
      'stderr' => 'quartus_json_creation_log.temp'
      },
      @create_command);
    if ($rc != 0) {
      print STDERR qq{Warning: could not gather information about the quartus compile for the static reports!\n};
    } else {
      push @cleanup_list, 'quartus_json_creation_log.temp';
      push @cleanup_list, 'quartus_database_output_tmp.txt';
    }
  } else {
    print STDERR qq{Warning: could not find standalone reports executable, static reports may be incomplete or nonexistent!\nEnsure your installation of oneAPI is complete!\n};
  }

  # clean up the temporary info-transferring files
  if (! acl::Common::get_save_temps()) {
    foreach my $cleanup_list_item (@cleanup_list) {
      unlink $cleanup_list_item;
    }
  }

  return;
}

=head2 get_file_list_from_dependency_files($work_dir, $dep_file, ...)

Parse each $dep_file, returning a list of all unique files given in the
dependency files.

=cut

sub get_file_list_from_dependency_files {
  my $work_dir = shift;
  my @filelist;
  my $ocl_header_filename = q{opencl_lib.h};
  # Temp file should be in the work dir
  my $dependency_files_list_filename = File::Spec->catfile($work_dir, q{temp_dependencies_list.txt.temp});
  # Make sure we have the full path so the future scripts can't get confused
  $dependency_files_list_filename = acl::File::abs_path($dependency_files_list_filename);
  for my $dependency_file (@_) {
    if( -f $dependency_file and open (my $fin, '<', $dependency_file) ) {
      foreach my $line (<$fin>) {
        # File/path may contain spaces and/or tabs.  Replace them with special
        # tokens, then swap back after splitting.
        $line =~ s/\\ /INTEL_FPGA_THERE_IS_A_SPACE_HERE/g;
        $line =~ s/(?!^)\011/INTEL_FPGA_THERE_IS_A_TAB_HERE/g;
        $line =~ s/\015/INTEL_FPGA_THERE_IS_A_CR_HERE/g;
        foreach my $file (split '\s', $line) {
          $file =~ s/INTEL_FPGA_THERE_IS_A_SPACE_HERE/ /g;
          $file =~ s/INTEL_FPGA_THERE_IS_A_TAB_HERE/\011/g;
          $file =~ s/INTEL_FPGA_THERE_IS_A_CR_HERE/\015/g;
          # The dependency file may include relative paths to source files,
          # which causes problems later on in the flow if we're running from a
          # different directory - expanding to absolute paths here resolves the
          # issue.  See case:1409393632.
          if (-f $file) {
            # "Unknown" files are included when opaque objects
            # (such as image objects) are in the source code
            if ( $file =~ m/$ocl_header_filename$/ ) {
              next;
            }
            $file = acl::File::abs_path($file);
            push @filelist, $file;
          }
        }
      }
      close($fin);
    }
  }
  open( DEPENDENCY_FILES, "> $dependency_files_list_filename" );
  foreach my $filename ( (sort @filelist) ) {
    print DEPENDENCY_FILES "$filename\n";
  }
  close DEPENDENCY_FILES;
  return $dependency_files_list_filename;
}


=head2 append_to_log($filename, $filename, ..., $filename, $logfile)

Prints each of the givenfiles to the given logfile.

=cut

sub append_to_log {
  my $logfile= pop @_;
  open my $logfh, '>>', $logfile;
  foreach my $infile (@_) {
    open my $tempfh, '<', $infile;
    while(my $l = <$tempfh>) {
      print $logfh $l;
    }
    close $tempfh;
  }
  close $logfh;
}


=head2 move_to_err($filename, $filename, $filename, ...)

Prints each file given in to stderr, and then unlinks the given files.

=cut

sub move_to_err {
  foreach my $infile (@_) {
    open my $errfh, '<', $infile; ## We currently can't guarantee existence of $infile # or mydie("Couldn't open $infile for appending.");
    while(my $l = <$errfh>) {
      print STDERR $l;
    }
    close $errfh;
    unlink $infile;
  }
}


=head2 append_to_err($filename, $filename, $filename, ...)

Prints each file given in to stderr. Does NOT unlink the files

=cut

sub append_to_err {
  foreach my $infile (@_) {
    open my $errfh, '<', $infile; ## We currently can't guarantee existence of $infile # or mydie("Couldn't open $infile for appending.");
    while(my $l = <$errfh>) {
      print STDERR $l;
    }
    close $errfh;
  }
}


=head2 append_to_out($filename, $filename, $filename, ...)

Prints each file given in to stdout. Does NOT unlink the files

=cut

sub append_to_out {
  foreach my $infile (@_) {
    open my $outfh, '<', $infile; ## We currently can't guarantee existence of $infile # or mydie("Couldn't open $infile for appending.");
    while(my $l = <$outfh>) {
      print STDOUT $l;
    }
    close $outfh;
  }
}


=head2 filter_llvm_time_passes($logfile, $time_log, $time_passes)

This functions filters output from LLVM's --time-passes
into the time log. The source log file is modified to not
contain this output as well.
TODO: move this function to Util.pm when Util.pm created in FB:512992

=cut

sub filter_llvm_time_passes {
  my ($logfile) = shift;
  my ($time_passes) = shift;

  if ($time_passes) {
    open (my $L, '<', $logfile);
    my @lines = <$L>;
    close ($L);
    # Look for the specific output pattern that corresponds to the
    # LLVM --time-passes report.
    for (my $i = 0; $i <= $#lines;) {
      my $l = $lines[$i];
      if ($l =~ m/^\s+\.\.\. Pass execution timing report \.\.\.\s+$/ ||
          $l =~ m/LLVM IR Parsing/) {
        # We are in a --time-passes section.
        my $start_line = $i - 1; # -1 because there's a ===----=== line before that's part of the --time-passes output

        # The end of the section is the SECOND blank line.
        for(my $j = 0; $j < 2; ++$j) {
          for(++$i; $i <= $#lines && $lines[$i] !~ m/^$/; ++$i) {
          }
        }
        my $end_line = $i;

        my @time_passes = splice (@lines, $start_line, $end_line - $start_line + 1);
        acl::Common::write_time_log(join ("", @time_passes));

        # Continue processing the rest of the lines, taking into account that
        # a chunk of the array just got removed.
        $i = $start_line;
      }
      else {
        ++$i;
      }
    }

    # Now rewrite the log file without the --time-passes output.
    open ($L, '>', $logfile);
    print $L join ("", @lines);
    close ($L);
  }
}

=head2 display_hls_error_message ($title, $out, $err, $keep_log, $logs_are_temporary, $retcode, $time_passes, $cleanup_list_ref)

This function displays the error message (if any) for the HLS flow. The message
depends on whether stdout/stderr have been redirected to $err and $out and
whether we have asked to keep them around or not. If we ask to delete them, we
redirect their content back to stderr/stdout and we push them to a cleanup list
to be deleted later. The function also calls filter_llvm_time_passes() if
needed (when --time-passes is used)

=cut

sub display_hls_error_message {
  my ($title, $out, $err, $keep_log, $out_is_temporary, $err_is_temporary, $move_err_to_out, $retcode, $time_passes, $cleanup_list_ref) = @_;
  my $loginfo = "";
  my $message = "";

  # Handle time-passes report
  if ($time_passes && $err) {
    acl::Report::filter_llvm_time_passes($err, $time_passes); #Will only execute in regtest_mode
  }

  # Handle File deletion/appendation
  if ($move_err_to_out && $err && $out && ($err ne $out)) {
    acl::Common::move_to_log('', $err, $out);
  } elsif ($err_is_temporary && $err) {
    acl::Report::append_to_err($err);
    push @$cleanup_list_ref, $err;
  }

  if ($out_is_temporary && $out && ($err ne $out)) {
    acl::Report::append_to_out($out);
    push @$cleanup_list_ref, $out;
  }

  # Handle error message
  if (!$out_is_temporary && !$err_is_temporary) {
    if ($retcode != 0) {
      if($err && $out && ($err ne $out)) {
        $keep_log = 1;
        $loginfo = "\nSee $err and $out for details.";
      } elsif ($err) {
        $keep_log = 1;
        $loginfo = "\nSee $err for details.";
      } elsif ($out) {
        $keep_log = 1;
        $loginfo = "\nSee $out for details.";
      }
      $message = "HLS $title FAILED.$loginfo\n";
      print $message;
    }
  } elsif (!$out_is_temporary && $err_is_temporary) {
    if ($retcode != 0) {
      if ($out) {
        $keep_log = 1;
        $loginfo = "\nSee $out for details.";
        $message = "HLS $title FAILED.$loginfo\n";
      } else {
        $message = "HLS $title FAILED.\n";
      }
      print $message;
    }
  }
  elsif ($out_is_temporary && !$err_is_temporary) {
    if ($retcode != 0) {
      if ($err) {
        $keep_log = 1;
        $loginfo = "\nSee $err for details.";
        $message = "HLS $title FAILED.$loginfo\n";
      } else {
        $message = "HLS $title FAILED.\n";
      }
      print $message;
    }
  } else {
    if ($retcode != 0) {
      $message = "HLS $title FAILED.\n";
      print $message;
    }
  }

  $_[3] = $keep_log; #update the actual reference
  return $message;
}

sub print_area_estimate_or_ii_message {
  my ($log_file, $prog, $prj_dir) = @_;
  
  open my $logfh, '<', $log_file;
  while (defined(my $line = <$logfh>)) {
    if ($line =~ m/"--?(Xs)?dont-error-if-large-area-est"/) {
      print "$prog: Large area estimate detected, see $prj_dir/reports/report.html for details.\n";
    } elsif ($line =~ /achieve user-specified II/) {
      print "$prog: Unable to achieve user specified II, see $prj_dir/reports/report.html for details.\n";
      last;
    }
  }
  close $logfh;
  return;
}

1;
