﻿// -----------------------------------------------------------------------------------------
// QSVEnc/NVEnc by rigaya
// -----------------------------------------------------------------------------------------
// The MIT License
//
// Copyright (c) 2011-2016 rigaya
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// --------------------------------------------------------------------------------------------

#include <set>
#include <iostream>
#include <iomanip>
#include "rgy_util.h"
#include "rgy_filesystem.h"
#include "rgy_version.h"
#if !CLFILTERS_AUF
#include "rgy_avutil.h"
#endif
#include "rgy_prm.h"
#include "rgy_err.h"
#include "rgy_perf_monitor.h"
#include "rgy_ini.h"
#if ENABLE_VPP_FILTER_AFS
#include "afs_stg.h"
#endif

RGY_VPP_RESIZE_TYPE getVppResizeType(RGY_VPP_RESIZE_ALGO resize) {
    if (resize == RGY_VPP_RESIZE_AUTO) {
        return RGY_VPP_RESIZE_TYPE_AUTO;
    } else if (resize < RGY_VPP_RESIZE_OPENCL_CUDA_MAX) {
        return RGY_VPP_RESIZE_TYPE_OPENCL;
#if ENCODER_QSV
    } else if (resize < RGY_VPP_RESIZE_MFX_MAX) {
        return RGY_VPP_RESIZE_TYPE_MFX;
#endif
#if ENCODER_NVENC && (!defined(_M_IX86) || FOR_AUO)
    } else if (resize < RGY_VPP_RESIZE_NPPI_MAX) {
        return RGY_VPP_RESIZE_TYPE_NPPI;
#endif
    } else {
        return RGY_VPP_RESIZE_TYPE_UNKNOWN;
    }
}

ColorspaceConv::ColorspaceConv() :
    from(),
    to(),
    sdr_source_peak(FILTER_DEFAULT_COLORSPACE_NOMINAL_SOURCE_PEAK),
    approx_gamma(false),
    scene_ref(false) {

}
bool ColorspaceConv::operator==(const ColorspaceConv &x) const {
    return from == x.from
        && to == x.to
        && sdr_source_peak == x.sdr_source_peak
        && approx_gamma == x.approx_gamma
        && scene_ref == x.scene_ref;
}
bool ColorspaceConv::operator!=(const ColorspaceConv &x) const {
    return !(*this == x);
}

TonemapHable::TonemapHable() :
    a(FILTER_DEFAULT_HDR2SDR_HABLE_A),
    b(FILTER_DEFAULT_HDR2SDR_HABLE_B),
    c(FILTER_DEFAULT_HDR2SDR_HABLE_C),
    d(FILTER_DEFAULT_HDR2SDR_HABLE_D),
    e(FILTER_DEFAULT_HDR2SDR_HABLE_E),
    f(FILTER_DEFAULT_HDR2SDR_HABLE_F) {}

bool TonemapHable::operator==(const TonemapHable &x) const {
    return a == x.a
        && b == x.b
        && c == x.c
        && d == x.d
        && e == x.e
        && f == x.f;
}
bool TonemapHable::operator!=(const TonemapHable &x) const {
    return !(*this == x);
}
TonemapMobius::TonemapMobius() :
    transition(FILTER_DEFAULT_HDR2SDR_MOBIUS_TRANSITION),
    peak(FILTER_DEFAULT_HDR2SDR_MOBIUS_PEAK) {
}
bool TonemapMobius::operator==(const TonemapMobius &x) const {
    return transition == x.transition
        && peak == x.peak;
}
bool TonemapMobius::operator!=(const TonemapMobius &x) const {
    return !(*this == x);
}
TonemapReinhard::TonemapReinhard() :
    contrast(FILTER_DEFAULT_HDR2SDR_REINHARD_CONTRAST),
    peak(FILTER_DEFAULT_HDR2SDR_REINHARD_PEAK) {
}
bool TonemapReinhard::operator==(const TonemapReinhard &x) const {
    return contrast == x.contrast
        &&peak == x.peak;
}
bool TonemapReinhard::operator!=(const TonemapReinhard &x) const {
    return !(*this == x);
}

HDR2SDRParams::HDR2SDRParams() :
    tonemap(HDR2SDR_DISABLED),
    hable(),
    mobius(),
    reinhard(),
    ldr_nits(FILTER_DEFAULT_COLORSPACE_LDRNITS),
    hdr_source_peak(FILTER_DEFAULT_COLORSPACE_HDR_SOURCE_PEAK),
    desat_base(FILTER_DEFAULT_HDR2SDR_DESAT_BASE),
    desat_strength(FILTER_DEFAULT_HDR2SDR_DESAT_STRENGTH),
    desat_exp(FILTER_DEFAULT_HDR2SDR_DESAT_EXP) {

}
bool HDR2SDRParams::operator==(const HDR2SDRParams &x) const {
    return tonemap == x.tonemap
        && hable == x.hable
        && mobius == x.mobius
        && reinhard == x.reinhard
        && ldr_nits == x.ldr_nits
        && hdr_source_peak == x.hdr_source_peak
        && desat_base == x.desat_base
        && desat_strength == x.desat_strength
        && desat_exp == x.desat_exp;
}
bool HDR2SDRParams::operator!=(const HDR2SDRParams &x) const {
    return !(*this == x);
}

LUT3DParams::LUT3DParams() :
    interp(FILTER_DEFAULT_LUT3D_INTERP),
    table_file() {

}
bool LUT3DParams::operator==(const LUT3DParams &x) const {
    return interp == x.interp
        && table_file == x.table_file;
}
bool LUT3DParams::operator!=(const LUT3DParams &x) const {
    return !(*this == x);
}

VppColorspace::VppColorspace() :
    enable(false),
    hdr2sdr(),
    lut3d(),
    convs() {

}

bool VppColorspace::operator==(const VppColorspace &x) const {
    if (enable != x.enable
        || x.hdr2sdr != this->hdr2sdr
        || x.lut3d != this->lut3d
        || x.convs.size() != this->convs.size()) {
        return false;
    }
    for (size_t i = 0; i < x.convs.size(); i++) {
        if (x.convs[i].from != this->convs[i].from
            || x.convs[i].to != this->convs[i].to) {
            return false;
        }
    }
    return true;
}
bool VppColorspace::operator!=(const VppColorspace &x) const {
    return !(*this == x);
}

VppDelogo::VppDelogo() :
    enable(false),
    logoFilePath(),
    logoSelect(),
    posX(0), posY(0),
    depth(FILTER_DEFAULT_DELOGO_DEPTH),
    Y(0), Cb(0), Cr(0),
    mode(DELOGO_MODE_REMOVE),
    autoFade(false),
    autoNR(false),
    NRArea(0),
    NRValue(0),
    log(false) {
}

bool VppDelogo::operator==(const VppDelogo& x) const {
    return enable == x.enable
        && logoFilePath == x.logoFilePath
        && logoSelect == x.logoSelect
        && posX == x.posX
        && posY == x.posY
        && depth == x.depth
        && Y == x.Y
        && Cb == x.Cb
        && Cr == x.Cr
        && mode == x.mode
        && autoFade == x.autoFade
        && autoNR == x.autoNR
        && NRArea == x.NRArea
        && NRValue == x.NRValue
        && log == x.log;
}
bool VppDelogo::operator!=(const VppDelogo& x) const {
    return !(*this == x);
}

tstring VppDelogo::print() const {
    tstring str = _T("");
    switch (mode) {
    case DELOGO_MODE_ADD:
        str += _T(", add");
        break;
    case DELOGO_MODE_REMOVE:
    default:
        break;
    }
    if (posX || posY) {
        str += strsprintf(_T(", pos=%d:%d"), posX, posY);
    }
    if (depth != FILTER_DEFAULT_DELOGO_DEPTH) {
        str += strsprintf(_T(", dpth=%d"), depth);
    }
    if (Y || Cb || Cr) {
        str += strsprintf(_T(", YCbCr=%d:%d:%d"), Y, Cb, Cr);
    }
    if (autoFade) {
        str += _T(", auto_fade");
    }
    if (autoNR) {
        str += _T(", auto_nr");
    }
    if ((autoFade || autoNR) && log) {
        str += _T(", log");
    }
    if (NRValue) {
        str += strsprintf(_T(", nr_value=%d"), NRValue);
    }
    if (NRArea) {
        str += strsprintf(_T(", nr_area=%d"), NRArea);
    }
    return str;
}

VppAfs::VppAfs() :
    enable(false),
    tb_order(FILTER_DEFAULT_AFS_TB_ORDER),
    clip(scan_clip(FILTER_DEFAULT_AFS_CLIP_TB, FILTER_DEFAULT_AFS_CLIP_TB, FILTER_DEFAULT_AFS_CLIP_LR, FILTER_DEFAULT_AFS_CLIP_LR)),
    method_switch(FILTER_DEFAULT_AFS_METHOD_SWITCH),
    coeff_shift(FILTER_DEFAULT_AFS_COEFF_SHIFT),
    thre_shift(FILTER_DEFAULT_AFS_THRE_SHIFT),
    thre_deint(FILTER_DEFAULT_AFS_THRE_DEINT),
    thre_Ymotion(FILTER_DEFAULT_AFS_THRE_YMOTION),
    thre_Cmotion(FILTER_DEFAULT_AFS_THRE_CMOTION),
    analyze(FILTER_DEFAULT_AFS_ANALYZE),
    shift(FILTER_DEFAULT_AFS_SHIFT),
    drop(FILTER_DEFAULT_AFS_DROP),
    smooth(FILTER_DEFAULT_AFS_SMOOTH),
    force24(FILTER_DEFAULT_AFS_FORCE24),
    tune(FILTER_DEFAULT_AFS_TUNE),
    rff(FILTER_DEFAULT_AFS_RFF),
    timecode(FILTER_DEFAULT_AFS_TIMECODE),
    log(FILTER_DEFAULT_AFS_LOG) {
    check();
}

bool VppAfs::operator==(const VppAfs &x) const {
    return enable == x.enable
        && tb_order == x.tb_order
        && clip.bottom == x.clip.bottom
        && clip.left == x.clip.left
        && clip.top == x.clip.top
        && clip.right == x.clip.right
        && method_switch == x.method_switch
        && coeff_shift == x.coeff_shift
        && thre_shift == x.thre_shift
        && thre_deint == x.thre_deint
        && thre_Ymotion == x.thre_Ymotion
        && thre_Cmotion == x.thre_Cmotion
        && analyze == x.analyze
        && shift == x.shift
        && drop == x.drop
        && smooth == x.smooth
        && force24 == x.force24
        && tune == x.tune
        && rff == x.rff
        && timecode == x.timecode
        && log == x.log;
}
bool VppAfs::operator!=(const VppAfs &x) const {
    return !(*this == x);
}

void VppAfs::check() {
    if (!shift) {
        method_switch = 0;
        coeff_shift = 0;
    }
    drop &= shift;
    smooth &= drop;
}

void VppAfs::set_preset(int preset) {
    switch (preset) {
    case AFS_PRESET_DEFAULT: //デフォルト
        method_switch = FILTER_DEFAULT_AFS_METHOD_SWITCH;
        coeff_shift   = FILTER_DEFAULT_AFS_COEFF_SHIFT;
        thre_shift    = FILTER_DEFAULT_AFS_THRE_SHIFT;
        thre_deint    = FILTER_DEFAULT_AFS_THRE_DEINT;
        thre_Ymotion  = FILTER_DEFAULT_AFS_THRE_YMOTION;
        thre_Cmotion  = FILTER_DEFAULT_AFS_THRE_CMOTION;
        analyze       = FILTER_DEFAULT_AFS_ANALYZE;
        shift         = FILTER_DEFAULT_AFS_SHIFT;
        drop          = FILTER_DEFAULT_AFS_DROP;
        smooth        = FILTER_DEFAULT_AFS_SMOOTH;
        force24       = FILTER_DEFAULT_AFS_FORCE24;
        tune          = FILTER_DEFAULT_AFS_TUNE;
        break;
    case AFS_PRESET_TRIPLE: //動き重視
        method_switch = 0;
        coeff_shift   = 192;
        thre_shift    = 128;
        thre_deint    = 48;
        thre_Ymotion  = 112;
        thre_Cmotion  = 224;
        analyze       = 1;
        shift         = false;
        drop          = false;
        smooth        = false;
        force24       = false;
        tune          = false;
        break;
    case AFS_PRESET_DOUBLE://二重化
        method_switch = 0;
        coeff_shift   = 192;
        thre_shift    = 128;
        thre_deint    = 48;
        thre_Ymotion  = 112;
        thre_Cmotion  = 224;
        analyze       = 2;
        shift         = true;
        drop          = true;
        smooth        = true;
        force24       = false;
        tune          = false;
        break;
    case AFS_PRESET_ANIME: //映画/アニメ
        method_switch = 64;
        coeff_shift   = 128;
        thre_shift    = 128;
        thre_deint    = 48;
        thre_Ymotion  = 112;
        thre_Cmotion  = 224;
        analyze       = 3;
        shift         = true;
        drop          = true;
        smooth        = true;
        force24       = false;
        tune          = false;
        break;
    case AFS_PRESET_MIN_AFTERIMG:      //残像最小化
        method_switch = 0;
        coeff_shift   = 192;
        thre_shift    = 128;
        thre_deint    = 48;
        thre_Ymotion  = 112;
        thre_Cmotion  = 224;
        analyze       = 4;
        shift         = true;
        drop          = true;
        smooth        = true;
        force24       = false;
        tune          = false;
        break;
    case AFS_PRESET_FORCE24_SD:        //24fps固定
        method_switch = 64;
        coeff_shift   = 128;
        thre_shift    = 128;
        thre_deint    = 48;
        thre_Ymotion  = 112;
        thre_Cmotion  = 224;
        analyze       = 3;
        shift         = true;
        drop          = true;
        smooth        = false;
        force24       = true;
        tune          = false;
        break;
    case AFS_PRESET_FORCE24_HD:        //24fps固定 (HD)
        method_switch = 92;
        coeff_shift   = 192;
        thre_shift    = 448;
        thre_deint    = 48;
        thre_Ymotion  = 112;
        thre_Cmotion  = 224;
        analyze       = 3;
        shift         = true;
        drop          = true;
        smooth        = true;
        force24       = true;
        tune          = false;
        break;
    case AFS_PRESET_FORCE30:           //30fps固定
        method_switch = 92;
        coeff_shift   = 192;
        thre_shift    = 448;
        thre_deint    = 48;
        thre_Ymotion  = 112;
        thre_Cmotion  = 224;
        analyze       = 3;
        shift         = false;
        drop          = false;
        smooth        = false;
        force24       = false;
        tune          = false;
        break;
    default:
        break;
    }
}

int VppAfs::read_afs_inifile(const TCHAR *inifile) {
    if (!rgy_file_exists(inifile)) {
        return 1;
    }
#if ENABLE_VPP_FILTER_AFS
    const auto filename = tchar_to_string(inifile);
    const auto section = AFS_STG_SECTION;

    clip.top      = GetPrivateProfileIntA(section, AFS_STG_UP, clip.top, filename.c_str());
    clip.bottom   = GetPrivateProfileIntA(section, AFS_STG_BOTTOM, clip.bottom, filename.c_str());
    clip.left     = GetPrivateProfileIntA(section, AFS_STG_LEFT, clip.left, filename.c_str());
    clip.right    = GetPrivateProfileIntA(section, AFS_STG_RIGHT, clip.right, filename.c_str());
    method_switch = GetPrivateProfileIntA(section, AFS_STG_METHOD_WATERSHED, method_switch, filename.c_str());
    coeff_shift   = GetPrivateProfileIntA(section, AFS_STG_COEFF_SHIFT, coeff_shift, filename.c_str());
    thre_shift    = GetPrivateProfileIntA(section, AFS_STG_THRE_SHIFT, thre_shift, filename.c_str());
    thre_deint    = GetPrivateProfileIntA(section, AFS_STG_THRE_DEINT, thre_deint, filename.c_str());
    thre_Ymotion  = GetPrivateProfileIntA(section, AFS_STG_THRE_Y_MOTION, thre_Ymotion, filename.c_str());
    thre_Cmotion  = GetPrivateProfileIntA(section, AFS_STG_THRE_C_MOTION, thre_Cmotion, filename.c_str());
    analyze       = GetPrivateProfileIntA(section, AFS_STG_MODE, analyze, filename.c_str());

    shift    = 0 != GetPrivateProfileIntA(section, AFS_STG_FIELD_SHIFT, shift, filename.c_str());
    drop     = 0 != GetPrivateProfileIntA(section, AFS_STG_DROP, drop, filename.c_str());
    smooth   = 0 != GetPrivateProfileIntA(section, AFS_STG_SMOOTH, smooth, filename.c_str());
    force24  = 0 != GetPrivateProfileIntA(section, AFS_STG_FORCE24, force24, filename.c_str());
    rff      = 0 != GetPrivateProfileIntA(section, AFS_STG_RFF, rff, filename.c_str());
    log      = 0 != GetPrivateProfileIntA(section, AFS_STG_LOG, log, filename.c_str());
    // GetPrivateProfileIntA(section, AFS_STG_DETECT_SC, fp->check[4], filename.c_str());
    tune     = 0 != GetPrivateProfileIntA(section, AFS_STG_TUNE_MODE, tune, filename.c_str());
    // GetPrivateProfileIntA(section, AFS_STG_LOG_SAVE, fp->check[6], filename.c_str());
    // GetPrivateProfileIntA(section, AFS_STG_TRACE_MODE, fp->check[7], filename.c_str());
    // GetPrivateProfileIntA(section, AFS_STG_REPLAY_MODE, fp->check[8], filename.c_str());
    // GetPrivateProfileIntA(section, AFS_STG_YUY2UPSAMPLE, fp->check[9], filename.c_str());
    // GetPrivateProfileIntA(section, AFS_STG_THROUGH_MODE, fp->check[10], filename.c_str());

    // GetPrivateProfileIntA(section, AFS_STG_PROC_MODE, g_afs.ex_data.proc_mode, filename.c_str());
    return 0;
#else
    return 1;
#endif
}

tstring VppAfs::print() const {
#define ON_OFF(b) ((b) ? _T("on") : _T("off"))
    return strsprintf(
        _T("afs: clip(T %d, B %d, L %d, R %d), switch %d, coeff_shift %d\n")
        _T("                    thre(shift %d, deint %d, Ymotion %d, Cmotion %d)\n")
        _T("                    level %d, shift %s, drop %s, smooth %s, force24 %s\n")
        _T("                    tune %s, tb_order %d(%s), rff %s, timecode %s, log %s"),
        clip.top, clip.bottom, clip.left, clip.right,
        method_switch, coeff_shift,
        thre_shift, thre_deint, thre_Ymotion, thre_Cmotion,
        analyze, ON_OFF(shift), ON_OFF(drop), ON_OFF(smooth), ON_OFF(force24),
        ON_OFF(tune), tb_order, tb_order ? _T("tff") : _T("bff"), ON_OFF(rff), ON_OFF(timecode), ON_OFF(log));
#undef ON_OFF
}

VppNnedi::VppNnedi() :
    enable(false),
    field(VPP_NNEDI_FIELD_USE_AUTO),
    nns(32),
    nsize(VPP_NNEDI_NSIZE_32x4),
    quality(VPP_NNEDI_QUALITY_FAST),
    precision(VPP_FP_PRECISION_AUTO),
    pre_screen(VPP_NNEDI_PRE_SCREEN_NEW_BLOCK),
    errortype(VPP_NNEDI_ETYPE_ABS),
    weightfile(_T("")) {

}

bool VppNnedi::isbob() {
    return field == VPP_NNEDI_FIELD_BOB_AUTO
        || field == VPP_NNEDI_FIELD_BOB_BOTTOM_TOP
        || field == VPP_NNEDI_FIELD_BOB_TOP_BOTTOM;
}

bool VppNnedi::operator==(const VppNnedi &x) const {
    return enable == x.enable
        && field == x.field
        && nns == x.nns
        && nsize == x.nsize
        && quality == x.quality
        && pre_screen == x.pre_screen
        && errortype == x.errortype
        && precision == x.precision
        && weightfile == x.weightfile;
}
bool VppNnedi::operator!=(const VppNnedi &x) const {
    return !(*this == x);
}

tstring VppNnedi::print() const {
    return strsprintf(
        _T("nnedi: field %s, nns %d, nsize %s, quality %s, prec %s\n")
        _T("                       pre_screen %s, errortype %s, weight \"%s\""),
        get_cx_desc(list_vpp_nnedi_field, field),
        nns,
        get_cx_desc(list_vpp_nnedi_nsize, nsize),
        get_cx_desc(list_vpp_nnedi_quality, quality),
        get_cx_desc(list_vpp_fp_prec, precision),
        get_cx_desc(list_vpp_nnedi_pre_screen, pre_screen),
        get_cx_desc(list_vpp_nnedi_error_type, errortype),
        ((weightfile.length()) ? weightfile.c_str() : _T("internal")));
}

VppYadif::VppYadif() :
    enable(false),
    mode(VPP_YADIF_MODE_AUTO) {

}

bool VppYadif::operator==(const VppYadif& x) const {
    return enable == x.enable
        && mode == x.mode;
}
bool VppYadif::operator!=(const VppYadif& x) const {
    return !(*this == x);
}

tstring VppYadif::print() const {
    return strsprintf(
        _T("yadif: mode %s"),
        get_cx_desc(list_vpp_yadif_mode, mode));
}

VppSelectEvery::VppSelectEvery() :
    enable(false),
    step(1),
    offset(0) {
}

bool VppSelectEvery::operator==(const VppSelectEvery& x) const {
    return enable == x.enable
        && step == x.step
        && offset == x.offset;
}
bool VppSelectEvery::operator!=(const VppSelectEvery& x) const {
    return !(*this == x);
}

tstring VppSelectEvery::print() const {
    return strsprintf(_T("selectevery %d (offset %d)"), step, offset);
}

VppDecimate::VppDecimate() :
    enable(false),
    cycle(FILTER_DEFAULT_DECIMATE_CYCLE),
    threDuplicate(FILTER_DEFAULT_DECIMATE_THRE_DUP),
    threSceneChange(FILTER_DEFAULT_DECIMATE_THRE_SC),
    blockX(FILTER_DEFAULT_DECIMATE_BLOCK_X),
    blockY(FILTER_DEFAULT_DECIMATE_BLOCK_Y),
    preProcessed(FILTER_DEFAULT_DECIMATE_PREPROCESSED),
    chroma(FILTER_DEFAULT_DECIMATE_CHROMA),
    log(FILTER_DEFAULT_DECIMATE_LOG) {

}

bool VppDecimate::operator==(const VppDecimate& x) const {
    return enable == x.enable
        && cycle == x.cycle
        && threDuplicate == x.threDuplicate
        && threSceneChange == x.threSceneChange
        && blockX == x.blockX
        && blockY == x.blockY
        && preProcessed == x.preProcessed
        && chroma == x.chroma
        && log == x.log;
}
bool VppDecimate::operator!=(const VppDecimate& x) const {
    return !(*this == x);
}

tstring VppDecimate::print() const {
    return strsprintf(_T("decimate: cycle %d, threDup %.2f, threSC %.2f\n")
        _T("                         block %dx%d, chroma %s, log %s"),
        cycle,
        threDuplicate, threSceneChange,
        blockX, blockY,
        /*preProcessed ? _T("on") : _T("off"),*/
        chroma ? _T("on") : _T("off"),
        log ? _T("on") : _T("off"));
}


VppMpdecimate::VppMpdecimate() :
    enable(false),
    lo(FILTER_DEFAULT_MPDECIMATE_LO),
    hi(FILTER_DEFAULT_MPDECIMATE_HI),
    max(FILTER_DEFAULT_MPDECIMATE_MAX),
    frac(FILTER_DEFAULT_MPDECIMATE_FRAC),
    log(FILTER_DEFAULT_MPDECIMATE_LOG) {

}

bool VppMpdecimate::operator==(const VppMpdecimate& x) const {
    return enable == x.enable
        && lo == x.lo
        && hi == x.hi
        && max == x.max
        && frac == x.frac
        && log == x.log;
}
bool VppMpdecimate::operator!=(const VppMpdecimate& x) const {
    return !(*this == x);
}

tstring VppMpdecimate::print() const {
    return strsprintf(_T("mpdecimate: hi %d, lo %d, frac %.2f, max %d, log %s"),
        hi, lo, frac, max,
        log ? _T("on") : _T("off"));
}

VppPad::VppPad() :
    enable(false),
    left(0),
    top(0),
    right(0),
    bottom(0) {

}

bool VppPad::operator==(const VppPad& x) const {
    return enable == x.enable
        && left == x.left
        && top == x.top
        && right == x.right
        && bottom == x.bottom;
}
bool VppPad::operator!=(const VppPad& x) const {
    return !(*this == x);
}

tstring VppPad::print() const {
    return strsprintf(_T("(right=%d, left=%d, top=%d, bottom=%d)"),
        right, left, top, bottom);
}

VppKnn::VppKnn() :
    enable(false),
    radius(FILTER_DEFAULT_KNN_RADIUS),
    strength(FILTER_DEFAULT_KNN_STRENGTH),
    lerpC(FILTER_DEFAULT_KNN_LERPC),
    weight_threshold(FILTER_DEFAULT_KNN_WEIGHT_THRESHOLD),
    lerp_threshold(FILTER_DEFAULT_KNN_LERPC_THRESHOLD) {
}

bool VppKnn::operator==(const VppKnn &x) const {
    return enable == x.enable
        && radius == x.radius
        && strength == x.strength
        && lerpC == x.lerpC
        && weight_threshold == x.weight_threshold
        && lerp_threshold == x.lerp_threshold;
}
bool VppKnn::operator!=(const VppKnn &x) const {
    return !(*this == x);
}

tstring VppKnn::print() const {
    return strsprintf(
        _T("denoise(knn): radius %d, strength %.2f, lerp %.2f\n")
        _T("                              th_weight %.2f, th_lerp %.2f"),
        radius, strength, lerpC,
        weight_threshold, lerp_threshold);
}

VppPmd::VppPmd() :
    enable(false),
    strength(FILTER_DEFAULT_PMD_STRENGTH),
    threshold(FILTER_DEFAULT_PMD_THRESHOLD),
    applyCount(FILTER_DEFAULT_PMD_APPLY_COUNT),
    useExp(FILTER_DEFAULT_PMD_USE_EXP) {

}

bool VppPmd::operator==(const VppPmd& x) const {
    return enable == x.enable
        && strength == x.strength
        && threshold == x.threshold
        && applyCount == x.applyCount
        && useExp == x.useExp;
}
bool VppPmd::operator!=(const VppPmd& x) const {
    return !(*this == x);
}

tstring VppPmd::print() const {
    return strsprintf(_T("denoise(pmd): strength %d, threshold %d, apply %d, exp %d"),
        (int)strength, (int)threshold, applyCount, useExp);
}

VppSmooth::VppSmooth() :
    enable(false),
    quality(FILTER_DEFAULT_SMOOTH_QUALITY),
    qp(FILTER_DEFAULT_SMOOTH_QP),
    prec(VPP_FP_PRECISION_AUTO),
    useQPTable(false),
    strength(FILTER_DEFAULT_SMOOTH_STRENGTH),
    threshold(FILTER_DEFAULT_SMOOTH_THRESHOLD),
    bratio(FILTER_DEFAULT_SMOOTH_B_RATIO),
    maxQPTableErrCount(FILTER_DEFAULT_SMOOTH_MAX_QPTABLE_ERR) {

}

bool VppSmooth::operator==(const VppSmooth &x) const {
    return enable == x.enable
        && quality == x.quality
        && qp == x.qp
        && prec == x.prec
        && useQPTable == x.useQPTable
        && strength == x.strength
        && threshold == x.threshold
        && bratio == x.bratio
        && maxQPTableErrCount == x.maxQPTableErrCount;
}
bool VppSmooth::operator!=(const VppSmooth &x) const {
    return !(*this == x);
}

tstring VppSmooth::print() const {
    //return strsprintf(_T("smooth: quality %d, qp %d, threshold %.1f, strength %.1f, mode %d, use_bframe_qp %s"), quality, qp, threshold, strength, mode, use_bframe_qp ? _T("yes") : _T("no"));
    tstring str = strsprintf(_T("smooth: quality %d, qp %d, prec %s"), quality, qp, get_cx_desc(list_vpp_fp_prec, prec));
    if (useQPTable) {
        str += strsprintf(_T(", use QP table on"));
    }
    return str;
}

VppConvolution3d::VppConvolution3d() :
    enable(false),
    fast(false),
    matrix(VppConvolution3dMatrix::Standard),
    threshYspatial(FILTER_DEFAULT_CONVOLUTION3D_THRESH_Y_SPATIAL),
    threshCspatial(FILTER_DEFAULT_CONVOLUTION3D_THRESH_C_SPATIAL),
    threshYtemporal(FILTER_DEFAULT_CONVOLUTION3D_THRESH_Y_TEMPORAL),
    threshCtemporal(FILTER_DEFAULT_CONVOLUTION3D_THRESH_C_TEMPORAL) {

}

bool VppConvolution3d::operator==(const VppConvolution3d &x) const {
    return enable == x.enable
        && fast == x.fast
        && matrix == x.matrix
        && threshYspatial == x.threshYspatial
        && threshCspatial == x.threshCspatial
        && threshYtemporal == x.threshYtemporal
        && threshCtemporal == x.threshCtemporal;
}
bool VppConvolution3d::operator!=(const VppConvolution3d &x) const {
    return !(*this == x);
}

tstring VppConvolution3d::print() const {
    tstring str = strsprintf(_T("convolution3d: matrix %s, mode %s\n")
        _T("                       threshold spatial luma %d, chroma %d, temporal luma %d, chroma %d"),
        get_cx_desc(list_vpp_convolution3d_matrix, (int)matrix),
        fast ? _T("fast") : _T("normal"),
        threshYspatial, threshCspatial, threshYtemporal, threshCtemporal);
    return str;
}

VppSubburn::VppSubburn() :
    enable(false),
    filename(),
    charcode(),
    fontsdir(),
    trackId(0),
    assShaping(1),
    scale(0.0),
    transparency_offset(0.0),
    brightness(FILTER_DEFAULT_TWEAK_BRIGHTNESS),
    contrast(FILTER_DEFAULT_TWEAK_CONTRAST),
    ts_offset(0.0),
    vid_ts_offset(true) {
}

bool VppSubburn::operator==(const VppSubburn &x) const {
    return enable == x.enable
        && filename == x.filename
        && charcode == x.charcode
        && fontsdir == x.fontsdir
        && trackId == x.trackId
        && assShaping == x.assShaping
        && scale == x.scale
        && transparency_offset == x.transparency_offset
        && brightness == x.brightness
        && contrast == x.contrast
        && ts_offset == x.ts_offset
        && vid_ts_offset == x.vid_ts_offset;
}
bool VppSubburn::operator!=(const VppSubburn &x) const {
    return !(*this == x);
}

tstring VppSubburn::print() const {
    tstring str = strsprintf(_T("subburn: %s, scale x%.2f"),
        (filename.length() > 0)
        ? filename.c_str()
        : strsprintf(_T("track #%d"), trackId).c_str(),
        scale);
    if (transparency_offset != 0.0) {
        str += strsprintf(_T(", transparency %.2f"), transparency_offset);
    }
    if (brightness != FILTER_DEFAULT_TWEAK_BRIGHTNESS) {
        str += strsprintf(_T(", brightness %.2f"), brightness);
    }
    if (contrast != FILTER_DEFAULT_TWEAK_CONTRAST) {
        str += strsprintf(_T(", contrast %.2f"), contrast);
    }
    if (ts_offset != 0.0) {
        str += strsprintf(_T(", ts_offset %.2f"), ts_offset);
    }
    if (!vid_ts_offset) {
        str += _T(", vid_ts_offset off");
    }
    return str;
}

VppUnsharp::VppUnsharp() :
    enable(false),
    radius(FILTER_DEFAULT_UNSHARP_RADIUS),
    weight(FILTER_DEFAULT_UNSHARP_WEIGHT),
    threshold(FILTER_DEFAULT_UNSHARP_THRESHOLD) {

}

bool VppUnsharp::operator==(const VppUnsharp &x) const {
    return enable == x.enable
        && radius == x.radius
        && weight == x.weight
        && threshold == x.threshold;
}
bool VppUnsharp::operator!=(const VppUnsharp &x) const {
    return !(*this == x);
}

tstring VppUnsharp::print() const {
    return strsprintf(_T("unsharp: radius %d, weight %.1f, threshold %.1f"),
        radius, weight, threshold);
}

VppEdgelevel::VppEdgelevel() :
    enable(false),
    strength(FILTER_DEFAULT_EDGELEVEL_STRENGTH),
    threshold(FILTER_DEFAULT_EDGELEVEL_THRESHOLD),
    black(FILTER_DEFAULT_EDGELEVEL_BLACK),
    white(FILTER_DEFAULT_EDGELEVEL_WHITE) {
}

bool VppEdgelevel::operator==(const VppEdgelevel &x) const {
    return enable == x.enable
        && strength == x.strength
        && threshold == x.threshold
        && black == x.black
        && white == x.white;
}
bool VppEdgelevel::operator!=(const VppEdgelevel &x) const {
    return !(*this == x);
}

tstring VppEdgelevel::print() const {
    return strsprintf(_T("edgelevel: strength %.1f, threshold %.1f, black %.1f, white %.1f"),
        strength, threshold, black, white);
}

VppWarpsharp::VppWarpsharp() :
    enable(false),
    threshold(FILTER_DEFAULT_WARPSHARP_THRESHOLD),
    blur(FILTER_DEFAULT_WARPSHARP_BLUR),
    type(FILTER_DEFAULT_WARPSHARP_TYPE),
    depth(FILTER_DEFAULT_WARPSHARP_DEPTH),
    chroma(FILTER_DEFAULT_WARPSHARP_CHROMA) {
}

bool VppWarpsharp::operator==(const VppWarpsharp& x) const {
    return enable == x.enable
        && threshold == x.threshold
        && blur == x.blur
        && type == x.type
        && depth == x.depth
        && chroma == x.chroma;
}
bool VppWarpsharp::operator!=(const VppWarpsharp& x) const {
    return !(*this == x);
}

tstring VppWarpsharp::print() const {
    return strsprintf(_T("warpsharp: threshold %.1f, blur %d, type %d, depth %.1f, chroma %d"),
        threshold, blur, type, depth, chroma);
}

VppTweak::VppTweak() :
    enable(false),
    brightness(FILTER_DEFAULT_TWEAK_BRIGHTNESS),
    contrast(FILTER_DEFAULT_TWEAK_CONTRAST),
    gamma(FILTER_DEFAULT_TWEAK_GAMMA),
    saturation(FILTER_DEFAULT_TWEAK_SATURATION),
    hue(FILTER_DEFAULT_TWEAK_HUE),
    swapuv(false) {
}

bool VppTweak::operator==(const VppTweak &x) const {
    return enable == x.enable
        && brightness == x.brightness
        && contrast == x.contrast
        && gamma == x.gamma
        && saturation == x.saturation
        && hue == x.hue
        && swapuv == x.swapuv;
}
bool VppTweak::operator!=(const VppTweak &x) const {
    return !(*this == x);
}

tstring VppTweak::print() const {
    return strsprintf(_T("tweak: brightness %.2f, contrast %.2f, saturation %.2f, gamma %.2f, hue %.2f, swapuv %s"),
        brightness, contrast, saturation, gamma, hue, swapuv ? _T("on") : _T("off"));
}

VppTransform::VppTransform() :
    enable(false),
    transpose(false),
    flipX(false),
    flipY(false) {
}

int VppTransform::rotate() const {
    if (transpose) {
        if (!flipY && flipX) {
            return 270;
        } else if (flipY && !flipX) {
            return 90;
        }
    } else if (flipY && flipX) {
        return 180;
    }
    return 0;
}

bool VppTransform::setRotate(int rotate) {
    switch (rotate) {
    case 90:
        transpose = true;
        flipY = true;
        break;
    case 180:
        flipX = true;
        flipY = true;
        break;
    case 270:
        transpose = true;
        flipX = true;
        break;
    default:
        return false;
    }
    return true;
}

bool VppTransform::operator==(const VppTransform &x) const {
    return enable == x.enable
        && transpose == x.transpose
        && flipX == x.flipX
        && flipY == x.flipY;
}
bool VppTransform::operator!=(const VppTransform &x) const {
    return !(*this == x);
}

tstring VppTransform::print() const {
#define ON_OFF(b) ((b) ? _T("on") : _T("off"))
    const auto rotation = rotate();
    if (rotation) {
        return strsprintf(_T("rotate: %d"), rotation);
    } else {
        return strsprintf(_T("transform: transpose %s, flipX %s, flipY %s"),
            ON_OFF(transpose), ON_OFF(flipX), ON_OFF(flipY));
    }
#undef ON_OFF
}

VppDeband::VppDeband() :
    enable(false),
    range(FILTER_DEFAULT_DEBAND_RANGE),
    threY(FILTER_DEFAULT_DEBAND_THRE_Y),
    threCb(FILTER_DEFAULT_DEBAND_THRE_CB),
    threCr(FILTER_DEFAULT_DEBAND_THRE_CR),
    ditherY(FILTER_DEFAULT_DEBAND_DITHER_Y),
    ditherC(FILTER_DEFAULT_DEBAND_DITHER_C),
    sample(FILTER_DEFAULT_DEBAND_MODE),
    seed(FILTER_DEFAULT_DEBAND_SEED),
    blurFirst(FILTER_DEFAULT_DEBAND_BLUR_FIRST),
    randEachFrame(FILTER_DEFAULT_DEBAND_RAND_EACH_FRAME) {

}

bool VppDeband::operator==(const VppDeband &x) const {
    return enable == x.enable
        && range == x.range
        && threY == x.threY
        && threCb == x.threCb
        && threCr == x.threCr
        && ditherY == x.ditherY
        && ditherC == x.ditherC
        && sample == x.sample
        && seed == x.seed
        && blurFirst == x.blurFirst
        && randEachFrame == x.randEachFrame;
}
bool VppDeband::operator!=(const VppDeband &x) const {
    return !(*this == x);
}

tstring VppDeband::print() const {
    return strsprintf(_T("deband: mode %d, range %d, threY %d, threCb %d, threCr %d\n")
        _T("                       ditherY %d, ditherC %d, blurFirst %s, randEachFrame %s"),
        sample, range,
        threY, threCb, threCr,
        ditherY, ditherC,
        blurFirst ? _T("yes") : _T("no"),
        randEachFrame ? _T("yes") : _T("no"));
}

RGYParamVpp::RGYParamVpp() :
    resize_algo(RGY_VPP_RESIZE_AUTO),
    resize_mode(RGY_VPP_RESIZE_MODE_DEFAULT),
    colorspace(),
    delogo(),
    afs(),
    nnedi(),
    yadif(),
    rff(false),
    selectevery(),
    decimate(),
    mpdecimate(),
    pad(),
    convolution3d(),
    knn(),
    pmd(),
    smooth(),
    subburn(),
    unsharp(),
    edgelevel(),
    warpsharp(),
    tweak(),
    transform(),
    deband(),
    checkPerformance(false) {

}


AudioSelect::AudioSelect() :
    trackID(0),
    decCodecPrm(),
    encCodec(),
    encCodecPrm(),
    encCodecProfile(),
    encBitrate(0),
    encSamplingRate(0),
    addDelayMs(0),
    extractFilename(),
    extractFormat(),
    filter(),
    streamChannelSelect(),
    streamChannelOut(),
    bsf(),
    disposition(),
    lang(),
    selectCodec(),
    metadata() {
    memset(streamChannelSelect, 0, sizeof(streamChannelSelect));
    memset(streamChannelOut, 0, sizeof(streamChannelOut));
}

AudioSource::AudioSource() :
    filename(),
    select() {

}

SubtitleSelect::SubtitleSelect() :
    trackID(0),
    encCodec(),
    encCodecPrm(),
    decCodecPrm(),
    asdata(false),
    bsf(),
    disposition(),
    lang(),
    selectCodec(),
    metadata() {

}

SubSource::SubSource() :
    filename(),
    select() {

}

DataSelect::DataSelect() :
    trackID(0),
    disposition(),
    lang(),
    selectCodec(),
    metadata() {

}

VMAFParam::VMAFParam() :
    enable(false),
    model(VMAF_DEFAULT_MODEL_VERSION),
    threads(0),
    subsample(1),
    phone_model(false),
    enable_transform(false) {
};

bool VMAFParam::operator==(const VMAFParam &x) const {
    return enable == x.enable
        && model == x.model
        && threads == x.threads
        && subsample == x.subsample
        && phone_model == x.phone_model
        && enable_transform == x.enable_transform;
}
bool VMAFParam::operator!=(const VMAFParam &x) const {
    return !(*this == x);
}
tstring VMAFParam::print() const {
    auto str = strsprintf(_T("vmaf %s, threads %d, subsample %d"),
        model.c_str(), threads, subsample);
    if (phone_model) {
        str += _T(", phone_model");
    }
    if (enable_transform) {
        str += _T(", transform");
    }
    return str;
}

RGYVideoQualityMetric::RGYVideoQualityMetric() :
    ssim(false),
    psnr(false),
    vmaf() {

}
bool RGYVideoQualityMetric::enabled() const {
    return ssim || psnr || vmaf.enable;
}
tstring RGYVideoQualityMetric::enabled_metric() const {
    if (!enabled()) return _T("none");
    tstring str;
    if (ssim) str += _T(",ssim");
    if (psnr) str += _T(",psnr");
    if (vmaf.enable) str += _T(",vmaf");
    return (str.length() > 0) ? str.substr(1) : _T("unknown");
}

GPUAutoSelectMul::GPUAutoSelectMul() : cores(0.001f), gen(1.0f), gpu(1.0f), ve(1.0f) {}

bool GPUAutoSelectMul::operator==(const GPUAutoSelectMul &x) const {
    return cores == x.cores
        && gen == x.gen
        && gpu == x.gpu
        && ve == x.ve;
}
bool GPUAutoSelectMul::operator!=(const GPUAutoSelectMul &x) const {
    return !(*this == x);
}

RGYParamInput::RGYParamInput() :
    resizeResMode(RGYResizeResMode::Normal) {

}

RGYParamInput::~RGYParamInput() {};

RGYParamCommon::RGYParamCommon() :
    inputFilename(),
    outputFilename(),
    muxOutputFormat(),
    out_vui(),
    inputOpt(),
    maxCll(),
    masterDisplay(),
    atcSei(RGY_TRANSFER_UNKNOWN),
    hdr10plusMetadataCopy(false),
    dynamicHdr10plusJson(),
    doviRpuFile(),
    doviProfile(0),
    videoCodecTag(),
    videoMetadata(),
    formatMetadata(),
    seekSec(0.0f),               //指定された秒数分先頭を飛ばす
    nSubtitleSelectCount(0),
    ppSubtitleSelectList(nullptr),
    subSource(),
    audioSource(),
    nAudioSelectCount(0), //pAudioSelectの数
    ppAudioSelectList(nullptr),
    nDataSelectCount(0),
    ppDataSelectList(nullptr),
    nAttachmentSelectCount(0),
    ppAttachmentSelectList(nullptr),
    audioResampler(RGY_RESAMPLER_SWR),
    inputRetry(0),
    demuxAnalyzeSec(-1),
    demuxProbesize(-1),
    AVMuxTarget(RGY_MUX_NONE),                       //RGY_MUX_xxx
    videoTrack(0),
    videoStreamId(0),
    nTrimCount(0),
    pTrimList(nullptr),
    copyChapter(false),
    keyOnChapter(false),
    chapterNoTrim(false),
    caption2ass(FORMAT_INVALID),
    audioIgnoreDecodeError(DEFAULT_IGNORE_DECODE_ERROR),
    muxOpt(),
    disableMp4Opt(false),
    chapterFile(),
    AVInputFormat(nullptr),
    AVSyncMode(RGY_AVSYNC_ASSUME_CFR),     //avsyncの方法 (RGY_AVSYNC_xxx)
    timecode(false),
    timecodeFile(),
    metric() {

}

RGYParamCommon::~RGYParamCommon() {};

RGYParamControl::RGYParamControl() :
    threadCsp(0),
    simdCsp(RGY_SIMD::SIMD_ALL),
    logfile(),              //ログ出力先
    loglevel(RGY_LOG_INFO),                 //ログ出力レベル
    logAddTime(false),
    logFramePosList(false),     //framePosList出力
    logPacketsList(false),
    logMuxVidTsFile(nullptr),
    threadOutput(RGY_OUTPUT_THREAD_AUTO),
    threadAudio(RGY_AUDIO_THREAD_AUTO),
    threadInput(RGY_INPUT_THREAD_AUTO),
    threadParams(),
    procSpeedLimit(0),      //処理速度制限 (0で制限なし)
    perfMonitorSelect(0),
    perfMonitorSelectMatplot(0),
    perfMonitorInterval(RGY_DEFAULT_PERF_MONITOR_INTERVAL),
    parentProcessID(0),
    lowLatency(false),
    gpuSelect(),
    skipHWDecodeCheck(false),
    avsdll(),
    enableOpenCL(true),
    outputBufSizeMB(RGY_OUTPUT_BUF_MB_DEFAULT) {

}
RGYParamControl::~RGYParamControl() {};


bool trim_active(const sTrimParam *pTrim) {
    if (pTrim == nullptr) {
        return false;
    }
    if (pTrim->list.size() == 0) {
        return false;
    }
    if (pTrim->list[0].start == 0 && pTrim->list[0].fin == TRIM_MAX) {
        return false;
    }
    return true;
}

//block index (空白がtrimで削除された領域)
//       #0       #0         #1         #1       #2    #2
//   |        |----------|         |----------|     |------
std::pair<bool, int> frame_inside_range(int frame, const std::vector<sTrim> &trimList) {
    int index = 0;
    if (trimList.size() == 0) {
        return std::make_pair(true, index);
    }
    if (frame < 0) {
        return std::make_pair(false, index);
    }
    for (; index < (int)trimList.size(); index++) {
        if (frame < trimList[index].start) {
            return std::make_pair(false, index);
        }
        if (frame <= trimList[index].fin) {
            return std::make_pair(true, index);
        }
    }
    return std::make_pair(false, index);
}

bool rearrange_trim_list(int frame, int offset, std::vector<sTrim> &trimList) {
    if (trimList.size() == 0)
        return true;
    if (frame < 0)
        return false;
    for (uint32_t i = 0; i < trimList.size(); i++) {
        if (trimList[i].start >= frame) {
            trimList[i].start = clamp(trimList[i].start + offset, 0, TRIM_MAX);
        }
        if (trimList[i].fin && trimList[i].fin >= frame) {
            trimList[i].fin = (int)clamp((int64_t)trimList[i].fin + offset, 0, (int64_t)TRIM_MAX);
        }
    }
    return false;
}

tstring print_metadata(const std::vector<tstring> &metadata) {
    tstring str;
    for (const auto &m : metadata) {
        str += _T(" \"") + m + _T("\"");
    }
    return str;
}

bool metadata_copy(const std::vector<tstring> &metadata) {
    return std::find(metadata.begin(), metadata.end(), RGY_METADATA_COPY) != metadata.end();
}

bool metadata_clear(const std::vector<tstring> &metadata) {
    return std::find(metadata.begin(), metadata.end(), RGY_METADATA_CLEAR) != metadata.end();
}

#if !FOR_AUO
unique_ptr<RGYHDR10Plus> initDynamicHDR10Plus(const tstring &dynamicHdr10plusJson, shared_ptr<RGYLog> log) {
    unique_ptr<RGYHDR10Plus> hdr10plus;
    if (!rgy_file_exists(dynamicHdr10plusJson)) {
        log->write(RGY_LOG_ERROR, RGY_LOGT_HDR10PLUS, _T("Cannot find the file specified : %s.\n"), dynamicHdr10plusJson.c_str());
    } else {
        hdr10plus = std::unique_ptr<RGYHDR10Plus>(new RGYHDR10Plus());
        auto ret = hdr10plus->init(dynamicHdr10plusJson);
        if (ret == RGY_ERR_NOT_FOUND) {
            log->write(RGY_LOG_ERROR, RGY_LOGT_HDR10PLUS, _T("Cannot find \"%s\" required for --dhdr10-info.\n"), RGYHDR10Plus::HDR10PLUS_GEN_EXE_NAME);
            hdr10plus.reset();
        } else if (ret != RGY_ERR_NONE) {
            log->write(RGY_LOG_ERROR, RGY_LOGT_HDR10PLUS, _T("Failed to initialize hdr10plus reader: %s.\n"), get_err_mes((RGY_ERR)ret));
            hdr10plus.reset();
        }
        log->write(RGY_LOG_DEBUG, RGY_LOGT_HDR10PLUS, _T("initialized hdr10plus reader: %s\n"), dynamicHdr10plusJson.c_str());
    }
    return hdr10plus;
}
#endif
