package hiro.yoshioka.sql;

import hiro.yoshioka.ast.sql.oracle.WolfSQLParserConstants;
import hiro.yoshioka.sdh.ResultSetDataHolder;
import hiro.yoshioka.sdh2.ResultSetDataHolder2;
import hiro.yoshioka.sql.engine.GettingResourceRequest;
import hiro.yoshioka.sql.engine.Request;
import hiro.yoshioka.sql.engine.SQLOperationType;
import hiro.yoshioka.sql.engine.TransactionRequest;
import hiro.yoshioka.sql.resource.DBColumn;
import hiro.yoshioka.sql.resource.DBResourceType;
import hiro.yoshioka.sql.resource.DBSequence;
import hiro.yoshioka.sql.resource.IDBColumn;
import hiro.yoshioka.sql.resource.IDBResource;
import hiro.yoshioka.sql.resource.IDBSchema;
import hiro.yoshioka.sql.resource.IDBTable;
import hiro.yoshioka.util.SQLDataType;
import hiro.yoshioka.util.SQLUtil;
import hiro.yoshioka.util.StringUtil;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class FileMakerSQL extends GeneralSQL {
	static final String DEFAULT_SCHEMA_NAME = "PUBLIC";
	static final String SQL_EXISTS_SCHEMA = "SELECT COUNT(*) CNT FROM INFORMATION_SCHEMA.SYSTEM_SCHEMAS WHERE UPPER(TABLE_SCHEM) = ?";
	static final String SQL_CREATE_SCHEMA_PRE = "CREATE SCHEMA ";
	static final String SQL_CREATE_SCHEMA_POST = " AUTHORIZATION DBA";
	static final String SQL_SELECT_TABLES = "SELECT TABLE_NAME FROM INFORMATION_SCHEMA.SYSTEM_TABLES "
			+ " WHERE TABLE_TYPE='TABLE' AND upper(TABLE_SCHEM) = UPPER(?)";
	static final String SQL_EXISTS_TABLE = "SELECT COUNT(*) CNT FROM INFORMATION_SCHEMA.SYSTEM_TABLES WHERE upper(TABLE_SCHEM) = ? AND UPPER(TABLE_NAME) =?";
	static final String SQL_DROP_TABLE = "DROP TABLE ";

	static final String _SELECT_DIFF_TABLE = " SELECT                                                                          "
			+ "   TABLE_NAME,            NULL AS TABLE_COMMENT,                                 "
			+ "   COLUMN_NAME,           TYPE_NAME,                                             "
			+ "   COLUMN_SIZE AS COLUMN_LENGTH,                                                 "
			+ "   NULL AS FLOATPART,                                                            "
			+ "   CASE NULLABLE WHEN 1 THEN NULL ELSE 'NN' END not_null,                        "
			+ "   NULL AS PRI_KEY,                                                              "
			+ "   NULL AS uniq_KEY,                                                             "
			+ "   ORDINAL_POSITION AS COLUMN_NUMBER,                                             "
			+ "   REMARKS AS COLUMN_COMMENT                                                      "
			+ " FROM                                                                            "
			+ "   INFORMATION_SCHEMA.SYSTEM_COLUMNS                                             "
			+ " WHERE                                                                           "
			+ "   UPPER(TABLE_SCHEM) = UPPER(?)                                           ";

	static final String _SQL_COLUMN_PK = "SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.SYSTEM_PRIMARYKEYS  "
			+ " WHERE TABLE_SCHEM = ? AND TABLE_NAME=?";

	static final String _SELECT_ALL_SEQUENCE = "SELECT  "
			+ "  SEQUENCE_SCHEMA, " + "  SEQUENCE_NAME, "
			+ "  DTD_IDENTIFIER, " + "  MAXIMUM_VALUE \"MAX\", "
			+ "  MINIMUM_VALUE \"MIN\", " + "  INCREMENT, "
			+ "  CYCLE_OPTION CYCLE, " + "  START_WITH "
			+ " FROM INFORMATION_SCHEMA.SYSTEM_SEQUENCES  "
			+ " WHERE SEQUENCE_SCHEMA = ? ";
	static final String _SELECT_SEQUENCE_CNT = "SELECT COUNT(*) AS \"CNT\" FROM INFORMATION_SCHEMA.SYSTEM_SEQUENCES WHERE UPPER(SEQUENCE_SCHEMA) = UPPER(?) AND UPPER(SEQUENCE_NAME) =UPPER(?)";

	public static String getSuggestURL() {
		return "jdbc:filemaker://localhost/<DB_NAME>";
	}

	protected FileMakerSQL(Driver ds) {
		super(ds);
	}

	@Override
	public boolean doOperation(SQLOperationType Operation, Request request)
			throws SQLException {
		boolean retCode = true;
		TransactionRequest treq = null;
		if (request instanceof TransactionRequest) {
			treq = (TransactionRequest) request;
			setMaxRowNum(treq.getMaxRownum());
		}
		switch (Operation) {
		case COUNT:
			String sql = null;
			if (StringUtil.isEmpty(treq.getSQLStatement())) {
				IDBTable table = treq.getIDBTable();

				sql = String.format("SELECT COUNT(*) CNT FROM \"%s\"",
						table.getName());
			}
			treq.setResultCount(count(sql));
			break;
		default:
			return super.doOperation(Operation, request);
		}

		return retCode;
	}

	@Override
	protected boolean createSchema(String schemaName, Properties properties)
			throws SQLException {
		String st = String.format("CREATE SCHEMA %s AUTHORIZATION DBA",
				schemaName);
		executePrepare(st, EMPTY);
		return true;
	}

	@Override
	public ResultSet getAllData(IDBTable table) throws SQLException {
		IDBResource schema = table.getParent();

		getStatement(_con, ResultSet.TYPE_FORWARD_ONLY,
				ResultSet.CONCUR_READ_ONLY);

		String sql = String.format("SELECT * FROM \"%s\"", table.getName());
		return fActiveStatement.executeQuery(sql);
	}

	@Override
	public String getDefaultSchemaName() {
		return FileMakerSQL.DEFAULT_SCHEMA_NAME;
	}

	@Override
	public boolean existsTable(String schemaName, String tableName)
			throws SQLException {
		ResultSetDataHolder rdh = executePrepareQuery(
				SQL_EXISTS_TABLE,
				new String[] { schemaName.toUpperCase(),
						tableName.toUpperCase() });
		return rdh.getIntDataDefaultZero(0, "CNT") > 0;
	}

	protected boolean dropTable(String schemaName, IDBTable table,
			boolean cascade, boolean quoteTableName) throws SQLException {
		// if (StringUtil.isEmpty(schemaName)) {
		// schemaName = HSQL.DEFAULT_SCHEMA_NAME;
		// }
		return super.dropTable(schemaName, table, cascade, true);
	}

	@Override
	public void setTableColumns(String schema, IDBTable table)
			throws SQLException {
		String quote_table_name = String.format("\"%s\"", table.getName());

		Connection con = this.getOrCreateExtraConnection();
		DatabaseMetaData meta = con.getMetaData();
		ResultSet rs_m = meta.getColumnPrivileges(null, null, table.getName(),
				null);
		Set<String> insertableColSets = new HashSet<String>();
		while (rs_m.next()) {
			String col = rs_m.getString("COLUMN_NAME");
			String privilege = rs_m.getString("PRIVILEGE");
			if ("INSERT".equalsIgnoreCase(privilege)) {
				insertableColSets.add(col);
			}
		}
		rs_m.close();

		ResultSet rs = null;
		Statement st = null;
		StringBuilder sql = new StringBuilder();
		try {
			st = this.getOrCreateExtraConnection().createStatement();
			st.setMaxRows(1);

			rs = st.executeQuery(String.format("SELECT * FROM \"%s\"",
					table.getName()));
			ResultSetMetaData md = rs.getMetaData();
			for (int i = 1; i <= md.getColumnCount(); i++) {
				DBColumn col = new DBColumn(table);
				col.setName(md.getColumnName(i));
				col.setDataType(SQLDataType.parse(md.getColumnType(i)));

				try {
					col.setSize(md.getPrecision(i));
				} catch (Exception e) {
					col.setSize(0);
				}
				try {
					col.setDecimalDigits(md.getScale(i));
				} catch (Exception e) {
					col.setDecimalDigits(0);
				}

				col.setNullable((short) md.isNullable(i));
				if (!insertableColSets.contains(col.getName())) {
					col.setInsertDisabled(true);
				}
				col.setDataTypeString(md.getColumnTypeName(i));
				col.setMaxColumnNameLength(SQLUtil.getMaxColumnNameBytes(this
						.getOrCreateExtraConnection()));

				table.putResource(col.getUName(), col);
			}
		} catch (SQLException e) {
			fLogger.info("sql=" + sql); //$NON-NLS-1$
			fLogger.info(StringUtil.EMPTY_STRING, e);
			return;
		} finally {
			if (rs != null) {
				rs.close();
			}
			if (st != null) {
				st.close();
			}
		}
	}

	@Override
	protected void getSequence(GettingResourceRequest request)
			throws SQLException {
		ResultSetDataHolder rdh = null;
		IDBSchema[] schemas = getRoot().getSchemas();
		for (int i = 0; i < schemas.length; i++) {
			if (schemas[i].getName().length() == 0) {
				continue;
			}
			if (request.targetType.isOnlySchema()
					&& !schemas[i].equals(request.selectionResource)) {
				continue;
			}

			rdh = executePrepareQuery(_SELECT_ALL_SEQUENCE,
					new String[] { schemas[i].getName() });
			DBSequence sequence = null;
			for (int j = 0; j < rdh.getRowCount(); j++) {
				sequence = new DBSequence(schemas[i]);
				String name = rdh.getStringData(j, "SEQUENCE_NAME");
				sequence.setName(name);
				// String text = _SELECT_SEQUENCE_BODY + schemas[i].getName()
				// + "." + name;
				// rdh2 = executePrepareQuery(text, EMPTY);
				if (rdh.getRowCount() > 0) {
					setResourceProperties(sequence, 0, rdh);
				}
				schemas[i].putSequence(sequence);
				// setViewText(trigger);
			}
			if (request.canceld()) {
				return;
			}
		}
	}

	protected String getSupportToken() {
		StringBuffer buf = new StringBuffer();
		Pattern p = Pattern.compile("\"(\\w+)\"");
		String[] str = WolfSQLParserConstants.tokenImage;
		for (int i = 0; i < str.length; i++) {
			Matcher m = p.matcher(str[i]);
			if (m.matches()) {
				buf.append(m.group(1)).append(",");
			}
		}
		if (buf.length() > 0) {
			buf.setLength(buf.length() - 1);
		}
		return buf.toString();
	}

	public ResultSetDataHolder2 executeBatUpdate(String[] statements)
			throws SQLException {
		Statement statement = null;
		ResultSetDataHolder2 sdh = null;
		if (_con == null) {
			return sdh;
		}
		try {
			boolean[] ret = new boolean[statements.length];
			long time = System.currentTimeMillis();
			for (int i = 0; i < ret.length; i++) {
				ret[i] = execute(statements[i]);
			}

			time = System.currentTimeMillis() - time;

			sdh = createRdh2(new String[] { "Returns", "Statement" }, null);
			for (int i = 0; i < ret.length; i++) {
				sdh.addRow(new String[] { String.valueOf(ret[i]), statements[i] });
			}

			sdh.setWrapTime(time);
			return sdh;
		} finally {
			setTransactionTime(true);

			if (statement != null) {
				statement.close();
			}
		}
	}

	// ----------------------------------------------------------------
	// [4] DIFF
	// ----------------------------------------------------------------
	@Override
	public ResultSetDataHolder2 getDiffInfoOf(DBResourceType resourceType,
			String schemaName, boolean withValue) {
		ResultSetDataHolder2 ret = null;
		try {
			switch (resourceType) {
			case TABLE:
				ret = executePrepareQuery(_SELECT_DIFF_TABLE,
						new String[] { schemaName });
				break;
			}
		} catch (SQLException e) {
			fLogger.error(StringUtil.EMPTY_STRING, e);
		}
		if (ret != null) {
			ret.setTableNameE(String.format("DEF_%s_%s", schemaName,
					resourceType.name()));
		}
		fLogger.debug("[" + _info.getDisplayString() + "] ret:[" + ret + "]");
		return ret;
	}

	public void test() {
		try {
			DatabaseMetaData meta = getOrCreateExtraConnection().getMetaData();

			ResultSet rs0 = meta.getTables(null, null, "%",
					new String[] { "TABLE" });
			ResultSetMetaData rmeta = rs0.getMetaData();

			System.out.println("3.-----------------------------------");
			ResultSet rs = meta.getColumnPrivileges(null, null, "イベント", null);
			ResultSetDataHolder2 rdh = RS2RDH(rs, true);
			System.out.println(rdh);

			rs = meta.getColumns(null, null, "イベント", null);
			System.out.println("4.-----------------------------------");
			rdh = RS2RDH(rs, true);
			System.out.println(rdh);

		} catch (SQLException e) {
			// TODO 自動生成された catch ブロック
			e.printStackTrace();
		}

	}

}