/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   common.c

   */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif /* HAVE_SYS_TIME_H */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <ctype.h>
#ifndef __W32__
#include <unistd.h>
#else
#include <process.h>
#include <io.h>
#endif /* __W32__ */
#include "timidity.h"
#include "common.h"
#include "output.h"
#include "controls.h"
#include "arc.h"
#include "nkflib.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "wrd.h"
#include "support.h"
//#include "strtab.h"
#include "common_prv.h" 

/* RAND_MAX must defined in stdlib.h
 * Why RAND_MAX is not defined at SunOS?
 */
#if defined(sun) && !defined(SOLARIS) && !defined(RAND_MAX)
#define RAND_MAX ((1<<15)-1)
#endif

#ifndef O_BINARY
#define O_BINARY 0
#endif

static const char *note_name[] =
{
    "C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"
};


#ifndef TMP_MAX
#define TMP_MAX 238328
#endif

int
tmdy_mkstemp(tmdy_struct_ex_t * tmdy_struct, char *tmpl)
{
  char *XXXXXX;
//  static uint32 value;
  uint32 random_time_bits;
  int count, fd = -1;
  int save_errno = errno;

  /* These are the characters used in temporary filenames.  */
  static const char letters[] =
    "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";

  /* This is where the Xs start.  */
  XXXXXX = strstr(tmpl, "XXXXXX");
  if (XXXXXX == NULL) {
    errno = EINVAL;
    return -1;
  }

  /* Get some more or less random data.  */
#if HAVE_GETTIMEOFDAY
  {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    random_time_bits = (uint32)((tv.tv_usec << 16) ^ tv.tv_sec);
  }
#else
  random_time_bits = (uint32)time(NULL);
#endif

  TMDY_COMMON->value += random_time_bits ^ getpid();

  for (count = 0; count < TMP_MAX; TMDY_COMMON->value += 7777, ++count) {
    uint32 v = TMDY_COMMON->value;

    /* Fill in the random bits.  */
    XXXXXX[0] = letters[v % 62];
    v /= 62;
    XXXXXX[1] = letters[v % 62];
    v /= 62;
    XXXXXX[2] = letters[v % 62];

    v = (v << 16) ^ TMDY_COMMON->value;
    XXXXXX[3] = letters[v % 62];
    v /= 62;
    XXXXXX[4] = letters[v % 62];
    v /= 62;
    XXXXXX[5] = letters[v % 62];

#ifdef _MSC_VER
#define S_IRUSR 0
#define S_IWUSR 0
#endif

	fd = open(tmpl, O_RDWR | O_CREAT | O_EXCL | O_BINARY, S_IRUSR | S_IWUSR);

    if (fd >= 0) {
      errno = save_errno;
      return fd;
    }
    if (errno != EEXIST){
      return -1;
    }
  }

  /* We got out of the loop because we ran out of combinations to try.  */
  errno = EEXIST;
  return -1;
}


static char *
url_dumpfile(tmdy_struct_ex_t * tmdy_struct, URL url, const char *ext)
{
  char filename[1024];
  char *tmpdir;
  int fd;
  FILE *fp;
  int n;
  char buff[BUFSIZ];

#ifdef TMPDIR
  tmpdir = TMPDIR;
#else
  tmpdir = getenv("TMPDIR");
#endif
  if(tmpdir == NULL || strlen(tmpdir) == 0)
    tmpdir = PATH_STRING "tmp" PATH_STRING;
  if(IS_PATH_SEP(tmpdir[strlen(tmpdir) - 1]))
    snprintf(filename, sizeof(filename), "%sXXXXXX.%s", tmpdir, ext);
  else
    snprintf(filename, sizeof(filename), "%s" PATH_STRING "XXXXXX.%s",
	     tmpdir, ext);

  fd = tmdy_mkstemp(tmdy_struct, filename);

  if (fd == -1)
    return NULL;

  if ((fp = fdopen(fd, "w")) == NULL) {
    close(fd);
    unlink(filename);
    return NULL;
  }

  while((n = TMDY_ARC->url->url_read(tmdy_struct, url, buff, sizeof(buff))) > 0)
    fwrite(buff, 1, n, fp);
  fclose(fp);
  return safe_strdup(tmdy_struct, filename);
}


/* Try to open a file for reading. If the filename ends in one of the
   defined compressor extensions, pipe the file through the decompressor */
struct timidity_file *try_to_open(tmdy_struct_ex_t *tmdy_struct, char *name, int decompress)
{
    struct timidity_file *tf;
    URL url;
    int len;

    if((url = TMDY_ARC->arc->url_arc_open(tmdy_struct, name)) == NULL)
      if((url = TMDY_ARC->url->url_open(tmdy_struct, name)) == NULL)
	return NULL;

    tf = (struct timidity_file *)safe_malloc(tmdy_struct, sizeof(struct timidity_file));
    tf->url = url;
    tf->tmpname = NULL;

    len = strlen(name);
    if(decompress && len >= 3 && strcasecmp(name + len - 3, ".gz") == 0)
    {
	int method;

	if(!IS_URL_SEEK_SAFE(tf->url))
	{
	    if((tf->url = TMDY_ARC->url->url_cache_open(tmdy_struct, tf->url, 1)) == NULL)
	    {
		close_file(tmdy_struct, tf);
		return NULL;
	    }
	}

	method = TMDY_ARC->arc->skip_gzip_header(tmdy_struct, tf->url);
	if(method == ARCHIVEC_DEFLATED)
	{
	    TMDY_ARC->url->url_cache_disable(tmdy_struct, tf->url);
	    if((tf->url = TMDY_ARC->url->url_inflate_open(tmdy_struct, tf->url, -1, 1)) == NULL)
	    {
		close_file(tmdy_struct, tf);
		return NULL;
	    }

	    /* success */
	    return tf;
	}
	/* fail */
	TMDY_ARC->url->url_rewind(tmdy_struct, tf->url);
	TMDY_ARC->url->url_cache_disable(tmdy_struct, tf->url);
    }

#ifdef __W32__
    /* Sorry, DECOMPRESSOR_LIST and PATCH_CONVERTERS are not worked yet. */
    return tf;
#endif /* __W32__ */

#if defined(DECOMPRESSOR_LIST)
    if(decompress)
    {
	static const char *decompressor_list[] = DECOMPRESSOR_LIST;
    const char  **dec;
	char tmp[1024], *tmpname;

	/* Check if it's a compressed file */
	for(dec = decompressor_list; *dec; dec += 2)
	{
	    if(!check_file_extension(tmdy_struct, name, (char*)*dec, 0))
		continue;

	    tf->tmpname = url_dumpfile(tmdy_struct, tf->url, *dec);
	    if (tf->tmpname == NULL) {
		close_file(tmdy_struct, tf);
		return NULL;
	    }

	    TMDY_ARC->url->url_close(tmdy_struct, tf->url);
	    snprintf(tmp, sizeof(tmp), *(dec+1), tf->tmpname);
	    if((tf->url = TMDY_ARC->url->url_pipe_open(tmdy_struct, tmp)) == NULL)
	    {
		close_file(tmdy_struct, tf);
		return NULL;
	    }

	    break;
	}
    }
#endif /* DECOMPRESSOR_LIST */

#if defined(PATCH_CONVERTERS)
    if(decompress == 2)
    {
	static char *decompressor_list[] = PATCH_CONVERTERS, **dec;
	char tmp[1024], *tmpname;

	/* Check if it's a compressed file */
	for(dec = decompressor_list; *dec; dec += 2)
	{
	    if(!check_file_extension(tmdy_struct, name, *dec, 0))
		continue;

	    tf->tmpname = url_dumpfile(tmdy_struct, tf->url, *dec);
	    if (tf->tmpname == NULL) {
		close_file(tmdy_struct, tf);
		return NULL;
	    }

	    TMDY_ARC->url->url_close(tmdy_struct, tf->url);
	    sprintf(tmp, *(dec+1), tf->tmpname);
	    if((tf->url = TMDY_ARC->url->url_pipe_open(tmdy_struct, tmp)) == NULL)
	    {
		close_file(tmdy_struct, tf);
		return NULL;
	    }

	    break;
	}
    }
#endif /* PATCH_CONVERTERS */
    
    return tf;
}

static int is_url_prefix(tmdy_struct_ex_t * tmdy_struct, char *name)
{
    int i;

    static const char *url_proto_names[] =
    {
	"file:",
#ifdef SUPPORT_SOCKET
	"http://",
	"ftp://",
	"news://",
#endif /* SUPPORT_SOCKET */
	"mime:",
	NULL
    };
    for(i = 0; url_proto_names[i]; i++)
	if(strncmp(name, url_proto_names[i], strlen(url_proto_names[i])) == 0)
	    return 1;

#ifdef __W32__
    /* [A-Za-z]: (for Windows) */
    if((('A' <= name[0] && name[0] <= 'Z') ||
	('a' <= name[0] && name[0] <= 'z')) &&
       name[1] == ':')
	return 1;
#endif /* __W32__ */

    return 0;
}

struct timidity_file *open_with_mem(tmdy_struct_ex_t * tmdy_struct, char *mem, int32 memlen, int noise_mode)
{
    URL url;
    struct timidity_file *tf;

    errno = 0;
    if((url = TMDY_ARC->url->url_mem_open(tmdy_struct, mem, memlen, 0)) == NULL)
    {
	if(noise_mode >= 2)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "Can't open.");
	return NULL;
    }
    tf = (struct timidity_file *)safe_malloc(tmdy_struct, sizeof(struct timidity_file));
    tf->url = url;
    tf->tmpname = NULL;
    return tf;
}

/* This is meant to find and open files for reading, possibly piping
   them through a decompressor. */
struct timidity_file *open_file(tmdy_struct_ex_t * tmdy_struct, char *name, int decompress, int noise_mode)
{
  struct timidity_file *tf;
  PathList *plp=(TMDY_COMMON->pathlist);
  int l;

  TMDY_COMMON->open_file_noise_mode = noise_mode;
  if (!name || !(*name))
    {
      if(noise_mode)
        (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "Attempted to open nameless file.");
      return 0;
    }

  /* First try the given name */
  strncpy(TMDY_COMMON->current_filename, TMDY_ARC->url->url_unexpand_home_dir(tmdy_struct, name), 1023);
  TMDY_COMMON->current_filename[1023]='\0';

  if(noise_mode)
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "Trying to open %s", TMDY_COMMON->current_filename);
  if ((tf=try_to_open(tmdy_struct, TMDY_COMMON->current_filename, decompress))){
    return tf;
  }

#ifdef __MACOS__
  if(errno)
#else
  if(errno && errno != ENOENT)
#endif
    {
	if(noise_mode)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "%s: %s",
		      TMDY_COMMON->current_filename, strerror(errno));
	return 0;
    }

  if (!IS_PATH_SEP(name[0]) && !is_url_prefix(tmdy_struct, name))
    while (plp)  /* Try along the path then */
      {
	*TMDY_COMMON->current_filename=0;
	l=strlen(plp->path);
	if(l)
	  {
	    strcpy(TMDY_COMMON->current_filename, plp->path);
	    if(!IS_PATH_SEP(TMDY_COMMON->current_filename[l-1]) &&
	       TMDY_COMMON->current_filename[l-1] != '#' &&
	       name[0] != '#')
		strcat(TMDY_COMMON->current_filename, PATH_STRING);
	  }
	strcat(TMDY_COMMON->current_filename, name);
	if(noise_mode)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		      "Trying to open %s", TMDY_COMMON->current_filename);
	if ((tf=try_to_open(tmdy_struct, TMDY_COMMON->current_filename, decompress))){
	  return tf;
	}
#ifdef __MACOS__
	if(errno)
#else
	if(errno && errno != ENOENT)
#endif
	{
	    if(noise_mode)
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "%s: %s",
			  TMDY_COMMON->current_filename, strerror(errno));
	    return 0;
	  }
	plp=plp->next;
      }

  /* Nothing could be opened. */

  *TMDY_COMMON->current_filename=0;

  if (noise_mode>=2)
      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "%s: %s", name,
		errno ? strerror(errno) : "Can't open file");

  return 0;
}

/* This closes files opened with open_file */
void close_file(tmdy_struct_ex_t * tmdy_struct, struct timidity_file *tf)
{
    int save_errno = errno;
    if(tf->url != NULL)
    {
#ifndef __W32__
	if(tf->tmpname != NULL)
	{
	    int i;
	    /* dispose the pipe garbage */
	    for(i = 0; tf_getc(tmdy_struct, tf) != EOF && i < 0xFFFF; i++)
		;
	}
#endif /* __W32__ */
	TMDY_ARC->url->url_close(tmdy_struct, tf->url);
    }
    if(tf->tmpname != NULL)
    {
	unlink(tf->tmpname); /* remove temporary file */
	free(tf->tmpname);
    }
    free(tf);
    errno = save_errno;
}

/* This is meant for skipping a few bytes. */
void skip(tmdy_struct_ex_t * tmdy_struct, struct timidity_file *tf, size_t len)
{
    TMDY_ARC->url->url_skip(tmdy_struct, tf->url, (long)len);
}
int tf_getc(tmdy_struct_ex_t * tmdy_struct, struct timidity_file *tf)
{
	    return TMDY_ARC->url->url_getc(tmdy_struct, tf->url);
}
char *tf_gets(tmdy_struct_ex_t * tmdy_struct, char *buff, int n, struct timidity_file *tf)
{
    return TMDY_ARC->url->url_gets(tmdy_struct, tf->url, buff, n);
}

long tf_read(tmdy_struct_ex_t * tmdy_struct, void *buff, int32 size, int32 nitems, struct timidity_file *tf)
{
    return TMDY_ARC->url->url_nread(tmdy_struct, tf->url, buff, size * nitems) / size;
}

long tf_seek(tmdy_struct_ex_t * tmdy_struct, struct timidity_file *tf, long offset, int whence)
{
    long prevpos;

    prevpos = TMDY_ARC->url->url_seek(tmdy_struct, tf->url, offset, whence);
    if(prevpos == -1)
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
		  "Warning: Can't seek file position");
    return prevpos;
}

long tf_tell(tmdy_struct_ex_t * tmdy_struct, struct timidity_file *tf)
{
    long pos;

    pos = TMDY_ARC->url->url_tell(tmdy_struct, tf->url);
    if(pos == -1)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
		  "Warning: Can't get current file position");
	return (long)tf->url->nread;
    }

    return pos;
}

void safe_exit(tmdy_struct_ex_t * tmdy_struct, int status)
{
    if( (tmdy_struct->output->play_mode) != NULL &&
    	(tmdy_struct->output->play_mode)->fd != -1)
    {
	(tmdy_struct->output->play_mode)->acntl(tmdy_struct, PM_REQ_DISCARD, NULL);
	(tmdy_struct->output->play_mode)->close_output(tmdy_struct);
    }
    (TMDY_CONTROLS->ctl)->close(tmdy_struct);
    tmdy_struct->wrd->wrdt->close(tmdy_struct);
    exit(status);
    /*NOTREACHED*/
}

/* This'll allocate memory or die. */
void *safe_malloc(tmdy_struct_ex_t * tmdy_struct, size_t count)
{
    void *p;
//    static int errflag = 0;

    if(TMDY_COMMON->safe_malloc_errflag)
	safe_exit(tmdy_struct, 10);
    if(count > MAX_SAFE_MALLOC_SIZE)
    {
	TMDY_COMMON->safe_malloc_errflag = 1;
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
		  "Strange, I feel like allocating %d bytes. "
		  "This must be a bug.", count);
    }
    else {
      if(count == 0)
	/* Some malloc routine return NULL if count is zero, such as
	 * malloc routine from libmalloc.a of Solaris.
	 * But TiMidity doesn't want to return NULL even if count is zero.
	 */
	count = 1;
      if((p = (void *)malloc(count)) != NULL){
		return p;
	  }
      TMDY_COMMON->safe_malloc_errflag = 1;
      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
		"Sorry. Couldn't malloc %d bytes.", count);
    }
#ifdef ABORT_AT_FATAL
    abort();
#endif /* ABORT_AT_FATAL */
    safe_exit(tmdy_struct, 10);
    /*NOTREACHED*/
}

void *safe_large_malloc(tmdy_struct_ex_t * tmdy_struct, size_t count)
{
    void *p;
//    static int safe_large_malloc_errflag = 0;

    if(TMDY_COMMON->safe_large_malloc_errflag)
	safe_exit(tmdy_struct, 10);
    if(count == 0)
      /* Some malloc routine return NULL if count is zero, such as
       * malloc routine from libmalloc.a of Solaris.
       * But TiMidity doesn't want to return NULL even if count is zero.
       */
      count = 1;
    if((p = (void *)malloc(count)) != NULL){
      return p;
    }
    TMDY_COMMON->safe_large_malloc_errflag = 1;
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
	      "Sorry. Couldn't malloc %d bytes.", count);

#ifdef ABORT_AT_FATAL
    abort();
#endif /* ABORT_AT_FATAL */
    safe_exit(tmdy_struct, 10);
    /*NOTREACHED*/
}

void *safe_realloc(tmdy_struct_ex_t * tmdy_struct, void *ptr, size_t count)
{
    void *p;
//    static int safe_realloc_errflag = 0;

    if(TMDY_COMMON->safe_realloc_errflag)
	safe_exit(tmdy_struct, 10);
    if(count > MAX_SAFE_MALLOC_SIZE)
    {
	TMDY_COMMON->safe_realloc_errflag = 1;
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
		  "Strange, I feel like allocating %d bytes. "
		  "This must be a bug.", count);
    }
    else {
      if (ptr == NULL){
		return safe_malloc(tmdy_struct, count);
      }
      if(count == 0)
	/* Some malloc routine return NULL if count is zero, such as
	 * malloc routine from libmalloc.a of Solaris.
	 * But TiMidity doesn't want to return NULL even if count is zero.
	 */
	count = 1;
      if((p = (void *)realloc(ptr, count)) != NULL){
			return p;
      }
      TMDY_COMMON->safe_realloc_errflag = 1;
      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
		"Sorry. Couldn't malloc %d bytes.", count);
    }
#ifdef ABORT_AT_FATAL
    abort();
#endif /* ABORT_AT_FATAL */
    safe_exit(tmdy_struct, 10);
    /*NOTREACHED*/
}

/* This'll allocate memory or die. */
char *safe_strdup(tmdy_struct_ex_t * tmdy_struct, const char *s)
{
    char *p;
//    static int safe_strdup_errflag = 0;

    if(TMDY_COMMON->safe_strdup_errflag)
	safe_exit(tmdy_struct, 10);

    if(s == NULL)
	p = strdup("");
    else
	p = strdup(s);
    if(p != NULL){
		return p;
    }
    TMDY_COMMON->safe_strdup_errflag = 1;
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL, "Sorry. Couldn't alloc memory.");
#ifdef ABORT_AT_FATAL
    abort();
#endif /* ABORT_AT_FATAL */
    safe_exit(tmdy_struct, 10);
    /*NOTREACHED*/
}

/* free ((void **)ptr_list)[0..count-1] and ptr_list itself */
void free_ptr_list(tmdy_struct_ex_t * tmdy_struct, void *ptr_list, int count)
{
	int i;
	for(i = 0; i < count; i++)
		free(((void **)ptr_list)[i]);
	free(ptr_list);
}

/* This adds a directory to the path list */
void add_to_pathlist(tmdy_struct_ex_t * tmdy_struct, char *s)
{
    PathList *cur, *prev, *plp;

    /* Check duplicated path in the pathlist. */
    plp = prev = NULL;
    for(cur = (TMDY_COMMON->pathlist); cur; prev = cur, cur = cur->next)
	if(pathcmp(tmdy_struct, s, cur->path, 0) == 0)
	{
	    plp = cur;
	    break;
	}

    if(plp) /* found */
    {
	if(prev == NULL) /* first */
	    (TMDY_COMMON->pathlist) = (TMDY_COMMON->pathlist)->next;
	else
	    prev->next = plp->next;
    }
    else
    {
	/* Allocate new path */
	plp = safe_malloc(tmdy_struct, sizeof(PathList));
	plp->path = safe_strdup(tmdy_struct, s);
    }

    plp->next = (TMDY_COMMON->pathlist);
    (TMDY_COMMON->pathlist) = plp;
}

void clean_up_pathlist(tmdy_struct_ex_t * tmdy_struct)
{
  PathList *cur, *next;

  cur = (TMDY_COMMON->pathlist);
  while (cur) {
    next = cur->next;
#ifdef DEFAULT_PATH
    if (cur == &(TMDY_COMMON->defaultpathlist)) {
      cur = next;
      continue;
    }
#endif
    free(cur->path);
    free(cur);
    cur = next;
  }

#ifdef DEFAULT_PATH
  (TMDY_COMMON->pathlist) = &(TMDY_COMMON->defaultpathlist);
#else
  (TMDY_COMMON->pathlist) = NULL;
#endif
}

#ifndef HAVE_VOLATILE
/*ARGSUSED*/
int volatile_touch(void *dmy) {return 1;}
#endif /* HAVE_VOLATILE */

/* code converters */
static const unsigned char
      w2k[] = {128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
               144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
               160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
               176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
               225,226,247,231,228,229,246,250,233,234,235,236,237,238,239,240,
               242,243,244,245,230,232,227,254,251,253,255,249,248,252,224,241,
               193,194,215,199,196,197,214,218,201,202,203,204,205,206,207,208,
               210,211,212,213,198,200,195,222,219,221,223,217,216,220,192,209};

static void code_convert_cp1251(tmdy_struct_ex_t * tmdy_struct, char *in, char *out, int maxlen)
{
    int i;
    if(out == NULL)
        out = in;
    for(i = 0; i < maxlen && in[i]; i++)
    {
	if(in[i] & 0200)
	    out[i] = w2k[in[i] & 0177];
	else
	    out[i] = in[i];
    }
    out[i]='\0';
}

static void code_convert_dump(tmdy_struct_ex_t * tmdy_struct, char *in, char *out, int maxlen, char *ocode)
{
    if(ocode == NULL)
	ocode = TMDY_COMMON->output_text_code;

    if(ocode != NULL && ocode != (char *)-1
	&& (strstr(ocode, "ASCII") || strstr(ocode, "ascii")))
    {
	int i;

	if(out == NULL)
	    out = in;
	for(i = 0; i < maxlen && in[i]; i++)
	    if(in[i] < ' ' || in[i] >= 127)
		out[i] = '.';
	    else
		out[i] = in[i];
	out[i]='\0';
    }
    else /* "NOCNV" */
    {
	if(out == NULL){
	    return;
	}
	strncpy(out, in, maxlen);
	out[maxlen] = '\0';
    }
}

#ifdef JAPANESE
static void code_convert_japan(tmdy_struct_ex_t * tmdy_struct, char *in, char *out, int maxlen,
			       char *icode, char *ocode)
{
//    static char *mode = NULL, *wrd_mode = NULL;

    if(ocode != NULL && ocode != (char *)-1)
    {
	TMDY_UTILS->nkflib->nkf_convert(tmdy_struct, in, out, maxlen, icode, ocode);
	if(out != NULL)
	    out[maxlen] = '\0';
	return;
    }

    if(TMDY_COMMON->mode == NULL || TMDY_COMMON->wrd_mode == NULL)
    {
	TMDY_COMMON->mode = TMDY_COMMON->output_text_code;
	if(TMDY_COMMON->mode == NULL || strstr(TMDY_COMMON->mode, "AUTO"))
	{
#ifndef __W32__
	    TMDY_COMMON->mode = getenv("LANG");
#else
	    TMDY_COMMON->mode = "SJIS";
	    TMDY_COMMON->wrd_mode = "SJISK";
#endif
	    if(TMDY_COMMON->mode == NULL || *TMDY_COMMON->mode == '\0')
	    {
		TMDY_COMMON->mode = "ASCII";
		TMDY_COMMON->wrd_mode = TMDY_COMMON->mode;
	    }
	}

	if(strstr(TMDY_COMMON->mode, "ASCII") ||
	   strstr(TMDY_COMMON->mode, "ascii"))
	{
	    TMDY_COMMON->mode = "ASCII";
	    TMDY_COMMON->wrd_mode = TMDY_COMMON->mode;
	}
	else if(strstr(TMDY_COMMON->mode, "NOCNV") ||
		strstr(TMDY_COMMON->mode, "nocnv"))
	{
	    TMDY_COMMON->mode = "NOCNV";
	    TMDY_COMMON->wrd_mode = TMDY_COMMON->mode;
	}
#ifndef	HPUX
	else if(strstr(TMDY_COMMON->mode, "EUC") ||
		strstr(TMDY_COMMON->mode, "euc") ||
		strstr(TMDY_COMMON->mode, "ujis") ||
		strcmp(TMDY_COMMON->mode, "japanese") == 0)
	{
	    TMDY_COMMON->mode = "EUC";
	    TMDY_COMMON->wrd_mode = "EUCK";
	}
	else if(strstr(TMDY_COMMON->mode, "SJIS") ||
		strstr(TMDY_COMMON->mode, "sjis"))
	{
	    TMDY_COMMON->mode = "SJIS";
	    TMDY_COMMON->wrd_mode = "SJISK";
	}
#else
	else if(strstr(TMDY_COMMON->mode, "EUC") ||
		strstr(TMDY_COMMON->mode, "euc") ||
		strstr(TMDY_COMMON->mode, "ujis"))
	{
	    TMDY_COMMON->mode = "EUC";
	    TMDY_COMMON->wrd_mode = "EUCK";
	}
	else if(strstr(TMDY_COMMON->mode, "SJIS") ||
		strstr(TMDY_COMMON->mode, "sjis") ||
		strcmp(TMDY_COMMON->mode, "japanese") == 0)
	{
	    TMDY_COMMON->mode = "SJIS";
	    TMDY_COMMON->wrd_mode = "SJISK";
	}
#endif	/* HPUX */
	else if(strstr(TMDY_COMMON->mode,"JISk")||
		strstr(TMDY_COMMON->mode,"jisk"))
	{
	    TMDY_COMMON->mode = "JISK";
	    TMDY_COMMON->wrd_mode = TMDY_COMMON->mode;
	}
	else if(strstr(TMDY_COMMON->mode, "JIS") ||
		strstr(TMDY_COMMON->mode, "jis"))
	{
	    TMDY_COMMON->mode = "JIS";
	    TMDY_COMMON->wrd_mode = "JISK";
	}
	else if(strcmp(TMDY_COMMON->mode, "ja") == 0)
	{
	    TMDY_COMMON->mode = "EUC";
	    TMDY_COMMON->wrd_mode = "EUCK";
	}
	else
	{
	    TMDY_COMMON->mode = "NOCNV";
	    TMDY_COMMON->wrd_mode = TMDY_COMMON->mode;
	}
    }

    if(ocode == NULL)
    {
	if(strcmp(TMDY_COMMON->mode, "NOCNV") == 0)
	{
	    if(out == NULL){
			return;
		}
	    strncpy(out, in, maxlen);
	    out[maxlen] = '\0';
	}
	else if(strcmp(TMDY_COMMON->mode, "ASCII") == 0)
	    code_convert_dump(tmdy_struct, in, out, maxlen, "ASCII");
	else
	{
	    TMDY_UTILS->nkflib->nkf_convert(tmdy_struct, in, out, maxlen, icode, TMDY_COMMON->mode);
	    if(out != NULL)
		out[maxlen] = '\0';
	}
    }
    else if(ocode == (char *)-1)
    {
	if(strcmp(TMDY_COMMON->wrd_mode, "NOCNV") == 0)
	{
	    if(out == NULL){
			return;
		}
	    strncpy(out, in, maxlen);
	    out[maxlen] = '\0';
	}
	else if(strcmp(TMDY_COMMON->wrd_mode, "ASCII") == 0)
	    code_convert_dump(tmdy_struct, in, out, maxlen, "ASCII");
	else
	{
	    TMDY_UTILS->nkflib->nkf_convert(tmdy_struct, in, out, maxlen, icode, TMDY_COMMON->wrd_mode);
	    if(out != NULL)
		out[maxlen] = '\0';
	}
    }
}
#endif /* JAPANESE */

void code_convert(tmdy_struct_ex_t * tmdy_struct, char *in, char *out, int outsiz, char *icode, char *ocode)
{
#if !defined(MIME_CONVERSION) && defined(JAPANESE)
    int i;
    /* check ASCII string */
    for(i = 0; in[i]; i++)
	if(in[i] < ' ' || in[i] >= 127)
	    break;
    if(!in[i])
    {
	if(out == NULL)
	    return;
	strncpy(out, in, outsiz - 1);
	out[outsiz - 1] = '\0';
	return; /* All ASCII string */
    }
#endif /* MIME_CONVERSION */

    if(ocode != NULL && ocode != (char *)-1)
    {
	if(strcasecmp(ocode, "nocnv") == 0)
	{
	    if(out == NULL)
		return;
	    outsiz--;
	    strncpy(out, in, outsiz);
	    out[outsiz] = '\0';
	    return;
	}

	if(strcasecmp(ocode, "ascii") == 0)
	{
	    code_convert_dump(tmdy_struct, in, out, outsiz - 1, "ASCII");
	    return;
	}

	if(strcasecmp(ocode, "1251") == 0)
	{
	    code_convert_cp1251(tmdy_struct, in, out, outsiz - 1);
	    return;
	}
    }

#if defined(JAPANESE)
    code_convert_japan(tmdy_struct, in, out, outsiz - 1, icode, ocode);
#else
    code_convert_dump(tmdy_struct, in, out, outsiz - 1, ocode);
#endif
}

/* EAW -- insert stuff from playlist files
 *
 * Tue Apr 6 1999: Modified by Masanao Izumo <mo@goice.co.jp>
 *                 One pass implemented.
 */
static char **expand_file_lists(tmdy_struct_ex_t * tmdy_struct, char **files, int *nfiles_in_out)
{
    int nfiles;
    int i;
    char input_line[256];
    char *pfile;
    static const char *testext = ".m3u.pls.asx.M3U.PLS.ASX";
    struct timidity_file *list_file;
    char *one_file[1];
    int one;

/* Recusive global *
    static StringTable e_f_l_st;
    static int error_outflag = 0;
    static int depth = 0;
*/
    if(TMDY_COMMON->depth >= 16)
    {
	if(!TMDY_COMMON->error_outflag)
	{
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "Probable loop in playlist files");
	    TMDY_COMMON->error_outflag = 1;
	}
	return NULL;
    }

    if(TMDY_COMMON->depth == 0)
    {
	TMDY_COMMON->error_outflag = 0;
	TMDY_UTILS->strtab->init_string_table(tmdy_struct, &TMDY_COMMON->e_f_l_st);
    }
    nfiles = *nfiles_in_out;

    /* Expand playlist recursively */
    for(i = 0; i < nfiles; i++)
    {
	/* extract the file extension */
	pfile = strrchr(files[i], '.');

	if(*files[i] == '@' || (pfile != NULL && strstr(testext, pfile)))
	{
	    /* Playlist file */
            if(*files[i] == '@')
		list_file = open_file(tmdy_struct, files[i] + 1, 1, 1);
            else
		list_file = open_file(tmdy_struct, files[i], 1, 1);
            if(list_file)
	    {
                while(tf_gets(tmdy_struct, input_line, sizeof(input_line), list_file)
		      != NULL) {
            	    if(*input_line == '\n' || *input_line == '\r')
			continue;
		    if((pfile = strchr(input_line, '\r')))
		    	*pfile = '\0';
		    if((pfile = strchr(input_line, '\n')))
		    	*pfile = '\0';
		    one_file[0] = input_line;
		    one = 1;
		    TMDY_COMMON->depth++;
		    expand_file_lists(tmdy_struct, one_file, &one);
		    TMDY_COMMON->depth--;
		}
                close_file(tmdy_struct, list_file);
            }
        }
	else /* Other file */
	    TMDY_UTILS->strtab->put_string_table(tmdy_struct, &TMDY_COMMON->e_f_l_st, files[i], strlen(files[i]));
    }

    if(TMDY_COMMON->depth){
		return NULL;
	}
    *nfiles_in_out = TMDY_COMMON->e_f_l_st.nstring;
    return TMDY_UTILS->strtab->make_string_array(tmdy_struct, &TMDY_COMMON->e_f_l_st);
}

char **expand_file_archives(tmdy_struct_ex_t * tmdy_struct, char **files, int *nfiles_in_out)
{
    int nfiles;
    char **new_files;
    int    new_nfiles;

    /* First, expand playlist files */
    nfiles = *nfiles_in_out;
    files = expand_file_lists(tmdy_struct, files, &nfiles);
    if(files == NULL)
      {
	*nfiles_in_out = 0;
	return NULL;
      }

    /* Second, expand archive files */
    new_nfiles = nfiles;
    (TMDY_COMMON->open_file_noise_mode) = OF_NORMAL;
    new_files = TMDY_ARC->arc->expand_archive_names(tmdy_struct, &new_nfiles, files);
    free(files[0]);
    free(files);

    *nfiles_in_out = new_nfiles;
    return new_files;
}

#ifdef RAND_MAX
int int_rand(tmdy_struct_ex_t * tmdy_struct, int n)
{
    if(n < 0)
    {
	if(n == -1)
	    srand(time(NULL));
	else
	    srand(-n);
	return n;
    }
    return (int)(n * (double)rand() * (1.0 / (RAND_MAX + 1.0)));
}
#else
int int_rand(tmdy_struct_ex_t * tmdy_struct, int n)
{
    static unsigned long rnd_seed = 0xabcd0123;

    if(n < 0)
    {
	if(n == -1)
	    rnd_seed = time(NULL);
	else
	    rnd_seed = -n;
	return n;
    }

    rnd_seed *= 69069UL;
    return (int)(n * (double)(rnd_seed & 0xffffffff) *
		 (1.0 / (0xffffffff + 1.0)));
}
#endif /* RAND_MAX */

int check_file_extension(tmdy_struct_ex_t * tmdy_struct, char *filename, char *ext, int decompress)
{
    int len, elen, i;
#if defined(DECOMPRESSOR_LIST)
    char *dlist[] = DECOMPRESSOR_LIST;
#endif /* DECOMPRESSOR_LIST */

    len = strlen(filename);
    elen = strlen(ext);
    if(len > elen && strncasecmp(filename + len - elen, ext, elen) == 0)
	return 1;

    if(decompress)
    {
	/* Check gzip'ed file name */

	if(len > 3 + elen &&
	   strncasecmp(filename + len - elen - 3 , ext, elen) == 0 &&
	   strncasecmp(filename + len - 3, ".gz", 3) == 0)
	    return 1;

#if defined(DECOMPRESSOR_LIST)
	for(i = 0; dlist[i]; i += 2)
	{
	    int dlen;

	    dlen = strlen(dlist[i]);
	    if(len > dlen + elen &&
	       strncasecmp(filename + len - elen - dlen , ext, elen) == 0 &&
	       strncasecmp(filename + len - dlen, dlist[i], dlen) == 0)
		return 1;
	}
#endif /* DECOMPRESSOR_LIST */
    }
    return 0;
}

void randomize_string_list(tmdy_struct_ex_t * tmdy_struct, char **strlist, int n)
{
    int i, j;
    char *tmp;
    for(i = 0; i < n; i++)
    {
    j = int_rand(tmdy_struct, n - i);
	tmp = strlist[j];
	strlist[j] = strlist[n - i - 1];
	strlist[n - i - 1] = tmp;
    }
}

int pathcmp(tmdy_struct_ex_t * tmdy_struct, const char *p1, const char *p2, int ignore_case)
{
    int c1, c2;

#ifdef __W32__
    ignore_case = 1;	/* Always ignore the case */
#endif

    do {
	c1 = *p1++ & 0xff;
	c2 = *p2++ & 0xff;
	if(ignore_case)
	{
	    c1 = tolower(c1);
	    c2 = tolower(c2);
	}
	if(IS_PATH_SEP(c1)) c1 = *p1 ? 0x100 : 0;
	if(IS_PATH_SEP(c2)) c2 = *p2 ? 0x100 : 0;
    } while(c1 == c2 && c1 /* && c2 */);

    return c1 - c2;
}

static int pathcmp_qsort(tmdy_struct_ex_t * tmdy_struct, const char **p1,
			 const char **p2)
{
    return pathcmp(tmdy_struct, *(const char **)p1, *(const char **)p2, 1);
}

void sort_pathname(tmdy_struct_ex_t * tmdy_struct, char **files, int nfiles)
{
    qsort(files, nfiles, sizeof(char *),
	  (int (*)(const void *, const void *))pathcmp_qsort);
}

char *pathsep_strchr(tmdy_struct_ex_t * tmdy_struct, const char *path)
{
#ifdef PATH_SEP2
    while(*path)
    {
        if(*path == PATH_SEP || *path == PATH_SEP2)
	    return (char *)path;
	path++;
    }
    return NULL;
#else
    return strchr(path, PATH_SEP);
#endif
}

char *pathsep_strrchr(tmdy_struct_ex_t * tmdy_struct, const char *path)
{
#ifdef PATH_SEP2
    char *last_sep = NULL;
    while(*path)
    {
        if(*path == PATH_SEP || *path == PATH_SEP2)
	    last_sep = (char *)path;
	path++;
    }
    return last_sep;
#else
    return strrchr(path, PATH_SEP);
#endif
}

int str2mID(tmdy_struct_ex_t * tmdy_struct, char *str)
{
    int val;

    if(strncasecmp(str, "gs", 2) == 0)
	val = 0x41;
    else if(strncasecmp(str, "xg", 2) == 0)
	val = 0x43;
    else if(strncasecmp(str, "gm", 2) == 0)
	val = 0x7e;
    else
    {
	int i, v;
	val = 0;
	for(i = 0; i < 2; i++)
	{
	    v = str[i];
	    if('0' <= v && v <= '9')
		v = v - '0';
	    else if('A' <= v && v <= 'F')
		v = v - 'A' + 10;
	    else if('a' <= v && v <= 'f')
		v = v - 'a' + 10;
	    else
		return 0;
	    val = (val << 4 | v);
	}
    }
    return val;
}


void ts_add_to_pathlist(tmdy_struct_ex_t * tmdy_struct, char *s){
	timidity_mutex_lock(TMDY_COMMON->busy);
	add_to_pathlist(tmdy_struct, s);
	timidity_mutex_unlock(TMDY_COMMON->busy);
}
void ts_clean_up_pathlist(tmdy_struct_ex_t * tmdy_struct){
	timidity_mutex_lock(TMDY_COMMON->busy);
	clean_up_pathlist(tmdy_struct);
	timidity_mutex_unlock(TMDY_COMMON->busy);
}
struct timidity_file *ts_open_file(tmdy_struct_ex_t * tmdy_struct, char *name,
				       int decompress, int noise_mode){
	struct timidity_file *ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_tf);
	ts_buf=open_file(tmdy_struct, name,decompress, noise_mode);
	timidity_mutex_unlock(TMDY_COMMON->busy_tf);
	return ts_buf;
}
struct timidity_file *ts_open_with_mem(tmdy_struct_ex_t * tmdy_struct, char *mem, int32 memlen,int noise_mode){
	struct timidity_file *ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_tf);
	ts_buf=open_with_mem(tmdy_struct, mem, memlen, noise_mode);
	timidity_mutex_unlock(TMDY_COMMON->busy_tf);
	return ts_buf;
}
void ts_close_file(tmdy_struct_ex_t * tmdy_struct, struct timidity_file *tf){
	timidity_mutex_lock(TMDY_COMMON->busy_tf);
	close_file(tmdy_struct, tf);
	timidity_mutex_unlock(TMDY_COMMON->busy_tf);
}
void ts_skip(tmdy_struct_ex_t * tmdy_struct, struct timidity_file *tf, size_t len){
	timidity_mutex_lock(TMDY_COMMON->busy_tf);
	skip(tmdy_struct, tf, len);
	timidity_mutex_unlock(TMDY_COMMON->busy_tf);
}
int ts_tf_getc(tmdy_struct_ex_t *tmdy_struct, struct timidity_file *tf){
        int ts_buf;
        timidity_mutex_lock(TMDY_COMMON->busy_tf);
        ts_buf=tf_getc(tmdy_struct, tf);
	timidity_mutex_unlock(TMDY_COMMON->busy_tf);
        return ts_buf;
}
char *ts_tf_gets(tmdy_struct_ex_t *tmdy_struct, char *buff, int n, struct timidity_file *tf){
	char *ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_tf);
	ts_buf=tf_gets(tmdy_struct, buff, n, tf);
	timidity_mutex_unlock(TMDY_COMMON->busy_tf);
	return ts_buf;
}
long ts_tf_read(tmdy_struct_ex_t * tmdy_struct, void *buff, int32 size, int32 nitems,
		    struct timidity_file *tf){
	long ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_tf);
	ts_buf=tf_read(tmdy_struct, buff, size, nitems,tf);
	timidity_mutex_unlock(TMDY_COMMON->busy_tf);
	return ts_buf;
}
long ts_tf_seek(tmdy_struct_ex_t * tmdy_struct, struct timidity_file *tf, long offset, int whence){
	long ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_tf);
	ts_buf=tf_seek(tmdy_struct, tf, offset, whence);
	timidity_mutex_unlock(TMDY_COMMON->busy_tf);
	return ts_buf;
}
long ts_tf_tell(tmdy_struct_ex_t * tmdy_struct, struct timidity_file *tf){
	long ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_tf);
	ts_buf=tf_tell(tmdy_struct, tf);
	timidity_mutex_unlock(TMDY_COMMON->busy_tf);
	return ts_buf;
}
int ts_int_rand(tmdy_struct_ex_t * tmdy_struct, int n){	/* random [0..n-1] */
	int ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy);
	ts_buf=int_rand(tmdy_struct, n);
	timidity_mutex_unlock(TMDY_COMMON->busy);
	return ts_buf;
}
int ts_check_file_extension(tmdy_struct_ex_t * tmdy_struct, char *filename, char *ext, int decompress){
	int ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy);
	ts_buf=check_file_extension(tmdy_struct, filename, ext, decompress);
	timidity_mutex_unlock(TMDY_COMMON->busy);
	return ts_buf;
}
void *ts_safe_malloc(tmdy_struct_ex_t * tmdy_struct, size_t count){
	void *ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_m);
	ts_buf=safe_malloc(tmdy_struct, count);
	timidity_mutex_unlock(TMDY_COMMON->busy_m);
	return ts_buf;
}
void *ts_safe_realloc(tmdy_struct_ex_t * tmdy_struct, void *old_ptr, size_t new_size){
	void *ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_m);
	ts_buf=safe_realloc(tmdy_struct, old_ptr, new_size);
	timidity_mutex_unlock(TMDY_COMMON->busy_m);
	return ts_buf;
}
void *ts_safe_large_malloc(tmdy_struct_ex_t * tmdy_struct, size_t count){
	void *ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_m);
	ts_buf=safe_large_malloc(tmdy_struct, count);
	timidity_mutex_unlock(TMDY_COMMON->busy_m);
	return ts_buf;
}
char *ts_safe_strdup(tmdy_struct_ex_t * tmdy_struct, const char *s){
	char *ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy_m);
	ts_buf=safe_strdup(tmdy_struct, s);
	timidity_mutex_unlock(TMDY_COMMON->busy_m);
	return ts_buf;
}
void ts_free_ptr_list(tmdy_struct_ex_t * tmdy_struct, void *ptr_list, int count){
	timidity_mutex_lock(TMDY_COMMON->busy);
	free_ptr_list(tmdy_struct, ptr_list, count);
	timidity_mutex_unlock(TMDY_COMMON->busy);
}
char **ts_expand_file_archives(tmdy_struct_ex_t * tmdy_struct, char **files, int *nfiles_in_out){
	char **ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy);
	ts_buf=expand_file_archives(tmdy_struct, files, nfiles_in_out);
	timidity_mutex_unlock(TMDY_COMMON->busy);
	return ts_buf;
}
void ts_randomize_string_list(tmdy_struct_ex_t * tmdy_struct, char **strlist, int nstr){
	timidity_mutex_lock(TMDY_COMMON->busy);
	randomize_string_list(tmdy_struct, strlist, nstr);
	timidity_mutex_unlock(TMDY_COMMON->busy);
}
int ts_pathcmp(tmdy_struct_ex_t * tmdy_struct, const char *path1, const char *path2, int ignore_case){
	int ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy);
	ts_buf=pathcmp(tmdy_struct, path1, path2, ignore_case);
	timidity_mutex_unlock(TMDY_COMMON->busy);
	return ts_buf;
}
void ts_sort_pathname(tmdy_struct_ex_t * tmdy_struct, char **files, int nfiles){
	timidity_mutex_lock(TMDY_COMMON->busy);
	sort_pathname(tmdy_struct, files, nfiles);
	timidity_mutex_unlock(TMDY_COMMON->busy);
}
int  ts_load_table(tmdy_struct_ex_t * tmdy_struct, char *file){
	int  ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy);
	ts_buf=load_table(tmdy_struct, file);
	timidity_mutex_unlock(TMDY_COMMON->busy);
	return ts_buf;
}
char *ts_pathsep_strrchr(tmdy_struct_ex_t * tmdy_struct, const char *path){
	char *ts_buf;
	ts_buf=pathsep_strrchr(tmdy_struct, path);
	return ts_buf;
}
char *ts_pathsep_strchr(tmdy_struct_ex_t * tmdy_struct, const char *path){
	char *ts_buf;
	ts_buf=pathsep_strchr(tmdy_struct, path);
	return ts_buf;
}
int ts_str2mID(tmdy_struct_ex_t * tmdy_struct, char *str){
	int ts_buf;
	timidity_mutex_lock(TMDY_COMMON->busy);
	ts_buf=str2mID(tmdy_struct, str);
	timidity_mutex_unlock(TMDY_COMMON->busy);
	return ts_buf;
}
void ts_code_convert(tmdy_struct_ex_t * tmdy_struct, char *in, char *out, int outsiz,
			 char *in_code, char *out_code){
	timidity_mutex_lock(TMDY_COMMON->busy);
	code_convert(tmdy_struct, in, out, outsiz, in_code, out_code);
	timidity_mutex_unlock(TMDY_COMMON->busy);
}
void ts_safe_exit(tmdy_struct_ex_t * tmdy_struct, int status){
	timidity_mutex_lock(TMDY_COMMON->busy_m);
	safe_exit(tmdy_struct, status);
	timidity_mutex_unlock(TMDY_COMMON->busy_m);
}


common_ex_t* new_common(tmdy_struct_ex_t *tmdy_struct){

	common_ex_t* common_ex;

	common_ex=(common_ex_t *)malloc(sizeof(common_ex_t));

	timidity_mutex_init(common_ex->busy);
	timidity_mutex_init(common_ex->busy_m);
	timidity_mutex_init(common_ex->busy_tf);
	
common_ex->note_name=note_name;
common_ex->timidity_version=timidity_version;
common_ex->output_text_code = NULL;

common_ex->add_to_pathlist=ts_add_to_pathlist;
common_ex->clean_up_pathlist=ts_clean_up_pathlist;
common_ex->open_file=ts_open_file;
common_ex->open_with_mem=ts_open_with_mem;
common_ex->close_file=ts_close_file;
common_ex->skip=ts_skip;
common_ex->tf_getc=ts_tf_getc;
common_ex->tf_gets=ts_tf_gets;
common_ex->tf_read=ts_tf_read;
common_ex->tf_seek=ts_tf_seek;
common_ex->tf_tell=ts_tf_tell;
common_ex->int_rand=ts_int_rand;	/* random [0..n-1] */
common_ex->check_file_extension=ts_check_file_extension;

common_ex->safe_malloc=ts_safe_malloc;
common_ex->safe_realloc=ts_safe_realloc;
common_ex->safe_large_malloc=ts_safe_large_malloc;
common_ex->safe_strdup=ts_safe_strdup;
common_ex->free_ptr_list=ts_free_ptr_list;
common_ex->expand_file_archives=ts_expand_file_archives;
common_ex->randomize_string_list=ts_randomize_string_list;
common_ex->pathcmp=ts_pathcmp;
common_ex->sort_pathname=ts_sort_pathname;
common_ex-> load_table=ts_load_table;
common_ex->pathsep_strrchr=ts_pathsep_strrchr;
common_ex->pathsep_strchr=ts_pathsep_strchr;
common_ex->str2mID=ts_str2mID;

common_ex->code_convert=ts_code_convert;

common_ex->safe_exit=ts_safe_exit;

common_ex->open_file_noise_mode = OF_NORMAL;

#ifdef DEFAULT_PATH
    /* The paths in this list will be tried whenever we're reading a file */
common_ex->defaultpathlist.path=DEFAULT_PATH;
common_ex->defaultpathlist.next=0;
	
common_ex->pathlist=&(common_ex->defaultpathlist); /* This is a linked list */
#else
common_ex->PathList *pathlist=0;
#endif

common_ex->safe_malloc_errflag = 0;
common_ex->safe_large_malloc_errflag = 0;
common_ex->safe_realloc_errflag = 0;
common_ex->safe_strdup_errflag = 0;
common_ex->mode = NULL;
common_ex->wrd_mode = NULL;
common_ex->error_outflag = 0;
common_ex->depth = 0;
	
	
TMDY_UTILS->mblock->init_mblock(tmdy_struct, &(common_ex->tmpbuffer));
	return common_ex;
}
void destroy_common(common_ex_t* common){	
	timidity_mutex_destroy(common->busy);
	timidity_mutex_destroy(common->busy_m);
	timidity_mutex_destroy(common->busy_tf);
	free(common);
}
