/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   instrum.h

   */

#ifndef ___INSTRUM_H_
#define ___INSTRUM_H_

typedef struct s_Sample {
  splen_t
    loop_start, loop_end, data_length;
  int32
    sample_rate, low_freq, high_freq, root_freq;
  int8 panning, note_to_use;
  int32
    envelope_rate[6], envelope_offset[6],
	modenv_rate[6], modenv_offset[6];
  FLOAT_T
    volume;
  sample_t
    *data;
  int32
    tremolo_sweep_increment, tremolo_phase_increment,
    vibrato_sweep_increment, vibrato_control_ratio;
  int16
    tremolo_depth;
  int16 vibrato_depth;
  uint8
    modes, data_alloced,
    low_vel, high_vel;
  int32 cutoff_freq;	/* in Hz, [1, 20000] */
  int16 resonance;	/* in centibels, [0, 960] */
  /* in cents, [-12000, 12000] */
  int16 tremolo_to_pitch, tremolo_to_fc, modenv_to_pitch, modenv_to_fc,
	  envelope_keyf[6], envelope_velf[6], modenv_keyf[6], modenv_velf[6],
	  vel_to_fc, key_to_fc;
  int16 vel_to_resonance;	/* in centibels, [-960, 960] */
  int8 envelope_velf_bpo, modenv_velf_bpo,
	  key_to_fc_bpo, vel_to_fc_threshold;	/* in notes */
  int32 vibrato_delay, tremolo_delay, envelope_delay, modenv_delay;	/* in samples */
  int16 scale_tuning;	/* in cents/key */
  int8 inst_type;
} Sample;

/* Bits in modes: */
#define MODES_16BIT	(1<<0)
#define MODES_UNSIGNED	(1<<1)
#define MODES_LOOPING	(1<<2)
#define MODES_PINGPONG	(1<<3)
#define MODES_REVERSE	(1<<4)
#define MODES_SUSTAIN	(1<<5)
#define MODES_ENVELOPE	(1<<6)
#define MODES_CLAMPED	(1<<7) /* ?? (for last envelope??) */

#define INST_GUS	0
#define INST_SF2	1
#define INST_MOD	2
#define INST_PCM	3	/* %sample */

typedef struct {
  int type;
  int samples;
  Sample *sample;
  char *instname;
} Instrument;

typedef struct {
  char *name;
  char *comment;
  Instrument *instrument;
  int8 note, pan, strip_loop, strip_envelope, strip_tail, loop_timeout,
	font_preset,font_keynote,legato,tva_level;
  uint8 font_bank;
  uint8 instype; /* 0: Normal
		    1: %font
		    2: %sample
		    3-255: reserved
		    */
  int16 amp,cutoff_freq,resonance;
  int tunenum;
  float *tune;
	int envratenum, envofsnum;
	int **envrate, **envofs;
	int tremnum, vibnum;
	struct Quantity_ **trem, **vib;
} ToneBankElement;

/* A hack to delay instrument loading until after reading the
   entire MIDI file. */
#define MAGIC_LOAD_INSTRUMENT ((Instrument *)(-1))
#define MAGIC_ERROR_INSTRUMENT ((Instrument *)(-2))
#define IS_MAGIC_INSTRUMENT(ip) ((ip) == MAGIC_LOAD_INSTRUMENT || (ip) == MAGIC_ERROR_INSTRUMENT)

typedef struct s_AlternateAssign {
    /* 128 bit vector:
     * bits[(note >> 5) & 0x3] & (1 << (note & 0x1F))
     */
    uint32 bits[4];
    struct s_AlternateAssign* next;
} AlternateAssign;

typedef struct {
  ToneBankElement tone[128];
  AlternateAssign *alt;
} ToneBank;

typedef struct s_SpecialPatch /* To be used MIDI Module play mode */
{
    int type;
    int samples;
    Sample *sample;
    char *name;
    int32 sample_offset;
} SpecialPatch;

enum instrument_mapID
{
    INST_NO_MAP = 0,
    SC_55_TONE_MAP,
    SC_55_DRUM_MAP,
    SC_88_TONE_MAP,
    SC_88_DRUM_MAP,
    SC_88PRO_TONE_MAP,
    SC_88PRO_DRUM_MAP,
    XG_NORMAL_MAP,
    XG_SFX64_MAP,
    XG_SFX126_MAP,
    XG_DRUM_MAP,
    NUM_INST_MAP
};
#define NSPECIAL_PATCH 256

#define MAX_MREL 5000
#define DEFAULT_MREL 800

#define SPECIAL_PROGRAM -1

/*instrum.c local structs definition*/
#define INSTRUMENT_HASH_SIZE 128
struct InstrumentCache
{
    char *name;
    int panning, amp, note_to_use, strip_loop, strip_envelope, strip_tail;
    Instrument *ip;
    struct InstrumentCache *next;
};
struct inst_map_elem
{
    int set, elem, mapped;
};
/*sndfont.c local structs definition*/
typedef struct s_SFPatchRec {
	int preset, bank, keynote; /* -1 = matches all */
} SFPatchRec;
typedef struct s_SampleList {
	Sample v;
	struct s_SampleList *next;
	int32 start;
	int32 len;
	int32 cutoff_freq;
	int16 resonance;
	int16 root, tune;
	char low, high;		/* key note range */
	int8 reverb_send,chorus_send;
	/* Depend on (TMDY_OUTPUT->play_mode)->rate */
	int32 vibrato_freq;
	int32 attack;
	int32 hold;
	int32 sustain;
	int32 decay;
	int32 release;

	int32 modattack;
	int32 modhold;
	int32 modsustain;
	int32 moddecay;
	int32 modrelease;
} SampleList;
typedef struct s_InstList {
	SFPatchRec pat;
	int pr_idx;
	int samples;
	int order;
	SampleList *slist;
	struct s_InstList *next;
} InstList;
typedef struct s_SFExclude {
	SFPatchRec pat;
	struct s_SFExclude *next;
} SFExclude;
typedef struct s_SFOrder {
	SFPatchRec pat;
	int order;
	struct s_SFOrder *next;
} SFOrder;
#define INSTHASHSIZE 127
typedef struct s_SFInsts {
	struct timidity_file *tf;
	char *fname;
	int8 def_order, def_cutoff_allowed, def_resonance_allowed;
	uint16 version, minorversion;
	int32 samplepos, samplesize;
	InstList *instlist[INSTHASHSIZE];
	char **inst_namebuf;
	SFExclude *sfexclude;
	SFOrder *sforder;
	struct s_SFInsts *next;
	FLOAT_T amptune;
	MBlockList pool;
} SFInsts;
#ifdef CFG_FOR_SF
typedef struct x_cfg_info_t_ {
	char m_bank[128][128];
	char m_preset[128][128];
	char m_rom[128][128];
	char *m_str[128][128];
	char d_preset[128][128];
	char d_keynote[128][128];
	char d_rom[128][128];
	char *d_str[128][128];
} x_cfg_info_t;
#endif
/* from smplfile.c local structs definition*/
typedef int (*SampleImporterDiscriminateProc)(tmdy_struct_ex_t *tmdy_struct, char *sample_file);
	/* returns 0 if file may be loadable */
typedef int (*SampleImporterSampleLoaderProc)(tmdy_struct_ex_t *tmdy_struct, char *sample_file, Instrument *inst);
	/* set inst->samples, inst->sample and returns 0 if loaded */
	/* inst is pre-allocated, and is freed by caller if loading failed */
	/* -1 to let caller give up testing other importers */

typedef struct {
	char				*extension;		/* file extension excluding '.' */
	SampleImporterDiscriminateProc	discriminant;
	SampleImporterSampleLoaderProc	load;
	/* either extension or discriminant may be NULL */
	int					added;			/* for get_importers()'s internal use */
} SampleImporter, *SampleImporterRef;

/**** main structure ****/


struct s_instrum_ex_t {
	timidity_mutex_t  busy;
	
	ToneBank
 	 	*tonebank[128],
  		*drumset[128];
	Instrument *default_instrument;
	SpecialPatch *special_patch[NSPECIAL_PATCH];

	int default_program[MAX_CHANNELS];
	int antialiasing_allowed;
	int fast_decay;
	int free_instruments_afterwards;
	int cutoff_allowed;
/* sndfont.c */
	void (*add_soundfont)(tmdy_struct_ex_t *tmdy_struct, char *sf_file, int sf_order,
			  int cutoff_allowed, int resonance_allowed,
			  int amp);
	void (*remove_soundfont)(tmdy_struct_ex_t *tmdy_struct, char *sf_file);
	void (*init_load_soundfont)(tmdy_struct_ex_t *tmdy_struct);
	Instrument *(*load_soundfont_inst)(tmdy_struct_ex_t *tmdy_struct, int order, int bank, int preset,
				       int keynote);
	Instrument *(*extract_soundfont)(tmdy_struct_ex_t *tmdy_struct, char *sf_file, int bank, int preset,
				     int keynote);
	int (*exclude_soundfont)(tmdy_struct_ex_t *tmdy_struct, int bank, int preset, int keynote);
	int (*order_soundfont)(tmdy_struct_ex_t *tmdy_struct, int bank, int preset, int keynote, int order);
	char *(*soundfont_preset_name)(tmdy_struct_ex_t *tmdy_struct, int bank, int preset, int keynote,
				   char **sndfile);
//	void (*free_soundfont_inst)(tmdy_struct_ex_t *tmdy_struct);	

/* instrum.c */
	int (*load_missing_instruments)(tmdy_struct_ex_t *tmdy_struct, int *rc);
	void (*free_instruments)(tmdy_struct_ex_t *tmdy_struct, int reload_default_inst);
	void (*free_special_patch)(tmdy_struct_ex_t *tmdy_struct, int id);
	int (*set_default_instrument)(tmdy_struct_ex_t *tmdy_struct, char *name);
	void (*clear_magic_instruments)(tmdy_struct_ex_t *tmdy_struct);
	Instrument *(*load_instrument)(tmdy_struct_ex_t *tmdy_struct, int dr, int b, int prog);
	void (*alloc_instrument_bank)(tmdy_struct_ex_t *tmdy_struct, int dr, int bankset);
	int (*instrument_map)(tmdy_struct_ex_t *tmdy_struct, int mapID, int *set_in_out, int *elem_in_out);
	void (*set_instrument_map)(tmdy_struct_ex_t *tmdy_struct, int mapID,
			       int set_from, int elem_from,
			       int set_to, int elem_to);
	void (*free_instrument_map)(tmdy_struct_ex_t *tmdy_struct);
	AlternateAssign *(*add_altassign_string)(tmdy_struct_ex_t *tmdy_struct, AlternateAssign *old,
					     char **params, int n);
	AlternateAssign *(*find_altassign)(tmdy_struct_ex_t *tmdy_struct, AlternateAssign *altassign, int note);
	void (*free_instrument)(tmdy_struct_ex_t *tmdy_struct, Instrument *ip);

	char *default_instrument_name;
	int progbase;
	int32 modify_release;

	/* from smplfile.c */
	Instrument *(*extract_sample_file)(tmdy_struct_ex_t* tmdy_struct, char *);

	
	/**** local variables *****/
	/*instrum.c */
	struct InstrumentCache *instrument_cache[INSTRUMENT_HASH_SIZE];
	ToneBank standard_tonebank, standard_drumset;
	struct inst_map_elem *inst_map_table[NUM_INST_MAP][128];
	char *last_name;
	/* sndfont. */
	SFInsts *sfrecs;
	SFInsts *current_sfrec;
#ifdef CFG_FOR_SF
	FILE *x_out;
	char x_sf_file_name[1024];
	int x_pre_bank;
	int x_pre_preset;
	int x_sort;
	x_cfg_info_t x_cfg_info;
	int x_cfg_info_init_flag;
	struct URL_module *(*url_module_list);
#endif
	/* from smplfile.c */
	SampleImporter	sample_importers[3];
};



extern instrum_ex_t* instrum_ex;

instrum_ex_t* new_instrum(tmdy_struct_ex_t *tmdy_struct);
void destroy_instrum(instrum_ex_t* instrum);

extern void new_init_sndfont(instrum_ex_t* instrum_ex);
extern void new_init_smplefile(instrum_ex_t* instrum_ex);
	
	
#endif /* ___INSTRUM_H_ */
