/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <stdlib.h>

#include "timidity.h"
#include "common.h"
#include "arc.h"
#include "arc_prv.h"
#include "strtab.h"
#include "zip.h"
#include "zip_prv.h"
#include "unlzh.h"
#include "unlzh_prv.h"
#include "explode.h"
#include "explode_prv.h"


char *arc_lib_version = ARC_LIB_VERSION;

#define GZIP_ASCIIFLAG		(1u<<0)
#define GZIP_MULTIPARTFLAG	(1u<<1)
#define GZIP_EXTRAFLAG		(1u<<2)
#define GZIP_FILEFLAG		(1u<<3)
#define GZIP_COMMFLAG		(1u<<4)
#define GZIP_ENCFLAG		(1u<<5)

#ifndef TRUE
#define TRUE			1
#endif /* TRUE */
#ifndef FALSE
#define FALSE			0
#endif /* FALSE */
#define ABORT			-1

ArchiveHandler arc_handler;
static MBlockList arc_buffer;

typedef struct s_ArchiveFileList
{
    char *archive_name;
    ArchiveEntryNode *entry_list;
    struct s_ArchiveFileList *next;
} ArchiveFileList;
static ArchiveFileList *arc_filelist = NULL;

static struct
{
    char *ext;
    int type;
} archive_ext_list[] =
{
    {".tar",	ARCHIVE_TAR},
    {".tar.gz",	ARCHIVE_TGZ},
    {".tgz",	ARCHIVE_TGZ},
    {".zip",	ARCHIVE_ZIP},
    {".neo",	ARCHIVE_ZIP},
    {".lzh",	ARCHIVE_LZH},
    {".lha",	ARCHIVE_LZH},
    {".mime",	ARCHIVE_MIME},
    {PATH_STRING, ARCHIVE_DIR},
    {NULL, -1}
};

int skip_gzip_header(tmdy_struct_ex_t *tmdy_struct, URL url)
{
    unsigned char flags;
    int m1, method;

    /* magic */
    m1 = TMDY_ARC->url->url_getc(tmdy_struct, url);
    if(m1 == 0)
    {
	url_skip(tmdy_struct, url, 128 - 1);
	m1 = TMDY_ARC->url->url_getc(tmdy_struct, url);
    }
    if(m1 != 0x1f)
	return -1;
    if(TMDY_ARC->url->url_getc(tmdy_struct, url) != 0x8b)
	return -1;

    /* method */
    method = TMDY_ARC->url->url_getc(tmdy_struct, url);
    switch(method)
    {
      case 8:			/* deflated */
	method = ARCHIVEC_DEFLATED;
	break;
      default:
	return -1;
    }
    /* flags */
    flags = TMDY_ARC->url->url_getc(tmdy_struct, url);
    if(flags & GZIP_ENCFLAG)
	return -1;
    /* time */
    TMDY_ARC->url->url_getc(tmdy_struct, url); TMDY_ARC->url->url_getc(tmdy_struct, url); TMDY_ARC->url->url_getc(tmdy_struct, url); TMDY_ARC->url->url_getc(tmdy_struct, url);

    TMDY_ARC->url->url_getc(tmdy_struct, url);		/* extra flags */
    TMDY_ARC->url->url_getc(tmdy_struct, url);		/* OS type */

    if(flags & GZIP_MULTIPARTFLAG)
    {
	/* part number */
	TMDY_ARC->url->url_getc(tmdy_struct, url); TMDY_ARC->url->url_getc(tmdy_struct, url);
    }

    if(flags & GZIP_EXTRAFLAG)
    {
	unsigned short len;
	int i;

	/* extra field */

	len = TMDY_ARC->url->url_getc(tmdy_struct, url);
	len |= ((unsigned short)TMDY_ARC->url->url_getc(tmdy_struct, url)) << 8;
	for(i = 0; i < len; i++)
	    TMDY_ARC->url->url_getc(tmdy_struct, url);
    }

    if(flags & GZIP_FILEFLAG)
    {
	/* file name */
	int c;

	do
	{
	    c = TMDY_ARC->url->url_getc(tmdy_struct, url);
	    if(c == EOF)
		return -1;
	} while(c != '\0');
    }

    if(flags & GZIP_COMMFLAG)
    {
	/* comment */
	int c;

	do
	{
	    c = TMDY_ARC->url->url_getc(tmdy_struct, url);
	    if(c == EOF)
		return -1;
	} while(c != '\0');
    }

    return method;
}

int parse_gzip_header_bytes(tmdy_struct_ex_t *tmdy_struct, char *gz, long maxparse, int *hdrsiz)
{
    URL url = TMDY_ARC->url->url_mem_open(tmdy_struct, gz, maxparse, 0);
    int method;

    if(!url)
	return -1;
    method = skip_gzip_header(tmdy_struct, url);
    *hdrsiz = url_tell(tmdy_struct, url);
    url_close(tmdy_struct, url);
    return method;
}

//void (* arc_error_handler)(tmdy_struct_ex_t *tmdy_struct,char *error_message) = NULL;
    
static void arc_cant_open(tmdy_struct_ex_t *tmdy_struct,char *s)
{
    if(TMDY_ARC->arc->arc_error_handler != NULL)
    {
	char buff[BUFSIZ];
	snprintf(buff, sizeof(buff), "%s: Can't open", s);
	TMDY_ARC->arc->arc_error_handler(tmdy_struct,buff);
    }
}

int get_archive_type(tmdy_struct_ex_t *tmdy_struct, char *archive_name)
{
    int i, len;
    char *p;
    int archive_name_length, delim;

#ifdef SUPPORT_SOCKET
    int type = TMDY_ARC->url->url_check_type(tmdy_struct, archive_name);
    if(type == URL_news_t)
	return ARCHIVE_MIME;
    if(type == URL_newsgroup_t)
	return ARCHIVE_NEWSGROUP;
#endif /* SUPPORT_SOCKET */

    if(strncmp(archive_name, "mail:", 5) == 0 ||
       strncmp(archive_name, "mime:", 5) == 0)
	return ARCHIVE_MIME;

    if((p = strrchr(archive_name, '#')) != NULL)
    {
	archive_name_length = p - archive_name;
	delim = '#';
    }
    else
    {
	archive_name_length = strlen(archive_name);
	delim = '\0';
    }

    for(i = 0; archive_ext_list[i].ext; i++)
    {
	len = strlen(archive_ext_list[i].ext);
	if(len <= archive_name_length &&
	   strncasecmp(archive_name + archive_name_length - len,
		       archive_ext_list[i].ext, len) == 0 &&
	   archive_name[archive_name_length] == delim)
	    return archive_ext_list[i].type; /* Found */
    }

    if(TMDY_ARC->url->url_check_type(tmdy_struct, archive_name) == URL_dir_t)
	return ARCHIVE_DIR;

    return -1;			/* Not found */
}

static ArchiveFileList *find_arc_filelist(tmdy_struct_ex_t *tmdy_struct, char *basename)
{
    ArchiveFileList *p;

    for(p = arc_filelist; p; p = p->next)
    {
	if(strcmp(basename, p->archive_name) == 0)
	    return p;
    }
    return NULL;
}

ArchiveEntryNode *arc_parse_entry(tmdy_struct_ex_t *tmdy_struct, URL url, int archive_type)
{
    ArchiveEntryNode *entry_first, *entry_last, *entry;
    ArchiveEntryNode *(* next_header_entry)(tmdy_struct_ex_t *tmdy_struct);
    int gzip_method;
    URL orig;

    orig = NULL;
    switch(archive_type)
    {
      case ARCHIVE_TAR:
	next_header_entry = next_tar_entry;
	break;
      case ARCHIVE_TGZ:
	gzip_method = skip_gzip_header(tmdy_struct, url);
	if(gzip_method != ARCHIVEC_DEFLATED)
	{
	    url_close(tmdy_struct, url);
	    return NULL;
	}
	orig = url;
	if((url = TMDY_ARC->url->url_inflate_open(tmdy_struct, orig, -1, 0)) == NULL)
	    return NULL;
	next_header_entry = next_tar_entry;
	break;
      case ARCHIVE_ZIP:
	next_header_entry = next_zip_entry;
	break;
      case ARCHIVE_LZH:
	next_header_entry = next_lzh_entry;
	break;
      case ARCHIVE_MIME:
	if(!IS_URL_SEEK_SAFE(url))
	{
	    orig = url;
	    if((url = TMDY_ARC->url->url_cache_open(tmdy_struct, orig, 0)) == NULL)
		return NULL;
	}
	next_header_entry = next_mime_entry;
	break;
      default:
	return NULL;
    }

    arc_handler.isfile = (url->type == URL_file_t);
    arc_handler.url = url;
    arc_handler.counter = 0;
    entry_first = entry_last = NULL;
    arc_handler.pos = 0;
    while((entry = next_header_entry(tmdy_struct)) != NULL)
    {
	if(entry_first != NULL)
	    entry_last->next = entry;
	else
	    entry_first = entry_last = entry;
	while(entry_last->next)
	    entry_last = entry_last->next;
	arc_handler.counter++;
    }
    url_close(tmdy_struct, url);
    if(orig)
	url_close(tmdy_struct, orig);
    return entry_first;		/* Note that NULL if no archive file */
}

static ArchiveFileList *add_arc_filelist(tmdy_struct_ex_t *tmdy_struct, char *basename, int archive_type)
{
    URL url;
    ArchiveFileList *afl;
    ArchiveEntryNode *entry;

    switch(archive_type)
    {
      case ARCHIVE_TAR:
      case ARCHIVE_TGZ:
      case ARCHIVE_ZIP:
      case ARCHIVE_LZH:
      case ARCHIVE_MIME:
	break;
      default:
	return NULL;
    }

    if((url = TMDY_ARC->url->url_open(tmdy_struct, basename)) == NULL)
    {
	arc_cant_open(tmdy_struct,basename);
	return NULL;
    }

    entry = arc_parse_entry(tmdy_struct,url, archive_type);

    afl = (ArchiveFileList *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(ArchiveFileList));
    afl->archive_name = TMDY_COMMON->safe_strdup(tmdy_struct, basename);
    afl->entry_list = entry;
    afl->next = arc_filelist;
    arc_filelist = afl;
    return afl;
}

static ArchiveFileList *regist_archive(tmdy_struct_ex_t *tmdy_struct, char *archive_filename)
{
    int archive_type;

    if((archive_type = get_archive_type(tmdy_struct,archive_filename)) < 0)
	return NULL;		/* Unknown archive */
    archive_filename = TMDY_ARC->url->url_expand_home_dir(tmdy_struct, archive_filename);
    if(find_arc_filelist(tmdy_struct,archive_filename))
	return NULL;		/* Already registerd */
    return add_arc_filelist(tmdy_struct,archive_filename, archive_type);
}

static int arc_expand_newfile(tmdy_struct_ex_t *tmdy_struct, StringTable *s, ArchiveFileList *afl,
			      char *pattern)
{
    ArchiveEntryNode *entry;
    char *p;

    for(entry = afl->entry_list; entry; entry = entry->next)
    {
	if(arc_case_wildmat(tmdy_struct, entry->name, pattern))
	{
	    p = TMDY_UTILS->mblock->new_segment(tmdy_struct, &arc_buffer, strlen(afl->archive_name) +
			    strlen(entry->name) + 2);
	    strcpy(p, afl->archive_name);
	    strcat(p, "#");
	    strcat(p, entry->name);
	    if(TMDY_UTILS->strtab->put_string_table(tmdy_struct, s, p, strlen(p)) == NULL)
		return -1;
	}
    }
    return 0;
}

char **expand_archive_names(tmdy_struct_ex_t *tmdy_struct, int *nfiles_in_out, char **files)
{
    int i, nfiles, arc_type;
    char *infile_name;
    char *base, *pattern, *p, buff[BUFSIZ];
    char *one_file[1];
    int one;
    ArchiveFileList *afl;

    /* Recusive global */
    static MBlockList *pool;
    static StringTable stab;
    static int error_flag = 0;
    static int depth = 0;

    if(depth == 0)
    {
	error_flag = 0;
	TMDY_UTILS->strtab->init_string_table(tmdy_struct, &stab);
	pool = &arc_buffer;
    }

    nfiles = *nfiles_in_out;

    for(i = 0; i < nfiles; i++)
    {
	infile_name = TMDY_ARC->url->url_expand_home_dir(tmdy_struct, files[i]);
	if((p = strrchr(infile_name, '#')) == NULL)
	{
	    base = infile_name;
	    pattern = "*";
	}
	else
	{
	    int len = p - infile_name;
	    base = TMDY_UTILS->mblock->new_segment(tmdy_struct, pool, len + 1); /* +1 for '\0' */
	    memcpy(base, infile_name, len);
	    base[len] = '\0';
	    pattern = p + 1;
	}

	if((afl = find_arc_filelist(tmdy_struct,base)) != NULL)
	{
	    if(arc_expand_newfile(tmdy_struct,&stab, afl, pattern) == -1)
		goto abort_expand;
	    continue;
	}

	arc_type = get_archive_type(tmdy_struct,base);
	if(arc_type == -1)
	{
	    if(TMDY_UTILS->strtab->put_string_table(tmdy_struct, &stab, infile_name, strlen(infile_name))
	       == NULL)
		goto abort_expand;
	    continue;
	}

#ifdef SUPPORT_SOCKET
	if(arc_type == ARCHIVE_NEWSGROUP)
	{
	    URL url;
	    int len1, len2;
	    char *news_prefix;

	    if((url = TMDY_ARC->url->url_newsgroup_open(tmdy_struct, base)) == NULL)
	    {
		arc_cant_open(tmdy_struct,base);
		continue;
	    }

	    strcpy(buff, base);
	    p = strchr(buff + 7, '/') + 1; /* news://..../ */
	    *p = '\0';
	    news_prefix = TMDY_UTILS->mblock->strdup_mblock(tmdy_struct, pool, buff);
	    len1 = strlen(news_prefix);

	    while(url_gets(url, buff, sizeof(buff)))
	    {
		len2 = strlen(buff);
		p = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, pool, len1 + len2 + 1);
		strcpy(p, news_prefix);
		strcpy(p + len1, buff);
		one_file[0] = p;
		one = 1;
		depth++;
		expand_archive_names(tmdy_struct,&one, one_file);
		depth--;
	    }
	    url_close(url);
	    if(error_flag)
		goto abort_expand;
	    continue;
	}
#endif /* SUPPORT_SOCKET */

	if(arc_type == ARCHIVE_DIR)
	{
	    URL url;
	    int len1, len2;

	    if((url = TMDY_ARC->url->url_dir_open(tmdy_struct, base)) == NULL)
	    {
		arc_cant_open(tmdy_struct,base);
		continue;
	    }

	    if(strncmp(base, "dir:", 4) == 0)
		base += 4;
	    len1 = strlen(base);
	    if(IS_PATH_SEP(base[len1 - 1]))
		len1--;
	    while(url_gets(tmdy_struct, url, buff, sizeof(buff)))
	    {
		if(strcmp(buff, ".") == 0 || strcmp(buff, "..") == 0)
		    continue;

		len2 = strlen(buff);
		p = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, pool, len1 + len2 + 2);
		strcpy(p, base);
		p[len1] = PATH_SEP;
		strcpy(p + len1 + 1, buff);
		one_file[0] = p;
		one = 1;
		depth++;
		expand_archive_names(tmdy_struct,&one, one_file);
		depth--;
	    }
	    url_close(tmdy_struct, url);
	    if(error_flag)
		goto abort_expand;
	    continue;
	}

	if((afl = add_arc_filelist(tmdy_struct,base, arc_type)) != NULL)
	{
	    if(arc_expand_newfile(tmdy_struct,&stab, afl, pattern) == -1)
		goto abort_expand;
	}
    }

    if(depth)
	return NULL;
    *nfiles_in_out = stab.nstring;
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, pool);
    return TMDY_UTILS->strtab->make_string_array(tmdy_struct, &stab); /* It is possible NULL */

  abort_expand:
    error_flag = 1;
    if(depth)
	return NULL;
    TMDY_UTILS->strtab->delete_string_table(tmdy_struct, &stab);
    TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);
    *nfiles_in_out = 0;
    return NULL;
}

ArchiveEntryNode *new_entry_node(tmdy_struct_ex_t *tmdy_struct, char *name, int len)
{
    ArchiveEntryNode *entry;
    entry = (ArchiveEntryNode *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(ArchiveEntryNode));
    memset(entry, 0, sizeof(ArchiveEntryNode));
    entry->name = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, len + 1);
    memcpy(entry->name, name, len);
    entry->name[len] = '\0';
    return entry;
}

void free_entry_node(tmdy_struct_ex_t *tmdy_struct, ArchiveEntryNode *entry)
{
    free(entry->name);
    if(entry->cache != NULL)
	free(entry->cache);
    free(entry);
}


static char *compress_buff;
long   compress_buff_len;
static long arc_compress_func(tmdy_struct_ex_t *tmdy_struct, char *buff, long size, void *user_val)
{
    if(compress_buff_len <= 0)
	return 0;
    if(size > compress_buff_len)
	size = compress_buff_len;
    memcpy(buff, compress_buff, size);
    compress_buff += size;
    compress_buff_len -= size;
    return size;
}

void *arc_compress(tmdy_struct_ex_t *tmdy_struct, void *buff, long bufsiz,
		   int compress_level, long *compressed_size)
{
    DeflateHandler compressor;
    long allocated, offset, space, nbytes;
    char *compressed;

    compress_buff = (char *)buff;
    compress_buff_len = bufsiz;
    compressor = TMDY_ARC->zip->open_deflate_handler(tmdy_struct, arc_compress_func, NULL,
				      compress_level);
    allocated = 1024;
    compressed = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, allocated);
    offset = 0;
    space = allocated;
    while((nbytes = zip_deflate(tmdy_struct, compressor, compressed + offset, space)) > 0)
    {
	offset += nbytes;
	space -= nbytes;
	if(space == 0)
	{
	    space = allocated;
	    allocated += space;
	    compressed = (char *)TMDY_COMMON->safe_realloc(tmdy_struct, compressed, allocated);
	}
    }
    close_deflate_handler(tmdy_struct, compressor);
    if(offset == 0)
    {
	free(buff);
	return NULL;
    }
    *compressed_size = offset;
    return compressed;
}

void *arc_decompress(tmdy_struct_ex_t *tmdy_struct, void *buff, long bufsiz, long *decompressed_size)
{
    InflateHandler decompressor;
    long allocated, offset, space, nbytes;
    char *decompressed;

    compress_buff = (char *)buff;
    compress_buff_len = bufsiz;
    decompressor = TMDY_ARC->zip->open_inflate_handler(tmdy_struct, arc_compress_func, NULL);
    allocated = 1024;
    decompressed = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, allocated);
    offset = 0;
    space = allocated;
    while((nbytes = TMDY_ARC->zip->zip_inflate(tmdy_struct, decompressor, decompressed + offset, space)) > 0)
    {
	offset += nbytes;
	space -= nbytes;
	if(space == 0)
	{
	    space = allocated;
	    allocated += space;
	    decompressed = (char *)TMDY_COMMON->safe_realloc(tmdy_struct, decompressed, allocated);
	}
    }
    TMDY_ARC->zip->close_inflate_handler(tmdy_struct, decompressor);
    if(offset == 0)
    {
	free(buff);
	return NULL;
    }
    *decompressed_size = offset;
    return decompressed;
}

void free_archive_files(tmdy_struct_ex_t *tmdy_struct)
{
    ArchiveEntryNode *entry, *ecur;
    ArchiveFileList *acur;

    while(arc_filelist)
    {
	acur = arc_filelist;
	arc_filelist = arc_filelist->next;
	entry = acur->entry_list;
	while(entry)
	{
	    ecur = entry;
	    entry = entry->next;
	    free_entry_node(tmdy_struct, ecur);
	}
	free(acur->archive_name);
	free(acur);
    }
}

/******************************************************************************
* url_arc
*****************************************************************************/
typedef struct s_URL_arc
{
    char common[sizeof(struct s_URL)];
    URL instream;
    long pos, size;
    int comptype;
    void *decoder;
} URL_arc;

static long url_arc_read(tmdy_struct_ex_t *tmdy_struct, URL url, void *buff, long n);
static long url_arc_tell(tmdy_struct_ex_t *tmdy_struct, URL url);
static void url_arc_close(tmdy_struct_ex_t *tmdy_struct, URL url);

static long archiver_read_func(tmdy_struct_ex_t *tmdy_struct, char *buff, long buff_size, void *v)
{
    URL_arc *url;
    long n;

    url = (URL_arc *)v;

    if(url->size < 0)
	n = buff_size;
    else
    {
	n = url->size - url->pos;
	if(n > buff_size)
	    n = buff_size;
    }

    if(n <= 0)
	return 0;
    n = url_read(tmdy_struct, url->instream, buff, n);
    if(n <= 0)
	return n;

    return n;
}

URL url_arc_open(tmdy_struct_ex_t *tmdy_struct, char *name)
{
    URL_arc *url;
    char *base, *p;
    int len;
    ArchiveFileList *afl;
    ArchiveEntryNode *entry;
    URL instream;

    if((p = strrchr(name, '#')) == NULL)
	return NULL;
    len = p - name;
    base = TMDY_UTILS->mblock->new_segment(tmdy_struct, &arc_buffer, len + 1);
    memcpy(base, name, len);
    base[len] = '\0';
    base = TMDY_ARC->url->url_expand_home_dir(tmdy_struct, base);

    if((afl = find_arc_filelist(tmdy_struct,base)) == NULL)
	afl = regist_archive(tmdy_struct,base);
    if(afl == NULL)
	return NULL;
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &arc_buffer);	/* free `base' */
    name += len + 1;

    for(entry = afl->entry_list; entry; entry = entry->next)
    {
	if(strcasecmp(entry->name, name) == 0)
	    break;
    }
    if(entry == NULL)
	return NULL;

    if(entry->cache != NULL)
	instream = TMDY_ARC->url->url_mem_open(tmdy_struct, (char *)entry->cache + entry->start,
				entry->compsize, 0);
    else
    {
	if((instream = TMDY_ARC->url->url_file_open(tmdy_struct, base)) == NULL)
	    return NULL;
	url_seek(tmdy_struct, instream, entry->start, 0);
    }

    url = (URL_arc *)alloc_url(tmdy_struct, sizeof(URL_arc));
    if(url == NULL)
    {
    *(TMDY_ARC->url->url_errno_p) = errno;
	return NULL;
    }

    /* open decoder */
    switch(entry->comptype)
    {
      case ARCHIVEC_STORED:	/* No compression */
      case ARCHIVEC_LZHED_LH0:	/* -lh0- */
      case ARCHIVEC_LZHED_LZ4:	/* -lz4- */
	url->decoder = NULL;

      case ARCHIVEC_DEFLATED:	/* deflate */
	url->decoder =
	    (void *)TMDY_ARC->zip->open_inflate_handler(tmdy_struct, archiver_read_func, url);
	if(url->decoder == NULL)
	{
	    url_arc_close(tmdy_struct, (URL)url);
	    return NULL;
	}
	break;

      case ARCHIVEC_IMPLODED_LIT8:
      case ARCHIVEC_IMPLODED_LIT4:
      case ARCHIVEC_IMPLODED_NOLIT8:
      case ARCHIVEC_IMPLODED_NOLIT4:
	url->decoder =
	    (void *)TMDY_ARC->explode->open_explode_handler(tmdy_struct, archiver_read_func,
					 entry->comptype - ARCHIVEC_IMPLODED - 1,
					 entry->compsize, entry->origsize, url);
	if(url->decoder == NULL)
	{
	    url_arc_close(tmdy_struct, (URL)url);
	    return NULL;
	}
	break;

      case ARCHIVEC_LZHED_LH1:	/* -lh1- */
      case ARCHIVEC_LZHED_LH2:	/* -lh2- */
      case ARCHIVEC_LZHED_LH3:	/* -lh3- */
      case ARCHIVEC_LZHED_LH4:	/* -lh4- */
      case ARCHIVEC_LZHED_LH5:	/* -lh5- */
      case ARCHIVEC_LZHED_LZS:	/* -lzs- */
      case ARCHIVEC_LZHED_LZ5:	/* -lz5- */
      case ARCHIVEC_LZHED_LHD:	/* -lhd- */
      case ARCHIVEC_LZHED_LH6:	/* -lh6- */
      case ARCHIVEC_LZHED_LH7:	/* -lh7- */

	timidity_mutex_lock(TMDY_ARC->unlzh->busy);
	url->decoder =
	    (void *)open_unlzh_handler(tmdy_struct, 
				       archiver_read_func,
				       TMDY_ARC->unlzh->lzh_methods[entry->comptype - ARCHIVEC_LZHED - 1],
				       entry->compsize, entry->origsize, url);
    timidity_mutex_unlock(TMDY_ARC->unlzh->busy);
	if(url->decoder == NULL)
	{
	    url_arc_close(tmdy_struct, (URL)url);
	    return NULL;
	}
	break;
      default:
	url_arc_close(tmdy_struct, (URL)url);
	return NULL;
    }


    /* common members */
    URLm(url, type)      = URL_arc_t;
    URLm(url, url_read)  = url_arc_read;
    URLm(url, url_gets)  = NULL;
    URLm(url, url_fgetc) = NULL;
    URLm(url, url_seek)  = NULL;
    URLm(url, url_tell)  = url_arc_tell;
    URLm(url, url_close) = url_arc_close;

    /* private members */
    url->instream = instream;
    url->pos = 0;
    url->size = entry->origsize;
    url->comptype = entry->comptype;
    return (URL)url;
}

static long url_arc_read(tmdy_struct_ex_t *tmdy_struct, URL url, void *vp, long bufsiz)
{
    URL_arc *urlp = (URL_arc *)url;
    long n = 0;
    int comptype;
    void *decoder;
    char *buff = (char *)vp;

    if(urlp->pos == -1)
	return 0;

    comptype = urlp->comptype;
    decoder = urlp->decoder;
    switch(comptype)
    {
      case ARCHIVEC_STORED:
      case ARCHIVEC_LZHED_LH0:	/* -lh0- */
      case ARCHIVEC_LZHED_LZ4:	/* -lz4- */
	n = archiver_read_func(tmdy_struct,buff, bufsiz, (void *)urlp);
	break;

      case ARCHIVEC_DEFLATED:
	n = TMDY_ARC->zip->zip_inflate(tmdy_struct, (InflateHandler)decoder, buff, bufsiz);
	break;

      case ARCHIVEC_IMPLODED_LIT8:
      case ARCHIVEC_IMPLODED_LIT4:
      case ARCHIVEC_IMPLODED_NOLIT8:
      case ARCHIVEC_IMPLODED_NOLIT4:
	n = explode(tmdy_struct, (ExplodeHandler)decoder, buff, bufsiz);
	break;

      case ARCHIVEC_LZHED_LH1:	/* -lh1- */
      case ARCHIVEC_LZHED_LH2:	/* -lh2- */
      case ARCHIVEC_LZHED_LH3:	/* -lh3- */
      case ARCHIVEC_LZHED_LH4:	/* -lh4- */
      case ARCHIVEC_LZHED_LH5:	/* -lh5- */
      case ARCHIVEC_LZHED_LZS:	/* -lzs- */
      case ARCHIVEC_LZHED_LZ5:	/* -lz5- */
      case ARCHIVEC_LZHED_LHD:	/* -lhd- */
      case ARCHIVEC_LZHED_LH6:	/* -lh6- */
      case ARCHIVEC_LZHED_LH7:	/* -lh7- */
	n = unlzh(tmdy_struct, (UNLZHHandler)decoder, buff, bufsiz);
	break;

      case ARCHIVEC_UU:		/* uu encoded */
      case ARCHIVEC_B64:	/* base64 encoded */
      case ARCHIVEC_QS:		/* quoted string encoded */
      case ARCHIVEC_HQX:	/* HQX encoded */
	n = url_read(tmdy_struct, (URL)decoder, buff, bufsiz);
	break;
    }

    if(n > 0)
	urlp->pos += n;
    return n;
}

static long url_arc_tell(tmdy_struct_ex_t *tmdy_struct, URL url)
{
    return ((URL_arc *)url)->pos;
}

static void url_arc_close(tmdy_struct_ex_t *tmdy_struct, URL url)
{
    URL_arc *urlp = (URL_arc *)url;
    void *decoder;
    int save_errno = errno;

    /* 1. close decoder 
	* 2. close decode_stream
	    * 3. free url
		*/

    decoder = urlp->decoder;
    if(decoder != NULL)
    {
	switch(urlp->comptype)
	{
	  case ARCHIVEC_DEFLATED:
	    TMDY_ARC->zip->close_inflate_handler(tmdy_struct, (InflateHandler)decoder);
	    break;

	  case ARCHIVEC_IMPLODED_LIT8:
	  case ARCHIVEC_IMPLODED_LIT4:
	  case ARCHIVEC_IMPLODED_NOLIT8:
	  case ARCHIVEC_IMPLODED_NOLIT4:
	    close_explode_handler(tmdy_struct, (ExplodeHandler)decoder);
	    break;

	  case ARCHIVEC_LZHED_LH1: /* -lh1- */
	  case ARCHIVEC_LZHED_LH2: /* -lh2- */
	  case ARCHIVEC_LZHED_LH3: /* -lh3- */
	  case ARCHIVEC_LZHED_LH4: /* -lh4- */
	  case ARCHIVEC_LZHED_LH5: /* -lh5- */
	  case ARCHIVEC_LZHED_LZS: /* -lzs- */
	  case ARCHIVEC_LZHED_LZ5: /* -lz5- */
	  case ARCHIVEC_LZHED_LHD: /* -lhd- */
	  case ARCHIVEC_LZHED_LH6: /* -lh6- */
	  case ARCHIVEC_LZHED_LH7: /* -lh7- */
	    close_unlzh_handler(tmdy_struct, (UNLZHHandler)decoder);
	    break;

	  case ARCHIVEC_UU:	/* uu encoded */
	  case ARCHIVEC_B64:	/* base64 encoded */
	  case ARCHIVEC_QS:	/* quoted string encoded */
	  case ARCHIVEC_HQX:	/* HQX encoded */
	    url_close(tmdy_struct, (URL)decoder);
	    break;
	}
    }

    if(urlp->instream != NULL)
	url_close(tmdy_struct, urlp->instream);
    free(urlp);
    errno = save_errno;
}



/************** wildmat ***************/
/* What character marks an inverted character class? */
#define NEGATE_CLASS		'!'

/* Is "*" a common pattern? */
#define OPTIMIZE_JUST_STAR

/* Do tar(1) matching rules, which ignore a trailing slash? */
#undef MATCH_TAR_PATTERN

/* Define if case is ignored */
#define MATCH_CASE_IGNORE

#include <ctype.h>
#define TEXT_CASE_CHAR(c) (toupper(c))
#define CHAR_CASE_COMP(a, b) (TEXT_CASE_CHAR(a) == TEXT_CASE_CHAR(b))

static char *ParseHex(tmdy_struct_ex_t *tmdy_struct, char *p, int *val)
{
    int i, v;

    *val = 0;
    for(i = 0; i < 2; i++)
    {
	v = *p++;
	if('0' <= v && v <= '9')
	    v = v - '0';
	else if('A' <= v && v <= 'F')
	    v = v - 'A' + 10;
	else if('a' <= v && v <= 'f')
	    v = v - 'a' + 10;
	else
	    return NULL;
	*val = (*val << 4 | v);
    }
    return p;
}

/*
 *  Match text and p, return TRUE, FALSE, or ABORT.
 */
static int DoMatch(tmdy_struct_ex_t *tmdy_struct, char *text, char *p)
{
    register int	last;
    register int	matched;
    register int	reverse;

    for ( ; *p; text++, p++) {
	if (*text == '\0' && *p != '*')
	    return ABORT;
	switch (*p) {
	case '\\':
	    p++;
	    if(*p == 'x')
	    {
		int c;
		if((p = ParseHex(tmdy_struct, ++p, &c)) == NULL)
		    return ABORT;
		if(*text != c)
		    return FALSE;
		continue;
	    }
	    /* Literal match with following character. */

	    /* FALLTHROUGH */
	default:
	    if (*text != *p)
		return FALSE;
	    continue;
	case '?':
	    /* Match anything. */
	    continue;
	case '*':
	    while (*++p == '*')
		/* Consecutive stars act just like one. */
		continue;
	    if (*p == '\0')
		/* Trailing star matches everything. */
		return TRUE;
	    while (*text)
		if ((matched = DoMatch(tmdy_struct, text++, p)) != FALSE)
		    return matched;
	    return ABORT;
	case '[':
	    reverse = p[1] == NEGATE_CLASS ? TRUE : FALSE;
	    if (reverse)
		/* Inverted character class. */
		p++;
	    matched = FALSE;
	    if (p[1] == ']' || p[1] == '-')
		if (*++p == *text)
		    matched = TRUE;
	    for (last = *p; *++p && *p != ']'; last = *p)
		/* This next line requires a good C compiler. */
		if (*p == '-' && p[1] != ']'
		    ? *text <= *++p && *text >= last : *text == *p)
		    matched = TRUE;
	    if (matched == reverse)
		return FALSE;
	    continue;
	}
    }

#ifdef	MATCH_TAR_PATTERN
    if (*text == '/')
	return TRUE;
#endif	/* MATCH_TAR_ATTERN */
    return *text == '\0';
}

static int DoCaseMatch(tmdy_struct_ex_t *tmdy_struct, char *text, char *p)
{
    register int	last;
    register int	matched;
    register int	reverse;

    for(; *p; text++, p++)
    {
	if(*text == '\0' && *p != '*')
	    return ABORT;
	switch (*p)
	{
	  case '\\':
	    p++;
	    if(*p == 'x')
	    {
		int c;
		if((p = ParseHex(tmdy_struct, ++p, &c)) == NULL)
		    return ABORT;
		if(!CHAR_CASE_COMP(*text, c))
		    return FALSE;
		continue;
	    }
	    /* Literal match with following character. */

	    /* FALLTHROUGH */
	  default:
	    if(!CHAR_CASE_COMP(*text, *p))
		return FALSE;
	    continue;
	  case '?':
	    /* Match anything. */
	    continue;
	  case '*':
	    while(*++p == '*')
		/* Consecutive stars act just like one. */
		continue;
	    if(*p == '\0')
		/* Trailing star matches everything. */
		return TRUE;
	    while(*text)
		if((matched = DoCaseMatch(tmdy_struct, text++, p)) != FALSE)
		    return matched;
	    return ABORT;
	case '[':
	    reverse = p[1] == NEGATE_CLASS ? TRUE : FALSE;
	    if(reverse)
		/* Inverted character class. */
		p++;
	    matched = FALSE;
	    if(p[1] == ']' || p[1] == '-')
	    {
		if(*++p == *text)
		    matched = TRUE;
	    }
	    for(last = TEXT_CASE_CHAR(*p); *++p && *p != ']';
		last = TEXT_CASE_CHAR(*p))
	    {
		/* This next line requires a good C compiler. */
		if(*p == '-' && p[1] != ']')
		{
		    p++;
		    if(TEXT_CASE_CHAR(*text) <= TEXT_CASE_CHAR(*p) &&
		       TEXT_CASE_CHAR(*text) >= last)
			matched = TRUE;
		}
		else
		{
		    if(CHAR_CASE_COMP(*text, *p))
			matched = TRUE;
		}
	    }
	    if(matched == reverse)
		return FALSE;
	    continue;
	}
    }

#ifdef	MATCH_TAR_PATTERN
    if (*text == '/')
	return TRUE;
#endif	/* MATCH_TAR_ATTERN */
    return *text == '\0';
}

/*
**  User-level routine.  Returns TRUE or FALSE.
*/
int arc_wildmat(tmdy_struct_ex_t *tmdy_struct, char *text, char *p)
{
#ifdef	OPTIMIZE_JUST_STAR
    if (p[0] == '*' && p[1] == '\0')
	return TRUE;
#endif /* OPTIMIZE_JUST_STAR */
    return DoMatch(tmdy_struct, text, p) == TRUE;
}

int arc_case_wildmat(tmdy_struct_ex_t *tmdy_struct, char *text, char *p)
{
#ifdef	OPTIMIZE_JUST_STAR
    if (p[0] == '*' && p[1] == '\0')
	return TRUE;
#endif /* OPTIMIZE_JUST_STAR */
    return DoCaseMatch(tmdy_struct, text, p) == TRUE;
}



arc_ex_t* new_arc(tmdy_struct_ex_t *tmdy_struct){
	int i;
	arc_ex_t* arc_ex;

	arc_ex=(arc_ex_t *)malloc(sizeof(arc_ex_t));
	
	timidity_mutex_init(arc_ex->busy);
	
arc_ex->expand_archive_names=expand_archive_names;
arc_ex->url_arc_open=url_arc_open;
arc_ex->free_archive_files=free_archive_files;
arc_ex->skip_gzip_header=skip_gzip_header;
arc_ex->parse_gzip_header_bytes=parse_gzip_header_bytes;
arc_ex->get_archive_type=get_archive_type;
arc_ex->arc_compress=arc_compress;
arc_ex->arc_decompress=arc_decompress;
arc_ex->arc_case_wildmat=arc_case_wildmat;
arc_ex->arc_wildmat=arc_wildmat;
arc_ex->arc_error_handler=NULL;

		return arc_ex;
}
void destroy_arc(arc_ex_t* arc){
	timidity_mutex_destroy(arc->busy);
	free(arc);
}
