/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    output.c

    Audio output (to file / device) functions.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef STDC_HEADERS
#include <string.h>
#include <ctype.h>
#elif HAVE_STRINGS_H
#include <strings.h>
#endif

#include "timidity.h"
#include "common.h"
#include "output.h"
#include "tables.h"
#include "controls.h"
#include "audio_cnv.h"

#include "output_prv.h" //guha

extern tmdy_struct_ex_t *tmdy_struct; //guha

int audio_buffer_bits = DEFAULT_AUDIO_BUFFER_BITS;

/* These are very likely mutually exclusive.. */
#if defined(AU_AUDRIV)
extern PlayMode audriv_play_mode;
#define DEV_PLAY_MODE &audriv_play_mode

#elif defined(AU_SUN)
extern PlayMode sun_play_mode;
#define DEV_PLAY_MODE &sun_play_mode

#elif defined(AU_OSS)
extern PlayMode oss_play_mode;
#define DEV_PLAY_MODE &oss_play_mode

#elif defined(AU_HPUX_AUDIO)
extern PlayMode hpux_play_mode;
#define DEV_PLAY_MODE &hpux_play_mode

#elif defined(AU_W32)
extern PlayMode w32_play_mode;
#define DEV_PLAY_MODE &w32_play_mode

#elif defined(AU_BSDI)
extern PlayMode bsdi_play_mode;
#define DEV_PLAY_MODE &bsdi_play_mode

#elif defined(__MACOS__)
extern PlayMode mac_play_mode;
#define DEV_PLAY_MODE &mac_play_mode

#elif defined(AU_DARWIN)
extern PlayMode darwin_play_mode;
#define DEV_PLAY_MODE &darwin_play_mode
#endif

#ifdef AU_ALSA
extern PlayMode alsa_play_mode;
#endif /* AU_ALSA */

#ifdef AU_HPUX_ALIB
extern PlayMode hpux_nplay_mode;
#endif /* AU_HPUX_ALIB */

#ifdef AU_ARTS
extern PlayMode arts_play_mode;
#endif /* AU_ARTS */

#ifdef AU_ESD
extern PlayMode esd_play_mode;
#endif /* AU_ESD */

#ifdef AU_PORTAUDIO
extern PlayMode portaudio_play_mode;
#endif /* AU_PORTAUDIO */

#ifdef AU_JACK
extern PlayMode jack_play_mode;
#endif /* AU_NAS */

#ifdef AU_NAS
extern PlayMode nas_play_mode;
#endif /* AU_NAS */

#ifndef __MACOS__
/* These are always compiled in. */
extern PlayMode raw_play_mode, wave_play_mode, au_play_mode, aiff_play_mode;
extern PlayMode list_play_mode;
#ifdef AU_VORBIS
extern PlayMode vorbis_play_mode;
#endif /* AU_VORBIS */
#ifdef AU_GOGO
extern PlayMode gogo_play_mode;
#endif /* AU_GOGO */
#endif /* !__MACOS__ */

extern PlayMode modmidi_play_mode;

/*****************************************************************/
/* Some functions to convert signed 32-bit data to other formats */

void s32tos8(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
    int8 *cp=(int8 *)(lp);
    int32 l, i;

    for(i = 0; i < c; i++)
    {
	l=(lp[i])>>(32-8-GUARD_BITS);
	if (l>127) l=127;
	else if (l<-128) l=-128;
	cp[i] = (int8)(l);
    }
}

void s32tou8(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
    uint8 *cp=(uint8 *)(lp);
    int32 l, i;

    for(i = 0; i < c; i++)
    {
	l=(lp[i])>>(32-8-GUARD_BITS);
	if (l>127) l=127;
	else if (l<-128) l=-128;
	cp[i] = 0x80 ^ ((uint8) l);
    }
}

void s32tos16(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
  int16 *sp=(int16 *)(lp);
  int32 l, i;

  for(i = 0; i < c; i++)
    {
      l=(lp[i])>>(32-16-GUARD_BITS);
      if (l > 32767) l=32767;
      else if (l<-32768) l=-32768;
      sp[i] = (int16)(l);
    }
}

void s32tou16(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
  uint16 *sp=(uint16 *)(lp);
  int32 l, i;

  for(i = 0; i < c; i++)
    {
      l=(lp[i])>>(32-16-GUARD_BITS);
      if (l > 32767) l=32767;
      else if (l<-32768) l=-32768;
      sp[i] = 0x8000 ^ (uint16)(l);
    }
}

void s32tos16x(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
  int16 *sp=(int16 *)(lp);
  int32 l, i;

  for(i = 0; i < c; i++)
    {
      l=(lp[i])>>(32-16-GUARD_BITS);
      if (l > 32767) l=32767;
      else if (l<-32768) l=-32768;
      sp[i] = XCHG_SHORT((int16)(l));
    }
}

void s32tou16x(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
  uint16 *sp=(uint16 *)(lp);
  int32 l, i;

  for(i = 0; i < c; i++)
    {
      l=(lp[i])>>(32-16-GUARD_BITS);
      if (l > 32767) l=32767;
      else if (l<-32768) l=-32768;
      sp[i] = XCHG_SHORT(0x8000 ^ (uint16)(l));
    }
}

void s32toulaw(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
    int8 *up=(int8 *)(lp);
    int32 l, i;

    for(i = 0; i < c; i++)
    {
	l=(lp[i])>>(32-16-GUARD_BITS);
	if (l > 32767) l=32767;
	else if (l<-32768) l=-32768;
	up[i] = AUDIO_S2U(l);
    }
}

void s32toalaw(tmdy_struct_ex_t *tmdy_struct, int32 *lp, int32 c)
{
    int8 *up=(int8 *)(lp);
    int32 l, i;

    for(i = 0; i < c; i++)
    {
	l=(lp[i])>>(32-16-GUARD_BITS);
	if (l > 32767) l=32767;
	else if (l<-32768) l=-32768;
	up[i] = AUDIO_S2A(l);
    }
}

/* return: number of bytes */
int32 general_output_convert(tmdy_struct_ex_t *tmdy_struct, int32 *buf, int32 count)
{
    int32 bytes;

    if(!((TMDY_OUTPUT->play_mode)->encoding & PE_MONO))
	count *= 2; /* Stereo samples */
    bytes = count;
    if((TMDY_OUTPUT->play_mode)->encoding & PE_16BIT)
    {
	bytes *= 2;
	if((TMDY_OUTPUT->play_mode)->encoding & PE_BYTESWAP)
	{
	    if((TMDY_OUTPUT->play_mode)->encoding & PE_SIGNED)
		s32tos16x(tmdy_struct, buf, count);
	    else
		s32tou16x(tmdy_struct, buf, count);
	}
	else if((TMDY_OUTPUT->play_mode)->encoding & PE_SIGNED)
	    s32tos16(tmdy_struct, buf, count);
	else
	    s32tou16(tmdy_struct, buf, count);
    }
    else if((TMDY_OUTPUT->play_mode)->encoding & PE_ULAW)
	s32toulaw(tmdy_struct, buf, count);
    else if((TMDY_OUTPUT->play_mode)->encoding & PE_ALAW)
	s32toalaw(tmdy_struct, buf, count);
    else if((TMDY_OUTPUT->play_mode)->encoding & PE_SIGNED)
	s32tos8(tmdy_struct, buf, count);
    else
	s32tou8(tmdy_struct, buf, count);
    return bytes;
}

int validate_encoding(tmdy_struct_ex_t *tmdy_struct, int enc, int include_enc, int exclude_enc)
{
    const char *orig_enc_name, *enc_name;
    int orig_enc;

    orig_enc = enc;
    orig_enc_name = output_encoding_string(tmdy_struct, enc);
    enc |= include_enc;
    enc &= ~exclude_enc;
    if(enc & (PE_ULAW|PE_ALAW))
	enc &= ~(PE_16BIT|PE_SIGNED|PE_BYTESWAP);
    if(!(enc & PE_16BIT))
	enc &= ~PE_BYTESWAP;
    enc_name = output_encoding_string(tmdy_struct, enc);
    if(strcmp(orig_enc_name, enc_name) != 0)
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NOISY,
		  "Notice: Audio encoding is changed `%s' to `%s'",
		  orig_enc_name, enc_name);
    return enc;
}

const char *output_encoding_string(tmdy_struct_ex_t *tmdy_struct, int enc)
{
    if(enc & PE_MONO)
    {
	if(enc & PE_16BIT)
	{
	    if(enc & PE_SIGNED)
		return "16bit (mono)";
	    else
		return "unsigned 16bit (mono)";
	}
	else
	{
	    if(enc & PE_ULAW)
		return "U-law (mono)";
	    else if(enc & PE_ALAW)
		return "A-law (mono)";
	    else if(enc & PE_SIGNED)
		return "8bit (mono)";
	    else
		return "unsigned 8bit (mono)";
	}
    }
    else if(enc & PE_16BIT)
    {
	if(enc & PE_BYTESWAP)
	{
	    if(enc & PE_SIGNED)
		return "16bit (swap)";
	    else
		return "unsigned 16bit (swap)";
	}
	else if(enc & PE_SIGNED)
	    return "16bit";
	else
	    return "unsigned 16bit";
    }
    else
	if(enc & PE_ULAW)
	    return "U-law";
	else if(enc & PE_ALAW)
	    return "A-law";
	else if(enc & PE_SIGNED)
	    return "8bit";
	else
	    return "unsigned 8bit";
    /*NOTREACHED*/
}

/* mode
  0,1: Default mode.
  2: Remove the directory path of input_filename, then add output_dir.
  3: Replace directory separator characters ('/','\',':') with '_', then add output_dir.
 */
char *create_auto_output_name(tmdy_struct_ex_t *tmdy_struct, const char *input_filename, char *ext_str, char *output_dir, int mode)
{
  char *output_filename;
  char *ext, *p;
  int32 dir_len = 0;
  char ext_str_tmp[65];

  output_filename = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, (output_dir?strlen(output_dir):0) + strlen(input_filename) + 6);
  if(output_filename==NULL)
    return NULL;
  output_filename[0] = '\0';
  if(output_dir!=NULL && (mode==2 || mode==3)) {
    strcat(output_filename,output_dir);
    dir_len = strlen(output_filename);
#ifndef __W32__
    if(dir_len>0 && output_filename[dir_len-1]!=PATH_SEP){
#else
      if(dir_len>0 && output_filename[dir_len-1]!='/' && output_filename[dir_len-1]!='\\' && output_filename[dir_len-1]!=':'){
#endif
	strcat(output_filename,PATH_STRING);
	dir_len++;
      }
    }
    strcat(output_filename, input_filename);

    if((ext = strrchr(output_filename, '.')) == NULL)
      ext = output_filename + strlen(output_filename);
    else {
      /* strip ".gz" */
      if(strcasecmp(ext, ".gz") == 0) {
	*ext = '\0';
	if((ext = strrchr(output_filename, '.')) == NULL)
	  ext = output_filename + strlen(output_filename);
      }
    }

    /* replace '\' , '/' or PATH_SEP between '#' and ext */
    p = strrchr(output_filename,'#');
    if(p!=NULL){
      char *p1;
#ifdef _mbsrchr
#define STRCHR(a,b) _mbschr(a,b)
#else
#define STRCHR(a,b) strchr(a,b)
#endif
#ifndef __W32__
      p1 = p + 1;
      while((p1 = STRCHR(p1,PATH_SEP))!=NULL && p1<ext){
        *p1 = '_';
	p1++;
      }
#else
      p1 = p + 1;
      while((p1 = STRCHR(p1,'\\'))!=NULL && p1<ext){
      	*p1 = '_';
	p1++;
      }
      p1 = p;
      while((p1 = STRCHR(p1,'/'))!=NULL && p1<ext){
	*p1 = '_';
	p1++;
      }
#endif
#undef STRCHR
    }

    /* replace '.' and '#' before ext */
    for(p = output_filename; p < ext; p++)
#ifndef __W32__
      if(*p == '.' || *p == '#')
#else
	if(*p == '#')
#endif
	  *p = '_';

    if(mode==2){
      char *p1,*p2,*p3;
#ifndef __W32__
      p = strrchr(output_filename+dir_len,PATH_SEP);
#else
#ifdef _mbsrchr
#define STRRCHR _mbsrchr
#else
#define STRRCHR strrchr
#endif
      p1 = STRRCHR(output_filename+dir_len,'/');
      p2 = STRRCHR(output_filename+dir_len,'\\');
      p3 = STRRCHR(output_filename+dir_len,':');
#undef STRRCHR
      p1>p2 ? (p1>p3 ? (p = p1) : (p = p3)) : (p2>p3 ? (p = p2) : (p = p3));
#endif
      if(p!=NULL){
	for(p1=output_filename+dir_len,p2=p+1; *p2; p1++,p2++)
	  *p1 = *p2;
	*p1 = '\0';
      }
    }

    if(mode==3){
      for(p=output_filename+dir_len; *p; p++)
#ifndef __W32__
	if(*p==PATH_SEP)
#else
	  if(*p=='/' || *p=='\\' || *p==':')
#endif
	    *p = '_';
    }

    if((ext = strrchr(output_filename, '.')) == NULL)
      ext = output_filename + strlen(output_filename);
    if(*ext){
      strncpy(ext_str_tmp,ext_str,64);
      ext_str_tmp[64]=0;
      if(isupper(*(ext + 1))){
	for(p=ext_str_tmp;*p;p++)
	  *p = toupper(*p);
	*p = '\0';
      } else {
	for(p=ext_str_tmp;*p;p++)
	  *p = tolower(*p);
	*p = '\0';
      }
      strcpy(ext+1,ext_str_tmp);
    }
    return output_filename;
}




output_ex_t* new_output(tmdy_struct_ex_t *tmdy_struct){
	int i;
	output_ex_t* output_ex;

	output_ex=(output_ex_t *)malloc(sizeof(output_ex_t));
	
	output_ex->s32tos8=s32tos8;
	output_ex->s32tou8=s32tou8;

	output_ex->s32tos16=s32tos16;
	output_ex->s32tou16=s32tou16;

	output_ex->s32tos16x=s32tos16x;
	output_ex->s32tou16x=s32tou16x;

	output_ex->s32toulaw=s32toulaw;

	output_ex->s32toalaw=s32toalaw;

	output_ex->general_output_convert=general_output_convert;
	output_ex->validate_encoding=validate_encoding;
	output_ex->output_encoding_string=output_encoding_string;

	output_ex->create_auto_output_name=create_auto_output_name;

	
	i=0;
#ifdef DEV_PLAY_MODE
	output_ex->play_mode_list[i++] =  DEV_PLAY_MODE;
#endif

#ifdef AU_ALSA
	output_ex->play_mode_list[i++] =  &alsa_play_mode;
#endif /* AU_ALSA */

#ifdef AU_HPUX_ALIB
	output_ex->play_mode_list[i++] =  &hpux_nplay_mode;
#endif /* AU_HPUX_ALIB */

#if defined(AU_ARTS)
	output_ex->play_mode_list[i++] =  &arts_play_mode;
#endif /* AU_ARTS */

#if defined(AU_ESD)
	output_ex->play_mode_list[i++] =  &esd_play_mode;
#endif /* AU_ESD */

#if defined(AU_PORTAUDIO)
	output_ex->play_mode_list[i++] =  &portaudio_play_mode;
#endif /* AU_PORTAUDIO */

#if defined(AU_JACK)
	output_ex->play_mode_list[i++] =  &jack_play_mode;
#endif /* AU_PORTAUDIO */

#if defined(AU_NAS)
	output_ex->play_mode_list[i++] =  &nas_play_mode;
#endif /* AU_NAS */

#ifndef __MACOS__
	output_ex->play_mode_list[i++] =  &wave_play_mode;
	output_ex->play_mode_list[i++] =  &raw_play_mode;
	output_ex->play_mode_list[i++] =  &au_play_mode;
	output_ex->play_mode_list[i++] =  &aiff_play_mode;
#ifdef AU_VORBIS
	output_ex->play_mode_list[i++] =  &vorbis_play_mode;
#endif /* AU_VORBIS */
#ifdef AU_GOGO
	output_ex->play_mode_list[i++] =  &gogo_play_mode;
#endif /* AU_GOGO */
	output_ex->play_mode_list[i++] =  &list_play_mode;
#endif /* __MACOS__ */
	output_ex->play_mode_list[i++] =  &modmidi_play_mode;
	output_ex->play_mode_list[i++] =  0;

	output_ex->play_mode = NULL;
	output_ex->target_play_mode = NULL;

	return output_ex;
}
void destroy_output(output_ex_t* output){	
	free(output);
}


