unit gamecode;

{ functions and procedures for the game }

interface

const
  { codes for special game-modes }
  sm_None       = 0; { none }
  sm_Scroll     = 1; { display hidden scroll text at top of screen }
  sm_InverseDir = 2; { inverse direction, move left-wards }
  sm_Extrahard  = 3; { extremely difficult }

procedure InitialiseGame;
procedure CleanupGame;

procedure DisplayCaveSlice;
procedure CaveStep;
procedure ChangeCaveDirection;

procedure HandlePlayerInput;
procedure SnakeStep;

procedure BackgroundEffect;

Function HiddenString(Input : String) : String;


var
  Difficulty : Byte;  { 0=easy, 1=normal, 2=hard }
  StartLevel : Byte;  { start at a level higher than 1 }
  SpecialMode : Byte; { secret modes }

  PlayerXpos : Byte;  { graphics unit needs it too }
  Level : Byte;       { graphics unit needs it too }

  GameOver : Boolean;

implementation

uses
  smallcrt, graphics, grafdata, keyboard, hiscores, menucode;

const
  c_SnakeYsize = 4;
  c_LevelLength = 4 * VGAx;

  { lookup-table for sinus-wave, a sinus with amplitude 100 in 256 steps,
    only the first 64 steps }
{  SnakeSinus : Array[0..63] of Byte =
  (2,3,2,3,2,3,2,3,2,2,3,2,2,3,2,2,
   3,2,2,2,2,2,2,3,2,2,2,1,2,2,2,2,
   1,2,2,1,2,1,2,1,1,2,1,1,1,1,1,1,
   1,1,1,1,0,1,1,0,1,0,0,1,0,0,0,0);

  { lookup-table for sinus-wave, a sinus with amplitude 75 in 256 steps,
    only the first 64 steps }
  SnakeSinus : Array[0..63] of Byte =
  (2,2,2,1,2,2,2,2,1,2,2,2,2,1,2,2,
   1,2,2,1,2,2,1,2,1,2,1,2,1,1,2,1,
   1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,0,
   1,1,0,1,0,1,0,1,0,0,0,1,0,0,0,0);

  { lookup-table for sinus-wave, a sinus with amplitude 50 in 256 steps,
    only the first 64 steps }
{  SnakeSinus : Array[0..63] of Byte =
  (1,1,2,1,1,1,2,1,1,1,1,2,1,1,1,1,
   1,1,1,2,1,1,1,1,1,1,1,1,1,1,0,1,
   1,1,1,1,0,1,1,1,0,1,1,0,1,0,1,0,
   1,0,0,1,0,1,0,0,0,0,1,0,0,0,0,0);
}

var
  { player variables }
  PlayerYpos : Byte;
  PlayerYlookup : Byte;
  PlayerScore : Word;
  { level }
  LevelCountDown : Word;
  { tunnel variables }
  TunnelYpos : Byte;
  TunnelHeightCurrent, TunnelHeightTarget : Byte;
  TunnelCountdown : Word;
  TunnelDirection : Integer;

  { obstacle variables }
  ObstacleYpos : Byte;
  ObstacleWidth, ObstacleHeigth : Byte;
  ObstacleCountDown : Word;

  { background variables }
  ColorBackground : Byte;

procedure PauseGame;
const
  xy_offset = (12*8*VGAx) + (17*8);
var
  GraphicsBackup : Array[0..(8*5*9)] of Byte;
  X, Y : Word;
  FlushKey : Char;
  ContinueGame : Boolean;
begin
  { make back-up of background }
  for y := 0 to 8 do
    for x := 0 to (8*5)-1 do
      GraphicsBackup[(8*5*y) + x] := Mem[VirtSeg:xy_offset + (y*VGAx) + x];

  { display pause }
  WriteText(17, 12, 'pause', cl_White);
  V_Flip;

  { incase game-paused while holding down spacebar }
  repeat until not SpacebarPressed;

  { wait for keypress }
  ContinueGame := False;
  repeat
    if keypressed then
    begin
      FlushKey := ReadKey;
      ContinueGame := True;
    end;
    if SpacebarPressed Then
      ContinueGame := True;
  until ContinueGame;
  { put back the original background }
  for y := 0 to 8 do
    for x := 0 to (8*5)-1 do
      Mem[VirtSeg:xy_offset + (y*VGAx) + x] := GraphicsBackup[(8*5*y) + x];
end;

Function HiddenString(Input : String) : String;
{ This function takes a scrambled string and makes it readable again
  This is because all strings are stored in the final .EXE file and can
  be found with a hex editor. Some strings must not be readable, such as
  the credits and the cheat codes :) }
Var
  i : Byte;
  tmpString : String;
Begin
  tmpString := '';
  For i := 1 to length(input) Do
   tmpString := tmpString + Chr(Ord(Input[i])-61);

  { return function result }
  HiddenString := tmpString;

End;


procedure InitialiseObstacle;
var
  tmpByte : Byte;
begin
  { set thickness }
  { on higher levels, less width else it's imposible to play }
  tmpByte := Level div 2;
  ObstacleWidth := 10 - tmpByte; {startout as 10, gradually thinner to width=5 on level 10 }
  { decide at what y-position (within the tunnel) the obstacle will be }
  ObstacleHeigth := (TunnelHeightCurrent div 3); { two-thirds of the tunnel size }
  tmpByte := 2 * ObstacleHeigth;
  ObstacleYpos := TunnelYpos + Random(tmpByte);

  { set countdown until next obstacle }
  {level 1             ..  10
  Easy   21+(2*rnd(29)     21+(2*rnd(10)
  Normal 18+(2*rnd(29)     18+(2*rnd(10)
  Hard   15+(2*rnd(29)     15+(2*rnd(10)
  }
  tmpByte := 15 + (3 * (2-Difficulty)); { tmpByte is invert Difficulty, hard=15, easy=21 }
  ObstacleCountdown := tmpByte * (2 + Random(30-(Level*2)));

end;

procedure DisplayCurrentLevelNr;
var
  strTemp : String;
begin
  { erase previous levelnumber }
  DrawBox(288, VGAy-8, 16, 8, cl_DGrey);
  { display current levelnumber }
  Str(Level, strTemp);
  WriteText(36, 24, strTemp, cl_White);
end;

procedure DisplayPlayerScore;
var
  strScore : String;
  i : byte;
begin
  { erase previous score }
  DrawBox(56, VGAy-8, 40, 8, cl_DGrey);
  { display score }
  Str(PlayerScore, strScore);
  WriteText(7, 24, strScore, cl_White);
end;

procedure GotoLevel(pLevel : Byte);
var
  tmpByte : Byte;
begin
  Level := pLevel;

  { check if at end of game }
  if (Level > 10) then
  begin
    Case Level of
    11 : begin
           { set tunnel at max heigth }
           TunnelHeightCurrent := 170;
           TunnelHeightTarget := TunnelHeightCurrent;
           TunnelYpos := 10;
           { scroll one entire screen untill end of game.. }
           LevelCountdown := VGAx;
           { no more obstacles }
           ObstacleCountdown := c_LevelLength;
         end;
    12 : begin
           { end game the player has completed all 20 levels!! }
           GameOver := True;
         end;
    end; { case }
    Exit;
  end;

  { set length of next level }
  LevelCountdown := c_LevelLength;
  InitGraphics(Level);

  { calculate how high the tunnel should be for this level and difficulty }
  {
  {level 1   .. 10
  Easy   134 -> 80
  Normal 114 -> 65
  Hard   95  -> 50
  XTRA   75  -> 35
  }
  tmpByte := Round(Level * ((10 + (2-Difficulty)) / 2));
  TunnelHeightTarget := 100 + ((2-Difficulty) * 20) - tmpByte;

  { how many frames does it take for HeightDifference to become 0 }
  TunnelCountdown := 2 * (TunnelHeightCurrent - TunnelHeightTarget);
  TunnelDirection := 0; {at start do not move}

  { display 'level:##' }
  DisplayCurrentLevelNr;

end;

procedure EraseStartText;
const
  x1 = 11*8;
  x2 = ((11*8)+(18*8)-1);
  y  = 12*8;
var
  i : Integer;
begin
  { erase special-mode text }
  if SpecialMode = sm_InverseDir then
    for i := x1 to x2 do
      DrawBox(x2-(i-x1), y, 1, 8, 236+(i mod 20)) { erase text }
  else
    for i := x1 to x2 do
      DrawBox(i, y, 1, 8, 236+(i mod 20)); { erase text }

end;

procedure InitialiseGame;
var
  i : Word;
begin

  PlayerXpos := 80;
  { special modes }
  Case SpecialMode of
    sm_Scroll     : InitScrollText;
    sm_InverseDir : PlayerXpos := (VGAx-80);
    sm_Extrahard  : Difficulty := 3;
  end;

  { randomize, else cave is the same every game }
  Randomize;
  { player variables }
  PlayerYpos := 100;
  PlayerYlookup := 63;
  PlayerScore := 0;
  GameOver := False;

  { tunnel variables }
  TunnelYpos := 10;
  TunnelHeightCurrent := 170;

  { obstacle variables }
  InitialiseObstacle;
  ObstacleWidth := 0;

  { background variables }
  ColorBackground := 0;
  { install keyboard handler }
  Xkey;

  { level }
  GotoLevel(StartLevel);

  { fill entire screen with pattern }
  for i := 0 to VGAx-1 do
    PutCaveSlice(i);

  { draw box for score and level }
  DrawBox(0, 191, VGAx, 9, cl_DGrey);

  { draw initial snake and carve out initial tunnel }
  if (SpecialMode = sm_InverseDir) then
  begin
    { 'carve out' an initial tunnel }
    for i := 0 to VGAx-1 do
      DrawBox((VGAx-1)-i, 10, 1, VGAy-29, 236+(i mod 20)); {background effect}
    { draw initial snake on right side }
    for i := PlayerXpos to VGAx-1 do
      PutSnakeSlice(i, PlayerYpos);
  end
  else
  begin
    { 'carve out' an initial tunnel }
    for i := 0 to VGAx-1 do
      DrawBox(i, 10, 1, VGAy-29, 236+(i mod 20)); {background effect}
    { on left side }
    for i := 0 to PlayerXpos do
      PutSnakeSlice(i, PlayerYpos);
  end;

  { set shadow color according to difficulty }
  SetFontShadow(SnakeArray[(Difficulty*4)+3]);
  { 'score:' and 'level:'}
  WriteText( 1, 24, 'Score:0', cl_White);
  WriteText(30, 24, 'Level:', cl_White);
  DisplayCurrentLevelNr;

  { display special-mode text if needed }
  Case SpecialMode of
  sm_Scroll     : WriteTextBlink(16, 12, 'SCROLL TEXT');
  sm_InverseDir : WriteTextBlink(11, 12, 'INVERSE DIRECTION');
  sm_Extrahard  : WriteTextBlink(11, 12, 'MISSION IMPOSSIBLE');
  end;

  { erase text if needed }
  if (SpecialMode <> sm_None) then
    EraseStartText;

  { 'get ready'}
  WriteTextBlink(16, 12, 'GET READY!');

  { erase 'get ready'}
  EraseStartText;

end;

procedure CleanupGame;
var
  strName, strCode : String;
begin

  { check if player complete entire game }
  if (Level = 12) then
  begin
    { draw a surrounding box }
    Drawbox((6*8)-2, (3*8)-2, (29*8)+4, (11*8)+4, cl_White);
    Drawbox((6*8)-1, (3*8)-1, (29*8)+2, (11*8)+2, 0);
    { player finished all 10 levels!!! }
    Case Difficulty of
    0: begin
         strName := 'CONGRATULATIONS!';
         strCode := HiddenString('h'); {Shift+S}
       end;
    1: begin
         strName := '  INCREDIBLE!';
         strCode := HiddenString('h'); {Shift+i}
       end;
    2: begin
         strName := '  ASTOUNDING!';
         strCode := HiddenString('h'); {Shift+M}
       end;
    3: begin
         WriteTextBlink(14,  4, 'UN');
         WriteTextBlink(16,  4, 'BE');
         WriteTextBlink(18,  4, 'LIEVABLE!');
         WriteTextBlink(10,  6, 'This is truly amazing');
         WriteTextBlink( 9,  8, 'Have some more points!!');

         PlayerScore := 50000;
         DisplayPlayerScore;

         WriteTextBlink(10, 10, 'No mortal can complete');
         WriteTextBlink(12, 11, 'MISSION IMPOSSIBLE');
         WriteTextBlink(13, 12, 'You are god-like');

       end;
    end;
    { normal endings () }
    If (Difficulty <> 3) then
    begin
      { display text }
      WriteTextBlink(13,  4, strName);
      WriteTextBlink( 8,  6, 'You completed '+Diff[Difficulty]+' mode');
      WriteTextBlink(10,  7, 'For this amazing feat');
      WriteTextBlink( 7,  8, 'i''ll let you in on a secret');
      { re-use the strName variable }
      strName := HiddenString(']]]]'); {Start the game by pressing}
      WriteTextBlink( 7, 10, strName);
      WriteTextBlink(16, 11, strcode);
      WriteTextBlink( 9, 12, 'and see what happens..');
    end;
  end
  else
  begin
    { player did not reach end of game }
    { draw skull and show virtual screen }
    PutSkull(PlayerYpos);

    WriteTextBlink(16, 12, 'GAME OVER');
  end;

  { incase player still holds down spacebar wait until released }
  repeat until not SpaceBarPressed;

  { is ExtraHard-mode then difficulty back to 'hard' so score can be saved there }
  if (SpecialMode = sm_Extrahard) then
    Difficulty := 2;

  { remove keyboard handler }
  Nkey;

  if (PlayerScore > HiscoresArray[Difficulty].Score) then
  begin
    WriteTextBlink(13, 15, 'New hi-score!!');
    strName := InputString(13, 17, cl_White);
    { store in array}
    HiScoresArray[Difficulty].Name := strName;
    HiScoresArray[Difficulty].Score := PlayerScore;
    if not SaveScores then
      WriteText(4, 19, 'Couldn''t save hi-scores to disk', cl_White);
  end;
  V_Flip;

  repeat until keypressed;
end;

procedure DisplayCaveSlice;
{ display one 'slice' of the case on the left or righthand side
  of the screen (incase specialmode=inverse direction)
  When the screen scrolls a tunnel effect is automatically created }
var
  i, Color : Byte;
  Xpos : Word;
begin
  { scroll the screen }
  If (SpecialMode = sm_InverseDir) then
  begin
    HScrollScreen(False);  { scroll to left }
    Xpos := 0;
  end
  else
  begin
    HScrollScreen(True);  { scroll to right }
    Xpos := (VGAx-1);
  end;

  { draw solid cave }
  PutCaveSlice(Xpos); { 0 of vgax-1}


  { 'carve out' the tunnel }
  for i := TunnelYpos to TunnelYpos+TunnelHeightCurrent do
    mem[VirtSeg:(i*VGAx)+Xpos] := 236+ColorBackground;

  { draw obstacle when obstacle is active }
  if (ObstacleWidth > 0) Then
  begin
    Color := ObstacleArray[Level];
    for i := 1 to ObstacleHeigth do
      mem[VirtSeg:((i+ObstacleYpos)*VGAx)+Xpos] := Color;
    Dec(ObstacleWidth);
  end;

  { display scroll text if active }
  If SpecialMode = sm_Scroll then
    DisplayScrollText;

end;

procedure CaveStep;
{ process all tunnel and object variables }
var
  tmpByte : Byte;
begin
  { move tunnel in direction }
  Case TunnelDirection of
{  0 : {nothing, not moving }
  1 : if (TunnelCountdown mod 2)=0 Then Inc(TunnelYpos); { moving down, slow }
  2 : Inc(TunnelYpos); { moving down }
  3 : if (TunnelCountdown mod 2)=0 Then Dec(TunnelYpos); { moving up, slow }
  4 : Dec(TunnelYpos); { moving up }
  end;

  { if tunnel too high }
  if (TunnelYpos < 10) then
    ChangeCaveDirection;

  { if tunnel too low }
  if (TunnelYpos+TunnelHeightCurrent > (VGAy-20))
  then
    { change direction to opposite direction }
    ChangeCaveDirection;

  { countdown to next obstacle }
  Dec(ObstacleCountdown);
  if (ObstacleCountdown = 0) then
    InitialiseObstacle;

  { adjust tunnel height if needed (when advancing a level) }
  if (TunnelHeightCurrent > TunnelHeightTarget) Then
    { gradually }
    if ((TunnelCountdown mod 4) = 0) then
    begin
      TunnelHeightCurrent := TunnelHeightCurrent - 2;
      Inc(TunnelYpos);
    end;

  { check if advance to next level }
  Dec(LevelCountdown);
  if (LevelCountdown = 0) Then
    GotoLevel(Level+1);

  { countdown to change-tunnel-direction }
  Dec(TunnelCountdown);
  if (TunnelCountdown = 0) Then
    ChangeCaveDirection;

end;

procedure ChangeCaveDirection;
{ TunnelDirection is a code
  0 = do not move up or down
  1 = slowly move down
  2 = move down
  3 = slowly move up
  4 = move up
   }
begin
  { change the direction in which the tunnel is moving }
  TunnelDirection := Random(5);

  { if tunnel too high }
  if (TunnelYpos < 10) then
    if (TunnelDirection >= 3) then
      TunnelDirection := TunnelDirection - 2;

  { if tunnel too low }
  if (TunnelYpos+TunnelHeightCurrent > (VGAy-20)) then
    if (TunnelDirection = 1) or (TunnelDirection = 2) then
      TunnelDirection := TunnelDirection + 2;

  { set TunnelCountdown until next change-tunnel-direction }
  TunnelCountdown := 10 * (1 + Random(5));

end;

procedure HandlePlayerInput;
{ process player input }
Var
  FlushKey : Char;
begin
  { if spacebar pressed down then move up, else move down }
  if SpaceBarPressed then
  begin
    { PlayerYlookup must be between 0 and 127 }
    if (PlayerYlookup > 0) then
      Dec(PlayerYlookup)
  end
  else
    if (PlayerYlookup < 127) then
      Inc(PlayerYlookup);

  { incase PlayerYlookup then move down, else move up }
  If (PlayerYlookup < 64) Then
    PlayerYpos := PlayerYpos - SnakeSinus[PlayerYlookup]
  Else
    PlayerYpos := PlayerYpos + SnakeSinus[127-PlayerYlookup];
  { if ESC or P then pause the game }
  if keypressed then
  begin
    FlushKey := ReadKey;
    If (FlushKey = #27)
    or (FlushKey = 'P')
    or (FlushKey = 'p') then
      PauseGame;
  end;
end;

procedure SnakeStep;
{ check if no collision, and put snake on screen }
var
  i : Byte;
begin
  { check collision by looking at pixels on position of snakehead }
  for i := 0 to c_SnakeYsize-1 do
    if (mem[VirtSeg:((i + PlayerYpos) * VGAx) + PlayerXpos] < 236) then
      { check if game is over }
      GameOver := True;

  { draw new snakehead }
  PutSnakeSlice(PlayerXPos, PlayerYpos);

  { every 4 frames, player scores 1 point }
  if (ColorBackground mod 4 = 0) Then
  Begin
    { add score }
    Inc(PlayerScore);
    DisplayPlayerScore;
  end;

end;

procedure BackgroundEffect;
begin
  { colors for background effect }
  ColorBackground := (ColorBackground + 1) mod 20;

  if ((ColorBackground mod 2) = 0) then
    { cycle the colors for background effect, starting at 236, 20 colors }
    CycleColors(236, 19);
end;

begin
  StartLevel := 1;
  Difficulty := 1;
  SpecialMode := sm_None;
end.