/*
SRShelfBar.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRShelfBar.h"

@interface SRShelfBar (private)

// Buttons
- (void)_updaetLeftButtons;
- (void)_updaetRightButtons;

@end

@implementation SRShelfBar

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _type = SRShelfTypeShelf;
    _attrTitle = nil;
    _leftButtons = [[NSMutableArray array] retain];
    _rightButtons = [[NSMutableArray array] retain];
    
    return self;
}

- (void)dealloc
{
    [_attrTitle release], _attrTitle = nil;
    [_leftButtons release], _leftButtons = nil;
    [_rightButtons release], _rightButtons = nil;
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- Attributes --
//--------------------------------------------------------------//

- (NSString*)title
{
    return [_attrTitle string];
}

- (void)setTitle:(NSString*)title
{
    [_attrTitle release], _attrTitle = nil;
    if (!title) {
        return;
    }
    
    static NSDictionary*    _attrs = nil;
    if (!_attrs) {
        NSShadow*   shadow;
        shadow = [[NSShadow alloc] init];
        [shadow setShadowColor:[NSColor blackColor]];
        [shadow setShadowOffset:NSMakeSize(0, -1)];
        [shadow setShadowBlurRadius:0.0f];
        
        _attrs = [[NSDictionary dictionaryWithObjectsAndKeys:
                [NSColor whiteColor], NSForegroundColorAttributeName, 
                [NSFont boldSystemFontOfSize:12.0f], NSFontAttributeName, 
//                shadow, NSShadowAttributeName, 
                nil] retain];
        [shadow release];
    }
    
    // Create attributed title
    _attrTitle = [[NSAttributedString alloc] 
            initWithString:title attributes:_attrs];
    
    // Update itself
    [self setNeedsDisplay:YES];
}

- (SRShelfType)type
{
    return _type;
}

- (void)setType:(SRShelfType)type
{
    if (_type == type) {
        return;
    }
    
    _type = type;
    
    // Update itself
    [self setNeedsDisplay:YES];
}

- (int)titleWidth
{
    return _titleWidth;
}

- (void)setTitleWidth:(int)titleWidth
{
    if (_titleWidth == titleWidth) {
        return;
    }
    
    _titleWidth = titleWidth;
    
    // Update itself
    [self _updaetLeftButtons];
    [self _updaetRightButtons];
    [self setNeedsDisplay:YES];
}

//--------------------------------------------------------------//
#pragma mark -- Buttons --
//--------------------------------------------------------------//

- (void)_updaetLeftButtons
{
    // Get frame
    NSRect  frame;
    frame = [self frame];
    
    // Update left buttons
    NSEnumerator*   enumerator;
    NSView*         button;
    int             x;
    x = _titleWidth - 9;
    enumerator = [_leftButtons reverseObjectEnumerator];
    while (button = [enumerator nextObject]) {
        // For divider
        if ([button isKindOfClass:[NSImageView class]]) {
            x -= 5;
        }
        
        NSSize  buttonSize;
        buttonSize = [button frame].size;
        
        x -= buttonSize.width + 2;
        
        NSRect  buttonFrame;
        buttonFrame.origin.x = x;
        buttonFrame.origin.y = floor((frame.size.height - buttonSize.height) / 2.0f);
        buttonFrame.size = buttonSize;
        
        [button setFrame:buttonFrame];
        [self addSubview:button];
        
        // For divider
        if ([button isKindOfClass:[NSImageView class]]) {
            x -= 7;
        }
    }
}

- (void)_updaetRightButtons
{
    // Get frame
    NSRect  frame;
    frame = [self frame];
    
    // Update right buttons
    NSEnumerator*   enumerator;
    NSView*         button;
    int             x;
    x = _titleWidth + 9;
    enumerator = [_rightButtons objectEnumerator];
    while (button = [enumerator nextObject]) {
        // For divider
        if ([button isKindOfClass:[NSImageView class]]) {
            x += 5;
        }
        
        NSSize  buttonSize;
        buttonSize = [button frame].size;
        
        NSRect  buttonFrame;
        buttonFrame.origin.x = x;
        buttonFrame.origin.y = floor((frame.size.height - buttonSize.height) / 2.0f);
        buttonFrame.size = buttonSize;
        
        [button setFrame:buttonFrame];
        [self addSubview:button];
        
        x += buttonFrame.size.width + 2;
        
        // For divider
        if ([button isKindOfClass:[NSImageView class]]) {
            x += 7;
        }
    }
}

- (NSArray*)leftButtons
{
    return _leftButtons;
}

- (void)setLeftButtons:(NSArray*)buttons
{
    // Clear old buttons
    [_leftButtons removeAllObjects];
    
    // Copy new buttons
    if (buttons) {
        [_leftButtons addObjectsFromArray:buttons];
    }
    
    // Update buttons
    [self _updaetLeftButtons];
}

- (NSArray*)rightButtons
{
    return _rightButtons;
}

- (void)setRightButtons:(NSArray*)buttons
{
    // Clear old buttons
    [_rightButtons makeObjectsPerformSelector:@selector(removeFromSuperview)];
    [_rightButtons removeAllObjects];
    
    // Copy new buttons
    if (buttons) {
        [_rightButtons addObjectsFromArray:buttons];
    }
    
    // Update button frames
    [self _updaetRightButtons];
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (NSImage*)imageForType:(SRShelfType)type
{
    switch (type) {
    case SRShelfTypeOcean: {
        return [NSImage imageNamed:@"shelfBarOcean"];
    }
    case SRShelfTypeGraphite: {
        return [NSImage imageNamed:@"shelfBarGraphite"];
    }
    case SRShelfTypeShelf: {
        return [NSImage imageNamed:@"shelfBarShelf"];
    }
    }
    
    return nil;
}

- (void)drawRect:(NSRect)rect
{
    // Get bounds
    NSRect  bounds;
    bounds = [self bounds];
    
    // Decide image
    NSImage*    image;
    image = [self imageForType:_type];
    
    // Draw image
    int     x = 0;
    NSRect  srcRect, destRect;
    srcRect.origin = NSZeroPoint;
    srcRect.size = [image size];
    destRect.origin.y = 0;
    destRect.size = srcRect.size;
    while (x < bounds.size.width) {
        destRect.origin.x = x;
        [image drawInRect:destRect fromRect:srcRect operation:NSCompositeCopy fraction:1.0f];
        
        x += srcRect.size.width;
    }
    
    // Draw title
    if (_attrTitle) {
        [_attrTitle drawAtPoint:NSMakePoint(8, 6)];
    }
    
#if 0
    // Draw divider
    if (_titleWidth > 0) {
        NSImage*    dividerImage;
        dividerImage = [NSImage imageNamed:@"shelfBarDivider"];
        
        srcRect.origin = NSZeroPoint;
        srcRect.size = [dividerImage size];
        destRect.origin.x = _titleWidth;
        destRect.origin.y = 4;
        destRect.size = srcRect.size;
        [dividerImage drawInRect:destRect fromRect:srcRect operation:NSCompositeSourceOver fraction:1.0f];
    }
#endif
}

@end
