/*
SRURLAutoComplete.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRURLAutoComplete.h"

#import "FoundationEx.h"
#import "WebKitEx.h"

static NSMutableArray*  _historyURLs = nil;
//static NSMutableArray*  _comboBoxes = nil;
static NSMutableArray*  _URLAutoCompleteDataSources = nil;

@implementation SRURLAutoCompleteDataSource

#pragma mark -
//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

//use +URLAutoCompleteDataSource and -invalidate
//instead of -init and -release

+ (SRURLAutoCompleteDataSource*)URLAutoCompleteDataSource
{
	SRURLAutoCompleteDataSource*	src= [[[SRURLAutoCompleteDataSource alloc] init]autorelease];
	if (!_URLAutoCompleteDataSources) {
		_URLAutoCompleteDataSources = [[NSMutableArray array] retain];
	}
	[_URLAutoCompleteDataSources addObject:src];
	return src;
}

- (void)invalidate
{
    [_URLAutoCompleteDataSources removeObject:self];
}

+ (void)_addURLStringToHistory:(NSString *)URLString
{
    // Add domain name
    if([URLString hasPrefix:@"http://"]) {
        NSString*   hostString;
        NSRange     slashRange;
        
        slashRange = [URLString rangeOfString:@"/"
                                      options:NSLiteralSearch
                                        range:NSMakeRange(7,[URLString length]-7)];
        
        if(slashRange.length) {
            hostString = [URLString substringToIndex:slashRange.location];
        }
        else {
            hostString = nil;
        }
        
        if((hostString != nil) && ![_historyURLs containsObject:hostString]) {
            [_historyURLs addObject:hostString];
        }
    }
    
    // Add URL string
    if (![_historyURLs containsObject:URLString]) {
        [_historyURLs addObject:URLString];
    }
}

- (void)_copyHistoryURLsForAutoComplete
{
    if (!_historyURLs) {
        _historyURLs = [[NSMutableArray array] retain];
        
        // Get shared history
        WebHistory* history;
        history = [WebHistory optionalSharedHistory];
        
        // Copy history URLs
        NSArray*        days;
        NSEnumerator*   dayEnum;
        NSCalendarDate* day;
        days = [history orderedLastVisitedDays];
        dayEnum = [days objectEnumerator];
        
        while (day = [dayEnum nextObject]) {
            NSArray*        items;
            NSEnumerator*   itemEnum;
            WebHistoryItem* item;
            items = [history orderedItemsLastVisitedOnDay:day];
            itemEnum = [items objectEnumerator];
            
            while (item = [itemEnum nextObject]) {
                [SRURLAutoCompleteDataSource _addURLStringToHistory:[item URLString]];
            }
        }
    }
    
    [_completedHistoryURLs setArray:_historyURLs];
}

- (id)init
{
    self = [super init];
    if (self) {
        // Initialize instance variables
        _completedHistoryURLs = [[NSMutableArray array] retain];
        _URLStringDict = [[NSMutableDictionary dictionary] retain];
        
        [self _copyHistoryURLsForAutoComplete];
        
        // Register notification
        static BOOL _isRegistered = NO;
        if (!_isRegistered) {
            NSNotificationCenter*   center;
            center = [NSNotificationCenter defaultCenter];
            
            [center addObserver:[self class] 
                    selector:@selector(webHistoryItemsAdded:) 
                    name:WebHistoryItemsAddedNotification 
                    object:nil];
            [center addObserver:[self class] 
                    selector:@selector(webHistoryItemsRemoved:) 
                    name:WebHistoryItemsRemovedNotification 
                    object:nil];
            [center addObserver:[self class] 
                    selector:@selector(webHistoryAllItemsRemoved:) 
                    name:WebHistoryAllItemsRemovedNotification 
                    object:nil];
            
            _isRegistered = YES;
        }
    }
    
    return self;
}

- (void)dealloc
{
    [_completedHistoryURLs release];
    [_URLStringDict release];
    
    [super dealloc];
}

- (void)setComboBox:(NSComboBox*)comboBox
{
	_comboBox = comboBox;
}

- (NSComboBox*)comboBox
{
	return _comboBox;
}

- (NSString*)absoluteURLStringOfCompletedString:(NSString*)completedString
{
    // Sanitize string
    // Remove new line characters
    NSCharacterSet* newLineCharacterSet;
    newLineCharacterSet = [NSCharacterSet newLineCharacterSet];
    if ([completedString rangeOfCharacterFromSet:newLineCharacterSet 
            options:NSLiteralSearch].location != NSNotFound)
    {
        completedString = [completedString stringByTrimmingCharactersInSet:newLineCharacterSet];
    }
    
#if 1
    // Remove last white spaces
    int length;
    for (length = [completedString length]; length > 0; length--) {
        unichar c;
        c = [completedString characterAtIndex:length - 1];
        if (c != 0x0020 && c != 0x0009) { // space and tab
            break;
        }
    }
    if (length < [completedString length]) {
        completedString = [completedString substringWithRange:NSMakeRange(0, length)];
    }
#else
    completedString = [completedString stringByAddingPercentEscapesUsingEncoding:NSASCIIStringEncoding];
#endif
    
    // For empty string
    if ([completedString length] == 0) {
        return nil;
    }
    
    // If passed string is well-formed, use it
    NSURL*  url;
    url = [NSURL _web_URLWithUserTypedString:completedString];
    if (url && [url scheme]) {
        return completedString;
    }
    
    // Get URL string from dictioanry
    NSString*   URLString;
    URLString = [_URLStringDict objectForKey:completedString];
    if (!URLString) {
        NSString *firstComponent = [[completedString pathComponents] objectAtIndex:0];
        
        if ([firstComponent rangeOfString:@"."].length) {
            URLString = [NSString stringWithFormat:@"http://%@", completedString];
        }
        else {
            NSRange pathRange = NSMakeRange([firstComponent length],
                                            [completedString length] - [firstComponent length]);
            NSString *relativePath = [completedString substringWithRange:pathRange];
            URLString = [NSString stringWithFormat:@"http://www.%@.com%@", firstComponent, relativePath];
        }
    }
    
    return URLString;
}

#pragma mark -
//--------------------------------------------------------------//
// WebHistory notification
//--------------------------------------------------------------//
//noteNumberOfItemsChanged is unnecessarily


+ (void)webHistoryItemsAdded:(NSNotification*)notification
{
    // Get items
    NSArray*    items;
    items = [[notification userInfo] objectForKey:WebHistoryItemsKey];
    
    // Add items
    NSEnumerator*   enumerator;
    WebHistoryItem* item;
    enumerator = [items objectEnumerator];
    while (item = [enumerator nextObject]) {
        // Get URL string
        NSString*   URLString;
        URLString = [item URLString];
        
        [self _addURLStringToHistory:URLString];
    }
}

+ (void)webHistoryItemsRemoved:(NSNotification*)notification
{
    // Get items
    NSArray*    items;
    items = [[notification userInfo] objectForKey:WebHistoryItemsKey];
    
    // Remove items
    NSEnumerator*   enumerator;
    WebHistoryItem* item;
    enumerator = [items objectEnumerator];
    while (item = [enumerator nextObject]) {
        // Get URL string
        NSString*   URLString;
        URLString = [item URLString];
        
        [_historyURLs removeObject:URLString];
    }
}

+ (void)webHistoryAllItemsRemoved:(NSNotification*)notification
{
    // Remove all objects
    [_historyURLs removeAllObjects];
}

#pragma mark -
//--------------------------------------------------------------//
// NSComboBox data source
//--------------------------------------------------------------//

- (int)numberOfItemsInComboBox:(NSComboBox*)comboBox
{
    return [_completedHistoryURLs count];
}

- (id)comboBox:(NSComboBox*)comboBox objectValueForItemAtIndex:(int)index
{
    if (index < 0 || index >= [_completedHistoryURLs count]) {
        return [comboBox stringValue];
    }
    return [_completedHistoryURLs objectAtIndex:index];
}

- (NSString*)comboBox:(NSComboBox*)comboBox completedString:(NSString*)uncompletedString
{
    // Remove all objects in completed list
    [_completedHistoryURLs removeAllObjects];
    [_URLStringDict removeAllObjects];
    
    // Check all URLs
    NSMutableArray*         URLs;
    NSMutableDictionary*    completedURLStringDict;
    NSEnumerator*           URLEnum;
    NSString*               URLString;
    URLs = [NSMutableArray array];
    completedURLStringDict = [NSMutableDictionary dictionary];
    URLEnum = [_historyURLs objectEnumerator];
    while (URLString = [URLEnum nextObject]) {
        
        // Case uncomplete string starts with 'http://'
        if ([uncompletedString hasPrefix:@"http://"]) {
            if ([URLString hasPrefix:uncompletedString]) {
                [URLs addObject:URLString];
                [_URLStringDict setObject:URLString forKey:uncompletedString];
                [completedURLStringDict setObject:uncompletedString forKey:URLString];
            }
        }
        else {
            // Make URL string without 'http://'
            NSString*   URLStringWithoutHttp;
            URLStringWithoutHttp = [URLString substringFromIndex:7];
            if ([URLStringWithoutHttp hasPrefix:uncompletedString]) {
                [URLs addObject:URLString];
                [_URLStringDict setObject:URLString forKey:URLStringWithoutHttp];
                [completedURLStringDict setObject:URLStringWithoutHttp forKey:URLString];
            }
            // Make URL string without 'http://www.'
            else if ([URLString hasPrefix:@"http://www."]) {
                NSString*   URLStringWithoutWWW;
                URLStringWithoutWWW = [URLString substringFromIndex:11];
                if ([URLStringWithoutWWW hasPrefix:uncompletedString]) {
                    [URLs addObject:URLString];
                    [_URLStringDict setObject:URLString forKey:URLStringWithoutWWW];
                    [completedURLStringDict setObject:URLStringWithoutWWW forKey:URLString];
                }
            }
        }
    }
    
    // Sort array
    [_completedHistoryURLs setArray:[URLs 
            sortedArrayUsingSelector:@selector(URLStringCompare:)]];
    
    // Notify the change of number of items
    [comboBox noteNumberOfItemsChanged];
    
    // Check number of completed URLs
    if ([_completedHistoryURLs count] == 0) {
        return uncompletedString;
    }
    
    return [completedURLStringDict objectForKey:[_completedHistoryURLs objectAtIndex:0]];
}

@end

#pragma mark -
//--------------------------------------------------------------//
// NSString URL comparisoin extension
//--------------------------------------------------------------//

@implementation NSString (SRURLComparisonMethods)

- (NSComparisonResult)URLStringCompare:(NSString *)string
{
    NSString *ourString = self;
    NSString *otherString = string;
    
    if([self hasPrefix:@"http://www."]) {
        ourString = [self substringWithRange:NSMakeRange(11,[self length]-11)];
    }
    else if([self hasPrefix:@"http://"]) {
        ourString = [self substringWithRange:NSMakeRange(7,[self length]-7)];
    }
    
    if([string hasPrefix:@"http://www."]) {
        otherString = [string substringWithRange:NSMakeRange(11,[string length]-11)];
    }
    else if([string hasPrefix:@"http://"]) {
        otherString = [string substringWithRange:NSMakeRange(7,[string length]-7)];
    }
    
    return [ourString caseInsensitiveCompare:otherString];
}

@end
