/*
SRContextMenu.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRContextMenu.h"

static NSMenu*  _s_webViewContextMenu = nil;
static NSMenu*  _s_bookmarkContextMenu = nil;
static NSMenu*  _s_historyContextMenu = nil;
static NSMenu*  _s_downloadContextMenu = nil;
static NSMenu*  _s_tabContextMenu = nil;

@implementation SRContextMenu

//--------------------------------------------------------------//
// Initialize
//--------------------------------------------------------------//

+ (void)_loadContextMenu
{
    // Load context menu
    SRContextMenu*  contextMenu;
    contextMenu = [[SRContextMenu alloc] init];
    if (![NSBundle loadNibNamed:@"ContextMenu" owner:contextMenu]) {
        // Fatal
        SR_FATAL(@"Could not load ContextMenu.nib");
    }
}

- (void)awakeFromNib
{
    // Set to static variables
    _s_webViewContextMenu = _webViewContextMenu;
    _s_bookmarkContextMenu = _bookmarkContextMenu;
    _s_historyContextMenu = _historyContextMenu;
    _s_downloadContextMenu = _downloadContextMenu;
    _s_tabContextMenu = _tabContextMenu;
}

//--------------------------------------------------------------//
// Accessing context menu
//--------------------------------------------------------------//

+ (NSMenu*)webViewContextMenu
{
    if (!_s_webViewContextMenu) {
        [self _loadContextMenu];
    }
    
    return _s_webViewContextMenu;
}

+ (NSMenu*)bookmarkContextMenu
{
    if (!_s_bookmarkContextMenu) {
        [self _loadContextMenu];
    }
    
    return _s_bookmarkContextMenu;
}

+ (NSMenu*)historyContextMenu
{
    if (!_s_historyContextMenu) {
        [self _loadContextMenu];
    }
    
    return _s_historyContextMenu;
}

+ (NSMenu*)downloadContextMenu
{
    if (!_s_downloadContextMenu) {
        [self _loadContextMenu];
    }
    
    return _s_downloadContextMenu;
}

+ (NSMenu*)tabContextMenu
{
    if (!_s_tabContextMenu) {
        [self _loadContextMenu];
    }
    
    return _s_tabContextMenu;
}

//--------------------------------------------------------------//
// Copying menu
//--------------------------------------------------------------//

+ (NSMenu*)copyMenuFrom:(NSMenu*)menu 
        ofTags:(NSArray*)tags 
        target:(id)target
{
    // Create menu
    NSMenu* newMenu;
    newMenu = [[NSMenu alloc] init];
    [newMenu autorelease];
    
    // Check menu items
    int i;
    for (i = 0; i < [menu numberOfItems]; i++) {
        // Get menu item
        id<NSMenuItem>  menuItem;
        SR_VERIFY(i >= 0, @"menu index is less than 0");
        menuItem = [menu itemAtIndex:i];
        
        // Case of separator item
        if ([menuItem isSeparatorItem]) {
            int numberOfItems;
            numberOfItems = [newMenu numberOfItems];
            
            // If no menu items, do not add separator
            if (numberOfItems == 0) {
                continue;
            }
            
            // If last item is separator, do not add separtor
            id<NSMenuItem>  lastItem;
            SR_VERIFY(numberOfItems - 1 >= 0, @"menu index is less than 0");
            lastItem = [newMenu itemAtIndex:numberOfItems - 1];
            if ([lastItem isSeparatorItem]) {
                continue;
            }
            
            [newMenu addItem:[NSMenuItem separatorItem]];
            continue;
        }
        
        // Check tag is conteind
        if ([tags containsObject:[NSNumber numberWithInt:[menuItem tag]]]) {
            // Copy menu item
            NSMenuItem* newMenuItem;
            newMenuItem = [[NSMenuItem alloc] 
                    initWithTitle:[menuItem title] 
                            action:[menuItem action] 
                            keyEquivalent:[menuItem keyEquivalent]];
            [newMenuItem autorelease];
            [newMenuItem setTag:[menuItem tag]];
            [newMenuItem setTarget:target];
            
            [newMenu addItem:newMenuItem];
        }
    }
    
    // Get number of items
    int numberOfItems;
    numberOfItems = [newMenu numberOfItems];
    
    // If no items, return nil
    if (numberOfItems == 0) {
        return nil;
    }
    
    // If last item is separator, remove it
    id<NSMenuItem>  lastItem;
    SR_VERIFY(numberOfItems - 1 >= 0, @"menu index is less than 0");
    lastItem = [newMenu itemAtIndex:numberOfItems - 1];
    if ([lastItem isSeparatorItem]) {
        [newMenu removeItemAtIndex:numberOfItems - 1];
    }
    
    return newMenu;
}

+ (NSMenuItem*)copyMenuItemFrom:(NSMenu*)menu 
        ofTag:(int)tag 
        target:(id)target
{
    // Get menu item
    id<NSMenuItem>  menuItem;
    menuItem = [menu itemWithTag:tag];
    if (!menuItem) {
        return nil;
    }
    
    // Copy menu item
    NSMenuItem* newMenuItem;
    newMenuItem = [[NSMenuItem alloc] 
            initWithTitle:[menuItem title] 
                    action:[menuItem action] 
                    keyEquivalent:[menuItem keyEquivalent]];
    [newMenuItem autorelease];
    [newMenuItem setTag:[menuItem tag]];
    [newMenuItem setTarget:target];
    
    return newMenuItem;
}

//--------------------------------------------------------------//
// Menu item utitlity
//--------------------------------------------------------------//
+ (void)insertMenuItem:(NSMenuItem*)menuItem inItems:(NSMutableArray*)items afterTag:(int)tag
{
    NSEnumerator*   enumerator;
    NSMenuItem*     item;
    enumerator = [items objectEnumerator];
    while (item = [enumerator nextObject]) {
        if ([item tag] == tag) {
            break;
        }
    }
    
    if (item) {
        [items insertObject:menuItem atIndex:[items indexOfObject:item] + 1];
    }
    else {
        [items addObject:menuItem];
    }
}

+ (void)removeMenuItem:(NSMenuItem*)menuItem fromItems:(NSMutableArray*)items ofTag:(int)tag
{
    NSEnumerator*   enumerator;
    NSMenuItem*     item;
    enumerator = [items objectEnumerator];
    while (item = [enumerator nextObject]) {
        if ([item tag] == tag) {
            break;
        }
    }
    
    if (item) {
        [items removeObject:item];
    }
}

@end
