package org.seasar.transaction;

import java.util.ArrayList;
import java.util.List;

import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.RollbackException;
import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

import org.seasar.log.Logger;
import org.seasar.message.MessageFormatter;
import org.seasar.util.SeasarRuntimeException;

public final class TransactionImpl implements Transaction {

    private static final int VOTE_READONLY = 0;
    private static final int VOTE_COMMIT = 1;
    private static final int VOTE_ROLLBACK = 2;
    private static Logger _logger = Logger.getLogger(TransactionImpl.class);
    private Xid _xid;
    private int _status = Status.STATUS_NO_TRANSACTION;
    private List _xaResourceInfoList = new ArrayList();
    private List _synchronizationList = new ArrayList();
    private boolean _suspended = false;
    private int _branchId = 0;

    public TransactionImpl(final Xid xid) {
        init(xid);
    }

    public void begin() throws SystemException {
        if (_logger.isDebugEnabled()) {
        	_logger.debug("Transaction.begin()");
        }
    	_status = Status.STATUS_ACTIVE;
        List list = XAResourceManager.getInstance().getXAResourceList();
        if (list == null) {
            return;
        }
        try {
            for (int i = 0; i < list.size(); i++) {
                try {
                    enlistResource((XAResource) list.get(i));
                } catch (RollbackException ex) {
                    Logger.getLogger(getClass()).log(ex);
                }
            }
        } finally {
            list.clear();
        }
    }

    public void suspend() {
        checkIsNotSuspended();
        checkStatus();
        for (int i = 0; i < _xaResourceInfoList.size(); i++) {
            XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(i);
            try {
                xari.end(XAResource.TMSUSPEND);
            } catch (XAException ex) {
                Logger.getLogger(getClass()).log(ex);
            }
        }
        _suspended = true;
    }

    public void resume() {
        checkIsSuspended();
        for (int i = 0; i < _xaResourceInfoList.size(); i++) {
            XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(i);
            try {
                xari.start(XAResource.TMRESUME);
            } catch (XAException ex) {
                Logger.getLogger(getClass()).log(ex);
            }
        }
        _suspended = false;
    }

    public void commit() throws RollbackException, HeuristicMixedException,
            HeuristicRollbackException, SecurityException, IllegalStateException,
            SystemException {

        if (_logger.isDebugEnabled()) {
        	_logger.debug("Transaction.commit()");
        }
        checkIsNotSuspended();
        checkStatus();
        beforeCompletion();
        endResources(XAResource.TMSUCCESS);
        if (_status == Status.STATUS_ACTIVE) {
            if (_xaResourceInfoList.size() == 0) {
                _status = Status.STATUS_COMMITTED;
            } else if (_xaResourceInfoList.size() == 1) {
                commitOnePhase();
            } else {
                switch (prepareResources()) {
                    case VOTE_READONLY:
                        _status = Status.STATUS_COMMITTED;
                        break;
                    case VOTE_COMMIT:
                        commitTwoPhase();
                        break;
                    case VOTE_ROLLBACK:
                        rollbackForVoteOK();
                        afterCompletion();
                        clear();
                        throw new RollbackException(
                                MessageFormatter.getMessage("ESSR0303", new Object[]{toString()}));
                }
            }
        }
        afterCompletion();
        clear();
    }

    public void rollback() throws IllegalStateException, SecurityException,
            SystemException {

        if (_logger.isDebugEnabled()) {
        	_logger.debug("Transaction.rollback()");
        }
        checkIsNotSuspended();
        checkStatusForRollback();
        beforeCompletion();
        endResources(XAResource.TMFAIL);
        rollbackResources();
        afterCompletion();
        clear();
    }

    public void setRollbackOnly() throws IllegalStateException, SystemException {
        checkIsNotSuspended();
        checkStatusForSetRollbackOnly();
        _status = Status.STATUS_MARKED_ROLLBACK;
    }

    public boolean enlistResource(final XAResource xaResource) throws RollbackException,
            IllegalStateException, SystemException {

        if (xaResource == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"xaResource"});
        }
        checkIsNotSuspended();
        checkStatusForEnlistResource();
        Xid xid = null;
        for (int i = 0; i < _xaResourceInfoList.size(); i++) {
            XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(i);
            if (xaResource.equals(xari._xaResource)) {
                return false;
            } else if (xaResource.getClass().getName().startsWith("oracle")) {
                continue;
            } else {
                try {
                    if (xaResource.isSameRM(xari._xaResource)) {
                        xid = xari._xid;
                        break;
                    }
                } catch (XAException ex) {
                    throw new IllegalStateException(ex.toString());
                }
            }
        }
        int flag = xid == null ? XAResource.TMNOFLAGS : XAResource.TMJOIN;
        boolean commitTarget = xid == null ? true : false;
        if (xid == null) {
            xid = createXidBranch();
        }
        try {
            xaResource.start(xid, flag);
            _xaResourceInfoList.add(new XAResourceInfo(xaResource, xid, commitTarget));
            return true;
        } catch (XAException ex) {
            throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
        }
    }

    public boolean delistResource(final XAResource xaResource, final int flag)
             throws IllegalStateException, SystemException {

        if (xaResource == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"xaResource"});
        }
        checkIsNotSuspended();
        checkStatusForDelistResource();
        for (int i = 0; i < _xaResourceInfoList.size(); i++) {
            XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(i);
            if (xaResource.equals(xari._xaResource)) {
                try {
                    xari.end(flag);
                    return true;
                } catch (XAException ex) {
                    Logger.getLogger(getClass()).log(ex);
                    _status = Status.STATUS_MARKED_ROLLBACK;
                    return false;
                }
            }
        }
        throw new SeasarRuntimeException("ESSR0001", new Object[]{xaResource});
    }

    public int getStatus() {
        return _status;
    }

    public void registerSynchronization(final Synchronization sync)
             throws RollbackException, IllegalStateException, SystemException {

        if (sync == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"sync"});
        }
        checkIsNotSuspended();
        checkStatusForRegisterSynchronization();
        _synchronizationList.add(sync);
    }

    public Xid getXid() {
        return _xid;
    }

    public boolean isSuspended() {
        return _suspended;
    }

    public int hashCode() {
        return _xid.hashCode();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof TransactionImpl)) {
            return false;
        }
        return _xid.equals(((TransactionImpl) o)._xid);
    }

    public String toString() {
        return _xid.toString();
    }

    public void init(final Xid xid) {
        _xid = xid;
        _xaResourceInfoList.clear();
        _synchronizationList.clear();
        _suspended = false;
    }
    
    public void destroy() {
        _status = Status.STATUS_NO_TRANSACTION;
    }

    private void endResources(final int flag) {
        for (int i = 0; i < _xaResourceInfoList.size(); i++) {
            XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(i);
            try {
                xari.end(flag);
            } catch (XAException ex) {
                Logger.getLogger(TransactionImpl.class).log("ESSR0017", new Object[]{ex}, ex);
                _status = Status.STATUS_MARKED_ROLLBACK;
            }
        }
    }

    private void beforeCompletion() {
        for (int i = 0; i < _synchronizationList.size(); i++) {
            Synchronization sync = (Synchronization) _synchronizationList.get(i);
            try {
                sync.beforeCompletion();
            } catch (Throwable t) {
                Logger.getLogger(TransactionImpl.class).log("ESSR0017", new Object[]{t}, t);
                _status = Status.STATUS_MARKED_ROLLBACK;
                break;
            }
        }
    }

    private void afterCompletion() {
        for (int i = 0; i < _synchronizationList.size(); i++) {
            Synchronization sync = (Synchronization) _synchronizationList.get(i);
            try {
                sync.afterCompletion(_status);
            } catch (Throwable t) {
                Logger.getLogger(TransactionImpl.class).log("ESSR0017", new Object[]{t}, t);
            }
        }
    }

    private void clear() {
        _synchronizationList.clear();
        _xaResourceInfoList.clear();
    }

    private int prepareResources() {
        _status = Status.STATUS_PREPARING;
        int vote = VOTE_READONLY;
        for (int i = 0; i < _xaResourceInfoList.size(); i++) {
            XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(i);
            if (!xari._commitTarget) {
                continue;
            }
            try {
                if (xari.prepare() == XAResource.XA_OK) {
                    xari._voteOk = true;
                    vote = VOTE_COMMIT;
                }
            } catch (XAException ex) {
                _status = Status.STATUS_MARKED_ROLLBACK;
                return VOTE_ROLLBACK;
            }
        }
        if (_status == Status.STATUS_PREPARING) {
            _status = Status.STATUS_PREPARED;
        }
        return vote;
    }

    private void commitOnePhase() {
        _status = Status.STATUS_COMMITTING;
        XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(0);
        try {
            xari.commit(true);
        } catch (XAException ex) {
            Logger.getLogger(getClass()).log(ex);
            _status = Status.STATUS_UNKNOWN;
        }
        if (_status == Status.STATUS_COMMITTING) {
            _status = Status.STATUS_COMMITTED;
        }
    }

    private void commitTwoPhase() {
        _status = Status.STATUS_COMMITTING;
        for (int i = 0; i < _xaResourceInfoList.size(); i++) {
            XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(i);
            if (!xari._commitTarget || !xari._voteOk) {
                continue;
            }
            try {
                xari.commit(false);
            } catch (XAException ex) {
                Logger.getLogger(getClass()).log(ex);
                _status = Status.STATUS_UNKNOWN;
            }
        }
        if (_status == Status.STATUS_COMMITTING) {
            _status = Status.STATUS_COMMITTED;
        }
    }

    private void rollbackForVoteOK() {
        _status = Status.STATUS_ROLLING_BACK;
        for (int i = 0; i < _xaResourceInfoList.size(); i++) {
            XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(i);
            try {
                if (xari._voteOk) {
                    xari.rollback();
                }
            } catch (XAException ex) {
                Logger.getLogger(TransactionImpl.class).log("ESSR0017", new Object[]{ex}, ex);
                _status = Status.STATUS_UNKNOWN;
            }
        }
        if (_status == Status.STATUS_ROLLING_BACK) {
            _status = Status.STATUS_ROLLEDBACK;
        }
    }

    private void rollbackResources() {
        _status = Status.STATUS_ROLLING_BACK;
        for (int i = 0; i < _xaResourceInfoList.size(); i++) {
            XAResourceInfo xari = (XAResourceInfo) _xaResourceInfoList.get(i);
            try {
                if (xari._commitTarget) {
                    xari.rollback();
                }
            } catch (XAException ex) {
                Logger.getLogger(TransactionImpl.class).log("ESSR0017", new Object[]{ex}, ex);
                _status = Status.STATUS_UNKNOWN;
            }
        }
        if (_status == Status.STATUS_ROLLING_BACK) {
            _status = Status.STATUS_ROLLEDBACK;
        }
    }

    private Xid createXidBranch() {
        return new XidImpl(_xid, ++_branchId);
    }

    private void checkStatus() throws IllegalStateException {
        switch (_status) {
            case Status.STATUS_ACTIVE:
                break;
            case Status.STATUS_PREPARING:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0304", null));
            case Status.STATUS_PREPARED:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0305", null));
            case Status.STATUS_COMMITTING:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0306", null));
            case Status.STATUS_COMMITTED:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0307", null));
            case Status.STATUS_MARKED_ROLLBACK:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0308", null));
            case Status.STATUS_ROLLING_BACK:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0309", null));
            case Status.STATUS_ROLLEDBACK:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0310", null));
            case Status.STATUS_NO_TRANSACTION:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0311", null));
            case Status.STATUS_UNKNOWN:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0312", null));
            default:
                throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0313", new Object[]{String.valueOf(_status)}));
        }
    }

    private void checkStatusForRollback() throws IllegalStateException {
        switch (_status) {
            case Status.STATUS_ACTIVE:
            case Status.STATUS_MARKED_ROLLBACK:
                break;
            default:
                checkStatus();
        }
    }

    private void checkStatusForEnlistResource() throws IllegalStateException {
        switch (_status) {
            case Status.STATUS_ACTIVE:
            case Status.STATUS_PREPARING:
                break;
            default:
                checkStatus();
        }
    }

    private void checkStatusForDelistResource() throws IllegalStateException {
        switch (_status) {
            case Status.STATUS_ACTIVE:
            case Status.STATUS_MARKED_ROLLBACK:
                break;
            default:
                checkStatus();
        }
    }

    private void checkStatusForSetRollbackOnly() throws IllegalStateException {
        switch (_status) {
            case Status.STATUS_ACTIVE:
            case Status.STATUS_PREPARING:
            case Status.STATUS_PREPARED:
                break;
            default:
                checkStatus();
        }
    }

    private void checkStatusForRegisterSynchronization() throws IllegalStateException {
        switch (_status) {
            case Status.STATUS_ACTIVE:
            case Status.STATUS_PREPARING:
                break;
            default:
                checkStatus();
        }
    }

    private void checkIsNotSuspended() throws IllegalStateException {
        if (_suspended) {
            throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0314", null));
        }
    }

    private void checkIsSuspended() throws IllegalStateException {
        if (!_suspended) {
            throw new IllegalStateException(
                        MessageFormatter.getMessage("ESSR0315", null));
        }
    }

    private class XAResourceInfo {

        private final XAResource _xaResource;
        private final Xid _xid;
        private final boolean _commitTarget;
        private boolean _voteOk = false;

        XAResourceInfo(final XAResource xaResource, final Xid xid, final boolean commitTarget) {
            _xaResource = xaResource;
            _xid = xid;
            _commitTarget = commitTarget;
        }

        void start(final int flag) throws XAException {
            _xaResource.start(_xid, flag);
        }

        void end(final int flag) throws XAException {
            _xaResource.end(_xid, flag);
        }

        int prepare() throws XAException {
            return _xaResource.prepare(_xid);
        }

        void commit(final boolean onePhase) throws XAException {
            _xaResource.commit(_xid, onePhase);
        }

        void rollback() throws XAException {
            _xaResource.rollback(_xid);
        }

        void forget() throws XAException {
            _xaResource.forget(_xid);
        }
    }
}
