package org.seasar.transaction;

import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

import org.seasar.message.MessageFormatter;

public abstract class AbstractXAResource implements XAResource {

    public final static int RS_NONE = 0;
    public final static int RS_ACTIVE = 1;
    public final static int RS_SUSPENDED = 2;
    public final static int RS_FAIL = 3;
    public final static int RS_SUCCESS = 4;
    public final static int RS_PREPARED = 5;

    private Xid _currentXid;
    private int _status = RS_NONE;
    private int _timeout = 0;

    public AbstractXAResource() {
    }

    public void start(final Xid xid, final int flags) throws XAException {
        switch (flags) {
            case TMNOFLAGS:
                checkNullCurrentXid();
                doBegin(xid);
                _currentXid = xid;
                _status = RS_ACTIVE;
                break;
            case TMRESUME:
                checkSameCurrentXid(xid);
                checkStatusForResume();
                doResume(xid);
                _status = RS_ACTIVE;
                break;
            case TMJOIN:
                throw new XAException(MessageFormatter.getMessage("ESSR0317", null));
            default:
                throw new XAException(MessageFormatter.getMessage("ESSR0022",
                        new Object[]{String.valueOf(flags)}));
        }
    }

    public void end(final Xid xid, final int flags) throws XAException {
        checkStatusIsActive();
        checkSameCurrentXid(xid);
        switch (flags) {
            case TMSUSPEND:
                doSuspend(xid);
                _status = RS_SUSPENDED;
                break;
            case TMFAIL:
                doFail(xid);
                _status = RS_FAIL;
                break;
            case TMSUCCESS:
                doSuccess(xid);
                _status = RS_SUCCESS;
                break;
            default:
                throw new XAException(MessageFormatter.getMessage("ESSR0022",
                        new Object[]{String.valueOf(flags)}));
        }
    }

    public void commit(final Xid xid, final boolean onePhase) throws XAException {
        if (onePhase) {
            checkStatusIsSuccess();
        } else {
            checkStatusForTwoPhaseCommit();
        }
        checkSameCurrentXid(xid);
        doCommit(xid, onePhase);
        init();
    }

    public void forget(final Xid xid) throws XAException {
        checkSameCurrentXid(xid);
        doForget(xid);
        init();
    }

    public int prepare(final Xid xid) throws XAException {
        checkStatusIsSuccess();
        checkSameCurrentXid(xid);
        int ret = doPrepare(xid);
        if (ret == XA_OK) {
            _status = RS_PREPARED;
        } else {
            _status = RS_NONE;
        }
        return ret;
    }

    public Xid[] recover(final int flag) throws XAException {
        return null;
    }

    public void rollback(final Xid xid) throws XAException {
        checkStatusForRollback();
        checkSameCurrentXid(xid);
        doRollback(xid);
        init();
    }

    public boolean isSameRM(final XAResource xar) throws XAException {
        return false;
    }

    public int getTransactionTimeout() throws XAException {
        return _timeout;
    }

    public boolean setTransactionTimeout(final int timeout) throws XAException {
        _timeout = timeout;
        return true;
    }

    public Xid getCurrentXid() {
        return _currentXid;
    }

    public int getStatus() {
        return _status;
    }

    protected abstract void doBegin(final Xid xid) throws XAException;

    protected abstract void doResume(final Xid xid) throws XAException;

    protected abstract void doSuspend(final Xid xid) throws XAException;

    protected abstract void doFail(final Xid xid) throws XAException;

    protected abstract void doSuccess(final Xid xid) throws XAException;

    protected abstract void doCommit(final Xid xid, final boolean onePhase) throws XAException;

    protected abstract void doForget(final Xid xid) throws XAException;

    protected abstract int doPrepare(final Xid xid) throws XAException;

    protected abstract void doRollback(final Xid xid) throws XAException;

    private void init() {
        _currentXid = null;
        _status = RS_NONE;
    }

    private void checkNullCurrentXid() throws XAException {
        if (_currentXid != null) {
            throw new XAException(MessageFormatter.getMessage("ESSR0316", null));
        }
    }

    private void checkSameCurrentXid(final Xid xid) throws XAException {
        if (_currentXid != xid) {
            throw new XAException(MessageFormatter.getMessage("ESSR0319", new Object[]{xid, _currentXid}));
        }
    }

    private void checkStatusIsSuccess() throws XAException {
        switch (_status) {
            case RS_SUCCESS:
                break;
            default:
                throw new XAException(MessageFormatter.getMessage("ESSR0320", null));
        }
    }

    private void checkStatusForTwoPhaseCommit() throws XAException {
        switch (_status) {
            case RS_PREPARED:
                break;
            default:
                throw new XAException(MessageFormatter.getMessage("ESSR0321", null));
        }
    }

    private void checkStatusForRollback() throws XAException {
        switch (_status) {
            case RS_FAIL:
            case RS_PREPARED:
                break;
            default:
                throw new XAException(MessageFormatter.getMessage("ESSR0322", null));
        }
    }

    private void checkStatusForResume() throws XAException {
        switch (_status) {
            case RS_SUSPENDED:
                break;
            default:
                throw new XAException(MessageFormatter.getMessage("ESSR0323", null));
        }
    }

    private void checkStatusIsActive() throws XAException {
        switch (_status) {
            case RS_ACTIVE:
                break;
            default:
                throw new XAException(MessageFormatter.getMessage("ESSR0324", null));
        }
    }
}