package org.seasar.sql;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.naming.NamingException;

import org.seasar.log.Logger;
import org.seasar.naming.NamingContext;
import org.seasar.naming.NamingService;
import org.seasar.system.Lifecycle;
import org.seasar.util.ElementHandler;
import org.seasar.util.SeasarException;
import org.seasar.util.XMLHandler;
import org.seasar.util.XMLHandlerParser;
import org.seasar.util.XMLHandlerRule;
import org.xml.sax.Attributes;

public final class ConnectionPoolService implements Lifecycle {

    private static final String XML_FILE = "/connectionpool-config.xml";
    private static XMLHandlerRule _xmlHandlerRule = new XMLHandlerRule();
    private List _connectionPoolMetaDataList;

    public ConnectionPoolService() {
        setupXMLHandlerRule();
    }

    public void start() throws SeasarException {
        _connectionPoolMetaDataList = (List)
        	XMLHandlerParser.parse(XML_FILE, _xmlHandlerRule);
        bindDataSources();
    }

    public void stop() {
    	unbindDataSources();
    	_connectionPoolMetaDataList.clear();
    	_connectionPoolMetaDataList = null;
        ConnectionPoolManager.getInstance().close();
    }

    private static void setupXMLHandlerRule() {
    	_xmlHandlerRule.addElementHandler("/connectionPools",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                    xmlHandler.push(new ArrayList());
                }
            });
        _xmlHandlerRule.addElementHandler("/connectionPools/connectionPool",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                    String dataSourceName = attributes.getValue("dataSourceName");
                    String jndiName = attributes.getValue("jndiName");
                    String xaDataSourceClassName = attributes.getValue("xaDataSourceClassName");
                    int timeout = Integer.parseInt(attributes.getValue("timeout"));
                    int poolSize = Integer.parseInt(attributes.getValue("poolSize"));
                    xmlHandler.push(new ConnectionPoolMetaData(dataSourceName, jndiName,
                            xaDataSourceClassName, timeout, poolSize));
                }
                public void end(XMLHandler xmlHandler, String body) {
                    ConnectionPoolMetaData cpmd = (ConnectionPoolMetaData) xmlHandler.pop();
                    List connectionPoolMetaDataList = (List) xmlHandler.peek();
                    connectionPoolMetaDataList.add(cpmd);
                }
            });
        _xmlHandlerRule.addElementHandler("/connectionPools/connectionPool/properties",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                	ConnectionPoolMetaData cpmd = (ConnectionPoolMetaData) xmlHandler.peek();
                	Properties properties = new Properties();
                    cpmd.setProperties(properties);
                    xmlHandler.push(properties);
                }
                public void end(XMLHandler xmlHandler, String body) {
                    xmlHandler.pop();
                }
            });
        _xmlHandlerRule.addElementHandler("/connectionPools/connectionPool/properties/property",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                    String name = attributes.getValue("name");
                    String value = attributes.getValue("value");
                    Properties properties = (Properties) xmlHandler.peek();
                    properties.setProperty(name, value);
                }
            });
    }
    
    private void bindDataSources() throws SeasarException {
    	NamingContext ctx = NamingService.getNamingServer().getNamingContext();
    	for (int i = 0; i < _connectionPoolMetaDataList.size(); ++i) {
    		ConnectionPoolMetaData cpmd = (ConnectionPoolMetaData) _connectionPoolMetaDataList.get(i);
            DataSourceImpl ds = new DataSourceImpl();
            ds.setDataSourceName(cpmd.getDataSourceName());
            try {
                ctx.bind(cpmd.getJNDIName(), ds);
                ctx.bind("connectionPoolMetaData/" + cpmd.getDataSourceName(), cpmd);
            } catch (NamingException ex) {
                throw SeasarException.convertSeasarException(ex);
            }
    	}
    }
    
    private void unbindDataSources() {
    	NamingContext ctx = NamingService.getNamingServer().getNamingContext();
    	for (int i = 0; i < _connectionPoolMetaDataList.size(); ++i) {
    		ConnectionPoolMetaData cpmd = (ConnectionPoolMetaData) _connectionPoolMetaDataList.get(i);
            try {
                ctx.unbind(cpmd.getJNDIName());
                ctx.unbind("connectionPoolMetaData/" + cpmd.getDataSourceName());
            } catch (Exception ex) {
                Logger.getLogger(getClass()).log(ex);
            }
    	}
    }
}