package org.seasar.nazuna;

import java.lang.reflect.UndeclaredThrowableException;
import java.rmi.ConnectException;
import java.rmi.NoSuchObjectException;
import java.rmi.RemoteException;
import java.rmi.UnmarshalException;
import java.util.List;
import java.util.Map;

import org.seasar.log.Logger;
import org.seasar.util.Assertion;
import org.seasar.util.EArrayList;
import org.seasar.util.SeasarException;
import org.seasar.util.ThrowableUtil;

public final class RemoteLocation implements NazunaLocation {

	private String _name;
	private List _urls = new EArrayList();
	private int _urlIndex = 0;

	public RemoteLocation(String name) {
		Assertion.assertNotNull("name", name);

		_name = name;
	}

	public final String getName() {
		return _name;
	}

	public final void addURL(NazunaURL url) {
		_urls.add(url);
	}

	public final int getURLSize() {
		return _urls.size();
	}

	public final NazunaURL getURL(int index) {
		return (NazunaURL) _urls.get(index);
	}

	public final Object executeRulet(String className, Object[] args)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		Object ret = null;
		try {
			ret = url.executeRulet(className, args);
		} catch (UndeclaredThrowableException ex) {
			return executeRuletFailover(url, ex, className, args);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeRuletFailoverIOException(
					url,
					ex,
					className,
					args);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final Object execute(String className, Map parameters)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		Object ret = null;
		try {
			ret = url.execute(className, parameters);
		} catch (UndeclaredThrowableException ex) {
			return executeFailover(url, ex, className, parameters);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeFailoverIOException(
					url,
					ex,
					className,
					parameters);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final Object execute(String className, Object[] args)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		Object ret = null;
		try {
			ret = url.execute(className, args);
		} catch (UndeclaredThrowableException ex) {
			return executeFailover(url, ex, className, args);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeFailoverIOException(url, ex, className, args);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final List executeQuery(String className, Map parameters)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		List ret = null;
		try {
			ret = url.executeQuery(className, parameters);
		} catch (UndeclaredThrowableException ex) {
			return executeQueryFailover(url, ex, className, parameters);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeQueryFailoverIOException(
					url,
					ex,
					className,
					parameters);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final List executeQuery(String className, Object[] args)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		List ret = null;
		try {
			ret = url.executeQuery(className, args);
		} catch (UndeclaredThrowableException ex) {
			return executeQueryFailover(url, ex, className, args);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeQueryFailoverIOException(
					url,
					ex,
					className,
					args);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final NzRecordSet executeRSQuery(String className, Map parameters)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		NzRecordSet ret = null;
		try {
			ret = url.executeRSQuery(className, parameters);
		} catch (UndeclaredThrowableException ex) {
			return executeRSQueryFailover(url, ex, className, parameters);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeRSQueryFailoverIOException(
					url,
					ex,
					className,
					parameters);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final NzRecordSet executeRSQuery(String className, Object[] args)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		NzRecordSet ret = null;
		try {
			ret = url.executeRSQuery(className, args);
		} catch (UndeclaredThrowableException ex) {
			return executeRSQueryFailover(url, ex, className, args);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeRSQueryFailoverIOException(
					url,
					ex,
					className,
					args);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final Object executeSingleQuery(String className, Map parameters)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		Object ret = null;
		try {
			ret = url.executeSingleQuery(className, parameters);
		} catch (UndeclaredThrowableException ex) {
			return executeSingleQueryFailover(url, ex, className, parameters);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeSingleQueryFailoverIOException(
					url,
					ex,
					className,
					parameters);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final Object executeSingleQuery(String className, Object[] args)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		Object ret = null;
		try {
			ret = url.executeSingleQuery(className, args);
		} catch (UndeclaredThrowableException ex) {
			return executeSingleQueryFailover(url, ex, className, args);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeSingleQueryFailoverIOException(
					url,
					ex,
					className,
					args);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final int executeUpdate(String className, Map parameters)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		int ret = -1;
		try {
			ret = url.executeUpdate(className, parameters);
		} catch (UndeclaredThrowableException ex) {
			return executeUpdateFailover(url, ex, className, parameters);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeUpdateFailoverIOException(
					url,
					ex,
					className,
					parameters);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	public final int executeUpdate(String className, Object[] args)
		throws SeasarException {

		NazunaURL url = getURL(_urlIndex);
		int ret = -1;
		try {
			ret = url.executeUpdate(className, args);
		} catch (UndeclaredThrowableException ex) {
			return executeUpdateFailover(url, ex, className, args);
		} catch (SeasarException ex) {
			if (ex.getMessageCode().equals("ESSR0040")) {
				return executeUpdateFailoverIOException(
					url,
					ex,
					className,
					args);
			} else {
				throw ex;
			}
		}
		_urlIndex = (_urlIndex + 1) % _urls.size();
		return ret;
	}

	private static boolean isFailoverThrowable(Throwable t) {
		if (!(t instanceof RemoteException)) {
			return false;
		}
		if (t instanceof UnmarshalException) {
			String s = ThrowableUtil.getStackTraceString(t);
			return s.indexOf("SocketException") > 0;
		}
		return t instanceof ConnectException
			|| t instanceof NoSuchObjectException;
	}

	private Object executeRuletFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Object[] args)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return executeRulet(className, args);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private Object executeRuletFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Object[] args)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return executeRulet(className, args);
			}
		}
		throw ex;
	}

	private Object executeFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Map parameters)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return execute(className, parameters);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private Object executeFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Map parameters)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return execute(className, parameters);
			}
		}
		throw ex;
	}

	private Object executeFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Object[] args)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return execute(className, args);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private Object executeFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Object[] args)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return execute(className, args);
			}
		}
		throw ex;
	}

	private List executeQueryFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Map parameters)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return executeQuery(className, parameters);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private List executeQueryFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Map parameters)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return executeQuery(className, parameters);
			}
		}
		throw ex;
	}

	private List executeQueryFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Object[] args)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return executeQuery(className, args);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private List executeQueryFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Object[] args)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return executeQuery(className, args);
			}
		}
		throw ex;
	}

	private NzRecordSet executeRSQueryFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Map parameters)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return executeRSQuery(className, parameters);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private NzRecordSet executeRSQueryFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Map parameters)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return executeRSQuery(className, parameters);
			}
		}
		throw ex;
	}

	private NzRecordSet executeRSQueryFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Object[] args)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return executeRSQuery(className, args);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private NzRecordSet executeRSQueryFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Object[] args)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return executeRSQuery(className, args);
			}
		}
		throw ex;
	}

	private Object executeSingleQueryFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Map parameters)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return executeSingleQuery(className, parameters);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private Object executeSingleQueryFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Map parameters)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return executeSingleQuery(className, parameters);
			}
		}
		throw ex;
	}

	private Object executeSingleQueryFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Object[] args)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return executeSingleQuery(className, args);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private Object executeSingleQueryFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Object[] args)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return executeSingleQuery(className, args);
			}
		}
		throw ex;
	}

	private int executeUpdateFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Map parameters)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return executeUpdate(className, parameters);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private int executeUpdateFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Map parameters)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return executeUpdate(className, parameters);
			}
		}
		throw ex;
	}

	private int executeUpdateFailover(
		NazunaURL url,
		UndeclaredThrowableException ex,
		String className,
		Object[] args)
		throws SeasarException {

		Throwable t = ThrowableUtil.getCause(ex);
		synchronized (_urls) {
			if (_urls.size() > 1
				&& _urls.indexOf(url) >= 0
				&& isFailoverThrowable(t)) {
				fail(url, t);
				return executeUpdate(className, args);
			}
		}
		if (t instanceof RuntimeException) {
			throw (RuntimeException) t;
		} else if (t instanceof Error) {
			throw (Error) t;
		} else {
			throw SeasarException.convertSeasarException(t);
		}
	}

	private int executeUpdateFailoverIOException(
		NazunaURL url,
		SeasarException ex,
		String className,
		Object[] args)
		throws SeasarException {

		synchronized (_urls) {
			if (_urls.size() > 1 && _urls.indexOf(url) >= 0) {
				fail(url, ex);
				return executeUpdate(className, args);
			}
		}
		throw ex;
	}

	private void fail(NazunaURL url, Throwable t) throws SeasarException {
		Logger.getLogger(getClass()).error(t);
		Logger.getLogger(getClass()).log(
			"WSSR0006",
			new Object[] { url.getSpec()});
		_urls.remove(_urlIndex);
		_urlIndex = 0;
	}

}
