package org.seasar.eclipse;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.internal.core.ClasspathEntry;

public final class SeasarProject
	extends PlatformObject
	implements IProjectNature {

	public static final String NATURE_ID =
		SeasarPlugin.PLUGIN_ID + ".seasarnature";
	private static final String PROPERTIES_FILENAME = ".seasarproject";
	private static final String KEY_CONTEXT_PATH = "contextPath";
	private static final String CALL_START =
		"  <Call name=\"addWebApplication\">\n";
	private static final String CALL_END = "  </Call>\n";
	private static final String ARG_START = "    <Arg>";
	private static final String ARG_END = "</Arg>\n";
	private static final String CONFIGURE_END = "</Configure>";
	private static final String SET_CONTENTS =
		"    <Set name=\"extractWAR\">true</Set>\n"
			+ "    <Set name=\"defaultsDescriptor\">webdefault.xml</Set>\n";

	private IJavaProject _javaProject;
	private IProject _project;
	private Properties _properties = new Properties();

	public static void addSeasarNature(IJavaProject javaProject)
		throws CoreException {

		IProject project = javaProject.getProject();
		IProjectDescription description = project.getDescription();
		String[] prevNatureIds = description.getNatureIds();
		for (int i = 0; i < prevNatureIds.length; ++i) {
			if (prevNatureIds[i].equals(NATURE_ID)) {
				return;
			}
		}
		String[] newNatureIds = new String[prevNatureIds.length + 1];
		System.arraycopy(
			prevNatureIds,
			0,
			newNatureIds,
			0,
			prevNatureIds.length);
		newNatureIds[prevNatureIds.length] = NATURE_ID;
		description.setNatureIds(newNatureIds);
		project.setDescription(description, null);
	}

	public static void removeSeasarNature(IJavaProject javaProject)
		throws CoreException {

		IProject project = javaProject.getProject();
		IProjectDescription description = project.getDescription();
		String[] prevNatureIds = description.getNatureIds();
		for (int i = 0; i < prevNatureIds.length; ++i) {
			if (prevNatureIds[i].equals(NATURE_ID)) {
				String[] newNatures = new String[prevNatureIds.length - 1];
				System.arraycopy(prevNatureIds, 0, newNatures, 0, i);
				System.arraycopy(
					prevNatureIds,
					i + 1,
					newNatures,
					i,
					prevNatureIds.length - (i + 1));
				description.setNatureIds(newNatures);
				project.setDescription(description, null);
			}
		}
	}

	public static SeasarProject getSeasarProject(IJavaProject javaProject)
		throws CoreException {

		SeasarProject result =
			(SeasarProject) javaProject.getProject().getNature(NATURE_ID);
		if (result == null) {
			return null;
		}
		result.setJavaProject(javaProject);
		result.loadProperties();
		return result;
	}

	public static SeasarProject getSeasarProject(IProject project)
		throws CoreException {

		IJavaProject javaProject = JavaCore.create(project);
		return getSeasarProject(javaProject);
	}

	public static void updateWebApplicationInJettyXML(
		File jettyXMLFile,
		String contextPath,
		String root)
		throws CoreException, IOException {

		String text = FileUtil.readText(jettyXMLFile);
		StringBuffer buf = new StringBuffer(text.length());
		int webApplicationStartPos = findWebApplicationStartPos(text, root);
		if (webApplicationStartPos < 0) {
			appendWebApplicationWhenNotFound(buf, text, contextPath, root);
		} else {
			appendWebApplicationWhenFound(
				buf,
				text,
				webApplicationStartPos,
				contextPath,
				root);
		}
		FileUtil.writeText(jettyXMLFile, buf.toString());
	}

	public static void removeWebApplicationInJettyXML(
		File jettyXMLFile,
		String root)
		throws CoreException, IOException {

		String text = FileUtil.readText(jettyXMLFile);
		int webApplicationStartPos = findWebApplicationStartPos(text, root);
		if (webApplicationStartPos < 0) {
			return;
		}
		StringBuffer buf = new StringBuffer(text.length());
		int pos = text.lastIndexOf('\n', webApplicationStartPos);
		buf.append(text.substring(0, pos));
		buf.append(getCallEndRemainText(text, webApplicationStartPos));
		FileUtil.writeText(jettyXMLFile, buf.toString());
	}

	public void configure() throws CoreException {
	}

	public void deconfigure() throws CoreException {
	}

	public IJavaProject getJavaProject() {
		return _javaProject;
	}

	public void setJavaProject(IJavaProject javaProject) {
		_javaProject = javaProject;
	}

	public IProject getProject() {
		return _javaProject.getProject();
	}

	public void setProject(IProject project) {
		_project = project;
	}

	public String getContextPath() {
		return getProperty(KEY_CONTEXT_PATH);
	}

	public void setContextPath(String contextPath) {
		setProperty(KEY_CONTEXT_PATH, contextPath);
	}

	public void storeProperties() throws IOException {
		OutputStream os = new FileOutputStream(getPropertiesFile());
		_properties.store(os, null);
	}

	public void removeSeasarNature() throws CoreException {
		removeSeasarNature(getJavaProject());
	}

	public void configureAll() throws CoreException, IOException {
		clearDefaultSourceEntriesFromClasspath();
		configureWebinfFolder();
		copyWebXMLToWebinf();
		copySeasarJarsToLib();
		addLibJarsToProjectClasspath();
		addRefSeasarLibJarsToProjectClasspath();
		setClassesAsOutputFolder();
		setSrcAsSourceFolder();
		copySeasarClassesFilesToSrc();
		updateWebApplicationInJettyXML();
		_project.refreshLocal(IResource.DEPTH_INFINITE, null);
	}

	public void updateWebApplicationInJettyXML()
		throws CoreException, IOException {
		backupJettyXML();
		File jettyXMLFile = getJettyXMLFile();
		updateWebApplicationInJettyXML(
			jettyXMLFile,
			getContextPath(),
			getRoot());
	}

	public void removeWebApplicationInJettyXML()
		throws CoreException, IOException {
		backupJettyXML();
		File jettyXMLFile = getJettyXMLFile();
		removeWebApplicationInJettyXML(jettyXMLFile, getRoot());
	}

	private static void appendWebApplicationWhenNotFound(
		StringBuffer buf,
		String text,
		String contextPath,
		String root) {

		int configureEndStartPos = text.indexOf(CONFIGURE_END);
		buf.append(text.substring(0, configureEndStartPos));
		appendWebApplication(buf, contextPath, root);
		buf.append('\n');
		buf.append(CONFIGURE_END);
	}

	private static void appendWebApplicationWhenFound(
		StringBuffer buf,
		String text,
		int startPos,
		String contextPath,
		String root) {

		buf.append(text.substring(0, startPos));
		appendWebApplication(buf, contextPath, root);
		buf.append(getCallEndRemainText(text, startPos));
	}

	private static void appendWebApplication(
		StringBuffer buf,
		String contextPath,
		String root) {
		buf.append(CALL_START);
		buf.append(ARG_START);
		buf.append(contextPath);
		buf.append(ARG_END);
		buf.append(ARG_START);
		buf.append(root);
		buf.append(ARG_END);
		buf.append(SET_CONTENTS);
		buf.append(CALL_END);
	}

	private String getRoot() throws IOException {
		return _project.getLocation().toFile().getCanonicalPath();
	}

	private static int findWebApplicationStartPos(String text, String root)
		throws IOException {
		int rootStartPos = text.indexOf(root);
		if (rootStartPos < 0) {
			return rootStartPos;
		}
		return text.lastIndexOf(CALL_START, rootStartPos);
	}

	private static String getCallEndRemainText(String text, int startPos) {
		int callEndStartPos = text.indexOf(CALL_END, startPos);
		return text.substring(callEndStartPos + CALL_END.length());
	}

	private static IClasspathEntry[] convertClasspathEntryFromListToArray(List cp) {
		return (IClasspathEntry[]) cp.toArray(new IClasspathEntry[cp.size()]);
	}

	private static IClasspathEntry newLibraryEntry(
		IPath path,
		IPath sourceAttachmentPath,
		IPath sourceAttachmentRootPath) {

		return new ClasspathEntry(
			IPackageFragmentRoot.K_BINARY,
			IClasspathEntry.CPE_LIBRARY,
			path,
			ClasspathEntry.NO_EXCLUSION_PATTERNS,
			sourceAttachmentPath,
			sourceAttachmentRootPath,
			null,
			false);
	}

	private List getRawClasspathAsList() throws CoreException {
		IClasspathEntry[] entries = _javaProject.getRawClasspath();
		List cp = new ArrayList(entries.length + 1);
		cp.addAll(Arrays.asList(entries));
		return cp;
	}

	private void setRawClasspath(List cp) throws CoreException {
		_javaProject.setRawClasspath(
			convertClasspathEntryFromListToArray(cp),
			null);
	}

	private IFolder getWebinfFolder() {
		return getProject().getFolder("WEB-INF");
	}

	private File getWebinfDir() {
		return getWebinfFolder().getLocation().toFile();
	}

	private IPath getWebinfPath() throws IOException {
		return new Path("WEB-INF");
	}

	private IFolder getClassesFolder() {
		return getWebinfFolder().getFolder("classes");
	}

	private IFolder getSrcFolder() {
		return getWebinfFolder().getFolder("src");
	}

	private File getSrcDir() {
		return getSrcFolder().getLocation().toFile();
	}

	private IFolder getLibFolder() {
		return getWebinfFolder().getFolder("lib");
	}

	private File getLibDir() {
		return getLibFolder().getLocation().toFile();
	}

	private IPath getLibPath() throws IOException {
		return getWebinfPath().append("lib");
	}

	private void configureFolder(IFolder folder) throws CoreException {
		if (!folder.exists()) {
			folder.create(false, true, null);
		}
	}

	private void configureWebinfFolder() throws CoreException {
		IFolder webinfFolder = getWebinfFolder();
		configureFolder(webinfFolder);
		configureFolder(webinfFolder.getFolder("classes"));
		configureFolder(webinfFolder.getFolder("lib"));
		configureFolder(webinfFolder.getFolder("src"));
	}

	private void copySeasarJarsToLib() throws CoreException, IOException {
		File libDir = getLibDir();
		SeasarPlugin.getDefault().copySeasarLibJarsToDir(libDir);
	}

	private void copySeasarClassesFilesToSrc()
		throws CoreException, IOException {
		File srcDir = getSrcDir();
		SeasarPlugin.getDefault().copySeasarClassesFilesToDir(srcDir);
		SeasarPlugin.getDefault().copyLog4jPropertiesToDir(srcDir);
	}

	private void copyWebXMLToWebinf() throws CoreException, IOException {
		File webinfDir = getWebinfDir();
		SeasarPlugin.getDefault().copyWebXMLToDir(webinfDir);
	}

	private void clearDefaultSourceEntriesFromClasspath()
		throws CoreException {
		IClasspathEntry[] entries = _javaProject.getRawClasspath();
		List cp = new ArrayList(entries.length + 1);
		for (int i = 0; i < entries.length; ++i) {
			if (entries[i].getEntryKind() != IClasspathEntry.CPE_SOURCE) {
				cp.add(entries[i]);
			}
		}
		setRawClasspath(cp);
	}

	private void addRefSeasarLibJarsToProjectClasspath()
		throws CoreException, IOException {

		IPath[] jars = SeasarPlugin.getDefault().getRefSeasarLibJars();
		List cp = getRawClasspathAsList();
		for (int i = 0; i < jars.length; ++i) {
			cp.add(JavaCore.newVariableEntry(jars[i], null, null));
		}
		setRawClasspath(cp);
	}

	private void addLibJarsToProjectClasspath()
		throws CoreException, IOException {

		File libDir = getLibDir();
		File[] jars = JarZipFilter.listFiles(libDir);
		List cp = getRawClasspathAsList();
		IPath libPath = getLibPath();
		for (int i = 0; i < jars.length; ++i) {
			IPath jarPath = libPath.append(jars[i].getName());
			IPath srcPath = null;
			if (jars[i].getName().equals("seasar.jar")) {
				srcPath = SeasarPlugin.getDefault().getSeasarSrcPath();
			}
			cp.add(newLibraryEntry(jarPath, srcPath, null));
		}
		setRawClasspath(cp);
	}

	private void setClassesAsOutputFolder() throws CoreException {
		IFolder classesFolder = getClassesFolder();
		_javaProject.setOutputLocation(classesFolder.getFullPath(), null);
	}

	private void setFolderAsSourceEntry(IFolder folder) throws CoreException {
		IClasspathEntry[] entries = _javaProject.getRawClasspath();
		IClasspathEntry[] newEntries = new IClasspathEntry[entries.length + 1];
		System.arraycopy(entries, 0, newEntries, 0, entries.length);
		newEntries[entries.length] =
			JavaCore.newSourceEntry(folder.getFullPath());
		_javaProject.setRawClasspath(newEntries, null);
	}

	private void setSrcAsSourceFolder() throws CoreException {
		setFolderAsSourceEntry(getSrcFolder());
	}

	private void backupJettyXML() throws CoreException, IOException {
		String backup = getJettyXML() + ".backup";
		File jettyXMLFile = getJettyXMLFile();
		if (!jettyXMLFile.exists()) {
			String msg =
				"jetty.xml file is not found in "
					+ jettyXMLFile.getAbsolutePath();
			SeasarPlugin.handleException(msg);
		}
		File backupFile = new File(backup);
		FileUtil.copyFile(jettyXMLFile, backupFile);
	}

	private File getJettyXMLFile() {
		return new File(getJettyXML());
	}

	private String getJettyXML() {
		return SeasarPlugin.getDefault().getJettyXML();
	}

	private File getPropertiesFile() {
		return getProject().getLocation().append(PROPERTIES_FILENAME).toFile();
	}

	private void loadProperties() {
		File file = getPropertiesFile();
		if (file.exists()) {
			try {
				InputStream is = new FileInputStream(file);
				_properties.load(is);
			} catch (IOException ex) {
				SeasarPlugin.handleException(ex);
			}
		}
	}

	private String getProperty(String key) {
		return _properties.getProperty(key);
	}

	private void setProperty(String key, String value) {
		_properties.setProperty(key, value);
	}
}
