package org.seasar.util;

import java.util.Map;
import java.util.Stack;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

public final class XMLHandler extends DefaultHandler {

    private XMLHandlerRule _xmlHandlerRule = null;
    private StringBuffer _bodyBuffer = null;
    private Stack _bodyBufferStack = new Stack();
    private String _locationPath = "";
    private String _qName = "";
    private Stack _qNameStack = new Stack();
    private String _locationDetailPath = "";
    private Object _result;
    private Stack _stack = new Stack();
    private Map _pathCounts = new SMap();
    private boolean _usePathMatching = true;

    public XMLHandler(XMLHandlerRule xmlHandlerRule) {
  	    this(xmlHandlerRule, true);
    }

    public XMLHandler(XMLHandlerRule xmlHandlerRule, boolean usePathMatching) {
    	Assertion.assertNotNull("xmlHandlerRule", xmlHandlerRule);

  	    _xmlHandlerRule = xmlHandlerRule;
  	    _usePathMatching = usePathMatching;
    }

    public void push(final Object o) {
        if (_stack.empty()) {
            _result = o;
        }
        _stack.push(o);
    }
  
    public Object getResult() {
        return _result;
    }

    public Object pop() {
        return _stack.pop();
    }

    public Object peek() {
        return _stack.peek();
    }

    public Object peek(final int n) {
        return _stack.get(_stack.size() - n - 1);
    }

    public Object peek(final Class clazz) {
    	for (int i = _stack.size() - 1; i >= 0; --i) {
    		Object o = _stack.get(i);
    		if (clazz.isInstance(o)) {
    			return o;
    		}
    	}
        return null;
    }

    public Object peekFirst() {
        return _stack.get(0);
    }

    public void startElement(final String namespaceURI, final String localName,
        final String qName, final Attributes attributes) throws SAXException {

        _bodyBufferStack.push(_bodyBuffer);
        _bodyBuffer = new StringBuffer();
        _qNameStack.push(_qName);
        _qName = qName;
        _locationPath += "/" + qName;
        int pathCount = incrementPathCount();
        _locationDetailPath += "/" + qName + "[" + pathCount + "]";
        ElementHandler eh = null;
        if (_usePathMatching) {
        	eh = _xmlHandlerRule.getElementHandler(_locationPath);
        } else {
        	eh = _xmlHandlerRule.getElementHandler(qName);
        }
        if (eh != null) {
        	try {
            	eh.start(this, attributes);
        	} catch (Exception ex) {
        		throw new SeasarRuntimeException("ESSR0347", new Object[]{_locationDetailPath, ex}, ex);
        	}
        }
    }

    public void characters(final char[] buffer, final int start, final int length)
    		throws SAXException {
    			
        _bodyBuffer.append(buffer, start, length);
        ElementHandler eh = null;
        if (_usePathMatching) {
        	eh = _xmlHandlerRule.getElementHandler(_locationPath);
        } else {
        	eh = _xmlHandlerRule.getElementHandler(_qName);
        }
        if (eh != null) {
        	try {
        		String body = new String(buffer, start, length).trim();
        		if (body.length() > 0) {
            		eh.appendBody(this, body);
        		}
        	} catch (Exception ex) {
        		throw new SeasarRuntimeException("ESSR0347", new Object[]{_locationDetailPath, ex}, ex);
        	}
        }
    }

    public void endElement(final String namespaceURI, final String localName,
    		final String qName) throws SAXException {
    			
        ElementHandler eh = null;
        if (_usePathMatching) {
        	eh = _xmlHandlerRule.getElementHandler(_locationPath);
        } else {
        	eh = _xmlHandlerRule.getElementHandler(qName);
        }
        if (eh != null) {
        	try {
            	eh.end(this, _bodyBuffer.toString().trim());
            } catch (Exception ex) {
            	throw new SeasarRuntimeException("ESSR0347", new Object[]{_locationDetailPath, ex}, ex);
        	}
        }
        _bodyBuffer = (StringBuffer) _bodyBufferStack.pop();
        _locationPath = _locationPath.substring(0, _locationPath.lastIndexOf('/'));
        _locationDetailPath = _locationDetailPath.substring(0, _locationDetailPath.lastIndexOf('/'));
        _qName = (String) _qNameStack.pop();
    }

    public void endDocument() {
        _bodyBuffer = null;
        _bodyBufferStack.clear();
        _stack.clear();
    }
    
    public String getLocationPath() {
    	return _locationPath;
    }
    
    public String getLocationDetailPath() {
    	return _locationDetailPath;
    }
    
    private int incrementPathCount() {
        Integer pathCount = (Integer) _pathCounts.get(_locationPath);
        if (pathCount == null) {
        	pathCount = MathUtil.ONE_INTEGER;
        } else {
        	pathCount = MathUtil.increment(pathCount);
        }
        _pathCounts.put(_locationPath, pathCount);
        return pathCount.intValue();
    }
}