package org.seasar.sql;

import java.io.Serializable;
import java.sql.SQLException;
import java.util.Properties;

import javax.sql.XAConnection;
import javax.sql.XADataSource;

import org.seasar.util.Reflector;
import org.seasar.util.SeasarRuntimeException;

public class ConnectionPoolMetaData implements Serializable {

    final static long serialVersionUID = 3767222791192220772L;
    private final static int DEFAULT_TIMEOUT = 60 * 10;
    private final String _dataSourceName;
    private final String _jndiName;
    private final String _xaDataSourceClassName;
    private final int _timeout;
    private final int _poolSize;
    private final int _hashCode;
    private Properties _properties;
    private transient XADataSource _xaDataSource;

    public ConnectionPoolMetaData(final String dataSourceName, final String jndiName,
            final String xaDataSourceClassName, final int timeout, final int poolSize) {

        this(dataSourceName, jndiName, xaDataSourceClassName, timeout, poolSize, null);
    }

    public ConnectionPoolMetaData(final String dataSourceName, final String jndiName,
            final String xaDataSourceClassName, final int timeout,
            final int poolSize, final Properties properties) {

        if (dataSourceName == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"dataSourceName"});
        }
        _dataSourceName = dataSourceName;
        _hashCode = dataSourceName.hashCode();
        if (jndiName == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"jndiName"});
        }
        _jndiName = jndiName;
        if (xaDataSourceClassName == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"xaDataSourceClassName"});
        }
        _xaDataSourceClassName = xaDataSourceClassName;
        if (timeout > 0) {
            _timeout = timeout;
        } else {
            _timeout = DEFAULT_TIMEOUT;
        }
        if (poolSize > 0) {
            _poolSize = poolSize;
        } else {
            _poolSize = 1;
        }
        _properties = properties;
    }

    public String getDataSourceName() {
        return _dataSourceName;
    }

    public String getJNDIName() {
        return _jndiName;
    }

    public int getTimeout() {
        return _timeout;
    }

    public int getPoolSize() {
        return _poolSize;
    }

    public void setProperties(final Properties properties) {
        _properties = properties;
    }

    public XAConnectionImpl getXAConnection() throws SQLException {
        XADataSource xaDs = getXADataSource();
        XAConnection xaCon = xaDs.getXAConnection();
        if (xaCon instanceof XAConnectionImpl) {
            return (XAConnectionImpl) xaCon;
        } else {
            return new XAConnectionImpl(xaCon);
        }
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof ConnectionPoolMetaData)) {
            return false;
        }
        return _dataSourceName.equals(((ConnectionPoolMetaData) o)._dataSourceName);
    }

    public int hashCode() {
        return _hashCode;
    }

    public XADataSource getXADataSource() throws SQLException {
        if (_xaDataSource != null) {
            return _xaDataSource;
        }
        synchronized (this) {
            if (_xaDataSource == null) {
                _xaDataSource = newXADataSource();
            }
        }
        return _xaDataSource;
    }

    private XADataSource newXADataSource() {
        if (_properties == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"properties"});
        }
        XADataSource ds = (XADataSource) Reflector.newInstance(_xaDataSourceClassName);
        Reflector.setProperties(ds, _properties);
        return ds;
    }
}
