package org.seasar.nazuna;

import java.lang.reflect.Field;
import java.lang.reflect.Method;

import org.seasar.util.Assertion;
import org.seasar.util.Reflector;
import org.seasar.util.SeasarException;
import org.seasar.util.SeasarRuntimeException;

public final class RuletFacade {
	
    private static final String DEFAULT_ARG_NAME = "DEFAULT_ARG";
	private Rulet _rulet;
	private Method _doExecuteMethod;
	private Method _doValidateMethod;
    private Object[] _defaultArgs;
	private NazunaTransAttribute _transAttribute;
	
	private RuletFacade(Rulet rulet, Method doExecuteMethod, Method doValidateMethod,
			Object[] defaultArgs, NazunaTransAttribute transAttribute) {
				
		_rulet = rulet;
		_doExecuteMethod = doExecuteMethod;
		_doValidateMethod = doValidateMethod;
        _defaultArgs = defaultArgs;
		_transAttribute = transAttribute;
	}
	
	public static final RuletFacade create(Class ruletClass) {
		Assertion.assertNotNull("ruletClass", ruletClass);
		
		Rulet rulet = (Rulet) Reflector.newInstance(ruletClass);
		RuletConfig ruletConfig = new RuletConfig();
		rulet.setRuletConfig(ruletConfig);
		Method doExecuteMethod = getMethod(ruletClass, "doExecute");
		Assertion.assertFound("doExecute", doExecuteMethod);
		Method doValidateMethod = getMethod(ruletClass, "doValidate");
        Object[] defaultArgs = getDefaultArgs(ruletClass,
            doExecuteMethod.getParameterTypes().length);
		NazunaTransAttribute transAttribute = getTransAttribute(rulet);
		return new RuletFacade(rulet, doExecuteMethod, doValidateMethod,
            defaultArgs, transAttribute);
	}
	
	public final Rulet getRulet() {
		return _rulet;
	}
	
	public final Method getDoExecuteMethod() {
		return _doExecuteMethod;
	}
	
	public final Method getDoValidateMethod() {
		return _doValidateMethod;
	}
	
	public final NazunaTransAttribute getTransAttribute() {
		return _transAttribute;
	}
	
	public final void setTransAttribute(NazunaTransAttribute transAttribute) {
		Assertion.assertNotNull("transAttribute", transAttribute);
		
		_transAttribute = transAttribute;
	}

	public final void init() throws SeasarException {
		_rulet.init();
	}
	
	public final void destroy() throws SeasarException {
		_rulet.destroy();
	}
	
	public final Object execute(Object[] args) throws SeasarException {
        args = convertArgs(args);
		if (_doValidateMethod != null) {
			Reflector.invoke(_doValidateMethod, _rulet, args);
		}
		return _transAttribute.invoke(_doExecuteMethod, _rulet, args);
	}
	
	private static final Method getMethod(Class ruletClass, String methodName) {
		Method[] methods = ruletClass.getDeclaredMethods();
		for (int i = 0; i < methods.length; ++i) {
			if (methods[i].getName().equals(methodName)) {
				return methods[i];
			}
		}
		Class superClass = ruletClass.getSuperclass();
		if (superClass != null && !superClass.equals(Rulet.class)) {
			return getMethod(superClass, methodName);
		}
		return null;
	}
    
    private static final Object getFieldValue(Class ruletClass, String fieldName) {
        Field[] fields = ruletClass.getDeclaredFields();
        for (int i = 0; i < fields.length; ++i) {
            if (fields[i].getName().equals(fieldName)) {
                try {
                    return fields[i].get(ruletClass);
                } catch (IllegalAccessException ex) {
                    throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
                }
                
            }
        }
        Class superClass = ruletClass.getSuperclass();
        if (superClass != null && !superClass.equals(Rulet.class)) {
            return getFieldValue(superClass, fieldName);
        }
        return null;
    }
	
	private static final NazunaTransAttribute getTransAttribute(Rulet rulet) {
		if (rulet instanceof RequiredTx) {
			return NazunaTransAttribute.REQUIRED;
		} else if (rulet instanceof RequiresNewTx) {
			return NazunaTransAttribute.REQUIRES_NEW;
		} else if (rulet instanceof MandatoryTx) {
			return NazunaTransAttribute.MANDATORY;
		} else {
			return NazunaTransAttribute.SUPPORTS;
		}
	}
    
    private static final Object[] getDefaultArgs(Class ruletClass, int argSize) {
        Object[] args = new Object[argSize];
        for (int i = 0; i < argSize; ++i) {
            args[i] = getFieldValue(ruletClass, DEFAULT_ARG_NAME + i);
        }
        return args;
    }
    
    private final Object[] convertArgs(Object[] args) {
		Class[] parameterTypes = _doExecuteMethod.getParameterTypes();
		for (int i = 0; i < args.length; ++i) {
			args[i] = NazunaUtil.adjustValue(args[i], parameterTypes[i]);
		}
		Object[] args2 = args;
        if (args.length < _defaultArgs.length) {
			args2 = new Object[_defaultArgs.length];
			System.arraycopy(args, 0, args2, 0, args.length);
        }
        for (int i = args.length; i < _defaultArgs.length; ++i) {
            args2[i] = _defaultArgs[i];
        }
        return args2;
    }
}

