/**@file
 *			Patches for Anthy, by G-HAL
 *@brief	iconv 
 *@date		Sun,07 Jun,2009 - Wed,17 Jun,2009
 *@date		Fri,19 Jun,2009 - Sat,20 Jun,2009
 *@date		Wed,13 Oct,2010 - Thu,14 Oct,2010
 *@author	Copyright(C)2009-2010 G-HAL
 */
#if defined(HAVE_CONFIG_H)
# include "config.h"
#endif

#if defined(USE_ICONV)

#if defined(HAVE_STDLIB_H)
# include <stdlib.h>
#endif
#if defined(HAVE_MALLOC_H)
# include <malloc.h>
#endif
#if !defined(__STDC_LIMIT_MACROS)
# define __STDC_LIMIT_MACROS
#endif
#if !defined(__STDC_CONSTANT_MACROS)
# define __STDC_CONSTANT_MACROS
#endif
#if defined(HAVE_STDINT_H)
# include <stdint.h>
#endif
#if !defined(__STDC_FORMAT_MACROS)
# define __STDC_FORMAT_MACROS
#endif
#if defined(HAVE_INTTYPES_H)
# include <inttypes.h>
#endif
#if defined(HAVE_STRING_H)
# include <string.h>
#endif
#if defined(HAVE_STRINGS_H)
# include <strings.h>
#endif
#if defined(HAVE_ARPA_INET_H)
# include <arpa/inet.h>
#endif
#if defined(HAVE_SYS_TYPES_H)
# include <sys/types.h>
#endif
#if defined(HAVE_NETINET_IN_H)
# include <netinet/in.h>
#endif
#if defined(HAVE_ALLOCA_H)	/* Patched by G-HAL, Wed,13 Oct,2010 */
# include <alloca.h>
#elif defined(HAVE_ALLOCA)
#else
# error	"alloca() was not found."
#endif

#include "anthy/anthy.h"
#include "anthy/settings.h"
#include "anthy/logger.h"
#include "anthy/xchar.h"
#include "anthy/xstr.h"

#if defined(HAVE_ICONV_H)
# include <iconv.h>
#endif
#if defined(HAVE_ICONV_OPEN) && !defined(HAVE_LIBICONV_OPEN)
# define	libiconv_open	iconv_open
#endif
#if defined(HAVE_ICONV) && !defined(HAVE_LIBICONV)
# define	libiconv	iconv
#endif
#if defined(HAVE_ICONV_CLOSE) && !defined(HAVE_LIBICONV_CLOSE)
# define	libiconv_close	iconv_close
#endif



/** @name ʸ @{ */

#define						UCS4_LEN	4

/** ʸ UCS-4ɽ */
static uint8_t				GETA_UCS4_str[] = { UINT8_C(0x00), UINT8_C(0x00), UINT8_C(0x30), UINT8_C(0x13) };
static const size_t			GETA_UCS4_len = 4;

/** ʸ UTF-8ɽ */
static uint8_t				GETA_UTF8_str[MAX_BYTES_PER_XCHAR] = { UINT8_C(0xE3), UINT8_C(0x80), UINT8_C(0x93), UINT8_C(0x00) };
static size_t				GETA_UTF8_len   = 3;

/** @} */



/** @name 󥳡ǥ󥰤Ū @{ */

static iconv_t	cd_euc_to_utf8;			/**< EUC  UTF-8 Ѵǥץ */
static iconv_t	cd_utf8_to_euc;			/**< UTF-8  EUC Ѵǥץ */

/** @} */


/** @name 󥳡ǥ󥰤ưŪ @{ */

/** 󥳡ǥѴΥǡå */
struct encoding_converter_table_t {
	int			initialized;										/**< ѥե饰 */
	const char*	encoding_name;										/**< 󥳡ǥ̾ */
	iconv_t		cd_to_internal;										/**< iconv ꥳ   Ѵǥץ */
	iconv_t		cd_internal_to;										/**< iconv   ꥳ Ѵǥץ */
};

/** 󥳡ǥѴѴơ֥ */
static struct encoding_converter_table_t	converter_tbl[ANTHY_ENCODING_MAX] = {
	[ANTHY_COMPILED_ENCODING]      = { .initialized = 0, .encoding_name = ICONV_SRCENCODING, },
	[ANTHY_ENCODING_EUCJP]         = { .initialized = 0, .encoding_name = NULL, },
	[ANTHY_ENCODING_UTF8]          = { .initialized = 0, .encoding_name = ICONV_UTF8, },

  #if defined(ICONV_ISO2022JP)
	[ANTHY_ENCODING_ISO2022JP]     = { .initialized = 0, .encoding_name = ICONV_ISO2022JP, },
  #else
	[ANTHY_ENCODING_ISO2022JP]     = { .initialized = 0, .encoding_name = "", },
  #endif
  #if defined(ICONV_EUCJP0213)
	[ANTHY_ENCODING_EUCJP0213]     = { .initialized = 0, .encoding_name = ICONV_EUCJP0213, },
  #else
	[ANTHY_ENCODING_EUCJP0213]     = { .initialized = 0, .encoding_name = "", },
  #endif

	[ANTHY_ENCODING_EUCJP0212]     = { .initialized = 0, .encoding_name = ICONV_EUCJP, },

  #if defined(ICONV_EUCJPMS)
	[ANTHY_ENCODING_EUCJPMS]       = { .initialized = 0, .encoding_name = ICONV_EUCJPMS, },
  #else
	[ANTHY_ENCODING_EUCJPMS]       = { .initialized = 0, .encoding_name = "", },
  #endif
  #if defined(ICONV_SHIFTJIS)
	[ANTHY_ENCODING_SHIFTJIS]      = { .initialized = 0, .encoding_name = ICONV_SHIFTJIS, },
  #else
	[ANTHY_ENCODING_SHIFTJIS]      = { .initialized = 0, .encoding_name = "", },
  #endif
  #if defined(ICONV_SHIFTJIS0213)
	[ANTHY_ENCODING_SHIFTJIS0213]  = { .initialized = 0, .encoding_name = ICONV_SHIFTJIS0213, },
  #else
	[ANTHY_ENCODING_SHIFTJIS0213]  = { .initialized = 0, .encoding_name = "", },
  #endif
  #if defined(ICONV_SJISMS)
	[ANTHY_ENCODING_SHIFTJISMS]    = { .initialized = 0, .encoding_name = ICONV_SJISMS, },
  #else
	[ANTHY_ENCODING_SHIFTJISMS]    = { .initialized = 0, .encoding_name = "", },
  #endif

	[ANTHY_ENCODING_USERDEFINE]    = { .initialized = 0, .encoding_name = NULL, },
};

/** xcharžѥХåեʸ */
#if defined(CHARARRAY_IS_CONTINUOUS8BIT) && defined(ICONV_UCS4HE)
static const char	encoding_name_ucs4internal[] = ICONV_UCS4HE;
#else
static const char	encoding_name_ucs4internal[] = ICONV_UCS4BE;
#endif

/** @} */





/** @name  @{ */

/** ʸƬΣʸΥǡĹ
 *@param[in]			c						ʸ
 *@param				max_len					ʸĹ
 *@param				encoding				ʸΥ󥳡ǥ
 *@return										ǡĹ
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 */
static size_t get_char_len( const char* const c, size_t max_len, enum ANTHY_ENCODING encoding )
{
	size_t ret = 0;
	const unsigned char	chr = (1 <= max_len) ? c[0] : 0;
	switch (encoding) {
	case ANTHY_ENCODING_UTF8:
		if (UINT8_C(0xC0) <= chr) {
			if ((UINT8_C(0xC0) <= chr) && (chr <= UINT8_C(0xDF))) {
				ret = 2;
			} else if ((UINT8_C(0xE0) <= chr) && (chr <= UINT8_C(0xEF))) {
				ret = 3;
			} else if ((UINT8_C(0xF0) <= chr) && (chr <= UINT8_C(0xF7))) {
				ret = 4;
			} else if ((UINT8_C(0xF8) <= chr) && (chr <= UINT8_C(0xFB))) {
				ret = 5;	/* Illigal data. */
			} else if ((UINT8_C(0xFC) <= chr) && (chr <= UINT8_C(0xFD))) {
				ret = 6;	/* Illigal data. */
			} else {
				ret = 1;	/* Broken data. UTF-16 BOM ? */
			}
		} else if ((UINT8_C(0x80) <= chr) && (chr <= UINT8_C(0xBF))) {
			ret = 1;	/* It is a trailer. */
		} else {
			ret = 1;
		}
		break;
	case ANTHY_ENCODING_EUCJP:
	case ANTHY_ENCODING_EUCJP0213:
	case ANTHY_ENCODING_EUCJP0212:
	case ANTHY_ENCODING_EUCJPMS:
		if ((UINT8_C(0xA1) <= chr) && (chr <= UINT8_C(0xFE))) {
			ret = 2;
		} else if (UINT8_C(0x8E) == chr) {
			ret = 2;
		} else if (UINT8_C(0x8F) == chr) {
			ret = 3;
		} else {
			ret = 1;
		}
		break;
	case ANTHY_ENCODING_SHIFTJIS:
		if ((UINT8_C(0x81) <= chr) && (chr <= UINT8_C(0x9F))) {
			ret = 2;
		} else if ((UINT8_C(0xE0) <= chr) && (chr <= UINT8_C(0xEF))) {
			ret = 2;
		} else {
			ret = 1;
		}
		break;
	case ANTHY_ENCODING_SHIFTJIS0213:
	case ANTHY_ENCODING_SHIFTJISMS:
		if ((UINT8_C(0x81) <= chr) && (chr <= UINT8_C(0x9F))) {
			ret = 2;
		} else if ((UINT8_C(0xE0) <= chr) && (chr <= UINT8_C(0xFC))) {
			ret = 2;
		} else {
			ret = 1;
		}
		break;
	case ANTHY_ENCODING_USERDEFINE:
	case ANTHY_ENCODING_ISO2022JP:
	case ANTHY_COMPILED_ENCODING:
	case ANTHY_ENCODING_MAX:
	default:
		ret = 1;
		break;
	}
	return (ret < max_len) ? ret : max_len;
}

/** @} */



/** @name 󥳡ǥ󥰤Ū @{ */

/** UTF-8  UCS-4 أʸѴ
 *@param[in]			s						Ѵʸ
 *@param[in,out]		res						Ѵʸ
 *@return										Ѵμʸ
 *
 *@comment
 *	 Anthy  xstr.c 餽Τޤޥԡ
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 */
const char* anthy_utf8_to_ucs4_xchar( const char* const s, xchar* const res )
{
	const unsigned char* str = (const unsigned char *)s;
	int i, len;
	xchar cur;
	cur = str[0];
	if (str[0] < UINT8_C(0x80)) {
		len = 1;
	} else if (str[0] < UINT8_C(0xE0)) {
		cur &= UINT8_C(0x1f);
		len = 2;
	} else if (str[0] < UINT8_C(0xF0)) {
		cur &= UINT8_C(0x0f);
		len = 3;
	} else if (str[0] < UINT8_C(0xF8)) {
		cur &= UINT8_C(0x07);
		len = 4;
	} else if (str[0] < UINT8_C(0xFC)) {
		cur &= UINT8_C(0x03);
		len = 5;
	} else {
		cur &= UINT8_C(0x01);
		len = 6;
	}
	str ++;
	for (i = 1; i < len; i++) {
		cur <<= 6;
		cur |= (str[0] & UINT8_C(0x3F));
		str++;
	}
	*res = cur;
	return (const char *)str;
}

/** UCS-4  UTF-8 أʸѴ
 *@param[in]			xc						Ѵʸ
 *@param[in,out]		buf						Ѵʸ
 *@return										Ѵʸ
 *
 *@comment
 *	 Anthy  xstr.c 餽Τޤޥԡ
 *
 *@comment
 *	Patched by G-HAL
 *		Wed,17 Jun,2009
 *		Thu,14 Oct,2010
 */
static int anthy_xchar_to_utf8_str( xchar xc, uint8_t* const buf )
{
	int i, len;
	if (xc < UINT32_C(0x80)) {
		buf[0] = UINT8_C(0);
		len = 1;
	} else if (xc < UINT32_C(0x800)) {
		buf[0] = UINT8_C(0xC0);
		len = 2;
	} else if (xc < UINT32_C(0x10000)) {
		buf[0] = UINT8_C(0xE0);
		len = 3;
	} else if (xc < UINT32_C(0x200000)) {
		buf[0] = UINT8_C(0xF0);
		len = 4;
	} else if (xc < UINT32_C(0x400000)) {
		buf[0] = UINT8_C(0xF8);
		len = 5;
	} else {
		buf[0] = UINT8_C(0xFC);
		len = 6;
	}
	for (i = len - 1; 0 < i; --i) {
		buf[i] = (xc & UINT8_C(0x3F)) | UINT8_C(0x80);
		xc >>= 6;
	}
	buf[0] += xc;
	buf[len] = 0;
	return len;
}



/** EUC-JP  UTF-8 Ѵ
 *@param[in]			s						Ѵʸ
 *@return										Ѵʸ
 *
 *@comment
 *	ѺѤ߸ͤ free() 뤳ȡ
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 */
char* const anthy_conv_euc_to_utf8( const char* const s )
{
	const size_t	in_len   = strlen( s );
	size_t			in_left  = in_len;
	const char*		in_ptr   = s;
	const size_t	out_max  = (in_len + 1) * 4;	/* +1 ʼ4  UTF-8 ΣʸκĹ */
	size_t			out_left = out_max;
	char* const		out_buf  = (char*) malloc( out_max );
	char*			out_ptr  = out_buf;

	if (NULL == out_buf) {
		return NULL;
	}
	{	iconv_t* const cd = &cd_euc_to_utf8;
		libiconv( *cd, NULL, NULL, NULL, NULL );
		while (1) {
			const ssize_t result = libiconv( *cd, &in_ptr, &in_left, &out_ptr, &out_left );
			if (0 <= result) {
				break;
			}
			{	const size_t geta_size = GETA_UTF8_len;
				if (out_left < geta_size) {
					anthy_log( 1, "Out of memory in anthy_iconv.c:anthy_conv_euc_to_utf8(%zu).\n", in_len );
					break;
				}
				memcpy( out_ptr, GETA_UTF8_str, geta_size );
				out_left -= geta_size;
				out_ptr  += geta_size;
			}
			{	const size_t c_len = get_char_len( in_ptr, in_left, ANTHY_ENCODING_EUCJP );
				in_left -= c_len;
				in_ptr  += c_len;
			}
		}
		libiconv( *cd, NULL, NULL, &out_ptr, &out_left );
		((uint8_t*)out_ptr)[0] = '\0';
		out_left -= 1;
		out_ptr  += 1;
	}
	{	const size_t len = out_max - out_left;
		char* const buf = realloc( out_buf, len );
		if (NULL == buf) {
			anthy_log( 1, "Out of memory at realloc() in anthy_iconv.c:anthy_conv_euc_to_utf8(%zu).\n", in_len );
			return out_buf;
		}
		return buf;
	}
}


/** UTF-8  EUC-JP Ѵ
 *@param[in]			s						Ѵʸ
 *@return										Ѵʸ
 *
 *@comment
 *	ѺѤ߸ͤ free() 뤳ȡ
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 */
char* const anthy_conv_utf8_to_euc( const char* const s )
{
	const size_t	in_len   = strlen( s );
	size_t			in_left  = in_len;
	const char*		in_ptr   = s;
	const size_t	out_max  = (in_len + 1) * 3;	/* +1 ʼ3  EUC-JP ΣʸκĹ */
	size_t			out_left = out_max;
	char* const		out_buf  = (char*) malloc( out_max );
	char*			out_ptr  = out_buf;

	if (NULL == out_buf) {
		return NULL;
	}
	{	iconv_t* const cd = &cd_utf8_to_euc;
		libiconv( *cd, NULL, NULL, NULL, NULL );
		while (1) {
			const ssize_t result = libiconv( *cd, &in_ptr, &in_left, &out_ptr, &out_left );
			if (0 <= result) {
				break;
			}
			{	const char*	geta_ptr  = (const char*)GETA_UTF8_str;
				size_t		geta_left = GETA_UTF8_len;
				libiconv( *cd, &geta_ptr, &geta_left, &out_ptr, &out_left );
			}
			{	const size_t c_len = get_char_len( in_ptr, in_left, ANTHY_UTF8_ENCODING );
				in_left -= c_len;
				in_ptr  += c_len;
			}
		}
		libiconv( *cd, NULL, NULL, &out_ptr, &out_left );
		((uint8_t*)out_ptr)[0] = '\0';
		out_left -= 1;
		out_ptr  += 1;
	}
	{	const size_t len = out_max - out_left;
		char* const buf = realloc( out_buf, len );
		if (NULL == buf) {
			anthy_log( 1, "Out of memory at realloc() in anthy_iconv.c:anthy_conv_utf8_to_euc(%zu).\n", in_len );
			return out_buf;
		}
		return buf;
	}
}

/** @} */



/** @name 󥳡ǥ󥰤ưŪ @{ */

/** ʸ UCS-4 Ѵ
 *@param[in,out]		xs						Ѵ
 *@param				n						buf κ祵
 *@param[in]			buf						Ѵ
 *@param				from_encoding			Ѵʸ
 *@return										xs سǼʸ
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 *		Sun,14 Jun,2009 - Mon,15 Jun,2009
 */
int anthy_snputcstr( xstr* const xs, int n, const char* const buf, enum ANTHY_ENCODING from_encoding )
{
	if (NULL == xs) {
		return 0;
	}
	if ((NULL == buf) || (NULL == xs->str) || (ANTHY_ENCODING_MAX <= from_encoding) || (from_encoding < 0)) {
		xs->len = 0;
		return 0;
	}
	{	const size_t	in_len   = strlen( buf );
		size_t			in_left  = in_len;
		const char*		in_ptr   = buf;
	  #if defined(ICONV_UCS4HE) && defined(XCHARARRAY_IS_CONTINUOUS32BIT)
		const size_t	out_max  = n * UCS4_LEN;
		size_t			out_left = out_max;
		char* const		out_buf  = (char*) xs->str;
		char*			out_ptr  = out_buf;
	  #else
		const size_t	out_max  = n * UCS4_LEN;
		size_t			out_left = out_max;
		char* const		out_buf  = (char*) alloca( out_max );
		char*			out_ptr  = out_buf;
	  #endif

		if (NULL == out_buf) {
			xs->len = 0;
			return 0;
		}
		{	struct encoding_converter_table_t* const	cvt = &(converter_tbl[from_encoding]);
			iconv_t* const cd = &(cvt->cd_to_internal);
			libiconv( *cd, NULL, NULL, NULL, NULL );
			while (1) {
				const ssize_t result = libiconv( *cd, &in_ptr, &in_left, &out_ptr, &out_left );
				if (0 <= result) {
					break;
				}
				{	const size_t geta_size = GETA_UCS4_len;
					if (out_left < geta_size) {
						anthy_log( 1, "Out of memory in anthy_iconv.c:anthy_cstr_to_xstr(%zu).\n", in_len );
						break;
					}
					memcpy( out_ptr, GETA_UCS4_str, geta_size );
					out_left -= geta_size;
					out_ptr  += geta_size;
				}
				{	const size_t c_len = get_char_len( in_ptr, in_left, from_encoding );
					in_left -= c_len;
					in_ptr  += c_len;
				}
			}
			libiconv( *cd, NULL, NULL, &out_ptr, &out_left );
		}
		xs->len = (out_max - out_left) / UCS4_LEN;
	  #if defined(ICONV_UCS4HE) && defined(XCHARARRAY_IS_CONTINUOUS32BIT)
	  #else
		{	size_t i;
			for (i = 0, out_ptr = out_buf; i < xs->len; ++i, out_ptr += UCS4_LEN) {
			  #if defined(CHARARRAY_IS_CONTINUOUS8BIT)
			  # if defined(ICONV_UCS4HE)
				xs->str[i] = *((uint32_t*)out_ptr);
			  # else
				xs->str[i] = ntohl( *((uint32_t*)out_ptr) );
			  # endif
			  #else
			  # if defined(ICONV_UCS4BE)
				xs->str[i]
						= (((uint32_t)(((uint8_t*)out_ptr)[0])) << 24)
						+ (((uint32_t)(((uint8_t*)out_ptr)[1])) << 16)
						+ (((uint32_t)(((uint8_t*)out_ptr)[2])) <<  8)
						+ (((uint32_t)(((uint8_t*)out_ptr)[3])) <<  0);
			  # else
				xs->str[i]
						= (((uint32_t)(((uint8_t*)out_ptr)[0])) <<  0)
						+ (((uint32_t)(((uint8_t*)out_ptr)[1])) <<  8)
						+ (((uint32_t)(((uint8_t*)out_ptr)[2])) << 16)
						+ (((uint32_t)(((uint8_t*)out_ptr)[3])) << 24);
			  # endif
			  #endif
			}
		}
	  #endif
	}
	return xs->len;
}


/** UCS-4 ʸѴ
 *@param[in,out]		buf						
 *@param				n						buf κ祵
 *@param[in]			xs						Ѵ
 *@param				to_encoding				Ѵʸ
 *@return										buf سǼ
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 *		Sun,14 Jun,2009 - Mon,15 Jun,2009
 *		Thu,14 Oct,2010
 */
int anthy_snputxstr( char* const buf, int n, const xstr* const xs, enum ANTHY_ENCODING to_encoding )
{
	if ((NULL == xs) || (xs->len <= 0) || (NULL == xs->str) || (ANTHY_ENCODING_MAX <= to_encoding) || (to_encoding < 0)) {
		buf[0] = '\0';
		return 0;
	}
	{	const size_t	out_max  = n;
		size_t			out_left = out_max;
		char* const		out_buf  = buf;
		char*			out_ptr  = out_buf;
	  #if defined(ICONV_UCS4HE) && defined(XCHARARRAY_IS_CONTINUOUS32BIT)
		const size_t	in_len   = xs->len * UCS4_LEN;
		size_t			in_left  = in_len;
		char* const		in_buf   = (char*) xs->str;
		char*			in_ptr   = in_buf;
	  #else
		const size_t	in_len   = (xs->len + 1) * UCS4_LEN;	/* +1  '\0' ʼη */
		size_t			in_left  = in_len;
		char* const		in_buf   = (char*) alloca( in_len );
		char*			in_ptr;
	  #endif

		if (NULL == in_buf) {
			buf[0] = '\0';
			return 0;
		}
	  #if defined(ICONV_UCS4HE) && defined(XCHARARRAY_IS_CONTINUOUS32BIT)
	  #else
		{	size_t i;
			for (i = 0, in_ptr = in_buf; i < xs->len; ++i, in_ptr += UCS4_LEN) {
			  #if defined(CHARARRAY_IS_CONTINUOUS8BIT)
			  # if defined(ICONV_UCS4HE)
				*((uint32_t*)in_ptr) = xs->str[i];
			  # else
				*((uint32_t*)in_ptr) = htonl( xs->str[i] );
			  # endif
			  #else
			  # if defined(ICONV_UCS4BE)
				((uint8_t*)in_ptr)[0] = (((uint32_t)xs->str[i]) >> 24) & UINT8_C(0xFF);
				((uint8_t*)in_ptr)[1] = (((uint32_t)xs->str[i]) >> 16) & UINT8_C(0xFF);
				((uint8_t*)in_ptr)[2] = (((uint32_t)xs->str[i]) >>  8) & UINT8_C(0xFF);
				((uint8_t*)in_ptr)[3] = (((uint32_t)xs->str[i]) >>  0) & UINT8_C(0xFF);
			  # else
				((uint8_t*)in_ptr)[0] = ((((uint32_t)xs->str[i]) & UINT32_C(0x000000FF)) >>  0);
				((uint8_t*)in_ptr)[1] = ((((uint32_t)xs->str[i]) & UINT32_C(0x0000FF00)) >>  8);
				((uint8_t*)in_ptr)[2] = ((((uint32_t)xs->str[i]) & UINT32_C(0x00FF0000)) >> 16);
				((uint8_t*)in_ptr)[3] = ((((uint32_t)xs->str[i]) & UINT32_C(0xFF000000)) >> 24);
			  # endif
			  #endif
			}
		  #if defined(CHARARRAY_IS_CONTINUOUS8BIT)
			*((uint32_t*)in_ptr) = UINT32_C(0);
		  #else
			((uint8_t*)in_ptr)[0] = UINT8_C(0);
			((uint8_t*)in_ptr)[1] = UINT8_C(0);
			((uint8_t*)in_ptr)[2] = UINT8_C(0);
			((uint8_t*)in_ptr)[3] = UINT8_C(0);
		  #endif
			in_ptr = in_buf;
		}
	  #endif
		{	struct encoding_converter_table_t* const	cvt = &(converter_tbl[to_encoding]);
			iconv_t* const cd = &(cvt->cd_internal_to);
			libiconv( *cd, NULL, NULL, NULL, NULL );
			while (1) {
				const ssize_t result = libiconv( *cd, (const char**)&in_ptr, &in_left, &out_ptr, &out_left );
				if (0 <= result) {
					break;
				}
				{	const char*	geta_ptr  = (const char*)GETA_UCS4_str;
					size_t		geta_left = GETA_UCS4_len;
					libiconv( *cd, &geta_ptr, &geta_left, &out_ptr, &out_left );
				}
				{
					in_left -= UCS4_LEN;
					in_ptr  += UCS4_LEN;
				}
			}
			libiconv( *cd, NULL, NULL, &out_ptr, &out_left );
		}
	  #if defined(ICONV_UCS4HE) && defined(XCHARARRAY_IS_CONTINUOUS32BIT)
		out_buf[ (0 < out_left) ? (out_max - out_left) : (out_max - 1) ] = '\0';
		return out_max - out_left;
	  #else
		out_buf[out_max - 1] = '\0';
		return out_max - out_left - 1; /* -1  '\0' */
	  #endif
	}
}



/** ʸ UCS-4 Ѵ
 *@param[in]			c						Ѵʸ
 *@param				from_encoding			Ѵʸ
 *@return										Ѵʸ
 *
 *@comment
 *	ѺѤ߸ͤ anthy_free_xstr() 뤳ȡ
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 */
xstr* const anthy_cstr_to_xstr( const char* const c, enum ANTHY_ENCODING from_encoding )
{
	xstr* const xs = (xstr* const) malloc( sizeof(struct xstr_) );
	if (NULL == xs) {
		return NULL;
	}
	{	const size_t	in_len   = strlen( c );
		const size_t	out_max  = (in_len + 1) * UCS4_LEN;	/* +1 ʼ */
		int				len;
		xs->str = (xchar*) malloc( out_max );
		if (NULL == xs->str) {
			free( xs );
			return NULL;
		}
		len = anthy_snputcstr( xs, out_max, c, from_encoding );
		{	xchar* const	buf = realloc( xs->str, sizeof(xchar) * (len + 1) );
			if (NULL == buf) {
				anthy_log( 1, "Out of memory at realloc() in anthy_iconv.c:anthy_cstr_to_xstr(%zu,%zu,%zu).\n", in_len, out_max, (sizeof(xchar) * len) );
			} else {
				xs->str = buf;
				xs->str[len] = 0;
			}
		}
	}
	return xs;
}


/** UCS-4 ʸѴ
 *@param[in]			xs						Ѵʸ
 *@param				to_encoding				Ѵʸ
 *@return										Ѵʸ
 *
 *@comment
 *	ѺѤ߸ͤ free() 뤳ȡ
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 */
char* const anthy_xstr_to_cstr( const xstr* const xs, enum ANTHY_ENCODING to_encoding )
{
	if ((NULL == xs) || (xs->len <= 0) || (NULL == xs->str)) {
		return NULL;
	}
	{	const size_t	in_len   = (xs->len + 1) * UCS4_LEN;	/* +1  '\0' ʼη */
		const size_t	out_max  = (xs->len + 1) * MAX_BYTES_PER_XCHAR + MAX_BYTES_BASE;
		char* const		out_buf  = (char*) malloc( out_max );
		const int		len = anthy_snputxstr( out_buf, out_max, xs, to_encoding );
		char* const		buf = realloc( out_buf, (len + 1) );	/* +1  '\0' */
		if (NULL == buf) {
			anthy_log( 1, "Out of memory at realloc() in anthy_iconv.c:anthy_xstr_to_cstr(%zu).\n", in_len );
			return out_buf;
		}
		return buf;
	}
}



/** @} */



/** ꤷʸɤѴơ֥
 *@param				encoding				ʸ
 *@return				0						
 *@return				1						
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 *		Tue,16 Jun,2009
 *		Wed,17 Jun,2009
 */
int anthy_open_iconv( enum ANTHY_ENCODING encoding )
{
	if ((ANTHY_ENCODING_MAX <= encoding) || (encoding < 0)) {
		return 1;
	}
	{	struct encoding_converter_table_t* const	cvt = &(converter_tbl[encoding]);
		if (cvt->initialized) {
			return 0;
		}
		if ((NULL == cvt->encoding_name) || ('\0' == cvt->encoding_name[0])) {
			anthy_log( 1, "iconv encoding name is empty in %d.\n", encoding );
			return 1;
		}

		cvt->cd_to_internal = libiconv_open( encoding_name_ucs4internal, cvt->encoding_name );
		if ((iconv_t)(-1) == cvt->cd_to_internal) {
			anthy_log( 1, "iconv initialization failed.(%s->internal)\n", cvt->encoding_name );
			return 1;
		}
		cvt->cd_internal_to = libiconv_open( cvt->encoding_name, encoding_name_ucs4internal );
		if ((iconv_t)(-1) == cvt->cd_internal_to) {
			libiconv_close( cvt->cd_to_internal );
			anthy_log( 1, "iconv initialization failed.(internal->%s)\n", cvt->encoding_name );
			return 1;
		}

		libiconv( cvt->cd_to_internal, NULL, NULL, NULL, NULL );
		libiconv( cvt->cd_internal_to, NULL, NULL, NULL, NULL );

		cvt->initialized = 1;
	}
	return 0;
}


/** ꤷʸɤѴơ֥뤬ѤǤ뤫ݤ
 *@param				encoding				ʸ
 *@return				1						Ѳǽ
 *@return				0						Բǽ
 *
 *@comment
 *	Patched by G-HAL
 *		Sat,20 Jun,2009
 */
int anthy_isuseable_iconv( enum ANTHY_ENCODING encoding )
{
	if ((ANTHY_ENCODING_MAX <= encoding) || (encoding < 0)) {
		return 0;
	}
	{	struct encoding_converter_table_t* const	cvt = &(converter_tbl[encoding]);
		if (cvt->initialized) {
			return 1;
		}
	}
	return 0;
}


/** 桼ʸɤѴơ֥
 *@param[in]			encoding_name			ʸ̾
 *@return				0						
 *@return				1						
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 */
int anthy_open_iconv_userdefineencoding( const char* const encoding_name )
{
	struct encoding_converter_table_t* const	cvt = &(converter_tbl[ANTHY_ENCODING_USERDEFINE]);
	if (cvt->initialized) {
		libiconv_close( cvt->cd_internal_to );
		libiconv_close( cvt->cd_to_internal );
		cvt->initialized = 0;
	}

	if (cvt->encoding_name) {
		free( cvt->encoding_name );
	}
	if ((NULL == encoding_name) || ('\0' == encoding_name[0])) {
		anthy_log( 1, "iconv initialization failed. UserDefineEncodingName is empty.\n" );
		return 1;
	}
	cvt->encoding_name = strdup( encoding_name );

	return anthy_open_iconv( ANTHY_ENCODING_USERDEFINE );
}


/** xchar饹
 *@retval				0						
 *@retval				1						
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 *		Wed,17 Jun,2009
 *		Sat,20 Jun,2009
 *		Thu,14 Oct,2010
 */
int anthy_init_iconv( void )
{
	{
	  #if defined(CHARARRAY_IS_CONTINUOUS8BIT)
	  #  if defined(ICONV_UCS4HE)
		*((uint32_t*)GETA_UCS4_str) = anthy_settings.GETA_UCS4;
	  #  else
		*((uint32_t*)GETA_UCS4_str) = htonl( anthy_settings.GETA_UCS4 );
	  #  endif
	  #else
	  # if defined(ICONV_UCS4BE)
		((uint8_t*)GETA_UCS4_str)[0] = (((uint32_t)anthy_settings.GETA_UCS4) >> 24) & UINT8_C(0xFF);
		((uint8_t*)GETA_UCS4_str)[1] = (((uint32_t)anthy_settings.GETA_UCS4) >> 16) & UINT8_C(0xFF);
		((uint8_t*)GETA_UCS4_str)[2] = (((uint32_t)anthy_settings.GETA_UCS4) >>  8) & UINT8_C(0xFF);
		((uint8_t*)GETA_UCS4_str)[3] = (((uint32_t)anthy_settings.GETA_UCS4) >>  0) & UINT8_C(0xFF);
	  # else
		((uint8_t*)GETA_UCS4_str)[0] = ((((uint32_t)anthy_settings.GETA_UCS4) & UINT32_C(0x000000FF)) >>  0);
		((uint8_t*)GETA_UCS4_str)[1] = ((((uint32_t)anthy_settings.GETA_UCS4) & UINT32_C(0x0000FF00)) >>  8);
		((uint8_t*)GETA_UCS4_str)[2] = ((((uint32_t)anthy_settings.GETA_UCS4) & UINT32_C(0x00FF0000)) >> 16);
		((uint8_t*)GETA_UCS4_str)[3] = ((((uint32_t)anthy_settings.GETA_UCS4) & UINT32_C(0xFF000000)) >> 24);
	  # endif
	  #endif
		GETA_UTF8_len = anthy_xchar_to_utf8_str( anthy_settings.GETA_UCS4, GETA_UTF8_str );
	}
	{
		const char*	encoding_name_default_eucjp = ICONV_EUCJP;
		switch (anthy_settings.anthy_mode.default_eucjp_encoding) {
		case ANTHY_ENCODING_EUCJP0213:
			#if defined(ICONV_EUCJP0213)
			encoding_name_default_eucjp = ICONV_EUCJP0213;
			#else
			encoding_name_default_eucjp = ICONV_EUCJP;
			#endif
			break;
		case ANTHY_ENCODING_EUCJP0212:
			encoding_name_default_eucjp = ICONV_EUCJP;
			break;
		case ANTHY_ENCODING_EUCJPMS:
			#if defined(ICONV_EUCJPMS)
			encoding_name_default_eucjp = ICONV_EUCJPMS;
			#else
			encoding_name_default_eucjp = ICONV_EUCJP;
			#endif
			break;
		case ANTHY_COMPILED_ENCODING:
		case ANTHY_ENCODING_EUCJP:
		case ANTHY_ENCODING_ISO2022JP:
		case ANTHY_ENCODING_SHIFTJIS:
		case ANTHY_ENCODING_SHIFTJIS0213:
		case ANTHY_ENCODING_SHIFTJISMS:
		case ANTHY_ENCODING_UTF8:
		case ANTHY_ENCODING_USERDEFINE:
		case ANTHY_ENCODING_MAX:
		default:
			anthy_log( 1, "Illigal parameter at ANTHY_ENCODING_EUCJP_AS_*. Fall back on EUC-JP JIS X 0212.\n" );
			break;
		}
		converter_tbl[ANTHY_ENCODING_EUCJP].encoding_name = encoding_name_default_eucjp;
	}
	{	int	failed = 0;
		failed |= anthy_open_iconv( ANTHY_COMPILED_ENCODING );
		failed |= anthy_open_iconv( ANTHY_ENCODING_EUCJP );
		failed |= anthy_open_iconv( ANTHY_ENCODING_UTF8 );
		if (anthy_settings.anthy_mode.encoding_name && anthy_settings.anthy_mode.encoding_name[0]) {
			failed |= anthy_open_iconv_userdefineencoding( anthy_settings.anthy_mode.encoding_name );
		}
		failed |= anthy_open_iconv( anthy_settings.anthy_mode.encoding );
		if (failed) {
			return 1;
		}
	}

	cd_euc_to_utf8      = libiconv_open( ICONV_UTF8, converter_tbl[ANTHY_ENCODING_EUCJP].encoding_name );
	cd_utf8_to_euc      = libiconv_open( converter_tbl[ANTHY_ENCODING_EUCJP].encoding_name, ICONV_UTF8 );
	if (
		((iconv_t)(-1) == cd_euc_to_utf8)
		|| ((iconv_t)(-1) == cd_utf8_to_euc)
	) {
		anthy_log( 1, "iconv initialization failed.(euc<->utf8)\n" );
		return 1;
	}
	libiconv( cd_euc_to_utf8, NULL, NULL, NULL, NULL );
	libiconv( cd_utf8_to_euc, NULL, NULL, NULL, NULL );

	return 0;
}

/** anthy_iconv饹õ
 *@retval				0						
 *@retval				1						
 *
 *@comment
 *	Patched by G-HAL
 *		Sun,07 Jun,2009 - Wed,10 Jun,2009
 */
int anthy_quit_iconv( void )
{
	libiconv_close( cd_utf8_to_euc );
	libiconv_close( cd_euc_to_utf8 );

	{	size_t i;
		for (i = 0; i < ANTHY_ENCODING_MAX; ++i) {
			struct encoding_converter_table_t* const	cvt = &(converter_tbl[i]);
			if (cvt->initialized) {
				libiconv_close( cvt->cd_internal_to );
				libiconv_close( cvt->cd_to_internal );
				cvt->initialized = 0;
			}
		}
	}

	return 0;
}
#endif /* defined(USE_ICONV) */
/* [ End of File ] */
/* vim:ts=4 sw=4 nomodified:
 */
