// ==============================================================
// D3D9Client.cpp
// Part of the ORBITER VISUALISATION PROJECT (OVP)
// Dual licensed under GPL v3 and LGPL v3
// Copyright (C) 2006-2016 Martin Schweiger
//				 2012-2016 Jarmo Nikkanen
// ==============================================================


#define STRICT 1
#define ORBITER_MODULE

#include <set> // ...for Brush-, Pen- and Font-accounting
#include "Orbitersdk.h"
#include "D3D9Client.h"
#include "D3D9Config.h"
#include "D3D9Util.h"
#include "D3D9Catalog.h"
#include "D3D9Surface.h"
#include "D3D9TextMgr.h"
#include "D3D9Frame.h"
#include "D3D9Pad.h"
#include "CSphereMgr.h"
#include "Scene.h"
#include "Mesh.h"
#include "VVessel.h"
#include "VStar.h"
#include "Texture.h"
#include "MeshMgr.h"
#include "Particle.h"
#include "TileMgr.h"
#include "RingMgr.h"
#include "HazeMgr.h"
#include "Log.h"
#include "VideoTab.h"
#include "GDIPad.h"
#include "windows.h"
#include "psapi.h"
#include "FileParser.h"
#include "OapiExtension.h"
#include "DebugControls.h"
#include "Tilemgr2.h"

#if defined(_MSC_VER) && (_MSC_VER <= 1700 ) // Microsoft Visual Studio Version 2012 and lower
#define round(v) floor(v+0.5)
#endif

// ==============================================================
// Structure definitions

struct D3D9Client::RenderProcData {
	__gcRenderProc proc;
	void *pParam;
	DWORD id;
};

struct D3D9Client::GenericProcData {
	__gcGenericProc proc;
	void *pParam;
	DWORD id;
};

using namespace oapi;

HINSTANCE g_hInst = 0;
D3D9Client *g_client = 0;
D3D9Catalog<D3D9Mesh*>			 *MeshCatalog;
D3D9Catalog<LPDIRECT3DTEXTURE9>	 *TileCatalog;
D3D9Catalog<LPD3D9CLIENTSURFACE> *SurfaceCatalog;

DWORD uCurrentMesh = 0;
vObject *pCurrentVisual = 0;
_D3D9Stats D3D9Stats;

#ifdef _NVAPI_H
 StereoHandle pStereoHandle = 0;
#endif

bool bFreeze = false;
bool bFreezeEnable = false;

// Module local constellation marker storage
static GraphicsClient::LABELSPEC *g_cm_list = NULL;
static DWORD g_cm_list_count = 0;

// Debuging Brush-, Pen- and Font-accounting
std::set<Font *> g_fonts;
std::set<Pen *> g_pens;
std::set<Brush *> g_brushes;

extern "C" {
	_declspec(dllexport) DWORD NvOptimusEnablement = 0x00000001;
}

// ==============================================================
// API interface
// ==============================================================

// ==============================================================
// Initialise module

void LogAttrips(DWORD AF, DWORD x, DWORD y, char *orig)
{
	char buf[512];
	sprintf_s(buf, 512, "%s (%d,%d)[0x%X]: ", orig, x, y, AF);
	if (AF&OAPISURFACE_TEXTURE)		 strcat_s(buf, 512, "OAPISURFACE_TEXTURE ");
	if (AF&OAPISURFACE_RENDERTARGET) strcat_s(buf, 512, "OAPISURFACE_RENDERTARGET ");
	if (AF&OAPISURFACE_GDI)			 strcat_s(buf, 512, "OAPISURFACE_GDI ");
	if (AF&OAPISURFACE_SKETCHPAD)	 strcat_s(buf, 512, "OAPISURFACE_SKETCHPAD ");
	if (AF&OAPISURFACE_MIPMAPS)		 strcat_s(buf, 512, "OAPISURFACE_MIPMAPS ");
	if (AF&OAPISURFACE_NOMIPMAPS)	 strcat_s(buf, 512, "OAPISURFACE_NOMIPMAPS ");
	if (AF&OAPISURFACE_ALPHA)		 strcat_s(buf, 512, "OAPISURFACE_ALPHA ");
	if (AF&OAPISURFACE_NOALPHA)		 strcat_s(buf, 512, "OAPISURFACE_NOALPHA ");
	if (AF&OAPISURFACE_UNCOMPRESS)	 strcat_s(buf, 512, "OAPISURFACE_UNCOMPRESS ");
	if (AF&OAPISURFACE_SYSMEM)		 strcat_s(buf, 512, "OAPISURFACE_SYSMEM ");
	LogDbg("BlueViolet", buf);
}


void MissingRuntimeError()
{
	MessageBoxA(NULL, "DirectX Runtimes may be missing. See /Doc/D3D9Client.pdf for more information", "D3D9Client Initialization Failed",MB_OK);
}

int PrintModules(DWORD pAdr)
{
	HMODULE hMods[1024];
	HANDLE hProcess;
	DWORD cbNeeded;
	unsigned int i;

	// Get a handle to the process.

	hProcess = OpenProcess( PROCESS_QUERY_INFORMATION | PROCESS_VM_READ, FALSE, GetProcessId(GetCurrentProcess()));

	if (NULL==hProcess) return 1;

	if (EnumProcessModules(hProcess, hMods, sizeof(hMods), &cbNeeded)) {
		for ( i = 0; i < (cbNeeded / sizeof(HMODULE)); i++ ) {
			char szModName[MAX_PATH];
			if (GetModuleFileNameExA(hProcess, hMods[i], szModName, sizeof(szModName))) {
				MODULEINFO mi;
				GetModuleInformation(hProcess, hMods[i], &mi, sizeof(MODULEINFO));
				DWORD Base = (DWORD)mi.lpBaseOfDll;
				if (pAdr>Base && pAdr<(Base+mi.SizeOfImage)) LogErr("%s EntryPoint=0x%8.8X, Base=0x%8.8X, Size=%u", szModName, mi.EntryPoint, mi.lpBaseOfDll, mi.SizeOfImage);
				else										 LogOk("%s EntryPoint=0x%8.8X, Base=0x%8.8X, Size=%u", szModName, mi.EntryPoint, mi.lpBaseOfDll, mi.SizeOfImage);
			}
		}
	}
	CloseHandle( hProcess );
	return 0;
}

bool bException = false;
bool bNVAPI = false;
DWORD ECode=0, EAddress=0;


int ExcHandler(EXCEPTION_POINTERS *p)
{
	EXCEPTION_RECORD *pER = p->ExceptionRecord;
	CONTEXT *pEC = p->ContextRecord;
	ECode = pER->ExceptionCode;
	EAddress = (DWORD)pER->ExceptionAddress;
	LogErr("Orbiter Version %d",oapiGetOrbiterVersion());
	LogErr("D3D9Client Build [%s]",__DATE__);
	LogErr("Exception Code=0x%8.8X, Address=0x%8.8X", ECode, EAddress);
	LogErr("EAX=0x%8.8X EBX=0x%8.8X ECX=0x%8.8X EDX=0x%8.8X ESI=0x%8.8X EDI=0x%8.8X EBP=0x%8.8X ESP=0x%8.8X EIP=0x%8.8X", pEC->Eax, pEC->Ebx, pEC->Ecx, pEC->Edx, pEC->Esi, pEC->Edi, pEC->Ebp, pEC->Esp, pEC->Eip);
	PrintModules(EAddress);
	bException = true;
	return 1;
}


DLLCLBK void InitModule(HINSTANCE hDLL)
{

#ifdef _DEBUG
	_CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
	// _CrtSetBreakAlloc(8351);
#endif

	D3D9InitLog("Modules/D3D9Client/D3D9ClientLog.html");

	if (!D3DXCheckVersion(D3D_SDK_VERSION, D3DX_SDK_VERSION)) {
		MissingRuntimeError();
		return;
	}

	QueryPerformanceFrequency((LARGE_INTEGER*)&qpcFrq);
	QueryPerformanceCounter((LARGE_INTEGER*)&qpcStart);

#ifdef _NVAPI_H
	if (NvAPI_Initialize()==NVAPI_OK) {
		LogAlw("[nVidia API Initialized]");
		bNVAPI = true;
	}
	else LogAlw("[nVidia API Not Available]");
#else
	LogAlw("[Not Compiled With nVidia API]");
#endif

	Config			= new D3D9Config();
	TileCatalog		= new D3D9Catalog<LPDIRECT3DTEXTURE9>();
	MeshCatalog		= new D3D9Catalog<D3D9Mesh*>;
	SurfaceCatalog	= new D3D9Catalog<LPD3D9CLIENTSURFACE>();

	DebugControls::Create();
	AtmoControls::Create();

	g_hInst = hDLL;
	g_client = new D3D9Client(hDLL);

	if (oapiRegisterGraphicsClient(g_client)==false) {
		delete g_client;
		g_client = 0;
	}
}

// ==============================================================
// Clean up module



DLLCLBK void ExitModule(HINSTANCE hDLL)
{
	LogAlw("--------------ExitModule------------");

	if (bException) LogErr("!!! Abnormal Program Termination !!!");

	delete TileCatalog;
	delete MeshCatalog;
	delete SurfaceCatalog;
	delete Config;

	DebugControls::Release();
	AtmoControls::Release();

	if (g_client) {
		oapiUnregisterGraphicsClient(g_client);
		g_client = 0;
	}

#ifdef _NVAPI_H
	if (bNVAPI) if (NvAPI_Unload()==NVAPI_OK) LogAlw("[nVidia API Unloaded]");
#endif

	LogAlw("Log Closed");
	D3D9CloseLog();
}

// ==============================================================
// D3D9Client class implementation
// ==============================================================

D3D9Client::D3D9Client (HINSTANCE hInstance) :
	GraphicsClient(hInstance),
	vtab(NULL),
	scenarioName("(none selected)"),
	pLoadLabel(""),
	pLoadItem(""),
	pDevice     (NULL),
	pDefaultTex (NULL),
	pScatterTest(NULL),
	pFramework  (NULL),
	pItemsSkp   (NULL),
	texmgr      (NULL),
	hLblFont1   (NULL),
	hLblFont2   (NULL),
	caps      (),
	parser    (),
	hRenderWnd(),
	scene     (),
	meshmgr   (),
	bControlPanel (false),
	bScatterUpdate(false),
	bFullscreen   (false),
	bAAEnabled    (false),
	bFailed       (false),
	bRunning      (false),
	bHalt         (false),
	bVertexTex    (false),
	bVSync        (false),
	bRendering	  (false),
	viewW         (0),
	viewH         (0),
	viewBPP       (0),
	frame_timer   (0),
	loadd_x       (0),
	loadd_y       (0),
	loadd_w       (0),
	loadd_h       (0),
	LabelPos (0)

{
}

// ==============================================================

D3D9Client::~D3D9Client()
{

	LogAlw("D3D9Client destructor called");
	SAFE_DELETE(vtab);

	// Free constellation names memory (if allocted)
	if (g_cm_list) {
		for (DWORD n = 0; n < g_cm_list_count; ++n) {
			delete[] g_cm_list[n].label[0];
			delete[] g_cm_list[n].label[1];
		}
		delete[] g_cm_list;
	}

	//
	// Orbiter seems not to release all resources :(
	//

	// --- Fonts
	if (g_fonts.size()) {
		LogErr("%u un-released fonts!", g_fonts.size());
		for (auto it = g_fonts.begin(); it != g_fonts.end(); ) {
			clbkReleaseFont(*it++);
		}
		g_fonts.clear();
	}
	// --- Brushes
	if (g_brushes.size()) {
		LogErr("%u un-released brushes!", g_brushes.size());
		for (auto it = g_brushes.begin(); it != g_brushes.end(); ) {
			clbkReleaseBrush(*it++);
		}
		g_brushes.clear();
	}
	// --- Pens
	if (g_pens.size()) {
		LogErr("%u un-released pens!", g_pens.size());
		for (auto it = g_pens.begin(); it != g_pens.end(); ) {
			clbkReleasePen(*it++);
		}
		g_pens.clear();
	}
}


// ==============================================================
// Overridden
//
const void *D3D9Client::GetConfigParam (DWORD paramtype) const
{
	return (paramtype >= CFGPRM_SHOWBODYFORCEVECTORSFLAG)
		 ? (paramtype >= CFGPRM_GETSELECTEDMESH)
		 ? DebugControls::GetConfigParam(paramtype)
		 : OapiExtension::GetConfigParam(paramtype)
		 : GraphicsClient::GetConfigParam(paramtype);
}


// ==============================================================
// This is called only once when the launchpad will appear
// This callback will initialize the Video tab only
//
bool D3D9Client::clbkInitialise()
{
	_TRACE;
	LogAlw("================ clbkInitialise ===============");
	LogAlw("Orbiter Version = %d",oapiGetOrbiterVersion());

	// Perform default setup
	if (GraphicsClient::clbkInitialise()==false) return false;
	//Create the Launchpad video tab interface
	vtab = new VideoTab(this, ModuleInstance(), OrbiterInstance(), LaunchpadVideoTab());

	return true;
}


// ==============================================================
// This is called when a simulation session will begin
//
HWND D3D9Client::clbkCreateRenderWindow()
{
	_TRACE;

	LogAlw("================ clbkCreateRenderWindow ===============");

	uEnableLog		 = Config->DebugLvl;
	pSplashScreen    = NULL;
	pBackBuffer      = NULL;
	pTextScreen      = NULL;
	hRenderWnd       = NULL;
	pDefaultTex		 = NULL;
	hLblFont1		 = NULL;
	hLblFont2		 = NULL;
	bControlPanel    = false;
	bFullscreen      = false;
	bFailed			 = false;
	bRunning		 = false;
	bHalt			 = false;
	bVertexTex		 = false;
	viewW = viewH    = 0;
	viewBPP          = 0;
	frame_timer		 = 0;
	scene            = NULL;
	meshmgr          = NULL;
	texmgr           = NULL;
	pFramework       = NULL;
	pDevice			 = NULL;
	parser			 = NULL;
	pBltGrpTgt		 = NULL;	// Let's set this NULL here, constructor is called only once. Not when exiting and restarting a simulation.
	pNoiseTex		 = NULL;
	surfBltTgt		 = NULL;	// This variable is not used, set it to NULL anyway
	hMainThread		 = GetCurrentThread();

	memset2(&D3D9Stats, 0, sizeof(D3D9Stats));

	D3DXMatrixIdentity(&ident);

	oapiDebugString()[0] = '\0';

	TileCatalog->Clear();
	MeshCatalog->Clear();
	SurfaceCatalog->Clear();

	hRenderWnd = GraphicsClient::clbkCreateRenderWindow();

	LogAlw("Window Handle = 0x%X",hRenderWnd);
	SetWindowText(hRenderWnd, "[D3D9Client]");

	LogOk("Starting to initialize device and 3D environment...");

	pFramework = new CD3DFramework9();

	if (pFramework->GetDirect3D()==NULL) return NULL;

	WriteLog("[DirectX 9 Initialized]");

	HRESULT hr = pFramework->Initialize(hRenderWnd, GetVideoData());

	if (hr!=S_OK) {
		LogErr("ERROR: Failed to initialize 3D Framework");
		return NULL;
	}

	RECT rect;
	GetClientRect(hRenderWnd, &rect);
	HDC hWnd = GetDC(hRenderWnd);
	HBRUSH hBr = CreateSolidBrush(RGB(0,0,0));
	FillRect(hWnd, &rect, hBr);
	DeleteObject(hBr);
	ReleaseDC(hRenderWnd, hWnd);
	ValidateRect(hRenderWnd, NULL);	// avoids white flash after splash screen

	caps = pFramework->caps;

	WriteLog("[3DDevice Initialized]");

	pDevice		= pFramework->GetD3DDevice();
	viewW		= pFramework->GetWidth();
	viewH		= pFramework->GetHeight();
	bFullscreen = (pFramework->IsFullscreen() == TRUE);
	bAAEnabled  = (pFramework->IsAAEnabled() == TRUE);
	viewBPP		= 32;
	bVertexTex  = (pFramework->HasVertexTextureSup() == TRUE);
	bVSync		= (pFramework->GetVSync() == TRUE);

	char fld[] = "D3D9Client";

	HR(D3DXCreateTextureFromFileA(pDevice, "Textures/D3D9Noise.dds", &pNoiseTex));

	D3D9ClientSurface::D3D9TechInit(this, pDevice, fld);

	HR(pDevice->GetRenderTarget(0, &pBackBuffer));
	HR(pDevice->GetDepthStencilSurface(&pDepthStencil));

	LogAlw("Render Target = 0x%X", pBackBuffer);
	LogAlw("DepthStencil = 0x%X", pDepthStencil);

	meshmgr		= new MeshManager(this);
	texmgr	    = new TextureManager(this);

	// Bring Sketchpad Online
	D3D9PadFont::D3D9TechInit(pDevice);
	D3D9PadPen::D3D9TechInit(pDevice);
	D3D9PadBrush::D3D9TechInit(pDevice);
	D3D9Text::D3D9TechInit(this, pDevice);
	D3D9Pad::D3D9TechInit(this, pDevice);

	deffont = (oapi::Font*) new D3D9PadFont(20, true, "fixed");
	defpen  = (oapi::Pen*)  new D3D9PadPen(1, 1, 0x00FF00);

	pDefaultTex = SURFACE(clbkLoadTexture("Null.dds"));
	if (pDefaultTex==NULL) LogErr("Null.dds not found");

	int x=0;
	if (viewW>1282) x=4;

	hLblFont1 = CreateFont(24+x, 0, 0, 0, 700, false, false, 0, 0, 3, 2, 1, 49, "Courier New");
	hLblFont2 = CreateFont(18+x, 0, 0, 0, 700, false, false, 0, 0, 3, 2, 1, 49, "Courier New");

	SplashScreen();  // Warning D3D9ClientSurface is not yet fully initialized here

	ShowWindow(hRenderWnd, SW_SHOW);

	OutputLoadStatus("Building Shader Programs...",0);

	D3D9Effect::D3D9TechInit(this, pDevice, fld);

	// Device-specific initialisations

	TileManager::GlobalInit(this);
	TileManager2Base::GlobalInit(this);
	PlanetRenderer::GlobalInit(this);
	RingManager::GlobalInit(this);
	HazeManager::GlobalInit(this);
	HazeManager2::GlobalInit(this);
	D3D9ParticleStream::GlobalInit(this);
	CSphereManager::GlobalInit(this);
	vStar::GlobalInit(this);
	vObject::GlobalInit(this);
	vVessel::GlobalInit(this);
	OapiExtension::GlobalInit(*Config);

	OutputLoadStatus("SceneTech.fx",1);
	Scene::D3D9TechInit(pDevice, fld);

	// Create scene instance
	scene = new Scene(this, viewW, viewH);

	WriteLog("[D3D9Client Initialized]");
	LogOk("...3D environment initialised");

#ifdef _NVAPI_H
	if (bNVAPI) {
		NvU8 bEnabled = 0;
		NvAPI_Status nvStereo = NvAPI_Stereo_IsEnabled(&bEnabled);

		if (nvStereo!=NVAPI_OK) {
			if (nvStereo==NVAPI_STEREO_NOT_INITIALIZED) LogWrn("Stereo API not initialized");
			if (nvStereo==NVAPI_API_NOT_INITIALIZED) LogErr("nVidia API not initialized");
			if (nvStereo==NVAPI_ERROR) LogErr("nVidia API ERROR");
		}

		if (bEnabled) {
			LogAlw("[nVidia Stereo mode is Enabled]");
			if (NvAPI_Stereo_CreateHandleFromIUnknown(pDevice, &pStereoHandle)!=NVAPI_OK) {
				LogErr("Failed to get StereoHandle");
			}
			else {
				if (pStereoHandle) {
					if (NvAPI_Stereo_SetConvergence(pStereoHandle, float(Config->Convergence))!=NVAPI_OK) {
						LogErr("SetConvergence Failed");
					}
					if (NvAPI_Stereo_SetSeparation(pStereoHandle, float(Config->Separation))!=NVAPI_OK) {
						LogErr("SetSeparation Failed");
					}
				}
			}
		}
		else LogAlw("[nVidia Stereo mode is Disabled]");
	}
#endif

	// Create status queries -----------------------------------------
	//
	if (pDevice->CreateQuery(D3DQUERYTYPE_OCCLUSION, NULL) == S_OK) LogAlw("D3DQUERYTYPE_OCCLUSION is supported by device");
	else LogAlw("D3DQUERYTYPE_OCCLUSION not supported by device");

	if (pDevice->CreateQuery(D3DQUERYTYPE_PIPELINETIMINGS, NULL)==S_OK) LogAlw("D3DQUERYTYPE_PIPELINETIMINGS is supported by device");
	else LogAlw("D3DQUERYTYPE_PIPELINETIMINGS not supported by device");

	if (pDevice->CreateQuery(D3DQUERYTYPE_BANDWIDTHTIMINGS, NULL) == S_OK) LogAlw("D3DQUERYTYPE_BANDWIDTHTIMINGS is supported by device");
	else LogAlw("D3DQUERYTYPE_BANDWIDTHTIMINGS not supported by device");

	if (pDevice->CreateQuery(D3DQUERYTYPE_PIXELTIMINGS, NULL) == S_OK) LogAlw("D3DQUERYTYPE_PIXELTIMINGS is supported by device");
	else LogAlw("D3DQUERYTYPE_PIXELTIMINGS not supported by device");

	return hRenderWnd;
}


// ==============================================================
// This is called when the simulation is ready to go but the clock
// is not yet ticking
//
void D3D9Client::clbkPostCreation()
{
	_TRACE;
	LogAlw("================ clbkPostCreation ===============");

	if (parser) return;

	parser = new FileParser(scenarioName);
	parser->LogContent();

	if (scene) scene->Initialise();

	MakeGenericProcCall(GENERICPROC_LOAD_ORBITERGUI);
	MakeGenericProcCall(GENERICPROC_LOAD);

	bRunning = true;

	LogAlw("=============== Loading Completed and Visuals Created ================");

	WriteLog("[Scene Initialized]");
}


// ==============================================================
// Called when simulation session is about to be closed
//
void D3D9Client::clbkCloseSession(bool fastclose)
{
	_TRACE;
	__TRY {

		LogAlw("================ clbkCloseSession ===============");


		// Check the status of RenderTarget Stack ------------------------------------------------
		//
		if (RenderStack.empty() == false) {
			LogErr("RenderStack contains %d items:", RenderStack.size());
			while (!RenderStack.empty()) {
				LogErr("RenderTarget=0x%X, DepthStencil=0x%X", RenderStack.front().pColor, RenderStack.front().pDepthStencil);
				RenderStack.pop_front();
			}
		}

		// Disable rendering and some other systems
		//
		bRunning = false;

		// At first, shutdown tile loaders -------------------------------------------------------
		//
		if (TileBuffer::ShutDown()==false) LogErr("Failed to Shutdown TileBuffer()");
		if (TileManager2Base::ShutDown()==false) LogErr("Failed to Shutdown TileManager2Base()");

		// Close dialog if Open and disconnect a visual form debug controls
		DebugControls::Close();

		// Disconnect textures from pipeline (Unlikely nesseccary)
		D3D9Effect::ShutDown();


		// DEBUG: List all textures connected to meshes
		/* DWORD cnt = MeshCatalog->CountEntries();
		for (DWORD i=0;i<cnt;i++) {
			D3D9Mesh *x = (D3D9Mesh*)MeshCatalog->Get(i);
			if (x) x->DumpTextures();
		} */
		//GraphicsClient::clbkCloseSession(fastclose);


		// Sent a signal for user applications to delete fonts, pens, and brushes.
		//
		MakeGenericProcCall(GENERICPROC_UNLOAD);
		MakeGenericProcCall(GENERICPROC_UNLOAD_ORBITERGUI);


		SAFE_DELETE(parser);
		LogAlw("================= Deleting Scene ================");
		Scene::GlobalExit();
		SAFE_DELETE(scene);
		LogAlw("============== Deleting Mesh Manager ============");
		SAFE_DELETE(meshmgr);
		WriteLog("[Session Closed. Scene deleted.]");
	}
	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("Exception in clbkCloseSession()");
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}
}

// ==============================================================

void D3D9Client::clbkDestroyRenderWindow (bool fastclose)
{
	_TRACE;
	oapiWriteLog("D3D9: [Destroy Render Window Called]");
	LogAlw("============= clbkDestroyRenderWindow ===========");

#ifdef _NVAPI_H
	if (bNVAPI) {
		if (pStereoHandle) {
			if (NvAPI_Stereo_DestroyHandle(pStereoHandle)!=NVAPI_OK) {
				LogErr("Failed to destroy stereo handle");
			}
		}
	}
#endif

	__TRY {

		LogAlw("=========== Clearing Texture Repository =========");
		SAFE_DELETE(texmgr);

		LogAlw("===== Calling GlobalExit() for sub-systems ======");
		HazeManager::GlobalExit();
		HazeManager2::GlobalExit();
		TileManager::GlobalExit();
		TileManager2Base::GlobalExit();
		PlanetRenderer::GlobalExit();
		D3D9ParticleStream::GlobalExit();
		CSphereManager::GlobalExit();
		vStar::GlobalExit();
		vVessel::GlobalExit();
		vObject::GlobalExit();

		SAFE_DELETE(defpen);
		SAFE_DELETE(deffont);

		DeleteObject(hLblFont1);
		DeleteObject(hLblFont2);

		D3D9Pad::GlobalExit();
		D3D9Text::GlobalExit();
		D3D9Effect::GlobalExit();
		D3D9ClientSurface::GlobalExit();

		SAFE_RELEASE(pSplashScreen);	// Splash screen related
		SAFE_RELEASE(pTextScreen);		// Splash screen related
		SAFE_RELEASE(pBackBuffer);
		SAFE_RELEASE(pDepthStencil);

		LPD3D9CLIENTSURFACE pBBuf = GetBackBufferHandle();

		SAFE_DELETE(pBBuf);
		SAFE_DELETE(pDefaultTex);
		SAFE_RELEASE(pNoiseTex);

		LogAlw("============ Checking Object Catalogs ===========");

		// Check surface catalog --------------------------------------------------------------------------------------
		//
		auto it = SurfaceCatalog->begin();
		if (it != SurfaceCatalog->end())
		{
			LogErr("UnDeleted Surface(s) Detected");
			while (it != SurfaceCatalog->end())
			{
				LogErr("Surface 0x%X (%s) (%u,%u)", *it, (*it)->GetName(), (*it)->GetWidth(), (*it)->GetHeight());
				delete *it;
				it = SurfaceCatalog->begin();
			}
		}

		// Check tile catalog --------------------------------------------------------------------------------------
		//
		DWORD nt = TileCatalog->CountEntries();
		if (nt) LogErr("SurfaceTile catalog contains %u unreleased entries",nt);

		SurfaceCatalog->Clear();
		MeshCatalog->Clear();
		TileCatalog->Clear();

		pFramework->DestroyObjects();

		SAFE_DELETE(pFramework);

		// Close Render Window -----------------------------------------
		GraphicsClient::clbkDestroyRenderWindow(fastclose);

		hRenderWnd		 = NULL;
		pDevice			 = NULL;
		bFailed			 = false;
		viewW = viewH    = 0;
		viewBPP          = 0;
	}

	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("Exception in clbkDestroyRenderWindow()");
		EmergencyShutdown();
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}
}

// ==============================================================

void D3D9Client::clbkSurfaceDeleted(LPD3D9CLIENTSURFACE hSurf)
{
	LPDIRECT3DSURFACE9 pSurf = hSurf->GetSurface();
	if (pSurf == NULL) return;

	for each (RenderTgtData data in RenderStack)
	{
		if (data.pColor == pSurf) {
			LogErr("Deleting a surface located in render stack 0x%X", DWORD(hSurf));
		}
	}
}

// ==============================================================

void D3D9Client::PushSketchpad(SURFHANDLE surf, D3D9Pad *pSkp)
{
	if (surf) {
		LPDIRECT3DSURFACE9 pTgt = SURFACE(surf)->GetSurface();
		LPDIRECT3DSURFACE9 pDep = SURFACE(surf)->GetDepthStencil();
		PushRenderTarget(pTgt, pDep, RENDERPASS_SKETCHPAD);
		RenderStack.front().pSkp = pSkp;
	}
}

// ==============================================================

void D3D9Client::PushRenderTarget(LPDIRECT3DSURFACE9 pColor, LPDIRECT3DSURFACE9 pDepthStencil, int code)
{
	static char *labels[] = { "NULL", "MAIN", "ENV", "CUSTOMCAM", "SHADOWMAP", "PICK", "SKETCHPAD", "OVERLAY" };

	RenderTgtData data;
	data.pColor = pColor;
	data.pDepthStencil = pDepthStencil;
	data.pSkp = NULL;
	data.code = code;

	if (pColor) {
		D3DSURFACE_DESC desc;
		pColor->GetDesc(&desc);
		D3DVIEWPORT9 vp = { 0, 0, desc.Width, desc.Height, 0.0f, 1.0f };
		pDevice->SetViewport(&vp);
	}

	pDevice->SetRenderTarget(0, pColor);
	pDevice->SetDepthStencilSurface(pDepthStencil);

	RenderStack.push_front(data);
	LogDbg("Plum", "PUSH:RenderStack[%d]={0x%X, 0x%X} %s", RenderStack.size(), DWORD(data.pColor), DWORD(data.pDepthStencil), labels[data.code]);
}

// ==============================================================

void D3D9Client::AlterRenderTarget(LPDIRECT3DSURFACE9 pColor, LPDIRECT3DSURFACE9 pDepthStencil)
{
	D3DSURFACE_DESC desc;
	pColor->GetDesc(&desc);
	D3DVIEWPORT9 vp = { 0, 0, desc.Width, desc.Height, 0.0f, 1.0f };

	pDevice->SetViewport(&vp);
	pDevice->SetRenderTarget(0, pColor);
	pDevice->SetDepthStencilSurface(pDepthStencil);
}

// ==============================================================

void D3D9Client::PopRenderTargets()
{
	static char *labels[] = { "NULL", "MAIN", "ENV", "CUSTOMCAM", "SHADOWMAP", "PICK", "SKETCHPAD", "OVERLAY" };

	assert(RenderStack.empty() == false);

	RenderStack.pop_front();

	if (RenderStack.empty()) {
		pDevice->SetRenderTarget(0, NULL);
		pDevice->SetDepthStencilSurface(NULL);
		LogDbg("Orange", "POP: Last one out ------------------------------------");
		return;
	}

	RenderTgtData data = RenderStack.front();

	if (data.pColor) {
		D3DSURFACE_DESC desc;
		data.pColor->GetDesc(&desc);
		D3DVIEWPORT9 vp = { 0, 0, desc.Width, desc.Height, 0.0f, 1.0f };

		pDevice->SetViewport(&vp);
		pDevice->SetRenderTarget(0, data.pColor);
		pDevice->SetDepthStencilSurface(data.pDepthStencil);
	}

	LogDbg("Plum", "POP:RenderStack[%d]={0x%X, 0x%X, 0x%X} %s", RenderStack.size(), DWORD(data.pColor), DWORD(data.pDepthStencil), DWORD(data.pSkp), labels[data.code]);
}

// ==============================================================

LPDIRECT3DSURFACE9 D3D9Client::GetTopDepthStencil()
{
	if (RenderStack.empty()) return NULL;
	return RenderStack.front().pDepthStencil;
}

// ==============================================================

LPDIRECT3DSURFACE9 D3D9Client::GetTopRenderTarget()
{
	if (RenderStack.empty()) return NULL;
	return RenderStack.front().pColor;
}

// ==============================================================

D3D9Pad *D3D9Client::GetTopInterface()
{
	if (RenderStack.empty()) return NULL;
	return RenderStack.front().pSkp;
}


// ==============================================================

void D3D9Client::clbkUpdate(bool running)
{
	_TRACE;
	double tot_update = D3D9GetTime();
	if (bFailed==false && bRunning) scene->Update();
	D3D9SetTime(D3D9Stats.Timer.Update, tot_update);
}

// ==============================================================

double frame_time = 0.0;
double scene_time = 0.0;

void D3D9Client::clbkRenderScene()
{
	_TRACE;

	if (pDevice==NULL || scene==NULL) return;
	if (bFailed) return;
	if (!bRunning) return;

	__TRY {

		if (Config->PresentLocation == 1) PresentScene();

		scene_time = D3D9GetTime();

		if (pDevice->TestCooperativeLevel()!=S_OK) {
			bFailed=true;
			MessageBoxA(pFramework->GetRenderWindow(),"Connection to Direct3DDevice is lost\nExit the simulation with Ctrl+Q and restart.\n\nAlt-Tabing not supported in a true fullscreen mode.\nDialog windows won't work with multi-sampling in a true fullscreen mode.","D3D9Client: Lost Device",0);
			return;
		}

		if (bHalt) {
			pDevice->BeginScene();
			RECT rect2 = {0,viewH-60,viewW,viewH-20};
			pFramework->GetLargeFont()->DrawTextA(0, "Critical error has occured", 26, &rect2, DT_CENTER | DT_TOP, D3DCOLOR_XRGB(255, 0, 0));
			rect2.left-=4; rect2.top-=4;
			pFramework->GetLargeFont()->DrawTextA(0, "Critical error has occured", 26, &rect2, DT_CENTER | DT_TOP, D3DCOLOR_XRGB(255, 255, 255));
			pDevice->EndScene();
			return;
		}

		UINT mem = pDevice->GetAvailableTextureMem()>>20;
		if (mem<32) TileBuffer::HoldThread(true);

		scene->RenderMainScene();		// Render the main scene

		VESSEL *hVes = oapiGetFocusInterface();

		if (hVes && Config->LabelDisplayFlags)
		{
			char Label[7] = "";
			if (Config->LabelDisplayFlags & D3D9Config::LABEL_DISPLAY_RECORD && hVes->Recording()) strcpy_s(Label, 7, "Record");
			if (Config->LabelDisplayFlags & D3D9Config::LABEL_DISPLAY_REPLAY && hVes->Playback()) strcpy_s(Label, 7, "Replay");

			if (Label[0]!=0) {
				pDevice->BeginScene();
				RECT rect2 = {0,viewH-60,viewW,viewH-20};
				pFramework->GetLargeFont()->DrawTextA(0, Label, 6, &rect2, DT_CENTER | DT_TOP, D3DCOLOR_XRGB(0, 0, 0));
				rect2.left-=4; rect2.top-=4;
				pFramework->GetLargeFont()->DrawTextA(0, Label, 6, &rect2, DT_CENTER | DT_TOP, D3DCOLOR_XRGB(255, 255, 255));
				pDevice->EndScene();
			}
		}

		D3D9SetTime(D3D9Stats.Timer.Scene, scene_time);


		if (bControlPanel) RenderControlPanel();

		// Compute total frame time
		D3D9SetTime(D3D9Stats.Timer.FrameTotal, frame_time);
		frame_time = D3D9GetTime();

		memset2(&D3D9Stats.Old, 0, sizeof(D3D9Stats.Old));
		memset2(&D3D9Stats.Surf, 0, sizeof(D3D9Stats.Surf));

	}

	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("Exception in clbkRenderScene()");
		EmergencyShutdown();
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}
}

// ==============================================================

void D3D9Client::clbkTimeJump(double simt, double simdt, double mjd)
{
	_TRACE;
	GraphicsClient::clbkTimeJump (simt, simdt, mjd);
}

// ==============================================================

void D3D9Client::PresentScene()
{
	double time = D3D9GetTime();

	if (bFullscreen == false) {
		RenderWithPopupWindows();
		pDevice->Present(0, 0, 0, 0);
	}
	else {
		if (!RenderWithPopupWindows()) pDevice->Present(0, 0, 0, 0);
	}

	D3D9SetTime(D3D9Stats.Timer.Display, time);
}

// ==============================================================

double framer_rater_limit = 0.0;

bool D3D9Client::clbkDisplayFrame()
{
	_TRACE;
//	static int iRefrState = 0;
	double time = D3D9GetTime();

	if (!bRunning) {
		RECT txt = { loadd_x, loadd_y, loadd_x+loadd_w, loadd_y+loadd_h };
		pDevice->StretchRect(pSplashScreen, NULL, pBackBuffer, NULL, D3DTEXF_POINT);
		pDevice->StretchRect(pTextScreen, NULL, pBackBuffer, &txt, D3DTEXF_POINT);
	}

	if (Config->PresentLocation == 0) PresentScene();

	double frmt = (1000000.0/Config->FrameRate) - (time - framer_rater_limit);

	framer_rater_limit = time;

	if (Config->EnableLimiter && Config->FrameRate>0 && bVSync==false) {
		if (frmt>0) frame_timer++;
		else        frame_timer--;
		if (frame_timer>40) frame_timer=40;
		Sleep(frame_timer);
	}

	return true;
}

// ==============================================================

void D3D9Client::clbkPreOpenPopup ()
{
	_TRACE;
	GetDevice()->SetDialogBoxMode(true);
}

// =======================================================================

static DWORD g_lastPopupWindowCount = 0;
static void FixOutOfScreenPositions (const HWND *hWnd, DWORD count)
{
	// Only check if a popup window is *added*
	if (count > g_lastPopupWindowCount)
	{
		for (DWORD i=0; i<count; ++i)
		{
			RECT rect;
			GetWindowRect(hWnd[i], &rect);

			int x = -1, y, w, h; // x != -1 indicates "position change needed"
			if (rect.left < 0) {
				x = 0;
				y = rect.top;
			}
			if (rect.top  < 0) {
				x = rect.left;
				y = 0;
			}

			// For the rest we need monitor information...
			HMONITOR monitor = MonitorFromWindow(hWnd[i], MONITOR_DEFAULTTONEAREST);
			MONITORINFO info;
			info.cbSize = sizeof(MONITORINFO);
			GetMonitorInfo(monitor, &info);

			int monitorWidth = info.rcMonitor.right - info.rcMonitor.left; // info.rcWork....
			int monitorHeight = info.rcMonitor.bottom - info.rcMonitor.top;

			if (rect.right > monitorWidth) {
				x = monitorWidth - (rect.right - rect.left);
				y = rect.top;
			}
			if (rect.bottom > monitorHeight) {
				x = rect.left;
				y = monitorHeight - (rect.bottom - rect.top);
			}

			if (x != -1) {
				w = rect.right - rect.left,
				h = rect.bottom - rect.top;
				MoveWindow(hWnd[i], x, y, w, h, FALSE);
			}
		}

	}
	g_lastPopupWindowCount = count;
}

// =======================================================================

bool D3D9Client::RenderWithPopupWindows()
{
	_TRACE;

	const HWND *hPopupWnd;
	DWORD count = GetPopupList(&hPopupWnd);

	if (bFullscreen) {
		if (count) GetDevice()->SetDialogBoxMode(true);
		else       GetDevice()->SetDialogBoxMode(false);
	}

	// Let the OapiExtension manager know about this..
	OapiExtension::HandlePopupWindows(hPopupWnd, count);

	FixOutOfScreenPositions(hPopupWnd, count);

	if (!bFullscreen) {
		for (DWORD i=0;i<count;i++) {
			DWORD val = GetWindowLongA(hPopupWnd[i], GWL_STYLE);
			if ((val&WS_SYSMENU)==0) {
				SetWindowLongA(hPopupWnd[i], GWL_STYLE, val|WS_SYSMENU);
			}
		}
	}

	return false;
}

#pragma region Particle stream functions

// =======================================================================
// Particle stream functions
// ==============================================================

ParticleStream *D3D9Client::clbkCreateParticleStream(PARTICLESTREAMSPEC *pss)
{
	LogErr("UnImplemented Feature Used clbkCreateParticleStream");
	return NULL;
}

// =======================================================================

ParticleStream *D3D9Client::clbkCreateExhaustStream(PARTICLESTREAMSPEC *pss,
	OBJHANDLE hVessel, const double *lvl, const VECTOR3 *ref, const VECTOR3 *dir)
{
	_TRACE;
	ExhaustStream *es = new ExhaustStream (this, hVessel, lvl, ref, dir, pss);
	scene->AddParticleStream (es);
	return es;
}

// =======================================================================

ParticleStream *D3D9Client::clbkCreateExhaustStream(PARTICLESTREAMSPEC *pss,
	OBJHANDLE hVessel, const double *lvl, const VECTOR3 &ref, const VECTOR3 &dir)
{
	_TRACE;
	ExhaustStream *es = new ExhaustStream (this, hVessel, lvl, ref, dir, pss);
	scene->AddParticleStream (es);
	return es;
}

// ======================================================================

ParticleStream *D3D9Client::clbkCreateReentryStream (PARTICLESTREAMSPEC *pss,
	OBJHANDLE hVessel)
{
	_TRACE;
	ReentryStream *rs = new ReentryStream (this, hVessel, pss);
	scene->AddParticleStream (rs);
	return rs;
}

#pragma endregion

// ==============================================================

ScreenAnnotation* D3D9Client::clbkCreateAnnotation()
{
	_TRACE;
	return GraphicsClient::clbkCreateAnnotation();
}

#pragma region Mesh functions

// ==============================================================

void D3D9Client::clbkStoreMeshPersistent(MESHHANDLE hMesh, const char *fname)
{
	_TRACE;
	if (fname) {
		LogAlw("Storing a mesh 0x%X (%s)",hMesh,fname);
		if (hMesh==NULL) LogErr("D3D9Client::clbkStoreMeshPersistent(%s) hMesh is NULL",fname);
	}
	else {
		LogAlw("Storing a mesh 0x%X",hMesh);
		if (hMesh==NULL) LogErr("D3D9Client::clbkStoreMeshPersistent() hMesh is NULL");
	}

	if (hMesh==NULL) return;

	int idx = meshmgr->StoreMesh(hMesh, fname);

	if (idx>=0) {
		if (fname) LogWrn("MeshGroup(%d) in a mesh %s is larger than 1km",idx,fname);
		else	   LogWrn("MeshGroup(%d) in a mesh 0x%X is larger than 1km",idx,hMesh);
	}
}

// ==============================================================

bool D3D9Client::clbkSetMeshTexture(DEVMESHHANDLE hMesh, DWORD texidx, SURFHANDLE surf)
{
	_TRACE;
	if (hMesh && surf) return ((D3D9Mesh*)hMesh)->SetTexture(texidx, SURFACE(surf));
	return false;
}

// ==============================================================

int D3D9Client::clbkSetMeshMaterial(DEVMESHHANDLE hMesh, DWORD matidx, const MATERIAL *mat)
{
	_TRACE;
	D3D9Mesh *mesh = (D3D9Mesh*)hMesh;
	DWORD nmat = mesh->GetMaterialCount();
	if (matidx >= nmat) return 4; // "index out of range"
	D3D9MatExt meshmat;
	//mesh->GetMaterial(&meshmat, matidx);
	CreateMatExt((const D3DMATERIAL9 *)mat, &meshmat);
	mesh->SetMaterial(&meshmat, matidx);
	return 0;
}

// ==============================================================

int D3D9Client::clbkMeshMaterial (DEVMESHHANDLE hMesh, DWORD matidx, MATERIAL *mat)
{
	_TRACE;
	D3D9Mesh *mesh = (D3D9Mesh*)hMesh;
	DWORD nmat = mesh->GetMaterialCount();
	if (matidx >= nmat) return 4; // "index out of range"
	const D3D9MatExt *meshmat = mesh->GetMaterial(matidx);
	if (meshmat) GetMatExt(meshmat, (D3DMATERIAL9 *)mat);
	return 0;
}

// ==============================================================

bool D3D9Client::clbkSetMeshProperty(DEVMESHHANDLE hMesh, DWORD prop, DWORD value)
{
	_TRACE;
	D3D9Mesh *mesh = (D3D9Mesh*)hMesh;
	switch (prop) {
		case MESHPROPERTY_MODULATEMATALPHA:
			mesh->EnableMatAlpha(value!=0);
			return true;
	}
	return false;
}

// ==============================================================
// Returns a mesh for a visual

MESHHANDLE D3D9Client::clbkGetMesh(VISHANDLE vis, UINT idx)
{
	_TRACE;
	if (vis==NULL) {
		LogErr("NULL visual in clbkGetMesh(NULL,%u)",idx);
		return NULL;
	}
	MESHHANDLE hMesh = ((vObject*)vis)->GetMesh(idx);
	if (hMesh==NULL) LogWrn("clbkGetMesh() returns NULL");
	return hMesh;
}

// =======================================================================

int D3D9Client::clbkEditMeshGroup(DEVMESHHANDLE hMesh, DWORD grpidx, GROUPEDITSPEC *ges)
{
	_TRACE;
	return ((D3D9Mesh*)hMesh)->EditGroup(grpidx, ges);
}

int D3D9Client::clbkGetMeshGroup (DEVMESHHANDLE hMesh, DWORD grpidx, GROUPREQUESTSPEC *grs)
{
	_TRACE;
	return ((D3D9Mesh*)hMesh)->GetGroup (grpidx, grs);
}

#pragma endregion

// ==============================================================

void D3D9Client::clbkNewVessel(OBJHANDLE hVessel)
{
	_TRACE;
	if (scene) scene->NewVessel(hVessel);
}

// ==============================================================

void D3D9Client::clbkDeleteVessel(OBJHANDLE hVessel)
{
	_TRACE;
	__TRY {
		if (scene) scene->DeleteVessel(hVessel);
	}
	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("Exception in clbkDeleteVessel()");
		EmergencyShutdown();
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}
}


// ==============================================================
// copy video options from the video tab

void D3D9Client::clbkRefreshVideoData()
{
	_TRACE;
	if (vtab) vtab->UpdateConfigData();
}

// ==============================================================

bool D3D9Client::clbkUseLaunchpadVideoTab() const
{
	_TRACE;
	return true;
}

// ==============================================================
// Fullscreen mode flag

bool D3D9Client::clbkFullscreenMode() const
{
	_TRACE;
	return bFullscreen;
}

// ==============================================================
// return the dimensions of the render viewport

void D3D9Client::clbkGetViewportSize(DWORD *width, DWORD *height) const
{
	_TRACE;
	*width = viewW, *height = viewH;
}

// ==============================================================
// Returns a specific render parameter

bool D3D9Client::clbkGetRenderParam(DWORD prm, DWORD *value) const
{
	_TRACE;
	switch (prm) {
		case RP_COLOURDEPTH:
			*value = viewBPP;
			return true;

		case RP_ZBUFFERDEPTH:
			*value = GetFramework()->GetZBufferBitDepth();
			return true;

		case RP_STENCILDEPTH:
			*value = GetFramework()->GetStencilBitDepth();
			return true;

		case RP_MAXLIGHTS:
			*value = MAX_SCENE_LIGHTS;
			return true;

		case RP_REQUIRETEXPOW2:
			*value = 0;
			return true;
	}
	return false;
}

// ==============================================================
// Responds to visual events

int D3D9Client::clbkVisEvent(OBJHANDLE hObj, VISHANDLE vis, DWORD msg, UINT context)
{
	_TRACE;
	VisObject *vo = (VisObject*)vis;
	vo->clbkEvent(msg, context);
	if (DebugControls::IsActive()) {
		if (msg==EVENT_VESSEL_INSMESH || msg==EVENT_VESSEL_DELMESH) {
			if (DebugControls::GetVisual()==vo) DebugControls::UpdateVisual();
		}
	}
	return 1;
}


// ==============================================================
//

void D3D9Client::EmergencyShutdown()
{
	bool b = oapiSaveScenario("D3D9ClientRescue","Contains the current simulation state before shutdown due to an unexpected error");
	if (b) LogAlw("ORBITER SCENARIO SAVED SUCCESSFULLY (D3D9ClientRescue.scn)");
	else   LogErr("FAILED TO SAVE A SCENARIO");
	bHalt = true;
}


// ==============================================================
// Message handler for render window

LRESULT D3D9Client::RenderWndProc (HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	_TRACE;

	static bool bTrackMouse = false;
	static short xpos=0, ypos=0;

	if (hRenderWnd!=hWnd && uMsg!= WM_NCDESTROY) {
		LogErr("Invalid Window !! RenderWndProc() called after calling clbkDestroyRenderWindow() uMsg=0x%X", uMsg);
		return 0;
	}

	if (bRunning && DebugControls::IsActive()) {
		// Must update camera to correspond MAIN_SCENE due to Pick() function,
		// because env-maps have altered camera settings
		GetScene()->UpdateCameraFromOrbiter(RENDERPASS_PICKSCENE);
	}

	__TRY {

		switch (uMsg) {

			case WM_MOUSELEAVE:
			{
				if (bTrackMouse && bRunning) GraphicsClient::RenderWndProc (hWnd, WM_LBUTTONUP, 0, 0);
				return 0;
			}

			case WM_MBUTTONDOWN:
			{
				break;
			}

			case WM_LBUTTONDOWN:
			{
				bTrackMouse = true;
				xpos = GET_X_LPARAM(lParam);
				ypos = GET_Y_LPARAM(lParam);
				TRACKMOUSEEVENT te; te.cbSize = sizeof(TRACKMOUSEEVENT); te.dwFlags = TME_LEAVE; te.hwndTrack = hRenderWnd;
				TrackMouseEvent(&te);

				bool bShift = (GetAsyncKeyState(VK_SHIFT) & 0x8000) != 0;
				bool bCtrl = (GetAsyncKeyState(VK_CONTROL) & 0x8000) != 0;

				// No Debug Controls
				if (bShift && bCtrl && !DebugControls::IsActive() && !oapiCameraInternal()) {

					D3D9Pick pick = GetScene()->PickScene(xpos, ypos);

					if (!pick.pMesh) break;

					OBJHANDLE hObj = pick.vObj->Object();
					if (oapiGetObjectType(hObj) == OBJTP_VESSEL) {
						oapiSetFocusObject(hObj);
					}

					break;
				}

				// With Debug Controls
				if (DebugControls::IsActive()) {

					DWORD flags = *(DWORD*)GetConfigParam(CFGPRM_GETDEBUGFLAGS);

					if (flags&DBG_FLAGS_PICK) {

						D3D9Pick pick = GetScene()->PickScene(xpos, ypos);

						if (!pick.pMesh) break;

						//sprintf_s(oapiDebugString(),256,"vObj=0x%X, Mesh=0x%X, Grp=%d, Face=%d", pick.vObj, pick.pMesh, pick.group, pick.face);

						if (bShift && bCtrl) {
							OBJHANDLE hObj = pick.vObj->Object();
							if (oapiGetObjectType(hObj)==OBJTP_VESSEL) {
								oapiSetFocusObject(hObj);
								break;
							}
						}
						else if (pick.group>=0) {
							DebugControls::SetVisual(pick.vObj);
							DebugControls::SelectMesh(pick.pMesh);
							DebugControls::SelectGroup(pick.group);
							DebugControls::SetGroupHighlight(true);
						}
					}
				}

				break;
			}

			case WM_LBUTTONUP:
			{
				if (DebugControls::IsActive()) {
					DWORD flags = *(DWORD*)GetConfigParam(CFGPRM_GETDEBUGFLAGS);
					if (flags&DBG_FLAGS_PICK) {
						DebugControls::SetGroupHighlight(false);
					}
				}
				bTrackMouse = false;
				break;
			}

			case WM_KEYDOWN:
			{
				if (DebugControls::IsActive()) {
					if (wParam == 'F') {
						if (bFreeze) bFreezeEnable = bFreeze = false;
						else bFreezeEnable = true;
					}
				}
				bool bShift = (GetAsyncKeyState(VK_SHIFT) & 0x8000)!=0;
				bool bCtrl  = (GetAsyncKeyState(VK_CONTROL) & 0x8000)!=0;
				if (wParam=='C' && bShift && bCtrl) bControlPanel = !bControlPanel;
				break;
			}

			case WM_MOUSEWHEEL:
			{
				if (DebugControls::IsActive()) {
					short d = GET_WHEEL_DELTA_WPARAM(wParam);
					if (d<-1) d=-1;
					if (d>1) d=1;
					double speed = *(double *)GetConfigParam(CFGPRM_GETCAMERASPEED);
					speed *= (DebugControls::GetVisualSize()/100.0);
					if (scene->CameraPan(_V(0,0,double(d))*2.0, speed)) return 0;
				}
			}

			case WM_MOUSEMOVE:
				if (DebugControls::IsActive()) {

					double x = double(GET_X_LPARAM(lParam) - xpos);
					double y = double(GET_Y_LPARAM(lParam) - ypos);
					xpos = GET_X_LPARAM(lParam);
					ypos = GET_Y_LPARAM(lParam);

					if (bTrackMouse) {
						double speed = *(double *)GetConfigParam(CFGPRM_GETCAMERASPEED);
						speed *= (DebugControls::GetVisualSize()/100.0);
						if (scene->CameraPan(_V(-x,y,0)*0.05, speed)) return 0;
					}

					//GetScene()->PickSurface(xpos, ypos);
				}
				break;

			case WM_MOVE:
				// If in windowed mode, move the Framework's window
				break;

			case WM_SYSCOMMAND:
				switch (wParam) {
					case SC_KEYMENU:
						// trap Alt system keys
						return 1;
					case SC_MOVE:
					case SC_SIZE:
					case SC_MAXIMIZE:
					case SC_MONITORPOWER:
						// Prevent moving/sizing and power loss in fullscreen mode
						if (bFullscreen) return 1;
						break;
				}
				break;

			case WM_SYSKEYUP:
				if (bFullscreen) return 0;  // trap Alt-key
				break;

		}
	}
	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("D3D9Client::RenderWndProc(hWnd=0x%X, uMsg=%u, wParam=%u, lParam=%u)",hWnd, uMsg, wParam, lParam);
		EmergencyShutdown();
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}

	__TRY {
		if (!bRunning && uMsg>=0x0200 && uMsg<=0x020E) return 0;
		return GraphicsClient::RenderWndProc (hWnd, uMsg, wParam, lParam);
	}
	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("GraphicsClient::RenderWndProc(hWnd=0x%X, uMsg=%u, wParam=%u, lParam=%u)",hWnd, uMsg, wParam, lParam);
		EmergencyShutdown();
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}

	return 0;
}


// ==============================================================
// Message handler for Launchpad "video" tab

BOOL D3D9Client::LaunchpadVideoWndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	_TRACE;
	if (vtab) return vtab->WndProc(hWnd, uMsg, wParam, lParam);
	else return false;
}

// =======================================================================

void D3D9Client::clbkRender2DPanel (SURFHANDLE *hSurf, MESHHANDLE hMesh, MATRIX3 *T, float alpha, bool additive)
{
	_TRACE;

	SURFHANDLE surf = NULL;
	DWORD ngrp = oapiMeshGroupCount(hMesh);

	if (ngrp==0) return;

	float sx = 1.0f/(float)(T->m11),  dx = (float)(T->m13);
	float sy = 1.0f/(float)(T->m22),  dy = (float)(T->m23);
	float vw = (float)viewW;
	float vh = (float)viewH;

	D3DXMATRIX mVP;
	D3DXMatrixOrthoOffCenterRH(&mVP, (0.0f-dx)*sx, (vw-dx)*sx, (vh-dy)*sy, (0.0f-dy)*sy, -100.0f, 100.0f);
	D3D9Effect::SetViewProjMatrix(&mVP);

	for (DWORD i=0;i<ngrp;i++) {

		float scale = 1.0f;

		MESHGROUP *gr = oapiMeshGroup(hMesh, i);

		if (gr->UsrFlag & 2) continue; // skip this group

		DWORD TexIdx = gr->TexIdx;

		if (TexIdx >= TEXIDX_MFD0) {
			int mfdidx = TexIdx - TEXIDX_MFD0;
			surf = GetMFDSurface(mfdidx);
			if (!surf) surf = (SURFHANDLE)pDefaultTex;
		} else if (hSurf) {
			surf = hSurf[TexIdx];
		}
		else surf = oapiGetTextureHandle (hMesh, gr->TexIdx+1);

		for (unsigned int k=0;k<gr->nVtx;k++) gr->Vtx[k].z = 0.0f;

		D3D9Effect::Render2DPanel(gr, SURFACE(surf), &ident, alpha, scale, additive);
	}
}

// =======================================================================

void D3D9Client::clbkRender2DPanel (SURFHANDLE *hSurf, MESHHANDLE hMesh, MATRIX3 *T, bool additive)
{
	_TRACE;
	clbkRender2DPanel (hSurf, hMesh, T, 1.0f, additive);
}

// =======================================================================

DWORD D3D9Client::clbkGetDeviceColour (BYTE r, BYTE g, BYTE b)
{
	_TRACE;
	return ((DWORD)r << 16) + ((DWORD)g << 8) + (DWORD)b;
}



#pragma region Surface, Blitting and Filling Functions



// =======================================================================
// Surface functions
// =======================================================================

bool D3D9Client::clbkSaveSurfaceToImage(SURFHANDLE  surf,  const char *fname, ImageFileFormat  fmt, float quality)
{
	_TRACE;
	if (surf==NULL) surf = pFramework->GetBackBufferHandle();

	LPDIRECT3DSURFACE9 pRTG = NULL;
	LPDIRECT3DSURFACE9 pSystem = NULL;
	LPDIRECT3DSURFACE9 pSurf = SURFACE(surf)->pSurf;

	if (pSurf==NULL) return false;

	bool bRet = false;
	ImageData ID;
	D3DSURFACE_DESC desc;
	D3DLOCKED_RECT pRect;

	SURFACE(surf)->GetDesc(&desc);

	if (desc.Pool!=D3DPOOL_SYSTEMMEM) {

		HR(pDevice->CreateRenderTarget(desc.Width, desc.Height, D3DFMT_X8R8G8B8, D3DMULTISAMPLE_NONE, 0, false, &pRTG, NULL));
		HR(pDevice->CreateOffscreenPlainSurface(desc.Width, desc.Height, D3DFMT_X8R8G8B8, D3DPOOL_SYSTEMMEM, &pSystem, NULL));
		HR(pDevice->StretchRect(pSurf, NULL, pRTG, NULL, D3DTEXF_NONE));
		HR(pDevice->GetRenderTargetData(pRTG, pSystem));

		if (pSystem->LockRect(&pRect, NULL, 0)==S_OK) {

			ID.bpp = 24;
			ID.height = desc.Height;
			ID.width = desc.Width;
			ID.stride = ((ID.width * ID.bpp + 31) & ~31) >> 3;
			ID.bufsize = ID.stride * ID.height;

			BYTE *tgt = ID.data = new BYTE[ID.bufsize];
			BYTE *src = (BYTE *)pRect.pBits;

			for (DWORD k=0;k<desc.Height;k++) {
				for (DWORD i=0;i<desc.Width;i++) {
					tgt[0+i*3] = src[0+i*4];
					tgt[1+i*3] = src[1+i*4];
					tgt[2+i*3] = src[2+i*4];
				}
				tgt += ID.stride;
				src += pRect.Pitch;
			}

			bRet = WriteImageDataToFile(ID, fname, fmt, quality);

			delete []ID.data;
			pSystem->UnlockRect();
		}

		pRTG->Release();
		pSystem->Release();
		return bRet;
	}

	if (pSurf->LockRect(&pRect, NULL, D3DLOCK_READONLY)==S_OK) {

		ID.bpp = 24;
		ID.height = desc.Height;
		ID.width = desc.Width;
		ID.stride = ((ID.width * ID.bpp + 31) & ~31) >> 3;
		ID.bufsize = ID.stride * ID.height;

		BYTE *tgt = ID.data = new BYTE[ID.bufsize];
		BYTE *src = (BYTE *)pRect.pBits;

		for (DWORD k=0;k<desc.Height;k++) {
			for (DWORD i=0;i<desc.Width;i++) {
				tgt[0+i*3] = src[0+i*4];
				tgt[1+i*3] = src[1+i*4];
				tgt[2+i*3] = src[2+i*4];
			}
			tgt += ID.stride;
			src += pRect.Pitch;
		}

		bRet = WriteImageDataToFile(ID, fname, fmt, quality);

		delete []ID.data;
		pSurf->UnlockRect();
		return bRet;
	}

	return false;
}

// ==============================================================

SURFHANDLE D3D9Client::clbkLoadTexture(const char *fname, DWORD flags)
{
	_TRACE;
	LPD3D9CLIENTSURFACE pTex = NULL;

	char path[MAX_PATH];

	if (TexturePath(fname, path)==false) {
		LogWrn("Texture %s not found.", fname);
		return NULL;
	}

	if (flags & 8) {
		if (texmgr->GetTexture(fname, &pTex, flags)==false) return NULL;
	}
	else {
		if (texmgr->LoadTexture(fname, &pTex, flags)!=S_OK) return NULL;
	}

	return pTex;
}

// ==============================================================

SURFHANDLE D3D9Client::clbkLoadSurface (const char *fname, DWORD attrib)
{
	_TRACE;
	DWORD flags = 0;

	// Process flag conflicts and issues ---------------------------------------------------------------------------------
	//
	flags = OAPISURFACE_RENDERTARGET|OAPISURFACE_SYSMEM;

	if ((attrib&flags)==flags) {
		LogErr("clbkLoadSurface() Can not combine OAPISURFACE_RENDERTARGET | OAPISURFACE_SYSMEM");
		attrib-=OAPISURFACE_SYSMEM;
	}

	if (attrib==OAPISURFACE_SKETCHPAD) {
		attrib |= OAPISURFACE_RENDERTARGET;
		attrib &= ~(OAPISURFACE_SYSMEM|OAPISURFACE_GDI);
	}

	D3D9ClientSurface *surf = new D3D9ClientSurface(pDevice, fname);
	surf->LoadSurface(fname, attrib);
	return surf;
}

// ==============================================================

void D3D9Client::clbkReleaseTexture(SURFHANDLE hTex)
{
	_TRACE;
	__TRY {

		if (texmgr->IsInRepository(hTex)) return;	// Do not release surfaces stored in repository

		if (SURFACE(hTex)->Release()) {
			for (auto it = MeshCatalog->cbegin(); it != MeshCatalog->cend(); ++it) {
				if (*it && (*it)->HasTexture(hTex)) {
					LogErr( "Something is attempting to delete a texture (%s) that is currently used by a mesh. Attempt rejected to prevent a CTD",
						    (*it)->GetName() );
					return;
				}
			}
			delete SURFACE(hTex);
		}
	}

	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("Exception in clbkReleaseTexture()");
		EmergencyShutdown();
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}
}

// ==============================================================

SURFHANDLE D3D9Client::clbkCreateSurfaceEx(DWORD w, DWORD h, DWORD attrib)
{
	_TRACE;

#ifdef _DEBUG
	LogAttrips(attrib, w, h, "CreateSrfEx");
#endif // _DEBUG

	if (w == 0 || h == 0) return NULL;	// Inline engine returns NULL for a zero surface

	if (attrib == 0x8C) attrib = 0x5;

	D3D9ClientSurface *surf = new D3D9ClientSurface(pDevice, "clbkCreateSurfaceEx");
	surf->CreateSurface(w, h, attrib);
	return surf;
}


// =======================================================================

SURFHANDLE D3D9Client::clbkCreateSurface(DWORD w, DWORD h, SURFHANDLE hTemplate)
{
	_TRACE;
	if (w == 0 || h == 0) return NULL;	// Inline engine returns NULL for a zero surface
	D3D9ClientSurface *surf = new D3D9ClientSurface(pDevice, "clbkCreateSurface");
	surf->MakeEmptySurfaceEx(w, h);
	return surf;
}

// =======================================================================

SURFHANDLE D3D9Client::clbkCreateSurface(HBITMAP hBmp)
{
	_TRACE;
	SURFHANDLE hSurf = GraphicsClient::clbkCreateSurface(hBmp);
	return hSurf;
}

// =======================================================================

SURFHANDLE D3D9Client::clbkCreateTexture(DWORD w, DWORD h)
{
	_TRACE;
	if (w == 0 || h == 0) return NULL;	// Inline engine returns NULL for a zero surface
	D3D9ClientSurface *pSurf = new D3D9ClientSurface(pDevice, "clbkCreateTexture");
	// DO NOT USE ALPHA
	pSurf->MakeEmptyTextureEx(w, h);
	return (SURFHANDLE)pSurf;
}

// =======================================================================

void D3D9Client::clbkIncrSurfaceRef(SURFHANDLE surf)
{
	_TRACE;
	if (surf==NULL) { LogErr("D3D9Client::clbkIncrSurfaceRef() Input Surface is NULL");	return; }
	SURFACE(surf)->IncRef();
}

// =======================================================================

bool D3D9Client::clbkReleaseSurface(SURFHANDLE surf)
{
	_TRACE;

	__TRY {

		if (surf==NULL) { LogErr("D3D9Client::clbkReleaseSurface() Input Surface is NULL");	return false; }

		if (texmgr->IsInRepository(surf)) return false;	// Do not release surfaces stored in repository

		bool bRel = SURFACE(surf)->Release();

		if (bRel) {

			for (auto it = MeshCatalog->cbegin(); it != MeshCatalog->cend(); ++it) {
				if (*it && (*it)->HasTexture(surf)) {
					LogErr( "Orbiter is attempting to delete a texture (%s) that is currently used by a mesh. Attempt rejected to prevent a CTD",
						    (*it)->GetName() );
					return true;
				}
			}
			delete SURFACE(surf);
		}
		return bRel;
	}

	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("Exception in clbkReleaseSurface()");
		EmergencyShutdown();
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}

	return false;
}

// =======================================================================

bool D3D9Client::clbkGetSurfaceSize(SURFHANDLE surf, DWORD *w, DWORD *h)
{
	_TRACE;
	if (surf==NULL) surf = pFramework->GetBackBufferHandle();
	*w = SURFACE(surf)->GetWidth();
	*h = SURFACE(surf)->GetHeight();
	return true;
}

// =======================================================================

bool D3D9Client::clbkSetSurfaceColourKey(SURFHANDLE surf, DWORD ckey)
{
	_TRACE;
	if (surf==NULL) { LogErr("Surface is NULL"); return false; }
	SURFACE(surf)->SetColorKey(ckey);
	return true;
}



// =======================================================================
// Blitting functions
// =======================================================================

int D3D9Client::clbkBeginBltGroup(SURFHANDLE tgt)
{
	_TRACE;
	if (pBltGrpTgt) return -1;

	if (tgt==RENDERTGT_NONE) {
		pBltGrpTgt = NULL;
		return -2;
	}

	if (tgt==RENDERTGT_MAINWINDOW) pBltGrpTgt = pFramework->GetBackBufferHandle();
	else						   pBltGrpTgt = tgt;

	if (SURFACE(pBltGrpTgt)->IsRenderTarget()==false) {
		pBltGrpTgt = NULL;
		return -3;
	}

	return 0;
}

int D3D9Client::clbkEndBltGroup()
{
	_TRACE;
	if (pBltGrpTgt==NULL) return -2;
	SURFACE(pBltGrpTgt)->EndBlitGroup(); // Flush queue and release GPU
	pBltGrpTgt = NULL;
	return 0;
}


bool D3D9Client::CheckBltGroup(SURFHANDLE src, SURFHANDLE tgt) const
{
	if (tgt==pBltGrpTgt) {
		if (SURFACE(src)->pTex!=NULL) {
			if (SURFACE(src)->ColorKey) { // Use blit groups only for color keyed source surfaces
				if (SURFACE(pBltGrpTgt)->BeginBlitGroup()==S_OK) return true;
			}
		}
	}
	// Flush queue, release GPU and disable blit group
	SURFACE(pBltGrpTgt)->EndBlitGroup();
	pBltGrpTgt = NULL;
	return false;
}



bool D3D9Client::clbkBlt(SURFHANDLE tgt, DWORD tgtx, DWORD tgty, SURFHANDLE src, DWORD flag) const
{
	_TRACE;

	__TRY {

		double time = D3D9GetTime();

		if (src==NULL) { LogErr("D3D9Client::clbkBlt() Source surface is NULL"); return false; }
		if (tgt==NULL) tgt = pFramework->GetBackBufferHandle();

		int w = SURFACE(src)->GetWidth();
		int h = SURFACE(src)->GetHeight();

		RECT rs = { 0, 0, w, h };
		RECT rt = { tgtx, tgty, tgtx+w, tgty+h };

		if (pBltGrpTgt) {
			if (CheckBltGroup(src,tgt)) SURFACE(pBltGrpTgt)->AddQueue(SURFACE(src), &rs, &rt);
			else 						SURFACE(tgt)->CopyRect(SURFACE(src), &rs, &rt, flag);
		}
		else SURFACE(tgt)->CopyRect(SURFACE(src), &rs, &rt, flag);

		D3D9SetTime(D3D9Stats.Timer.BlitTime, time);
	}

	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("Exception in clbkBlt(0x%X, %u,%u, 0x%X, 0x%X)",tgt, tgtx, tgty, src, flag);
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}
	return true;
}

// =======================================================================

bool D3D9Client::clbkBlt(SURFHANDLE tgt, DWORD tgtx, DWORD tgty, SURFHANDLE src, DWORD srcx, DWORD srcy, DWORD w, DWORD h, DWORD flag) const
{
	_TRACE;

	__TRY {
		double time = D3D9GetTime();

		if (src==NULL) { LogErr("D3D9Client::clbkBlt() Source surface is NULL"); return false; }
		if (tgt==NULL) tgt = pFramework->GetBackBufferHandle();

		RECT rs = { srcx, srcy, srcx+w, srcy+h };
		RECT rt = { tgtx, tgty, tgtx+w, tgty+h };

		if (pBltGrpTgt) {
			if (CheckBltGroup(src,tgt)) SURFACE(pBltGrpTgt)->AddQueue(SURFACE(src), &rs, &rt);
			else 						SURFACE(tgt)->CopyRect(SURFACE(src), &rs, &rt, flag);
		}
		else SURFACE(tgt)->CopyRect(SURFACE(src), &rs, &rt, flag);

		D3D9SetTime(D3D9Stats.Timer.BlitTime, time);
	}

	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("Exception in clbkBlt(0x%X, %u,%u, 0x%X, %u,%u,%u,%u, 0x%X)",tgt,tgtx,tgty, src, srcx, srcy, w, h, flag);
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}
	return true;
}

// =======================================================================

bool D3D9Client::clbkScaleBlt (SURFHANDLE tgt, DWORD tgtx, DWORD tgty, DWORD tgtw, DWORD tgth,
                               SURFHANDLE src, DWORD srcx, DWORD srcy, DWORD srcw, DWORD srch, DWORD flag) const
{
	_TRACE;

	__TRY {
		double time = D3D9GetTime();

		if (src==NULL) { LogErr("D3D9Client::clbkScaleBlt() Source surface is NULL"); return false; }
		if (tgt==NULL) tgt = pFramework->GetBackBufferHandle();

		RECT rs = { srcx, srcy, srcx+srcw, srcy+srch };
		RECT rt = { tgtx, tgty, tgtx+tgtw, tgty+tgth };

		if (pBltGrpTgt) {
			if (CheckBltGroup(src,tgt)) SURFACE(pBltGrpTgt)->AddQueue(SURFACE(src), &rs, &rt);
			else 						SURFACE(tgt)->CopyRect(SURFACE(src), &rs, &rt, flag);
		}
		else SURFACE(tgt)->CopyRect(SURFACE(src), &rs, &rt, flag);

		D3D9SetTime(D3D9Stats.Timer.BlitTime, time);
	}

	__EXCEPT(ExcHandler(GetExceptionInformation()))
	{
		LogErr("Exception in clbkScaleBlt(0x%X, %u,%u,%u,%u, 0x%X, %u,%u,%u,%u, 0x%X)",tgt, tgtx,tgty,tgtw,tgth, src, srcx, srcy, srcw, srch, flag);
		FatalAppExitA(0,"Critical error has occured. See Orbiter.log for details");
	}

	return true;
}

// =======================================================================

bool D3D9Client::clbkCopyBitmap(SURFHANDLE pdds, HBITMAP hbm, int x, int y, int dx, int dy)
{
	_TRACE;
	return GraphicsClient::clbkCopyBitmap(pdds, hbm, x,y,dx,dy);;
}

// =======================================================================

bool D3D9Client::clbkFillSurface(SURFHANDLE tgt, DWORD col) const
{
	_TRACE;
	if (tgt==NULL) tgt = pFramework->GetBackBufferHandle();
	bool ret = SURFACE(tgt)->Clear(col);
	return ret;
}

// =======================================================================

bool D3D9Client::clbkFillSurface(SURFHANDLE tgt, DWORD tgtx, DWORD tgty, DWORD w, DWORD h, DWORD col) const
{
	_TRACE;
	if (tgt==NULL) tgt = pFramework->GetBackBufferHandle();
	RECT r = {tgtx, tgty, tgtx+w, tgty+h};
	bool ret = SURFACE(tgt)->Fill(&r, col);
	return ret;
}

#pragma endregion

// =======================================================================
// Constellation name functions
// =======================================================================

DWORD D3D9Client::GetConstellationMarkers(const LABELSPEC **cm_list) const
{
	if ( !g_cm_list ) {
		#pragma pack(1)
		// File entry struct
		typedef struct {
			double lng;    ///< longitude
			double lat;    ///< latitude
			char   abr[3]; ///< abbreviation (short name)
			size_t len;    ///< length of 'fullname'
		} ConstellEntry;
		#pragma pack()

		FILE* file = NULL;
		const size_t e_size = sizeof(ConstellEntry);
		const float sphere_r = 1e6f; // the actual render distance for the celestial sphere
		                             // is irrelevant, since it is rendered without z-buffer,
		                             // but it must be within the frustum limits - check this
		                             // in case the near and far planes are dynamically changed!

		if ( 0 != fopen_s(&file, ".\\Constell2.bin", "rb") || file == NULL ) {
			LogErr("Could not open 'Constell2.bin'");
			return 0;
		}

		ConstellEntry f_entry;
		LABELSPEC *p_out;

		// Get number of labels from file
		while ( !feof(file) && (1 == fread(&f_entry, e_size, 1 , file)) ) {
			++g_cm_list_count;
			fseek(file, f_entry.len, SEEK_CUR);
		}

		rewind(file);

		g_cm_list = new LABELSPEC[g_cm_list_count]();

		for (p_out = g_cm_list; !feof(file); ++p_out) {
			if ( 1 == fread(&f_entry, e_size, 1 , file)) {
				p_out->label[0] = new char[f_entry.len+1]();
				p_out->label[1] = new char[4]();
				// position
				double xz = sphere_r * cos(f_entry.lat);
				p_out->pos.x = xz * cos(f_entry.lng);
				p_out->pos.z = xz * sin(f_entry.lng);
				p_out->pos.y = sphere_r * sin(f_entry.lat);
				// fullname
				fread(p_out->label[0], sizeof(char), f_entry.len, file);
				// shortname
				p_out->label[1][0] = f_entry.abr[0];
				p_out->label[1][1] = f_entry.abr[1];
				p_out->label[1][2] = f_entry.abr[2];
			}
		}

		fclose(file);
	}

	*cm_list = g_cm_list;

	return g_cm_list_count;
}

// =======================================================================
// GDI functions
// =======================================================================

HDC D3D9Client::clbkGetSurfaceDC(SURFHANDLE surf)
{
	_TRACE;
	if (surf==NULL) surf = pFramework->GetBackBufferHandle();
	HDC hDC = SURFACE(surf)->GetDC();
	return hDC;
}

// =======================================================================

void D3D9Client::clbkReleaseSurfaceDC(SURFHANDLE surf, HDC hDC)
{
	_TRACE;
	if (hDC==NULL) { LogErr("D3D9Client::clbkReleaseSurfaceDC() Input hDC is NULL"); return; }
	if (surf==NULL) surf = pFramework->GetBackBufferHandle();
	SURFACE(surf)->ReleaseDC(hDC);
}

// =======================================================================

bool D3D9Client::clbkSplashLoadMsg (const char *msg, int line)
{
	_TRACE;
	return OutputLoadStatus (msg, line);
}

// =======================================================================

LPD3D9CLIENTSURFACE D3D9Client::GetDefaultTexture() const
{
	return pDefaultTex;
}

// =======================================================================

HWND D3D9Client::GetWindow()
{
	return pFramework->GetRenderWindow();
}

// =======================================================================

LPD3D9CLIENTSURFACE D3D9Client::GetBackBufferHandle() const
{
	_TRACE;
	return SURFACE(pFramework->GetBackBufferHandle());
}

// =======================================================================

void D3D9Client::MakeRenderProcCall(Sketchpad *pSkp, DWORD id, LPD3DXMATRIX pV, LPD3DXMATRIX pP)
{
	for (auto it = RenderProcs.cbegin(); it != RenderProcs.cend(); ++it) {
		if (it->id == id) {
			D3D9Pad *pSkp2 = (D3D9Pad *)pSkp;
			pSkp2->LoadDefaults();
			pSkp2->SetViewMatrix((FMATRIX4 *)pV);
			pSkp2->SetProjectionMatrix((FMATRIX4 *)pP);
			it->proc(pSkp, it->pParam);
		}
	}
}

// =======================================================================

void D3D9Client::MakeGenericProcCall(DWORD id)
{
	for (auto it = GenericProcs.cbegin(); it != GenericProcs.cend(); ++it) {
		if (it->id == id) {
			it->proc(0, NULL, it->pParam);
		}
	}
}

// =======================================================================

bool D3D9Client::RegisterRenderProc(__gcRenderProc proc, DWORD id, void *pParam)
{
	if (id)	{ // register (add)
		RenderProcData data = { proc, pParam, id };
		RenderProcs.push_back(data);
		return true;
	}
	else { // unregister (remove)
		for (auto it = RenderProcs.cbegin(); it != RenderProcs.cend(); ++it) {
			if (it->proc == proc) {
				RenderProcs.erase(it);
				return true;
			}
		}
	}
	return false;
}

// =======================================================================

bool D3D9Client::RegisterGenericProc(__gcGenericProc proc, DWORD id, void *pParam)
{
	if (id) { // register (add)
		GenericProcData data = { proc, pParam, id };
		GenericProcs.push_back(data);
		return true;
	}
	else { // unregister (remove)
		for (auto it = GenericProcs.cbegin(); it != GenericProcs.cend(); ++it) {
			if (it->proc == proc) {
				GenericProcs.erase(it);
				return true;
			}
		}
	}
	return false;
}

// =======================================================================

void D3D9Client::WriteLog(const char *msg) const
{
	_TRACE;
	char cbuf[256];
	sprintf_s(cbuf, 256, "D3D9: %s", msg);
	oapiWriteLog(cbuf);
}

// =======================================================================

bool D3D9Client::OutputLoadStatus(const char *txt, int line)
{

	if (bRunning) return false;

	if (line == 1) strcpy_s(pLoadItem, 127, txt); else
	if (line == 0) strcpy_s(pLoadLabel, 127, txt), pLoadItem[0] = '\0'; // New top line => clear 2nd line

	if (pTextScreen) {

		if (pDevice->TestCooperativeLevel()!=S_OK) {
			LogErr("TestCooperativeLevel() Failed");
			return false;
		}

		RECT txt = { loadd_x, loadd_y, loadd_x+loadd_w, loadd_y+loadd_h };

		pDevice->StretchRect(pSplashScreen, &txt, pTextScreen, NULL, D3DTEXF_POINT);

		HDC hDC;
		HR(pTextScreen->GetDC(&hDC));

		HFONT hO = (HFONT)SelectObject(hDC, hLblFont1);
		SetTextColor(hDC, 0xE0A0A0);
		SetBkMode(hDC,TRANSPARENT);
		SetTextAlign(hDC, TA_LEFT|TA_TOP);

		TextOut(hDC, 2, 2, pLoadLabel, strlen(pLoadLabel));

		SelectObject(hDC, hLblFont2);
		TextOut(hDC, 2, 36, pLoadItem, strlen(pLoadItem));

		HPEN pen = CreatePen(PS_SOLID,1,0xE0A0A0);
		HPEN po = (HPEN)SelectObject(hDC, pen);

		MoveToEx(hDC, 0, 32, NULL);
		LineTo(hDC, loadd_w, 32);

		SelectObject(hDC, po);
		SelectObject(hDC, hO);
		DeleteObject(pen);

		HR(pTextScreen->ReleaseDC(hDC));
		HR(pDevice->StretchRect(pSplashScreen, NULL, pBackBuffer, NULL, D3DTEXF_POINT));
		HR(pDevice->StretchRect(pTextScreen, NULL, pBackBuffer, &txt, D3DTEXF_POINT));

		IDirect3DSwapChain9 *pSwap;

		if (pDevice->GetSwapChain(0, &pSwap)==S_OK) {
			pSwap->Present(0, 0, 0, 0, D3DPRESENT_DONOTWAIT);
			pSwap->Release();
			return true;
		}

		// Prevent "Not Responding" during loading
		MSG msg;
		while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) DispatchMessage(&msg);
	}
	return false;
}

// =======================================================================

void D3D9Client::SplashScreen()
{

	loadd_x = 279*viewW/1280;
	loadd_y = 545*viewH/800;
	loadd_w = viewW/3;
	loadd_h = 80;

	RECT rS;

	GetWindowRect(hRenderWnd, &rS);

	LogAlw("Splash Window Size = [%u, %u]", rS.right - rS.left, rS.bottom - rS.top);
	LogAlw("Splash Window LeftTop = [%d, %d]", rS.left, rS.top);

	HR(pDevice->TestCooperativeLevel());
	HR(pDevice->Clear(0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER|D3DCLEAR_STENCIL, 0x0, 1.0f, 0L));
	HR(pDevice->CreateOffscreenPlainSurface(loadd_w, loadd_h, D3DFMT_X8R8G8B8, D3DPOOL_DEFAULT, &pTextScreen, NULL));
	HR(pDevice->CreateOffscreenPlainSurface(viewW, viewH, D3DFMT_X8R8G8B8, D3DPOOL_DEFAULT, &pSplashScreen, NULL));

	D3DXIMAGE_INFO Info;

	HMODULE hOrbiter =  GetModuleHandleA("orbiter.exe");
	HRSRC hRes = FindResourceA(hOrbiter, MAKEINTRESOURCEA(292), "IMAGE");
	HGLOBAL hImage = LoadResource(hOrbiter, hRes);
	LPVOID pData = LockResource(hImage);
	DWORD size = SizeofResource(hOrbiter, hRes);

	// Splash screen image is 1920 x 1200 pixel
	double scale = min(viewW / 1920.0, viewH / 1200.0);
	double _w = (1920.0 * scale);
	double _h = (1200.0 * scale);
	double _l = abs(viewW - _w)/2.0;
	double _t = abs(viewH - _h)/2.0;
	RECT imgRect = {
		static_cast<LONG>( round(_l) ),
		static_cast<LONG>( round(_t) ),
		static_cast<LONG>( round(_w + _l) ),
		static_cast<LONG>( round(_h + _t) )
	};
	HR(pDevice->ColorFill(pSplashScreen, NULL, D3DCOLOR_XRGB(0, 0, 0)));
	HR(D3DXLoadSurfaceFromFileInMemory(pSplashScreen, NULL, &imgRect, pData, size, NULL, D3DX_FILTER_LINEAR, 0, &Info));

	HDC hDC;
	HR(pSplashScreen->GetDC(&hDC));

	LOGFONTA fnt; memset2((void *)&fnt, 0, sizeof(LOGFONT));

	fnt.lfHeight		 = 18;
	fnt.lfWeight		 = 700;
	fnt.lfCharSet		 = ANSI_CHARSET;
	fnt.lfOutPrecision	 = OUT_DEFAULT_PRECIS;
	fnt.lfClipPrecision	 = CLIP_DEFAULT_PRECIS;
	fnt.lfQuality		 = ANTIALIASED_QUALITY;
	fnt.lfPitchAndFamily = DEFAULT_PITCH;
	strcpy_s(fnt.lfFaceName, "Courier New");

	HFONT hF = CreateFontIndirect(&fnt);

	HFONT hO = (HFONT)SelectObject(hDC, hF);
	SetTextColor(hDC, 0xE0A0A0);
	SetBkMode(hDC,TRANSPARENT);

	char *months[]={"???","Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec","???"};

	DWORD d = oapiGetOrbiterVersion();
	DWORD y = d/10000; d-=y*10000;
	DWORD m = d/100; d-=m*100;
	if (m>12) m=0;

#ifdef _DEBUG
	char dataA[]={"D3D9Client Beta R28.8 (Debug Build) [" __DATE__ "]"};
#else
	char dataA[]={"D3D9Client Beta R28.8 [" __DATE__ "]"};
#endif

	char dataB[128]; sprintf_s(dataB,128,"Build %s %lu 20%lu [%u]", months[m], d, y, oapiGetOrbiterVersion());
	char dataD[]={"Warning: Config folder not present in /Modules/Server/. Please create symbolic link."};

	int xc = viewW*750/1280;
	int yc = viewH*545/800;

	TextOut(hDC, xc, yc + 0*20, "ORBITER Space Flight Simulator",30);
	TextOut(hDC, xc, yc + 1*20, dataB, strlen(dataB));
	TextOut(hDC, xc, yc + 2*20, dataA, strlen(dataA));

	DWORD cattrib = GetFileAttributes("Modules/Server/Config");

	if ((cattrib&0x10)==0 || cattrib==INVALID_FILE_ATTRIBUTES) {
		SetTextAlign(hDC, TA_CENTER);
		TextOut(hDC, viewW/2, viewH-50, dataD, strlen(dataD));
	}

	SelectObject(hDC, hO);
	DeleteObject(hF);

	HR(pSplashScreen->ReleaseDC(hDC));


	RECT src = { loadd_x, loadd_y, loadd_x+loadd_w, loadd_y+loadd_h };
	pDevice->StretchRect(pSplashScreen, &src, pTextScreen, NULL, D3DTEXF_POINT);
	pDevice->StretchRect(pSplashScreen, NULL, pBackBuffer, NULL, D3DTEXF_POINT);
	pDevice->Present(0, 0, 0, 0);
}

// =======================================================================

HRESULT D3D9Client::BeginScene()
{
	bRendering = false;
	HRESULT hr = pDevice->BeginScene();
	if (hr == S_OK) bRendering = true;
	return hr;
}

// =======================================================================

void D3D9Client::EndScene()
{
	pDevice->EndScene();
	bRendering = false;
}

// =======================================================================

#pragma region Drawing_(Sketchpad)_Interface


double sketching_time;

// =======================================================================
// 2D Drawing Interface
//
oapi::Sketchpad *D3D9Client::clbkGetSketchpad(SURFHANDLE surf)
{
	_TRACE;
	oapi::Sketchpad *pSkp = NULL;

	if (GetCurrentThread() != hMainThread) {
		_wassert(L"Sketchpad called from a worker thread !", _CRT_WIDE(__FILE__), __LINE__);
		return NULL;
	}

	sketching_time = D3D9GetTime();

	if (surf == NULL) surf = GetBackBufferHandle();

	if (SURFACE(surf)->IsRenderTarget()) {

		// Get Pooled Sketchpad
		D3D9Pad *pPad = SURFACE(surf)->GetD3D9Pad();

		// Get Current interface if any
		D3D9Pad *pCur = GetTopInterface();

		// Do we have an existing SketchPad interface in use
		if (pCur) {
			if (pCur == pPad) _wassert(L"Sketchpad already exists for this surface", _CRT_WIDE(__FILE__), __LINE__);
			pCur->EndDrawing();	// Put the current one in hold
			LogDbg("Red", "Switching to another sketchpad in a middle");
		}

		// Push a new Sketchpad onto a stack
		PushSketchpad(surf, pPad);

		pPad->BeginDrawing();
		pPad->LoadDefaults();

		pSkp = pPad;
	}
	else {
		HDC hDC = SURFACE(surf)->GetDC();
		if (hDC) pSkp = new GDIPad(surf, hDC);
	}

	return pSkp;
}

// =======================================================================

void D3D9Client::clbkReleaseSketchpad(oapi::Sketchpad *sp)
{
	_TRACE;
	if (!sp) return;

	SURFHANDLE hSrf = sp->GetSurface();

	if (SURFACE(hSrf)->IsRenderTarget()) {

		D3D9Pad *pPad = ((D3D9Pad*)sp);

		if (GetTopInterface() != pPad) _wassert(L"Sketchpad release failed. Not a top one.", _CRT_WIDE(__FILE__), __LINE__);

		pPad->EndDrawing();

		PopRenderTargets();

		// Do we have an old interface ?
		D3D9Pad *pOld = GetTopInterface();
		if (pOld) {
			pOld->BeginDrawing();	// Continue with the old one
			LogDbg("Red", "Continue Previous Sketchpad");
		}
	}
	else {
		GDIPad *pGDI = (GDIPad *)sp;
		SURFACE(hSrf)->ReleaseDC(pGDI->GetDC());
		delete pGDI;
	}

	sketching_time = D3D9GetTime() - sketching_time;
}

// =======================================================================

Font *D3D9Client::clbkCreateFont(int height, bool prop, const char *face, Font::Style style, int orientation) const
{
	_TRACE;
	return *g_fonts.insert(new D3D9PadFont(height, prop, face, style, orientation)).first;
}

// =======================================================================

/*Font *D3D9Client::clbkCreateFont(int height, bool prop, const char *face, Font::Style style, int orientation) const
{
	_TRACE;
	DWORD flags = 0;
	flags |= (style & Font::BOLD) ? FNT_BOLD : 0;
	flags |= (style & Font::ITALIC) ? FNT_ITALIC : 0;
	flags |= (style & Font::UNDERLINE) ? FNT_UNDERLINE : 0;
	return clbkCreateFontEx(height, 0, prop, face, flags, orientation);
}*/

// =======================================================================

Font *D3D9Client::clbkCreateFontEx(int height, int width, bool prop, const char *face, DWORD flags, int orientation) const
{
	_TRACE;
	return NULL;
	//return *g_fonts.insert(new D3D9PadFont(height, width, prop, face, flags, orientation)).first;
}

// =======================================================================

void D3D9Client::clbkReleaseFont(Font *font) const
{
	_TRACE;
	g_fonts.erase(font);
	delete ((D3D9PadFont*)font);
}

// =======================================================================

Pen *D3D9Client::clbkCreatePen(int style, int width, DWORD col) const
{
	_TRACE;
	return *g_pens.insert(new D3D9PadPen(style, width, col)).first;
}

// =======================================================================

void D3D9Client::clbkReleasePen(Pen *pen) const
{
	_TRACE;
	g_pens.erase(pen);
	delete ((D3D9PadPen*)pen);
}

// =======================================================================

Brush *D3D9Client::clbkCreateBrush(DWORD col) const
{
	_TRACE;
	return *g_brushes.insert(new D3D9PadBrush(col)).first;
}

// =======================================================================

void D3D9Client::clbkReleaseBrush(Brush *brush) const
{
	_TRACE;
	g_brushes.erase(brush);
	delete ((D3D9PadBrush*)brush);
}

#pragma endregion


// ======================================================================
// class VisObject

VisObject::VisObject(OBJHANDLE hObj) : hObj(hObj)
{
	_TRACE;
}

// =======================================================================

VisObject::~VisObject ()
{
}
