/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina.translate;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.io.Reader;
import java.util.List;

import net.morilib.automata.DFA;
import net.morilib.automata.dfa.ConvertedDFA;
import net.morilib.automata.dfa.ConvertedRangeDFA;
import net.morilib.automata.dfa.MinimizedRangeDFA;
import net.morilib.nina.NinaNFA;
import net.morilib.nina.NinaParser;
import net.morilib.nina.Quadro;
import net.morilib.nina.cmd.NinatOptions;
import net.morilib.nina.translate.sh.ShNinatBuiltInCommands;
import net.morilib.nina.translate.sh.ShNinatFileSystem;
import net.morilib.sh.DefaultShRuntime;
import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShFacade;
import net.morilib.sh.ShFileSystem;
import net.morilib.sh.ShProcess;
import net.morilib.sh.ShRootEnvironment;
import net.morilib.sh.ShRuntime;
import net.morilib.sh.ShSyntaxException;
import net.morilib.sh.misc.XtraceStream;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/23
 */
public abstract class AbstractNinaTranslator
implements NinaTranslator {

	/**
	 * 
	 */
	protected NinatOptions options;

	/**
	 * 
	 */
	protected DFA<Object, ?, Void> dfa;

	/**
	 * 
	 */
	protected String machine;

	//
	private String fragment;

	/**
	 * 
	 * @param lang
	 * @return
	 */
	public static NinaTranslator getTranslator(String lang) {
		Class<?> c;
		String p;

		try {
			p = AbstractNinaTranslator.class.getPackage().getName();
			c = Class.forName(p + ".NinaTranslator" + lang);
			return (NinaTranslator)c.newInstance();
		} catch(ClassNotFoundException e) {
			return null;
		} catch(InstantiationException e) {
			throw new RuntimeException(e);
		} catch(IllegalAccessException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 
	 * @param dfa
	 */
	public abstract void printStates(PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	public abstract void printObjectStates(PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	public abstract void printClassStates(PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	public abstract void printAcceptStates(PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	public abstract void printAcceptToken(PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	public abstract void printActions(PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	public abstract void printImports(List<String> imp,
			PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	protected abstract InputStream openScript() throws IOException;

	/**
	 * 
	 * @param dfa
	 */
	protected abstract PrintStream openOutput() throws IOException;

	//
	private void processNFA(NinaNFA n, String p, String t,
			ShNinatBuiltInCommands cmd,
			ShEnvironment env) throws IOException {
		PrintStream ous = null;
		InputStream ins = null;
		ShFileSystem fs;
		XtraceStream qs;
		ShRuntime run;

		if(t == null) {
			dfa = ConvertedRangeDFA.convertDFA(n);
		} else {
			dfa = ConvertedDFA.convertDFA(n);
		}

		if(!options.getOption("minimize").equalsIgnoreCase("no")) {
			dfa = MinimizedRangeDFA.newInstance(dfa);
		}

		try {
			ins  = openScript();
			ous  = openOutput();
			fs   = new ShNinatFileSystem(fragment, options);
			cmd.putCommand("print_token", new ShProcess() {

				@Override
				public int main(ShEnvironment env, ShFileSystem fs,
						InputStream stdin, PrintStream stdout,
						PrintStream stderr,
						String... args) throws IOException {
					printAcceptToken(stdout);
					return 0;
				}

			});
			run  = new DefaultShRuntime(cmd);
			qs   = new XtraceStream(System.out);
			ShFacade.execute(env, fs, cmd, run, ins, System.in, ous,
					System.err, qs);
		} catch(ShSyntaxException e) {
			throw new RuntimeException(e);
		} finally {
			if(ins != null)  ins.close();
			if(ous != null)  ous.close();
		}
	}

	//
	private void processDFA(
			DFA<Object, Object, Void> d, String p,
			ShNinatBuiltInCommands cmd,
			ShEnvironment env) throws IOException {
		PrintStream ous = null;
		InputStream ins = null;
		ShFileSystem fs;
		XtraceStream qs;
		ShRuntime run;

		dfa = d;
		try {
			ins  = openScript();
			ous  = openOutput();
			fs   = new ShNinatFileSystem(fragment, options);
			cmd.putCommand("print_actions", new ShProcess() {

				@Override
				public int main(ShEnvironment env, ShFileSystem fs,
						InputStream stdin, PrintStream stdout,
						PrintStream stderr,
						String... args) throws IOException {
					printActions(stdout);
					return 0;
				}

			});
			run  = new DefaultShRuntime(cmd);
			qs   = new XtraceStream(System.out);
			ShFacade.execute(env, fs, cmd, run, ins, System.in, ous,
					System.err, qs);
		} catch(ShSyntaxException e) {
			throw new RuntimeException(e);
		} finally {
			if(ins != null)  ins.close();
			if(ous != null)  ous.close();
		}
	}

	private ShEnvironment getenv(String p, Quadro q) {
		ShEnvironment env;
		String s;
		char[] a;

		env  = new ShRootEnvironment();
		a    = options.getFilename().toCharArray();
		a[0] = Character.toUpperCase(a[0]);
		env.bind("CLASSNAME", new String(a));
		env.bind("PACKAGE", p);
		env.bind("FILENAME", options.getFilename());
		s = (s = options.getOption("bufsize")).equals("") ?
				"1024" : s;
		env.bind("BUFSIZE", s);
		env.bind("TEMPLATE", q.getOption("template"));
		env.bind("EXTENDS", q.getOption("extends"));
		env.bind("IMPLEMENTS", q.getOption("implements"));

		if((s = q.getType()) == null || s.equals("char")) {
			env.bind("TYPE", "");
			env.bind("CTYPE", "int");
		} else if(s.equals("string")) {
			env.bind("TYPE", s);
			env.bind("CTYPE", "String");
		} else if(s.equals("class")) {
			env.bind("TYPE", s);
			env.bind("CTYPE", "Object");
		} else {
			env.bind("TYPE", s);
			env.bind("CTYPE", s);
		}
		return env;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.compiler.NinaTranslater#translate(java.io.Reader, java.io.PrintWriter)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void translate(Reader rd,
			NinatOptions opts) throws IOException {
		ShNinatBuiltInCommands cmd;
		ShEnvironment env;
		final Quadro q;
		String p, m;
		Object o;

		options = opts;
		q = Quadro.read(rd);
		o = NinaParser.complie(q, opts.getLibraryList());
		if((m = q.getOption("machine")) == null) {
			// do nothing
		} else {
			m = "unknown";
		}

		fragment = q.getFragment();
		if((p = opts.getOption("package")) != null &&
				!p.equals("")) {
			// do nothing
		} else if((p = q.getRootPackage()) != null &&
				!p.equals("")) {
			// do nothing
		} else {
			p = "";
		}

		env = getenv(p, q);
		cmd = new ShNinatBuiltInCommands();
		cmd.putCommand("print_states", new ShProcess() {

			@Override
			public int main(ShEnvironment env, ShFileSystem fs,
					InputStream stdin, PrintStream stdout,
					PrintStream stderr,
					String... args) throws IOException {
				if(q.getType() == null) {
					printStates(stdout);
				} else if(q.getType().equals("class")) {
					printClassStates(stdout);
				} else {
					printObjectStates(stdout);
				}
				return 0;
			}

		});
		cmd.putCommand("print_accepts", new ShProcess() {

			@Override
			public int main(ShEnvironment env, ShFileSystem fs,
					InputStream stdin, PrintStream stdout,
					PrintStream stderr,
					String... args) throws IOException {
				printAcceptStates(stdout);
				return 0;
			}

		});
		cmd.putCommand("print_imports", new ShProcess() {

			@Override
			public int main(ShEnvironment env, ShFileSystem fs,
					InputStream stdin, PrintStream stdout,
					PrintStream stderr,
					String... args) throws IOException {
				printImports(q.getImports(), stdout);
				return 0;
			}

		});

		if(o instanceof NinaNFA) {
			machine = "nfa";
			processNFA((NinaNFA)o, p, q.getType(), cmd, env);
		} else if(o instanceof DFA) {
			machine = "dfa";
			processDFA((DFA<Object, Object, Void>)o, p, cmd, env);
		} else {
			opts.perror("machinenotsupport", m);
		}
	}

}
