/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import junit.framework.TestCase;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/13
 */
public class NinaMatcherTest extends TestCase {

	static void mat(NinaPattern m, String s) {
		NinaMatcher x;

		x = m.matcher(s);
		assertTrue(x.matches());
	}

	static void nmt(NinaPattern m, String s) {
		NinaMatcher x;

		x = m.matcher(s);
		assertFalse(x.matches());
	}

	static void lok(NinaPattern m, String s, String d) {
		NinaMatcher x;

		x = m.matcher(s);
		assertTrue(x.lookingAt());
		assertEquals(d, x.group());
	}

	static void fnd(NinaPattern m, String s, String d) {
		NinaMatcher x;

		x = m.matcher(s);
		assertTrue(x.find());
		assertEquals(d, x.group());
	}

	public void test0001() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >-----------------a-------------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[a-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "a");  
		mat(m, "aaa");
		mat(m, "aaaaa");
		nmt(m, "");
		nmt(m, "aa");
		nmt(m, "aaaa");
		mat(m, "a");  
		mat(m, "aba");
		mat(m, "abada");

		lok(m, "aza", "aza");
		lok(m, "azaz", "aza");
		assertFalse(m.matcher("baza").lookingAt());

		fnd(m, "baza", "aza");
		fnd(m, "zbazaz", "aza");
		assertFalse(m.matcher("bbb").find());
	}

	public void test0002() {
		String s =
				"#machine DFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >-----------------a-------------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[b-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		assertTrue(m.matcher("a").matches());  
		assertTrue(m.matcher("aba").matches());
		assertTrue(m.matcher("ababa").matches());
		assertFalse(m.matcher("").matches());
		assertFalse(m.matcher("ab").matches());
		assertFalse(m.matcher("abab").matches());
		assertTrue(m.matcher("a").matches());  
		assertTrue(m.matcher("aba").matches());
		assertTrue(m.matcher("abada").matches());

		lok(m, "aza", "aza");
		lok(m, "azaz", "aza");
		assertFalse(m.matcher("baza").lookingAt());

		fnd(m, "baza", "aza");
		fnd(m, "zbazaz", "aza");
		assertFalse(m.matcher("bbb").find());
	}

	public void test0003() {
		String s =
				"#machine NFABuilder\n" +
				" =======     *****                      @*****\n" +
				" =aaa  >----->b  >-------a-------------->aa  *\n" +
				" =======     **v**                      **^***\n" +
				"               \\---------b----------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "a");  
		mat(m, "b");
		nmt(m, "");
		nmt(m, "ab");
		nmt(m, "ba");
	}

	public void test0004() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------`a+`------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[c-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "a");
		mat(m, "aaca");
		mat(m, "aaacaa");
		nmt(m, "");
		nmt(m, "aac");
		nmt(m, "aaaac");

		lok(m, "aaza", "aaza");
		lok(m, "aazaz", "aaza");
		assertFalse(m.matcher("baza").lookingAt());

		fnd(m, "baaza", "aaza");
		fnd(m, "zbazaaz", "azaa");
		assertFalse(m.matcher("bbb").find());
	}

	public void test0005() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >-------------`(ab)+`------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------[c-z]--------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "ab");
		mat(m, "ababcab");
		mat(m, "abababcabab");
		nmt(m, "");
		nmt(m, "ababc");
		nmt(m, "ababababc");
		nmt(m, "abababacab");
	}

	public void test0006() {
		String s =
				"#machine DFABuilder\n" +
				"   /a------------------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" ==^====             **v***\n" +
				"   \\------------------b/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "aba");
	}

	public void test0011() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'\\n'------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'\\n'---------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "\n");
		mat(m, "\n\n\n");
		mat(m, "\n\n\n\n\n");
		nmt(m, "");
		nmt(m, "\n\n");
		nmt(m, "\n\n\n\n");
	}

	public void test0012() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'\\r'------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'\\r'---------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "\r");
		mat(m, "\r\r\r");
		mat(m, "\r\r\r\r\r");
		nmt(m, "");
		nmt(m, "\r\r");
		nmt(m, "\r\r\r\r");
	}

	public void test0013() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'\\t'------------>aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'\\t'---------------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "\t");
		mat(m, "\t\t\t");
		mat(m, "\t\t\t\t\t");
		nmt(m, "");
		nmt(m, "\t\t");
		nmt(m, "\t\t\t\t");
	}

	public void test0014() {
		String s =
				"#machine NFABuilder\n" +
				" =======                                @*****\n" +
				" =aaa  >----------------'\\n\\r\\t'-------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\-------------------'\\n\\r\\t'-----------/  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "\n\r\t");
		mat(m, "\n\r\t\n\r\t\n\r\t");
		mat(m, "\n\r\t\n\r\t\n\r\t\n\r\t\n\r\t");
		nmt(m, "");
		nmt(m, "\n\r\t\n\r\t");
		nmt(m, "\n\r\t\n\r\t\n\r\t\n\r\t");
	}

	public void test0015() {
		String s =
				"#machine DFABuilder\n" +
				"   /[\\d]---------------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" =v^====             **v***\n" +
				"  |\\---------------[\\D]/  \n" +
				"  |     ****  \n" +
				"  \\[\\D]->  *  \n" +
				"        ****  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "0a1a2a3b4c5d6e7f8g9");
	}

	public void test0016() {
		String s =
				"#machine DFABuilder\n" +
				"   /[\\w]---------------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" =v^====             **v***\n" +
				"  |\\---------------[\\W]/  \n" +
				"  |     ****  \n" +
				"  \\[\\W]->  *  \n" +
				"        ****  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "A,B,C,D,E,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z");
		mat(m, "a,b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,x,y,z");
		mat(m, "0,1,2,3,4,5,6,7,8,9,_");
	}

	public void test0017() {
		String s =
				"#machine DFABuilder\n" +
				"   /[\\s]---------------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" =v^====             **v***\n" +
				"  |\\---------------[\\S]/  \n" +
				"  |     ****  \n" +
				"  \\[\\S]->  *  \n" +
				"        ****  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "\t:\r:\n:\u000b:\f:\r: ");
	}

	public void test0018() {
		String s =
				"#machine DFABuilder\n" +
				"   /[\\p{Space}]--------\\  \n" +
				" ==^====             @*v***\n" +
				" =aaa  =             *aa  *\n" +
				" =v^====             **v***\n" +
				"  |\\---------------[\\S]/  \n" +
				"  |     ****  \n" +
				"  \\[\\S]->  *  \n" +
				"        ****  \n" +
				"";
		NinaPattern m;

		m = Nina.pattern(s);
		mat(m, "\t:\r:\n:\u000b:\f:\r: ");
	}

}
