/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.BufferedReader;
import java.io.IOException;

import net.morilib.automata.NFA;
import net.morilib.automata.NFAState;
import net.morilib.range.Range;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/13
 */
public class NFABuilder implements NinaAction {

	private NinaNFA nfa;
	private NinaState vertex;
	private String label;

	/**
	 * 
	 * @param q
	 */
	public NFABuilder() {
		nfa = new NinaNFA();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#labelAdded(int, java.lang.String, net.morilib.nina.NinaState)
	 */
	@Override
	public void labelAdded(NinaEvent q, NinaFrameReader rd,
			boolean accept) {
		BufferedReader br = new BufferedReader(rd);

		try {
			label = br.readLine();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		if(vertex != null) {
			// create a vertex
			q.setScratch(vertex);
			link(q, new NinaState(this, label));
		} else {
			// set initial state if it is not set
			vertex = new NinaState(this, label);
			nfa.initial = vertex;
		}

		if(accept) {
			// accepted state
			nfa.accept.put(vertex, null);
		}
	}

	@SuppressWarnings("unchecked")
	public void link(NinaEvent q, Object ver) {
		NinaState v = vertex;

		vertex = (NinaState)ver;
		if(q.getEdge() instanceof NFAProxy) {
			nfa.linkNFA(v, vertex,
					(NFA<Object, NFAState, Void>)q.getEdge());
		} else if(q.getEdge() instanceof Range) {
			nfa.linkAlphabet(v, vertex, (Range)q.getEdge());
		} else {
			nfa.linkAlphabet(v, vertex, q.getEdge());
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getLabel()
	 */
	@Override
	public String getLabel() {
		return label;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getVertex()
	 */
	@Override
	public Object getVertex() {
		return vertex;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setVertex(java.lang.Object)
	 */
	@Override
	public void setVertex(Object o) {
		vertex = (NinaState)o;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#doneBlockSearching(net.morilib.nina.NinaEvent)
	 */
	@Override
	public void doneBlockSearching(NinaEvent q) {
		// searching a block is done
		vertex = (NinaState)q.getScratch();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdge(net.morilib.nina.NinaEvent, java.lang.Character)
	 */
	@Override
	public void setEdge(NinaEvent q, Object o) {
		q.setEdge(o);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdgeResource(net.morilib.nina.NinaEvent, net.morilib.nina.NinaParser, java.lang.String)
	 */
	@Override
	public void setEdgeResource(NinaEvent q, final NinaParser p,
			final String s) {
		q.setEdge(new NFAProxy<Object, NFAState, Void>() {

			private NFA<Object, NFAState, Void> nfa = null;

			@SuppressWarnings("unchecked")
			@Override
			protected NFA<Object, NFAState, Void> proxy() {
				if(nfa == null) {
					nfa = (NFA<Object, NFAState, Void>)p._compileRes(
							s, new NFABuilder());
				}
				return nfa;
			}

		});
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#accept()
	 */
	@Override
	public Object accept() {
		return nfa;
	}

}
