/*
 * sys module program copyright (C) 2009 H.Niwa
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <errno.h>
#include <setjmp.h>
#include <sys/time.h>
#include <math.h>
#include <libgen.h>
#include <setjmp.h>

#ifndef __MINGW32__ 
#include <sys/times.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <sys/utsname.h>
#endif /* __MINGW32__ */

#include <string>
#include <complex>

#include "syserr.h"

#include "bin_node.h"
#include "gc.h"
#include "var.h"
#include "pred.h"
#include "context.h"
#include "unify.h"
#include "builtin.h"
#include "sysmodule.h"
#include "expression.h"
#include "let.h"
#include "func.h"
#include "token.h"
#include "module.h"
#include "code.h"
#include "timeout.h"
#include "help.h"
#include "sleep.h"
#include "complx.h"
#include "lisp.h"
#include "ncurlib.h"
#include "compiler.h"

#define MAXPATHLEN 4096

extern int TraceFlag;
extern jmp_buf program_jb;

extern void PushStack(Context* cx, Node* goals, Node* md, Node* env);
extern int PopStack(Context* cx, Node* &goals, Node* &md, Node* &env);

extern Node* UTF8Char(char* str);
extern Node* SJISChar(char* str);
extern Node* EUCChar(char* str);

extern FILE* MksTemp(char* templ);


int CmdArgs(Context* cx, Node* goalscar, List* module);
int DlibPath(Context* cx, Node* goalscar, List* module);
int CutAll(Context* cx, Node* goalscar, List* module);
int Write(Context* cx, Node* goalscar, List* module);
int WriteNl(Context* cx, Node* goalscar, List* module);

int eq(Context* cx, Node* n, List* module);
int noteq(Context* cx, Node* n, List* module);
int isNil(Context* cx, Node* n, List* module);
int isAtom(Context* cx, Node* n, List* module);
int isList(Context* cx, Node* n, List* module);
int isPred(Context* cx, Node* n, List* module);
int isVar(Context* cx, Node* n, List* module);
int isUndefVar(Context* cx, Node* n, List* module);
int isFloat(Context* cx, Node* n, List* module);
int isInteger(Context* cx, Node* n, List* module);


int isTrue(Context* cx, Node* goalscar, List* module);
int isFalse(Context* cx, Node* goalscar, List* module);
int isUnknown(Context* cx, Node* goalscar, List* module);

int Max(Context* cx, Node* goalscar, List* module);
int Min(Context* cx, Node* goalscar, List* module);
int Maxf(Context* cx, Node* goalscar, List* module);
int Minf(Context* cx, Node* goalscar, List* module);
int Sum(Context* cx, Node* goalscar, List* module);
int Sumf(Context* cx, Node* goalscar, List* module);
int Avg(Context* cx, Node* goalscar, List* module);
int Avgf(Context* cx, Node* goalscar, List* module);


int DoOpenR(Context* cx, Node* goalscar, List* module);
int DoOpenW(Context* cx, Node* goalscar, List* module);
int DoOpenWP(Context* cx, Node* goalscar, List* module);

int DoGetc(Context* cx, Node* goalscar, List* module);
int DoPutc(Context* cx, Node* goalscar, List* module);
int GetLine(Context* cx, Node* goalscar, List* module);
int SyntaxLine(Context* cx, Node* goalscar, List* module);
int TmpFile(Context* cx, Node* goalscar, List* module);
int Line(Context* cx, Node* goalscar, List* module);

int DoRegex(Context* cx, Node* goalscar, List* module);
int DoSub(Context* cx, Node* goalscar, List* module);
int DoGSub(Context* cx, Node* goalscar, List* module);
int Split(Context* cx, Node* goalscar, List* module);
int Toupper(Context* cx, Node* goalscar, List* module);
int Tolower(Context* cx, Node* goalscar, List* module);
int Length(Context* cx, Node* goalscar, List* module);
int Random(Context* cx, Node* goalscar, List* module);

int Char(Context* cx, Node* goalscar, List* module);
int Concat(Context* cx, Node* goalscar, List* module);
int ConcatCode(Context* cx, Node* goalscar, List* module);
int LeftStr(Context* cx, Node* goalscar, List* module);
int RightStr(Context* cx, Node* goalscar, List* module);
int SubStr(Context* cx, Node* goalscar, List* module);
int InsertStr(Context* cx, Node* goalscar, List* module);

int SetCode(Context* cx, Node* goalscar, List* module);
int CodeCharPrd(Context* cx, Node* goalscar, List* module);
int UTF8CharPrd(Context* cx, Node* goalscar, List* module);
int EUCCharPrd(Context* cx, Node* goalscar, List* module);
int SJISCharPrd(Context* cx, Node* goalscar, List* module);

int And(Context* cx, Node* goalscar, List* module);
int Or(Context* cx, Node* goalscar, List* module);
int Xor(Context* cx, Node* goalscar, List* module);
int BitNot(Context* cx, Node* goalscar, List* module);
int ShiftL(Context* cx, Node* goalscar, List* module);
int ShiftR(Context* cx, Node* goalscar, List* module);

int DoMkPred(Context* cx, Node* goalscar, List* module);

int DoCountNode(Context* cx, Node* goalscar, List* module);

int GetTime(Context* cx, Node* goalscar, List* module);
int Time(Context* cx, Node* goalscar, List* module);
int Date(Context* cx, Node* goalscar, List* module);
int Sleep(Context* cx, Node* goalscar, List* module);
int USleep(Context* cx, Node* goalscar, List* module);
int Pause(Context* cx, Node* goalscar, List* module);

int BaseName(Context* cx, Node* goalscar, List* module);
int DirName(Context* cx, Node* goalscar, List* module);
int Suffix(Context* cx, Node* goalscar, List* module);

#ifndef __MINGW32__
int ClearScreen(Context* cx, Node* goalscar, List* module);
int Uname(Context* cx, Node* goalscar, List* module);
#endif


// program start time
struct ProgTime progtime;


// lib path
Node*	dlibpathnode = Nil;

int FuncArg(Context* cx, Node*& args, Node* goalscar, List* module)
{
	Node* retn;
	int rn;
			
	cxpush(cx, goalscar);
	cxpush(cx, args);
	cxpush(cx, module);
	if ((rn=FuncPred(cx, args, module, retn))>0) {
		cxpop(cx);
		cxpop(cx);
		cxpop(cx);
		args = retn->Val();
	} else {
		cxpop(cx);
		cxpop(cx);
		cxpop(cx);
	}
	return rn;
}

void GetLibPath(char* dlibpath)
{
	if (dlibpath == NULL) {
		dlibpath = "";
	}
	dlibpathnode = Nil;
	if (dlibpath != NULL) {
		int	i, j;
		char*	pathbuf = new char[strlen(dlibpath)];
		
		for (i = 0, j = 0; i <= strlen(dlibpath); i++, j++) {
			switch (dlibpath[i]) {
#ifdef __MINGW32__
			case ';' :	// semicolon
#else 
			case ':' :	
#endif
			case 0 :
				pathbuf[j] = 0;
				dlibpathnode = Append(dlibpathnode, 
							MkList(mka(pathbuf)));
				j = -1;
				break;
			default :
				pathbuf[j] = dlibpath[i];
				break;
			}
		}
		delete pathbuf;
	}

}

void GetLibPath()
{
	char* dlibpath = getenv(DLIBPATH);
	if (dlibpath == NULL) {
		dlibpath = ".";
	}
	GetLibPath(dlibpath);
}

int sysmodule(Context* cx, Node* goalscar, Node* goalscdr, 
				Node* goals, List* module, int& r)
{
	Node* retn;
	int	rn;

	std::string	s;

	if (goalscar->Val()->Car()->kind() == ATOM) {
		((Atom*)(goalscar->Val()->Car()))->toString(s);

		if (s == "args") {
			r = CmdArgs(cx, goalscar, module);
			return 1;
		} else if (s == "DLIBPATH") {
			r = DlibPath(cx, goalscar, module);
			return 1;
#if 0
		} else if (s == "cutall") {
			r = CutAll(cx, goalscar, module);
			return 1;
#endif
		} else if (s == "mkpred") {
			r = DoMkPred(cx, goalscar, module);
			return 1;
		} else if (s == "writenl") {
			WriteNl(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "writeln") {
			WriteNl(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "print") {
			WriteNl(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "write") {
			Write(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "isNil") {
			r = isNil(cx, goalscar, module);
			return 1;
		} else if (s == "isAtom") {
			r = isAtom(cx, goalscar, module);
			return 1;
		} else if (s == "isList") {
			r = isList(cx, goalscar, module);
			return 1;
		} else if (s == "isPred") {
			r = isPred(cx, goalscar, module);
			return 1;
		} else if (s == "isVar") {
			r = isVar(cx, goalscar, module);
			return 1;
		} else if (s == "isUndefVar") {
			r = isUndefVar(cx, goalscar, module);
			return 1;
		} else if (s == "isFloat") {
			r = isFloat(cx, goalscar, module);
			return 1;
		} else if (s == "isInteger") {
			r = isInteger(cx, goalscar, module);
			return 1;
#ifndef __MINGW32__
		} else if (s == "isInf") {
			r = isInf(cx, goalscar, module);
			return 1;
		} else if (s == "isNan") {
			r = isNan(cx, goalscar, module);
			return 1;
		} else if (s == "finte") {
			r = Finite(cx, goalscar, module);
			return 1;
#endif
		} else if (s == "isTrue") {
			r = isTrue(cx, goalscar, module);
			return 1;
		} else if (s == "isFalse") {
			r = isFalse(cx, goalscar, module);
			return 1;
		} else if (s == "isUnknown") {
			r = isUnknown(cx, goalscar, module);
			return 1;
		} else if (s == "max") {
			r = Max(cx, goalscar, module);
			return 1;
		} else if (s == "min") {
			r = Min(cx, goalscar, module);
			return 1;
		} else if (s == "maxf") {
			r = Maxf(cx, goalscar, module);
			return 1;
		} else if (s == "minf") {
			r = Minf(cx, goalscar, module);
			return 1;
		} else if (s == "sum") {
			r = Sum(cx, goalscar, module);
			return 1;
		} else if (s == "sumf") {
			r = Sumf(cx, goalscar, module);
			return 1;
		} else if (s == "avg") {
			r = Avg(cx, goalscar, module);
			return 1;
		} else if (s == "avgf") {
			r = Avgf(cx, goalscar, module);
			return 1;
		} else if (s == "regex") {
			r = DoRegex(cx, goalscar, module);
			return 1;
		} else if (s == "sub") {
			r = DoSub(cx, goalscar, module);
			return 1;
		} else if (s == "gsub") {
			r = DoGSub(cx, goalscar, module);
			return 1;
		} else if (s == "split") {
			r = Split(cx, goalscar, module);
			return 1;
		} else if (s == "toupper") {
			r = Toupper(cx, goalscar, module);
			return 1;
		} else if (s == "tolower") {
			r = Tolower(cx, goalscar, module);
			return 1;
		} else if (s == "length") {
			r = Length(cx, goalscar, module);
			return 1;
		} else if (s == "random") {
			r = Random(cx, goalscar, module);
			return 1;

		} else if (s == "real") {
			r = CReal(cx, goalscar, module);
			return 1;
		} else if (s == "image") {
			r = CImage(cx, goalscar, module);
			return 1;
		} else if (s == "arg") {
			r = CArg(cx, goalscar, module);
			return 1;
		} else if (s == "norm") {
			r = CNorm(cx, goalscar, module);
			return 1;
		} else if (s == "conj") {
			r = CConj(cx, goalscar, module);
			return 1;
		} else if (s == "polar") {
			r = CPolar(cx, goalscar, module);
			return 1;

#if 0
		} else if (s == "sin") {
			r = Sin(cx, goalscar, module);
			return 1;
		} else if (s == "cos") {
			r = Cos(cx, goalscar, module);
			return 1;
		} else if (s == "tan") {
			r = Tan(cx, goalscar, module);
			return 1;
		} else if (s == "asin") {
			r = ASin(cx, goalscar, module);
			return 1;
		} else if (s == "acos") {
			r = ACos(cx, goalscar, module);
			return 1;
		} else if (s == "atan") {
			r = ATan(cx, goalscar, module);
			return 1;
		} else if (s == "atan2") {
			r = ATan2(cx, goalscar, module);
			return 1;
		} else if (s == "sinh") {
			r = Sinh(cx, goalscar, module);
			return 1;
		} else if (s == "cosh") {
			r = Cosh(cx, goalscar, module);
			return 1;
		} else if (s == "tanh") {
			r = Tanh(cx, goalscar, module);
			return 1;
		} else if (s == "asinh") {
			r = ASinh(cx, goalscar, module);
			return 1;
		} else if (s == "acosh") {
			r = ACosh(cx, goalscar, module);
			return 1;
		} else if (s == "atanh") {
			r = ATanh(cx, goalscar, module);
			return 1;
		} else if (s == "log") {
			r = Log(cx, goalscar, module);
			return 1;
		} else if (s == "log10") {
			r = Log10(cx, goalscar, module);
			return 1;
		} else if (s == "exp") {
			r = Exp(cx, goalscar, module);
			return 1;
		} else if (s == "exp2") {
			r = Exp2(cx, goalscar, module);
			return 1;
		} else if (s == "exp10") {
			r = Exp10(cx, goalscar, module);
			return 1;
		} else if (s == "pow") {
			r = Pow(cx, goalscar, module);
			return 1;
		} else if (s == "sqrt") {
			r = Sqrt(cx, goalscar, module);
			return 1;
		} else if (s == "abs") {
			r = Abs(cx, goalscar, module);
			return 1;
#else
		} else if (s == "sin") {
			r = CSin(cx, goalscar, module);
			return 1;
		} else if (s == "cos") {
			r = CCos(cx, goalscar, module);
			return 1;
		} else if (s == "tan") {
			r = CTan(cx, goalscar, module);
			return 1;
		} else if (s == "asin") {
			r = CASin(cx, goalscar, module);
			return 1;
		} else if (s == "acos") {
			r = CACos(cx, goalscar, module);
			return 1;
		} else if (s == "atan") {
			r = CATan(cx, goalscar, module);
			return 1;
		} else if (s == "atan2") {
			r = ATan2(cx, goalscar, module);
			return 1;
		} else if (s == "sinh") {
			r = CSinh(cx, goalscar, module);
			return 1;
		} else if (s == "cosh") {
			r = CCosh(cx, goalscar, module);
			return 1;
		} else if (s == "tanh") {
			r = CTanh(cx, goalscar, module);
			return 1;
		} else if (s == "asinh") {
			r = CASinh(cx, goalscar, module);
			return 1;
		} else if (s == "acosh") {
			r = CACosh(cx, goalscar, module);
			return 1;
		} else if (s == "atanh") {
			r = CATanh(cx, goalscar, module);
			return 1;
		} else if (s == "log") {
			r = CLog(cx, goalscar, module);
			return 1;
		} else if (s == "log10") {
			r = CLog10(cx, goalscar, module);
			return 1;
		} else if (s == "exp") {
			r = CExp(cx, goalscar, module);
			return 1;
		} else if (s == "exp2") {
			r = CExp2(cx, goalscar, module);
			return 1;
		} else if (s == "exp10") {
			r = CExp10(cx, goalscar, module);
			return 1;
		} else if (s == "pow") {
			r = CPow(cx, goalscar, module);
			return 1;
		} else if (s == "sqrt") {
			r = CSqrt(cx, goalscar, module);
			return 1;
		} else if (s == "abs") {
			r = CAbs(cx, goalscar, module);
			return 1;
#endif

		} else if (s == "PI") {
			r = PI(cx, goalscar, module);
			return 1;
		} else if (s == "e") {
			r = E(cx, goalscar, module);
			return 1;
		} else if (s == "int") {
			r = Int(cx, goalscar, module);
			return 1;
		} else if (s == "floor") {
			r = Floor(cx, goalscar, module);
			return 1;
		} else if (s == "ceil") {
			r = Ceil(cx, goalscar, module);
			return 1;
		} else if (s == "trunc") {
			r = Trunc(cx, goalscar, module);
			return 1;
		} else if (s == "car") {
			r = Car(cx, goalscar, module);
			return 1;
		} else if (s == "cdr") {
			r = Cdr(cx, goalscar, module);
			return 1;

		} else if (s == "caar") {
			r = Caar(cx, goalscar, module);
			return 1;
		} else if (s == "cadr") {
			r = Cadr(cx, goalscar, module);
			return 1;
		} else if (s == "cdar") {
			r = Cdar(cx, goalscar, module);
			return 1;
		} else if (s == "cddr") {
			r = Cddr(cx, goalscar, module);
			return 1;

		} else if (s == "caaar") {
			r = Caaar(cx, goalscar, module);
			return 1;
		} else if (s == "caadr") {
			r = Caadr(cx, goalscar, module);
			return 1;
		} else if (s == "cadar") {
			r = Cadar(cx, goalscar, module);
			return 1;
		} else if (s == "caddr") {
			r = Caddr(cx, goalscar, module);
			return 1;
		} else if (s == "cdaar") {
			r = Cdaar(cx, goalscar, module);
			return 1;
		} else if (s == "cdadr") {
			r = Cdadr(cx, goalscar, module);
			return 1;
		} else if (s == "cddar") {
			r = Cddar(cx, goalscar, module);
			return 1;
		} else if (s == "cdddr") {
			r = Cdddr(cx, goalscar, module);
			return 1;

		} else if (s == "caaaar") {
			r = Caaaar(cx, goalscar, module);
			return 1;
		} else if (s == "caaadr") {
			r = Caaadr(cx, goalscar, module);
			return 1;
		} else if (s == "caadar") {
			r = Caadar(cx, goalscar, module);
			return 1;
		} else if (s == "caaddr") {
			r = Caaddr(cx, goalscar, module);
			return 1;
		} else if (s == "cadaar") {
			r = Cadaar(cx, goalscar, module);
			return 1;
		} else if (s == "cadadr") {
			r = Cadadr(cx, goalscar, module);
			return 1;
		} else if (s == "caddar") {
			r = Caddar(cx, goalscar, module);
			return 1;
		} else if (s == "cadddr") {
			r = Cadddr(cx, goalscar, module);
			return 1;
		} else if (s == "cdaaar") {
			r = Cdaaar(cx, goalscar, module);
			return 1;
		} else if (s == "cdaadr") {
			r = Cdaadr(cx, goalscar, module);
			return 1;
		} else if (s == "cdadar") {
			r = Cdadar(cx, goalscar, module);
			return 1;
		} else if (s == "cdaddr") {
			r = Cdaddr(cx, goalscar, module);
			return 1;
		} else if (s == "cddaar") {
			r = Cddaar(cx, goalscar, module);
			return 1;
		} else if (s == "cddadr") {
			r = Cddadr(cx, goalscar, module);
			return 1;
		} else if (s == "cdddar") {
			r = Cdddar(cx, goalscar, module);
			return 1;
		} else if (s == "cddddr") {
			r = Cddddr(cx, goalscar, module);
			return 1;

		} else if (s == "cons") {
			r = Cons(cx, goalscar, module);
			return 1;
		} else if (s == "nth") {
			r = Nth(cx, goalscar, module);
			return 1;
		} else if (s == "code") {
			r = SetCode(cx, goalscar, module);
			return 1;
		} else if (s == "char") {
			r = CodeCharPrd(cx, goalscar, module);
			return 1;
		} else if (s == "byte") {
			r = Char(cx, goalscar, module);
			return 1;
		} else if (s == "asciichar") {
			r = Char(cx, goalscar, module);
			return 1;
		} else if (s == "utf8char") {
			r = UTF8CharPrd(cx, goalscar, module);
			return 1;
		} else if (s == "eucchar") {
			r = EUCCharPrd(cx, goalscar, module);
			return 1;
		} else if (s == "sjischar") {
			r = SJISCharPrd(cx, goalscar, module);
			return 1;
		} else if (s == "concat") {
			r = Concat(cx, goalscar, module);
			return 1;
		} else if (s == "concatcode") {
			r = ConcatCode(cx, goalscar, module);
			return 1;
		} else if (s == "leftstr") {
			r = LeftStr(cx, goalscar, module);
			return 1;
		} else if (s == "rightstr") {
			r = RightStr(cx, goalscar, module);
			return 1;
		} else if (s == "substr") {
			r = SubStr(cx, goalscar, module);
			return 1;
		} else if (s == "insertstr") {
			r = InsertStr(cx, goalscar, module);
			return 1;
		} else if (s == "bitand") {
			r = And(cx, goalscar, module);
			return 1;
		} else if (s == "bitor") {
			r = Or(cx, goalscar, module);
			return 1;
		} else if (s == "bitxor") {
			r = Xor(cx, goalscar, module);
			return 1;
		} else if (s == "bitnot") {
			r = BitNot(cx, goalscar, module);
			return 1;
		} else if (s == "shiftl") {
			r = ShiftL(cx, goalscar, module);
			return 1;
		} else if (s == "shiftr") {
			r = ShiftR(cx, goalscar, module);
			return 1;
		} else if (s == "eq") {
			r = eq(cx, goalscar, module);
			return 1;
		} else if (s == "noteq") {
			r = noteq(cx, goalscar, module);
			return 1;
		} else if (s == "is") {
			r = eq(cx, goalscar, module);
			return 1;
		} else if (s == "getc") {
			r = DoGetc(cx, goalscar, module);
			return 1;
		} else if (s == "putc") {
			r = DoPutc(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "getline") {
			r = GetLine(cx, goalscar, module);
			return 1;
		} else if (s == "syntax") {
			r = SyntaxLine(cx, goalscar, module);
			return 1;
		} else if (s == "tmpfile") {
			r = TmpFile(cx, goalscar, module);
			return 1;
		} else if (s == "line") {
			r = Line(cx, goalscar, module);
			return 1;
		} else if (s == "openr") {
			r = DoOpenR(cx, goalscar, module);
			return 1;
		} else if (s == "openw") {
			r = DoOpenW(cx, goalscar, module);
			return 1;
		} else if (s == "openwp") {
			r = DoOpenWP(cx, goalscar, module);
			return 1;
		} else if (s == "gettime") {
			r = GetTime(cx, goalscar, module);
			return 1;
		} else if (s == "time") {
			r = Time(cx, goalscar, module);
			return 1;
		} else if (s == "date") {
			r = Date(cx, goalscar, module);
			return 1;
		} else if (s == "sleep") {
			r = Sleep(cx, goalscar, module);
			return 1;
		} else if (s == "usleep") {
			r = USleep(cx, goalscar, module);
			return 1;
		} else if (s == "pause") {
			r = Pause(cx, goalscar, module);
			return 1;
		} else if (s == "basename") {
			r = BaseName(cx, goalscar, module);
			return 1;
		} else if (s == "dirname") {
			r = DirName(cx, goalscar, module);
			return 1;
		} else if (s == "suffix") {
			r = Suffix(cx, goalscar, module);
			return 1;
#ifndef __MINGW32__
		} else if (s == "clear") {
			r = ClearScreen(cx, goalscar, module);
			return 1;
		} else if (s == "uname") {
			r = Uname(cx, goalscar, module);
			return 1;
#endif
		} else if (s == "countnode") {
			r = DoCountNode(cx, goalscar, module);
			return 1;
		} else if (s == "gc") {
			r = 1;
			GC();
			return 1;
		}
	}
	r = -1;
	syserr("::sys %s: The predicate that did not exist in the sys module is used. \n"
			, s.c_str());
	return 1;
}

int CmdArgs(Context* cx, Node* goalscar, List* module)
{
	extern int	pargc;
	extern char**	pargv;

	Node* g = goalscar->Cdr();
	if (ListLength(g) != 1) {
		syserr("usage : <args VAR> \n");
		return 0;
	}

	Node* nvar = g->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("args: argument of args should be a variable. ");
		return 0;
	}

	Node*	n = Nil;

	for (int i=1; i < pargc; i++) {
		n = Append(n, MkList(mka(pargv[i])));
	}

	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)(nvar->Val()))->Set(n);

	PushStack(cx, Nil, Nil, env);
	return 1;
	
}

int DlibPath(Context* cx, Node* goalscar, List* module)
{
	extern int	pargc;
	extern char**	pargv;

	Node* g = goalscar->Cdr();
	if (ListLength(g) != 1) {
		syserr("usage : <DLIBPATH VAR> \n");
		return 0;
	}

	Node* nvar = g->Car()->Val();
	if (nvar->kind() != UNDEF) {
		if (nvar->kind() == ATOM) {	// set DLIBPATH
			std::string sdpath;
			
			((Atom*)nvar)->toString(sdpath);
			GetLibPath((char*)sdpath.c_str());
#ifndef __MINGW32__
			setenv(DLIBPATH, (char*)sdpath.c_str(), 1);
#else
			std::string s;
			s = DLIBPATH;
			s = s + "=";
			s = s + sdpath;
			putenv((char*)s.c_str());
#endif
			return 1;
		}
		syserr("DLIBPATH: argument of args should be a variable or an atom. \n");
		return 0;		
	}

	GetLibPath();

	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)(nvar->Val()))->Set(Dup(dlibpathnode));

	PushStack(cx, Nil, Nil, env);
	return 1;
	
}

int CutAll(Context* cx, Node* goalscar, List* module)
{
	if (goalscar->Cdr() != Nil) {
		syserr("usage : <cutall> \n");
		return 0;
	}
	cx->CutAll();
	
	return 1;
}


int DoMkPred(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <mkpred LIST> \n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car();
	Node* n1   = g->Cdr()->Car();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <mkpred LIST> \n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, n1, goalscar, module)) <= 0) {
		syserr("mkpred: failed in the evaluation of the argument. \n");
		return 0;
	}

	Node* prd = MkPred(n1);
		
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(prd);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}


int Write(Context* cx, Node* goalscar, List* module)
{
	Node*	n = goalscar->Cdr()->Val();
	int rn;

	if ((rn = FuncArg(cx, n, goalscar, module)) <= 0) {
		syserr("write: failed in the evaluation of the argument. \n");
		return 0;
	}

	n->Car()->print(cx->ioout);
	n->Cdr()->printcdr(cx->ioout);

	return 1;
}

int WriteNl(Context* cx, Node* goalscar, List* module)
{
	int rn;
	
	if ((rn=Write(cx, goalscar, module)) <= 0) {
		syserr("writenl: failed in the evaluation of the argument. \n");
		return 0;
	}
	
	fprintf(cx->ioout, "\n");
	return 1;
}

int isNil(Context* cx, Node* n, List* module)
{
	Node* goalscar = n;
	
	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

	int rn;

	if ((rn = FuncArg(cx, n, goalscar, module)) <= 0) {
		syserr("isNil: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n->Car() == Nil) {
		return 1;
	} else {
		return -1;
	}
}

int isAtom(Context* cx, Node* n, List* module)
{
	Node* goalscar = n;

	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

	int rn;

	if ((rn = FuncArg(cx, n, goalscar, module)) <= 0) {
		syserr("isAtom: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n->Car()->kind() == ATOM) {
		return 1;
	} else {
		return -1;
	}
}

int isList(Context* cx, Node* n, List* module)
{
	Node* goalscar = n;

	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

	int rn;

	if ((rn = FuncArg(cx, n, goalscar, module)) <= 0) {
		syserr("isList: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n->Car()->kind() == LIST) {
		return 1;
	} else {
		return -1;
	}
}

int isPred(Context* cx, Node* n, List* module)
{
	Node* goalscar = n;

	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

	if (n->Car()->kind() == PRED) {
		return 1;
	} else {
		return -1;
	}
}

int isVar(Context* cx, Node* n, List* module)
{
	Node* goalscar = n;

	n = n->Cdr();
	if (ListLength(n) != 1) {
		return -1;
	}

	int rn;

	if ((rn = FuncArg(cx, n, goalscar, module)) <= 0) {
		syserr("isVar: failed in the evaluation of the argument. \n");
		return 0;
	}

	if ((n->Car()->kind() == VAR) || (n->Car()->kind() == UNDEF)) {
		return 1;
	} else {
		return -1;
	}

}

int isUndefVar(Context* cx, Node* n, List* module)
{
	Node* goalscar = n;

	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

	int rn;

	if ((rn = FuncArg(cx, n, goalscar, module)) <= 0) {
		syserr("isUndefVar: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n->Car()->kind() == UNDEF) {
		return 1;
	} else {
		return -1;
	}
}

int isFloat(Context* cx, Node* n, List* module)
{
	Node* goalscar = n;
	long double	d;
	
	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

	int rn;

	if ((rn = FuncArg(cx, n, goalscar, module)) <= 0) {
		syserr("isFloat: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n->Car()->kind() != ATOM) {
		return -1;
	}
	if (((Atom*)(n->Car()))->toFloat(d)) {
		return 1;
	} else {
		return -1;
	}

}

int isInteger(Context* cx, Node* n, List* module)
{
	Node* goalscar = n;
	long long i;
	
	n = n->Cdr()->Val();

	if (ListLength(n) != 1) {
		return -1;
	}

	int rn;

	if ((rn = FuncArg(cx, n, goalscar, module)) <= 0) {
		syserr("isInteger: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n->Car()->kind() != ATOM) {
		return -1;
	}
	if (((Atom*)(n->Car()))->toInt(i)) {
		return 1;
	} else {
		return -1;
	}
}


int isTrue(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		return -1;
	}

	Node* gl = goalscar->Cdr()->Car()->Val();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	int r;
	if ((r=Unify(cx2, gl, module))) {
		cx->Merge(cx2);
	}

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	return r;
}

int isFalse(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		return -1;
	}

	Node* gl = goalscar->Cdr()->Car()->Val();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	int r;
	if ((r=Unify(cx2, gl, module))) {
		cx->Merge(cx2);
	}

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	if (r > 0) {
		return -1;
	} else if (r == 0) {
		return 1;
	}
	return r;
}

int isUnknown(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		return -1;
	}

	Node* gl = goalscar->Cdr()->Car()->Val();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	int r;
	if ((r=Unify(cx2, gl, module))) {
		cx->Merge(cx2);
	}

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	if (r > 0) {
		return -1;
	} else if (r == 0) {
		return -1;
	}
	return 1;
}


int Max(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <max VAR LIST> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <max VAR LIST> \n");
		return 0;
	}

	Node* nlist = goalscar->Cdr()->Val();
	if (nlist->kind() != LIST) {
		syserr("usage : <max VAR LIST> \n");
		return 0;
	}

	long long max=0, num;
	Node* n;
	n = nlist;

	if ((n->Car()->kind() != ATOM) || !((Atom*)(n->Car()))->toInt(max)) {
		syserr("max: argument have to be number. \n");
		return 0;
	}

	for ( ; n->kind() != ATOM; n = n->Cdr()) {
		int rn;
		Node* np = n->Car();

		if ((rn = FuncArg(cx, np, goalscar, module)) <= 0) {
			syserr("max: failed in the evaluation of the argument. \n");
			return 0;
		}

		np = np->Val();

		if ((np->Car()->kind() != ATOM) 
				|| !((Atom*)np->Car())->toInt(num)) {
			syserr("max: argument have to be number. \n");
			return 0;
		}
		if (num > max) {
			max = num;
		}
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(max));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int Min(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <min VAR LIST> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <min VAR LIST> \n");
		return 0;
	}

	Node* nlist = goalscar->Cdr()->Val();
	if (nlist->kind() != LIST) {
		syserr("usage : <min VAR LIST> \n");
		return 0;
	}

	long long min=0, num;
	Node* n;
	n = nlist;

	if ((n->Car()->kind() != ATOM) || !((Atom*)(n->Car()))->toInt(min)) {
		syserr("min: argument have to be number. \n");
		return 0;
	}

	for ( ; n->kind() != ATOM; n = n->Cdr()) {
		int rn;
		Node* np = n->Car();
		
		if ((rn = FuncArg(cx, np, goalscar, module)) <= 0) {
			syserr("min: failed in the evaluation of the argument. \n");
			return 0;
		}

		np = np->Val();
		if ((np->Car()->kind() != ATOM) 
				|| !((Atom*)np->Car())->toInt(num)) {
			syserr("min: argument have to be number. \n");
			return 0;
		}
		if (num < min) {
			min = num;
		}
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(min));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int Maxf(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <maxf VAR LIST> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <maxf VAR LIST> \n");
		return 0;
	}

	Node* nlist = goalscar->Cdr()->Val();
	if (nlist->kind() != LIST) {
		syserr("usage : <maxf VAR LIST> \n");
		return 0;
	}

	long double max=0, num;
	Node* n;
	n = nlist;

	if ((n->Car()->kind() != ATOM) || !((Atom*)(n->Car()))->toFloat(max)) {
		syserr("maxf: argument have to be number. \n");
		return 0;
	}

	for ( ; n->kind() != ATOM; n = n->Cdr()) {
		int rn;
		Node* np = n->Car();
		
		if ((rn = FuncArg(cx, np, goalscar, module)) <= 0) {
			syserr("maxf: failed in the evaluation of the argument. \n");
			return 0;
		}

		np = np->Val();
		if ((np->Car()->kind() != ATOM) 
				|| !((Atom*)np->Car())->toFloat(num)) {
			syserr("maxf: argument have to be number. \n");
			return 0;
		}
		if (num > max) {
			max = num;
		}
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(max));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int Minf(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <minf VAR LIST> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <minf VAR LIST> \n");
		return 0;
	}

	Node* nlist = goalscar->Cdr()->Val();
	if (nlist->kind() != LIST) {
		syserr("usage : <minf VAR LIST> \n");
		return 0;
	}

	long double min=0, num;
	Node* n;
	n = nlist;

	if ((n->Car()->kind() != ATOM) || !((Atom*)(n->Car()))->toFloat(min)) {
		syserr("minf: argument have to be number. \n");
		return 0;
	}

	for ( ; n->kind() != ATOM; n = n->Cdr()) {
		int rn;
		Node* np = n->Car();
		
		if ((rn = FuncArg(cx, np, goalscar, module)) <= 0) {
			syserr("minf: failed in the evaluation of the argument. \n");
			return 0;
		}

		np = np->Val();
		if ((np->Car()->kind() != ATOM) 
				|| !((Atom*)np->Car())->toFloat(num)) {
			syserr("minf: argument have to be number. \n");
			return 0;
		}
		if (num < min) {
			min = num;
		}
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(min));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int Sum(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <sum VAR LIST ...> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <sum VAR LIST ...> \n");
		return 0;
	}

	Node* nlist = goalscar->Cdr()->Val();
	if (nlist->kind() != LIST) {
		syserr("usage : <sum VAR LIST ...> \n");
		return 0;
	}

	long long sum = 0, num;
	Node* n;
	n = nlist;

	for (n = nlist; n->kind() != ATOM; n = n->Cdr()) {
		int rn;
		Node* np = n->Car();
		
		if ((rn = FuncArg(cx, np, goalscar, module)) <= 0) {
			syserr("sum: failed in the evaluation of the argument. \n");
			return 0;
		}

		np = np->Val();
		if ((np->kind() != ATOM) 
				|| !((Atom*)np)->toInt(num)) {
			syserr("sum: argument have to be number. \n");
			return 0;
		}
		sum += num;
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(sum));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int Sumf(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <sumf VAR LIST ...> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <sumf VAR LIST ...> \n");
		return 0;
	}

	Node* nlist = goalscar->Cdr()->Val();
	if (nlist->kind() != LIST) {
		syserr("usage : <sumf VAR LIST ...> \n");
		return 0;
	}

	long double sum = 0, num;
	Node* n;
	n = nlist;

	for (n = nlist; n->kind() != ATOM; n = n->Cdr()) {
		int rn;
		Node* np = n->Car();
		
		if ((rn = FuncArg(cx, np, goalscar, module)) <= 0) {
			syserr("sumf: failed in the evaluation of the argument. \n");
			return 0;
		}

		np = np->Val();
		if ((np->kind() != ATOM) 
				|| !((Atom*)np)->toFloat(num)) {
			syserr("sumf: argument have to be number. \n");
			return 0;
		}
		sum += num;
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(sum));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int Avg(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <avg VAR LIST ...> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <avg VAR LIST ...> \n");
		return 0;
	}

	Node* nlist = goalscar->Cdr()->Val();
	if (nlist->kind() != LIST) {
		syserr("usage : <avg VAR LIST ...> \n");
		return 0;
	}

	int count = 0;
	long long sum = 0, num;
	Node* n;
	n = nlist;

	for (n = nlist; n->kind() != ATOM; n = n->Cdr()) {
		int rn;
		Node* np = n->Car();
		
		if ((rn = FuncArg(cx, np, goalscar, module)) <= 0) {
			syserr("avg: failed in the evaluation of the argument. \n");
			return 0;
		}

		np = np->Val();
		if ((np->kind() != ATOM) 
				|| !((Atom*)np)->toInt(num)) {
			syserr("avg: argument have to be number. \n");
			return 0;
		}
		sum += num;
		count++;
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(sum/count));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int Avgf(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <avgf VAR LIST ...> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <avgf VAR LIST ...> \n");
		return 0;
	}

	Node* nlist = goalscar->Cdr()->Val();
	if (nlist->kind() != LIST) {
		syserr("usage : <avgf VAR LIST ...> \n");
		return 0;
	}

	int count = 0;
	long double sum = 0, num;
	Node* n;
	n = nlist;

	for (n = nlist; n->kind() != ATOM; n = n->Cdr()) {
		int rn;
		Node* np = n->Car();
		
		if ((rn = FuncArg(cx, np, goalscar, module)) <= 0) {
			syserr("avgf: failed in the evaluation of the argument. \n");
			return 0;
		}

		np = np->Val();
		if ((np->kind() != ATOM) 
				|| !((Atom*)np)->toFloat(num)) {
			syserr("avgf: argument have to be number. \n");
			return 0;
		}
		sum += num;
		count++;
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(sum/count));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}


static int CheckDelimiterOne(char* str, int i, char* delm, int j)
{
	int k, ns, nd;

	ns = CharLen(str[i]);
	nd = CharLen(delm[j]);
	if (i+ns > strlen(str)) {
		return 0;
	}
	if (j+nd > strlen(delm)) {
		return 0;
	}
	
	if (ns == nd) {
		for (k = 0; k < nd; k++) {
			if (str[i+k] != delm[j+k]) {
				return 0;
			}
		}
		return 1;
	} else {
		return 0;
	}				
}

static int CheckDelimiter(char* str, int i, char* delm)
{
	int j, nd;

	for (j = 0; j < strlen(delm); j += nd) {
		nd = CharLen(delm[j]);
		if (j+nd > strlen(delm)) {
			return 0;
		}
		if (CheckDelimiterOne(str, i, delm, j)) {
			return 1;
		}
	}
	return 0;
}


int Split(Context* cx, Node* goalscar, List* module)
{
	std::string Delimiters = " \t";

	Delimiters = Delimiters + CharSpace();

	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 3)) {
		syserr("usage : <split VAR STRINGS> or <split VAR STRINGS DELIMITERS>\n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <split VAR STRINGS> or <split VAR STRINGS DELIMITERS>\n");
		return 0;
	}

	Node* nstr = goalscar->Cdr()->Car()->Val();
	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("split: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <split VAR STRINGS> or <split VAR STRINGS DELIMITERS>\n");
		return 0;
	}
	std::string str;
	((Atom*)nstr)->toString(str);

	if (goalscar->Cdr()->Cdr() != Nil) {
		Node* ndelim = goalscar->Cdr()->Cdr()->Car()->Val();
		if (ndelim->kind() != ATOM) {
			syserr("usage : <split VAR STRINGS> or <split VAR STRINGS DELIMITERS>\n");
			return 0;
		}
		((Atom*)ndelim)->toString(Delimiters);
	}

	// skip space
	std::string tmpstr = "";
	int i, j, ns;
	Node* arg = Nil;
	for (i=0; i < str.length(); i += ns) {
		ns = CharLen(str[i]);
		if (i+ns > str.length()) {
			break;
		}
		if (!CheckDelimiter((char*)str.c_str(), i, 
				(char*)Delimiters.c_str())) {
			break;
		}
	}

	// appended terms
	for (; i < str.length(); i += ns) {
		ns = CharLen(str[i]);
		if (i+ns > str.length()) {
			break;
		}
		if (CheckDelimiter((char*)str.c_str(), i, 
					(char*)Delimiters.c_str())) {
			if (tmpstr != "") {
				arg = Append(arg, MkList(mka((char*)tmpstr.c_str())));
				tmpstr = "";
			}
		} else {
			for (j = 0; j < ns; j++) {
				tmpstr = tmpstr + str[i+j];
			}
		}
	}
	if (tmpstr != "") {
		arg = Append(arg, MkList(mka((char*)tmpstr.c_str())));
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(arg);

	PushStack(cx, Nil, Nil, env);
	
	return 1;
				
}

int Toupper(Context* cx, Node* goalscar, List* module)
{
	char* output;
	
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll != 2) {
		syserr("usage : <toupper VAR STRINGS> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <toupper VAR STRINGS> \n");
		return 0;
	}

	Node* nstr = goalscar->Cdr()->Car()->Val();
	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("toupper: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <toupper VAR STRINGS> \n");
		return 0;
	}
	std::string str;
	((Atom*)nstr)->toString(str);

	output = (char*)malloc(str.length()+1);

	CodeToupper((char*)str.c_str(), output);

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(output));

	free(output);

	PushStack(cx, Nil, Nil, env);
	
	return 1;
				
}


int Tolower(Context* cx, Node* goalscar, List* module)
{
	char* output;
	
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll != 2) {
		syserr("usage : <tolower VAR STRINGS> \n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <tolower VAR STRINGS> \n");
		return 0;
	}

	Node* nstr = goalscar->Cdr()->Car()->Val();
	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("tolower: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <tolower VAR STRINGS> \n");
		return 0;
	}
	std::string str;
	((Atom*)nstr)->toString(str);

	output = (char*)malloc(str.length()+1);

	CodeTolower((char*)str.c_str(), output);

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(output));

	free(output);

	PushStack(cx, Nil, Nil, env);
	
	return 1;
				
}

int Length(Context* cx, Node* goalscar, List* module)
{
	long long n;
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll != 2) {
		syserr("usage : <length VAR STRINGS>\n");
		return 0;
	}
	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <length VAR STRINGS>\n");
		return 0;
	}
	
	Node* nstr = goalscar->Cdr()->Car()->Val();

	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("length: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() == LIST) {
		n = ListLength(nstr);
	} else {
		n = 1;
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(n));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
				
}


int Random(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <random VAR>\n");
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();

	if (v->kind() == UNDEF) {
		long long rd;
		Node* env = Nil->Cons(Nil);
#ifndef __MINGW32__
		rd = (long long)random();
#else
		rd = (long long)rand();
#endif /* __MINGW32__ */

		SetEnv(env, v);
		((Undef*)v)->Set(mka(rd));
		PushStack(cx, Nil, Nil, env);
		
		return 1;
	} else {
		syserr("usage : <random VAR>\n");
		return 0;
	}
			
	syserr("usage : <random VAR>\n");
	return 0;
}


int Char(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <char VAR STRING>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <char VAR STRING>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("char: failed in the evaluation of the argument. \n");
		return 0;
	}

	
	if (nstr->kind() != ATOM) {
		syserr("usage : <char VAR STRING>\n");
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	int i;
	for (i = 0; i < s.length(); i++) {
		n = Append(n, MkList(mka((long long)(unsigned char)(s[i]))));
	}

	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int Concat(Context* cx, Node* goalscar, List* module)
{
	Node* env;
	
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <concat VAR STRING-LIST>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nlist = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <concat VAR STRING-LIST>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, nlist, goalscar, module)) <= 0) {
		syserr("concat: failed in the evaluation of the argument. \n");
		return 0;
	}


	if (nlist->kind() == ATOM) {
		env = Nil->Cons(Nil);

		SetEnv(env, nvar);
		((Undef*)nvar)->Set(nlist);
		PushStack(cx, Nil, Nil, env);
		
		return 1;
	}
		
	if (nlist->kind() != LIST) {
		syserr("usage : <concat VAR STRING-LIST>\n");
		return 0;
	}


	std::string s="";
	std::string sl="";
	Node*	n;
	for (n = nlist; n->kind() != ATOM; n=n->Cdr()) {
		n=n->Val();
		if (n->Car()->kind() == ATOM) {
			if (!((Atom*)(n->Car()))->toString(sl)) {
				syserr("usage : <concat VAR STRING-LIST>\n");
				return 0;
			}
			s = s+sl;
		} else if (n->Car()->kind() == LIST) {
			Node* n2;
			for (n2 = n->Car(); n2->kind() != ATOM; n2=n2->Cdr()) {
				if (n2->Car()->kind() != ATOM) {
					return 0;
				}
				if (!((Atom*)(n2->Car()))->toString(sl)) {
					syserr("usage : <concat VAR STRING-LIST>\n");
					return 0;
				}
				s = s+sl;
			}			
		} else {
			syserr("usage : <concat VAR STRING-LIST>\n");
			return 0;
		}
	}

	
	env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)s.c_str()));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int ConcatCode(Context* cx, Node* goalscar, List* module)
{
	Node* env;
	
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <concatcode VAR CODE-LIST>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nlist = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <concatcode VAR CODE-LIST>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, nlist, goalscar, module)) <= 0) {
		syserr("concatcode: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nlist->kind() != LIST) {
		syserr("usage : <concatcode VAR CODE-LIST>\n");
		return 0;
	}


	std::string s="";
	Node*	n;
	long long nn;
	for (n = nlist; n->kind() != ATOM; n=n->Cdr()) {
		n=n->Val();
		if (n->Car()->kind() == ATOM) {
			if (!((Atom*)(n->Car()))->toInt(nn)) {
				syserr("usage : <concatcode VAR CODE-LIST>\n");
				return 0;
			}
			s = s+char(nn);
		} else if (n->Car()->kind() == LIST) {
			Node* n2;
			for (n2 = n->Car(); n2->kind() != ATOM; n2=n2->Cdr()) {
				if (n2->Car()->kind() != ATOM) {
					syserr("usage : <concatcode VAR CODE-LIST>\n");
					return 0;
				}
				if (!((Atom*)(n2->Car()))->toInt(nn)) {
					syserr("usage : <concatcode VAR CODE-LIST>\n");
					return 0;
				}
				s = s+char(nn);
			}			
		} else {
			syserr("usage : <concatcode VAR CODE-LIST>\n");
			return 0;
		}
	}

	
	env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)s.c_str()));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int LeftStr(Context* cx, Node* goalscar, List* module)
{
	Node* env;
	
	if (ListLength(goalscar->Cdr()) != 3) {
		syserr("usage : <leftstr VAR STRING NUM>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();
	Node* nn   = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <leftstr VAR STRING NUM>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("leftstr: failed in the evaluation of the argument. \n");
		return 0;
	}


	if (nstr->kind() != ATOM) {
		syserr("usage : <leftstr VAR STRING NUM>\n");
		return 0;
	}
	std::string s;
	((Atom*)nstr)->toString(s);
	
	if ((rn = FuncArg(cx, nn, goalscar, module)) <= 0) {
		syserr("leftstr: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nn->kind() != ATOM) {
		syserr("usage : <leftstr VAR STRING NUM>\n");
		return 0;
	}
	long long n;
	if (!((Atom*)nn)->toInt(n)) {
		syserr("usage : <leftstr VAR STRING NUM>\n");
		return 0;
	}

	if (n < 0) {
		n = 0;
	}
	if (n > s.length()) {
		n = s.length();
	}
	
	s = s.substr(0, n);
	
	env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)s.c_str()));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int RightStr(Context* cx, Node* goalscar, List* module)
{
	Node* env;
	
	if (ListLength(goalscar->Cdr()) != 3) {
		syserr("usage : <rightstr VAR STRING NUM>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();
	Node* nn   = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <rightstr VAR STRING NUM>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("rightstr: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <rightstr VAR STRING NUM>\n");
		return 0;
	}
	std::string s;
	((Atom*)nstr)->toString(s);
	
	if ((rn = FuncArg(cx, nn, goalscar, module)) <= 0) {
		syserr("rightstr: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nn->kind() != ATOM) {
		syserr("usage : <rightstr VAR STRING NUM>\n");
		return 0;
	}
	long long n;
	if (!((Atom*)nn)->toInt(n)) {
		syserr("usage : <rightstr VAR STRING NUM>\n");
		return 0;
	}

	if (n < 0) {
		n = 0;
	}
	if (n > s.length()) {
		n = s.length();
	}
	long long p = s.length() - n;
	
	s = s.substr(p, n);
	
	env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)s.c_str()));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int SubStr(Context* cx, Node* goalscar, List* module)
{
	Node* env;
	
	if (ListLength(goalscar->Cdr()) != 4) {
		syserr("usage : <substr VAR STRING NUM LEN>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();
	Node* nn1  = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();
	Node* nn2  = goalscar->Cdr()->Cdr()->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <substr VAR STRING NUM LEN>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("substr: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <substr VAR STRING NUM LEN>\n");
		return 0;
	}
	std::string s;
	((Atom*)nstr)->toString(s);
	
	if ((rn = FuncArg(cx, nn1, goalscar, module)) <= 0) {
		syserr("substr: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nn1->kind() != ATOM) {
		syserr("usage : <substr VAR STRING NUM LEN>\n");
		return 0;
	}
	long long n1;
	if (!((Atom*)nn1)->toInt(n1)) {
		syserr("usage : <substr VAR STRING NUM LEN>\n");
		return 0;
	}
	n1 = n1 -1;
	if (n1 < 0) {
		n1 = 0;
	}

	if ((rn = FuncArg(cx, nn2, goalscar, module)) <= 0) {
		syserr("substr: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nn2->kind() != ATOM) {
		syserr("usage : <substr VAR STRING NUM LEN>\n");
		return 0;
	}
	long long n2;
	if (!((Atom*)nn2)->toInt(n2)) {
		syserr("usage : <substr VAR STRING NUM LEN>\n");
		return 0;
	}
	if (n2 < 0) {
		n2 = 0;
	}
	
	if (n1 > s.length()) {
		n1 = s.length();
	}
	if (n1+n2 > s.length()) {
		n2 = s.length()-n1;
	}

	s = s.substr(n1, n2);
	
	env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)s.c_str()));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int InsertStr(Context* cx, Node* goalscar, List* module)
{
	Node* env;
	
	if (ListLength(goalscar->Cdr()) != 4) {
		syserr("usage : <insertstr VAR STRING NUM INS-STR>\n");
		return 0;
	}

	Node* nvar  = goalscar->Cdr()->Car()->Val();
	Node* nstr1 = goalscar->Cdr()->Cdr()->Car()->Val();
	Node* nn    = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();
	Node* nstr2 = goalscar->Cdr()->Cdr()->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <insertstr VAR STRING NUM INS-STR>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, nstr1, goalscar, module)) <= 0) {
		syserr("insertstr: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr1->kind() != ATOM) {
		syserr("usage : <insertstr VAR STRING NUM INS-STR>\n");
		return 0;
	}
	std::string s1;
	((Atom*)nstr1)->toString(s1);

	if ((rn = FuncArg(cx, nn, goalscar, module)) <= 0) {
		syserr("insertstr: failed in the evaluation of the argument. \n");
		return 0;
	}

	
	if (nn->kind() != ATOM) {
		syserr("usage : <insertstr VAR STRING NUM INS-STR>\n");
		return 0;
	}
	long long n;
	if (!((Atom*)nn)->toInt(n)) {
		syserr("usage : <insertstr VAR STRING NUM INS-STR>\n");
		return 0;
	}

	if (n < 0) {
		n = 0;
	}
	
	if (n > s1.length()) {
		n = s1.length();
	}
	
	if ((rn = FuncArg(cx, nstr2, goalscar, module)) <= 0) {
		syserr("insertstr: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr2->kind() != ATOM) {
		syserr("usage : <insertstr VAR STRING NUM INS-STR>\n");
		return 0;
	}
	std::string s2;
	((Atom*)nstr2)->toString(s2);
	
	s1 = s1.insert(n, s2);
	
	env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)s1.c_str()));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}




int SetCode(Context* cx, Node* goalscar, List* module)
{
	extern std::string code;
	
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <code VAR> or <code CODE>\n");
		return 0;
	}

	Node* ncode = goalscar->Cdr()->Car()->Val();
	std::string scode;
	if (ncode->kind() == UNDEF) {
		Node* env = Nil->Cons(Nil);

		SetEnv(env, ncode);
		((Undef*)ncode)->Set(mka((char*)code.c_str()));
		PushStack(cx, Nil, Nil, env);

		return 1;
	}
	int rn;

	if ((rn = FuncArg(cx, ncode, goalscar, module)) <= 0) {
		syserr("code: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (ncode->kind() != ATOM) {
		syserr("usage : <code VAR> or <code CODE>\n");
		return 0;
	}
	
	((Atom*)ncode)->toString(scode);
	if ((scode == "EUCJP") || (scode == "EUC-JP")|| (scode == "EUC")) {
		code = "EUC";
	} else if ((scode == "SJIS") || (scode == "SHIFT-JIS")) {
		code = "SJIS";
	} else if ((scode == "UTF8") || (scode == "UTF-8")) {
		code = "UTF8";
	}
	return 1;	
}


int CodeCharPrd(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <char VAR STRING>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <char VAR STRING>\n");
		return 0;
	}
	
	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("char: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <char VAR STRING>\n");
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	n = CodeChar((char*)s.c_str());
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}


int UTF8CharPrd(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <utf8char VAR STRING>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <utf8char VAR STRING>\n");
		return 0;
	}
	
	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("utf8char: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <utf8char VAR STRING>\n");
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	n = UTF8Char((char*)s.c_str());
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int EUCCharPrd(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <eucchar VAR STRING>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <eucchar VAR STRING>\n");
		return 0;
	}
	
	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("eucchar: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <eucchar VAR STRING>\n");
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	n = EUCChar((char*)s.c_str());
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int SJISCharPrd(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <sjischar VAR STRING>\n");
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <sjischar VAR STRING>\n");
		return 0;
	}
	
	int rn;

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("sjischar: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <sjischar VAR STRING>\n");
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	n = SJISChar((char*)s.c_str());
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int And(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		syserr("usage : <bitand VAR NUM1 NUM2>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* n2   = g->Cdr()->Cdr()->Car()->Val();
	long long nn1, nn2;

	if (nvar->kind() != UNDEF) {
		syserr("usage : <bitand VAR NUM1 NUM2>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, n1, goalscar, module)) <= 0) {
		syserr("bitand: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n1->kind() != ATOM) {
		syserr("usage : <bitand VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if (!((Atom*)n1)->toInt(nn1)) {
		syserr("usage : <bitand VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if ((rn = FuncArg(cx, n2, goalscar, module)) <= 0) {
		syserr("bitand: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n2->kind() != ATOM) {
		syserr("usage : <bitand VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if (!((Atom*)n2)->toInt(nn2)) {
		syserr("usage : <bitand VAR NUM1 NUM2>\n");
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	long long nn3;
	if (nn1 != 1) {
		nn3 = -1;
	} else if (nn2 != 1) {
		nn3 = -1;
	} else {
		nn3 = 1;
	}
	
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn3));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Or(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		syserr("usage : <bitor VAR NUM1 NUM2>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* n2   = g->Cdr()->Cdr()->Car()->Val();
	long long nn1, nn2;

	
	if (nvar->kind() != UNDEF) {
		syserr("usage : <bitor VAR NUM1 NUM2>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, n1, goalscar, module)) <= 0) {
		syserr("bitor: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n1->kind() != ATOM) {
		syserr("usage : <bitor VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if (!((Atom*)n1)->toInt(nn1)) {
		syserr("usage : <bitor VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if ((rn = FuncArg(cx, n2, goalscar, module)) <= 0) {
		syserr("bitor: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n2->kind() != ATOM) {
		syserr("usage : <bitor VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if (!((Atom*)n2)->toInt(nn2)) {
		syserr("usage : <bitor VAR NUM1 NUM2>\n");
		return 0;
	}
	
	long long nn3;
	if (nn1 == 1) {
		nn3 = 1;
	} else if (nn2 == 1) {
		nn3 = 1;
	} else {
		nn3 = -1;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn3));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Xor(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		syserr("usage : <bitxor VAR NUM1 NUM2>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* n2   = g->Cdr()->Cdr()->Car()->Val();
	long long nn1, nn2, nn3;

	if (nvar->kind() != UNDEF) {
		syserr("usage : <bitxor VAR NUM1 NUM2>\n");
		return 0;
	}

	int rn;
	
	if ((rn = FuncArg(cx, n1, goalscar, module)) <= 0) {
		syserr("bitxor: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n1->kind() != ATOM) {
		syserr("usage : <bitxor VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if (!((Atom*)n1)->toInt(nn1)) {
		syserr("usage : <bitxor VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if ((rn = FuncArg(cx, n2, goalscar, module)) <= 0) {
		syserr("bitxor: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n2->kind() != ATOM) {
		syserr("usage : <bitxor VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if (!((Atom*)n2)->toInt(nn2)) {
		syserr("usage : <bitxor VAR NUM1 NUM2>\n");
		return 0;
	}
	
	if ((nn1 == 1) && (nn2 == 1)){
		nn3 = -1;
	} else if (nn1 == 1) {
		nn3 = 1;
	} else if (nn2 == 1) {
		nn3 = 1;
	} else {
		nn3 = -1;
	}

	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn3));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int BitNot(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <bitnot VAR NUM>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car();
	Node* n1   = g->Cdr()->Car();
	long long nn1;

	if (nvar->kind() != UNDEF) {
		syserr("usage : <bitnot VAR NUM>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, n1, goalscar, module)) <= 0) {
		syserr("bitor: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n1->kind() != ATOM) {
		syserr("usage : <bitnot VAR NUM>\n");
		return 0;
	}
	
	if (!((Atom*)n1)->toInt(nn1)) {
		syserr("usage : <bitnot VAR NUM>\n");
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(~nn1));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int ShiftL(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		syserr("usage : <shiftl VAR NUM SHIFT>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* nsft = g->Cdr()->Cdr()->Car()->Val();
	long long nn1, nnsft;

	if (nvar->kind() != UNDEF) {
		syserr("usage : <shiftl VAR NUM SHIFT>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, n1, goalscar, module)) <= 0) {
		syserr("bitor: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n1->kind() != ATOM) {
		syserr("usage : <shiftl VAR NUM SHIFT>\n");
		return 0;
	}
	
	if (!((Atom*)n1)->toInt(nn1)) {
		syserr("usage : <shiftl VAR NUM SHIFT>\n");
		return 0;
	}
	
	if ((rn = FuncArg(cx, nsft, goalscar, module)) <= 0) {
		syserr("bitor: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nsft->kind() != ATOM) {
		syserr("usage : <shiftl VAR NUM SHIFT>\n");
		return 0;
	}
	
	if (!((Atom*)nsft)->toInt(nnsft)) {
		syserr("usage : <shiftl VAR NUM SHIFT>\n");
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn1<<nnsft));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int ShiftR(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		syserr("usage : <shiftr VAR NUM SHIFT>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* nsft   = g->Cdr()->Cdr()->Car()->Val();
	long long nn1, nnsft;

	if (nvar->kind() != UNDEF) {
		syserr("usage : <shiftr VAR NUM SHIFT>\n");
		return 0;
	}

	int rn;

	if ((rn = FuncArg(cx, n1, goalscar, module)) <= 0) {
		syserr("bitor: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (n1->kind() != ATOM) {
		syserr("usage : <shiftr VAR NUM SHIFT>\n");
		return 0;
	}
	
	if (!((Atom*)n1)->toInt(nn1)) {
		syserr("usage : <shiftr VAR NUM SHIFT>\n");
		return 0;
	}
	
	if ((rn = FuncArg(cx, nsft, goalscar, module)) <= 0) {
		syserr("bitor: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nsft->kind() != ATOM) {
		syserr("usage : <shiftr VAR NUM SHIFT>\n");
		return 0;
	}
	
	if (!((Atom*)nsft)->toInt(nnsft)) {
		syserr("usage : <shiftr VAR NUM SHIFT>\n");
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn1>>nnsft));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}


int eq(Context* cx, Node* goalscar, List* module)
{
	Node*	env = Nil->Cons(Nil);
	
	Node* n = goalscar->Cdr()->Val();
	if (ListLength(n) != 2) {
		syserr("usage : <eq LIST1 LIST2>\n");
		return 0;
	}
	Node* n1 = n->Car();
	Node* n2 = n->Cdr()->Car();

	if (Unification(n1, n2, env, cx)) {
		PushStack(cx, Nil, Nil, env);
		return 1;
	}
	return -1;
}

int noteq(Context* cx, Node* goalscar, List* module)
{
	Node*	env = Nil->Cons(Nil);
	
	Node* n = goalscar->Cdr()->Val();
	if (ListLength(n) != 2) {
		syserr("usage : <noteq LIST1 LIST2>\n");
		return 0;
	}
	Node* n1 = n->Car();
	Node* n2 = n->Cdr()->Car();

	if (Unification(n1, n2, env, cx)) {
		PushStack(cx, Nil, Nil, env);
		return -1;
	}
	return 1;
}

int DoGetc(Context* cx, Node* goalscar, List* module)
{

	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <getc VAR> \n");
		return 0;
	}

	if (goalscar->Cdr()->Cdr() != Nil) {
		syserr("usage : <getc VAR> \n");
		return 0;
	}
	
	if (cx->ioin == stdin) {
		syserr("File is not opened");
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();

	if (v->kind() == UNDEF) {
		Node* env = Nil->Cons(Nil);

		int c = fgetc(cx->ioin);

		if (c == EOF) {
			return -1;
		}

		char ca[2];
		ca[0] = c;
		ca[1] = 0;		
		SetEnv(env, v);
		((Undef*)v)->Set(mka(ca));
		PushStack(cx, Nil, Nil, env);
		
		return 1;
	} else if (v->kind() == ATOM) {
		int c = fgetc(cx->ioin);

		if (c == EOF) {
			return -1;
		}

		std::string s;
		((Atom*)v)->toString(s);
		
		return ((s.c_str())[0] == c);
	} else {
		syserr("usage : <getc VAR> \n");
		return 0;
	}
			
	syserr("usage : <getc VAR> \n");
	return 0;
}

int DoPutc(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <putc STRINGS>\n");
		return 0;
	}

	Node* g = goalscar->Val();
	if (g->Cdr()->kind() == LIST) {
		if (g->Cdr()->Car()->kind() == ATOM) {
			std::string s;
			((Atom*)g->Cdr()->Car())->toString(s);
			putc(s.c_str()[0], cx->ioout);
			return 1;
		}
	}
	syserr("usage : <putc STRINGS>\n");
	return 0;
				
}

#define MAXCLINE	4096

int GetLine(Context* cx, Node* goalscar, List* module)
{
	char*	cline;
	
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 1) {
		syserr("usage : <getline VAR [PRED]>\n");
		return 0;
	}
	Node* nvar = goalscar->Car();
	if (nvar->kind() != UNDEF) {
		syserr("getline : the first argument is not a variable.");
		return 0;
	}

	Node* npred;
	if (ll >= 2) {
		npred = goalscar->Cdr();
	}

#ifndef __MINGW32__
	if (cx->ioin == stdin) {
		cline = readline(NULL);
		if (cline != 0) {
			add_history(cline);
		} else {
			cline = (char*)malloc(2);
			cline[0] = 0;
		}
	} else {
		cline = (char*)malloc(MAXCLINE);
		if (fgets(cline, MAXCLINE, cx->ioin) == NULL) {
			free(cline);
			return 0;
		}
		cline[MAXCLINE-1] = 0;
	}
#else
	cline = (char*)malloc(MAXCLINE);
	if (fgets(cline, MAXCLINE-1, cx->ioin) == NULL) {
		free(cline);
		return 0;
	}
	cline[MAXCLINE-1] = 0;
#endif /* __MINGW32__ */

	int n = strlen(cline);
	for (int i = n-1; i >= 0; i--) {
		int c = cline[i];
		if ((c == '\n') || (c == '\r')) {
			cline[i] = 0;
		} else {
			break;
		}
	}
	
	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(cline));

	if (ll >= 2) {
		static char tmpfilename[MAXPATHLEN];
		strncpy(tmpfilename, tmppath, MAXPATHLEN);
		strcat(tmpfilename, "/descXXXXXX");
		FILE* fd = MksTemp(tmpfilename);
		if (fd == NULL) {
			printf("tmpfile : cannot open tmp file \n");
			return 0;
		}
		
		int rn;
		Context* cx2 = new Context(cx->module);
		cx2->inherit = cx->inherit;
		cx2->ioin = cx->ioin;
		cx2->ioout = cx->ioout;
		cx2->tokenflag = cx->tokenflag;

		cx2->ioin = fd;

		fprintf(cx2->ioin, "%s", cline);
		free(cline);

		rewind(cx2->ioin);

		cxpush(cx2, goalscar);
		cxpush(cx2, nvar);
		cxpush(cx2, npred);
		cxpush(cx2, env);
//PrintNode("getline npred ", npred);
		if ((rn=Unify(cx2, npred, cx->module))>0) {
			cx->Merge(cx2);

			fclose(cx2->ioin);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			delete cx2;
			cx2 = 0;

			PushStack(cx, Nil, Nil, env);

			unlink(tmpfilename);
//printf("getline trace 0 : %d \n", rn);
			return rn;
		} else {
			fclose(cx2->ioin);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			delete cx2;
			cx2 = 0;

			unlink(tmpfilename);
//printf("getline trace 1 : %d \n", rn);
			return rn;
		}

	}

	return 1;
}

int SyntaxLine(Context* cx, Node* goalscar, List* module)
{
	std::string	sline;
	
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 2) {
		syserr("usage : <syntax VAR PRED>\n");
		return 0;
	}
	Node* nval = goalscar->Car()->Val();
#if 0
	if (nval->kind() != ATOM) {
		syserr("syntax : the first argument is not a Atom.");
		return 0;
	}

	((Atom*)nval)->toString(sline);
#endif
	
	Node* npred;

	npred = goalscar->Cdr();
	
	{
		static char tmpfilename[MAXPATHLEN];
		strncpy(tmpfilename, tmppath, MAXPATHLEN);
		strcat(tmpfilename, "/descXXXXXX");
		FILE* fd = MksTemp(tmpfilename);
		if (fd == NULL) {
			syserr("tmpfile : cannot open tmp file \n");
			return 0;
		}
		
		int rn;
		Context* cx2 = new Context(cx->module);
		cx2->inherit = cx->inherit;
		cx2->ioin = cx->ioin;
		cx2->ioout = cx->ioout;
		cx2->tokenflag = cx->tokenflag;

		cx2->ioin = fd;

//		fprintf(cx2->ioin, "%s", sline.c_str());
		nval->print(cx2->ioin);		

		rewind(cx2->ioin);

		cxpush(cx2, goalscar);
		cxpush(cx2, npred);
//PrintNode("syntax npred 1 ", npred);

		if ((rn=Unify(cx2, npred, cx->module))>0) {

//PrintNode("syntax npred 2 ", npred->Val());
			cx->Merge(cx2);

			fclose(cx2->ioin);
			cxpop(cx2);
			cxpop(cx2);
			delete cx2;
			cx2 = 0;

			unlink(tmpfilename);
//printf("syntax trace 0 : %d \n", rn);
			return rn;
		} else {
			fclose(cx2->ioin);
			cxpop(cx2);
			cxpop(cx2);
			delete cx2;
			cx2 = 0;

			unlink(tmpfilename);
//printf("syntax trace 1 : %d \n", rn);
			return rn;
		}

	}

	return 1;
}


int TmpFile(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <tmpfile VAR>\n");
		return 0;
	}
	Node* nvar = goalscar->Car();
	if (nvar->kind() != UNDEF) {
		syserr("tmpfile : the first argument is not a variable.");
		return 0;
	}

	static char tmpfilename[MAXPATHLEN];
	strncpy(tmpfilename, tmppath, MAXPATHLEN);
	strcat(tmpfilename, "/descXXXXXX");
	FILE* fd = MksTemp(tmpfilename);
	if (fd == NULL) {
		printf("tmpfile : cannot open tmp file \n");
		return 0;
	}

	fclose(fd);
	
	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)tmpfilename));

	unlink(tmpfilename);

	return 1;
}

int Line(Context* cx, Node* goalscar, List* module)
{
	long long	LineNo = 1;
	long	TelSav;
	int	c;

	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <line VAR>\n");
		return 0;
	}
	Node* nvar = goalscar->Car();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <line VAR>\n");
		return 0;
	}


	TelSav = ftell(cx->ioin);
	rewind(cx->ioin);
	while(TelSav != ftell(cx->ioin)){
		c = fgetc(cx->ioin);
		if(c == '\n') {
			LineNo++;
		}
		if (c == EOF) break;
	}
	fseek(cx->ioin, TelSav, 0);

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(LineNo));

	return 1;

}

int GetlineEval()
{
	char*	cline;
	std::string scline;
	int 	i;
	
	for(;;) {
		int assertflg = 0;
		
#ifndef __MINGW32__
		cline = readline("? ");
		if (cline != 0) {
			add_history(cline);
		} else {
			cline = (char*)malloc(2);
			cline[0] = 0;
		}
#else
		printf("? ");
		cline = (char*)malloc(MAXCLINE);
		if (fgets(cline, MAXCLINE-1, stdin) == NULL) {
			free(cline);
			return 0;
		}
		cline[MAXCLINE-1] = 0;
#endif /* __MINGW32__ */

		for (i = 0; i < strlen(cline); i++) {
			int rc = 1;
			if (cline[i] == ' ') {
				continue;
			}
			char* spc = CharSpace();
			if (cline[i] == spc[0]) {
				int j;
				for (j = 1; j < strlen(spc); j++) {
					if ((i+j < strlen(cline)) &&
							(cline[i+j] != spc[j])) {
						break;
					}
				}
				if (j == strlen(spc)) {
					i = i+j;
					continue;
				}
				rc = 1;
			}
			if (rc) {
				break;
			}
		}
		if (cline[i] == '/') {
			assertflg = 1;
			cline[i] = ' ';
		}
				
		// save tmpline
		static char tmpfilename[MAXPATHLEN];
		strncpy(tmpfilename, tmppath, MAXPATHLEN);
		strcat(tmpfilename, "/descXXXXXX");
		FILE* fd = MksTemp((char*)tmpfilename);
		if (fd == NULL) {
			printf("getline : cannot open tmp file \n");
			return 0;
		}

		extern FILE* RdFp;
		FILE* fdsave = RdFp;
		RdFp = fd;

		// if '<>' is none in cline, added it.
		std::string sline = cline;

		sline += " ;";

//printf("getlieneval %s \n", sline.c_str());

		std::string sline2 = "";

		for (i = 0; i < sline.length()-1; i++) {
			if (sline[i] == ' ') {
				continue;
			}
			if ((sline[i] == '<') || (sline[i] == ';')) {
				break;
			}
			if ((sline[i] == ':') && (sline[i+1] == ':')) {
				i++;
				char* spc = CharSpace();
				int nspc = strlen(spc);
				for ( ; i < sline.length(); i++) {
					if (isspace(sline[i])) {
						continue;
					}
					if (strncmp(&sline[i], spc, nspc) == 0) {
						i += nspc - 1;
						continue;
					}
					break;
				}
				sline2 = "::";
				for (i++; i < sline.length(); i++) {
					if (isspace(sline[i])) {
						i++;
						break;
					}
					if (sline[i] == '<') {
						break;
					}
					if (strncmp(&sline[i], spc, nspc) == 0) {
						i = i+nspc;
						break;
					}
					sline2 += sline[i];
				}

				if (sline[i] == '<') {
					break;
				}
			}
					

			sline2 += "<";
			sline2 = sline2 + sline.substr(i);

			for (int j = sline2.length(); j >= 0; j--) {
				if (sline2[j] == ';') {
					sline2[j] = '>';
					sline2 = sline2 + ";";
					break;
				}
			}
			sline = sline2;
			break;
		}

		if (assertflg) {
			fprintf(RdFp, "%s\n", sline.c_str());
		} else {
			fprintf(RdFp, "?%s\n", sline.c_str());
		}
		
		// eval
		// save tmpline
		rewind(RdFp);

		jmp_buf savejb;
		memcpy(&savejb, &program_jb, sizeof(jmp_buf));

		extern  int     NwccMain();	
		NwccMain();

		memcpy(&program_jb, &savejb, sizeof(jmp_buf));

		fclose(RdFp);
		RdFp = fdsave;
		
		free(cline);

		unlink(tmpfilename);
	}
}

int DoOpenR(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) < 2) {
		syserr("usage : <oepnr FILENAME [PRED]>\n");
		return 0;
	}

	Node* fname = goalscar->Cdr()->Car()->Val();

	int rn;

	if ((rn = FuncArg(cx, fname, goalscar, module)) <= 0) {
		syserr("openr: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (fname->kind() != ATOM) {
		syserr("usage : <oepnr FILENAME [PRED]>\n");
		return 0;
	}

	if (fname == Nil) {
		syserr("usage : <oepnr FILENAME [PRED]>\n");
		return 0;
	}
	
	std::string sfname;
	((Atom*)fname)->toString(sfname);

	FILE* fd;
	fd = fopen(sfname.c_str(), "rb");
	if (fd == NULL) {
		syserr("openr: can not open file %s \n", sfname.c_str());
		return 0;
	}
	

	Node*	gl = goalscar->Cdr()->Cdr();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = fd;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	int r;
	r=Unify(cx2, gl, cx->module);

	cxpop(cx2);

	if (cx2->ioin != stdin) {
		fclose(cx2->ioin);
	}

	delete cx2;
	cx2 = 0;

	return r;
}

int DoOpenW(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) < 2) {
		syserr("usage : <oepnw FILENAME [PRED]>\n");
		return 0;
	}

	Node* fname = goalscar->Cdr()->Car()->Val();

	int rn;

	if ((rn = FuncArg(cx, fname, goalscar, module)) <= 0) {
		syserr("openw: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (fname->kind() != ATOM) {
		syserr("usage : <oepnw FILENAME [PRED]>\n");
		return 0;
	}

	if (fname == Nil) {
		syserr("usage : <oepnw FILENAME [PRED]>\n");
		return 0;
	}
	
	std::string sfname;
	((Atom*)fname)->toString(sfname);
	FILE* fd;
	fd = fopen(sfname.c_str(), "w");
	if (fd == NULL) {
		syserr("openw: can not open file %s \n", sfname.c_str());
		return 0;
	}


	Node*	gl = goalscar->Cdr()->Cdr();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = fd;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;
	
	cxpush(cx2, gl);

	int r;
	r=Unify(cx2, gl, cx->module);

	cxpop(cx2);

	if (cx2->ioout != stdout) {
		fclose(cx2->ioout);
	}

	delete cx2;
	cx2 = 0;

	return r;
}

int DoOpenWP(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) < 2) {
		syserr("usage : <oepnwp FILENAME [PRED]>\n");
		return 0;
	}

	Node* fname = goalscar->Cdr()->Car()->Val();

	int rn;

	if ((rn = FuncArg(cx, fname, goalscar, module)) <= 0) {
		syserr("openwp: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (fname->kind() != ATOM) {
		syserr("usage : <oepnwp FILENAME [PRED]>\n");
		return 0;
	}

	if (fname == Nil) {
		syserr("usage : <oepnwp FILENAME [PRED]>\n");
		return 0;
	}
	
	std::string sfname;
	((Atom*)fname)->toString(sfname);
	FILE* fd;
	fd = fopen(sfname.c_str(), "w+");
	if (fd == NULL) {
		syserr("openwp: can not open file %s \n", sfname.c_str());
		return 0;
	}

	Node*	gl = goalscar->Cdr()->Cdr();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = fd;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;
	
	cxpush(cx2, gl);

	int r;
	r=Unify(cx2, gl, cx->module);

	cxpop(cx2);

	if (cx2->ioout != stdout) {
		fclose(cx2->ioout);
	}

	delete cx2;
	cx2 = 0;

	return r;
}



int DoRegex(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 5) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}

	Node* nptn = goalscar->Car()->Val();
	int rn;

	if ((rn = FuncArg(cx, nptn, goalscar, module)) <= 0) {
		syserr("regex: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nptn->kind() != ATOM) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}
	Node* nstr = goalscar->Cdr()->Car()->Val();

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("regex: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}
	Node* nfore = goalscar->Cdr()->Cdr()->Car()->Val();

	if ((rn = FuncArg(cx, nfore, goalscar, module)) <= 0) {
		return 0;
	}

	if (nfore->kind() != UNDEF) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}
	Node* nmatch = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();
	if (nmatch->kind() != UNDEF) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}
	Node* nrest = goalscar->Cdr()->Cdr()->Cdr()->Cdr()->Car()->Val();
	if (nrest->kind() != UNDEF) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}

	extern int Regex(std::string ptn, std::string str,
	                std::string& forestr, std::string& matchstr, 
	                	std::string& reststr);
	std::string	str, ptn,  fore, match, rest;

	((Atom*)nstr)->toString(str);
	((Atom*)nptn)->toString(ptn);

	Node* env = Nil->Cons(Nil);

	if (!Regex(ptn, str, fore, match, rest)) {
		return -1;
	}
	 
	SetEnv(env, nfore);
	((Undef*)nfore)->Set(mka((char*)fore.c_str()));

	SetEnv(env, nmatch);
	((Undef*)nmatch)->Set(mka((char*)match.c_str()));

	SetEnv(env, nrest);
	((Undef*)nrest)->Set(mka((char*)rest.c_str()));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int DoSub(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 4) {
		syserr("usage : <sub pattern strings replacestr outputstr>\n");
		return 0;
	}

	Node* nptn = goalscar->Car()->Val();

	int rn;
	
	if ((rn = FuncArg(cx, nptn, goalscar, module)) <= 0) {
		syserr("sub: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nptn->kind() != ATOM) {
		syserr("usage : <sub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* nstr = goalscar->Cdr()->Car()->Val();

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("sub: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <sub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* nreplace = goalscar->Cdr()->Cdr()->Car()->Val();

	if ((rn = FuncArg(cx, nreplace, goalscar, module)) <= 0) {
		syserr("sub: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nreplace->kind() != ATOM) {
		syserr("usage : <sub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* noutput = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();
	if (noutput->kind() != UNDEF) {
		syserr("usage : <sub pattern strings replacestr outputstr>\n");
		return 0;
	}

	extern int Sub(std::string ptn, std::string str,
	                std::string replacestr, std::string& outputstr);
	std::string	str, ptn,  replace, output;

	((Atom*)nptn)->toString(ptn);
	((Atom*)nstr)->toString(str);
	((Atom*)nreplace)->toString(replace);

	Node* env = Nil->Cons(Nil);

	if (!Sub(ptn, str, replace, output)) {
		return -1;
	}
	 
	SetEnv(env, noutput);
	((Undef*)noutput)->Set(mka((char*)output.c_str()));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int DoGSub(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 4) {
		syserr("usage : <gsub pattern strings replacestr outputstr>\n");
		return 0;
	}

	Node* nptn = goalscar->Car()->Val();

	int rn;
	
	if ((rn = FuncArg(cx, nptn, goalscar, module)) <= 0) {
		syserr("gsub: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nptn->kind() != ATOM) {
		syserr("usage : <gsub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* nstr = goalscar->Cdr()->Car()->Val();

	if ((rn = FuncArg(cx, nstr, goalscar, module)) <= 0) {
		syserr("gsub: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nstr->kind() != ATOM) {
		syserr("usage : <gsub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* nreplace = goalscar->Cdr()->Cdr()->Car()->Val();

	if ((rn = FuncArg(cx, nreplace, goalscar, module)) <= 0) {
		syserr("gsub: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nreplace->kind() != ATOM) {
		syserr("usage : <gsub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* noutput = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();
	if (noutput->kind() != UNDEF) {
		syserr("usage : <gsub pattern strings replacestr outputstr>\n");
		return 0;
	}

	extern int GSub(std::string ptn, std::string str,
	                std::string replacestr, std::string& outputstr);
	std::string	str, ptn,  replace, output;

	((Atom*)nptn)->toString(ptn);
	((Atom*)nstr)->toString(str);
	((Atom*)nreplace)->toString(replace);

	Node* env = Nil->Cons(Nil);

	if (!GSub(ptn, str, replace, output)) {
		return -1;
	}
	 
	SetEnv(env, noutput);
	((Undef*)noutput)->Set(mka((char*)output.c_str()));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int DoCountNode(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <countnode VAR>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <countnode VAR>\n");
		return 0;
	}

	Node* env = Nil->Cons(Nil);

	long long n = CountNode();
	
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(n));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int GetTime(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <gettime VAR>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <gettime VAR>\n");
		return 0;
	}

	struct timeval tv;
	gettimeofday (&tv, NULL);
	long long val = tv.tv_sec*1000000+tv.tv_usec;
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(val));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Time(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <time VAR>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <time VAR>\n");
		return 0;
	}

	struct timeval tv;
	gettimeofday (&tv, NULL);
	long long val = tv.tv_sec*1000000+tv.tv_usec - progtime.start_time;
	long double fval = (long double)val / 1000000.0;
	Node* env = Nil->Cons(Nil);

#ifndef __MINGW32__
	long clk_tck = sysconf(_SC_CLK_TCK);
	struct tms tmsbuf;
	times(&tmsbuf);
	long double fuval = (long double)(tmsbuf.tms_utime - progtime.start_utime)
				/ clk_tck;
	long double fsval = (long double)(tmsbuf.tms_stime - progtime.start_stime)
				/ clk_tck;

	Node* l = MkList(mka(fuval), mka(fsval), mka(fval));
#else
	Node* l = MkList(mka(fval));
#endif

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(l);
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Date(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <date VAR>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car();

	if (nvar->kind() != UNDEF) {
		syserr("usage : <date VAR>\n");
		return 0;
	}

	struct timeval tv;
	gettimeofday (&tv, NULL);
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	char* t = strdup(ctime((const time_t*)&tv.tv_sec));
	t[strlen(t)-1] = 0;
	
	((Undef*)nvar)->Set(mka(t));

	free(t);

	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Sleep(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <sleep SEC>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nval = g->Car();

	int rn;
	
	if ((rn = FuncArg(cx, nval, goalscar, module)) <= 0) {
		syserr("sleep: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nval->kind() != ATOM) {
		syserr("usage : <sleep SEC>\n");
		return 0;
	}

	long long t;
	if (!((Atom*)nval)->toInt(t)) {
		syserr("usage : <sleep SEC>\n");
		return 0;
	}
		

	CallSleep(t);
			
	return 1;

}

int USleep(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <usleep SEC>\n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nval = g->Car();

	int rn;
	
	if ((rn = FuncArg(cx, nval, goalscar, module)) <= 0) {
		syserr("usleep: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nval->kind() != ATOM) {
		syserr("usage : <usleep SEC>\n");
		return 0;
	}

	long long t;
	if (!((Atom*)nval)->toInt(t)) {
		syserr("usage : <usleep SEC>\n");
		return 0;
	}
		

	CalluSleep(t);
			
	return 1;

}


int Pause(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) != 0) {
		syserr("usage : <pause>\n");
		return 0;
	}

	fflush(stdin);
	
	char* buf = (char*)malloc(4096);

	fgets(buf, 4096-1, stdin);
	buf[4096-1] = 0;
	
	free(buf);

	return 1;

}

int BaseName(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 2) {
		syserr("usage : <basename VAR PATH>\n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <basename VAR PATH>\n");
		return 0;
	}
	Node* npath = goalscar->Cdr()->Car()->Val();

	int rn;
	
	if ((rn = FuncArg(cx, npath, goalscar, module)) <= 0) {
		syserr("basename: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (npath->kind() != ATOM) {
		syserr("usage : <basename VAR PATH>\n");
		return 0;
	}

	std::string spath;
	
	((Atom*)npath)->toString(spath);

	Node* env = Nil->Cons(Nil);
	 
	SetEnv(env, nvar);

	char* cpath = strdup(spath.c_str());
	((Undef*)nvar)->Set(mka(basename(cpath)));
	free(cpath);

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int DirName(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 2) {
		syserr("usage : <dirname VAR PATH>\n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <dirname VAR PATH>\n");
		return 0;
	}
	Node* npath = goalscar->Cdr()->Car()->Val();

	int rn;
	
	if ((rn = FuncArg(cx, npath, goalscar, module)) <= 0) {
		syserr("dirname: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (npath->kind() != ATOM) {
		syserr("usage : <dirname VAR PATH>\n");
		return 0;
	}

	std::string spath;
	
	((Atom*)npath)->toString(spath);

	Node* env = Nil->Cons(Nil);
	 
	SetEnv(env, nvar);

	char* cpath = strdup(spath.c_str());
	((Undef*)nvar)->Set(mka(dirname(cpath)));
	free(cpath);

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int Suffix(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 3) {
		syserr("usage : <suffix VAR PATH>\n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <suffix VAR PATH>\n");
		return 0;
	}
	Node* npath = goalscar->Cdr()->Car()->Val();

	int rn;
	
	if ((rn = FuncArg(cx, npath, goalscar, module)) <= 0) {
		syserr("suffix: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (npath->kind() != ATOM) {
		syserr("usage : <suffix VAR PATH>\n");
		return 0;
	}
	Node* nsuffix = goalscar->Cdr()->Cdr()->Car()->Val();

	if ((rn = FuncArg(cx, nsuffix, goalscar, module)) <= 0) {
		syserr("suffix: failed in the evaluation of the argument. \n");
		return 0;
	}

	if (nsuffix->kind() != ATOM) {
		syserr("usage : <suffix VAR PATH>\n");
		return 0;
	}

	std::string spath, ssuffix;
	
	((Atom*)npath)->toString(spath);
	((Atom*)nsuffix)->toString(ssuffix);

	int i, l = spath.length()-1;
	for (i = l; i >= 0; i--) {
		if (spath[i] == '.') {
			spath = spath.substr(0, i+1) + ssuffix;
			break;
		}
	}
	if (i == 0) {
		spath = spath + '.';
		spath = spath + ssuffix;
	}


	Node* env = Nil->Cons(Nil);
	 
	SetEnv(env, nvar);

	((Undef*)nvar)->Set(mka((char*)spath.c_str()));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

#ifndef __MINGW32__
int ClearScreen(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 0) {
		syserr("usage : <clear>\n");
		return 0;
	}

	printf("\x1b[2J");

	return 1;
}

int Uname(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 1) {
		syserr("usage : <uname VAR>\n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("usage : <uname VAR>\n");
		return 0;
	}

	struct utsname u;
	uname(&u);
	
	Node* n = MkList(mka(u.sysname), mka(u.nodename),
			 mka(u.release), mka(u.version),
			 mka(u.machine));
	
	Node* env = Nil->Cons(Nil);
	 
	SetEnv(env, nvar);

	((Undef*)nvar)->Set(n);

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

#endif

