/*
 * op call program copyright (C) 2009 H.Niwa
 */


/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <errno.h>
#include <setjmp.h>
#include <sys/time.h>
#include <math.h>
#include <libgen.h>
#include <setjmp.h>

#include <string>
#include <complex>

#include "syserr.h"

#include "bin_node.h"
#include "gc.h"
#include "var.h"
#include "pred.h"
#include "context.h"
#include "unify.h"
#include "builtin.h"
#include "sysmodule.h"
#include "expression.h"
#include "let.h"
#include "func.h"
#include "token.h"
#include "module.h"
#include "code.h"
#include "timeout.h"
#include "help.h"
#include "ncurlib.h"
#include "lisp.h"
#include "complx.h"
#include "compiler.h"
#include "matrix.h"
#include "lib_include.h"
#include "opcall.h"

extern int CmdArgs(Context* cx, Node* goalscar, List* module);
extern int DlibPath(Context* cx, Node* goalscar, List* module);
extern int CutAll(Context* cx, Node* goalscar, List* module);
extern int Write(Context* cx, Node* goalscar, List* module);
extern int WriteNl(Context* cx, Node* goalscar, List* module);

extern int eq(Context* cx, Node* n, List* module);
extern int noteq(Context* cx, Node* n, List* module);
extern int isNil(Context* cx, Node* n, List* module);
extern int isAtom(Context* cx, Node* n, List* module);
extern int isList(Context* cx, Node* n, List* module);
extern int isPred(Context* cx, Node* n, List* module);
extern int isVar(Context* cx, Node* n, List* module);
extern int isUndefVar(Context* cx, Node* n, List* module);
extern int isFloat(Context* cx, Node* n, List* module);
extern int isInteger(Context* cx, Node* n, List* module);


extern int isTrue(Context* cx, Node* goalscar, List* module);
extern int isFalse(Context* cx, Node* goalscar, List* module);
extern int isUnknown(Context* cx, Node* goalscar, List* module);

extern int Max(Context* cx, Node* goalscar, List* module);
extern int Min(Context* cx, Node* goalscar, List* module);
extern int Maxf(Context* cx, Node* goalscar, List* module);
extern int Minf(Context* cx, Node* goalscar, List* module);
extern int Sum(Context* cx, Node* goalscar, List* module);
extern int Sumf(Context* cx, Node* goalscar, List* module);
extern int Avg(Context* cx, Node* goalscar, List* module);
extern int Avgf(Context* cx, Node* goalscar, List* module);


extern int DoOpenR(Context* cx, Node* goalscar, List* module);
extern int DoOpenW(Context* cx, Node* goalscar, List* module);
extern int DoOpenWP(Context* cx, Node* goalscar, List* module);

extern int DoGetc(Context* cx, Node* goalscar, List* module);
extern int DoPutc(Context* cx, Node* goalscar, List* module);
extern int GetLine(Context* cx, Node* goalscar, List* module);
extern int SyntaxLine(Context* cx, Node* goalscar, List* module);
extern int TmpFile(Context* cx, Node* goalscar, List* module);
extern int Line(Context* cx, Node* goalscar, List* module);

extern int DoRegex(Context* cx, Node* goalscar, List* module);
extern int DoSub(Context* cx, Node* goalscar, List* module);
extern int DoGSub(Context* cx, Node* goalscar, List* module);
extern int Split(Context* cx, Node* goalscar, List* module);
extern int Toupper(Context* cx, Node* goalscar, List* module);
extern int Tolower(Context* cx, Node* goalscar, List* module);
extern int Length(Context* cx, Node* goalscar, List* module);
extern int Random(Context* cx, Node* goalscar, List* module);

extern int Char(Context* cx, Node* goalscar, List* module);
extern int Concat(Context* cx, Node* goalscar, List* module);
extern int ConcatCode(Context* cx, Node* goalscar, List* module);
extern int LeftStr(Context* cx, Node* goalscar, List* module);
extern int RightStr(Context* cx, Node* goalscar, List* module);
extern int SubStr(Context* cx, Node* goalscar, List* module);
extern int InsertStr(Context* cx, Node* goalscar, List* module);

extern int SetCode(Context* cx, Node* goalscar, List* module);
extern int CodeCharPrd(Context* cx, Node* goalscar, List* module);
extern int UTF8CharPrd(Context* cx, Node* goalscar, List* module);
extern int EUCCharPrd(Context* cx, Node* goalscar, List* module);
extern int SJISCharPrd(Context* cx, Node* goalscar, List* module);

extern int And(Context* cx, Node* goalscar, List* module);
extern int Or(Context* cx, Node* goalscar, List* module);
extern int Xor(Context* cx, Node* goalscar, List* module);
extern int BitNot(Context* cx, Node* goalscar, List* module);
extern int ShiftL(Context* cx, Node* goalscar, List* module);
extern int ShiftR(Context* cx, Node* goalscar, List* module);

extern int DoMkPred(Context* cx, Node* goalscar, List* module);

extern int DoCountNode(Context* cx, Node* goalscar, List* module);

extern int GetTime(Context* cx, Node* goalscar, List* module);
extern int Time(Context* cx, Node* goalscar, List* module);
extern int Date(Context* cx, Node* goalscar, List* module);
extern int Sleep(Context* cx, Node* goalscar, List* module);
extern int USleep(Context* cx, Node* goalscar, List* module);
extern int Pause(Context* cx, Node* goalscar, List* module);

extern int BaseName(Context* cx, Node* goalscar, List* module);
extern int DirName(Context* cx, Node* goalscar, List* module);
extern int Suffix(Context* cx, Node* goalscar, List* module);

#ifndef __MINGW32__
extern int ClearScreen(Context* cx, Node* goalscar, List* module);
extern int Uname(Context* cx, Node* goalscar, List* module);
#endif


int OpCall(Context* cx, Node* goalscar, List* module)
{
	int r;
	
	goalscar = goalscar->Cdr();
	if (ListLength(goalscar) <= 0) {
		syserr("usage: <opcall OPNUM [ARG..]>\n");
		return 0;
	}
	
	Node* nop = goalscar->Car();
	if (nop->kind() != ATOM) {
		syserr("usage: <opcall OPNUM [ARG..]>\n");
		return 0;
	}
		
	long long op;
	if (!((Atom*)nop)->toInt(op)) {
		syserr("usage: <opcall OPNUM [ARG..]>\n");
		return 0;
	}

	switch(op) {
	case ARGS :
		r = CmdArgs(cx, goalscar, module);
		return r;
	case DDLIBPATH :
		r = DlibPath(cx, goalscar, module);
		return r;
#if 0
	case CUTALL :
		r = CutAll(cx, goalscar, module);
		return r;
#endif
	case MKPRED :
		r = DoMkPred(cx, goalscar, module);
		return r;
	case WRITENL :
		WriteNl(cx, goalscar, module);
		r = 1;
		return r;
	case WRITELN :
		WriteNl(cx, goalscar, module);
		r = 1;
		return r;
	case PRINT :
		WriteNl(cx, goalscar, module);
		r = 1;
		return r;
	case WRITE :
		Write(cx, goalscar, module);
		r = 1;
		return r;
	case ISNIL :
		r = isNil(cx, goalscar, module);
		return r;
	case ISATOM :
		r = isAtom(cx, goalscar, module);
		return r;
	case ISLIST :
		r = isList(cx, goalscar, module);
		return r;
	case ISPRED :
		r = isPred(cx, goalscar, module);
		return r;
	case ISVAR :
		r = isVar(cx, goalscar, module);
		return r;
	case ISUNDEFVAR :
		r = isUndefVar(cx, goalscar, module);
		return r;
	case ISFLOAT :
		r = isFloat(cx, goalscar, module);
		return r;
	case ISINTEGER :
		r = isInteger(cx, goalscar, module);
		return r;
#ifndef __MINGW32__
	case ISINF :
		r = isInf(cx, goalscar, module);
		return r;
	case ISNAN :
		r = isNan(cx, goalscar, module);
		return r;
	case FINTE :
		r = Finite(cx, goalscar, module);
		return r;
#endif
	case ISTRUE :
		r = isTrue(cx, goalscar, module);
		return r;
	case ISFALSE :
		r = isFalse(cx, goalscar, module);
		return r;
	case ISUNKNOWN :
		r = isUnknown(cx, goalscar, module);
		return r;
	case MAX :
		r = Max(cx, goalscar, module);
		return r;
	case MIN :
		r = Min(cx, goalscar, module);
		return r;
	case MAXF :
		r = Maxf(cx, goalscar, module);
		return r;
	case MINF :
		r = Minf(cx, goalscar, module);
		return r;
	case SUM :
		r = Sum(cx, goalscar, module);
		return r;
	case SUMF :
		r = Sumf(cx, goalscar, module);
		return r;
	case AVG :
		r = Avg(cx, goalscar, module);
		return r;
	case AVGF :
		r = Avgf(cx, goalscar, module);
		return r;
	case REGEX :
		r = DoRegex(cx, goalscar, module);
		return r;
	case SUB :
		r = DoSub(cx, goalscar, module);
		return r;
	case GSUB :
		r = DoGSub(cx, goalscar, module);
		return r;
	case SPLIT :
		r = Split(cx, goalscar, module);
		return r;
	case TOUPPER :
		r = Toupper(cx, goalscar, module);
		return r;
	case TOLOWER :
		r = Tolower(cx, goalscar, module);
		return r;
	case LENGTH :
		r = Length(cx, goalscar, module);
		return r;
	case RANDOM :
		r = Random(cx, goalscar, module);
		return r;
	
	case REAL :
		r = CReal(cx, goalscar, module);
		return r;
	case IMAGE :
		r = CImage(cx, goalscar, module);
		return r;
	case ARG :
		r = CArg(cx, goalscar, module);
		return r;
	case NORM :
		r = CNorm(cx, goalscar, module);
		return r;
	case CONJ :
		r = CConj(cx, goalscar, module);
		return r;
	case POLAR :
		r = CPolar(cx, goalscar, module);
		return r;
	
	case SIN :
		r = CSin(cx, goalscar, module);
		return r;
	case COS :
		r = CCos(cx, goalscar, module);
		return r;
	case TAN :
		r = CTan(cx, goalscar, module);
		return r;
	case ASIN :
		r = CASin(cx, goalscar, module);
		return r;
	case ACOS :
		r = CACos(cx, goalscar, module);
		return r;
	case ATAN :
		r = CATan(cx, goalscar, module);
		return r;
	case ATAN2 :
		r = ATan2(cx, goalscar, module);
		return r;
	case SINH :
		r = CSinh(cx, goalscar, module);
		return r;
	case COSH :
		r = CCosh(cx, goalscar, module);
		return r;
	case TANH :
		r = CTanh(cx, goalscar, module);
		return r;
	case ASINH :
		r = CASinh(cx, goalscar, module);
		return r;
	case ACOSH :
		r = CACosh(cx, goalscar, module);
		return r;
	case ATANH :
		r = CATanh(cx, goalscar, module);
		return r;
	case LOG :
		r = CLog(cx, goalscar, module);
		return r;
	case LOG10 :
		r = CLog10(cx, goalscar, module);
		return r;
	case EXP :
		r = CExp(cx, goalscar, module);
		return r;
	case EXP2 :
		r = CExp2(cx, goalscar, module);
		return r;
	case EXP10 :
		r = CExp10(cx, goalscar, module);
		return r;
	case POW :
		r = CPow(cx, goalscar, module);
		return r;
	case SQRT :
		r = CSqrt(cx, goalscar, module);
		return r;
	case ABS :
		r = CAbs(cx, goalscar, module);
		return r;
	case DPI :
		r = PI(cx, goalscar, module);
		return r;
	case DE :
		r = E(cx, goalscar, module);
		return r;
	case INT :
		r = Int(cx, goalscar, module);
		return r;
	case FLOOR :
		r = Floor(cx, goalscar, module);
		return r;
	case CEIL :
		r = Ceil(cx, goalscar, module);
		return r;
	case TRUNC :
		r = Trunc(cx, goalscar, module);
		return r;
	case CAR :
		r = Car(cx, goalscar, module);
		return r;
	case CDR :
		r = Cdr(cx, goalscar, module);
		return r;
	
	case CAAR :
		r = Caar(cx, goalscar, module);
		return r;
	case CADR :
		r = Cadr(cx, goalscar, module);
		return r;
	case CDAR :
		r = Cdar(cx, goalscar, module);
		return r;
	case CDDR :
		r = Cddr(cx, goalscar, module);
		return r;
	case CAAAR :
		r = Caaar(cx, goalscar, module);
		return r;
	case CAADR :
		r = Caadr(cx, goalscar, module);
		return r;
	case CADAR :
		r = Cadar(cx, goalscar, module);
		return r;
	case CADDR :
		r = Caddr(cx, goalscar, module);
		return r;
	case CDAAR :
		r = Cdaar(cx, goalscar, module);
		return r;
	case CDADR :
		r = Cdadr(cx, goalscar, module);
		return r;
	case CDDAR :
		r = Cddar(cx, goalscar, module);
		return r;
	case CDDDR :
		r = Cdddr(cx, goalscar, module);
		return r;
	case CAAAAR :
		r = Caaaar(cx, goalscar, module);
		return r;
	case CAAADR :
		r = Caaadr(cx, goalscar, module);
		return r;
	case CAADAR :
		r = Caadar(cx, goalscar, module);
		return r;
	case CAADDR :
		r = Caaddr(cx, goalscar, module);
		return r;
	case CADAAR :
		r = Cadaar(cx, goalscar, module);
		return r;
	case CADADR :
		r = Cadadr(cx, goalscar, module);
		return r;
	case CADDAR :
		r = Caddar(cx, goalscar, module);
		return r;
	case CADDDR :
		r = Cadddr(cx, goalscar, module);
		return r;
	case CDAAAR :
		r = Cdaaar(cx, goalscar, module);
		return r;
	case CDAADR :
		r = Cdaadr(cx, goalscar, module);
		return r;
	case CDADAR :
		r = Cdadar(cx, goalscar, module);
		return r;
	case CDADDR :
		r = Cdaddr(cx, goalscar, module);
		return r;
	case CDDAAR :
		r = Cddaar(cx, goalscar, module);
		return r;
	case CDDADR :
		r = Cddadr(cx, goalscar, module);
		return r;
	case CDDDAR :
		r = Cdddar(cx, goalscar, module);
		return r;
	case CDDDDR :
		r = Cddddr(cx, goalscar, module);
		return r;
	case CONS :
		r = Cons(cx, goalscar, module);
		return r;
	case NTH :
		r = Nth(cx, goalscar, module);
		return r;
	case CODE :
		r = SetCode(cx, goalscar, module);
		return r;
	case CHAR :
		r = CodeCharPrd(cx, goalscar, module);
		return r;
	case BYTE :
		r = Char(cx, goalscar, module);
		return r;
	case ASCIICHAR :
		r = Char(cx, goalscar, module);
		return r;
	case UTF8CHAR :
		r = UTF8CharPrd(cx, goalscar, module);
		return r;
	case EUCCHAR :
		r = EUCCharPrd(cx, goalscar, module);
		return r;
	case SJISCHAR :
		r = SJISCharPrd(cx, goalscar, module);
		return r;
	case CONCAT :
		r = Concat(cx, goalscar, module);
		return r;
	case CONCATCODE :
		r = ConcatCode(cx, goalscar, module);
		return r;
	case LEFTSTR :
		r = LeftStr(cx, goalscar, module);
		return r;
	case RIGHTSTR :
		r = RightStr(cx, goalscar, module);
		return r;
	case SUBSTR :
		r = SubStr(cx, goalscar, module);
		return r;
	case INSERTSTR :
		r = InsertStr(cx, goalscar, module);
		return r;
	case BITAND :
		r = And(cx, goalscar, module);
		return r;
	case BITOR :
		r = Or(cx, goalscar, module);
		return r;
	case BITXOR :
		r = Xor(cx, goalscar, module);
		return r;
	case BITNOT :
		r = BitNot(cx, goalscar, module);
		return r;
	case SHIFTL :
		r = ShiftL(cx, goalscar, module);
		return r;
	case SHIFTR :
		r = ShiftR(cx, goalscar, module);
		return r;
	case EQ :
		r = eq(cx, goalscar, module);
		return r;
	case NOTEQ :
		r = noteq(cx, goalscar, module);
		return r;
	case IS :
		r = eq(cx, goalscar, module);
		return r;
	case GETC :
		r = DoGetc(cx, goalscar, module);
		return r;
	case PUTC :
		r = DoPutc(cx, goalscar, module);
		r = 1;
		return r;
	case GETLINE :
		r = GetLine(cx, goalscar, module);
		return r;
	case SYNTAX :
		r = SyntaxLine(cx, goalscar, module);
		return r;
	case TMPFILE :
		r = TmpFile(cx, goalscar, module);
		return r;
	case LINE :
		r = Line(cx, goalscar, module);
		return r;
	case OPENR :
		r = DoOpenR(cx, goalscar, module);
		return r;
	case OPENW :
		r = DoOpenW(cx, goalscar, module);
		return r;
	case OPENWP :
		r = DoOpenWP(cx, goalscar, module);
		return r;
	case GETTIME :
		r = GetTime(cx, goalscar, module);
		return r;
	case TIME :
		r = Time(cx, goalscar, module);
		return r;
	case DATE :
		r = Date(cx, goalscar, module);
		return r;
	case SLEEP :
		r = Sleep(cx, goalscar, module);
		return r;
	case USLEEP :
		r = USleep(cx, goalscar, module);
		return r;
	case PAUSE :
		r = Pause(cx, goalscar, module);
		return r;
	case BASENAME :
		r = BaseName(cx, goalscar, module);
		return r;
	case DIRNAME :
		r = DirName(cx, goalscar, module);
		return r;
	case SUFFIX :
		r = Suffix(cx, goalscar, module);
		return r;
#ifndef __MINGW32__
	case CLEAR :
		r = ClearScreen(cx, goalscar, module);
		return r;
	case UNAME :
		r = Uname(cx, goalscar, module);
		return r;
#endif
	case COUNTNODE :
		r = DoCountNode(cx, goalscar, module);
		return r;
	case DGC :
		r = 1;
		GC();
		return r;
#ifndef __MINGW32__
	case CURS_STARTCURSES :
		r = curs_startcurses(cx, goalscar, module);
		return r;
	case CURS_MVPRINTW :
		r = curs_mvprintw(cx, goalscar, module);
		return r;
	case CURS_PRINTW :
		r = curs_printw(cx, goalscar, module);
		return r;
	case CURS_REFRESH :
		r = curs_refresh(cx, goalscar, module);
		return r;
	case CURS_MOVE :
		r = curs_move(cx, goalscar, module);
		return r;
	case CURS_CBREAK :
		r = curs_cbreak(cx, goalscar, module);
		return r;
	case CURS_NOCBREAK :
		r = curs_nocbreak(cx, goalscar, module);
		return r;
	case CURS_RAW :
		r = curs_raw(cx, goalscar, module);
		return r;
	case CURS_NORAW :
		r = curs_noraw(cx, goalscar, module);
		return r;
	case CURS_ECHO :
		r = curs_echo(cx, goalscar, module);
		return r;
	case CURS_NOECHO :
		r = curs_noecho(cx, goalscar, module);
		return r;
	case CURS_ERASE :
		r = curs_erase(cx, goalscar, module);
		return r;
	case CURS_CLEAR :
		r = curs_clear(cx, goalscar, module);
		return r;
	case CURS_GETCH :
		r = curs_getch(cx, goalscar, module);
		return r;
	case CURS_UNGETCH :
		r = curs_ungetch(cx, goalscar, module);
		return r;
	case CURS_GETSTR :
		r = curs_getstr(cx, goalscar, module);
		return r;
	case CURS_INCH :
		r = curs_inch(cx, goalscar, module);
		return r;
	case CURS_INSTR :
		r = curs_instr(cx, goalscar, module);
		return r;
	case CURS_ISENDWIN :
		r = curs_isendwin(cx, goalscar, module);
		return r;
	case CURS_BEEP :
		r = curs_beep(cx, goalscar, module);
		return r;
	case CURS_FLASH :
		r = curs_flash(cx, goalscar, module);
		return r;
	case CURS_HLINE :
		r = curs_hline(cx, goalscar, module);
		return r;
	case CURS_VLINE :
		r = curs_vline(cx, goalscar, module);
		return r;
	case CURS_KEYPAD :
		r = curs_keypad(cx, goalscar, module);
		return r;
	case CURS_SCROLLOK :
		r = curs_scrollok(cx, goalscar, module);
		return r;
	case CURS_SCRL :
		r = curs_scrl(cx, goalscar, module);
		return r;
	case CURS_LINES :
		r = curs_LINES(cx, goalscar, module);
		return r;
	case CURS_COLS :
		r = curs_COLS(cx, goalscar, module);
		return r;
	case CURS_DELCH :
		r = curs_delch(cx, goalscar, module);
		return r;
	case CURS_MVDELCH :
		r = curs_mvdelch(cx, goalscar, module);
		return r;
	case CURS_CLRTOBOT :
		r = curs_clrtobot(cx, goalscar, module);
		return r;
	case CURS_CLRTOEOL :
		r = curs_clrtoeol(cx, goalscar, module);
		return r;
	case CURS_BORDER :
		r = curs_border(cx, goalscar, module);
		return r;
#endif /* __MINGW32__ */

	case COMPCHECKRESERVED :
		r = CompCheckReserved(cx, goalscar, module);
		return r;
	case COMPGETVAR :
		r = CompGetVar(cx, goalscar, module);
		return r;
	case COMPADDVAR :
		r = CompAddVar(cx, goalscar, module);
		return r;
	case COMPCHECKVAR :
		r = CompCheckVar(cx, goalscar, module);
		return r;
	case COMPNEWFUNC :
		r = CompNewFunc(cx, goalscar, module);
		return r;
	case COMPENDFUNC :
		r = CompEndFunc(cx, goalscar, module);
		return r;

	case MATRIXZERO :
		r = matrix_zero(cx, goalscar, module);
		return r;
	case MATRIXUNIT :
		r = matrix_unit(cx, goalscar, module);
		return r;
	case MATRIXSET :
		r = matrix_set(cx, goalscar, module);
		return r;
	case MATRIXDISPLAY :
		r = matrix_display(cx, goalscar, module);
		return r;
	case MATRIXADD :
		r = matrix_add(cx, goalscar, module);
		return r;
	case MATRIXSUB :
		r = matrix_sub(cx, goalscar, module);
		return r;
	case MATRIXMUL :
		r = matrix_mul(cx, goalscar, module);
		return r;
	case MATRIXMULSCALAR :
		r = matrix_mulscalar(cx, goalscar, module);
		return r;
	case MATRIXDIVSCALAR :
		r = matrix_divscalar(cx, goalscar, module);
		return r;
	case MATRIXTRA :
		r = matrix_tra(cx, goalscar, module);
		return r;
	case MATRIXEQ :
		r = matrix_equal(cx, goalscar, module);
		return r;
	case MATRIXNOTEQ :
		r = matrix_notequal(cx, goalscar, module);
		return r;
	case MATRIXDIM :
		r = matrix_dimentions(cx, goalscar, module);
		return r;
	case MATRIXGETROW :
		r = matrix_getrow(cx, goalscar, module);
		return r;
	case MATRIXGETCOLUMN :
		r = matrix_getcolumn(cx, goalscar, module);
		return r;
	case MATRIXSWAPROW :
		r = matrix_swaprow(cx, goalscar, module);
		return r;
	case MATRIXSWAPCOLUMN :
		r = matrix_swapcolumn(cx, goalscar, module);
		return r;
	case MATRIXRANGEROW :
		r = matrix_rangerow(cx, goalscar, module);
		return r;
	case MATRIXRANGECOLUMN :
		r = matrix_rangecolumn(cx, goalscar, module);
		return r;
	case MATRIXRANGE :
		r = matrix_range(cx, goalscar, module);
		return r;
	case MATRIXISMATRIX :
		r = matrix_ismatrix(cx, goalscar, module);
		return r;
	case MATRIXISVECTOR :
		r = matrix_isvector(cx, goalscar, module);
		return r;
	case MATRIXISSQUARE :
		r = matrix_issquare(cx, goalscar, module);
		return r;
	case MATRIXISNULL :
		r = matrix_isnull(cx, goalscar, module);
		return r;
	case MATRIXISDIAGONAL :
		r = matrix_isdiagonal(cx, goalscar, module);
		return r;
	case MATRIXISSYMMETRIC :
		r = matrix_issymmetric(cx, goalscar, module);
		return r;
	case MATRIXISREGULAR :
		r = matrix_isregular(cx, goalscar, module);
		return r;
	case MATRIXISSINGULAR :
		r = matrix_issingular(cx, goalscar, module);
		return r;
	case MATRIXDET :
		r = matrix_det(cx, goalscar, module);
		return r;
	case MATRIXINV :
		r = matrix_inv(cx, goalscar, module);
		return r;
	case MATRIXRANDOM :
		r = matrix_random(cx, goalscar, module);
		return r;
	case MATRIXGETVAL :
		r = matrix_getrow_column(cx, goalscar, module);
		return r;
	case MATRIXSETVAL :
		r = matrix_setrow_column(cx, goalscar, module);
		return r;
	case MATRIXSOLVE :
		r = matrix_solve(cx, goalscar, module);
		return r;
	case MATRIXEIGEN :
		r = matrix_eigen(cx, goalscar, module);
		return r;
	case MATRIXFOURIER :
		r = matrix_fourier(cx, goalscar, module);
		return r;
	case MATRIXINVFOURIER :
		r = matrix_invfourier(cx, goalscar, module);
		return r;
	}
	
	syserr("invalid op code : %d\n", op);
	return 0;			
}
