/*
 * built-in program copyright (C) 2009 H.Niwa
 */


/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <errno.h>
#include <setjmp.h>
#include <sys/time.h>
#include <math.h>
#include <libgen.h>
#include <setjmp.h>

#ifndef __MINGW32__
#include <readline/readline.h>
#include <readline/history.h>
#endif /* __MINGW32__ */

#include <string>

#include "syserr.h"

#include "bin_node.h"
#include "gc.h"
#include "var.h"
#include "pred.h"
#include "context.h"
#include "unify.h"
#include "builtin.h"
#include "expression.h"
#include "let.h"
#include "func.h"
#include "token.h"
#include "module.h"
#include "code.h"
#include "timeout.h"
#include "help.h"

#define MAXPATHLEN 4096

extern int TraceFlag;
extern jmp_buf program_jb;

int incflag = 0;

extern void PushStack(Context* cx, Node* goals, Node* md, Node* env);
extern int PopStack(Context* cx, Node* &goals, Node* &md, Node* &env);

extern Node* UTF8Char(char* str);
extern Node* SJISChar(char* str);
extern Node* EUCChar(char* str);

extern FILE* MksTemp(char* templ);

int ListLength(Node* n);

int CmdArgs(Context* cx, Node* goals);
int Eval(Context* cx, Node* goalscar);
int Include(Context* cx, Node* goals);
int Load(Context* cx, Node* goals);
int ChDir(Context* cx, Node* goals);
int GetPwd(Context* cx, Node* goals);
int Dir(Context* cx, Node* goals);
int DlibPath(Context* cx, Node* goals);
int Plist(Context* cx, Node* goals);
int Quit(Context* cx, Node* goals);
int DoNew(Context* cx, Node* goals);

int Quote(Context* cx, Node* goals);

int CutAll(Context* cx, Node* goals);
int GetModule(Context* cx, Node* goals);
int DoUnify(Context* cx, Node* goals, List* module);
int TimeOut(Context* cx, Node* goals, List* module);
int DoLoop(Context* cx, Node* goals, List* module);
int DoOrPred(Context* cx, Node* goals, List* module);
int DoAlt(Context* cx, Node* goals, List* module);
int DoNot(Context* cx, Node* goals, List* module);
int DoFor(Context* cx, Node* goals, List* module);
int DoForeach(Context* cx, Node* goals, List* module);
int DoFindAll(Context* cx, Node* goals, List* module);


int Write(Context* cx, Node* goalscar, List* module);
int WriteNl(Context* cx, Node* goalscar, List* module);

int wnl(Context* cx, Node* goalscar, List* module);
int wo(Context* cx, Node* goalscar, List* module);
int wx(Context* cx, Node* goalscar, List* module);
int wf(Context* cx, Node* goalscar, List* module);
int wg(Context* cx, Node* goalscar, List* module);
int wtab(Context* cx, Node* goalscar, List* module);

int fr(Context* cx, Node* n);
int fl(Context* cx, Node* n);

int eq(Context* cx, Node* n);
int noteq(Context* cx, Node* n);
int isNil(Node* n);
int isAtom(Node* n);
int isList(Node* n);
int isPred(Node* n);
int isVar(Node* n);
int isUndefVar(Node* n);
int isFloat(Node* n);
int isInteger(Node* n);

int isTrue(Context* cx, Node* goals);
int isFalse(Context* cx, Node* goals);
int isUnknown(Context* cx, Node* goals);

int DoOpenR(Context* cx, Node* goals);
int DoOpenW(Context* cx, Node* goals);
int DoOpenWP(Context* cx, Node* goals);

int DoGetc(Context* cx, Node* goals);
int DoPutc(Context* cx, Node* goals);
int GetLine(Context* cx, Node* goals);
int TmpFile(Context* cx, Node* goals);

int DoRegex(Context* cx, Node* goals);
int DoSub(Context* cx, Node* goals);
int DoGSub(Context* cx, Node* goals);
int Split(Context* cx, Node* goals);
int Length(Context* cx, Node* goals);

int DoSetVar(Context* cx, Node* goals, List* module);
int DoSetArray(Context* cx, Node* goals, List* module);
int DoAsserta(Context* cx, Node* goals, List* module);
int DoAssertz(Context* cx, Node* goals, List* module);
int DoRetract(Context* cx, Node* goals, List* module);
int DoRetractPred(Context* cx, Node* goals, List* module);

int Random(Context* cx, Node* goals);
int Sin(Context* cx, Node* goals);
int Cos(Context* cx, Node* goals);
int Tan(Context* cx, Node* goals);
int ASin(Context* cx, Node* goals);
int ACos(Context* cx, Node* goals);
int ATan(Context* cx, Node* goals);
int Log(Context* cx, Node* goals);
int Exp(Context* cx, Node* goals);
int Sqrt(Context* cx, Node* goals);
int Abs(Context* cx, Node* goals);
int Int(Context* cx, Node* goals);

int Car(Context* cx, Node* goals);
int Cdr(Context* cx, Node* goals);
int Cons(Context* cx, Node* goals);

int Char(Context* cx, Node* goals);
int Concat(Context* cx, Node* goals);

int SetCode(Context* cx, Node* goals);
int CodeCharPrd(Context* cx, Node* goals);
int UTF8CharPrd(Context* cx, Node* goals);
int EUCCharPrd(Context* cx, Node* goals);
int SJISCharPrd(Context* cx, Node* goals);

int And(Context* cx, Node* goals);
int Or(Context* cx, Node* goals);
int Xor(Context* cx, Node* goals);
int BitNot(Context* cx, Node* goals);
int ShiftL(Context* cx, Node* goals);
int ShiftR(Context* cx, Node* goals);

int DoMkPred(Context* cx, Node* goals);

int Tron();
int Troff();
int DoCountNode(Context* cx, Node* goals);


// lib path
Node*	dlibpathnode = Nil;

void GetPath(char* dlibpath)
{
	if (dlibpath == NULL) {
		dlibpath = "";
	}
	dlibpathnode = Nil;
	if (dlibpath != NULL) {
		int	i, j;
		char*	pathbuf = new char[strlen(dlibpath)];
		
		for (i = 0, j = 0; i <= strlen(dlibpath); i++, j++) {
			switch (dlibpath[i]) {
#ifdef __MINGW32__
			case ';' :	// semicolon
#else 
			case ':' :	
#endif
			case 0 :
				pathbuf[j] = 0;
				dlibpathnode = Append(dlibpathnode, 
							MkList(mka(pathbuf)));
				j = -1;
				break;
			default :
				pathbuf[j] = dlibpath[i];
				break;
			}
		}
		delete pathbuf;
	}

}

void GetPath()
{
	char* dlibpath = getenv(DLIBPATH);
	if (dlibpath == NULL) {
		dlibpath = ".";
	}
	GetPath(dlibpath);
}

char* tmppath;

void GetTmpPath()
{
	tmppath = getenv(TMPPATH);
	if (tmppath == NULL) {
		tmppath = "/tmp";
	}
}


int ListLength(Node* n)
{
	int	i;

	if ((n->kind() != LIST) && (n->kind() != PRED)) {
		return 0;
	}

	for (i = 0; n->kind() != ATOM; i++, n=n->Cdr()) 
		;

	return i;
}

int FuncArg(Context* cx, Node*& args, List* module)
{
	Node* retn;
	int rn;
		
	cxpush(cx, args);
	cxpush(cx, module);
	if ((rn=FuncPred(cx, args, module, retn))>0) {
		cxpop(cx);
		cxpop(cx);
		args = retn;
	} else {
		cxpop(cx);
		cxpop(cx);
	}
	return rn;
}


int builtin(Context* cx, Node* goalscar, List* module, int& r)
{
	Node* g = goalscar;

	if (g->Val()->Car()->kind() == ATOM) {
		std::string	s;
		((Atom*)(g->Val()->Car()))->toString(s);
		if (s == "help") {
			r = Help(cx, goalscar);
			return 1;
		} else if (s == "module") {
			r = GetModule(cx, goalscar);
			return 1;
		} else if (s == "list") {
			r = Plist(cx, goalscar);
			return 1;
		} else if (s == "bye") {
			r = Quit(cx, goalscar);
			return 1;
		} else if (s == "quit") {
			r = Quit(cx, goalscar);
			return 1;
		} else if (s == "new") {
			r = DoNew(cx, goalscar);
			return 1;
		} else if (s == "eval") {
			r = Eval(cx, goalscar);
			return 1;
		} else if (s == "print") {
			WriteNl(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "include") {
			r = Include(cx, goalscar);
			return 1;
		} else if (s == "load") {
			r = Load(cx, goalscar);
			return 1;
		} else if (s == "cd") {
			r = ChDir(cx, goalscar);
			return 1;
		} else if (s == "pwd") {
			r = GetPwd(cx, goalscar);
			return 1;
		} else if (s == "dir") {
			r = Dir(cx, goalscar);
			return 1;
		} else if (s == "ls") {
			r = Dir(cx, goalscar);
			return 1;
		} else if (s == "quote") {
			r = Quote(cx, goalscar);
			return 1;
		} else if (s == "obj") {
			r = DoUnify(cx, goalscar, module);
			return 1;
		} else if (s == "unify") {
			r = DoUnify(cx, goalscar, module);
			return 1;
		} else if (s == "timeout") {
			r = TimeOut(cx, goalscar, module);
			return 1;
		} else if (s == "loop") {
			r = DoLoop(cx, goalscar, module);
			return 1;
		} else if (s == "alt") {
			r = DoAlt(cx, goalscar, module);
			return 1;
		} else if (s == "for") {
			r = DoFor(cx, goalscar, module);
			return 1;
		} else if (s == "foreach") {
			r = DoForeach(cx, goalscar, module);
			return 1;
		} else if (s == "map") {
			r = DoForeach(cx, goalscar, module);
			return 1;
		} else if (s == "findall") {
			r = DoFindAll(cx, goalscar, module);
			return 1;
		} else if (s == "or") {
			r = DoOrPred(cx, goalscar, module);
			return 1;
		} else if (s == "not") {
			r = DoNot(cx, goalscar, module);
			return 1;
		} else if (s == "tron") {
			r = Tron();
			return 1;
		} else if (s == "troff") {
			r = Troff();
			return 1;
		} else if (s == "letf") {
			r = float_let(cx, goalscar);
			return 1;
		} else if (s == "rpnf") {
			r = float_rpn(cx, goalscar);
			return 1;
		} else if (s == "let") {
			r = int_let(cx, goalscar);
			return 1;
		} else if (s == "rpn") {
			r = int_rpn(cx, goalscar);
			return 1;
		} else if (s == "func") {
			r = Func(cx, goalscar, module);
			return 1;
		} else if (s == "f") {
			r = Func(cx, goalscar, module);
			return 1;
		} else if (s == "comparef") {
			r = float_compare(cx, goalscar);
			return 1;
		} else if (s == "compare") {
			r = int_compare(cx, goalscar);
			return 1;
		} else if (s == "asserta") {
			r = DoAsserta(cx, goalscar, module);
			return 1;
		} else if (s == "assertz") {
			r = DoAssertz(cx, goalscar, module);
			return 1;
		} else if (s == "assert") {
			r = DoAsserta(cx, goalscar, module);
			return 1;
		} else if (s == "erase") {
			r = DoRetract(cx, goalscar, module);
			return 1;
		} else if (s == "retract") {
			r = DoRetract(cx, goalscar, module);
			return 1;
		} else if (s == "retractpred") {
			r = DoRetractPred(cx, goalscar, module);
			return 1;
		}
	}
	r = -1;
	return 0;
}


int sysmodule(Context* cx, Node* goalscar, List* module, int& r)
{
	Node* retn;
	int	rn;

	std::string	s;

	if (goalscar->Val()->Car()->kind() == ATOM) {
		((Atom*)(goalscar->Val()->Car()))->toString(s);
//PrintNode("sysmodule goalscar ", goalscar);
//printf("sysmodule name %s \n", s.c_str());


		if (s == "args") {
			r = CmdArgs(cx, goalscar);
			return 1;
		} else if (s == "DLIBPATH") {
			r = DlibPath(cx, goalscar);
			return 1;
#if 0
		} else if (s == "cutall") {
			r = CutAll(cx, goalscar);
			return 1;
#endif
		} else if (s == "mkpred") {
			r = DoMkPred(cx, goalscar);
			return 1;
		} else if (s == "writenl") {
			WriteNl(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "writeln") {
			WriteNl(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "print") {
			WriteNl(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "write") {
			Write(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "wcr") {
			wnl(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "wnl") {
			wnl(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "w") {
			Write(cx, goalscar, module);
			r = 1;
			return 1;
		} else if (s == "wo") {
			r = wo(cx, goalscar, module);
			return 1;
		} else if (s == "wx") {
			r = wx(cx, goalscar, module);
			return 1;
		} else if (s == "wf") {
			r = wf(cx, goalscar, module);
			return 1;
		} else if (s == "wg") {
			r = wg(cx, goalscar, module);
			return 1;
		} else if (s == "wtab") {
			r = wtab(cx, goalscar, module);
			return 1;
		} else if (s == "fr") {
			r = fr(cx, goalscar->Val()->Cdr());
			return 1;
		} else if (s == "fl") {
			r = fl(cx, goalscar->Val()->Cdr());
			return 1;
		} else if (s == "isNil") {
			r = isNil(goalscar);
			return 1;
		} else if (s == "isAtom") {
			r = isAtom(goalscar);
			return 1;
		} else if (s == "isList") {
			r = isList(goalscar);
			return 1;
		} else if (s == "isPred") {
			r = isPred(goalscar);
			return 1;
		} else if (s == "isVar") {
			r = isVar(goalscar);
			return 1;
		} else if (s == "isUndefVar") {
			r = isUndefVar(goalscar);
			return 1;
		} else if (s == "isFloat") {
			r = isFloat(goalscar);
			return 1;
		} else if (s == "isInteger") {
			r = isInteger(goalscar);
			return 1;
		} else if (s == "isTrue") {
			r = isTrue(cx, goalscar);
			return 1;
		} else if (s == "isFalse") {
			r = isFalse(cx, goalscar);
			return 1;
		} else if (s == "isUnknown") {
			r = isUnknown(cx, goalscar);
			return 1;
		} else if (s == "regex") {
			r = DoRegex(cx, goalscar);
			return 1;
		} else if (s == "sub") {
			r = DoSub(cx, goalscar);
			return 1;
		} else if (s == "gsub") {
			r = DoGSub(cx, goalscar);
			return 1;
		} else if (s == "split") {
			r = Split(cx, goalscar);
			return 1;
		} else if (s == "length") {
			r = Length(cx, goalscar);
			return 1;
		} else if (s == "setvar") {
			r = DoSetVar(cx, goalscar, module);
			return 1;
		} else if (s == "setarray") {
			r = DoSetArray(cx, goalscar, module);
			return 1;
		} else if (s == "random") {
			r = Random(cx, goalscar);
			return 1;
		} else if (s == "sin") {
			r = Sin(cx, goalscar);
			return 1;
		} else if (s == "cos") {
			r = Cos(cx, goalscar);
			return 1;
		} else if (s == "tan") {
			r = Tan(cx, goalscar);
			return 1;
		} else if (s == "asin") {
			r = ASin(cx, goalscar);
			return 1;
		} else if (s == "acos") {
			r = ACos(cx, goalscar);
			return 1;
		} else if (s == "atan") {
			r = ATan(cx, goalscar);
			return 1;
		} else if (s == "log") {
			r = Log(cx, goalscar);
			return 1;
		} else if (s == "exp") {
			r = Exp(cx, goalscar);
			return 1;
		} else if (s == "sqrt") {
			r = Sqrt(cx, goalscar);
			return 1;
		} else if (s == "abs") {
			r = Abs(cx, goalscar);
			return 1;
		} else if (s == "int") {
			r = Int(cx, goalscar);
			return 1;
		} else if (s == "car") {
			r = Car(cx, goalscar);
			return 1;
		} else if (s == "cdr") {
			r = Cdr(cx, goalscar);
			return 1;
		} else if (s == "cons") {
			r = Cons(cx, goalscar);
			return 1;
		} else if (s == "code") {
			r = SetCode(cx, goalscar);
			return 1;
		} else if (s == "char") {
			r = CodeCharPrd(cx, goalscar);
			return 1;
		} else if (s == "byte") {
			r = Char(cx, goalscar);
			return 1;
		} else if (s == "asciichar") {
			r = Char(cx, goalscar);
			return 1;
		} else if (s == "utf8char") {
			r = UTF8CharPrd(cx, goalscar);
			return 1;
		} else if (s == "eucchar") {
			r = EUCCharPrd(cx, goalscar);
			return 1;
		} else if (s == "sjischar") {
			r = SJISCharPrd(cx, goalscar);
			return 1;
		} else if (s == "concat") {
			r = Concat(cx, goalscar);
			return 1;
		} else if (s == "bitand") {
			r = And(cx, goalscar);
			return 1;
		} else if (s == "bitor") {
			r = Or(cx, goalscar);
			return 1;
		} else if (s == "bitxor") {
			r = Xor(cx, goalscar);
			return 1;
		} else if (s == "bitnot") {
			r = BitNot(cx, goalscar);
			return 1;
		} else if (s == "shiftl") {
			r = ShiftL(cx, goalscar);
			return 1;
		} else if (s == "shiftr") {
			r = ShiftR(cx, goalscar);
			return 1;
		} else if (s == "eq") {
			r = eq(cx, goalscar);
			return 1;
		} else if (s == "noteq") {
			r = noteq(cx, goalscar);
			return 1;
		} else if (s == "is") {
			r = eq(cx, goalscar);
			return 1;
		} else if (s == "getc") {
			r = DoGetc(cx, goalscar);
			return 1;
		} else if (s == "putc") {
			r = DoPutc(cx, goalscar);
			r = 1;
			return 1;
		} else if (s == "getline") {
			r = GetLine(cx, goalscar);
			return 1;
		} else if (s == "tmpfile") {
			r = TmpFile(cx, goalscar);
			return 1;
		} else if (s == "TOKEN") {
			r = token(cx, goalscar, module);
			return 1;
		} else if (s == "SKIPSPACE") {
			tokenSkipSpace(cx);
			r = 1;
			return 1;
		} else if (s == "C") {
			r = tokenC(cx, goalscar);
			return 1;
		} else if (s == "N") {
			r = tokenN(cx, goalscar);
			return 1;
		} else if (s == "A") {
			r = tokenA(cx, goalscar);
			return 1;
		} else if (s == "AN") {
			r = tokenAN(cx, goalscar);
			return 1;
		} else if (s == "CR") {
			r = tokenCR(cx, goalscar);
			return 1;
		} else if (s == "CNTL") {
			r = tokenCNTL(cx, goalscar);
			return 1;
		} else if (s == "EOF") {
			r = tokenEof(cx, goalscar);
			return 1;
		} else if (s == "SPACE") {
			r = tokenSpace(cx, goalscar);
			return 1;
		} else if (s == "PUNCT") {
			r = tokenSpace(cx, goalscar);
			return 1;
		} else if (s == "STRINGS") {
			r = tokenStrings(cx, goalscar);
			return 1;
		} else if (s == "WORD") {
			r = tokenWORD(cx, goalscar);
			return 1;
		} else if (s == "NUM") {
			r = tokenNum(cx, goalscar);
			return 1;
		} else if (s == "FNUM") {
			r = tokenFNum(cx, goalscar);
			return 1;
		} else if (s == "ID") {
			r = tokenID(cx, goalscar);
			return 1;
		} else if (s == "RANGE") {
			r = tokenRange(cx, goalscar);
			return 1;
		} else if (s == "NONRANGE") {
			r = tokenNonRange(cx, goalscar);
			return 1;
		} else if (s == "GETTOKEN") {
			r = tokenGettoken(cx, goalscar);
			return 1;
		} else if (s == "openr") {
			r = DoOpenR(cx, goalscar);
			return 1;
		} else if (s == "openw") {
			r = DoOpenW(cx, goalscar);
			return 1;
		} else if (s == "openwp") {
			r = DoOpenWP(cx, goalscar);
			return 1;
		} else if (s == "countnode") {
			r = DoCountNode(cx, goalscar);
			return 1;
		} else if (s == "gc") {
			r = 1;
			GC();
			return 1;
		}
	}
	r = -1;
	return 0;
}

int CmdArgs(Context* cx, Node* goalscar)
{
	extern int	pargc;
	extern char**	pargv;

	Node* g = goalscar->Cdr();
	if (ListLength(g) != 1) {
		syserr("usage : <args VAR> \n");
		return 0;
	}

	Node* nvar = g->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("args: argument of args should be a variable. ");
		return 0;
	}

	Node*	n = Nil;

	for (int i=1; i < pargc; i++) {
		n = Append(n, MkList(mka(pargv[i])));
	}

	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)(nvar->Val()))->Set(n);

	PushStack(cx, Nil, Nil, env);
	return 1;
	
}

int Eval(Context* cx, Node* goalscar)
{
	if (goalscar->Cdr() != Nil) {
		syserr("usage : <eval> \n");
		return 0;
	}
	
	extern FILE* RdFp;
	FILE* fdsave = RdFp;
	RdFp = cx->ioin;

	cxpush(cx, Module);

	jmp_buf savejb;
	memcpy(&savejb, &program_jb, sizeof(jmp_buf));
	
	extern  int     NwccMain();	
	NwccMain();

	memcpy(&program_jb, &savejb, sizeof(jmp_buf));
	
	cxpop(cx);

	RdFp = fdsave;
	return 1;
}


int Include(Context* cx, Node* goalscar)
{
	Node* g = goalscar->Cdr();
	if (ListLength(g) != 1) {
		syserr("usage : <include OBJ-FILE> or <include MODULE-NAME> \n");
		return 0;
	}

	Node* nname = g->Car()->Val();
	if (nname->kind() != ATOM) {
		syserr("include : name argument of args should be a atom. \n");
		return 0;
	}

	std::string sfile;
	((Atom*)nname)->toString(sfile);

	FILE* fd;

	GetPath();

	char* dfile = strdup((char*)sfile.c_str());
	char* bfile = strdup((char*)sfile.c_str());
	char* dname = dirname(dfile);
	char* bname = basename(bfile);
	free(dfile);
	free(bfile);

	if ((dname[0] == '.') && (dname[1] == 0) 
			&& (sfile.c_str()[1] != '/') ) {
		Node* nd;
		fd = NULL;

		if (dlibpathnode == Nil) {
			fd = fopen(sfile.c_str(), "rb");
		} else {			
			for (nd = dlibpathnode; nd != Nil; nd=nd->Cdr()) {
				std::string s = "";

				if (nd->Car()->kind() != ATOM) {
					continue;
				}

				((Atom*)(nd->Car()))->toString(s);
#ifndef __MINGW32__
				s = s + "/" + sfile;
#else
				s = s + "\\" + sfile;
#endif

				fd = fopen(s.c_str(), "rb");
				if (fd != NULL) {
					break;
				}
			}
		}
	} else {
		fd = fopen(sfile.c_str(), "rb");
	}
	if (fd == NULL) {
		PrintNode("include : file name : ", nname);
		printf("include : Can not open include file \n");
		return 0;
	}

	extern FILE* RdFp;
	FILE* fdsave = RdFp;
	RdFp = fd;

	List* modulesave = Module;
	Module = (List*)Nil->Cons(Nil);

	cxpush(cx, Module);
	cxpush(cx, modulesave);

	jmp_buf savejb;
	memcpy(&savejb, &program_jb, sizeof(jmp_buf));

	extern  int     NwccMain();	
	NwccMain();

	memcpy(&program_jb, &savejb, sizeof(jmp_buf));

	cxpop(cx);
	cxpop(cx);

//printf("Include module "); Module->print(); printf("\n");
//printf("Include modulesave "); modulesave->print(); printf("\n");
//printf("Include assert "); MkList(MkPred(Cons(nname, Module)))->print(); printf("\n");

	Assert(modulesave, MkList(MkPred(Cons(nname, Module->Car()))));
	Module = modulesave;
	fclose(fd);
	RdFp = fdsave;
	
	incflag = 1;
	return 1;
	
}

int Load(Context* cx, Node* goalscar)
{
	Node* g = goalscar->Cdr();
	if (ListLength(g) != 1) {
		syserr("usage : <load OBJ-FILE> or <load MODULE-NAME> \n");
		return 0;
	}

	Node* nname = g->Car()->Val();
	if (nname->kind() != ATOM) {
		syserr("load : name argument of args should be a atom. \n");
		return 0;
	}

	std::string sfile;
	((Atom*)nname)->toString(sfile);

	FILE* fd;

	GetPath();

	char* dfile = strdup((char*)sfile.c_str());
	char* bfile = strdup((char*)sfile.c_str());
	char* dname = dirname(dfile);
	char* bname = basename(bfile);
	free(dfile);
	free(bfile);

	if ((dname[0] == '.') && (dname[1] == 0) 
			&& (sfile.c_str()[1] != '/') ) {
		Node* nd;
		fd = NULL;

		if (dlibpathnode == Nil) {
			fd = fopen(sfile.c_str(), "rb");
		} else {			
			for (nd = dlibpathnode; nd != Nil; nd=nd->Cdr()) {
				std::string s = "";

				if (nd->Car()->kind() != ATOM) {
					continue;
				}

				((Atom*)(nd->Car()))->toString(s);
#ifndef __MINGW32__
				s = s + "/" + sfile;
#else
				s = s + "\\" + sfile;
#endif

				fd = fopen(s.c_str(), "rb");
				if (fd != NULL) {
					break;
				}
			}
		}
	} else {
		fd = fopen(sfile.c_str(), "rb");
	}
	if (fd == NULL) {
		PrintNode("load : file name : ", nname);
		printf("load : Can not open load file \n");
		return 0;
	}

	extern FILE* RdFp;
	FILE* fdsave = RdFp;
	RdFp = fd;

	cxpush(cx, Module);

	jmp_buf savejb;
	memcpy(&savejb, &program_jb, sizeof(jmp_buf));
	
	extern  int     NwccMain();	
	NwccMain();

	memcpy(&program_jb, &savejb, sizeof(jmp_buf));

	cxpop(cx);

	fclose(fd);
	RdFp = fdsave;

	incflag = 1;
	
	return 1;
	
}

int DlibPath(Context* cx, Node* goalscar)
{
	extern int	pargc;
	extern char**	pargv;

	Node* g = goalscar->Cdr();
	if (ListLength(g) != 1) {
		syserr("usage : <DLIBPATH VAR> \n");
		return 0;
	}

	Node* nvar = g->Car()->Val();
	if (nvar->kind() != UNDEF) {
		if (nvar->kind() == ATOM) {	// set DLIBPATH
			std::string sdpath;
			
			((Atom*)nvar)->toString(sdpath);
			GetPath((char*)sdpath.c_str());
#ifndef __MINGW32__
			setenv(DLIBPATH, (char*)sdpath.c_str(), 1);
#else
			std::string s;
			s = DLIBPATH;
			s = s + "=";
			s = s + sdpath;
			putenv((char*)s.c_str());
#endif
			return 1;
		}
		syserr("DLIBPATH: argument of args should be a variable or an atom. \n");
		return 0;		
	}

	GetPath();

	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)(nvar->Val()))->Set(Dup(dlibpathnode));

	PushStack(cx, Nil, Nil, env);
	return 1;
	
}

int ChDir(Context* cx, Node* goalscar)
{
	extern int	pargc;
	extern char**	pargv;

	Node* g = goalscar->Cdr();
	if (ListLength(g) != 1) {
		syserr("usage : <cd PATH> \n");
		return 0;
	}

	Node* npath = g->Car()->Val();

	if (npath->kind() != ATOM) {
		syserr("usage : <cd PATH> \n");
		return 0;
	}

	std::string spath;
	((Atom*)npath)->toString(spath);

	int r = chdir(spath.c_str());
	if (r != 0) {
		return -1;
	}
	return 1;
	
}

int GetPwd(Context* cx, Node* goalscar)
{
	extern int	pargc;
	extern char**	pargv;

	Node* g = goalscar->Cdr();
	if (ListLength(g) != 1) {
		syserr("usage : <pwd VAR> \n");
		return 0;
	}

	Node* nvar = g->Car()->Val();
	if (nvar->kind() != UNDEF) {
		syserr("pwd: argument of args should be a variable. ");
		return 0;
	}

	Node*	npwd = Nil;

	extern char* GetCwd();
	
	char* cpwd = GetCwd();
	if (cpwd == NULL) {
		return 0;
	}
	
	npwd = mka(cpwd);

	free(cpwd);
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)(nvar->Val()))->Set(npwd);

	PushStack(cx, Nil, Nil, env);
	return 1;
	
}

int Dir(Context* cx, Node* goalscar)
{
	Node* g = goalscar->Cdr();
	if (goalscar->Cdr() != Nil) {
		syserr("usage : <dir/ls> \n");
		return 0;
	}

#ifdef __MINGW32__
	system("dir");
#else 
	system("ls");
#endif

	return 1;
}

int Plist(Context* cx, Node* goalscar)
{
	Node* g = goalscar->Cdr();
	if (goalscar->Cdr() != Nil) {
		syserr("usage : <list> \n");
		return 0;
	}

	PPmodule(cx->module);

	return 1;
}


int Quit(Context* cx, Node* goalscar)
{

	Node* g = goalscar->Cdr();
	if (goalscar->Cdr() != Nil) {
		syserr("usage : <quit> \n");
		return 0;
	}

	exit(0);
	return 1;
}

int DoNew(Context* cx, Node* goalscar)
{

	Node* g = goalscar->Cdr();
	if (goalscar->Cdr() != Nil) {
		syserr("usage : <new> \n");
		return 0;
	}

	Module = (List*)Nil->Cons(Nil);

	return 1;
}

int Quote(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <quote PRED> \n");
		return 0;
	}
	
	return 1;
}

int CutAll(Context* cx, Node* goalscar)
{
	if (goalscar->Cdr() != Nil) {
		syserr("usage : <cutall> \n");
		return 0;
	}
	cx->CutAll();
	
	return 1;
}

int GetModule(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <getmodule VAR> \n");
		return 0;
	}

	Node*	n = goalscar->Cdr()->Val();
	Node*	nvar = n->Car();
	n = n->Cdr();

	if (n != Nil) {
		return 0;
	}

	if (nvar->Val()->kind() == UNDEF) {
		Node* env = Nil->Cons(Nil);

		SetEnv(env, nvar->Val());
		((Undef*)(nvar->Val()))->Set(cx->module);

		PushStack(cx, Nil, Nil, env);
		return 1;
	}

	return 0;
}

int getvar(Node* var, Node* mod, Node*& val)
{
	val = Nil;

	if (mod->kind() != LIST) {
		return 0;
	}
	Node* m = (Node*)mod;

//PrintNode("getvar var ", var);

	for ( ; m->kind() != ATOM; m = m->Cdr()) {
//PrintNode("getvar m ", m);
		if (m->Car()->kind() == LIST) {
			if (m->Car()->Car()->kind() == PRED) {
				if (m->Car()->Car()->Car()->Eq(var)) {
					val = m->Car()->Car()->Cdr();
					return 1;
				}
			}
		}
	}
	return 0;
}

// <Unify module goals>
int DoUnify(Context* cx, Node* goalscar, List* module)
{
	int	rn;
	
	if (ListLength(goalscar->Cdr()) < 2) {
		syserr("usage : <unify MODULE PRED> \nusage : <obj OBJ PRED> \n");
		return 0;
	}
	
	Node*	n = goalscar->Cdr();
	Node*	marg = n->Car()->Val();

	if (marg->kind() == ATOM) {
		std::string argname;
		((Atom*)marg)->toString(argname);

		// sys module pred
		if (((Atom*)marg)->EqStr("sys")) {
			int r = 1;
			int rt;
//			cx->inherit = Cons(mka("sys"), cx->inherit);
			Node* g = goalscar->Cdr()->Cdr();
			if (g->Car()->kind() != PRED) {
				g = g->Car();
			}
			do {
				rt = sysmodule(cx, g->Car(), module, r);
				if (rt) {
					if (r>0) {
				    		if (TraceFlag) {
							PrintNode("sys module: ", goalscar->Val(), "...success");
						}
					} else {
					    if (TraceFlag) {
						if (r == 0) {
							PrintNode("sys module: ", goalscar->Val(), "...false");
						} else {	// rval == -1
							PrintNode("sys module: ", goalscar->Val(), "...unkown");
						}
					    }
					    return r;
					}
				} else {
					return -1;
				}
				g = g->Cdr();
			} while (g->kind() != ATOM);
			return 1;
		}
		if (!getvar(marg, module->Car(), marg)) {
			syserr("obj/unify : cannot find obj module %s \n", 
					argname.c_str());
			return 0;
		}
		marg = Append(Dup(marg), module->Car());
		marg = MkList(marg);
	}

	if (marg->kind() != LIST) {
		marg = MkList(marg);
	}

	List*	m = (List*)marg;
	
	n = n->Cdr();

	Node*	gl = n;

	if (gl->kind() != LIST) {
		gl = MkList(gl);
	}

	Context *cx2 = new Context(m);
	cx2->inherit = cx->inherit;
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

//PrintNode("DoUnify gl ", gl);
	cxpush(cx2, gl);
	cxpush(cx2, goalscar);
	cxpush(cx2, m);
	cxpush(cx2, n);
	if ((rn=Unify(cx2, gl, m))>0) {
		cx->Merge(cx2);

		cxpop(cx2);
		cxpop(cx2);
		cxpop(cx2);
		cxpop(cx2);
		delete cx2;
		cx2 = 0;

		return rn;
	} else {
		cxpop(cx2);
		cxpop(cx2);
		cxpop(cx2);
		cxpop(cx2);
		delete cx2;
		cx2 = 0;

		return rn;
	}
	return 0;
}

int TimeOut(Context* cx, Node* goalscar, List* module)
{
#if 1
	int	rn;
	
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <timeout TIMEOUT(usec) PRED> \n");
		return 0;
	}
	
	Node*	timearg = goalscar->Cdr()->Car();
	if (timearg->kind() != ATOM) {
		syserr("usage : <timeout *TIMEOUT(usec)* PRED> \n");
		return 0;
	}
	int tm;
	if (!((Atom*)timearg)->toInt(tm)) {
		syserr("usage : <timeout *TIMEOUT(usec)* PRED> \n");
		return 0;
	}
	
	Node*	predarg = goalscar->Cdr()->Cdr()->Car();
	if (predarg->kind() != PRED) {
		syserr("usage : <timeout TIMEOUT(usec) *PRED*> \n");
		return 0;
	}


	timeout* tim = RegistTime(tm);
	if (setjmp(tim->jbuf)) {
		PopContext(cx);

		delete tim;
		
		return -1;
	}

	Context* cx2 = new Context(module);
	cx2->inherit = cx->inherit;
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;

	cxpush(cx2, goalscar);
	cxpush(cx2, predarg);
	cxpush(cx2, module);
	if ((rn=Unify(cx2, predarg, module))>0) {
		cx->Merge(cx2);

		cxpop(cx2);
		cxpop(cx2);
		cxpop(cx2);
		delete cx2;
		cx2 = 0;

		delete tim;
		return rn;
	} else {
		cxpop(cx2);
		cxpop(cx2);
		cxpop(cx2);
		delete cx2;
		cx2 = 0;

		delete tim;
		return rn;
	}
#endif
}

// <loop preds>
// { preds }
int DoLoop(Context* cx, Node* goalscar, List* module)
{
	int	r;
	
	Node*	gl = MkList(goalscar->Cdr());
//printf("builtin DoLoop "); goalscar->print(); printf("\n");

	Context *cx2 = new Context(module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	do {
		long inp = ftell(cx2->ioin);
		
		r = Unify(cx2, gl, module);
//printf("DoLoop cx2 "); cx2->env_stack->print(); printf("\n");
//printf("DoLoop r %d gl ", r); gl->Val()->print(); printf("\n");

//printf("doloop %ld \n", inp);

		if (r>0) {
			if (cx->tokenflag) cx->token = cx2->token;
			inp = ftell(cx2->ioin);
		}

		cx2->Clear();

		fseek(cx2->ioin, inp, 0);

//printf("DoLoop gl clear "); gl->Val()->print(); printf("\n");
	} while (r>0);

	cxpop(cx2);
	
	delete cx2;
	cx2 = 0;

	return 1;
}

// <or preds>
// [ preds ]
int DoOrPred(Context* cx, Node* goalscar, List* module)
{
	Node*	gl;
	Context *cx2;
//printf("builtin DoOrPred "); goalscar->Cdr()->print(); printf("\n");

	for (goalscar = goalscar->Cdr(); goalscar->kind() != ATOM; 
					goalscar = goalscar->Cdr()) {
		int r = 0;
		
		gl = goalscar->Car();
		cx2 = new Context(module);
		cx2->ioin = cx->ioin;
		cx2->ioout = cx->ioout;
		cx2->tokenflag = cx->tokenflag;
		cx2->token = cx->token;

		cxpush(cx2, gl);
//PrintNode("DoOrPred gl ", gl);
		r = Unify(cx2, gl, module);

		cxpop(cx2);

		if (r>0) {
			cx->Merge(cx2);
			delete cx2;
			cx2 = 0;
			return 1;
		} else {
			cx2->Clear();
			delete cx2;
			cx2 = 0;
		}
	}
	return -1;
}

// <alt preds>
// [ preds ]
int DoAlt(Context* cx, Node* goalscar, List* module)
{
	int	rn;
	Node*	gl = MkList(goalscar->Cdr());
//printf("builtin DoAlt "); gl->print(); printf("\n");

	Context *cx2 = new Context(module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	if ((rn=Unify(cx2, gl, module))>0) {
		cx->Merge(cx2);
	}

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	return 1;
}

// <not preds>
int DoNot(Context* cx, Node* goalscar, List* module)
{
	Node*	gl = goalscar->Cdr();
	int	r;
	
//printf("builtin DoNot "); gl->print(); printf("\n");

	Context *cx2 = new Context(module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	r = Unify(cx2, gl, module);

//printf("Donot "); gl->print(); printf("\n");

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	if (r == 1) {
		return 0;
	} else if (r == -1) {
		return -1;
	} else {
		return 1;
	}
}

// <for (#var to_val) block> or <for (#var initval to_val) block>
int DoFor(Context* cx, Node* goalscar, List* module)
{
	int i;

	if (ListLength(goalscar) < 2) {
printf("for trace 0\n");
		syserr("usage : <for (VAR LOOP-COUNT) PRED> or <for (VAR FROM TO) PRED>\n");
		return 0;
	}
	Node*	gl = goalscar->Cdr()->Cdr();
	Node*	glargs = goalscar->Cdr()->Car();
	
	int nc = ListLength(glargs);
	if (!((nc == 2) || (nc == 3)) ) {
printf("for trace 1\n");
		syserr("usage : <for (VAR LOOP-COUNT) PRED> or <for (VAR FROM TO) PRED>\n");
		return 0;
	}
		
	Node* nvar = glargs->Car()->Val();
	if (nvar->kind() != UNDEF) {
printf("for trace 2\n");
		syserr("usage : <for (VAR LOOP-COUNT) PRED> or <for (VAR FROM TO) PRED>\n");
		return 0;
	}

	glargs = glargs->Cdr();

	Node* nto_val;
	int init_val = 0;
	int to_val;
	if (nc == 2) {
		init_val = 0;
		nto_val = glargs->Car()->Val();
		if (nto_val->kind() != ATOM) {
printf("for trace 3\n");
			syserr("usage : <for (VAR LOOP-COUNT) PRED> or <for (VAR FROM TO) PRED>\n");
			return 0;
		}
		if (!((Atom*)nto_val)->toInt(to_val)) {
printf("for trace 4\n");
			syserr("usage : <for (VAR LOOP-COUNT) PRED> or <for (VAR FROM TO) PRED>\n");
			return 0;
		}
	} else {
		Node* ninitval = glargs->Car()->Val();
		if (ninitval->kind() != ATOM) {
printf("for trace 5\n");
			syserr("usage : <for (VAR LOOP-COUNT) PRED> or <for (VAR FROM TO) PRED>\n");
			return 0;
		}
		if (!((Atom*)ninitval)->toInt(init_val)) {
printf("for trace 6\n");
			syserr("usage : <for (VAR LOOP-COUNT) PRED> or <for (VAR FROM TO) PRED>\n");
			return 0;
		}
		
		nto_val = glargs->Cdr()->Car()->Val();
		if (nto_val->kind() != ATOM) {
printf("for trace 7\n");
			syserr("usage : <for (VAR LOOP-COUNT) PRED> or <for (VAR FROM TO) PRED>\n");
			return 0;
		}
		if (!((Atom*)nto_val)->toInt(to_val)) {
printf("for trace 8\n");
			syserr("usage : <for (VAR LOOP-COUNT) PRED> or <for (VAR FROM TO) PRED>\n");
			return 0;
		}
	}
	
	for (i = init_val; i < to_val; i++) {
		Context *cx2 = new Context(module);
		cx2->ioin = cx->ioin;
		cx2->ioout = cx->ioout;
		cx2->tokenflag = cx->tokenflag;
		cx2->token = cx->token;

		Node* env = Nil->Cons(Nil);
		SetEnv(env, nvar);
		((Undef*)nvar)->Set(mka(i));

		cxpush(cx2, env);
		cxpush(cx2, nvar);
		cxpush(cx2, goalscar);
				
		int r = Unify(cx2, gl, module);

		if (r == 1) {
			if (cx->tokenflag) cx->token = cx2->token;
		}

		cx2->Clear();
		
		cxpop(cx2);
		cxpop(cx2);
		cxpop(cx2);
		
		UnsetEnv(env);
		
		delete cx2;
		cx2 = 0;

		if (r != 1) {
			return r;
		}
	}		
	return 1;
}

int DoForeach(Context* cx, Node* goalscar, List* module)
{
	int i;
	Node*	gl = goalscar->Cdr();
	Node*	glargs = gl->Car();
	int	rn;
	
	int narg = ListLength(glargs);
	if (narg < 2) {
		syserr("usage : <foreach (VAR LIST) PRED>\n");
		return 0;
	}

	Node* nvar = glargs->Car()->Val();
	if (nvar->kind() != UNDEF) {
		return 0;
	}

	glargs = glargs->Cdr();

	Node* nlist = glargs->Car()->Val();
	if (nlist->kind() != LIST) {
		return 0;
	}

	gl = gl->Cdr();

	for ( ; nlist->kind() != ATOM; nlist=nlist->Cdr()) {
		Context *cx2 = new Context(module);
		cx2->ioin = cx->ioin;
		cx2->ioout = cx->ioout;
		cx2->tokenflag = cx->tokenflag;
		cx2->token = cx->token;

		Node* env = Nil->Cons(Nil);
		SetEnv(env, nvar);
		((Undef*)nvar)->Set(nlist->Car());

		cxpush(cx2, env);
		cxpush(cx2, nvar);
		cxpush(cx2, goalscar);
		cxpush(cx2, nlist);
				
		rn = Unify(cx2, gl, module);

		if (rn == 1) {
			if (cx->tokenflag) cx->token = cx2->token;
		}
		
		cx2->Clear();

		cxpop(cx2);
		cxpop(cx2);
		cxpop(cx2);
		cxpop(cx2);
		
		UnsetEnv(env);
		
		delete cx2;
		cx2 = 0;
/*
		if (rn != 1) {
			return rn;
		}
*/
	}		
	return 1;
}

int DoFindAll(Context* cx, Node* goalscar, List* module)
{
	int i;
	Node*	gl = goalscar->Cdr();
	int	rn;

	if (gl == Nil) {
		syserr("usage : <findall PRED>\n");
		return 0;
	}

	gl = Append(gl, MkList(MkPred(mka("unknown"))));
//PrintNode("findall gl ", gl);

	Context *cx2 = new Context(module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);
	cxpush(cx2, goalscar);
				
	rn = Unify(cx2, gl, module);

	if (rn == 1) {
		if (cx->tokenflag) cx->token = cx2->token;
	}
		
	cx2->Clear();

	cxpop(cx2);
	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	return 1;
}

int DoAsserta(Context* cx, Node* goalscar, List* module)
{
	int i;
	Node*	gl = goalscar->Cdr();
	Node* nargs = gl->Val();

	if (nargs->Car()->kind() != PRED) {
		syserr("usage : <asserta PRED-LIST>\n");
		return 0;
	}
	
//PrintNode("DoAsserta 1 ", module);

	Asserta(module, nargs);
	
//	module->SetCar(Append(MkList(nargs), module->Car()));

//PrintNode("DoAsserta 2 ", module);

	return 1;
}

int DoAssertz(Context* cx, Node* goalscar, List* module)
{
	int i;
	Node*	gl = goalscar->Cdr();
	Node* nargs = gl->Val();
	
	if (nargs->Car()->kind() != PRED) {
		syserr("usage : <assert PRED-LIST>\n");
		return 0;
	}

//PrintNode("DoAssert nargs ", nargs);
//PrintNode("DoAssert 1 ", module);

	Assert(module, nargs);

//	module = (List*)Append(module->Car(), MkList(nargs));

//PrintNode("DoAssert 2 ", module);

	return 1;
}

int DoRetractPred(Context* cx, Node* goalscar, List* module)
{
	int i;
	Node*	gl = goalscar->Cdr()->Val();

	int l = ListLength(gl);
	if (l != 1) {
printf("DoRetract 1 \n");
		syserr("usage : <retract PRED>\n");
		return 0;
	}
	Node* nargs = gl->Car()->Val();
	if (nargs->kind() != PRED) {
//printf("DoRetract 2 \n");
		return 0;
	}

	List* mod = (List*)module->Car();
	List* modsave;
	Node* env = Nil->Cons(Nil);

	int r = Unification(nargs, Dup(mod->Car()->Car()), env, cx);
//PrintNode("DoRetract nargs ", nargs);
//PrintNode("DoRetract mod->Car() ", mod->Car());
	
	UnsetEnv(env);
	if (r) {
		module->SetCar(mod->Cdr());
		return 1;
	}

	modsave = mod;
	mod = (List*)mod->Cdr();
	for ( ; mod->kind() != ATOM; mod = (List*)mod->Cdr()) {
		r = Unification(nargs, Dup(mod->Car()->Car()), env, cx);
//PrintNode("DoRetract mod->Car() ", mod->Car());
		UnsetEnv(env);
		if (r) {
			modsave->SetCdr(mod->Cdr());
			return 1;
		}
		modsave = mod;
	}
//printf("DoRetract 3 \n");
	return 0;
}

int DoRetract(Context* cx, Node* goalscar, List* module)
{
	int i;
	Node*	gl = goalscar->Cdr()->Val();

	int l = ListLength(gl);
	if (l != 1) {
		syserr("usage : <retractpred PRED_NAME>\n");
		return 0;
	}
	Node* nargs = gl->Car()->Val();
	if (nargs->kind() != ATOM) {
		return 0;
	}

	List* mod = (List*)module->Car();
	List* modsave = mod;
	
	while (mod->Car()->Car()->Car()->Eq(nargs)) {
		module->SetCar(mod->Cdr());
		mod = (List*)mod->Cdr();
	}

	mod = (List*)mod->Cdr();
	for ( ; mod->kind() != ATOM; mod = (List*)mod->Cdr()) {
		if (mod->Car()->Car()->Car()->Eq(nargs)) {
			modsave->SetCdr(mod->Cdr());
		}
		modsave = mod;
	}
	return 1;
}

int Format(Context* cx, Node* goalscar)
{
	int i;
	Node*	gl = goalscar->Cdr()->Val();

	int l = ListLength(gl);
	if (l != 3) {
		syserr("usage : <format OUTPUT FORMAT ARG>\n");
		return 0;
	}
	Node* nout = gl->Car();
	if (nout->kind() != UNDEF) {
		return 0;
	}
	Node* nfmt = gl->Cdr()->Car();
	if (nfmt->kind() != ATOM) {
		return 0;
	}
	Node* narg = gl->Cdr()->Cdr()->Car();
	if (narg->kind() != ATOM) {
		return 0;
	}
	
}

int Write(Context* cx, Node* goalscar, List* module)
{
	Node*	n = goalscar->Cdr()->Val();
	int rn;

	cxpush(cx, goalscar);	
	cxpush(cx, module);	
	if ((rn = FuncArg(cx, n, module)) <= 0) {
		cxpop(cx);
		cxpop(cx);
		return rn;
	}
	cxpop(cx);
	cxpop(cx);

	n->Car()->print(cx->ioout);
	n->Cdr()->printcdr(cx->ioout);

	return 1;
}

int WriteNl(Context* cx, Node* goalscar, List* module)
{
	int rn;
	
	if ((rn=Write(cx, goalscar, module)) <= 0) {
		return rn;
	}
	
	fprintf(cx->ioout, "\n");
	return 1;
}


int wnl(Context* cx, Node* goalscar, List* module)
{
	if (ListLength(goalscar->Cdr()) > 0) {
		syserr("usage : <wnl>\n");
		return 0;
	}
	fprintf(cx->ioout, "\n");
	return 1;
}

int wo(Context* cx, Node* goalscar, List* module)
{
	Node* n1 = goalscar->Cdr();
	int rn;

	cxpush(cx, goalscar);
	cxpush(cx, module);	
	if ((rn = FuncArg(cx, n1, module)) <= 0) {
		cxpop(cx);
		cxpop(cx);
		return rn;
	}
	cxpop(cx);
	cxpop(cx);

	if (ListLength(n1) != 1) {
		syserr("usage : <wo NUM>\n");
		return 0;
	}
	
	if (n1->Car()->kind() != ATOM) {
		return 0;
	}
	int	n;
	if (!((Atom*)n1->Car())->toInt(n)) {
		return 0;
	}
		
	fprintf(cx->ioout, "%o", n);
	return 1;
}

int wx(Context* cx, Node* goalscar, List* module)
{
	Node* n1 = goalscar->Cdr();
	int rn;
	
	cxpush(cx, goalscar);
	cxpush(cx, module);	
	if ((rn = FuncArg(cx, n1, module)) <= 0) {
		cxpop(cx);
		cxpop(cx);
		return rn;
	}
	cxpop(cx);
	cxpop(cx);

	if (ListLength(n1) != 1) {
		syserr("usage : <wx NUM>\n");
		return 0;
	}
	
	if (n1->Car()->kind() != ATOM) {
		return 0;
	}
	int	n;
	if (!((Atom*)n1->Car())->toInt(n)) {
		return 0;
	}
		
	fprintf(cx->ioout, "%x", n);
	return 1;
}

int wf(Context* cx, Node* goalscar, List* module)
{
	Node* n1 = goalscar->Cdr();
	int rn;
	
	cxpush(cx, goalscar);
	cxpush(cx, module);	
	if ((rn = FuncArg(cx, n1, module)) <= 0) {
		cxpop(cx);
		cxpop(cx);
		return rn;
	}
	cxpop(cx);
	cxpop(cx);

	if (ListLength(n1) != 1) {
		syserr("usage : <wf NUM>\n");
		return 0;
	}
	
	if (n1->Car()->kind() != ATOM) {
		return 0;
	}
	double	n;
	if (!((Atom*)n1->Car())->toFloat(n)) {
		return 0;
	}
		
	fprintf(cx->ioout, "%f", n);
	return 1;

}

int wg(Context* cx, Node* goalscar, List* module)
{
	Node* n1 = goalscar->Cdr();
	int rn;
	
	cxpush(cx, goalscar);
	cxpush(cx, module);	
	if ((rn = FuncArg(cx, n1, module)) <= 0) {
		cxpop(cx);
		cxpop(cx);
		return rn;
	}
	cxpop(cx);
	cxpop(cx);

	if (ListLength(n1) != 1) {
		syserr("usage : <wg NUM>\n");
		return 0;
	}
	
	if (n1->Car()->kind() != ATOM) {
		return 0;
	}
	double	n;
	if (!((Atom*)n1->Car())->toFloat(n)) {
		return 0;
	}
		
	fprintf(cx->ioout, "%g", n);
	return 1;

}

int wtab(Context* cx, Node* goalscar, List* module)
{
	Node* n1 = goalscar->Cdr();
	int rn;
	
	cxpush(cx, goalscar);
	cxpush(cx, module);	
	if ((rn = FuncArg(cx, n1, module)) <= 0) {
		cxpop(cx);
		cxpop(cx);
		return rn;
	}
	cxpop(cx);
	cxpop(cx);

	if (ListLength(n1) != 0) {
		syserr("usage : <wtab>\n");
		return 0;
	}
	fprintf(cx->ioout, "\t");
	return 1;
}

int fr(Context* cx, Node* n1)
{
	std::string s1;
 	std::string s2 = "";
	int	n;
	int i, s1len;


	n1 = n1->Val();
	
	if (ListLength(n1) != 3) {
		syserr("usage : <fr VAR STRINGS WIDTH>\n");
		return 0;
	}
	
	if (n1->Car()->kind() != UNDEF) {
		return 0;
	}
	Node* nvar = n1->Car();

	if (n1->Cdr()->Car()->kind() != ATOM) {
		return 0;
	}
	if (!((Atom*)n1->Cdr()->Car())->toString(s1)) {
		return 0;
	}
	

	if (n1->Cdr()->Cdr()->Car()->kind() != ATOM) {
		return 0;
	} else if (!((Atom*)n1->Cdr()->Cdr()->Car())->toInt(n)) {
		return 0;
	}

	if (n <= 0) {
		return 0;
	}

	s1len = s1.length();
	i = s1len - n;
	if (i < 0) {
		for ( ; i < 0; i++) {
			s2 = s2 + " ";
		}
	}
	for ( ; i < s1len; i++) {
		s2 = s2 + s1[i];
	}
	
	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)s2.c_str()));

	PushStack(cx, Nil, Nil, env);

	return 1;
}

int fl(Context* cx, Node* n1)
{
	std::string s1;
 	std::string s2 = "";
	int	n;
	int i, s1len;


	n1 = n1->Val();
	
	if (ListLength(n1) != 3) {
		syserr("usage : <fl VAR STRINGS>\n");
		return 0;
	}
	
	if (n1->Car()->kind() != UNDEF) {
		return 0;
	}
	Node* nvar = n1->Car();

	if (n1->Cdr()->Car()->kind() != ATOM) {
		return 0;
	}
	if (!((Atom*)n1->Cdr()->Car())->toString(s1)) {
		return 0;
	}
	

	if (n1->Cdr()->Cdr()->Car()->kind() != ATOM) {
		return 0;
	} else if (!((Atom*)n1->Cdr()->Cdr()->Car())->toInt(n)) {
		return 0;
	}

	if (n <= 0) {
		return 0;
	}

	s1len = s1.length();
	for (i = 0; i < n; i++) {
		if (i >=s1len) {
			s2 = s2 + " ";
		} else {
			s2 = s2 + s1[i];
		}
	}
	
	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)s2.c_str()));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}


int eq(Context* cx, Node* goalscar)
{
	Node*	env = Nil->Cons(Nil);
	
	Node* n = goalscar->Cdr()->Val();
//printf("n "); n->print(); printf("\n");
	if (ListLength(n) != 2) {
		syserr("usage : <eq LIST1 LIST2>\n");
		return 0;
	}
	Node* n1 = n->Car();
	Node* n2 = n->Cdr()->Car();
//printf("n1 "); n1->print(); printf("\n");
//printf("n2 "); n2->print(); printf("\n");

	if (Unification(n1, n2, env, cx)) {
		PushStack(cx, Nil, Nil, env);
		return 1;
	}
	return -1;
}

int noteq(Context* cx, Node* goalscar)
{
	Node*	env = Nil->Cons(Nil);
	
	Node* n = goalscar->Cdr()->Val();
//printf("n "); n->print(); printf("\n");
	if (ListLength(n) != 2) {
		syserr("usage : <noteq LIST1 LIST2>\n");
		return 0;
	}
	Node* n1 = n->Car();
	Node* n2 = n->Cdr()->Car();
//printf("n1 "); n1->print(); printf("\n");
//printf("n2 "); n2->print(); printf("\n");

	if (Unification(n1, n2, env, cx)) {
		PushStack(cx, Nil, Nil, env);
		return -1;
	}
	return 1;
}

int isNil(Node* n)
{
	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

//printf("isNil "); n->print(); printf("\n");
	if (n->Car() == Nil) {
		return 1;
	} else {
		return -1;
	}
}

int isAtom(Node* n)
{
	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

//printf("isAtom "); n->print(); printf("\n");
	if (n->Car()->kind() == ATOM) {
		return 1;
	} else {
		return -1;
	}
}

int isList(Node* n)
{
//PrintNode("isList n ", n);
	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

//printf("isList "); n->print(); printf("\n");
	if (n->Car()->kind() == LIST) {
		return 1;
	} else {
		return -1;
	}
}

int isPred(Node* n)
{
	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

//printf("isPred "); n->print(); printf("\n");
	if (n->Car()->kind() == PRED) {
		return 1;
	} else {
		return -1;
	}
}

int isVar(Node* n)
{
	n = n->Cdr();
	if (ListLength(n) != 1) {
		return -1;
	}

//printf("isVar "); n->print(); printf("\n");
	if ((n->Car()->kind() == VAR) || (n->Car()->kind() == UNDEF)) {
		return 1;
	} else {
		return -1;
	}

}

int isUndefVar(Node* n)
{
	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

//printf("isUndefVar "); n->print(); printf("\n");
	if (n->Car()->kind() == UNDEF) {
		return 1;
	} else {
		return -1;
	}
}

int isFloat(Node* n)
{
	double	d;
	
	n = n->Cdr()->Val();
	if (ListLength(n) != 1) {
		return -1;
	}

//printf("isFloat "); n->print(); printf("\n");
	if (n->Car()->kind() != ATOM) {
		return 0;
	}
	if (((Atom*)(n->Car()))->toFloat(d)) {
		return 1;
	} else {
		return -1;
	}

}

int isInteger(Node* n)
{
	int	i;
	
	n = n->Cdr()->Val();
//PrintNode("isInteger ", n->Car());
	if (ListLength(n) != 1) {
		return -1;
	}

//printf("isInteger "); n->print(); printf("\n");
	if (n->Car()->kind() != ATOM) {
		return 0;
	}
	if (((Atom*)(n->Car()))->toInt(i)) {
		return 1;
	} else {
		return -1;
	}
}

int isTrue(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		return -1;
	}

	Node* gl = goalscar->Cdr()->Car()->Val();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	int r;
	if ((r=Unify(cx2, gl, cx->module))) {
		cx->Merge(cx2);
	}

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	return r;
}

int isFalse(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		return -1;
	}

	Node* gl = goalscar->Cdr()->Car()->Val();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	int r;
	if ((r=Unify(cx2, gl, cx->module))) {
		cx->Merge(cx2);
	}

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	if (r > 0) {
		return -1;
	} else if (r == 0) {
		return 1;
	}
	return r;
}

int isUnknown(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		return -1;
	}

	Node* gl = goalscar->Cdr()->Car()->Val();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	int r;
	if ((r=Unify(cx2, gl, cx->module))) {
		cx->Merge(cx2);
	}

	cxpop(cx2);

	delete cx2;
	cx2 = 0;

	if (r > 0) {
		return -1;
	} else if (r == 0) {
		return -1;
	}
	return 1;
}

int DoOpenR(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) < 2) {
		return -1;
	}

	Node* fname = goalscar->Cdr()->Car()->Val();
	if (fname->kind() != ATOM) {
		return -1;
	}
	
	std::string sfname;
	((Atom*)fname)->toString(sfname);

	FILE* fd;
	fd = fopen(sfname.c_str(), "rb");
	if (fd == NULL) {
		return -1;
	}
	

	Node*	gl = goalscar->Cdr()->Cdr();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = fd;
	cx2->ioout = cx->ioout;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;

	cxpush(cx2, gl);

	int r;
	r=Unify(cx2, gl, cx->module);

	cxpop(cx2);

	if (cx2->ioin != stdin) {
		fclose(cx2->ioin);
	}

	delete cx2;
	cx2 = 0;

	return r;
}

int DoOpenW(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) < 2) {
		return -1;
	}

	Node* fname = goalscar->Cdr()->Car()->Val();
	if (fname->kind() != ATOM) {
		return -1;
	}
	
	std::string sfname;
	((Atom*)fname)->toString(sfname);
	FILE* fd;
	fd = fopen(sfname.c_str(), "w");
	if (fd == NULL) {
		return -1;
	}


	Node*	gl = goalscar->Cdr()->Cdr();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = fd;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;
	
	cxpush(cx2, gl);

	int r;
	r=Unify(cx2, gl, cx->module);

	cxpop(cx2);

	if (cx2->ioout != stdout) {
		fclose(cx2->ioout);
	}

	delete cx2;
	cx2 = 0;

	return r;
}

int DoOpenWP(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) < 2) {
		return -1;
	}

	Node* fname = goalscar->Cdr()->Car()->Val();
	if (fname->kind() != ATOM) {
		return -1;
	}
	
	std::string sfname;
	((Atom*)fname)->toString(sfname);
	FILE* fd;
	fd = fopen(sfname.c_str(), "w+");
	if (fd == NULL) {
		return -1;
	}


	Node*	gl = goalscar->Cdr()->Cdr();

	Context *cx2 = new Context(cx->module);
	cx2->ioin = cx->ioin;
	cx2->ioout = fd;
	cx2->tokenflag = cx->tokenflag;
	cx2->token = cx->token;
	
	cxpush(cx2, gl);

	int r;
	r=Unify(cx2, gl, cx->module);

	cxpop(cx2);

	if (cx2->ioout != stdout) {
		fclose(cx2->ioout);
	}

	delete cx2;
	cx2 = 0;

	return r;
}

int DoGetc(Context* cx, Node* goalscar)
{

	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <getc VAR> \n");
		return 0;
	}

	if (goalscar->Cdr()->Cdr() != Nil) {
		return 0;
	}
	
	Node* v = goalscar->Cdr()->Car()->Val();

	if (v->kind() == UNDEF) {
		Node* env = Nil->Cons(Nil);

		int c = fgetc(cx->ioin);

		if (c == EOF) {
			return 0;
		}

		char ca[2];
		ca[0] = c;
		ca[1] = 0;		
		SetEnv(env, v);
		((Undef*)v)->Set(mka(ca));
		PushStack(cx, Nil, Nil, env);
		
		return 1;
	} else if (v->kind() == ATOM) {
		int c = fgetc(cx->ioin);

		if (c == EOF) {
			return 0;
		}

		std::string s;
		((Atom*)v)->toString(s);
		if (s.length() != 1) {
			return 0;
		}
		
		return ((s.c_str())[0] == c);
	} else {
		return 0;
	}
			
	return 0;
}

int DoPutc(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		syserr("usage : <putc STRINGS>\n");
		return 0;
	}

	Node* g = goalscar->Val();
	if (g->Cdr()->kind() == LIST) {
		if (g->Cdr()->Car()->kind() == ATOM) {
			std::string s;
			((Atom*)g->Cdr()->Car())->toString(s);
			putc(s.c_str()[0], cx->ioout);
			return 1;
		}
	}
	return 0;
				
}

#define MAXCLINE	4096

int GetLine(Context* cx, Node* goalscar)
{
	char*	cline;
	
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll < 1) {
		syserr("usage : <getline VAR [PRED]>\n");
		return 0;
	}
	Node* nvar = goalscar->Car();
	if (nvar->kind() != UNDEF) {
		syserr("getline : the first argument is not a variable.");
		return 0;
	}

	Node* npred;
	if (ll >= 2) {
		npred = goalscar->Cdr();
	}

#ifndef __MINGW32__
	if (cx->ioin == stdin) {
		cline = readline(NULL);
		if (cline != 0) {
			add_history(cline);
		} else {
			cline = (char*)malloc(2);
			cline[0] = 0;
		}
	} else {
		cline = (char*)malloc(MAXCLINE);
		if (fgets(cline, MAXCLINE, cx->ioin) == NULL) {
			free(cline);
			return 0;
		}
		cline[MAXCLINE-1] = 0;
	}
#else
	cline = (char*)malloc(MAXCLINE);
	if (fgets(cline, MAXCLINE-1, cx->ioin) == NULL) {
		free(cline);
		return 0;
	}
	cline[MAXCLINE-1] = 0;
#endif /* __MINGW32__ */

	int n = strlen(cline);
	for (int i = n-1; i >= 0; i--) {
		int c = cline[i];
		if ((c == '\n') || (c == '\r')) {
			cline[i] = 0;
		} else {
			break;
		}
	}
	
	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(cline));

	if (ll >= 2) {
		char tmpfilename[MAXPATHLEN];
		strncpy(tmpfilename, tmppath, MAXPATHLEN);
		strcat(tmpfilename, "/descXXXXXX");
		FILE* fd = MksTemp(tmpfilename);
		if (fd == NULL) {
			printf("tmpfile : cannot open tmp file \n");
			return 0;
		}
		
		int rn;
		Context* cx2 = new Context(cx->module);
		cx2->inherit = cx->inherit;
		cx2->ioin = cx->ioin;
		cx2->ioout = cx->ioout;
		cx2->tokenflag = cx->tokenflag;

		cx2->ioin = fd;

		fprintf(cx2->ioin, "%s", cline);
		free(cline);

		rewind(cx2->ioin);

		cxpush(cx2, goalscar);
		cxpush(cx2, nvar);
		cxpush(cx2, npred);
		cxpush(cx2, env);
		if ((rn=Unify(cx2, npred, cx->module))>0) {
			cx->Merge(cx2);

			fclose(cx2->ioin);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			delete cx2;
			cx2 = 0;

			PushStack(cx, Nil, Nil, env);

			unlink(tmpfilename);
			return rn;
		} else {
			fclose(cx2->ioin);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			cxpop(cx2);
			delete cx2;
			cx2 = 0;

			unlink(tmpfilename);
			return rn;
		}

	}

	return 1;
}

int TmpFile(Context* cx, Node* goalscar)
{
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll != 1) {
		syserr("usage : <tmpfile VAR>\n");
		return 0;
	}
	Node* nvar = goalscar->Car();
	if (nvar->kind() != UNDEF) {
		syserr("tmpfile : the first argument is not a variable.");
		return 0;
	}

	char tmpfilename[MAXPATHLEN];
	strncpy(tmpfilename, tmppath, MAXPATHLEN);
	strcat(tmpfilename, "/descXXXXXX");
	FILE* fd = MksTemp(tmpfilename);
	if (fd == NULL) {
		printf("tmpfile : cannot open tmp file \n");
		return 0;
	}

	fclose(fd);
	
	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)tmpfilename));

	unlink(tmpfilename);

	return 1;
}


int GetlineEval()
{
	char*	cline;
	std::string scline;
	int i;
	
	for(;;) {
#ifndef __MINGW32__
		cline = readline("? ");
		if (cline != 0) {
			add_history(cline);
		} else {
			cline = (char*)malloc(2);
			cline[0] = 0;
		}
#else
		printf("? ");
		cline = (char*)malloc(MAXCLINE);
		if (fgets(cline, MAXCLINE-1, stdin) == NULL) {
			free(cline);
			return 0;
		}
		cline[MAXCLINE-1] = 0;
#endif /* __MINGW32__ */

		// save tmpline
		char tmpfilename[MAXPATHLEN];
		strncpy(tmpfilename, tmppath, MAXPATHLEN);
		strcat(tmpfilename, "/descXXXXXX");
		FILE* fd = MksTemp((char*)tmpfilename);
		if (fd == NULL) {
			printf("getline : cannot open tmp file \n");
			return 0;
		}

		extern FILE* RdFp;
		FILE* fdsave = RdFp;
		RdFp = fd;

		// if '<>' is none in cline, added it.
		std::string sline = cline;
		
		for (;;) {
			
			for (i = strlen(cline)-1; i >= 0; i--) {
				if (cline[i] == ';') {
					break;
				}
			}
			if (cline[i] == ';') {
				break;
			}
#ifndef __MINGW32__
			cline = readline(NULL);
			if (cline != 0) {
				add_history(cline);
			} else {
				cline = (char*)malloc(2);
				cline[0] = 0;
			}
#else
			cline = (char*)malloc(MAXCLINE);
			if (fgets(cline, MAXCLINE-1, stdin) == NULL) {
				free(cline);
				return 0;
			}
			cline[MAXCLINE-1] = 0;
#endif /* __MINGW32__ */

			sline = sline + cline;
		}

		std::string sline2;

		for (i = 0; i < sline.length(); i++) {
			if (sline[i] == ' ') {
				continue;
			}
			if ((sline[i] == '<') || (sline[i] == ':') ||
					(sline[i] == ';')) {
				break;
			}
			sline2 = "<";
			sline2 = sline2 + sline;
			for (int j = sline2.length(); j >= 0; j--) {
				if (sline2[j] == ' ') {
					continue;
				}
				if (sline2[j] == ';') {
					sline2[j] = '>';
					sline2 = sline2 + ";";
					break;
				}
			}
			sline = sline2;
			break;
		}

		fprintf(RdFp, "?%s\n", sline.c_str());
		
		// eval
		// save tmpline
		rewind(RdFp);

		jmp_buf savejb;
		memcpy(&savejb, &program_jb, sizeof(jmp_buf));

		extern  int     NwccMain();	
		NwccMain();

		memcpy(&program_jb, &savejb, sizeof(jmp_buf));

		fclose(RdFp);
		RdFp = fdsave;
		
		free(cline);

		unlink(tmpfilename);
	}
}


int DoRegex(Context* cx, Node* goalscar)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 5) {
		return 0;
	}

	Node* nptn = goalscar->Car()->Val();
	if (nptn->kind() != ATOM) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}
	Node* nstr = goalscar->Cdr()->Car()->Val();
	if (nstr->kind() != ATOM) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}
	Node* nfore = goalscar->Cdr()->Cdr()->Car()->Val();
	if (nfore->kind() != UNDEF) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}
	Node* nmatch = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();
	if (nmatch->kind() != UNDEF) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}
	Node* nrest = goalscar->Cdr()->Cdr()->Cdr()->Cdr()->Car()->Val();
	if (nrest->kind() != UNDEF) {
		syserr("usage : <regex pattern strings forestr matchstr reststr> \n");
		return 0;
	}

	extern int Regex(std::string ptn, std::string str,
	                std::string& forestr, std::string& matchstr, 
	                	std::string& reststr);
	std::string	str, ptn,  fore, match, rest;

	((Atom*)nstr)->toString(str);
	((Atom*)nptn)->toString(ptn);

//printf("Regex ptn %s \n", ptn.c_str());
//printf("Regex str %s \n", str.c_str());
	
	Node* env = Nil->Cons(Nil);

	if (!Regex(ptn, str, fore, match, rest)) {
		return -1;
	}
	 
	SetEnv(env, nfore);
	((Undef*)nfore)->Set(mka((char*)fore.c_str()));

	SetEnv(env, nmatch);
	((Undef*)nmatch)->Set(mka((char*)match.c_str()));

	SetEnv(env, nrest);
	((Undef*)nrest)->Set(mka((char*)rest.c_str()));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int DoSub(Context* cx, Node* goalscar)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 4) {
		return 0;
	}

	Node* nptn = goalscar->Car()->Val();
	if (nptn->kind() != ATOM) {
		syserr("usage : <sub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* nstr = goalscar->Cdr()->Car()->Val();
	if (nstr->kind() != ATOM) {
		syserr("usage : <sub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* nreplace = goalscar->Cdr()->Cdr()->Car()->Val();
	if (nreplace->kind() != ATOM) {
		syserr("usage : <sub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* noutput = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();
	if (noutput->kind() != UNDEF) {
		syserr("usage : <sub pattern strings replacestr outputstr>\n");
		return 0;
	}

	extern int Sub(std::string ptn, std::string str,
	                std::string replacestr, std::string& outputstr);
	std::string	str, ptn,  replace, output;

	((Atom*)nptn)->toString(ptn);
	((Atom*)nstr)->toString(str);
	((Atom*)nreplace)->toString(replace);

//printf("Sub ptn %s \n", ptn.c_str());
//printf("Sub str %s \n", str.c_str());
	
	Node* env = Nil->Cons(Nil);

	if (!Sub(ptn, str, replace, output)) {
		return -1;
	}
	 
	SetEnv(env, noutput);
	((Undef*)noutput)->Set(mka((char*)output.c_str()));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

int DoGSub(Context* cx, Node* goalscar)
{
	goalscar = goalscar->Cdr()->Val();
	
	if (ListLength(goalscar) != 4) {
		return 0;
	}

	Node* nptn = goalscar->Car()->Val();
	if (nptn->kind() != ATOM) {
		syserr("usage : <gsub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* nstr = goalscar->Cdr()->Car()->Val();
	if (nstr->kind() != ATOM) {
		syserr("usage : <gsub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* nreplace = goalscar->Cdr()->Cdr()->Car()->Val();
	if (nreplace->kind() != ATOM) {
		syserr("usage : <gsub pattern strings replacestr outputstr>\n");
		return 0;
	}
	Node* noutput = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();
	if (noutput->kind() != UNDEF) {
		syserr("usage : <gsub pattern strings replacestr outputstr>\n");
		return 0;
	}

	extern int GSub(std::string ptn, std::string str,
	                std::string replacestr, std::string& outputstr);
	std::string	str, ptn,  replace, output;

	((Atom*)nptn)->toString(ptn);
	((Atom*)nstr)->toString(str);
	((Atom*)nreplace)->toString(replace);

//printf("GSub ptn %s \n", ptn.c_str());
//printf("GSub str %s \n", str.c_str());
	
	Node* env = Nil->Cons(Nil);

	if (!GSub(ptn, str, replace, output)) {
		return -1;
	}
	 
	SetEnv(env, noutput);
	((Undef*)noutput)->Set(mka((char*)output.c_str()));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
}

static int CheckDelimiterOne(char* str, int i, char* delm, int j)
{
	int k, ns, nd;

	ns = CharLen(str[i]);
	nd = CharLen(delm[j]);
	if (i+ns > strlen(str)) {
		return 0;
	}
	if (j+nd > strlen(delm)) {
		return 0;
	}
	
	if (ns == nd) {
		for (k = 0; k < nd; k++) {
			if (str[i+k] != delm[j+k]) {
				return 0;
			}
		}
		return 1;
	} else {
		return 0;
	}				
}

static int CheckDelimiter(char* str, int i, char* delm)
{
	int j, nd;

	for (j = 0; j < strlen(delm); j += nd) {
		nd = CharLen(delm[j]);
		if (CheckDelimiterOne(str, i, delm, j)) {
			return 1;
		}
	}
	return 0;
}

int Split(Context* cx, Node* goalscar)
{
	std::string Delimiters = " \t";

	Delimiters = Delimiters + CharSpace();

	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if ((ll != 2) && (ll != 3)) {
		syserr("usage : <split VAR STRINGS> or <split VAR STRINGS DELIMITERS>\n");
		return 0;
	}

	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		return 0;
	}

	Node* nstr = goalscar->Cdr()->Car()->Val();
	if (nstr->kind() != ATOM) {
		return 0;
	}
	std::string str;
	((Atom*)nstr)->toString(str);

	if (goalscar->Cdr()->Cdr() != Nil) {
		Node* ndelim = goalscar->Cdr()->Cdr()->Car()->Val();
		if (ndelim->kind() != ATOM) {
			return 0;
		}
		((Atom*)ndelim)->toString(Delimiters);
	}

	// skip space
	std::string tmpstr = "";
	int i, j, ns;
	Node* arg = Nil;
	for (i=0; i < str.length(); i += ns) {
		ns = CharLen(str[i]);
		if (!CheckDelimiter((char*)str.c_str(), i, 
				(char*)Delimiters.c_str())) {
			break;
		}
	}

	// appended terms
	for (; i < str.length(); i += ns) {
		ns = CharLen(str[i]);

		if (CheckDelimiter((char*)str.c_str(), i, 
					(char*)Delimiters.c_str())) {
			if (tmpstr != "") {
				arg = Append(arg, MkList(mka((char*)tmpstr.c_str())));
				tmpstr = "";
			}
		} else {
			for (j = 0; j < ns; j++) {
				tmpstr = tmpstr + str[i+j];
			}
		}
	}
	if (tmpstr != "") {
		arg = Append(arg, MkList(mka((char*)tmpstr.c_str())));
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(arg);

	PushStack(cx, Nil, Nil, env);
	
	return 1;
				
}

int Length(Context* cx, Node* goalscar)
{
	int n;
	goalscar = goalscar->Cdr()->Val();
	int ll = ListLength(goalscar);
	if (ll != 2) {
		syserr("usage : <length VAR STRINGS>\n");
		return 0;
	}
	Node* nvar = goalscar->Car()->Val();
	if (nvar->kind() != UNDEF) {
		return 0;
	}
	
	Node* nstr = goalscar->Cdr()->Car()->Val();
	if (nstr->kind() == LIST) {
		n = ListLength(nstr);
	} else {
		n = 1;
	}

	Node* env = Nil->Cons(Nil);
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(n));

	PushStack(cx, Nil, Nil, env);
	
	return 1;
				
}

int DoSetVar(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr();
//PrintNode("DoSetVar goalscar ", goalscar);
	int ll = ListLength(goalscar);
	if (ll < 2) {
//PrintNode("DoSetVar goalscar ", goalscar);
		return 0;
	}
	Node* nvar = goalscar->Car()->Val();
	Node* val  = goalscar->Cdr()->Car()->Val();
	Node* md   = module->Car();

//PrintNode("DoSetVar nvar ", nvar);
//PrintNode("DoSetVar val ", val);

	for ( ; md != Nil; md = md->Cdr()) {
		if (ModuleCompare(md->Car()->Car()->Car(), nvar) == 0) {
			((List*)md->Car())->SetCar(MkPred(Cons(nvar, MkList(val))));
			return 1;
		}
	}
	Asserta(module, MkList(MkPred(Cons(nvar, MkList(val)))));

	return 1;
}


int DoSetArray(Context* cx, Node* goalscar, List* module)
{
	goalscar = goalscar->Cdr();
//PrintNode("DoSetArray goalscar ", goalscar);
	int ll = ListLength(goalscar);
	if (ll < 3) {
//PrintNode("DoSetArray goalscar ", goalscar);
		return 0;
	}
	Node* nvar  = goalscar->Car()->Val();
	Node* val   = goalscar->Cdr()->Car()->Val();
	Node* index = goalscar->Cdr()->Cdr()->Val();
	Node* md = module->Car();

//PrintNode("DoSetArray val ", val);
//PrintNode("DoSetArray nvar ", nvar);
//PrintNode("DoSetArray index ", index);

	for ( ; md != Nil; md = md->Cdr()) {
		if (ModuleCompare(md->Car()->Car()->Car(), nvar) == 0) {
			if (md->Car()->Car()->Cdr()->Cdr()->Eq(index)) {
				((List*)md->Car())->SetCar(
					MkPred(Cons(nvar, Cons(val, index))));
				return 1;
			}
		}
	}
	Asserta(module, MkList(MkPred(Cons(nvar, Cons(val, index)))));

	return 1;
}

int Random(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();

	if (v->kind() == UNDEF) {
		int rd;
		Node* env = Nil->Cons(Nil);
#ifndef __MINGW32__
		rd = (int)random();
#else
		return 0;
#endif /* __MINGW32__ */

		SetEnv(env, v);
		((Undef*)v)->Set(mka(rd));
		PushStack(cx, Nil, Nil, env);
		
		return 1;
	} else {
		return 0;
	}
			
	return 0;
}


int Sin(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)sin(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}


int Cos(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)cos(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}


int Tan(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)tan(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}


int ASin(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)asin(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}


int ACos(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)acos(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}


int ATan(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)atan(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}


int Log(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)log(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}



int Exp(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)exp(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}


int Sqrt(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)sqrt(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}


int Abs(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((double)fabs(f)));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}


int Int(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();
	double f;

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	if ((ag->kind() != ATOM) || (!((Atom*)ag)->toFloat(f))) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);


	SetEnv(env, v);
	((Undef*)v)->Set(mka((int)f));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Car(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, v);
	((Undef*)v)->Set(ag->Car());
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Cdr(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag = goalscar->Cdr()->Cdr()->Car()->Val();

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, v);
	((Undef*)v)->Set(ag->Cdr());
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Cons(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		return 0;
	}

	Node* v = goalscar->Cdr()->Car()->Val();
	Node* ag1 = goalscar->Cdr()->Cdr()->Car()->Val();
	Node* ag2 = goalscar->Cdr()->Cdr()->Cdr()->Car()->Val();

	if (v->kind() != UNDEF) {
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, v);
	((Undef*)v)->Set(Cons(ag1, ag2));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}


int Char(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}
	
	if (nstr->kind() != ATOM) {
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	int i;
	for (i = 0; i < s.length(); i++) {
		n = Append(n, MkList(mka((int)(unsigned char)(s[i]))));
	}

	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int Concat(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nlist = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}
	
	if (nlist->kind() != LIST) {
		return 0;
	}


	std::string s="";
	Node*	n;
	int nn;
	for (n = nlist; n->kind() != ATOM; n=n->Cdr()) {
		n=n->Val();
		if (n->Car()->kind() == ATOM) {
			if (!((Atom*)(n->Car()))->toInt(nn)) {
				return 0;
			}
			s = s+char(nn);
		} else if (n->Car()->kind() == LIST) {
			Node* n2;
			for (n2 = n->Car(); n2->kind() != ATOM; n2=n2->Cdr()) {
				if (n2->Car()->kind() != ATOM) {
					return 0;
				}
				if (!((Atom*)(n2->Car()))->toInt(nn)) {
					return 0;
				}
				s = s+char(nn);
			}			
		} else {
			return 0;
		}
	}

	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka((char*)s.c_str()));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}


int SetCode(Context* cx, Node* goalscar)
{
	extern std::string code;
	
	if (ListLength(goalscar->Cdr()) != 1) {
		return 0;
	}

	Node* ncode = goalscar->Cdr()->Car()->Val();
	std::string scode;
	if (ncode->kind() == UNDEF) {
		Node* env = Nil->Cons(Nil);

		SetEnv(env, ncode);
		((Undef*)ncode)->Set(mka((char*)code.c_str()));
		PushStack(cx, Nil, Nil, env);

		return 1;
	} else if (ncode->kind() != ATOM) {
		return 0;
	}
	
	((Atom*)ncode)->toString(scode);
	if ((scode == "EUCJP") || (scode == "EUC-JP")|| (scode == "EUC")) {
		code = "EUC";
	} else if ((scode == "SJIS") || (scode == "SHIFT-JIS")) {
		code = "SJIS";
	} else if ((scode == "UTF8") || (scode == "UTF-8")) {
		code = "UTF8";
	}
	return 1;	
}


int CodeCharPrd(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}
	
	if (nstr->kind() != ATOM) {
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	n = CodeChar((char*)s.c_str());
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}


int UTF8CharPrd(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}
	
	if (nstr->kind() != ATOM) {
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	n = UTF8Char((char*)s.c_str());
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int EUCCharPrd(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}
	
	if (nstr->kind() != ATOM) {
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	n = EUCChar((char*)s.c_str());
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int SJISCharPrd(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* nvar = goalscar->Cdr()->Car()->Val();
	Node* nstr = goalscar->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}
	
	if (nstr->kind() != ATOM) {
		return 0;
	}

	std::string s;
	((Atom*)nstr)->toString(s);
	Node*	n=Nil;

	n = SJISChar((char*)s.c_str());
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(n);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int And(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* n2   = g->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}

	if (n1->kind() != ATOM) {
		return 0;
	}
	
	int nn1;
	if (!((Atom*)n1)->toInt(nn1)) {
		return 0;
	}
	
	if (n2->kind() != ATOM) {
		return 0;
	}
	
	int nn2;
	if (!((Atom*)n2)->toInt(nn2)) {
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	int nn3;
	if (nn1 != 1) {
		nn3 = -1;
	} else if (nn2 != 1) {
		nn3 = -1;
	} else {
		nn3 = 1;
	}
	
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn3));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Or(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* n2   = g->Cdr()->Cdr()->Car()->Val();


	if (nvar->kind() != UNDEF) {
		return 0;
	}

	if (n1->kind() != ATOM) {
		return 0;
	}
	
	int nn1;
	if (!((Atom*)n1)->toInt(nn1)) {
		return 0;
	}
	
	if (n2->kind() != ATOM) {
		return 0;
	}
	
	int nn2;
	if (!((Atom*)n2)->toInt(nn2)) {
		return 0;
	}
	
	int nn3;
	if (nn1 == 1) {
		nn3 = 1;
	} else if (nn2 == 1) {
		nn3 = 1;
	} else {
		nn3 = -1;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn3));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int Xor(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* n2   = g->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}

	if (n1->kind() != ATOM) {
		return 0;
	}
	
	int nn1;
	if (!((Atom*)n1)->toInt(nn1)) {
		return 0;
	}
	
	if (n2->kind() != ATOM) {
		return 0;
	}
	
	int nn2;
	if (!((Atom*)n2)->toInt(nn2)) {
		return 0;
	}
	
	int nn3;
	if ((nn1 == 1) && (nn2 == 1)){
		nn3 = -1;
	} else if (nn1 == 1) {
		nn3 = 1;
	} else if (nn2 == 1) {
		nn3 = 1;
	} else {
		nn3 = -1;
	}

	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn3));
	PushStack(cx, Nil, Nil, env);
		
	return 1;

}

int BitNot(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car();
	Node* n1   = g->Cdr()->Car();

	if (nvar->kind() != UNDEF) {
		return 0;
	}

	if (n1->kind() != ATOM) {
		return 0;
	}
	
	int nn1;
	if (!((Atom*)n1)->toInt(nn1)) {
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(~nn1));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int ShiftL(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* nsft = g->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}

	if (n1->kind() != ATOM) {
		return 0;
	}
	
	int nn1;
	if (!((Atom*)n1)->toInt(nn1)) {
		return 0;
	}
	
	if (nsft->kind() != ATOM) {
		return 0;
	}
	
	int nnsft;
	if (!((Atom*)nsft)->toInt(nnsft)) {
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn1<<nnsft));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int ShiftR(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 3) {
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car()->Val();
	Node* n1   = g->Cdr()->Car()->Val();
	Node* nsft   = g->Cdr()->Cdr()->Car()->Val();

	if (nvar->kind() != UNDEF) {
		return 0;
	}

	if (n1->kind() != ATOM) {
		return 0;
	}
	
	int nn1;
	if (!((Atom*)n1)->toInt(nn1)) {
		return 0;
	}
	
	if (nsft->kind() != ATOM) {
		return 0;
	}
	
	int nnsft;
	if (!((Atom*)nsft)->toInt(nnsft)) {
		return 0;
	}
	
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(nn1>>nnsft));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int DoMkPred(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 2) {
		syserr("usage : <mkpred LIST> \n");
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car();
	Node* n1   = g->Cdr()->Car();

	if (nvar->kind() != UNDEF) {
		return 0;
	}

	Node* prd = MkPred(n1);
		
	Node* env = Nil->Cons(Nil);

	SetEnv(env, nvar);
	((Undef*)nvar)->Set(prd);
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}

int Tron()
{
	TraceFlag = 1;
	return 1;
}

int Troff()
{
	TraceFlag = 0;
	return 1;
}


int DoCountNode(Context* cx, Node* goalscar)
{
	if (ListLength(goalscar->Cdr()) != 1) {
		return 0;
	}

	Node* g    = goalscar->Cdr()->Val();
	Node* nvar = g->Car();

	if (nvar->kind() != UNDEF) {
		return 0;
	}

	Node* env = Nil->Cons(Nil);

	int n = CountNode();
	
	SetEnv(env, nvar);
	((Undef*)nvar)->Set(mka(n));
	PushStack(cx, Nil, Nil, env);
		
	return 1;
}
