package org.zkforge.timeline;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.SortedSet;
import java.util.TimeZone;
import java.util.TreeSet;

import org.zkforge.json.simple.JSONArray;
import org.zkforge.timeline.data.OccurEvent;
import org.zkforge.timeline.decorator.HighlightDecorator;
import org.zkforge.timeline.event.BandScrollEvent;
import org.zkforge.timeline.impl.TimelineComponent;
import org.zkforge.timeline.util.TimelineUtil;
import org.zkoss.lang.Objects;
import org.zkoss.xml.HTMLs;
import org.zkoss.zk.au.AuScript;
import org.zkoss.zk.au.Command;
import org.zkoss.zk.ui.Component;
import org.zkoss.zk.ui.UiException;
import org.zkoss.zk.ui.event.Event;
import org.zkoss.zk.ui.event.EventListener;
import org.zkoss.zul.ListModel;
import org.zkoss.zul.ListModelList;
import org.zkoss.zul.event.ListDataEvent;
import org.zkoss.zul.event.ListDataListener;

/**
 * The Bandinfo component.
 * 
 * <p>
 * See also <a href="http://simile.mit.edu/timeline">MIT Timeline</a>
 * 
 * @author WeiXing Gu, China
 */
public class Bandinfo extends TimelineComponent {

	private ListModel _model;

	private SortedSet _events = new TreeSet();//

	private ArrayList _eventList = new ArrayList();// store the event's index;

	private Date _min;

	private Date _max;

	private transient ListDataListener _dataListener;

	private String _width = "70%";

	private String _intervalUnit = "month";

	private int _intervalPixels = 100;

	private boolean _highlight = true;

	private boolean _showEventText = true;

	private String _syncWith;

	private TimeZone _timeZone = TimeZone.getDefault();

	private float _trackHeight = (float) 1.5;

	private float _trackGap = (float) 0.5;

	private Date _date = new Date();

	private String _eventSourceUrl;

	// public String getOuterAttrs() {
	//
	// return "";
	// }

	// public String getContent() {
	//
	// return "";
	// }

	public String getInnerAttrs() {
		final String attrs = super.getInnerAttrs();
		final StringBuffer sb = new StringBuffer(64);
		if (attrs != null) {
			sb.append(attrs);
		}

		HTMLs.appendAttribute(sb, "z.pid", getParent().getUuid());
		HTMLs.appendAttribute(sb, "z.highlight", isHighlight());
		HTMLs.appendAttribute(sb, "z.width", getWidth());
		HTMLs.appendAttribute(sb, "z.intervalUnit", TimelineUtil
				.convertIntervalUnitFromName(_intervalUnit));
		HTMLs.appendAttribute(sb, "z.intervalPixels", getIntervalPixels());
		HTMLs.appendAttribute(sb, "z.showEventText", isShowEventText());
		HTMLs.appendAttribute(sb, "z.timeZone", _timeZone.getRawOffset()
				/ (1000 * 60 * 60));
		HTMLs.appendAttribute(sb, "z.trackHeight", String
				.valueOf(getTrackHeight()));
		HTMLs.appendAttribute(sb, "z.syncWith", findSyncWithIndex(_syncWith));
		HTMLs.appendAttribute(sb, "z.trackGap", String.valueOf(getTrackGap()));

		HTMLs.appendAttribute(sb, "z.date", TimelineUtil
				.formatDateTime(getDate()));
		HTMLs.appendAttribute(sb, "z.eventSourceUrl", _eventSourceUrl);
		return sb.toString();
	}

	public void setParent(Component parent) {
		if (parent != null && !(parent instanceof Timeline))
			throw new UiException("Unsupported parent for bandinfo: " + parent);
		super.setParent(parent);
	}

	/**
	 * @return the width
	 */
	public String getWidth() {
		return _width;
	}

	/**
	 * @param width
	 *            the width to set
	 */
	public void setWidth(String width) {
		if (!Objects.equals(_width, width)) {
			_width = width;
			// smartUpdate("z.width", _width);
			invalidate();
		}
	}

	/**
	 * @return the intervalPixels
	 */
	public int getIntervalPixels() {
		return _intervalPixels;
	}

	/**
	 * @param intervalPixels
	 *            the intervalPixels to set
	 */
	public void setIntervalPixels(int intervalPixels) {
		if (intervalPixels != _intervalPixels) {
			_intervalPixels = intervalPixels;
			// smartUpdate("z.intervalPixels", intervalPixels);
			invalidate();
		}
	}

	/**
	 * @return the intervalUnit
	 */
	public String getIntervalUnit() {
		return _intervalUnit;
	}

	/**
	 * @param intervalUnit
	 *            the intervalUnit to set
	 */
	public void setIntervalUnit(String intervalUnit) {
		if (!Objects.equals(intervalUnit, _intervalUnit)) {
			_intervalUnit = intervalUnit;
			// smartUpdate("z.intervalUnit", intervalUnit);
			invalidate();
		}

	}

	/**
	 * @return the showEventText
	 */
	public boolean isShowEventText() {
		return _showEventText;
	}

	/**
	 * @param showEventText
	 *            the showEventText to set
	 */
	public void setShowEventText(boolean showEventText) {
		if (showEventText != _showEventText) {
			_showEventText = showEventText;
			// smartUpdate("z.showEventText", showEventText);
			invalidate();
		}
	}

	private String findSyncWithIndex(String id) {
		Timeline parent = (Timeline) getParent();
		List l = parent.getChildren();
		for (int i = 0; i < l.size(); i++) {
			Bandinfo b = (Bandinfo) l.get(i);

			if (b.getId().equals(id)) {
				return String.valueOf(i);
			}
		}
		return "";
	}

	/**
	 * @return the _syncWith
	 */
	public String getSyncWith() {
		return _syncWith;
	}

	/**
	 * @param syncWith
	 *            the _syncWith to set
	 */
	public void setSyncWith(String syncWith) {
		if (!Objects.equals(syncWith, _syncWith)) {
			_syncWith = syncWith;
			// smartUpdate("z.syncWith", syncIndex);
			invalidate();
		}

	}

	/**
	 * @return the highlight
	 */
	public boolean isHighlight() {
		return _highlight;
	}

	/**
	 * @param highlight
	 *            the _highlight to set
	 */
	public void setHighlight(boolean highlight) {
		if (highlight != _highlight) {
			_highlight = highlight;
			// smartUpdate("z.highlight", highlight);
			invalidate();
		}
	}

	public TimeZone getTimeZone() {
		return _timeZone;
	}

	public void setTimeZone(TimeZone timeZone) {
		if (!Objects.equals(timeZone, _timeZone)) {
			_timeZone = timeZone;

			// smartUpdate("z.timeZone", timeZone.toString());
			invalidate();
		}
	}

	public float getTrackGap() {
		return _trackGap;
	}

	public void setTrackGap(float trackGap) {
		if (trackGap != _trackGap) {
			_trackGap = trackGap;
			// smartUpdate("z.trackGap", String.valueOf(trackGap));
			invalidate();
		}
	}

	public float getTrackHeight() {
		return _trackHeight;
	}

	public void setTrackHeight(float trackHeight) {
		if (trackHeight != _trackHeight) {
			_trackHeight = trackHeight;
			// smartUpdate("z.trackHeight", String.valueOf(trackHeight));
			invalidate();
		}
	}

	public Date getDate() {
		return _date;
	}

	public void setDate(Date date) {
		if (!Objects.equals(date, _date)) {
			_date = date;
			// smartUpdate("z.date", date.toString());
			invalidate();
		}
	}

	// -- Component --//
	public boolean insertBefore(Component child, Component insertBefore) {
		if (!(child instanceof Hotzone))
			throw new UiException("Unsupported child for timeline: " + child);
		return super.insertBefore(child, insertBefore);
	}

	public void addOccurEvent(OccurEvent event) {
		// if (!Objects.equals(event, _event)) {
		// _event = event;
		if (event == null)
			return;

		response("addOccurEvent" + event.getId(), new AuScript(this,
				"zkBandInfo.addOccurEvent(\"" + getUuid() + "\"" + ","
						+ event.toString() + ")"));

		// }
	}

	public void removeOccurEvent(OccurEvent event) {
		if (event == null)
			return;
		response("removeOccurEvent" + event.getId(), new AuScript(this,
				"zkBandInfo.removeOccurEvent(\"" + getUuid() + "\"" + ","
						+ "\"" + event.getId() + "\")"));
	}

	public void modifyOccurEvent(OccurEvent event) {
		if (event == null)
			return;
		response("modifyOccurEvent" + event.getId(), new AuScript(this,
				"zkBandInfo.modifyOccurEvent(\"" + getUuid() + "\"" + ","
						+ event.toString() + ")"));
	}

	// -- Component --//

	public String getEventSourceUrl() {
		return _eventSourceUrl;
	}

	public void setEventSourceUrl(String eventSourceUrl) {
		if (!Objects.equals(eventSourceUrl, _eventSourceUrl)) {
			_eventSourceUrl = eventSourceUrl;
			smartUpdate("z.eventSourceUrl", eventSourceUrl);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.zkoss.zk.ui.AbstractComponent#invalidate()
	 */
	// @Override
	public void invalidate() {
		// TODO Auto-generated method stub
		super.invalidate();
		if (getParent() != null)
			getParent().invalidate();
	}

	public void addManyOccurEvents(Iterator iter) {
		if (iter == null)
			return;
		JSONArray list = new JSONArray();
		while (iter.hasNext()) {
			OccurEvent e = (OccurEvent) iter.next();
			list.add(e);
		}

		response("addManyOccurEvent" + iter.hashCode(), new AuScript(this,
				"zkBandInfo.addManyOccurEvent(\"" + getUuid() + "\"" + ","
						+ list.toString() + ")"));

	}

	public void addHighlightDecorator(HighlightDecorator hd) {
		// decorators.add(hd);
		if (hd == null)
			return;
		response("addHighlightDecorator" + hd.getId(), new AuScript(this,
				"zkBandInfo.addHighlightDecorator(\"" + getUuid() + "\"" + ","
						+ hd.toString() + ")"));
	}

	public void removeHighlightDecorator(HighlightDecorator hd) {
		// decorators.remove(hd);
		if (hd == null)
			return;
		response("removeHighlightDecorator" + hd.getId(), new AuScript(this,
				"zkBandInfo.removeHighlightDecorator(\"" + getUuid() + "\""
						+ "," + hd.getId() + ")"));

	}

	public void showLoadingMessage(boolean show) {
		if (show) {
			response("showLoadingMessage", new AuScript(this,
					"zkTimeline.showLoadingMessage(\"" + getParent().getUuid()
							+ "\"" + ")"));
		} else {
			response("hideLoadingMessage", new AuScript(this,
					"zkTimeline.hideLoadingMessage(\"" + getParent().getUuid()
							+ "\"" + ")"));
		}

	}

	public void scrollToCenter(Date date) {
		if (date == null)
			return;
		response("scrollToCenter", new AuScript(this,
				"zkBandInfo.scrollToCenter(\"" + getUuid() + "\"" + ",\""
						+ date.toString() + "\")"));
	}

	static {
		new BandScrollCommand("onBandScroll", Command.IGNORE_OLD_EQUIV);
	}

	/**
	 * @return the model
	 */
	public ListModel getModel() {
		return _model;
	}

	/**
	 * @param model
	 *            the model to set
	 */
	public void setModel(ListModel model) {
		if (_model != null)
			_model.removeListDataListener(_dataListener);
		_model = model;

		if (_model != null) {
			_dataListener = new ListDataListener() {
				public void onChange(ListDataEvent event) {
					// TODO Auto-generated method stub
					onListDataChange(event);
				}
			};
			_model.addListDataListener(_dataListener);
			_events.clear();
			_eventList.clear();
			invalidate();

			int count = _model.getSize();
			for (int i = 0; i < count; i++) {
				Object o = _model.getElementAt(i);
				_events.add(o);
				_eventList.add(o);
			}
		}
		// listening band scroll event
		addEventListener("onBandScroll", new BandScrollListener());
	}

	private class BandScrollListener implements EventListener {

		public boolean isAsap() {
			// TODO Auto-generated method stub
			return true;
		}

		public void onEvent(Event event) {
			// TODO Auto-generated method stub
			BandScrollEvent e = (BandScrollEvent) event;
			Date newmin = e.getMin();
			Date newmax = e.getMax();
			if (_min == null && _max == null) {
				_min = newmin;
				_max = newmax;
				syncModel(_min, _max);
			}

			if (newmin.compareTo(_min) < 0) {
				syncModel(newmin, _min);
				_min = newmin;
			}

			if (newmax.compareTo(_max) > 0) {
				syncModel(_max, newmax);
				_max = newmax;
			}

		}
	};

	protected void syncModel(Date min, Date max) {
		OccurEvent e1 = new OccurEvent();
		e1.setStart(min);
		OccurEvent e2 = new OccurEvent();
		e2.setStart(max);
		SortedSet ss = _events.subSet(e1, e2);
		Iterator iter = ss.iterator();
		// System.out.println("live date :"+ss.size());
		addManyOccurEvents(iter);

	}

	protected void onListDataChange(ListDataEvent event) {
		// TODO Auto-generated method stub
		int lower = event.getIndex0();
		int upper = event.getIndex1();
		// System.out.println("lower=" + lower + "upper=" + upper);
		OccurEvent e1 = new OccurEvent();
		e1.setStart(_min);
		OccurEvent e2 = new OccurEvent();
		e2.setStart(_max);
		switch (event.getType()) {
		case ListDataEvent.INTERVAL_ADDED:
			for (int i = lower; i <= upper; i++) {
				OccurEvent oe = (OccurEvent) _model.getElementAt(i);
				_events.add(oe);
				_eventList.add(oe);

				if (oe.compareTo(e1) >= 0 && oe.compareTo(e2) <= 0)// lazy-load
					this.addOccurEvent(oe);// if the event is in [_min,_max]
				// then display it.
			}
			break;
		case ListDataEvent.INTERVAL_REMOVED:

			for (int i = upper; i >= lower; i--) {
				OccurEvent oe = (OccurEvent) _eventList.get(i);
				_events.remove(oe);
				_eventList.remove(oe);
				this.removeOccurEvent(oe);
				oe = null;
			}
			break;
		case ListDataEvent.CONTENTS_CHANGED:
			for (int i = lower; i <= upper; i++) {
				OccurEvent oe = (OccurEvent) _model.getElementAt(i);
				OccurEvent e = (OccurEvent) _eventList.get(i);

				_eventList.set(i, oe);

				_events.remove(e);
				_events.add(oe);
				this.removeOccurEvent(e);
				if (oe.compareTo(e1) >= 0 && oe.compareTo(e2) <= 0)
					// if the event is in [_min,_max]
					this.addOccurEvent(oe);// then display it.

			}
			break;

		}
	}

}
